package BM::Banners::LBannerAnalysis;
use strict;

use utf8;
use open ':utf8';

use std;
use base qw(BM::Banners::LBannerBase);

use List::Util qw(min max sum);

use BM::ContextSyns::ContextLinks;
use Utils::Words;
use Utils::Array;
use Utils::Urls;
use Utils::Dicts qw(load_words_dict);
use Utils::Sys;
use Utils::Sys qw(md5int);

########################################################
# Интерфейс
########################################################
#   get_banner_text_phl
#   parsed_banner_text
#   parse
#   categs_type
#   banner_type
#   native_town
#   has_vendors
#   get_phrases_minicategs
#   get_minicategs_siblings_hash
#   exminicategshash
#   vendorshash
#   goodswords
#   experiment
#   has_celebrities
#   banner_text_phrases
#   categs_minuswords
#   _compute_minicategs_stats
#   urlpage
#   textwords                                   снормализованные слова из текста баннера (title, body)
#   textwordshash                               хэшь из textwords
#   has_minuswords
#   minuswordshash
#   bannerwordshash
#   phraseswords
#   bannernormwordshash
#   bannersnormwordshash
#   homonymy_minuswordshash
#   update_banners_phrase_inf
#   diffphrsfilter
#   modelphrases
#   modelphraseshash
#   phrsfilterreason

#   get_catalogia_flags                 Получить флаги категорий

__PACKAGE__->mk_accessors(qw(
    cache_phrsfilterreason
));

########################################################
# Инициализация
########################################################

########################################################
# Инициализация класса
########################################################

our $diffphrs = {};
our $goods_words = {};
our $service_words = {};
our $info_words = {};
our $minus_media = {};
our $domain_flags = {};
#our $camp_categs = {};


# словари премиальных брендов
my %category_premium_brands;

sub class_init :RUN_ONCE {
    my ($class, $opt) = @_;
    my $log  = $opt->{logger};
    my $proj = $opt->{'proj'};

    $log->log('diff');

    open(F2, $opt->{'diff_words'});
    while(my $l = <F2>){
        chomp $l;
        my @a = split("\t",$l);
        next if @a < 2;
        my @w1 = map { word2snorm($_) } split(',', $a[0]);
        my @w2 = map { word2snorm($_) } split(',', $a[1]);  #!!!
        $_ =~ s/\s// for (@w1, @w2);
        for my $diffword (@w1) {
            $diffphrs->{$diffword} = [] unless $diffphrs->{$diffword};
            push(@{$diffphrs->{$diffword}}, @w2);
            for( @w2 ){
                $diffphrs->{$_} = [] unless $diffphrs->{$_};
                push(@{$diffphrs->{$_}}, $diffword);
            }
        }
    }
    close(F2);

    $goods_words   = load_words_dict($opt->{goods_words},   {snorm => 1});
    $service_words = load_words_dict($opt->{service_words}, {snorm => 1});
    $info_words    = load_words_dict($opt->{info_words},    {snorm => 1});

    open F2, $$opt{minus_media};
    while(<F2>) {
        chomp;
        my ($phr, $words) = split "\t";
        $$minus_media{$proj->phrase($phr)->snorm_phr} = [split ",", $words];
    }
    close F2;

    open F2, $$opt{dict_domain_flags};
    while(<F2>) {
        chomp;
        my ($domain, $flags) = split "\t";
        push @{$domain_flags->{$domain}}, split /,\s*/, $flags;
    }
    close F2;


=h
    if(open F2, $proj->banners_categories->camp_categs_file) {
        $camp_categs->{$_->[0]} = $_->[1] for map{chomp; [split "\t"]} <F2>;
        close F2;
    } else {
        $log->log("camp_categs_file not found");
    }
=cut

    $log->log('/diff');
}

########################################################
#Методы
########################################################

# заглушка;
# в бродматчинге используется объект BM::Banners::LBannerBM, где этот метод переопределен
sub experiment {
    my $self = shift;
    return {};
}

sub get_banner_text_phl {
    my ($self) = @_;
    my @texts = ();
    push @texts, $self->title;
    push @texts, $_ for split /(?:\.|\!|\?)\s+/, $self->body;
    return $self->proj->phrase_list( { phrases_arr=>\@texts } );
}

sub parsed_banner_text :CACHE {
    my ($self) = @_;
    #print $self->proj->stack_trace;
    my $ph = $self->banner_text_phrase->get_banner_prefiltered_phrase->post_prefilter_bnr;
    my $text = $ph->text;
    my ($brand, $model) = $self->parse;
    #print STDERR Dumper ([$brand, $model]);
    my $dict_notmodels = $self->proj->phrase->dict_notmodels; # если присутствует немодельное слово. то модель обнуляем
    my $hasnt_model = grep { exists( $dict_notmodels->{$_} ) } map { $self->proj->phrase($_)->norm_phr } split /\s+/, $text;
    return { brand => $brand, model => '', othertext => $text } if $hasnt_model;
    $text =~ s/\b$brand\b//ig if $brand;
    $text =~ s/\b$model\b//ig if $model;
    return { brand => $brand, model => $model, othertext => $text };
}

sub parse {
    my ( $self ) = @_;
    my $model;
    my $brand = $self->banner_text_phrase->get_brand;
    for my $phr ( $self->get_banner_text_phl->phrases ) {
        my %h = ();
        eval { %h = $phr->parse };
        #print Dumper (\%h);
        $model = $h{model} if ( $h{model} );
        return ($brand, $model) if $model;
    }
    return $brand, $model;
}


sub extend_diez_phr {
    my ($self, $phr) = @_;
    my $text = $self->title_body_phr->text;
    my $replacement = $phr->plus_text;
    $text =~ s/(\#[^\#]*\#)/ $replacement /g;
    return $self->proj->phrase($text);
}

sub sentences {
    my ($self) = @_;
    my ( $title, $body ) = map { (split /\+/)[0] } ( $self->title, $self->body );
    # разделитель используем чтобы не бить по предложениям числа с десятичной частью, отделенной точкой
    # и заменяем домены
    ( $title, $body ) = map { $_ = "$_"; s/(\d+)\.(\d+)/$1_DELIM_$2/g; s/\#/ /g; s/(?:^|(?:(?<=\W))(?P<tt>[a-z]{2,}))?\.ru(?=(?:[\s.,!?]|$))/"dmn_".($+{tt}||"")."_ru"/ieg; $_ } map { $_ || () } ( $title, $body );  ## no critic
    return map { s/_DELIM_/\./g; s/\s+/ /g; s/^ | $//g; $_ } map { split /[.!?:]/, $_ || () } map { $_ || ()} ( $title, $body );  ## no critic
}

sub categs_type :CACHE {
    my ($self) = @_;
    return 'movies' if $self->page && ($self->page->domain =~ /(^|\/|\.)ivi\.ru/);
    return 'movies' if grep { $_ eq 'Видеофильмы'} $self->upcategs;
}

# model
# brand
# subsection
sub banner_type :CACHE {
    my ($self) = @_;
    return $self->{external_type} if $self->{external_type};
    return $self->categs_type if $self->categs_type;
    my $prsinf = $self->parsed_banner_text;
    return 'model' if $prsinf->{'model'};
    return 'brand' if $prsinf->{'brand'};
    return 'subsection';
}

sub native_town :CACHE {
    my $self = shift;
    my $rinf = $self->region_inf;
    return if $rinf->{rule} ne 'native_town';
    return $rinf->{native_town};
}

sub has_vendors :CACHE {
    my ($self) = @_;
    my $res = keys %{$self->vendorshash} ? 1 : 0;
    return $res;
}

sub join_phrases_for_categorization {
    my $self = shift;
    my @phrases = $self->phrases;
    return $self->language->phrase(join (" ", map{$_->norm_phr} @phrases[0..min(9, $#phrases)]));
}

sub get_phrases_minicategs  :CACHE {
    my $self = shift;
    my $ph = $self->join_phrases_for_categorization;
    $ph->{dont_use_minicategs_cache} = 1;
    my $stats = $self->_compute_minicategs_stats([$ph]);
    my $MAX_CATEGS = 3;

    # искусственно ограничиваем количество категорий
    if (scalar keys %$stats > $MAX_CATEGS) {
        my @top = sort{$stats->{$b} <=> $stats->{$a}} keys %$stats;
        $stats = {map{$_=>$stats->{$_}} @top[0..$MAX_CATEGS-1]};
    }

    return sort keys %$stats;
}

sub get_minicategs_videodirectgroups_directids :CACHE {
    my ($self) = @_;
    return $self->proj->categs_tree->get_minicategs_videodirectgroups_directids_from_categs($self->get_minicategs);
}

sub get_domain_flags :CACHE {
    my ($self) = @_;
    my $match;
    # фильтруем параметры и вырезаем якорь
    my $fixed_url = lc($self->proj->page(url_to_unicode($self->url))->fixed_url);
    #вырезание якорей
    $fixed_url =~ s/#.*$//;
    #на всякий случай вырезание некорректных якорей перед параметрами
    $fixed_url =~ s/#[^?]*\?/\?/;
    my $page = $self->proj->page($fixed_url);

    my $domain = $page->domain;
    my $domain_2lvl = $page->domain_2lvl;

    #смотрим совпадение по домену, пути и параметрам
    if ( exists $domain_flags->{$domain . $page->uri} || exists $domain_flags->{$domain_2lvl . $page->uri} ) {
        my $match_domain = exists $domain_flags->{$domain_2lvl . $page->uri} ? $domain_2lvl : $domain;
        $match = $match_domain . $page->uri;
    }

    #не совпало - смотрим по домену и пути, поднимаясь по директориям
    unless ( $match ) {
        my $path = $self->proj->page($page->domain_path)->uri;
        while ($path && !exists $domain_flags->{$domain . $path} && !exists $domain_flags->{$domain_2lvl . $path} ) {
            $path =~ s/(\/$|[^\/]*$)//; #откусываем по куску от path, пробуем поймать флаги по частичному совпадению
        }
        if ($path) {
            my $match_domain = exists $domain_flags->{$domain_2lvl . $path} ? $domain_2lvl : $domain;
            $match = $match_domain . $path;
        }
    }
    #нет совпадения по куску урла - проверяем домены
    unless ($match) {
        $match = $domain;
        #нет для точного совпадения - берем родительский домен
        while ($match && !exists $domain_flags->{$match}) {
            #убираем самый нижний поддомен
            $match =~ s/^([-\w]+)//;
            # вообще мы здесь убираем точку, но тут стоит \W как защита от зацикливания при невалидных урлах
            $match =~ s/^\W//;
        }
    }
    my %flags = ();
    if ($match){
        foreach ( @{$domain_flags->{$match}} ) {
            $flags{$_} = $match;
        }
    }
    return \%flags;
}

my $catalogia_flags_plus = 0;
my $catalogia_flags_minus = 0;

sub get_catalogia_flags :CACHE {
    my ($self) = @_;
    return $self->_get_catalogia_flags;
}

sub get_catalogia_flags_analysis :CACHE {
    my ($self) = @_;
    $self->_get_catalogia_flags('analyse');
    return $self->{get_catalogia_flags_analysis};
}

sub _get_catalogia_flags {
    my ($self, $analyse) = @_;

    $analyse //= 0;

    my @analysis = ();
    my %prev = ();

    sub analysis_elem{
        #before перезаписывается в процессе
        my($name, $before, $after, $info) = @_;
        my %both = (%{$before},%{$after});
        my @added = sort grep {!exists $before->{$_} && exists $after->{$_}} keys %both;
        my @deleted = sort grep {exists $before->{$_} && !exists $after->{$_}} keys %both;
        %{$before} = ();
        $before->{$_} = $after->{$_} for keys %{$after};
        if ( $info ) {
            foreach ( @added, @deleted ) {
                $_ = $_ .' => '. $info->{$_} if exists $info->{$_};
            }
        }
        return {
            name => $name,
            add  => \@added,
            del  => \@deleted
        };
    };

    # флаги по тексту баннера
    #здесь не используется get_title_body_minicategs, потому что там менее надежные методы категоризации, чем это нужно для флагов
    my @minicategs = $self->preprocess_title_body->get_minicategs;
    my %h = map{$_=>1} $self->proj->categs_tree->get_catalogia_flags(@minicategs);
    # костыль для asocial, который нужен, потому что categs_tree->get_catalogia_flags отдает только флаги с префиксом _
    $h{asocial}++ if $self->proj->categs_tree->check_minicategs_flag(join("/", @minicategs), "asocial");

    push @analysis, analysis_elem('Категоризация текста баннера',\%prev,\%h) if $analyse;

    # флаги по нетекстовым способам категоризации баннера (внешняя категоризация, домен)
    # отключены по тикету CATALOGIA-1091
    #my @nontext_minicategs = $self->get_nontext_minicategs;
    #$h{$_}++ for $self->proj->categs_tree->get_catalogia_flags(@nontext_minicategs);
    # костыль для asocial, который нужен, потому что categs_tree->get_catalogia_flags отдает только флаги с префиксом _
    #$h{asocial}++ if $self->proj->categs_tree->check_minicategs_flag(join("/", @nontext_minicategs), "asocial");
    #push @analysis, analysis_elem('Категоризация баннера',\%prev,\%h) if $analyse;

    # флаги по словам
    my $markers = $self->proj->catalogia_markers->get_plus_markers($self->title_body_phr);
    $h{$_}++ for keys %$markers;

    push @analysis, analysis_elem('Плюс-слова из маркеров по тексту баннера',\%prev,\%h, $markers) if $analyse;

    # добавляем флаги по префильтрованному тексту, так как могут дописываться специальные слова для флагов (например, moderatebadwordtype)
    my $prefilter_markers = $self->proj->catalogia_markers->get_plus_markers($self->preprocess_title_body);
    $h{$_}++ for keys %$prefilter_markers;

    push @analysis, analysis_elem('Плюс-слова из маркеров по префильтрованному тексту баннера',\%prev,\%h, $prefilter_markers) if $analyse;


    #Исправление дыры в подстановке запрещенный текстов через шаблоны
    if($self->title_body_phr->text =~ /#/){
        #сначала проверяем категорийные флаги
        my $ct = $self->proj->categs_tree;
        for my $ph ( $self->phl->phrases ){
            $h{$_}++ for $ct->get_catalogia_flags($self->extend_diez_phr($ph)->get_minicategs);
        }

        # отдельно проверяем слова, разрываемые решётками
        my $text = $self->title_body_phr->text;
        $text =~ s/\#//g; # важно: не дописываем новых пробелов
        $h{$_}++ for $ct->get_catalogia_flags($self->proj->phrase($text)->get_minicategs);

        push @analysis, analysis_elem('Категории текста баннера после подстановки фраз в шаблон',\%prev,\%h) if $analyse;

        #а теперь проверяем маркеры
        my %template_markers = ();
        for my $ph ( $self->phl->phrases ){
            %template_markers = ( %template_markers, %{$self->proj->catalogia_markers->get_plus_markers($self->extend_diez_phr($ph))} );
        }
        #также проверим слова, разрываемые решетками
        %template_markers = ( %template_markers, %{$self->proj->catalogia_markers->get_plus_markers($self->proj->phrase($text))} );
        $h{$_}++ for keys %template_markers;

        push @analysis, analysis_elem('маркеры по тексту баннера после подстановки фраз в шаблон',\%prev,\%h, \%template_markers) if $analyse;
    }

    # модельный флаг
    # $h{model}++ if $self->banner_type eq "model";

    # минус-слова на флаги
    my $minus_markers = $self->proj->catalogia_markers->get_minus_markers($self->title_body_phr, [keys %h]);
    delete $h{$_} for keys %$minus_markers;

    push @analysis, analysis_elem('Фильтрация минус-словами из маркеров',\%prev,\%h, $minus_markers) if $analyse;

    # плюс-флаги и минус-флаги
    if(!$catalogia_flags_plus) {
        $catalogia_flags_plus = {};
        $catalogia_flags_minus = {};

        if(!open(F, $Utils::Common::options->{catalogia_flags_plus_minus})) {
            $self->log("ERROR: $!");
        } else {
            while(<F>) {
                chomp;
                my ($flag, $plus, $minus) = split "\t";

                next if !$flag;
                if(!$self->proj->categs_tree->get_flag_description("_$flag")) {
                    $self->log("WARNING: unknown flag '$flag' near '$plus\t$minus'");
                    next;
                }

                for my $pair ([$catalogia_flags_plus, $plus], [$catalogia_flags_minus, $minus]) {
                    my ($h, $text) = @$pair;

                    $h->{$flag} ||= {};

                    for my $other_flag (grep{$_} split /\s*,\s*/, ($text || "")) {
                        if(!$self->proj->categs_tree->get_flag_description("_$other_flag")) {
                            $self->log("WARNING: unknown flag '$other_flag' in '$text'");
                            next;
                        }

                        $h->{$flag}{$other_flag}++;
                    }
                }
            }

            close F;
        }
    }

    my %new_h = %h;
    for my $flag (keys %h) {
        $new_h{$_}++ for keys %{$catalogia_flags_plus->{$flag} || {}};
        delete $new_h{$_} for grep{$new_h{$_}} keys %{$catalogia_flags_minus->{$flag} || {}};
    }

    %h = %new_h;
    push @analysis, analysis_elem('Дополняющие и исключающие флаги',\%prev,\%h) if $analyse;

    my @regions_flags = $self->title_body_phr->get_regions_flags();
    $h{$_} = 1 for map {s/^_//; $_;} @regions_flags; ## no critic
    $h{'directmod_dict'} = 1 if $self->title_body_phr->check_directmod_dict_flag();

    push @analysis, analysis_elem('Региональные флаги и directmod_dict',\%prev,\%h) if $analyse;

    my $domain_flags_plus = {};
    my $domain_flags_minus = {};
    my $domain_flags = $self->get_domain_flags();
    for my $flag (keys %$domain_flags) {
        if ($flag =~ /^[-]/) {
            my $fltflag  = $flag;
            $fltflag =~ s/^([-])//;
            $domain_flags_minus->{$fltflag}++;
        }
        else {
            $domain_flags_plus->{$flag}++;
        }
    }
    $h{$_}++ for keys %{$domain_flags_plus};
    delete $h{$_} for grep{$h{$_}} keys %{$domain_flags_minus};

    push @analysis, analysis_elem('Доменные флаги',\%prev,\%h, $domain_flags) if $analyse;
    $self->{get_catalogia_flags_analysis} = \@analysis;

    push @analysis, analysis_elem('Итого',{},\%h) if $analyse;

    return sort keys %h;
}


our $bmbadctgs = { map {$_=>1} ( 'Универсальные магазины', ) };
sub get_minicategs_siblings_hash :CACHE {
    my $self = shift;
    my @cts = $self->get_minicategs;

    @cts = grep { ! $bmbadctgs->{$_} } @cts; #Удаляем часть широких категорий

    my %own = map { $_ => 'own' } @cts;
    my $h = {};

    for my $ct (@cts) {
        my $sibl_inf = $self->proj->categs_tree->get_minicategs_siblings_info($ct);
        while (my ($sibl, $name) = each %$sibl_inf) {
            $h->{$sibl} .= "sibling::$name/$ct/$sibl/" if !$own{$sibl};
        }
    }

    for my $ct (@cts) {
        for my $neph ($self->proj->categs_tree->get_minicateg_nephews($ct)) {
            $h->{$neph} .= "nephew/$ct/$neph/" if !$own{$neph};
        }
    }

    return { %$h, %own };
}

sub exminicategshash :CACHE {
    my ($self) = @_;

    my $ctree = $self->proj->categs_tree;

    my $h = $self->get_minicategs_siblings_hash;

    $h = { map{$_=>$h->{$_}} keys %$h};

    my @cts = $self->get_minicategs;
    @cts = grep { ! $bmbadctgs->{$_} } @cts; #Удаляем часть широких категорий

    for my $hier ($ctree->hier_minicategs(@cts)) {
        $h->{$hier} = "hier" if !$h->{$hier};
    }

    ##Добавляем путь на один уровень наверх
    #for my $c (@cts){
    #    $h->{$_} = "path" for grep { ! $h->{$_} } $ctree->get_minicateg_parent($c);
    #}

    return $h;
}

sub vendorshash :CACHE {
    my ($self) = @_;
    my $vndrs = $self->proj->models->vendors;
    return { map { $_ => 1 } grep {$vndrs->{$_}} ($self->textwords, $self->phraseswords) };
}

sub brands_phl :CACHE {
    my ($self) = @_;
    return $self->title_body_phr->get_brands_snorm_phl;
}

sub goodswords :CACHE {
    my ($self) = @_;
    my @gw = grep { $goods_words->{$_} } $self->textwords;
    unless(@gw){ #если не нашли упоминание товара
        my $vh = $self->vendorshash;
        if(keys %$vh){ #в тексте есть упоминание вендера
            my $mdl2type = $self->proj->models->vendormodel2modeltype; #преобразователь моделей в товар
            my $typesh = {}; #найденные товары
            my @tws = $self->textwords;
            for my $i (0..$#tws){
                my $w = $tws[$i];
                if( $vh->{$w} ){
                    my $wwcnt = min($i+4, $#tws);
                    my $mdltext = $w.' ';
                    for my $j ( $i+1 .. $wwcnt ){
                        my $admd = $tws[$j];
                        $admd =~ s/\-//g; #удаляем минусы, она часто встречаются
                        $mdltext .= $admd;
                        $typesh->{$mdl2type->{$mdltext}}++ if $mdl2type->{$mdltext};
                    }
                }
            }
            my @mdlgoods = keys %$typesh;
            push(@gw,@mdlgoods);
        }
    }
    return @gw;
}


sub servicewords :CACHE {
    my ($self) = @_;
    return ( grep { $service_words->{$_} } $self->textwords );
}
sub servicewordshash :CACHE {
    my ($self) = @_;
    return { map { $_ => 1 } $self->servicewords };
}
sub infowords :CACHE {
    my ($self) = @_;
    return ( grep { $info_words->{$_} } $self->textwords );
}
sub infowordshash :CACHE {
    my ($self) = @_;
    return { map { $_ => 1 } $self->infowords };
}

sub has_celebrities :CACHE {
    my ($self) = @_;
    return $self->banner_text_phrases->has_celebrities;
}

sub banner_text_phrases :CACHE {
    my ($self) = @_;
    my $proj = $self->proj;
    my $gdsph = $proj->phrase(join(' ',$self->goodswords));
    return $proj->phrase_list({ phrases_list => [$gdsph, $self->title_phr, $self->body_phr, $self->phrases] });
}

sub categs_minuswords :CACHE {
    my ($self) = @_;
    my $h = {};
    my $plus = {};

    # собираем плюс-слова категорий (чтобы вычесть их из минус-слов)
    for my $ct ($self->get_minicategs) {
        $plus->{$_}++ for $self->proj->categs_tree->get_minicateg_words($ct);
    }

    for my $ct ($self->get_minicategs) {
        my @a = $self->proj->categs_tree->get_minicateg_minus_words($ct);
        $h->{$_}++ for grep{!$self->bannerwordshash->{word2snorm $_} && !$plus->{word2snorm $_}} @a;
    }

    return sort keys %$h;
}

sub _compute_minicategs_stats {
    my ($self, $phrases) = @_;
    my $categs = {};
    for my $p (@$phrases) {
        $p->{dont_use_minicategs_cache} = 1 if $self->{dont_use_minicategs_cache};
        $categs->{$_}++ for $p->get_minicategs;
$self->{analysis} = $p->get_analysis;
    }
    return $categs;
}

sub urlpage: CACHE {
    my ($self) = @_;
    return $self->proj->page( $self->url );
}

sub textwords :CACHE {
    my ($self) = @_;
    return sort keys %{{ map {$_=>1} ($self->title_phr->snormwords, $self->body_phr->snormwords)}};
}

sub textwordshash :CACHE {
    my ($self) = @_;
    return { map { $_ => 1 } $self->textwords };
}

sub has_minuswords :CACHE {
    my ($self) = @_;
    return 1 if grep {$_} values %{$self->minuswordshash};
    return 0;
}

sub minuswordshash :CACHE {
    my ($self) = @_;
    my $bwh = $self->bannerwordshash;
    my %seen;
    for my $phr (@{$self->phl}) {
        for my $r ($phr->minus_phr->analyze_words) {
            $seen{$r->{word}} = 1 if !$bwh->{$r->{snorm}};
        }
    }
    return \%seen;
}

# хэш снорм-слов из текста и фраз
sub bannerwordshash :CACHE {
    my ($self) = @_;
    return { map { $_ => 1 } ($self->textwords, $self->phraseswords) };
}

sub phraseswords :CACHE {
    my ($self) = @_;
    return (map {$_->snormwords} $self->phrases);
}

sub bannernormwordshash :CACHE {
    my ($self) = @_;
    my @list = (
        $self->title_phr->normwords,
        $self->body_phr->normwords,
        map {$_->normwords} $self->phrases,
    );
    return { map { $_ => 1 } @list };
}

sub bannersnormwordshash :CACHE {
    my ($self) = @_;
    my @list = (
        $self->title_phr->snormwords,
        $self->body_phr->snormwords,
        map {$_->snormwords} $self->phrases,
    );
    return { map { $_ => 1 } @list };
}

# максимально допустимая длина фразы с учётом шаблонов title/body
sub max_phrase_length :CACHE {
    my ($self) = @_;
    my $max_phrase_length = 4096;

    my $title = $self->title;
    if($title =~ /\#/) {
        $title =~ s/\#[^\#]*\#//g;
        $max_phrase_length = min(33 - length($title), $max_phrase_length);
    }

    my $body = $self->body;
    if($body =~ /\#/) {
        $body =~ s/\#[^\#]*\#//g;
        $max_phrase_length = min(75 - length($title), $max_phrase_length);
    }

    return $max_phrase_length;
}

# список минус слов на баннер, уточняющих значения омонимов
sub homonymy_minuswordshash :CACHE {
    my ($self) = @_;

    my $result = {};
    my $homs = $self->proj->homonyms->words;
    my %nwh = %{$self->bannernormwordshash};

    for my $w (grep{$homs->{$_}} keys %nwh) {
        my %context = map{$_=>1} $self->proj->homonyms->get_context($w, keys %nwh);
        next if !%context;
        for my $c (keys %{$homs->{$w}}) {
            my $h = $context{$c} ? $self->proj->homonyms->minuswords : $homs;
            $result->{$_}++ for grep{!$self->bannerwordshash->{$_}} keys %{$h->{$w}{$c}};
        }
    }

    return $result;
}

#Слова, нетипичные для рекламы. Фильтруем текстом и фразами баннера.
sub banner_badwords_filter :CACHE {
    my ($self) = @_;
    my $badwds = $self->proj->phrase("видео своими +своими +руками !своими !руками описание руководство инструкция")->normwordshash;

    my $bnrh = $self->bannernormwordshash; #Список слов, который встречался в баннере
    my $curbadwds = { map {$_ => 1} grep { ! $bnrh->{$_} } keys %$badwds }; #Оставляем для фильтрации только те слова, которые не встречались в баннере

    return $curbadwds;
}


############################################################################
# / Специальные вызовы с внешними настройками поведения для баннеров
############################################################################


#Фильтр по несовместимым словам
#true если нет несовместимых слов
sub diffphrsfilter {
    my ($self, $ph) = @_;
    my $bwh = $self->textwordshash;
    my @w = grep {(!$bwh->{$_}) && $diffphrs->{$_}} $ph->snormwords;
    return 0 unless @w; #Если нет проверяемых слов - выходим
    for my $cw (@w){
        for my $ow (@{$diffphrs->{$cw}}){
            return $cw.':'.$ow if $bwh->{$ow};
        }
    }
    return 0;
}

sub modelphrases :CACHE {
    my ($self) = @_;
    my $proj = $self->proj;
    my @arr = ();
    for my $ph ($self->phrases){
        push(@arr, $ph) if $proj->models->is_model_type($ph);
    }
    return @arr;
}

sub modelphraseshash :CACHE {
    my ($self) = @_;
    return { map { $_->snorm_phr => 1 } $self->modelphrases };
}


# DEPRECATED, use arr version!
sub phrsfilterreason {
    my $self = shift;
    warn "Method `phrsfilterreason' is deprecated! Use filter_phrase_list instead";
    if($self->{cache_phrsfilterreason}){
        my $kk = $_[0]->snorm_phr;
        $self->{cache_phrsfilterreason_data}{$kk} //= [$self->phrsfilterreason_arr(@_)];
        return join('', map { '_'.$_ } @{ $self->{cache_phrsfilterreason_data}{$kk} });
    }
    my @reason = $self->phrsfilterreason_arr(@_);
    return join('', map { '_'.$_ } @reason);
}
# DEPRECATED, use filter_phrase_list
sub phrsfilterreason_arr {
    my $self = shift;
    my $ph = shift;
    my %par = @_;

    my $phl = $self->proj->phrase_list([$ph]);
    $self->filter_phrase_list($phl, %par);

    return @{$ph->{filterreason}};
}


my @phrase_filters = (
    ['is_good_phrase',
    sub {
        my ($self, $ph, %par) = @_;

        # плохая фраза
        my $badphrs = $ph->badphrsreason;
        $badphrs =~ s/_wide//g if $par{allow_wide};
        my @reason = ($badphrs =~ /_([^_]+)/g);
        return @reason;
    }],

    ['diffphrs',
    sub {
        my ($self, $ph, %par) = @_;

        #выкидываем фразы из несовместимых слов
        my $dfws = $self->diffphrsfilter($ph);
        if( $dfws ){
            return 'diffphrs:'.$dfws;
        }

        return ();
    }],

    ['badhomonym',
    sub {
        my ($self, $ph, %par) = @_;

        # снимаем омонимию
        my $hmwh = $self->homonymy_minuswordshash;
        if(grep{$hmwh->{$_}} $ph->snormwords) {
            return 'badhomonym';
        }

        return ();
    }],

    ['minusword',
    sub {
        my ($self, $ph, %par) = @_;

        #Учёт минусслов
        if($self->has_minuswords){
            my $mnh = $self->minuswordshash;
            if(grep { $mnh->{$_} } $ph->snormwords){
                return 'minusword';
            }
        }

        return ();
    }],

    ['badvendor',
    sub {
        my ($self, $ph, %par) = @_;

        #Если в тексте баннера явно указан вендор - не приписываем других вендоров
        if((grep{$self->proj->categs_tree->check_minicategs_flag($_, "vendor")} $self->get_minicategs) && $self->has_vendors) {
            my $bvh = $self->vendorshash;
            return 'badvendor' if grep {! $bvh->{$_} } $ph->vendors;
        }

        return () unless $self->lang eq 'tr';
        # Для турецких баннеров - дополнительная фильтрация фраз с чужими брендами
        my $brands_ph_phl = $ph->get_brands_snorm_phl;
        if ($brands_ph_phl->count) {
            #$self->proj->dd([ \%brands_own, $ph->text, [$ph->get_brands] ]);
            my %brands_own_snorm = map { $_->snorm_phr => 1 }  $self->brands_phl->phrases;
            my $brands_ph_phl = $ph->get_brands_snorm_phl;
            return 'badbrand_tr' if grep { not $brands_own_snorm{ $_->snorm_phr } }  $brands_ph_phl->phrases;
        }
        return ();
    }],

    ['widemdltype',
    sub {
        my ($self, $ph, %par) = @_;

        #Широкие типы товаров не могут предлагаться
        if( $self->proj->models->is_model_type($ph) ){
            return 'widemdltype' unless $self->modelphraseshash->{$ph->snorm_phr};
        }

        return ();
    }],

    ['categsminusword',
    sub {
        my ($self, $ph, %par) = @_;

        my %h = map{$_ => 1} $ph->snormwords;
        if(grep{$h{$_}} $self->categs_minuswords) {
            return 'categsminusword';
        }

        return ();
    }],

    ['badwordsfilter',
    sub {
        my ($self, $ph, %par) = @_;

        #Если в тексте баннера явно указан вендор - не приписываем других вендоров
        my $curbadwds = $self->banner_badwords_filter;
        return 'badwordsfilter' if grep { $curbadwds->{$_} } $ph->normwords;

        return ();
    }],

    ['geofilter',
    sub {
        my ($self, $ph, %par) = @_;

        if(!$par{no_geofilter}) {
            #Фильтрация по городам (geofilter)
            my $geo_reason = $self->geofilter($ph);
            return "geofilter:$geo_reason" if $geo_reason;
        }

        return ();
    }],

    ['automobile',
    sub {
        my ($self, $ph, %par) = @_;
        my $avto_reason = $self->filter_automobiles($ph);
        return $avto_reason if $avto_reason;
        return ();
    }],

    ['celebr',
    sub {
        my ($self, $ph, %par) = @_;

        #Отношения с селебретис
        if( (! $self->has_celebrities ) && $ph->has_celebrities && !$self->proj->categs_tree->check_minicategs_flag(join("/", $self->get_minicategs), "celebr") ){
            return 'celebr';
        }

        return ();
    }],

    ['nobmctg',
    sub {
        my ($self, $ph, %par) = @_;

        # Фраза/баннер с заведомо плохими категориями
        if(grep{$self->proj->categs_tree->check_minicategs_flag($_, "nobm")} ($self->get_minicategs, $ph->get_minicategs)) {
            return "nobmctg";
        }

        return ();
    }],

    ['antiword',
    sub {
        my ($self, $ph, %par) = @_;
        my @categs = $self->get_minicategs;

        for my $categ (@categs) {
            my $phl = $ph->get_categ_antiwords_phl($categ);
            return "antiword" if $phl && $phl->phrases;
        }

        return ();
    }],

    # если во фразе есть определенное слово, то оно должно быть в баннере (или его категориях)
    ['badword',
    sub {
        my ($self, $ph, %par) = @_;
        my $proj = $self->proj;

        my @ws = qw(игра рука фото видео);
        my %add = (
            'игра' => [ 'игровой' ],
        );
        my @reason;
        for my $w (@ws) {
            my $p0 = $self->proj->phrase($w);
            next if !$ph->snormwordshash->{$p0->snorm_phr};

            my @phrs = ($p0);
            push @phrs, $proj->phrase_list($add{$w})->phrases if $add{$w};

            next if grep { $self->bannersnormwordshash->{$_->snorm_phr} } @phrs;

            my %c = map { $_ => 1 } map { $_->search_category_by_snorm } @phrs;
            next if grep { $c{$_} } $self->get_minicategs;

            push @reason, "badword:$w";
        }

        return @reason;
    }],

    ['ctgfilter',
    sub {
        my ($self, $ph, %par) = @_;
        my $reason = "";
        my $ctree = $self->proj->categs_tree;
        my $allow_no_categs = $par{allow_no_categs};

        #Фильтрация по миникатегориям
        my %ctgfilter;  # информация о прохождении фильтра, для записи в matcher_inf.log
        if($self->get_minicategs){ #В тексте баннера есть упоминание слов одной из миникатегорий
            my @bnr_categs = $self->get_minicategs;

            if(@bnr_categs > 1 || !$self->proj->categs_tree->check_minicategs_flag($bnr_categs[0], "nofilter")) {
                my @phcats = grep {$_} $ph->get_minicategs;
                my $bnrgwf = $self->exminicategshash;

                my $bnrctgscount = keys %$bnrgwf;

                @phcats = $ph->get_uncertain_minicategs if !@phcats;

                $ctgfilter{bnr_categs} = \@bnr_categs;
                $ctgfilter{phr_categs} = \@phcats;

                if($bnrctgscount == 0){ #У баннера нет категорий
                    $reason = 'nobnrctgs' unless $allow_no_categs;
                } elsif(@phcats){
                    #Выкидываем, если есть несовпадающие категории
                    my @gctgs = (grep{  $bnrgwf->{$_}} @phcats);
                    my $gctgsh = { map {$_ => 1} @gctgs };
                    my @bctgs = (grep{! $bnrgwf->{$_}} @phcats);
                    if(@bctgs){ #Пробуем профильтровать плохие категории
                        @bctgs = grep {! $gctgsh->{ $ctree->cached_get_minicateg_parent($_) } } @bctgs; #Убираем из плохих те, у которых в родителях есть хорошие
                        my %gparent = map { $ctree->cached_get_minicateg_parent($_) => 1 } @gctgs;
                        @bctgs = grep { !$gparent{$_} } @bctgs;
                    }
                    my $bctgsh = { map {$_ => 1} @bctgs };

                    $reason = 'badminictg' if (grep{ $bctgsh->{$_}} @phcats); #Если есть дополнительные категории
                    $ctgfilter{type} = 'subset';
                    if ($self->{fltlog}) {
                        $ph->{fltlog} .= '[['.join( ',', map {"'$_'"} @phcats ).'], ['.join( ',', map {"'$_'"} @bnr_categs ).']]' if grep{!$$bnrgwf{$_}} @phcats;
                    }
                } elsif($ph->number_of_words < 2 && !$ph->look_like_a_model) {
                    # однословники без категорий выкидываем, не глядя в advq
                    $reason = 'nominictg' unless $allow_no_categs;
                } else {
                    my %tail_categs;
                    my $tot_count = $ph->get_search_count;
                    while (my ($tail, $count) = each %{$ph->tail2count}) {
                        my @ctg_ids = @{$ph->tail2categs->{$tail} // []};
                        $tail_categs{$_} += $count for grep { $_ } map { $ctree->get_minicateg_by_id($_) } @ctg_ids;
                    }

                    my $max_count = max(values %tail_categs) // 0;
                    @phcats = sort keys %tail_categs;
                    @phcats = grep { $tail_categs{$_} >= 0.01 * $tot_count } @phcats;
                    @phcats = grep { $tail_categs{$_} >= 0.1 * $max_count } @phcats;

                    #Выкидываем, если нет пересечения
                    if(@phcats || !$self->experiment->{dont_filter_noctg}) {

                        if( @phcats == 0 ){
                            $reason = 'nophrctgs' unless $allow_no_categs; #Нет категорий у фразы
                        }else{
                            my @gctgs = grep {$_ && $bnrgwf->{$_}} @phcats;
                            $reason = 'badminictg2' unless @gctgs; #Нет пересечения по категориям с баннером
                        }
                    }
                    if ($ph->{fltlog}) {
                        $ph->{fltlog} .= '[['.join( ',', map {"'$_'"} @phcats ).'], ['.join( ',', map {"'$_'"} @bnr_categs ).']]' if $reason =~ /badminictg2/;
                    }
                    # выкидываем, если слишком много advq-категорий
                    if(@phcats >= 6) {
                        $reason = 'advqctg6';
                    }
                    $ctgfilter{type} = 'advq_common';
                }

                # в exminicategs много категорий, оставляем только нужные нам
                $ctgfilter{exminicategs} = { map { $_ => $bnrgwf->{$_} } grep { defined $bnrgwf->{$_} } @phcats, @bnr_categs };
            } else {
                $ctgfilter{type} = 'no_filter';
            }
        } else {
            $ctgfilter{type} = 'no_minicategs' unless $allow_no_categs;
        }

        # проверяем в конце, чтобы сохранить ctgfilter
        my $good_pair = $par{allow_ctg_pair};
        if ($good_pair and $reason) {
            for my $cb ($self->get_minicategs) {
                for my $cp ($ph->get_minicategs) {
                    if ($good_pair->{$cb}{$cp}) {
                        $ctgfilter{type} = 'allow_ctg_pair';
                        $ctgfilter{allow_pair} = "$cb/$cp";
                        $reason = '';
                    }
                }
            }
        }

        $ph->{matcher_inf}{ctgfilter} = \%ctgfilter;

        return $reason if $reason;
        return ();
    }],
);

# на входе:
#   $phl  - PhraseList
# доп. параметры:
#   check_all =>  0/1  проверять всеми фильтрами (по умолчанию: 0)
#   forced => 0/1 не брать filterreason из кэша
#   no_geofilter => 0/1 отключить геофильтр (по умолчанию 0)
#   no_categs_cache => 0/1 отключить кэширование категорий в cdict (по умолчанию 0)
#   allow_no_categs => 0/1 разрешить некатегоризуемые фразы (по умолчанию 0)
#   allow_wide => 0/1 разрешить широкие фразы (по умолчанию 0)
#   allow_ctg_pair => $h - разрешаем заданные пары категорий { $ctg_b => { $ctg_p => 1 } }
#   timer => $timer_object
sub filter_phrase_list {
    my ($self, $phl, %par) = @_;

    my $proj = $self->proj;
    my $tm = $par{timer};

    $tm->time('filter_phrase_list::cache') if $tm;
    unless ($ENV{MR_BROADMATCH}) {
        $phl->cache_cdict_categs_atoms if !$par{no_categs_cache};
        $phl->cache_cdict_tail_categs;
        $phl->cache_is_good_phrase if !$par{no_categs_cache};  # TODO correct name for flag
        $phl->cache_cdict_regions_phrases;
    }

    for my $ph (@$phl) {
        my @reason;

        my $sn = $ph->snorm_phr;

        if(!$par{forced} && $self->{cache_phrsfilterreason} && $self->{cache_phrsfilterreason_arrs}{$sn}){
            $ph->{filterreason} = $self->{cache_phrsfilterreason_arrs}{$sn};
            next;
        }

        for my $h (@phrase_filters) {
            my ($name, $filter) = @$h;
            $tm->time("filter_phrase_list::$name") if $tm;
            my @curr_reason = $filter->($self, $ph, %par);
            if(@curr_reason) {
                push @reason, @curr_reason;
                last if !$par{check_all};
            }
        }

        if($self->{cache_phrsfilterreason}){
            $self->{cache_phrsfilterreason_arrs}{$sn} = \@reason;
        }

        $ph->{filterreason} = \@reason;
    }

    return $phl->lgrep(sub { ! @{$_->{filterreason}} });
}

sub _moderate_phrsfilterreason_goodtails :CACHE {
    my ($self) = @_;
    return $self->proj->phrase('купить заказать доставка недорого москва питер самара интернет магазин наличии стоимость отзывы оптовые опт характеристики качественный дорогой 1 2 3 4 5 6 7 8 9 0 11 12 13')->snormwordshash;
}

sub is_template :CACHE {
    my ($self) = @_;

    return ($self->text =~ /\#/ ? 1 : 0);
}

sub moderate_filter_phrase_list {
    my ($self, $sphl, %par) = @_;

    my $bnrtext = $self->title_phr." ".$self->body_phr;
    $bnrtext .= " $1$2" for $bnrtext =~ /(\S+)\-(\S+)/g; #Для слов через дефис добавляем слитное написание
    #$bnrtext =~ s/[-\+]/ /g;
    my $has_template = $bnrtext =~ /#.*#/;
    my $bnphr = $self->proj->phrase($bnrtext);

    my @new_phrases;
    for my $sph (@$sphl) {
        my $phtext = $sph->text;
        $phtext =~ s/ -.+/ /g;    #Удаляем минус-слова
        #$phtext =~ s/[-\+"!]/ /g; #Удаляем вспомагательный синтаксис
        $phtext =~ s/~0/ /g;      #Удаляем условие точного совпадения

        my $ph = $self->proj->phrase($phtext);
        $ph->{original_obj} = $sph;
        push @new_phrases, $ph;
    }
    my $phl = $self->proj->phrase_list({ phrases_list => \@new_phrases });

    $self->filter_phrase_list($phl, %par, no_geofilter => 1, template => $self->is_template);

    for my $ph (@$phl) {
        my %h = map{$_ => 1} @{$ph->{filterreason}};
        $h{wide_spam}++ if $ph->is_wide_spam_phrase;
        delete $h{$_} for grep{$h{$_}} qw{ nobnrctgs nophrctgs nobmctg badwords minusword doubleword };

        delete $h{nominictg} if $h{nominictg} && $ph->text =~ /^\d{7,}$/; #Разрешаем длинные последовательности цифр - часто бывают номерами деталей машин

        if($h{wide}) {
            my @words_without_bsstops = $ph->normwords_without_bsstops;
            $h{verybad}++ if @words_without_bsstops < 2;
        }

        if($h{badminictg2} || $h{badminictg}) {
            if($has_template || $ph->is_subphr( $bnphr, 'snorm' )){ #Фикс над багами категоризации
                delete $h{badminictg};
                delete $h{badminictg2};
            }else{
                my $spf = $self->_moderate_phrsfilterreason_goodtails;
                my @w = $ph->snormwords;
                my $bh = $bnphr->snormwordshash;
                my @gw = grep {$bh->{$_}} @w;  #Хорошие слова (пересекаются с баннером)
                my @bw = grep {!$bh->{$_}} @w; #Не из пересечения с баннером
                if(! ( grep {! $spf->{$_} } @bw )){ #Если нет слов не из разрешенного списка
                    delete $h{badminictg};
                    delete $h{badminictg2};
                }
            }
        }

        # размечаем флагами фразы у баннеров с шаблонами
        if($has_template) {
            my @flags = $self->proj->categs_tree->get_catalogia_flags($ph->get_minicategs);
            $h{$_}++ for @flags;
        }

        my $sph = $ph->{original_obj};
        $sph->{fltlog} = $ph->{fltlog};
        $sph->{moderate_filterreason} = [ sort keys %h ];
    }
}

#deprecated, use moderate_filter_phrase_list
sub moderate_phrsfilterreason {
    my ($self, $sph) = @_;

    my $bnrtext = $self->title_phr." ".$self->body_phr;
    $bnrtext .= " $1$2" for $bnrtext =~ /(\S+)\-(\S+)/g; #Для слов через дефис добавляем слитное написание
    $bnrtext =~ s/[-\+]/ /g;

    my $phtext = $sph->text;
    $phtext =~ s/ -.+/ /g;    #Удаляем минус-слова
    $phtext =~ s/[-\+"!]/ /g; #Удаляем вспомагательный синтаксис
    $phtext =~ s/~0/ /g;      #Удаляем условие точного совпадения

    my $ph = $self->proj->phrase($phtext);

    my $reason = $self->phrsfilterreason($ph);
    $reason =~ s/_geofilter:wrong_region:[^_]+//g; #Удаляем фильтрацию регионами, так как что-то неправильно подхватывается
    $reason =~ s/_geofilter:native_town//g; #Удаляем фильтрацию регионами, так как что-то неправильно подхватывается
    for my $br (qw{ _nobnrctgs _nophrctgs _nobmctg _badwords _minusword _doubleword }){
        $reason =~ s/$br//;
    }

    my $bnphr = $self->proj->phrase($bnrtext);

    $reason =~ s/_nominictg// if $phtext =~ /^\d{7,}$/; #Разрешаем длинные последовательности цифр - часто бывают номерами деталей машин

    $reason .= '_verybad' if $reason =~ /_wide/ && $ph->wordcount == 1;

    if( $ph->is_subphr( $bnphr, 'snorm' ) ){ #Фикс над багами категоризации
        $reason =~ s/_badminictg2?//;
    }else{
        my $spf = $self->_moderate_phrsfilterreason_goodtails;
        my @w = $ph->snormwords;
        my $bh = $bnphr->snormwordshash;
        my @gw = grep {$bh->{$_}} @w;  #Хорошие слова (пересекаются с баннером)
        my @bw = grep {!$bh->{$_}} @w; #Не из пересечения с баннером
        if(! ( grep {! $spf->{$_} } @bw )){ #Если нет слов не из разрешенного списка
            $reason =~ s/_badminictg2?//;
        }
    }

    $reason =~ s/_badminictg2?// if $bnrtext =~ /#.*#/; #Если баннер содержит шаблон, то считаем, что категоризация не влияет на качество

    $sph->{fltlog} = $ph->{fltlog};
    return $reason;
}

# информация для фильтра по маркам автомобилей
sub automobiles_inf :CACHE {
    my $self = shift;

    my @avto_ctg = ('Техническое обслуживание автомобилей', 'Запчасти для авто- и мототехники');
    my %avto_ctg = map { $_ => 1 } @avto_ctg;
    my $avto_dict = '.Автомобили';
    my $ctg_found = 0;

    for my $ctg ($self->get_minicategs) {
        $ctg =~ s/.* _ //;  # убираем виртуализацию
        my @path = $self->proj->categs_tree->_cat_path($ctg);
        push @path, $ctg;
        if (grep { $avto_ctg{$_} } @path) {
            $ctg_found = 1;
            last;
        }
    }
    return {} if !$ctg_found;

    # ищем марку авто в title+body
    my %avto_name;
    my @phrs = ($self->preprocess_title_body);
    for my $phr (@phrs) {
        my $atoms = $phr->search_atoms;
        while (my ($subphr, $dicts) = each %$atoms) {
            for my $dict (keys %$dicts) {
                $avto_name{$subphr} = 1 if $dict eq $avto_dict;
            }
        }
    }

    # случай native - все фразы содержат марку авто
    my $min_phrs = 5;
    my @own_phrs = $self->phl->phrases;
    if (@own_phrs >= $min_phrs) {
        my $avto_phl = $self->language->phrase_list([keys %avto_name])->snorm_pack_list;
        my %count;
        for my $phr (@own_phrs) {
            my $sub = $avto_phl->search_subphrases_in_phrase($phr, use_existing_phrases => 1);
            $count{$_->snorm_phr}++ for @$sub;
        }
        for my $phr (@$avto_phl) {
            $avto_name{$phr->text} = 'native' if ($count{$phr->snorm_phr} // 0) == @own_phrs;
        }
    }

    return \%avto_name;
}

sub filter_automobiles {
    my $self = shift;
    my $phr  = shift;

    my $avto_inf = $self->automobiles_inf;
    return () if !keys %$avto_inf;

    my $avto_dict = '.Автомобили';
    my $atoms = $phr->search_atoms;
    while (my ($subphr, $dicts) = each %$atoms) {
        for my $dict (keys %$dicts) {
            return 'automobiles' if $dict eq $avto_dict and !$avto_inf->{$subphr};
        }
    }
    return ();
}

# при определении региона учитываем иностранные города/регионы только для баннеров с категорией с флагом "geo"
sub geo_flag :CACHE {
    my $self = shift;
    my $flag = (grep { $self->proj->categs_tree->check_minicategs_flag($_, 'geo') } $self->get_minicategs) ? 1 : 0;
    return $flag;
}


sub region_inf :CACHE {
    my $self = shift;
    my $proj = $self->proj;

    my %inf;
    $inf{geo_flag} = $self->geo_flag;

    # пытаемся найти native_town
    my %cnt;
    my $totcnt = 0;
    my @checked_phrs = ($self->phrases, $self->preprocess_title_body);
    for my $phr (@checked_phrs) {
        my @phr_reg = $phr->get_regions;
        @phr_reg = grep { !$_->{is_world} } @phr_reg;
        my %seen_town = map { $_->{town} => 1 } @phr_reg;
        $cnt{$_}++ for keys %seen_town;
        $totcnt++;
    }
    my @native = grep {
        (@checked_phrs >= 10 and $cnt{$_} >= $totcnt - 1)
        or (@checked_phrs >= 6 and $cnt{$_} >= $totcnt)
    } keys %cnt;
    if (@native) {
        my $town = $native[0];
        $inf{names} = [ $town ];
        $inf{rule} = 'native_town';
        $inf{native_town} = $town;
        return \%inf;
    }

    # пытаемся определить регионы по title/body/phrases
    my @textreg;
    for my $phr (@checked_phrs) {
        my @phr_reg = $phr->get_regions;
        push @textreg, @phr_reg;
    }
    my %seen;
    @textreg = grep { !$seen{$_->{name}}++ } @textreg;
    @textreg = grep { !$_->{is_world} } @textreg if !$self->geo_flag;
    if (@textreg) {
        $inf{names} = [ map { $_->{name} } @textreg ];
        $inf{towns} = { map { $_->{name} => $_->{town} } @textreg };
        $inf{rule} = 'text';
        return \%inf;
    }

    # геотаргетинг
    return $self->geotargeting_region_inf;
}

sub geotargeting_region_inf :CACHE {
    my $self = shift;

    my %inf;

    $inf{geo_flag} = $self->geo_flag;
    $inf{rule} = 'geotargetting';
    $inf{names} = $self->get_targetting_regions;

    return \%inf;
}

my $cached_targetting2regions = {};

sub get_targetting_regions :CACHE {
    my ($self) = @_;

    my @geotarget;

    return [] unless $self->{targetting};

    if ( exists $cached_targetting2regions->{ $self->{targetting} } ) {
        @geotarget = @{ $cached_targetting2regions->{ $self->{targetting} } };
    }
    else {
        my @regid =  split /,/, $self->{targetting};
        @geotarget = $self->proj->geo->geoids2regions(\@regid);
        @{ $cached_targetting2regions->{ $self->{targetting} } } = @geotarget;
    }

    return \@geotarget;
}

# returns filter reason
# на входе:
#   $phr  =  фраза
# доп. параметры:
#   only_geotargeting => 0/1   при определении региона баннера смотреть только на геотаргетинг
sub geofilter {
    my $self = shift;
    my $phr = shift;
    my %par = @_;

    my $geo = $self->proj->geo;

    my $inf;
    if ($par{only_geotargeting}) {
        $inf = $self->geotargeting_region_inf;
    } else {
        $inf = $self->region_inf;
    }
    my $rule = $inf->{rule} || '';
    if ($rule eq 'native_town') {
        # во фразе должен содержаться native_town
        my $town = $inf->{native_town};
        if (!grep { $_->{town} eq $town } $phr->get_regions) {
            return "native_town:$town";
        }
    } elsif (($rule eq 'text' or $rule eq 'geotargetting') and @{$inf->{names}}) {
        # во фразе не должны содержаться города других регионов
        my @bnr_reg = @{$inf->{names}};
        my @phr_reg = map { $_->{name} } $phr->get_regions;
        # зарубежные регионы используем в фильтрации только для категорий с флагом "geo"
        if (!$inf->{geo_flag}) {
            @bnr_reg = grep { !$geo->is_world_region($_) } @bnr_reg;
            @phr_reg = grep { !$geo->is_world_region($_) } @phr_reg;
        }
        my %bnr_reg = map { $_ => 1 } @bnr_reg;

        my @wrong;
        # регион фразы плохой, если не является под-регионом баннера
        for my $reg (@phr_reg) {
            my $reginf = $geo->reg2reginf($reg);
            my @path = map { $geo->get_regtree->{$_}{name} } @{$reginf->{path}};
            push @wrong, $reg if !grep { $bnr_reg{$_} } @path, $reg;
        }
        if (@wrong) {
            return "wrong_region:".join(',', @wrong);
        }
    }

    return '';
}

sub is_media :CACHE {
    my ($self) = @_;
    my $markers = {map{$_=>1} qw(
        афиша киноафиша
        фильм кинофильм сериал сезон серия
        актер актриса
        театр спектакль
        роман биография
        песня
        книга
        смотреть)};

    for my $phrase ($self->phrases) {
        return 1 if grep{$markers->{$_}} $phrase->normwords;
    }

    return 0;
}

# фильтрация фраз по частоте с учетом геотаргетинга баннера
# параметры:
#   $phl
#   min_search_count => $cnt  -  порог на частоту (default: 1)
sub filter_by_search_count {
    my $self = shift;
    my $phl = shift;
    my %par = @_;
    my $min = $par{min_search_count} // 1;

    my $subtree = $self->geotargeting_subtree_hash;
    $phl->cache_regions_count;

    my @good;
    for my $phr (@$phl) {
        my $counts = $phr->get_regions_count;

        # идём по тому, кто меньше
        my $regions;
        my $walk_subtree;
        if (keys(%$subtree) < keys(%$counts)) {
            $regions = $subtree;
            $walk_subtree = 1;
        } else {
            $regions = $counts;
            $walk_subtree = 0;
        }
        my $sum = 0;
        while (my ($reg, $val) = each %$regions) {
            if ($walk_subtree) {
                $sum += ($counts->{$reg} // 0);
            } else {
                $sum += $val if $subtree->{$reg};  # $val = $counts->{$reg}
            }
            if ($sum >= $min) {
                # дальше можно не считать
                push @good, $phr;
                last;
            }
        }
    }
    return $self->proj->phrase_list(\@good);
}

sub get_related_minicategs_hash {
    my ($self) = @_;
    my $ctree = $self->proj->categs_tree;
    return $ctree->minicategs_to_related_minicategs_hash([$self->get_minicategs]);

    my @catnms = $self->get_minicategs;
    my %gvrt = map {$_ => 1} map { /(.*) _ /; $1 } grep { / _ / } @catnms;
    $self->proj->dd(\@catnms, \%gvrt);
    my %cth = ();
    $cth{$_}++ for @catnms; #Добавляем сами категории
    #Добавляем путь наверх
    $cth{$_}++ for map { $ctree->get_cat_path($_) } @catnms;
    #Добавляем подкатегории, сиблинги и невьюсы
    $cth{$_}++ for $ctree->get_related_minicategs( @catnms );
    #Фильтруем виртуалки
    %cth = map { $_ => 1 } grep { /(.*) _ / ? $gvrt{$1} : 1 } keys %cth;
    return \%cth;
}

sub get_related_minicategs {
    my ($self) = @_;
    return keys %{$self->get_related_minicategs_hash};
}

sub add_search_categs_minuswords_without_cache_strict {
    my ($self, $phl) = @_;
    my @ctgs = $self->get_related_minicategs;
    $_->{tail2categs} = undef for @$phl;
    return $phl->phtmap(sub { $_->add_search_categs_minuswords_strict(\@ctgs)->text });
}

sub add_search_categs_minuswords_strict {
    my ($self, $phl) = @_;
    my @ctgs = $self->get_related_minicategs;
    return $phl->cache_cdict_minicategs->cache_tail_categs->phtmap(sub { $_->add_search_categs_minuswords_strict(\@ctgs)->text });
}

# минус-слова

sub minuswords :CACHE {
    my ($self) = @_;
    return ( keys %{$self->minuswordshash} );
}

1;
