package BM::BannersMaker::BannerLandProject;
use strict;
use warnings;
use utf8;
use open ':utf8';

use base qw(BaseProject);

use Storable qw(dclone);
use Utils::Array qw{array_intersection};
use File::Basename qw(dirname);

use BM::BannersMaker::Product;
use BM::BannersMaker::ProductWear;
use BM::BannersMaker::ProductBooks;
use BM::BannersMaker::ProductFurniture;
use BM::BannersMaker::ProductHotels;
use BM::BannersMaker::ProductAviaTickets;
use BM::BannersMaker::ProductExternal;
use BM::BannersMaker::ProductDSE;
use BM::BannersMaker::ProductList;
use BM::BannersMaker::ProductSports;
use BM::BannersMaker::ProductBuildingMaterials;
use BM::BannersMaker::ProductRealty;
use BM::BannersMaker::ProductCars;
use BM::BannersMaker::ProductPromocodes;
use BM::BannersMaker::ProductChildrensGoods;
use BM::BannersMaker::ProductTiresDisks;
use BM::BannersMaker::ProductBeautyHealth;
use BM::BannersMaker::ProductAutoAccessories;
use BM::BannersMaker::ProductGames;
use BM::BannersMaker::ProductTravel;
use BM::BannersMaker::FeedDataSource;
use BM::BannersMaker::FeedDataSourceFactory;
use BM::BannersMaker::FeedDataSourceTravelBooking;

use BM::BannersMaker::FeedDataMapping;

use BM::BannersMaker::BadPhrases;
use BM::BannersMaker::DSETools;

use BM::BannersMaker::Tasks::DynTask;
use BM::BannersMaker::Tasks::DynGrpTask;
use BM::BannersMaker::Tasks::PerfTask;

use BM::BannersMaker::Feed;

use BM::BMClient::MarketSubphraser;

use Utils::UniDecode::UniDecode;


__PACKAGE__->mk_accessors(qw(
    temp_dir

    feed_data_source_factory
    feed_data_mapping fdm
    bad_dyn_phrases

    dse_tools

    market_subphraser
    direct_unidecode
));

sub init {
    my $self = shift;
    return if $self->get_calls_count_and_inc();

    $self->SUPER::init;

    my $options = $self->options;

    $self->feed_data_source_factory(BM::BannersMaker::FeedDataSourceFactory->new({
        proj => $self,
    }));
    $self->feed_data_mapping(BM::BannersMaker::FeedDataMapping->new({
        proj => $self,
    }));
    $self->fdm( $self->feed_data_mapping );  # alias

    $self->{bad_dyn_phrases} = BM::BannersMaker::BadPhrases->new({ proj => $self, dict_name => "bad_dyn_phrases"});

    $self->{dse_tools} = BM::BannersMaker::DSETools->new({ proj => $self, });

    $self->{market_subphraser} = BM::BMClient::MarketSubphraser->new({
        proj => $self,
        %{$options->{subphraser_params}},
        %{$options->{subphraser_market}},
    });
    (-d $_ || mkdir $_) for map { dirname($self->{market_subphraser}{$_}) } qw(comptrie_file staticmap_file);

    $self->{direct_unidecode} = Utils::UniDecode::UniDecode->new();

    # bannerland_dbh -> в IronProject
}


# передается ссылка на хеш параметров
# { url => , name => }
# { data => , feed_file_type => , csvmap => {...}, additional_data => {..} }
# feed_file_type поддерживает значения offers_tskv, yml, csv
# offers_tskv - товарные предложения в формате key=value\t...\n
# csv - поля, разделенные запятыми и взятые в кавычки, первая строка - заголовок с названиями полей
# если не указан feed_file_type, data может быть массивом хешей (готовый фид), или текстом, текст будет считаться текстом пейджа и feed_data_source автоматически попытается определить тип данных
# csvmap - ссылка на хеш с маппингом полей из csv в tskv
# additional_data - ссылка на хеш с именами доп.полей, которые нужно взять из фида, промапить и подцепить в Product
sub feed {
    my ( $self, $hpar ) = @_;
    if( ! ref($hpar) ){
        $hpar = { url => "$hpar" };
    }

    $hpar->{proj} = $self;
    return BM::BannersMaker::Feed->new( $hpar );
}


sub feed_data_source {
    my ( $self, $hpar ) = @_;
    return $self->feed_data_source_factory->new_fds($hpar);
}

# тестовый режим для тестирования классов динамических баннеров на каталогии
sub set_experimental_product_mode {
    my $self = shift;
    my $prev = $self->{experimental_product_mode};
    $self->{experimental_product_mode} = $_[0];
    return $prev;
}

sub product {
    my $self = shift;
    my $hpar = { proj => $self, data => [@_] };
    # дефолтный вариант - класс Product
    return BM::BannersMaker::Product->new( $hpar ) if ( ref($_[0]) ne 'HASH' || !$_[0]->{product_type} );

    # категоризованный offer
    my $offer_growscaled_minicategs = [ split( /\s*\/\s*/, $_[0]->{product_type} ) ];

    # категории по тематикам, по к-рым запускаются классы
    my $hbcats = $self->{options}{bannerland_categs};

    # экспериментальное тестирование тематик
    if ( $self->{experimental_product_mode} ){
        # сюда нужно помещать вызовы конструкторов классов экспериментальных тематик
    }

    # игры
    return BM::BannersMaker::ProductGames->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{games} )} );

    # авиабилеты
    return BM::BannersMaker::ProductAviaTickets->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{aviatickets} )} );

    # недвижимость
    return BM::BannersMaker::ProductRealty->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{realty} )} );

    # одежда
    return BM::BannersMaker::ProductWear->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{clothes} )} );

    # отели
    return BM::BannersMaker::ProductHotels->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{hotel} )} );

    # мебель
    return BM::BannersMaker::ProductFurniture->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{furniture} )} );

    # строительные материалы
    return BM::BannersMaker::ProductBuildingMaterials->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{building_materials} )} );

    # спортивные товары
    return BM::BannersMaker::ProductSports->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{sports} )} );

    # красота и здоровье
    return BM::BannersMaker::ProductBeautyHealth->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{beauty_health} )} );

    # детские товары
    return BM::BannersMaker::ProductChildrensGoods->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{childrens_goods} )} );

    # автомобили
    return BM::BannersMaker::ProductCars->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{cars} )} );

    # шины и диски
    return BM::BannersMaker::ProductTiresDisks->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{tires_disks} )} );

    # автотовары
    return BM::BannersMaker::ProductAutoAccessories->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{auto_accessories} )} );

    # купоны
    return BM::BannersMaker::ProductPromocodes->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{promocodes} )} );

    # фиктивный external-Product
    return BM::BannersMaker::ProductExternal->new($hpar) if @{array_intersection($offer_growscaled_minicategs, $hbcats->{external})};

    # фиктивный Product для DSE
    return BM::BannersMaker::ProductDSE->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{dse} )} );

    # Путешествия

    return BM::BannersMaker::ProductTravel->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{travel} )} );

    # книги
    return BM::BannersMaker::ProductBooks->new( $hpar ) if ( @{array_intersection( $offer_growscaled_minicategs, $hbcats->{books} )} );

    # дефолтный, если не найдено соответствий по категориям
    return BM::BannersMaker::Product->new( $hpar );
}

sub product_by_classname {
    my $self = shift;
    my $classname = shift;
    my $product_inf = shift;
    
    my $obj;
    my @args;
    my $need_pt_init = $product_inf->{need_pt_init};
    if ($need_pt_init) {
        @args = ({ proj => $self, data => [ $product_inf ] });
    } else {
        @args = ({ proj => $self, %$product_inf}, no_init => 1);
    }
    
    $obj = eval {$classname->new(@args)};
    if ($@) {
        $obj = eval {BM::BannersMaker::Product->new(@args)};
    }
    return $obj;
}

sub product_list {
    my $self = shift;
    return BM::BannersMaker::ProductList->new({ proj => $self, data => [@_] });
}

sub dyntask {
    my ($self, $taskinf) = @_;
    return BM::BannersMaker::Tasks::DynTask->new({ proj => $self, taskinf => dclone($taskinf), });
}

sub dyngrptask {
    my ($self, $data) = @_;
    $data->{proj} = $self;
    return BM::BannersMaker::Tasks::DynGrpTask->new($data);
}

sub perftask {
    my ($self, $taskinf, %params) = @_;
    $params{proj}    = $self;
    $params{taskinf} = $taskinf;
    return BM::BannersMaker::Tasks::PerfTask->new(\%params);
}

sub get_task_obj {
    my ($self, $task_type, $task_inf) = @_;
    if ($task_type eq 'perf') {
        return $self->perftask($task_inf);
    } else {
        return $self->dyntask($task_inf);
    }
}

#проверка текста на совместимость с движком( у них ucs-2)
#адаптация одноименной функции из pylib
sub is_bs_compatible {
    my $self = shift;
    my $text = shift // '';
    my $compat = 1;
    my $allowed_chars_hashref = $self->_bannerland_allowed_chars_hashref;
    if ( grep {!exists $allowed_chars_hashref->{$_}} split //, $text) {
        $compat = 0;
    }
    return $compat;
}

sub make_bs_compatible_or_empty {
    my $self = shift;
    my $text = shift // '';
    if ( !$self->is_bs_compatible($text) ) {
        $text = $self->direct_unidecode->Decode($text);
    }
    if ( !$self->is_bs_compatible($text) ) {
        $text = '';
    }
    return $text;
}

#рекурсивная версия, проверяет на совместимость с движком содержимое произвольной структуры (нужно для валидации структур, сериализуемых в json)
sub is_bs_compatible_recursive {
    my ($self, $r) = @_;
    my $valid = 1;
    if (ref($r) eq 'HASH') {
        for my $key (keys %$r) {
            #проверяем как ключ, так и значение. Ключ в хэше - всегда строка, риска повредить внутреннее состояние нет
            $valid = $self->is_bs_compatible( $key ) && $self->is_bs_compatible_recursive( $r->{$key} );
            last if !$valid;
        }
    }
    elsif (ref($r) eq 'ARRAY') {
        for my $elem (@$r) {
            $valid = $self->is_bs_compatible_recursive( $elem );
            last if !$valid;
        }
    }
    else {
        my $scalar = $r // '';
        # если это объект наподобие Phrase, то он интерполируется в строку. Если это скаляр, он тоже принудительно превращается в строку.
        # внутреннее состояние скаляра здесь меняется на стринг (да, даже если он не lvalue). Поэтому не вставляем сюда сам $r, чтобы не трогать скаляр (от этого зависит сериализация в json)
        $valid = $self->is_bs_compatible( ''.$scalar );
    }

    return $valid;
}

sub _bannerland_allowed_chars_hashref :CACHE {
    my ( $self ) = @_;
    my %result = map {$_ => 1} split //, $self->options->{bannerland_allowed_chars};
    return \%result;
}

sub body_builder: CACHE {
    my $self = shift;
    require BM::BannersMaker::BodyBuilder;
    return BM::BannersMaker::BodyBuilder->new({proj=>$self});
}

1;
