package BM::BannersMaker::DSETools;

use utf8;
use open ':utf8';

use Utils::Words;
use Utils::Urls qw(get_sec_level_domain);
use BM::PhraseParser;
use BM::BannersMaker::Product;
use HTML::Entities qw(decode_entities);

use POSIX;
use base qw(ObjLib::ProjPart);

my $PREPS_RE = $BM::PhraseParser::preps_re . '|из';

my $geofilter_mediagroups = [
    "ВИРТ _ ГОСТИНИЦЫ И КРАТКОСРОЧНАЯ АРЕНДА",
    "ПАССАЖИРСКИЕ ПЕРЕВОЗКИ",
    "ВИРТ _ АРЕНДА КОММЕРЧЕСКОЙ НЕДВИЖИМОСТИ",
    "УСЛУГИ ПО ТУРИЗМУ И ОТДЫХУ",
    "УСЛУГИ ТРАНСПОРТНЫЕ (РАЗНОЕ)",
    "ВИРТ _ БИЛЕТЫ НА ТРАНСПОРТ",
    "КРАТКОВРЕМЕННАЯ АРЕНДА ЖИЛОЙ НЕДВИЖИМОСТИ",
    "ВИРТ _ ЖИЛАЯ НЕДВИЖИМОСТЬ И ИПОТЕКА",
    "УСЛУГИ ПО АРЕНДЕ ЖИЛЫХ ПОМЕЩЕНИЙ",
    "УСЛУГИ ПО ОПЕРАЦИЯМ С ЗАГОРОДНОЙ НЕДВИЖИМОСТЬЮ",
    "УСЛУГИ ПО ОПЕРАЦИЯМ С НЕДВИЖИМОСТЬЮ ЗА РУБЕЖОМ",
    "УСЛУГИ ВОЗДУШНОГО ПАССАЖИРСКОГО ТРАНСПОРТА",
    "УСЛУГИ ПО ОПЕРАЦИЯМ С ЖИЛЫМИ ПОМЕЩЕНИЯМИ",
    "УСЛУГИ ЖЕЛЕЗНОДОРОЖНОГО ПАССАЖИРСКОГО ТРАНСПОРТА",
    "УСЛУГИ ПО ПРОВЕДЕНИЮ ДОСУГА",
    "ПРОКАТ АВТОМОБИЛЕЙ",
    "ВИРТ _ ТУРИЗМ",
    "УСЛУГИ ПО ОПЕРАЦИЯМ С НЕЖИЛЫМИ ПОМЕЩЕНИЯМИ",
    "ЛЕЧЕНИЕ И ОТДЫХ ЗА РУБЕЖОМ",
    "ГОСТИНИЦЫ"
];

my $geofilter_mediagroups_hash = { map {$_ => 1} @$geofilter_mediagroups };

my @titles_priority = qw(title kwyt_title landing_title links_title);

my %tld_variants_cache = ();

my $truncate_upper_bound = 500;

__PACKAGE__->mk_accessors(qw(
));


sub process_dse_base_row_phrases_step {
    my ($self, $params, $callback) = @_;
    my $proj = $self->proj;
    my $domain      = $params->{'domain'}      // '';
    my $url         = $params->{'url'}         // '';
    my $norm        = $params->{'norm'}        // '';

    my $bl_title = $params->{title};

    my $tld = (split /\./, $domain)[-1];
    unless ($tld_variants_cache{$tld}) {
        my @tld_variants = grep {$_} ($tld, $proj->phrase($tld)->translit_simple);
        push @tld_variants, grep {$_} map{$proj->phrase($_)->norm_phr} @tld_variants;
        push @tld_variants, map{"+$_"} @tld_variants;
        $tld_variants_cache{$tld} = \@tld_variants;
    }

    my $title_phr = $proj->phrase($bl_title =~ s/\.\.\.$//r);

    #проверка на регионы баннеров из определенных медиагрупп
    # выкидываем фразы, не соответствующие по гео
    my $need_region_check = 0;
    my %title_regions;
    my @mediagroups = $title_phr->get_mediagroups_for_categs([split /\//, $params->{categs}]);
    if (grep {exists $geofilter_mediagroups_hash->{$_}} @mediagroups) {
        %title_regions = map {$_->{name} => 1} $title_phr->get_regions_fast;
        $need_region_check = %title_regions;
    }

    my $norm_format = $proj->options->{DSE_Banners_params}{norm_format};

    my @title_words = $title_phr->normwords;
    my $title_wc = scalar @title_words;

    my $title_normwordshash = $title_phr->normwordshash;

    for my $norm_phr ( split_field($norm) ) {
        my $h = $self->decode_norm($norm_phr, $norm_format);

        $h->{'tlds'} = $tld_variants_cache{$tld};
        next unless $self->filter_norm($h);

        if ($need_region_check) {
            my %phrase_regions = map {$_->{name} => 1} $h->{norm_phr}->get_regions_fast;
            my ($query_region) = $proj->geo->geoids2regions([$h->{region}]);
            next if $query_region && exists $title_regions{$query_region} && !exists $phrase_regions{$query_region};
        }
        else {
            #выкидываем производные фразы с топонимами, если нет необходимости их добавлять
            next if $h->{orig} == 0;
        }

        my $rank = complex_rank($h);
        if ($rank > 0){

            my $phrase = $h->{norm_phr};
            my @intersection = grep {exists $title_normwordshash->{$_}} $phrase->normwords;
            my $ic = scalar @intersection;
            my $highlight = $title_wc ? $ic/$title_wc : 1;
            my $dup_rank = $rank * $highlight;

            $callback->( {
                domain => $domain,
                weak_cluster_key => $params->{weak_cluster_key},
                strong_cluster_key => $params->{strong_cluster_key},
                url => $url,
                phrase => $h->{'norm'},
                rank => $rank,
                dup_rank => $dup_rank,
            } );
        }
    }
}

sub process_dse_base_row_title_step {
    my ($self, $params, $callback) = @_;
    my $proj = $self->proj;
    my $domain      = $params->{'domain'}      // '';
    my $url         = $params->{'url'}         // '';
    my $norm        = $params->{'norm'}        // '';

    return unless filter_url($url);

    my $metalink = $proj->page(lc $url)->is_metalinks_url;
    $is_metalink = $metalink && $metalink !~ /^url_geo/ ? 1 : 0;
    return if $is_metalink;

    my $dyn_title_len = BM::BannersMaker::Product->title_len_from_type('dyn');
    my $perf_title_len = BM::BannersMaker::Product->title_len_from_type('perf');

    my $bl_title = $self->get_title($params, $dyn_title_len);
    return unless $bl_title;
    my $bl_title_perf = $self->get_title($params, $perf_title_len);
    return unless $bl_title_perf;
    my $bl_long_title = $self->get_title($params, 150);
    $bl_long_title = $bl_title_perf unless $bl_long_title;

    my $title_norm = $proj->phrase($bl_title =~ s/\.\.\.$//r)->norm_phr;

    my $strong_cluster_key = $title_norm;

    my $weak_cluster_key = $title_norm;
    $weak_cluster_key =~ s/\d+/DIGITS/g;

    $callback->( {
        title => $bl_title,
        title_perf => $bl_title_perf,
        long_title => $bl_long_title,
        url => $params->{'url'},
        norm => $params->{'norm'},
        domain => get_sec_level_domain($domain),
        strong_cluster_key => $strong_cluster_key,
        weak_cluster_key => $weak_cluster_key,
    } );
}

sub process_dse_base_row_categ_step {
    my ($self, $params, $callback) = @_;
    my $proj = $self->proj;
    my $domain           = $params->{'domain'}           // '';
    my $weak_cluster_key = $params->{'weak_cluster_key'} // '';
    my $title            = $params->{'title'}            // '';

    my $bnr = $proj->bf->lbanner({
        title => $title,
        title_extension => '',
        body => '',
        phrases => '',
        url => $domain,
    });

    my $categs = join(',',$bnr->get_minicategs);

    $callback->( {
        domain => $domain,
        title => $title,
        weak_cluster_key => $weak_cluster_key,
        categs => $categs,
        flags => '',
    } );

}

#Возвращаем лучший заголовок из списка. Причесывается регулярками, выбирается по принципу "самый близкий по размеру к тому, что нужно", обрезается, если нужно
sub best_title {
    my ($self, $titles, $hlim, $filter_func) = @_;
    my $llim = 12;
    my @lower_list;
    my @higher_list;
    my @trimmed_titles;
    # так как мы выбираем лучший заголовок, основываясь на размере, то задаем нижнюю и верхнюю длину фразы, после которых точно можно дальше не смотреть
    # подстраиваем их в процессе парсинга
    # это нужно на случай очень больших
    my $high_cut = 1000000000;
    my $low_cut = 0;
    for ( sort { length($a) <=> length($b) } @$titles) {
        my $title = $_;
        my $current_length = length($title);
        if ( ( $current_length <= $low_cut ) || ( $current_length >= $high_cut ) ) {
            next;
        }
        # транкейтим очень большие тексты сразу, чтобы регулярки дальше не зависали (маленькие тексты не изменятся)
        $title = $self->truncate_title($title, $truncate_upper_bound);
        #тяжелые фильтры выгоднее делать здесь, так как предыдущий фильтр очень строгий и сам почти все отбрасывает
        if (defined($filter_func) && ! $filter_func->($title)) {
            next;
        }
        $title = decode_entities($title) if $title;
        $title = $self->cut_title_meta($title);
        #убираем эскейпы из текста, например, \r\n
        $title =~ s/\\\w//g;
        $title =~ s/\\//g;
        #убираем пустые скобки
        $title =~ s/\(\s*\)//g;
        #убираем лишние пробелы
        $title =~ s/\s+/ /g;
        #убираем пробелы и знаки препинания в начале и конце строки
        $title = $self->remove_text_artifacts($title);
        push @trimmed_titles, $title;

        my $trimmed_length = length($title);
        # Если обрезанный заголовок укладывается в лимит, смотрим только то, что длиннее в необрезанном виде. Если не укладывается - смотрим только то, что короче
        # проверка на то, что текущая необрезанная длина больше минимума и меньше максимума, уже была
        if ( ( $trimmed_length < $hlim ) && ( $trimmed_length >= $llim ) ) {
            $low_cut = $current_length;
            next;
        }
        if ( ( $trimmed_length > $hlim ) ) {
            $high_cut = $current_length;
            next;
        }
        if ($trimmed_length == $hlim) {
            #дальше нет смысла искать, этот заголовок все равно выиграет
            last;
        }
    }
    @trimmed_titles = sort { length($a) <=> length($b) } @trimmed_titles;
    for my $title (@trimmed_titles) {
        next if length($title) < $llim;
        if ( length($title) <= $hlim ) {
            push @lower_list, $title;
        }
        else {
            push @higher_list, $title;
            last;#более длинные заголовки не нужны
        }
    }
    my $result = '';
    $result = pop(@lower_list) if !$result && scalar @lower_list;
    $result = shift(@higher_list) if !$result && scalar @higher_list;
    $result = $self->remove_unnecessary_words($result);
    $result = $self->truncate_title($result, $hlim);
    $result = $self->remove_unnecessary_words($result);
    #капитализация
    $result = ucfirst($self->proj->phrase($result)->casecorrection->text) if $result;
    return $result;
}

sub truncate_title {
    my ($self, $str, $limit, %par) = @_;
    return $self->_truncate_title_ignore_narrow($str, $limit) if $par{ignore_narrow};
    return $str if length($str) <= $limit;
    my $result = substr($str, 0, $limit + 1); # limit + 1, чтобы отличать предлоги

    my @tokens = $result =~ /(\s+|\w+|[^\s\w]+)/g; #либо только вайтспейс, либо только буквы, либо только пунктуация
    my $reslen = length($result);
    my $is_truncated_word = $tokens[-1] =~ /^\w/ ? 1 : 0;
    my $is_need_dots = 1;
    #раньше здесь были регулярки, которые просто отрезали вайтспейс с конца, но они по факту работали примерно за квадрат, поэтому теперь так
    while ( scalar(@tokens) && (($reslen > $limit-3 || $reslen > $limit && !$is_need_dots) ||
        ( ( $tokens[-1] =~ /^\w/ ) && ( stop4norm($tokens[-1]) || $tokens[-1] =~ /^($PREPS_RE)$/) )
    ) ) {
        #отрезаем последнее слово с конца строки вместе с пунктуацией и пробелами.
        #убираем непробелы с конца строки
        while ( scalar(@tokens) && ( $tokens[-1] =~ /^\S/ ) ) {
            $reslen -= length($tokens[-1]);
            $is_need_dots = ($tokens[-1] =~ /^($PREPS_RE)$/ && !$is_truncated_word) ? 0 : 1;
            $is_truncated_word = 0;
            pop @tokens;
        };
        while ( scalar(@tokens) && ( $tokens[-1] =~ /^\s/ ) ) {$reslen -= length($tokens[-1]); pop @tokens }; #убираем пробелы с конца строки
        while ( scalar(@tokens) && ( $tokens[-1] =~ /^\W/ ) ) {$reslen -= length($tokens[-1]); pop @tokens }; #убираем пунктуацию с конца строки
    }

    $result = join('', @tokens);
    if ($result) {
        $result .= '...' if ($is_need_dots);
    }
    return $result;
}

sub _truncate_title_ignore_narrow {
    my ($self, $str, $limit) = @_;
    my $narrow_symbols_max_count = $self->proj->options->{bannerland_narrow_symbols_max_count};
    return $str if $self->get_length($str, ignore_narrow => 1) <= $limit;
    my $result = substr($str, 0, $limit + $narrow_symbols_max_count + 1); # limit + 1, чтобы отличать предлоги

    my @tokens = $result =~ /(\s+|\w+|[^\s\w]+)/g; #либо только вайтспейс, либо только буквы, либо только пунктуация
    my $normal_symbols_cnt = $self->_get_normal_symbols_length($result);
    my $narrow_symbols_cnt = $self->_get_narrow_symbols_length($result);
    my $is_truncated_word = $tokens[-1] =~ /^\w/ ? 1 : 0;
    my $is_need_dots = 1;
    #раньше здесь были регулярки, которые просто отрезали вайтспейс с конца, но они по факту работали примерно за квадрат, поэтому теперь так
    while (scalar(@tokens) &&
        (
            (!$self->_is_title_params_valid($normal_symbols_cnt, $narrow_symbols_cnt + 3, $limit) ||
            !$self->_is_title_params_valid($normal_symbols_cnt, $narrow_symbols_cnt, $limit) && !$is_need_dots) ||
            (($tokens[-1] =~ /^\w/) && (stop4norm($tokens[-1]) || $tokens[-1] =~ /^($PREPS_RE)$/))
        )) {
        #отрезаем последнее слово с конца строки вместе с пунктуацией и пробелами.
        #убираем непробелы с конца строки
        while ( scalar(@tokens) && ( $tokens[-1] =~ /^\S/ ) ) {
            $normal_symbols_cnt -= $self->_get_normal_symbols_length($tokens[-1]);
            $narrow_symbols_cnt -= $self->_get_narrow_symbols_length($tokens[-1]);
            $is_need_dots = ($tokens[-1] =~ /^($PREPS_RE)$/ && !$is_truncated_word) ? 0 : 1;
            $is_truncated_word = 0;
            pop @tokens;
        };
        while ( scalar(@tokens) && ( $tokens[-1] =~ /^\s/ ) ) {
            $normal_symbols_cnt -= $self->_get_normal_symbols_length($tokens[-1]);
            $narrow_symbols_cnt -= $self->_get_narrow_symbols_length($tokens[-1]);
            pop @tokens
        }; #убираем пробелы с конца строки
        while ( scalar(@tokens) && ( $tokens[-1] =~ /^\W/ ) ) {
            $normal_symbols_cnt -= $self->_get_normal_symbols_length($tokens[-1]);
            $narrow_symbols_cnt -= $self->_get_narrow_symbols_length($tokens[-1]);
            pop @tokens
        }; #убираем пунктуацию с конца строки
    }

    $result = join('', @tokens);
    if ($result) {
        $result .= '...' if ($is_need_dots);
    }
    return $result;
}

sub _is_title_params_valid {
    my ($self, $normal_symbols_cnt, $narrow_symbols_cnt, $limit) = @_;
    my $narrow_symbols_max_count = $self->proj->options->{bannerland_narrow_symbols_max_count};
    return ($normal_symbols_cnt <= $limit && $narrow_symbols_cnt <= $narrow_symbols_max_count);
}

sub _get_normal_symbols_length {
    my $self = shift;
    my $text = shift;
    my $narrow_symbols = $self->proj->options->{bannerland_narrow_symbols};
    $text =~ s/[$narrow_symbols]//g;
    return length($text);
}

sub _get_narrow_symbols_length {
    my $self = shift;
    my $text = shift;
    my $narrow_symbols = $self->proj->options->{bannerland_narrow_symbols};
    $text =~ s/[^$narrow_symbols]//g;
    return length($text);
}

# доп. параметры:
#   ignore_narrow =>    не учитывать узкие символы в длине
sub get_length {
    my $self = shift;
    my $text = shift;
    my %par  = @_;
    my $src_text_length = length($text);
    return $src_text_length if !$par{ignore_narrow};
    my $narrow_symbols_max_count = $self->proj->options->{bannerland_narrow_symbols_max_count};
    my $narrow_symbols_len = $self->_get_narrow_symbols_length($text);
    $narrow_symbols_len = $narrow_symbols_max_count if $narrow_symbols_len > $narrow_symbols_max_count;
    return $src_text_length - $narrow_symbols_len;
}

sub get_title {
    my ($self, $params, $hlim) = @_;
    my $proj = $self->proj;
    my $bl_title = '';
    my $first = 1;
    for my $title_field(@titles_priority) {
        if ($first) {
            $first = 0;
            $bl_title = $self->best_title([split_field($params->{$title_field})], $hlim, sub {my $t = shift; return $self->filter_primary_title($t); } ) unless $bl_title;
        }
        else {
            $bl_title = $self->best_title([split_field($params->{$title_field})], $hlim, sub {my $t = shift; return $self->filter_additional_title($t); }) unless $bl_title;
        }
    }
    return $bl_title;
}

#в полях есть заэскейпленная табуляция, нужно по ней побить
sub split_field {
    my $text = shift;
    return () unless $text;
    return split /\t|\\t/, $text;
}

sub decode_norm {
    my ($self, $str, $format) = @_;

    my @key = @$format;
    my @value = split /:/, $str;

    my %result;
    @result{@key} = @value;

    my $norm = $result{norm};
    $norm =~ s/\s+~0$//;
    $result{norm_phr} = $self->proj->normed_phrase($norm);

    return \%result;
}

#filter one-words, all-digits and phrases with hits > 20000
#return true if filter is passed
sub filter_norm {
    my ($self, $h) = @_;
    my $proj = $self->proj;
    my $phr = $h->{norm_phr};
    my $norm = $phr->text;
    my @words = split /\s+/, $norm;
    my $wordcount = scalar( @words );
    my $wc_minlength = {
        1 => 5,
        2 => 6,
        3 => 9,
    };
    my $minlength;
    if ( $wordcount <= 3 ) {
        $minlength = $wc_minlength->{$wordcount};
    }

    my $is_domain_phrase = 0;
    if ( $wordcount == 2 ) {
        foreach my $tld ( @{$h->{'tlds'}} ) {
            if ( grep {$_ eq $tld} @words ) {
                $is_domain_phrase = 1;
                last;
            }
        }
    }

    my $result = length($norm) <= 2                              ||
                 $norm !~ /[^\s\d]/                              ||
                 $is_domain_phrase                               ||
                 $phr->is_porno_phrase                           ||
                 $norm =~ /^\d+ скидка$/                         ||
                 $norm =~ /^\d{2} искать$/                       ||
                 $norm =~ /знакомств|знакомиться/                ||
                 ( $wordcount == 1 && !$phr->look_like_a_model ) ||
                 ( $minlength && length($norm) < $minlength )
                 ? 0 : 1;
    return $result;
}

sub filter_url {
    my $url = shift;
    return lc($url) !~ /(article|\.pdf|\/about|\/blog|\/contact|\/help|forum|\/comments|\/review|\/otziv|\/otzyv|\/testimonials|\/opinion|\/responce|\/star-review|\/reviewandawards|\/showuserreviews|\/shops|\/news|\/digest[^a-zA-Z])/;
}

sub simple_rank {
    my $h = shift;
    return $h->{'clicks'};
}

sub complex_rank {
    my $h = shift;
    #return $h->{'hits'} * ( ( ( $h->{'clicks'} + 1 ) ** 1 ) / ( $h->{'position'} + 1 ) )
    return $h->{'hits'} * ( ( ( $h->{'clicks'} + 1 ) ** 2 ) / ( $h->{'position'} + 1 ) )
}

sub all_adjectives {
    my $self = shift;
    my $lc_title = shift;
    my $proj = $self->proj;

    for my $word (split /\s+/, $proj->phrase($lc_title)) {
        my $gnc = $proj->phrase($word)->get_gender_number_case(add_posp => 1);
        return 0 if (
            ($gnc->{posp} ne 'A') ||
            ( $word !~ /^[а-яё][-а-яё]+$/) ||
            (! $proj->phrase($word)->norm_phr )
        );
    }
    return 1;
}

sub filter_additional_title {
    my $self = shift;
    my $title = shift;
    my $lc_title = lc($title);

    return 0 if ($lc_title =~ /(товар не найден|страница не найдена)/);
    if ( $lc_title =~ /404|502/ && $lc_title =~ /(ошибка|error|bad gateway)/ ) {
        return 0;
    }
    return 0 if ($lc_title =~ /^каталог\s+/);
    return 0 if ($lc_title =~ /^(студенческая жизнь|результаты поиска|поиск по сайту|ошибка просмотра товара|совместные покупки: каталог|общий каталог|все предложения|заблокировано)$/);
    return 0 if ($lc_title =~ /(^|\s)технические работы(\s|$)/);
    return 0 if ($lc_title =~ /домен/ and $lc_title =~ /продается/);
    return 0 if ($lc_title =~ /временно не работает/);
    return 0 if ($lc_title =~ /оформление заказа/);
    return 0 if ($lc_title =~ /^[cс]мотреть\s+вс[её]/);
    return 0 if ($lc_title =~ /^показать\s+вс[её]/);
    return 0 if ($lc_title =~ /^(с|на|по|в|без|у|над)(\s+|$)/);
    return 0 if ($lc_title =~ /^(шириной|длиной|высотой|диаметром|площадью)(\s+|$)/);
    return 0 if ($lc_title =~ /^рекомендации\s+специалистов$/);
    return 0 if ($lc_title =~ /(сайт закрыт(\s|$)|welcome to nginx|test page for the nginx http server|доставка|о нас|о компании|контакты|отзыв|отзывы|рецензии|комментарии|форум|просмотреть|посмотреть|ОК|далее|дальше|обратно|вернуться|отмена|назад|вперед|влево|вправо|вниз|наверх|регистрация|пароль|прочие|прочее|фотогалерея|галерея|в корзину|обзор)/) ;

    return 0 if ($self->all_adjectives($lc_title));
    return 0 if ($self->is_title_has_discount_or_price($lc_title));

    return 1;
}

sub filter_primary_title {
    my $self = shift;
    my $title = shift;
    my $lc_title = lc($title);

    return 0 if ($lc_title =~ /(товар не найден|страница не найдена)/);
    if ( $lc_title =~ /404|502/ && $lc_title =~ /(ошибка|error|bad gateway)/ ) {
        return 0;
    }
    return 0 if ($lc_title =~ /^каталог\s+/);
    return 0 if ($lc_title =~ /^(студенческая жизнь|результаты поиска|поиск по сайту|ошибка просмотра товара|совместные покупки: каталог|общий каталог|все предложения|заблокировано)$/);
    return 0 if ($lc_title =~ /(^|\s)технические работы(\s|$)/);
    return 0 if ($lc_title =~ /домен/ and $lc_title =~ /продается/);
    return 0 if ($lc_title =~ /временно не работает/);
    return 0 if ($lc_title =~ /оформление заказа/);
    return 0 if ($lc_title =~ /^[cс]мотреть\s+вс[её]/);
    return 0 if ($lc_title =~ /^показать\s+вс[её]/);
    return 0 if ($lc_title =~ /^(с|на|по|в|без|у|над)(\s+|$)/);
    return 0 if ($lc_title =~ /^(шириной|длиной|высотой|диаметром|площадью)(\s+|$)/);
    return 0 if ($lc_title =~ /^рекомендации\s+специалистов$/);
    return 0 if ($lc_title =~ /(^отзыв|сайт закрыт(\s|$)|welcome to nginx|test page for the nginx http server|о нас|о компании|форум|отмена|регистрация|пароль|фотогалерея|галерея)/);

    return 0 if ($self->all_adjectives($lc_title));
    return 0 if ($self->is_title_has_discount_or_price($lc_title));

    return 1;
}

#убираем из заголовка служебную информацию (например "страница 2 из 4")
sub cut_title_meta {
    my ($self,$str) = @_;
    #такие регулярки тяжелые, для них нужно проверять целесообразность. Иначе работает очень долго
    if ( $str =~ /[сС](траниц|ТРАНИЦ)/ ) { #СИЛЬНО быстрее, чем с ключом i
        $str =~ s/(^|\W+)страница\s+\d+(\W+из\W+\d+)?(\W+|$)/ /i;
    }
    return $str;
}

# скидки и цены быстро становятся не актуальными, поэтому убираем заголовки с ними из dse
# https://st.yandex-team.ru/SUPBL-858, https://st.yandex-team.ru/SUPBL-825
sub is_title_has_discount_or_price {
    my $self = shift;
    my $lc_title = shift;
    my (undef, %metrics) = $self->proj->phrase($lc_title)->cut_all_metrical;
    return 1 if ($metrics{money} || $metrics{vol} =~ m/%|процент/ && $lc_title =~ m/скидк/);
    return 0;
}

sub remove_unnecessary_words {
    my $self = shift;
    my $text = shift;

    my $unnecessary_words_re = 'купить';
    $text =~ s/[\s,.;:-]*$unnecessary_words_re\W*$//i;
    my $orig_text = $text;

    if ($text !~ /^\s*$unnecessary_words_re\s+/i) {
        return $orig_text;
    }
    $text =~ s/^\s*$unnecessary_words_re\s+//i;
    $text =~ s/^[\s,.;:-]+//i; # удаляем знаки

    # Если предлог идет первым, лучше уже не сделать "Купить к нарезному оружию в Нижневартовске"
    if ($text =~ /^\s*($PREPS_RE)\s+/i) {
        return $orig_text;
    }
    my $main_parts = _get_text_main_parts($text);
    my $is_can_remove = 1;
    for my $txt_part (@$main_parts) {
        if (!$self->_is_possible_to_remove_unnecessary_words($txt_part)) {
            $is_can_remove = 0;
            last;
        }
    }
    if (!$is_can_remove) {
        $text = $orig_text;
    }
    return $text;
}

sub remove_text_artifacts {
    my $self = shift;
    my $text = shift;

    # удаляем знаки из начала строки
    $text =~ s/^[\s,.;:-]+//i;

    # удаляем знаки из конца строки
    $reversed_text = reverse $text;
    $reversed_text =~ s/^[\s,.;:-]+//i;
    $text = reverse $reversed_text;

    return $text;
}

sub _get_text_main_parts {
    my $text = shift;
    my $first_sentence = _get_first_sentence($text);
    my @main_parts = split(/\sи\s|,\s/, $first_sentence);
    @main_parts = map { _get_text_first_part($_) } @main_parts;
    return \@main_parts;
}

sub _get_first_sentence {
    my $text = shift;
    my ($first_sentence, undef) = split(/[\.\!?](?:\s|$)/, $text);
    return $first_sentence;
}

sub _get_text_first_part {
    my $text = shift;
    my ($first_part, undef) = split(/(\s($PREPS_RE))(\s|$)/i, $text, 2);
    return $first_part;
}

sub _is_possible_to_remove_unnecessary_words {
    my $self = shift;
    my $text = shift;
    my $context_gnc = $self->proj->phrase($text)->get_gender_number_case();
    if ($context_gnc->{'case'} eq 'nom') {
        return 1;
    }
    return 0;
}

1;
