package BM::BannersMaker::Feed;

use utf8;
use open ':utf8';

use std;
use base qw(ObjLib::ProjPart);

use Data::Dumper;
use JSON qw(to_json from_json);

use Encode;
use Encode qw{ _utf8_on _utf8_off };

use URI::Escape;
use LWP::UserAgent;
use IPC::Open2;
use Utils::Urls;
use Utils::Sys qw{unzipdata do_safely};
use Utils::Array qw{uniq_array uniq_array_ref in_array array_intersection};
use ObjLib::FileIter;
use Time::HiRes qw/gettimeofday tv_interval/;

use Scalar::Util qw(weaken);
use List::Util qw(min sum);

use Digest::MD5 qw(md5_hex);
use XML::Parser;

use XMLParser;

use BM::BannersMaker::Product;
use BM::BannersMaker::Tasks::PerfMergeFeed;
use BM::BannersMaker::Tasks::BasePerfTask;

use Utils::Hosts qw( get_curr_host );

########################################################
#Доступ к полям
########################################################

__PACKAGE__->mk_accessors(qw(
    name
    url
    fds
    source_letter
    offer_source
    mapping_done
    __next_data_pack_ith
));

no warnings 'utf8'; # из-за варнингов ловим segfault в bmapi, глушим варнинги до решения DYNSMART-863

########################################################
# Интерфейс
########################################################

#   data                  Указатель на массив товаров

########################################################
# Инициализация
########################################################

our @filters_log_info_fields = qw/image_url_validation_rejected_offers_cnt
                                  image_url_validation_rejected_images_cnt/;

my @json_fields = qw(additional_data images);

sub init {
    my ($self) = @_;

    $self->{$_} = 0 for @filters_log_info_fields;
    $self->{required_fields_filter_log} = ();

    # Создание фида по готовым данным (по содержимому)
    # text передаем через extfile
    if ( defined($self->{data}) && !ref($self->{data}) && !defined($self->{extfile}) ) {
        my $temp_extfile = $self->get_tempfile('feed_extfile.tmp', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
        open F, '>', $temp_extfile;
        # binmode нужен, поскольку двоичные файлы передаются, как текст, это используется в тестовом интерфейсе
        # sergio: комментирую этот вызов и ставлю флаг utf8, так как бились турецкие символы
#        binmode(F);
        binmode(F, ":utf8");
        _utf8_off($temp_extfile);
        print F $self->{data};
        close F;
        $self->{extfile} = $temp_extfile;
    }

    if ($self->{extfile}) {
        # Создание фида из файла
        $self->fds( $self->proj->feed_data_source( { map { $_ => $self->{$_} } qw{ url name login pass offerfilters download_timeout max_file_size max_file_size_type offer_tag tskvmap additional_data required_fields extfile business_type } } ) );
    } else {
        # Создание фида по урлу/бизнесшопу
        $self->fds( $self->proj->feed_data_source( { map { $_ => $self->{$_} } qw{ url name login pass offerfilters download_timeout max_file_size max_file_size_type offer_tag tskvmap additional_data required_fields business_type business_id shop_id datacamp_orig_feedurl last_valid_feed_type offer_source is_new_task force_dc is_new_feed} } ) );
    }
    $self->fds->{feed_file_type} = $self->{datatype} if $self->{datatype};
}

sub open_log_search_filtered {
    my ($self, $log_file) = @_;

    if (-f $log_file) {
        $self->proj->do_sys_cmd("mv $log_file $log_file\_tmp", no_die => 1);
    }

    open (my $fh, '>>', $log_file."_tmp");
    $self->{log_search_filtered_file} = $log_file;
    $self->{log_search_filtered_fh} = $fh;

    return 1;
}

sub close_log_search_filtered {
    my ($self) = @_;
    if ($self->{log_search_filtered_fh} && -f $self->{log_search_filtered_file}.'_tmp') {
        my $fh = $self->{log_search_filtered_fh};
        close $fh;
        $self->proj->do_sys_cmd('mv '.$self->{log_search_filtered_file}.'_tmp '.$self->{log_search_filtered_file});
        delete $self->{log_search_filtered_fh};
    }
}

sub DESTROY {
    my ($self) = @_;
    $self->close_log_search_filtered;
}

########################################################
# Методы
########################################################

sub get_offer_source {
    my $self = shift;
    return $self->offer_source // 'feed';
}

sub get_filters_full_log {
    my ($self) = @_;
    my %res = ();

    for my $field(keys (%{$self->{required_fields_filter_log} || {}})) {
        $res{"required_field: $field"} += $self->{required_fields_filter_log}{$field};
    }
    for my $fds_field( grep {$self->{$_} } @filters_log_info_fields) {
        $res{$fds_field} = $self->{$fds_field};
    }

    return \%res;
}

sub is_empty {
    my $self = shift;
    return !$self->fds->srcfilesize;
}

#####
# Получение содержимого фида после маппинга
#
sub tskv_view_debug {
    my ( $self ) = @_;
    my $filename = $self->offers_tskv_mpd_file->{filename};
    return $self->proj->file( $filename )->text;
}

sub _process_images {
    my ($self, $h) = @_;

    # гарантируем что в images первая картинка главная и нет дубликатов
    my $images = [];
    push @$images, $h->{picture} if $h->{picture};
    push @$images, @{$h->{images}} if $h->{images} && @{$h->{images}};
    $images = uniq_array_ref($images);

    my @good_images = grep { $self->proj->validate_url($_) } @$images;
    if (!@good_images or $good_images[0] ne $h->{picture}) {
        # Либо не провалидировался ни один урл, либо хотя бы главная картинка. Проваливаем валидацию.
        $h->{bad_flags} = $h->{bad_flags} ? $h->{bad_flags} . ",no_images" : "no_images";
        $self->{image_url_validation_rejected_offers_cnt}++;
    }
    # логируем разницу длинн массивов — столько картинок не провалидировалось.
    $self->{image_url_validation_rejected_images_cnt} += $#$images - $#good_images;

    # ограничиваем длинну images сверху
    my $max_feed_image_idx = $Utils::Common::options->{max_feed_images} - 1;
    @good_images = @good_images[0..min($max_feed_image_idx, $#good_images)];

    # сохраняем картинки, даже если там пустой массив
    $h->{images} = \@good_images;
}

#####
# Маппинг офферов и вывод в формате tskv во временный файл
#
# доп. параметры:
#   filename    =>  subj (если не задано, создаём временный файл)
#   pack_size   =>  размер пачки для next_data_pack (default: 1000)
sub offers_tskv_mpd_file {
    my $self = shift;
    my %par = (
        pack_size => 1000,
        @_,
    );
    my $proj = $self->proj;

    my $filename = $par{filename};
    unless ($filename) {
        $filename = $self->get_tempfile('offers_tskv_mpd', UNLINK => 1,
            DIR => $Utils::Common::options->{dirs}{banners_generation_files});
    }

    my $pack_size = $par{pack_size};
    my $total_offers = $self->get_total_offers_count;
    my $total_packs = int($total_offers / $pack_size) + (($total_offers % $pack_size == 0) ? 0 : 1);
    my $pack_number = 0;

    $proj->log("offers_tskv_mpd_file: create `$filename' with params:", \%par);

    my $timeout = 3600 * 20; #Ограничиваем время обработки, так как оно может быть большим, в т.ч. из-за прокачки урлов в больших фидах (фильтры по контенту или фоллбек категоризации)
    my $begtime = time;
    open(TEMP_F, "> $filename") or die("Can't open file $filename: $!");
    $self->iter_init; #Сбрасываем цикл обхода, если фид уже обходился
    while(defined( my $data = $self->next_data_pack($pack_size) )){
        $pack_number++;
        for my $h( @$data ) {
            # json поля сохраняем в _json и приводим к json
            for my $field (@json_fields) {
                if ($h->{$field}) {
                    my $data = delete $h->{$field};
                    $h->{$field . '_json'} = to_json($data);
                }
            }
            my $offer_line = join("\t", map { ($_//'')."=".($h->{$_}//'') } keys %$h);
            print TEMP_F $offer_line,"\n";
        }

        my $ocnt = $self->get_iter_offers_count;
        my $read_count = sum(0, map { $_->{read} // 0 } values %$ocnt);
        my $output_count = sum(0, map { $_->{output} // 0 } values %$ocnt);
        $proj->log("offers_tskv_mpd_file: pack $pack_number/$total_packs; offers (total/read/output): $total_offers => $read_count => $output_count");

        my $curtime = time;
        if($curtime - $begtime > $timeout){ #Прерываемся при превышении таймаута
            $proj->log("offers_tskv_mpd_file finished due to timeout");
            last;
        }
    }
    $self->iter_finalize;

    my $err = $self->get_filters_full_log();
    $proj->log("errors appeared while mapping to tskv:".Dumper($err)) if (%$err);
    close(TEMP_F);

    $proj->log("offers_tskv_mpd_file: done!");

    return {filename => $filename, offers_count => $self->get_iter_offers_count};
}

#####
# Получение хэша с информацией о категориях (в т.ч. path)
# Возвразает хэш (categ_id => hash_categ)
#
sub _get_categs_hash {
    my ($self, $categs_tskv) = @_;

    my $h_categs_id2h = {};
    if ( $categs_tskv ){
        my @arr_categ_hashes = ();
        for my $categ_h( @{$self->fds->tskv2arr( $categs_tskv )} ) {
            my $h_res = { map { my ($k, $v) = ($_, $categ_h->{$_}); $k =~ s/.*\://i; $k => $v } keys %$categ_h };
            push @arr_categ_hashes, $h_res;
        }

        $h_categs_id2h = { map { $_->{id} => $_ } grep { $_->{id} } @arr_categ_hashes };
        $_->{path} = '' for values %$h_categs_id2h;
        $_->{path} = $_->{category} for grep {! $_->{parentId}} values %$h_categs_id2h;
        my $have_progress = 1;
        while ( $have_progress ) {
            $have_progress = 0;
            for my $category_no_path (grep { ! $_->{path} } values %$h_categs_id2h){
                my $h_parent_categ = $h_categs_id2h->{$category_no_path->{parentId}};
                if ( $h_parent_categ->{path} ){
                    $category_no_path->{path} = $h_parent_categ->{path}.' / '.$category_no_path->{category};
                    $have_progress = 1;
                }
            }
        }
    }
    return $h_categs_id2h;
}

# промапить пачку tskv-строк с офферами
# работаем с пачками, т.к.
# 1) исторически сложилось -- есть группировочные мапы по таким пачкам
# 2) возможно, так будет лучше по производительности
# 3) можно при таймауте остановиться на середине
# параметры:
#   $offers_tskv --  список строк tskv-файла
#   $categs_inf -- хэш с информацией о категориях
# возвращает ссылку на список офферов (хэшей)
sub _tskv2feed {
    my $self = shift;
    my $offers_tskv = shift;
    my $categs_inf = shift;
    my %par = @_;

    my $offers = $self->fds->tskv2arr( $offers_tskv );
    my @res = ();
    for my $offer( @$offers ){
        my %h = %$offer;
        if (keys %$categs_inf and $offer->{categoryId} and $categs_inf->{$offer->{categoryId}}) {
            $h{categpath} = $categs_inf->{$offer->{categoryId}}{path};
        }
        # маппинг полей source_to_canonical
        if ( $offer->{tskvmap} ){
            my %hres_tskv_map = $self->do_smartmap( $offer->{tskvmap}, \%h );
            $h{$_} = $hres_tskv_map{$_} for ( keys %hres_tskv_map );
            delete $h{tskvmap};
        }
        $self->_append_feed_lang( \%h ) unless $h{feed_lang};
        # в фид дописываются поля minicategs и product_type
        # после добавления языка, так как от языка может зависеть product_type
        $self->_append_product_type( \%h ) unless $h{product_type};
        $self->_append_main_mirror(\%h) unless $par{no_mirrors};
        # (apovetkin) DYNSMART-1385: Хотфикс против sumochka.com
        # TODO заменить на нормальный механизм
        if ((defined $h{main_mirror} && $self->proj->options->{banned_domains}->{get_sec_level_domain($h{main_mirror})}) ||
            (defined $h{orig_domain} && $self->proj->options->{banned_domains}->{get_sec_level_domain($h{orig_domain})}) ||
            (defined $h{main_mirror} && $self->proj->options->{banned_domains}->{$h{main_mirror}}) ||
            (defined $h{orig_domain} && $self->proj->options->{banned_domains}->{$h{orig_domain}})
        ) {
            $self->proj->log("$h{orig_domain} is banned, skip offer");
            next;
        }
        $self->_append_source_inf(\%h);
        $self->_process_images(\%h);

        $h{available} ||= 'true' if $self->{force_dc};

        push @res, \%h;
    }

    # mapping to YABS section
    for my $offer(@res) {
        # маппинг полей source_to_yabs в отдельный хеш
        if ( $offer->{additional_data} ) {
            my %hres_ad_map = $self->do_smartmap( $offer->{additional_data}, $offer );
            $offer->{additional_data} = { %hres_ad_map };
        }
    }

    return \@res;
}

#####
# Проверка на наличие необходимых полей
#
sub check_required_fields {
    my ($self, $hfields, %par ) = @_;

    my @offer_fields = grep { $hfields->{$_} } keys %$hfields;
    my @rf_rules = grep { /\S/ } split /\s*\n\s*/, $self->fds->required_fields;
    my $res = 1;
    for my $rule ( @rf_rules ){
        my @rule_fields = split/\s+_OR_\s+/, $rule;
        if (! @{array_intersection( \@offer_fields, \@rule_fields )}) {
            $self->{required_fields_filter_log}{$rule}++ if $par{LOGGING};
            $res = 0;
        }
        last unless $res;
    }
    return $res;
}


sub _short_path {
    my ( $text ) = @_;
    my @temp = split /\s*\/\s*/, $text;
    my $path = $temp[-1];
    $path = $temp[-2].' '.$temp[-1] if defined($temp[-2]); # пока берем 2 последние секции
    return $path;
}

sub _append_feed_lang {
    my ($self, $hfields ) = @_;
    $hfields->{feed_lang} = $self->fds->feed_lang;
}

#####
# Добавляет product_type и minicategs к полям фида
#
our %categs_cache = (); # кэш для категоризации
our %dynbanners_categs_mapping = ();
our %forbidden_mono_categs = (); #категории из моно-тематик, в которые можно мапить только определенные типы фидов и нельзя все остальные
sub _append_product_type {
    my ($self, $hfields ) = @_;
    my @res = ();

    #проверка на монотематики: если тип фида однозначно мапится в тематику, сразу присваиваем product_type
    if ( defined $hfields->{feed_data_type} && defined $self->proj->options->{bannerland_mono_feed_types}{$hfields->{feed_data_type}} ) {
        my $type = $self->proj->options->{bannerland_mono_feed_types}{$hfields->{feed_data_type}};
        $hfields->{product_type} = $self->proj->options->{bannerland_categs}{$type}[0];
        return;
    }

    unless ( %dynbanners_categs_mapping ){
        open F, $self->proj->options->{dynbanners_categs_mapping};
        %dynbanners_categs_mapping = map { chomp; my ($c1, $c2) = split/\t/; $c1 => $c2 } <F>;
    }

    unless ( %forbidden_mono_categs ) {
        my $mono_feeds = $self->proj->options->{bannerland_mono_feed_types};
        foreach my $feed_type ( keys %{ $mono_feeds } ) {
            foreach my $categ ( @{ $self->proj->options->{bannerland_categs}{ $mono_feeds->{$feed_type} } } ) {
                $forbidden_mono_categs{$categ} = 1;
            }
        }
    }

    unless ( $hfields->{minicategs} ){
        # укорачиваем categspath, берем последнюю секцию и предпоследнюю, если последняя оказалась однословником
        if ( defined($hfields->{categpath}) ){
            $hfields->{categpath_short} = _short_path($hfields->{categpath});
        }
        if ( defined($hfields->{market_category}) ){
            $hfields->{marketpath_short} = _short_path($hfields->{market_category});
        }
        my $cache_key = '';
        $cache_key ||= $hfields->{$_} for qw/categpath market_category typePrefix/;
        $hfields->{minicategs} = $categs_cache{$cache_key} if $cache_key;
        unless ( $hfields->{minicategs} ){
            my @minicategs = ();
            my @mapped_categs = ();
            # 19.04.2017 переместил marketpath_short перед categpath_short из-за маркетного фида watchtown с наручными часами
            for my $fieldname ( qw/marketpath_short categpath_short typePrefix name model description/ ){
                my $text = $hfields->{$fieldname} || '';
                $text = $self->proj->phrase( $text )->head(10) if ( $fieldname eq 'description' );
                my @temp_minicategs = $self->proj->phrase($text)->get_minicategs;
                if (@temp_minicategs) {
                    @minicategs = @temp_minicategs if (!@minicategs); # Сохраняем первую успешную категоризацию, на случай если все @mapped_categs окажутся пустыми
                    @mapped_categs = grep {!exists $forbidden_mono_categs{$_}} map {$dynbanners_categs_mapping{$_} || ()} @temp_minicategs;
                    @minicategs = @temp_minicategs if @mapped_categs;
                }
                last if @mapped_categs;
            }

            $hfields->{minicategs} ||= join("/", @minicategs);
            $categs_cache{$cache_key} ||= $hfields->{minicategs} if ($cache_key && $cache_key =~ m/\w/i);
        }
    }
    if ( $hfields->{minicategs} ){
        my @mapped = grep {!exists $forbidden_mono_categs{$_} } map { $dynbanners_categs_mapping{$_} || () } split( "/", $hfields->{minicategs});
        $hfields->{product_type} = join( "/", uniq_array(@mapped) );
    }
}

sub _append_source_inf {
    my $self = shift;
    my $offer = shift;
    $offer->{source_letter} = $self->source_letter if $self->source_letter;
    $offer->{offer_source} //= $self->get_offer_source;
}

# добавляет к офферу отредирекченный домен и главное зеркало.
# чтобы не прокачивать все урлы, прокачивает только первые N урлов. После этого добавляет ко всем офферам самый топовый
sub _append_main_mirror {
    my $self = shift;
    my $offer = shift;
    return if !$offer->{url};
    return if $offer->{main_mirror};

    my $mirror_data = $self->fds->url_to_mirror_data($offer->{url});

    for my $field ( qw(main_mirror main_mirror_id orig_domain orig_domain_id) ) {
        $offer->{$field} = $mirror_data->{$field};
    }
}

#####
# Получить id фида в каком-то кэше
#
sub get_remotecache_id {
    my ( $self ) = @_;
    if ( $self->fds->{url} ) { return md5int_base64( $self->fds->{url} ); }
    if ( $self->fds->{data} ) { return md5int_base64( $self->fds->{data} ); }
    if ( $self->{data} ) { return md5int_base64( $self->{data} ); }

    # TODO: here to be other content!
    return 0;
}


#####
# Сброс буфера product type list
#

sub get_first_product_url :CACHE {
    my ($self) = @_;
    my $url;
    $self->proj->log('getting first product url');
    $self->iter_init;
    while (defined(my $ptl = $self->next_ptl_pack(100))) {
        if (@$ptl) {
            $url = $ptl->[0]->url;
            last;
        }
    }
    $self->iter_init;  # в любой непонятной ситуации делай iter_init
    return $url;
}

sub get_first_product_domain :CACHE {
    my $self = shift;
    my $url = $self->get_first_product_url;
    return if !$url;
    return $self->proj->page($url)->domain;
}

sub iter_init {
    my ($self) = @_;
    $self->__next_data_pack_ith(undef);
}

# хэш {$offer_source => $counters}, поля счётчиков:
# total - сколько всего офферов в файле, доступно после iter_finalize
# read - сколько офферов было прочитано в next_data_pack
# checked_required_fields - сколько прошло проверку required_fields
# output - сколько прошло фильтры (и возвращено функцией next_data_pack)
sub get_iter_offers_count {
    my $self = shift;
    return $self->__next_data_pack_ith->{offers_count} //= {};
}

sub update_iter_offers_count {
    my $self = shift;
    my $offers = shift;
    my $type = shift;
    my $counter = $self->get_iter_offers_count;
    for my $offer (@$offers) {
        my $source;
        if (!ref($offer)) {  # tskv line
            if ($offer =~ /(?:^|\t)offer_source=([^\t]*)/) {
                $source = $1;
            }
        } else {
            $source = $offer->{offer_source};
        }
        $source //= $self->get_offer_source;
        $counter->{$source}{$type}++;
    }
}

# дочитаем tskv до конца, чтобы определить общее кол-во офферов
sub iter_finalize {
    my $self = shift;
    $_->{total} = $_->{read} for values %{$self->get_iter_offers_count};
    while (defined (my $pack = $self->__next_data_pack_ith->{iter}->get_pack)) {
        $self->update_iter_offers_count($pack, 'total');
    }
}

#####
# Получить следующую порцию отфильтрованных офферов
# итерируемся пачками, так как есть проблемы с очень большими фидами
# в %par можем передать флаг для проверки обязательных полей (перфоманс)
# параметры:
#   $size - int          размер пачки (default: 1000)
#   no_mapping => bool   отключить маппинг (default: 0, т.е. маппинг работает)
#   check_required_fields => bool   проверка на обязательные поля (раньше было в _tskv2feed)
sub next_data_pack {
    my $self = shift;
    my $size = shift || 1000;
    my %par  = @_;

    my $do_mapping = ((not $par{no_mapping}) and (not $self->mapping_done));

    my $proj = $self->proj;

    my $ith = $self->__next_data_pack_ith;
    if ( !$ith ){
        $ith = {};
        $self->__next_data_pack_ith($ith);
        if (!$self->fds) {
            die "Error in conversion of data to feed: no feed_data_source!\n";
        }
        eval {
            $self->log("Getting offers_tskv");
            my $tskvfile = $self->fds->offers_tskv_light_file;
            $self->log("tskvfile: $tskvfile");
            $ith->{iter} = ObjLib::FileIter->new({ filename => $tskvfile });
            if ($do_mapping) {
                $self->log("Getting categs_tskv");
                my $categs_tskv = $self->fds->categs_tskv_light;
                $self->log("Getting categs inf");
                $ith->{categs_inf} = $self->_get_categs_hash($categs_tskv);
            }
        };
        if ($@) {
            die("Error in conversion of data to feed: ".$@."\n");
        }
        my $do_filter = 0;
        if ($self->{filters}) {
            $do_filter = 1;
            if (!$self->{filters_prepared}) {
                delete($_->{categoryId}) for grep { defined($_->{categoryId}) && $_->{categoryId}[0] == 1000000001 } values %{$self->{filters}};
                delete($_->{'categoryId =='}) for grep { defined($_->{'categoryId =='}) && $_->{'categoryId =='}[0] == 1000000001 } values %{$self->{filters}};
                $self->{filters} = $self->proj->filters( $self->{filters} );
                $self->{filters_prepared} = 1;
            }
            $ith->{filters} = $self->{filters};
        }

        $self->log("next_data_pack: do_mapping=$do_mapping, do_filter=$do_filter");
    }

    #Грузим пачку
    my $pack = $ith->{iter}->get_pack($size);

    if (!$pack) {
        # прочитали все данные
        $self->log('next_data_pack done!');
        $self->log('Cache stats:', $ith->{filters}->cache_stats) if $ith->{filters};
        return undef;
    }
    $self->update_iter_offers_count($pack, 'read');

    my $data = [];
    if ($do_mapping) {
        my $offers_tskv = join('', map { $_."\n" } @$pack);
        $data = $self->_tskv2feed($offers_tskv, $ith->{categs_inf}, no_mirrors => $par{no_mirrors});
    } else {
        # данные уже промаплены и провалидированы, просто парсим
        for my $line (@$pack) {
            my %h = map { split /=/, $_, 2 } split /\t/, $line;
            # json поля декодируем и сохраняем без _json
            for my $field (@json_fields) {
                if ($h{$field . "_json"}) {
                    my $json_str = delete $h{$field . "_json"};
                    $h{$field} = from_json($json_str);
                }
            }
            push @$data, \%h;
        }
    }

    # проверка на обязательные поля
    if ($par{check_required_fields} and $self->fds->required_fields ) {
        # логгируем информацию по отфильтрованным офферам
        $data = [ grep { $self->check_required_fields($_, LOGGING => 1) } @$data ];
    }
    $self->update_iter_offers_count($data, 'checked_required_fields');

    if ($ith->{filters} and @$data){
        $ith->{filters}->grep_filter_list($data, 'offerfilters'); #Пакетная обработка
        $data = [ grep { defined($_->{offerfilters}) } @$data ];

        # нам важна статистика кеширования контентных фильтров
        $proj->logger->debug('Cache stats:', $ith->{filters}->cache_stats);
    }
    $self->update_iter_offers_count($data, 'output');

    if ($par{init_specific_product_fields}) {
        for my $data_elem (@$data) {
            $data_elem->{__init_specific_product_fields} = $par{init_specific_product_fields};
        }
    }

    return $data;
}

#####
# Получить следующую порцию отфильтрованных product type list'ов
#
sub next_ptl_pack {
    my $self = shift;
    my $data = $self->next_data_pack(@_);
    return undef unless defined($data);

    if ($self->{use_as_name}) {
        my @use_as_name_fields = split /,/, $self->{use_as_name};
        $_->{use_as_name} = get_use_as_value(\@use_as_name_fields, $_) for @$data;
    }
    if ($self->{use_as_body}) {
        my @use_as_body_fields = split /,/, $self->{use_as_body};
        $_->{use_as_body} = get_use_as_value(\@use_as_body_fields, $_) for @$data;
    }

    return $self->proj->product_list( $data );
}

sub get_use_as_value {
    my ($use_as_fields, $offer) = @_;
    my @use_as_values = ();
    for my $field (@$use_as_fields) {
        if ($offer->{$field} || $offer->{lc($field)}) {
            push(@use_as_values, $offer->{$field} || $offer->{lc($field)});
        }
        else {
            @use_as_values = ();
            last;
        }
    }
    return join(" ", @use_as_values);
}

#####
# Смешиваем блоки офферов с одинаковыми categoryId друг с другом
#
sub shuf_offers_by_category_ids {
    my ($self, $tskv_file, $offers_limit) = @_;
    $offers_limit //= 10_000;

    my $res_file = $self->get_tempfile("offers_shuffled", UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});

    # читаем все офферы в @arr_offers а также извлекаем categoryId'ы
    my @arr_offers = ();
    my @categoryIds = ();
    open F, '<', $tskv_file;
    my $offers_cnt = 0;
    while (my $line = <F>) {
        last if $offers_cnt > $offers_limit;
        push @arr_offers, $line;
        push @categoryIds, $line =~ /categoryId\=(\d+)(?:\t|$)/i ? $1 : '-1';
        $offers_cnt++;
    }
    close F;

    # сортируем, чтобы сформировать цельные блоки категорий
    my @sorted_indices = sort { $categoryIds[$b] <=> $categoryIds[$a] } 0..$#arr_offers;
    # оставляем массивы согласованными
    @arr_offers = @arr_offers[ @sorted_indices ];
    @categoryIds = @categoryIds[ @sorted_indices ];
    # инициализируем указатели на блоки категорий
    my %categoryId_pointers = ();
    my $i = 0;
    my $offers_count = scalar(@arr_offers);
    while ($i < $offers_count) {
        $categoryId_pointers{$categoryIds[$i]} = $i;
        $i++;
        while ($i < $offers_count && $categoryIds[$i] == $categoryIds[$i-1]) {
            $i++;
        }
    }
    # формируем итоговый файл
    open F, '>', $res_file;
    my $k = 0;
    my @category_keys = reverse sort {$a <=> $b} keys %categoryId_pointers;
    my $categs_count = scalar(@category_keys);
    for($i = 0; $i < $offers_count; $i++) {
        my $offer_num = $categoryId_pointers{ $category_keys[$k] };
        print F $arr_offers[$offer_num];

        # сдвигаем указатель блока либо удаляем
        if ($offer_num + 1 >= $offers_count || $categoryIds[$offer_num] != $categoryIds[$offer_num+1]) {
            $category_keys[$k] = $category_keys[$categs_count-1]; #оптимизация, несущественно портит сэмпл
            $categs_count--;
            last unless $categs_count;
            $k = $k % $categs_count;
        } else {
            $categoryId_pointers{ $category_keys[$k] }++;
            $k = ($k + 1) % $categs_count;
        }
    }
    close F;

    return $res_file;
}

#####
# Получения хешика для превью для yml2directinf
# https://st.yandex-team.ru/DYNSMART-119
#
sub get_feed_offer_previews {
    my ($self, $count, $use_dse) = @_;
    my $proj = $self->proj;

    $count = 100 if (!$count || $count < 0);
    my $h_res = ();

    my $additional_data = ($proj->fdm->preview_mapping)[1];

    my $offers_tskv = $self->fds->offers_tskv_light_file;

    my $shuf_file = '';
    # если есть categoryId
    if ($proj->file($offers_tskv)->head_lines(100) =~ /categoryId/i) {
        # в начало помещаем как можно больше различных категорий
        $shuf_file = $self->shuf_offers_by_category_ids($offers_tskv, 10_000);
    } else {
        # просто шаффл
        my $tmp = $proj->file($offers_tskv)->shuf;
        $shuf_file = $tmp->{name};
    }
    $proj->log("get_feed_offer_previews: shuf file is done");

    # создаем фид с переупорядоченными исходными данными. Старый файл удаляем, если не extfile, и если шафл сгенерировал новый файл (в некоторых случаях он отдает старый)
    if ( $self->fds->{_offers_tskv_light_file} ne $shuf_file ) {
        unlink $self->fds->{_offers_tskv_light_file} if (!$self->fds->{extfile} || $self->fds->{extfile} ne $self->fds->{_offers_tskv_light_file});
        $self->fds->{_offers_tskv_light_file} = $shuf_file;
        $self->iter_init;
    }
    # за счет того, что мы специальным образом переупорядочили офферы,
    # идем по фиду до тех пор, пока не наберем $count офферов
    # выборка будет с максимальной дисперсией по категориям

    my $ignore_native_by_business_type = 0;
    if ($use_dse // 0) {
        ($ignore_native_by_business_type) = BM::BannersMaker::Tasks::BasePerfTask::get_generation_flags_for_business_type($self->fds->{business_type});
    }

    my $dse_fd;
    if ($ignore_native_by_business_type) {
        # генерим не по нативке, а по dse
        my $shuf_file_with_url = $proj->get_tempfile('offers_tskv_mpd', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
        # дописываем merge key
        open my $F, "<", $shuf_file;
        open my $FF, ">", $shuf_file_with_url;
        while (my $l = <$F>) {
            if ($l =~ /(?:\t|^)(offer:url|URL|Final URL)=(.+?)(?:\t|$)/) {
                print $FF "url=$2\t".$l;
            }
        }
        close $F;
        close $FF;

        my $shuf_file_with_merge = $proj->get_tempfile('offers_tskv_mpd', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
        my $perftask = $proj->perftask({});
        $perftask->add_merge_keys($shuf_file_with_url, $shuf_file_with_merge);

        if ($proj->file($shuf_file_with_merge)->size) {
            # получилось доклеить merge ключи, берём этот файл для генерации; если нет - оставляем предыдущий
            unlink $self->fds->{_offers_tskv_light_file} if (!$self->fds->{extfile} || $self->fds->{extfile} ne $self->fds->{_offers_tskv_light_file});
            $self->fds->{_offers_tskv_light_file} = $shuf_file_with_merge;
            $self->iter_init;

            my $product_url = $self->get_first_product_url;
            my $product_domain = $proj->page($product_url)->domain;
            if ($product_domain) {
                my $dse_feed_filename = $Utils::Common::options->{dirs}{banners_generation_files} . "/cache_dse_" . $perftask->get_source_key($product_domain);
                $dse_feed_filename =~ s/\./_/g;

                my $DAY = 60 * 60 * 24;
                my $days_expired = (-e $dse_feed_filename) ? ( $proj->file( $dse_feed_filename )->seconds_expired / $DAY ) : 7 * $DAY;
                if ($days_expired >= 7 * $DAY) {
                    my $dse_options = $proj->options->{DynSources}->{dse};
                    my $dse_feed_filename_unsorted = $perftask->get_source_file(
                        name             => 'source_dse',
                        yt_path          => $dse_options->{yt_path_domain},
                        sec_level_domain => $perftask->get_source_key($product_domain),
                        add_str => "product_type=dse\t",
                        directory_path => $Utils::Common::options->{dirs}{banners_generation_files},
                    );

                    $perftask->add_merge_keys($dse_feed_filename_unsorted, $dse_feed_filename);
                    unlink $dse_feed_filename_unsorted;
                }
                if (-e $dse_feed_filename) {
                    $dse_fd = BM::BannersMaker::Tasks::PerfMergeFeed->new({ proj => $proj, filename => $dse_feed_filename });
                }
            }
        }
    }

    my $sample_size = 0;
    my %pictures_urls = ();
    my @thumbs_by_offer_cnt = ();
    $self->iter_init;
    # просматриваем офферы пачками по $count штук
    while( $sample_size < $count && defined( my $ptl = $self->next_ptl_pack( int($count * 1.5), check_required_fields => 1, no_mirrors => 1)) ) {
        for my $pt(@$ptl) {
            next if !$proj->validate_url($pt->url);
            my $banners;
            if ($ignore_native_by_business_type && $pt->is_base_product) {
                if ($dse_fd) {
                    my $pt_dse = $dse_fd->get_by_key($pt->{merge_key});
                    $banners = $pt_dse->perf_banners if $pt_dse;
                }
            } else{
                $banners = $pt->perf_banners_single;
            }

            # берем заголовок
            $pt->{title} = $banners->[0]{title} || '';

            # берем оффер ид
            my $offer_id = $pt->{OfferID} || $pt->{id} || '';
            if ($offer_id && $pt->{title} && $pt->{picture} && @{$pt->{images}} && not $h_res->{$offer_id}) {
                # собираем в хеш данные по офферу + title для того, чтобы воспользоваться готовыми методами
                my @offer_fields = @BM::BannersMaker::Product::offer_fields;
                push @offer_fields, BM::BannersMaker::Product->get_common_offer_fields;
                push @offer_fields, 'title';
                my $offer = {map {$_ => $pt->{$_}} grep { defined $pt->{$_} } @offer_fields};
                my $offer_additional_data = $proj->fdm->add_values_to_map_hash( $additional_data, $offer );

                # получаем промапленный итоговый хеш без тумбов (thumbnails)
                my $h_res_ad_map = { $self->do_smartmap( $offer_additional_data, $pt ) };
                if ($pt->{additional_data} && $pt->{additional_data}{text}) {
                    # добавляем промапленные поля в text
                    $h_res_ad_map->{text}{$_} = $pt->{additional_data}{text}{$_} for grep { !$h_res_ad_map->{text}{$_} } keys %{$pt->{additional_data}{text}};
                }
                $h_res->{$offer_id} = $h_res_ad_map;

                # сохраняем, чтобы потом сходить в аватарницу всей пачкой
                $pictures_urls{$offer_id} = $pt->{images};

                $sample_size++;
                last if $sample_size == $count;
            }
        }
    }
    $proj->log("get_feed_offer_previews: ".(scalar keys %pictures_urls)." picture urls are to process");

    # добавляем тумбы пачкой - для оптимизации
    my @images = map { @$_ } values %pictures_urls;
    my $start_time = [gettimeofday()];
    my $img2ava = $proj->image('')->get_avatars(\@images);
    my $avatar_timings += tv_interval($start_time);
    while (my ($offer_id, $images) = each %pictures_urls) {
        my @avatars = grep { $_ } map { $img2ava->{$_}{avatars} } @$images;
        if (!@avatars) {
            $proj->log("get_feed_offer_previews: avatarnica did not respond for offer_id [$offer_id] images [" . join(', ', @$images) . "]");
            delete($h_res->{$offer_id});
            next;
        }
        $h_res->{$offer_id}->{image} = $avatars[0];
        $h_res->{$offer_id}->{images} = \@avatars;
    }
    $self->send_yml2directinf_timing("getting_thumbnails", $avatar_timings);
    $proj->log("get_feed_offer_previews: is done");
    return $h_res;
}


sub fake_yml2directinf {
    my $proj = shift;
    my $datacamp_crawler_url = shift;
    my $datacamp_crawler_domain = $proj->page($datacamp_crawler_url)->domain;
    return $proj->json_obj->encode({
        categs => [],
        file_data => 'TTTTTT', #Директ не может не сохранять данные и они не могут быть нулевой длины
        feed_type => 'YandexMarket',
        errors => [],
        warnings => [],
        all_elements_amount => 0,
        offer_examples => {},
        domain => {
            $datacamp_crawler_domain => 0
        },
    });
}

#####
# Ручка для получения информации для директа (с превью)
#
sub yml2directinf {
    my ($self, $h_params) = @_;
    my $start_time_total = [gettimeofday];

    # получаем базовую информацию
    my $res = $self->fds->get_directinf($h_params);

    my @errors_codes = ();
    if ($res->{errors} and (ref $res->{errors} eq 'ARRAY')) {
        @errors_codes = map { $_->{code} } (@{$res->{errors}});
    }
    if ( ( not @errors_codes ) and ( ($h_params->{gen_previews} || 0) eq '1') ) {
        my $start_time_previews = [gettimeofday];
        # получаем 100 случайных офферов для превью
        my $feed_offer_previews = '';
        do_safely( sub { $feed_offer_previews = $self->get_feed_offer_previews(100, $h_params->{use_dse} // 0); }, no_die => 1, timeout => 600, die_if_timeout => 0);
        $self->proj->log("yml2directinf: no previews!") unless ($feed_offer_previews);

        # запоминаем результат
        $res->{offer_examples} = {};
        $res->{offer_examples}{data_params} = {};
        if ($feed_offer_previews) {
            $res->{offer_examples}{data_params}{$_} = $feed_offer_previews->{$_} for (keys %$feed_offer_previews);
            $self->proj->log("yml2directinf: ".(scalar keys %$feed_offer_previews)." previews are ready");
        }
        my $timings_previews = tv_interval($start_time_previews);
        $self->send_yml2directinf_timing ("previews_only", $timings_previews);
    }

    # this debug needed to log all queries and their result
    my $logging_debug_info = {};
    $logging_debug_info->{url}                  = $self->{url};
    $logging_debug_info->{feed_id}              = $h_params->{feed_id};
    $logging_debug_info->{is_new_feed}          = $h_params->{is_new_feed};
    $logging_debug_info->{business_type}        = $self->{business_type};
    $logging_debug_info->{max_file_size}        = $self->{max_file_size};
    $logging_debug_info->{max_file_size_type}   = $self->{max_file_size_type};
    $logging_debug_info->{errors}               = $res->{errors};
    $logging_debug_info->{warnings}             = $res->{warnings};
    $logging_debug_info->{feed_type}            = $res->{feed_type};
    $logging_debug_info->{debug_info}           = delete $res->{debug_info};
    $logging_debug_info->{version}              = 5;
    $self->proj->log("YML2DIRECTINF_DEBUG: ".$self->proj->json_obj->encode($logging_debug_info));

    $res = $self->proj->json_obj->encode($res);
    my $timings_total = tv_interval($start_time_total);
    if (@errors_codes) {
        for my $e (@errors_codes) {
            if (defined $self->{solomon_client}) {
                $self->{solomon_client}->push_single_sensor({
                    cluster => "host_info",
                    service => "yml2directinf",
                    sensor  => "error_count",
                    labels  => {
                        host       => get_curr_host(),
                        error_type => "$e",
                    },
                    value   => 1,
                });
            }
        }
        $self->send_yml2directinf_timing ("with_errors", $timings_total);
    } else {
        $self->send_yml2directinf_timing ("with_previews", $timings_total);
    }
    return $res;
}

#####
# Получить полное количество офферов в фиде
#
sub get_total_offers_count :CACHE {
    my ($self) = @_;
    my $file = $self->fds->offers_tskv_light_file;
    return $self->proj->file($file)->wc_l;
}

#####
# Получить количество хороших (фильтрованных) офферов
#
sub get_offers_count {
    my ($self) = @_;
    my $offers_tskv = $self->fds->offers_tskv_light; # $self->offers_tskv_cached;
    my $filters = $self->proj->filters( $self->{filters} );
    return $filters->filter_tskv_count( $offers_tskv );
}

#####
# Получить информацию по тэгам офферов: частоту и пример значения
# Возвращает хэш в json'е
#
sub get_offers_tags_dict {
    my ($self) = @_;
    my $offers_tskv = $self->fds->offers_tskv_light; # $self->offers_tskv_cached;
    my $res = { stat => {}, examples => {},};
    for my $el (split /[\t\n]/, $offers_tskv){
        my ($k, $v) = split(/=/, $el, 2);
        next unless $k;
        next unless defined $v;
        $res->{stat}{$k}++;
        $res->{examples}{$k} ||= $v;
    }
    return $self->proj->json_obj->encode( $res );
}

#####
# Получить из name & categs поля title & body
#
sub name2titlebody {
    my ($self, $name, $categs) = @_;
    my $proj = $self->proj;

    my $len_name = length($name);
    my $categs_line = $categs;
    $categs_line =~ s/ \/ /. /g;
    my ($title, $body) = ('', '');
    if ( $len_name < 35 ){
        $title = $name;
        my ($bb1, $bb2) = $proj->phrase($categs_line)->smart_split_text(75);
        $body = $bb1;
    } else {
        my ($nn1, $nn2) = $proj->phrase($name)->smart_split_text(35);
        $title = $nn1;
        my ($bb1, $bb2) = $proj->phrase($categs_line)->smart_split_text(75 - $len_name + 35);
        $body = "$nn2 $bb1";
    }
    return ($title, $body);
}

#####
# Получить ошибку
#
sub error_text {
    my ($self) = @_;
    return $self->fds->{error} || '';
}

sub do_smartmap {
    my $self = shift;
    my @args = @_;
    $self->proj->fdm->do_smartmap(@args, feed => $self);
}

sub send_yml2directinf_timing {
    my ( $self, $metric, $time_value ) = @_;
    if (defined $self->{solomon_client}) {
        $self->{solomon_client}->push_single_sensor({
            cluster => "host_info",
            service => "yml2directinf",
            sensor  => "timings",
            labels  => {
                host   => get_curr_host(),
                metric => $metric,
            },
            value   => $time_value,
        });
    }
}

1;
