package BM::BannersMaker::FeedDataMapping;

use base qw(ObjLib::ProjPart);
use utf8;
use open ':utf8';

use std;

use Data::Dumper;
use Utils::Array;
use Utils::Sys qw(
    merge_hashes
);
use Utils::Funcs qw(get_urls_from_str);
use HTML::TreeBuilder;

sub image_possible_keys :CACHE {
    my $self = shift;
    my @list = ('picture', 'image', 'Image URL', 'g:image_link', 'image_link', 'image_url', 'imageLink', 'img',
        'pictures', 'images', 'image_urls', 'imgs', 'ProductImages:ImageURL1024x783',
        'bigimage' ,'images:image', 'offer:original_pictures:url',
    );
    return +{ map { lc($_) => 1 } @list };
}

sub is_image_field {
    my ($self, $key, $value) = @_;
    my $image_possible_keys = $self->image_possible_keys;
    my $lc_key = lc($key);
    return (($image_possible_keys->{$lc_key} or $lc_key =~ /\bimage\b/) && $value && $value !~ /^null$/i)
}

no warnings 'utf8'; # из-за варнингов ловим segfault в bmapi, глушим варнинги до решения DYNSMART-863

#####
# для переименования полей в FeedDataSource
#
sub param_rename :CACHE {
    return +{
        'пол'=>'sex',
        'gender'=>'sex',
        'цвет'=>'color',
        'сезон'=>'season',
        'сезонность'=>'season',
        'возраст'=>'age',
        'материал'=>'material',
        'корпус'=>'corpus',
        'тип элемента' => 'type',
        'коллекция'=>'collection',
        'размер' => 'size',
        'длина' => 'length',
        'ширина' => 'width',
        'высота' => 'height'
    };
}

#####
# для цветов на латинице
#
sub latin_colors_map :CACHE {
    my $self = shift;
    return +{
        'white'=>'белый',
        'blue'=>'синий',
        'cyan'=>'голубой',
        'black'=>'черный',
        'brown'=>'коричневый',
        'red'=>'красный',
        'yellow'=>'желтый',
        'orange'=>'оранжевый',
        'green'=>'зеленый',
        'indigo'=>'синий',
        'индиго'=>'синий',
        'violet'=>'фиолетовый',
        'purple'=>'сиреневый',
        'pink'=>'розовый',
        'silver'=> 'серебряный',
        'beige'=>'бежевый',
        'brown'=>'коричневый',
        'gray'=>'серый',
        'grey'=>'серый',
        'gold'=>'золотой',
        'magenta'=>'пурпурный'
    };
}

sub simple_color_map :CACHE {
    return +{
        'алый' => 'красный',
        'белый цвет шерсть' => 'белый',
        'бирюзовый' => 'синий',
        'бордовый' => 'красный',
        'бурый' => 'коричневый',
        'вишневый' => 'красный',
        'волна морской цвет' => 'синий',
        'голубой лед цвет' => 'синий',
        'изумрудный' => 'зеленый',
        'индиго' => 'синий',
        'коралловый' => 'красный',
        'королевский синий' => 'синий',
        'кость слоновый цвет' => 'белый',
        'кремовый' => 'бежевый',
        'лайма цвет' => 'зеленый',
        'лиловый' => 'фиолетовый',
        'лимонный' => 'желтый',
        'молочно-белый' => 'белый',
        'молочный' => 'белый',
        'мятный' => 'зеленый',
        'оливковый' => 'зеленый',
        'персиковый' => 'розовый',
        'песочный' => 'бежевый',
        'пудровый' => 'розовый',
        'пурпурный' => 'фиолетовый',
        'рыжий' => 'оранжевый',
        'салатовый' => 'зеленый',
        'серебряный' => 'серебристый',
        'сиреневый' => 'фиолетовый',
        'сливовый' => 'фиолетовый',
        'телесный' => 'бежевый',
        'фуксия' => 'розовый',
        'хаки' => 'зеленый',
        'ягодный' => 'фиолетовый',
        'экрю' => 'бежевый',
        'васильковый' => 'голубой',
    };
}

sub test_smartmap {
    my ( $self, $textdata, $source_to_canonical, $source_to_yabs, $offer_tag, $required_fields ) = @_;
    my %hpar = ();
    $hpar{data} = $textdata;
    $hpar{tskvmap} = $source_to_canonical if $source_to_canonical;
    $hpar{additional_data} = $source_to_yabs if $source_to_yabs;
    $hpar{offer_tag} = $offer_tag if $offer_tag;
    $hpar{required_fields} = $required_fields if $required_fields;
    my $feed = $self->proj->feed( \%hpar );
    return $feed;

}

# к каждому полю правила добавляет его значение из хеша, в FeedDataSource
sub add_values_to_map_hash {
    my ($self, $map_config_text, $h) = @_;
    my @res = ();
    my $fields_from = '';
    for my $rule ( split /\s*\n\s*/, $map_config_text ){
        next unless $rule;
        my @arr_rules = split/\s*\=\>\s*/, $rule || ();
        if ( scalar( @arr_rules ) == 1 || $arr_rules[0] =~ /^_FUNC_/ || $arr_rules[1] =~ /^\"/ || (scalar(@arr_rules) > 2 && $arr_rules[0] =~ /^\"/ ) ){
            push @res, $rule;
            next;
        }
        my $fields_from = $arr_rules[0];
        my @kv_found = ();
        for my $field ( grep { exists $h->{$_} } map { $_ =~ s/^\s+|\s+$//g; $_ } split(/\s*\,\s*/, $fields_from) ) { ##no critic
            if (ref($h->{$field}) eq 'ARRAY') {
                push @kv_found, $field . '=' . join(',', @{$h->{$field}});
            } elsif (!ref($h->{$field})) {
                push @kv_found, $field . '=' . $h->{$field};
            }
        }
        next unless @kv_found;
        my $kv_found_text = join(',', @kv_found);
        $arr_rules[0] = $kv_found_text;
        my $rule_with_values = join(' => ', @arr_rules);
        push @res, $rule_with_values;
    }
    return join("\n", @res);
}

#####
# Осуществляем маппинг
# в доп. параметрах можно передать feed => $feed_object (для adv_type, например)
#
# Это основной метод маппинга!
# Параметры:
#   $maptext --  текст с правилами маппинга, в котором к полям приклеены значения (функцией add_values_to_map)
#   $hoffer  --  хэш с данными оффера, сейчас не используется
#
sub do_smartmap {
    my ( $self, $maptext, $hoffer, %par) = @_;
    my %hres = ();
    $maptext =~ s/^\s+|\s+$//g;
    for my $maprule ( split/\s*\n\s*/, $maptext ) {
        my @arr_maprule = split/\s*\=\>\s*/, $maprule;
        # функция
        if ( $arr_maprule[-1] =~ /^_FUNC_/ ) {
            my $func_with_params = $arr_maprule[-1];
            my ( $func_name, @arr_params_line ) = split/(?<=[^,])\s+/, $func_with_params;
            my $func_params_line = join(' ', @arr_params_line );
#       print STDERR Dumper ( [ ( $func_name, $func_params_line ) ] );;
            my @func_params = ();
            push @func_params, $hoffer;
            push @func_params, $arr_maprule[0] if ((@arr_maprule > 1) and ($func_name ne '_FUNC_FORMAT_STR_'));
            push @func_params, ( _params_line_to_arr($func_params_line) ) if $func_params_line;
#       print STDERR Dumper ( [ \@arr_maprule, $func_name, $func_params_line, \@func_params ] );
            my %hret = $self->$func_name( @func_params );
            if ( scalar(@arr_maprule) > 2 ) {

                # двигаемся по уровням ключа хеша
                my $sub_hres = \%hres;
                for my $key_part ( @arr_maprule[1..$#arr_maprule-2] ){
                    $sub_hres->{$key_part} ||= {};
                    $sub_hres = $sub_hres->{$key_part};
                }

                $sub_hres->{$arr_maprule[-2]}{$_} = $hret{$_} for ( keys %hret );
            } else {
                $hres{$_} = $hret{$_} for ( keys %hret );
            }
            next;
        }
        # приписывается фиксированное значени (литерал)
        if ( @arr_maprule == 2 && $arr_maprule[1] =~ /^\"(.+)\"/ ) {
            $hres{$arr_maprule[0]} = $1;
            next;
        }
        # обычный прямой маппинг
        if ( @arr_maprule == 2 && $arr_maprule[0] =~ /^[^=]+=(.+)$/) {
            $hres{$arr_maprule[1]} = $1;
            next;
        }
        my @kv = split(/\=/, $arr_maprule[0], 2);
        if ($arr_maprule[0] =~ /^\"/) {
            my $const_value = substr($arr_maprule[0], 1, length($arr_maprule[0])-2);
            @kv = ('', $const_value);
        }
        # многоуровневый хеш
        if ( @arr_maprule > 2 ) {
            shift @arr_maprule;
            my $firstrule = shift @arr_maprule;
            my $hrule = {};
            $hres{$firstrule} ||= {};
            $hrule = $hres{$firstrule};
            for my $rulename ( @arr_maprule[0..$#arr_maprule-1] ){
                $hrule->{$rulename} ||= {};
                $hrule = $hrule->{$rulename};
            }
            $hrule->{$arr_maprule[-1]} = $kv[1];
            next;
        }
    }
    return %hres;
}

#####
# Бьет строку по запятым. Не бьет, если запятые внутри кавычек (чтобы в кавычках можно было передавать любые литералы)
# также удаляет пробелы, если они не внутри кавычек
#
sub _params_line_to_arr {
    my ($line) = @_;
    my $qopen = 0;
    my @res = ();
    my @buffer = ();
    for my $sym ( split//,$line ){
        if ( $sym eq ',' ){
            if ( $qopen ){
                push @buffer, $sym;
            } else {
                push @res, join('', @buffer) if @buffer;
                @buffer = ();
            }
        } elsif ( $sym eq '"' ){
            $qopen = ($qopen == 0 ? 1:0);
            next;
        } else {
            push @buffer, $sym if ( $sym ne ' ' || $qopen );
        }
    }
    push @res, join('', @buffer) if ( @buffer && !$qopen );
    return @res;
}


############################################################################
# Ресурсы для маппинга
############################################################################

#####
# Получить маппинг для некоторых полей из таблицы PerfParamMapping
#
#TODO: переписать это как в DYNSMART-557
sub facilities_dict :CACHE {
    my ($self) = @_;
    my $proj = $self->proj;
    my $dbt;
    if ($ENV{MR_BROADMATCH}) {
        $dbt = $proj->elem_factory->elem_from_db_dump('bannerland', 'PerfParamMapping');
    } else {
        $dbt = $proj->dbtable('PerfParamMapping', 'ID', 'bannerland_dbh');
    }
    my %h = map { my $desr = $_->{desr}; map { $_ => $desr } split(',', lc($_->{src})) } map {@$_} $dbt->List();
    return \%h;
}

#####
# Получение информации о регионе по region name
# Локально кэшируется
#
my $region_cache = {};
sub _get_region_inf {
    my ( $self, $reg_name ) = @_;
    return () unless $reg_name;
    my $reg_inf = $region_cache->{$reg_name};
    unless( $reg_inf ){
        $reg_inf = {};
        my @reg_ids = $self->proj->phrase( $reg_name )->get_geobase_region_ids;
        $reg_inf->{geoid} = $reg_ids[0] || '';
        $region_cache->{$reg_name} = $reg_inf;
    }
    return %$reg_inf;
}

###############################################################################
# Функции для маппинга
###############################################################################

#####
# зануляет поле, если в нем есть невалидные с точки зрения Директа символы
#
sub _FUNC_BS_COMPATIBLE_ {
    my ( $self, @params ) = @_;
    return () unless @params > 2;
    my ( $h_offer, $source_field_kv, $target_field_name ) = @params;
    my %h_res = ();

    my $text = ( split/\=/, $source_field_kv, 2 )[1];
    if ( !$self->proj->is_bs_compatible( $text ) ) {
        $text = '';
    }
    $h_res{$target_field_name} = $text;
    return %h_res;
}

#####
# Объединяет несколько полей через разделитель в одно поле
#
sub _FUNC_CONCAT_ {
    my ( $self, @params ) = @_;
    return () unless @params > 3;
    my ($h_offer, $source_fields_text, $delim, $target_field_name, @necessary_fields) = @params;
    my %h_res = ();

    my @source_fields_kv = split /\s*\,\s*/, $source_fields_text;
    if ( @necessary_fields ) {
        @necessary_fields = map { $_ =~ s/^necessary:\s*//; $_; } @necessary_fields; ##no critic
        my @existing_fields = map { (split/\=/)[0] } @source_fields_kv;
        my @not_found = @{array_minus(\@necessary_fields, \@existing_fields)};
        return () if ( @necessary_fields and @not_found );
    }
    $delim =~ s/\\//g; # экранированные спецсимволы
    $delim = ',' if $delim eq '_comma_';
    $delim = ' ' if $delim eq '_space_';
    $delim = ';' if $delim eq '_semicolon_';
    my @values = map { (split/\=/)[1] } @source_fields_kv;
    return () unless @values;
    my $value = join( $delim, @values );
    $h_res{$target_field_name} = $value;
    return %h_res;
}

# Для фидов AutoRu если есть информация о руле (право- или леворульная), унифицируем её в нужном нам виде
# Дубль кода _FUNC_CONCAT_, но с обработкой поля 'wheel' (т.к. текущие правила маппинга не допускают суперпозиции функций)
sub _FUNC_CONCAT_WITH_WHEEL_ {
    my ( $self, @params ) = @_;
    return () unless @params > 3;
    my ($h_offer, $source_fields_text, $delim, $target_field_name, @necessary_fields) = @params;
    my %h_res = ();

    my @source_fields_kv = split /\s*\,\s*/, $source_fields_text;
    if ( @necessary_fields ) {
        @necessary_fields = map { $_ =~ s/^necessary:\s*//; $_; } @necessary_fields; ##no critic
        my @existing_fields = map { (split/\=/)[0] } @source_fields_kv;
        my @not_found = @{array_minus(\@necessary_fields, \@existing_fields)};
        return () if ( @necessary_fields and @not_found );
    }
    $delim =~ s/\\//g; # экранированные спецсимволы
    $delim = ',' if $delim eq '_comma_';
    $delim = ' ' if $delim eq '_space_';
    $delim = ';' if $delim eq '_semicolon_';

    my @values = ();
    for my $source_elem (@source_fields_kv) {
        my @split_val = split /\=/, $source_elem;
        if (($split_val[0] eq "wheel") && ($split_val[1])) {

            my $left_wheel = $split_val[1] =~ m/\bлевый\b/i;
            my $right_wheel = $split_val[1] =~ m/\bправый\b/i;
            if (($left_wheel) && (!$right_wheel)) {
                push @values, "Левый руль";
            } elsif ((!$left_wheel) && ($right_wheel)) {
                push @values, "Правый руль";
            }

        } else {
            push @values, $split_val[1];
        }
    }

    return () unless @values;
    my $value = join( $delim, @values );
    $h_res{$target_field_name} = $value;
    return %h_res;
}

#####
# Сплитит цену и валюту
#
sub _FUNC_SPLIT_PRICE_ {
    my ( $self, @params ) = @_;
    return () unless @params > 3;
    my ($h_offer, $source_field_kv, $price_field, $currency_field) = @params;
    my %h_res = ();

    my $value_to_split = (split/\=/, $source_field_kv)[1];
    my $currency_re = $self->proj->options->{currency_re};
    if ($value_to_split =~ /^(\d+[\.,]?\d*)\s*($currency_re)?$/i) {
        $h_res{$price_field} = $1 if $price_field ne '_';
        $h_res{$currency_field} = uc $2 if $2 and $currency_field ne '_';
    }
    return %h_res;
}

#####
# Вызывает _FUNC_SPLIT_PRICE_ если в оффере есть другое указанное поле
#
sub _FUNC_SPLIT_PRICE_IF_ANOTHER_FIELD_EXISTS_ {
    my ( $self, @params ) = @_;
    return '' if (scalar @params < 5);

    my ($offer, $source_field, $another_field, $price_field, $currency_field) = @params;
    return $self->_FUNC_SPLIT_PRICE_($offer, $source_field, $price_field, $currency_field) if ($offer->{$another_field});
    return ();
}

#####
# Вызывает _FUNC_SPLIT_PRICE_ если в оффере нет более приоритетного поля
#
sub _FUNC_SPLIT_PRICE_IF_NO_MORE_PRIORITY_FIELD_ {
    my ( $self, @params ) = @_;
    return '' if (scalar @params < 5);

    my ($offer, $source_field, $more_priority_field, $price_field, $currency_field) = @params;
    if (exists $offer->{$more_priority_field} && defined $offer->{$more_priority_field} && length($offer->{$more_priority_field})) {
        return ();
    }
    return $self->_FUNC_SPLIT_PRICE_($offer, $source_field, $price_field, $currency_field);
}

#####
# Отображение некоторых полей в нормы
#
sub _FUNC_NORM_FACILITY_ {
    my ( $self, @params ) = @_;
    return () unless @params > 2;
    my ( $h_offer, $source_field_kv, $target_field_name ) = @params;
    my %h_res = ();

    my $facilities_text = ( split/\=/, $source_field_kv )[1];
    return () unless $facilities_text;
    my $facilities_dict = $self->facilities_dict;
    return () unless %$facilities_dict;
    my $facilities_text_mapped = join(',', grep {$_} map { $facilities_dict->{$_} } split(';', lc($facilities_text)));
    return () unless $facilities_text_mapped;
    $h_res{$target_field_name} = $facilities_text_mapped;
    return %h_res;
}

sub _FUNC_GET_FIRST_PICTURE_ {
    my ( $self, @params ) = @_;
    return () unless @params > 2;
    my ( $h_offer, $source_field_kv, $target_field_name ) = @params;
    my %h_res = ();
    my $elem_text = ( split/\=/, $source_field_kv, 2)[1];
    my @images = get_urls_from_str($elem_text);
    $h_res{$target_field_name} = $images[0];
    return %h_res;
}

#####
# Получение geo_id по region name
#
sub _FUNC_GET_GEOBASE_ID_ {
    my ( $self, @params ) = @_;
    return () unless @params > 2;
    my ($h_offer, $source_field_kv, $target_field_name) = @params;
    my %h_res = ();

    my $region = ( split/\=/, $source_field_kv )[1];
    return () unless $region;
    my %h_reg_inf = $self->_get_region_inf( $region );
    return () unless %h_reg_inf;
    $h_res{$target_field_name} = $h_reg_inf{geoid};
    return %h_res;
}

sub _format_str {
    my ( $proj, $allow_empty, $case_insensitive, $translit, @params ) = @_;
    return () unless @params > 2;
    my ($h_offer, $line_template, $target_field ) = @params;

    my @line = ();
    my @subst = ();
    @subst = $line_template =~ /\[(.+?)\]/g;
    if ($case_insensitive) {
        $h_offer = {map{lc $_ => $h_offer->{$_}} keys %$h_offer};
    }
    my $res_line = $line_template;
    for my $fieldname ( @subst ){
        my $fieldvalue = $h_offer->{$case_insensitive ? lc $fieldname : $fieldname};
        if ($fieldvalue) {
            if ($translit) {
                $fieldvalue = $proj->phrase($fieldvalue)->translit_simple(tolang=>'en');
            }
            $res_line =~ s/\[$fieldname\]/$fieldvalue/;
        } elsif ($allow_empty) {
            $res_line =~ s/\S*?\[$fieldname\]\S*?(\s|$)//;
        } else {
            return ();
        }
    }
    $res_line =~ s/^\s+|\s+$//g;
    return ($target_field => $res_line);
}

#####
# Подставляет в переданный шаблон строки значния полей, указанных в шаблоне
#
sub _FUNC_FORMAT_STR_ {
    my ( $self, @params ) = @_;
    return _format_str($self->proj, 1, 0, 0, @params);
}

#####
# Подставляет в переданный шаблон строки значния полей, указанных в шаблоне
# Если хотя бы одно из значений полей отсутствует или пустое, то ничего не возвращает
# Все поля матчатся независимо от регистра
#
sub _FUNC_FORMAT_STR_NO_EMPTY_CI_ {
    my ( $self, @params ) = @_;
    return _format_str($self->proj, 0, 1, 0, @params);
}

#####
# Подставляет в переданный шаблон строки значния полей, указанных в шаблоне
# Если хотя бы одно из значений полей отсутствует или пустое, то ничего не возвращает
# Все поля матчатся независимо от регистра
# Значения полей переводятся транаслитом в латиницу
#
sub _FUNC_FORMAT_STR_NO_EMPTY_CI_TRANSLIT_ {
    my ( $self, @params ) = @_;
    return _format_str($self->proj, 0, 1, 1, @params);
}


# зануление поля
sub _FUNC_NULLIFY_FIELD {
    my ( $self, @params ) = @_;
    return () unless @params > 1;
    my ($h_offer, $field_name) = @params;
    return ( $field_name => '' );
}


#####
# Нормализация единицы валюты
#
sub _FUNC_CONVERT_CURRENCY_CODE_ {
    my ( $self, @params ) = @_;
    return () unless @params > 2;

    my ($h_offer, $source_field_kv, $target_field) = @params;
    my $source_field_value = ( split/\=/, $source_field_kv )[1];
    return ( $target_field => 'TRY' ) if ( $source_field_value eq 'TL' );
    return ( $target_field => 'EUR' ) if ( $source_field_value eq 'EURO' );
    return ( $target_field => $source_field_value );
}

#####
# Получить тип недвижимости: ЖК или квартира
#
sub _FUNC_GET_REALTY_TYPE_ {
    my ( $self, $offer, $offer_type, $res_key) = @_;

    $offer_type = ( split('=', $offer_type) )[1];
    my %h_res = ();
    $h_res{$res_key} = $offer_type eq '1' ? 'building': 'apartment';
    return %h_res;
}

#####
# Получить строку о застройщике и проектной декларации
#
sub _FUNC_GET_REALTY_LEGAL_ {
    my ($self, $offer, $organization, $res_key) = @_;
    my %h_res = ();
    $organization =~ s/^.*?=//;
    $organization =~ s/^\s+|\s+$//g;
    $h_res{$res_key} = "Застройщик $organization. Проектная декларация на сайте " . $self->proj->options->{yandex_realty_disclaimer_site};
    return %h_res;
}

#####
# Получить currencyId по домену (пока только для турков)
#
sub _FUNC_GET_CURRENCYID_BY_DOMAIN_ {
    my ($self, $offer, $url, $res_key) = @_;
    return {} unless $url;
    $url =~ s/^[^=]*?=//;
    my $domain = $self->proj->page($url)->domain;
    return {} unless $domain;
    my %h_res = ();
    $h_res{$res_key} = "TRY" if ($domain =~ /\.tr$/);
    return %h_res;
}

#####
# Получить название страны по коду
# не смартмаппинг
#
my %h_country_codes = ();
sub _FUNC_COUNTRY_BY_2CODE_ {
    my ( $self, @params ) = @_;
    return '' unless @params > 0;
    my $code = (@params);

    $code = lc $code;
    unless ( %h_country_codes ){
        my $phl = $self->proj->dict_manager->get_dict("country_codes", '')->phrase_list;
        for ( @$phl ){
            my @temp = split/\t/, $_->text;
            $h_country_codes{$temp[0]} = $temp[1];
        }
    }
    return '' unless defined $h_country_codes{$code};
    return $h_country_codes{$code};
}

#####
# Ширина картинки по урлу
#
sub _FUNC_GET_IMAGE_WIDTH_ {
    my ( $self, @params ) = @_;
    my ($hoffer, $url_line, $target_field) = @params;
    my $url = '';
    $url = ( split('=', $url_line, 2) )[1] if $url_line;
    return '' unless $url;

    my $image = $self->proj->image($url);
    #print STDERR "url = '$url'\n";
    #my @size = $image->get_image_size;
    #print STDERR Dumper \@size;
    return ($target_field, ($image->get_image_size)[0]);
}

#####
# Высота картинки по урлу
#
sub _FUNC_GET_IMAGE_HEIGHT_ {
    my ( $self, @params ) = @_;
    my ($hoffer, $url_line, $target_field) = @params;
    my $url = '';
    $url = ( split('=', $url_line, 2) )[1] if $url_line;
    return '' unless $url;

    my $image = $self->proj->image($url);
    #print STDERR "url = '$url'\n";
    #my @size = $image->get_image_size;
    #print STDERR Dumper \@size;
    return ($target_field, ($image->get_image_size)[1]);
}


#####
# Дописывает http к урлу, если его там нет
#
sub _FUNC_ADD_HTTP_IF_NEEDED_ {
    my ( $self, @params ) = @_;
    return '' if (scalar @params < 3);

    my $url = $params[1];
    $url =~ s/^.*?=//;
    if ($url and ($url !~ /^(https?|ftp):\/\//i)) {
        $url =~ s/^:?\/\/?([^\/])/http:\/\/$1/i; # фикс, если ссылка начинается на //, /, :/, ://
        if ( $url !~ /^(https?|ftp):\/\// ) {
            $url = 'http://'.$url;  # фиксим, если она начинается со всего остального
        }
    }
    my $target_field = $params[2];
    return ( $target_field => $url );
}

#####
#  Удаляет флаг no_images, если есть поле images
#
sub _FUNC_DELETE_FLAG_NO_IMAGES_ {
    my ( $self, $h_offer ) = @_;
    return () unless ($h_offer->{bad_flags});

    my $bad_flags = $h_offer->{bad_flags};
    $bad_flags =~ s/^.+=//;
    my @new_flags = split ",", $bad_flags;
    @new_flags = grep { $_ ne 'no_images' } @new_flags;
    my %h_res = ();
    if (@new_flags) {
        $h_res{'bad_flags'} = join ",", @new_flags;
    } else {
        delete $h_offer->{bad_flags};
    }
    return %h_res;
}

sub _FUNC_PROCESS_DATACAMP_PRICE_ {
    my ( $self, @params ) = @_;
    return '' if (scalar @params < 3);

    my ($h_offer, $source_field_kv, $target_field_name) = @params;
    my $price = (split/\=/, $source_field_kv, 2)[1];
    return ( $target_field_name => $price / 10_000_000 );
}

sub _FUNC_PROCESS_DATACAMP_CATEGPATH_ {
    my ( $self, @params ) = @_;
    return '' if (scalar @params < 3);

    my ($h_offer, $source_field_kv, $target_field_name) = @params;
    my $datacamp_categpath = (split/\=/, $source_field_kv, 2)[1];
    $datacamp_categpath =~ s!\\! / !g; # convert datacamp categpath to own format (change categs delim "\" => " / ")
    return ( $target_field_name => $datacamp_categpath );
}

#####
# Обрезает текст под требуемое количество символов. Если на входе html-код, выпарсивает оттуда текст и обрезает. Берет только тот текст, который внутри валидных тегов
#
sub _FUNC_CUT_HTML_EXPLICIT_TEXT_ {
    my ( $self, @params ) = @_;
    return () unless @params > 3;
    my ( $h_offer, $source_field_kv, $length, $target_field_name ) = @params;
    my %h_res = ();

    my $text = ( split/\=/, $source_field_kv, 2)[1];
    #HTML check
    if ( $text =~ /<[^<>]+>/ ) {
        $text =~ s/</ </g;
        $text =~ s/\s+/ /g;
        my $root = HTML::TreeBuilder->new();
        $root->parse_content($text);
        _remove_implicit_html_content_from_tree($root);
        $text = $root->as_text;
        $root->delete();
    }
    $text = substr($text, 0, $length - 3) . '...' if ( length ($text) > $length );

    $h_res{$target_field_name} = $text;
    return %h_res;
}

sub _FUNC_GET_CAR_YEAR_ {
    my ( $self, @params ) = @_;
    return () if (scalar @params < 3);
    my ( $h_offer, $source_field_kv, $target_field_name ) = @params;
    my %h_res = ();
    my $text = ( split/\=/, $source_field_kv, 2)[1];
    if ($text and ($text =~ /\s*(20\d{2}|19\d{2})\s*/)) {
        $h_res{$target_field_name} = $1;
    } else {
        $h_res{$target_field_name} = '';
    }
    return %h_res;
}

sub _remove_implicit_html_content_from_tree {
    my $elem = shift;
    my $nullify = 0;
    if ($elem->implicit()) {
        $nullify = 1;
    }
    foreach my $child ( @{$elem->content || [] }) {
        if (ref($child)) {
            _remove_implicit_html_content_from_tree($child);
        }
        elsif ( $nullify ) {
            $child = '';
        }
    }
}


#####
#  Обрезает текст под требуемое количество символов
#
sub _FUNC_CUT_STR_ {
    my ( $self, @params ) = @_;
    return () unless @params > 3;
    my ( $h_offer, $source_field_kv, $length, $target_field_name ) = @params;
    my %h_res = ();

    my $text = ( split/\=/, $source_field_kv, 2)[1];
    $text = substr($text, 0, $length - 3) . '...' if ( length ($text) > $length );

    $h_res{$target_field_name} = $text;
    return %h_res;
}

#####
#  Обрезает текст под требуемое количество символов, разделяя его по заданном символу
#
sub _FUNC_CUT_STR_BY_DELIM_ {
    my ( $self, @params ) = @_;
    return () unless @params > 4;
    my ( $h_offer, $source_field_kv, $max_length, $delim, $target_field_name ) = @params;

    my %h_res = ();
    $delim = '/;/' if ($delim eq '_slash_semicolon_');
    my $text = ( split/\=/, $source_field_kv, 2)[1];
    my @text_parts = split $delim, $text;
    my $result = '';
    if ( length ($text) <= $max_length ) {
        $result = $text;
    } else {
        my $i = 0;
        while (($i <= $#text_parts) and (length($result) <= $max_length)) {
            my $to_add = (length($result) == 0) ? '' : $delim;
            $to_add .= $text_parts[$i];
            if (length ($result) + length($to_add) <= $max_length) {
                $result .= $to_add;
            }
            $i++;
        }
    }

    $h_res{$target_field_name} = $result if ($result);
    return %h_res;
}

#############################################################################
# Кастомные маппинги
#############################################################################

my %feed_data_type2mapping_function = (
    'GoogleHotels'   => 'google_hotels_mapping',
    'GoogleFlights'  => 'avia_tickets_mapping',
    'AutoRu'         => 'cars_mapping',
    'GoogleMerchant' => 'google_merchant_mapping',
    'GoogleCustom'   => 'google_custom_mapping',
    'GoogleTravel'   => 'google_travel_mapping',
    'YandexCustom'   => 'yandex_custom_mapping',
    'YandexMarket'   => 'yabs_mapping',
    'DatacampYandexMarket'   => 'datacamp_yabs_mapping',
    'TravelBooking'  => 'travel_booking_mapping',
    'YandexRealty'   => 'realty_mapping'
);

sub get_mapping_by_feed_data_type {
    my ( $self, $feed_data_type ) = @_;
    die "Empty feed_data_type" unless ($feed_data_type);
    die "No mapping for this feed_data_type '$feed_data_type'" unless (exists $feed_data_type2mapping_function{$feed_data_type});
    my $mapping_method = $feed_data_type2mapping_function{$feed_data_type};
    die "No such mapping method '$mapping_method'" unless ($self->can($mapping_method));
    return $self->$mapping_method;
}

sub get_origin_mapping_h {
    my ( $self, $feed_data_type ) = @_;
    die "Empty feed_data_type" unless ($feed_data_type);
    my $feed_fields_by_type = $self->proj->options->{feed_fields_by_type};
    unless (exists $feed_fields_by_type->{$feed_data_type}) {
        $self->proj->log( "No origin mapping for this feed_data_type '$feed_data_type'");
        return {};
    }
    my $origin_fields_h = $feed_fields_by_type->{$feed_data_type};
    my $mapping_h = {};
    for my $key (keys %$origin_fields_h) {
        my $fields_set = $origin_fields_h->{$key};
        for my $field_name (@$fields_set) {
            $mapping_h->{lc($field_name)} = $field_name;
        }
    }
    return $mapping_h;
}

# маппинг для GoogleHotels
sub google_hotels_mapping {
    my ( $self ) = @_;

    my $str_source_to_canonical = '
        product_type => "Отели, гостиницы"
        Property name => name
        Destination URL => url
        Final URL => url
        Property ID => OfferID
        Destination name => location
        Image URL => picture
        Star rating => class
        Price => _FUNC_SPLIT_PRICE_ price, currencyId
        Property Type => property_type
        Facilities => facilities
        phrases => custom_phrases
        Contextual keywords => custom_phrases
    ';

    my $str_source_to_yabs = '
        Facilities => _FUNC_NORM_FACILITY_ facility
        Facilities => text => facility
        Star rating => text => class
        Score => total
        Max score => max_score
        Price => price => _FUNC_SPLIT_PRICE_ current, _
        Price => text => _FUNC_SPLIT_PRICE_ _, currency_iso_code
        Destination name => text => location
        Destination name => _FUNC_GET_GEOBASE_ID_ location

    ';
    my $required_fields = '
        Destination name
        Final URL _OR_ Destination URL
        Property name
        Property ID
        Image URL
    ';
    my $offer_tag = '';
    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}

# Маппинг для фидов типа GoogleFlights
sub avia_tickets_mapping {
    my ( $self ) = @_;

    my $str_source_to_canonical = '
        _FUNC_FORMAT_STR_NO_EMPTY_CI_TRANSLIT_ "[Destination ID]", OfferID
        _FUNC_FORMAT_STR_NO_EMPTY_CI_TRANSLIT_ "[Origin name]-[Destination ID]", OfferID
        _FUNC_FORMAT_STR_NO_EMPTY_CI_TRANSLIT_ "[Origin ID]-[Destination ID]", OfferID
        _FUNC_FORMAT_STR_NO_EMPTY_CI_ "[Origin name] - [Destination name]", name
        _FUNC_FORMAT_STR_NO_EMPTY_CI_ "[Origin name] [Destination name]", location
        Origin name => origin
        Destination name => destination
        Destination URL => url
        Final URL => url
        Image URL => picture
        Flight price => _FUNC_SPLIT_PRICE_ price, currencyId
        product_type => "Авиабилеты"
        phrases => custom_phrases
        Contextual keywords => custom_phrases
    ';

    my $str_source_to_yabs = '
        _FUNC_FORMAT_STR_NO_EMPTY_CI_ "[Origin Name] – [Destination Name]", name
        Origin name => text => from
        Destination name => text => to
        Flight price => price => _FUNC_SPLIT_PRICE_ current, _
        Flight price => text => _FUNC_SPLIT_PRICE_ _, currency_iso_code
    ';
    my $required_fields = '
        Destination ID
        Destination name
        Final URL _OR_ Destination URL
        Image URL
    ';
    my $offer_tag = '';
    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}

#####
# маппинг для фидов YandexRealty
#
sub realty_mapping {
    my ( $self ) = @_;

    my $str_source_to_canonical = '
        _FUNC_FORMAT_STR_ "[building-name] ["area:value"] [area:unit] [rooms]-к м.[location:metro:name]", name
        internal-id => OfferID
        location:locality-name => location
        location:sub-locality-name => sub_location
        image => _FUNC_GET_FIRST_PICTURE_ picture
        price:value => price
        price:currency => currencyId
        product_type => "Недвижимость"
        location:address => address
        location:metro:name => metro
        location:metro:time-on-foot => time-on-foot
        location:metro:time-on-transport => time-on-transport
        area:value => area

    ';

    my $str_source_to_yabs = '
        location:locality-name => _FUNC_GET_GEOBASE_ID_ location
        location:locality-name => text => location
        location:sub-locality-name => text => sub_location
        building-name => text => building_name
        location:metro:name => text => metro
        location:address => text => address
        price:value => price => current
        price:currency => text => currency_iso_code
        offer_type => text => _FUNC_GET_REALTY_TYPE_ realty_type
        sales-agent:organization => text => _FUNC_GET_REALTY_LEGAL_ legal
    ';
    my $required_fields = '
        sales-agent:organization
        image
        url
        OfferID
        sub_location _OR_ location
    ';
    my $offer_tag = 'offer';
    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}


# маппинг по умолчанию для фидов формата AutoRu
sub cars_mapping {
    my ( $self ) = @_;

    my $str_source_to_canonical = '
        minicategs => "Автомобили"
        product_type => "Автомобили"
        vin => OfferID
        unique_id => OfferID
        currency => currencyId
        folder_id => model
        mark_id => vendor
        year => _FUNC_GET_CAR_YEAR_ year
        images:image => _FUNC_GET_FIRST_PICTURE_ picture
        images => _FUNC_GET_FIRST_PICTURE_ picture
    ';

    my $str_source_to_yabs = '
        price => price => current
        currency => text => currency_iso_code
        folder_id => attribute => model
        folder_id => attribute => market => Model
        mark_id => attribute => market => Vendor
        folder_id => text => model
        mark_id => text => vendor
        year => text => _FUNC_GET_CAR_YEAR_ year
        run, run-metric => text => _FUNC_CONCAT_ _space_, run, "necessary:run"
        modification_id, year, body_type, wheel => text => _FUNC_CONCAT_WITH_WHEEL_ _semicolon_, facilities
        mark_id, folder_id, year => text => _FUNC_CONCAT_ _space_, name, "necessary:mark_id", "necessary:folder_id"
    ';
    my $required_fields = '
        unique_id _OR_ vin
        url
        images
    ';
    my $offer_tag = 'car';
    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}

# маппинг для фидов GoogleMerchant
sub google_merchant_mapping {
    my ( $self ) = @_;

    my $str_source_to_canonical = '
    id => OfferID
    g:id => OfferID
    title => name
    g:title => name
    g:availability => availability
    g:link => url
    g:link => link
    link => url
    g:image_link => image_link
    g:image_link => picture
    image_link => picture
    price => _FUNC_SPLIT_PRICE_ price, currencyId
    g:price => _FUNC_SPLIT_PRICE_ price, currencyId
    sale_price => _FUNC_SPLIT_PRICE_ price, _
    g:sale_price => _FUNC_SPLIT_PRICE_ price, _
    brand => vendor
    g:brand => vendor
    brand => brand
    g:brand => brand
    _FUNC_NULLIFY_FIELD product_type
    link => _FUNC_GET_CURRENCYID_BY_DOMAIN_ currencyId
    g:link => _FUNC_GET_CURRENCYID_BY_DOMAIN_ currencyId
    phrases => custom_phrases
    Contextual keywords => custom_phrases
    g:size => size
    g:size_system => size_unit
    size => size
    size_system => size_unit
    g:color => color
    g:description => description
    g:item_group_id => group_id
    ';

    my $str_source_to_yabs = '
    price => price => _FUNC_SPLIT_PRICE_IF_NO_MORE_PRIORITY_FIELD_ sale_price, current, _
    g:price => price => _FUNC_SPLIT_PRICE_IF_NO_MORE_PRIORITY_FIELD_ g:sale_price, current, _
    sale_price => price => _FUNC_SPLIT_PRICE_ current, _
    g:sale_price => price => _FUNC_SPLIT_PRICE_ current, _
    price => text => _FUNC_SPLIT_PRICE_IF_NO_MORE_PRIORITY_FIELD_ sale_price, _, currency_iso_code
    g:price => text => _FUNC_SPLIT_PRICE_IF_NO_MORE_PRIORITY_FIELD_ g:sale_price, _, currency_iso_code
    sale_price => text => _FUNC_SPLIT_PRICE_ _, currency_iso_code
    g:sale_price => text => _FUNC_SPLIT_PRICE_ _, currency_iso_code
    price => price => _FUNC_SPLIT_PRICE_IF_ANOTHER_FIELD_EXISTS_ sale_price, old, _
    g:price => price => _FUNC_SPLIT_PRICE_IF_ANOTHER_FIELD_EXISTS_ g:sale_price, old, _
    g:link => text => _FUNC_GET_CURRENCYID_BY_DOMAIN_ currency_iso_code
    link => text => _FUNC_GET_CURRENCYID_BY_DOMAIN_ currency_iso_code
    brand => attribute => market => Vendor
    g:brand => attribute => market => Vendor
    ';
    my $required_fields = '';
    my $offer_tag = 'item';
    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}


# маппинг для фидов TravelBooking
sub travel_booking_mapping {
    my ( $self ) = @_;

    my $str_source_to_canonical = '
        product_type => "Отели, гостиницы"
        Property Name => name
        Destination URL => url
        Final URL => url
        Property ID => OfferID
        Destination Name => location
        Image URL => picture
        Star Rating => class
        Price => _FUNC_SPLIT_PRICE_ price, currencyId
        Property Type => _FUNC_BS_COMPATIBLE_ property_type
        Rating => class
        Country => country
        Facilities => facilities
    ';

    my $str_source_to_yabs = '
        Facilities => _FUNC_NORM_FACILITY_ facility
        Facilities => text => facility
        Star Rating => text => class
        Score => total
        Max Score => max_score
        Price => price => _FUNC_SPLIT_PRICE_ current, _
        Price => text => _FUNC_SPLIT_PRICE_ _, currency_iso_code
        Destination Name => text => location
        Destination Name => _FUNC_GET_GEOBASE_ID_ location
    ';
    my $required_fields = '';
    my $offer_tag = 'listing';
    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}

#Маппинг для превью yml2directinf
sub preview_mapping {
    my ( $self ) = @_;

    my $str_source_to_canonical = '';
    my $str_source_to_yabs = '
    url => click_url => image_small
    url => click_url => image_big
    url => click_url => image_huge
    url => click_url => image_orig
    url => click_url => text_body
    url => click_url => text_name
    url => click_url => price_current
    url => click_url => price_old
    url => click_url => image_small
    url => count
    url => target_url
    currencyId => text => currency_iso_code
    price => price => current
    oldprice => price => old
    title => text => name
    ';
    my $required_fields = '';
    my $offer_tag = '';
    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );

    # в Feed.pm ходим целой пачкой картинок за аватарками
}

# маппинг для фидов формата GoogleCustom
sub google_custom_mapping {
    my ( $self ) = @_;

    my $str_source_to_canonical = '
        Destination URL => url
        Final URL => url
        Image URL => picture
        Item description => description
        Item Category => category
        Item title => name
        ID => OfferID
        _FUNC_FORMAT_STR_NO_EMPTY_CI_ "[ID]-[ID2]", OfferID
        Price => _FUNC_SPLIT_PRICE_ price, currencyId
        Price => _FUNC_SPLIT_PRICE_IF_ANOTHER_FIELD_EXISTS_ "Sale Price", oldprice, _
        Sale Price => _FUNC_SPLIT_PRICE_ price, _
        Sale Price => _FUNC_SPLIT_PRICE_ sale_price, _
        product_type => "Custom"
        phrases => custom_phrases
        Contextual keywords => custom_phrases
    ';

    my $str_source_to_yabs = '
        Item description => text => _FUNC_CUT_HTML_EXPLICIT_TEXT_ 160, description_for_direct
        Price => price => _FUNC_SPLIT_PRICE_ current, _
        Price => text => _FUNC_SPLIT_PRICE_ _, currency_iso_code
        Sale Price => price => _FUNC_SPLIT_PRICE_ current, _
        Sale Price => text => _FUNC_SPLIT_PRICE_ _, currency_iso_code
        Price => price => _FUNC_SPLIT_PRICE_IF_ANOTHER_FIELD_EXISTS_ "Sale Price", old, _
        adv_type => "custom"
    ';
    my $offer_tag = '';
    my $required_fields = '
        ID
        Final URL _OR_ Destination URL
        Image URL
    ';
    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}

# маппинг для фидов формата GoogleTravel
sub google_travel_mapping {
    my ( $self ) = @_;

    my $str_source_to_canonical = '
        _FUNC_FORMAT_STR_NO_EMPTY_CI_TRANSLIT_ "[Destination ID]", OfferID
        _FUNC_FORMAT_STR_NO_EMPTY_CI_TRANSLIT_ "[Origin ID]-[Destination ID]", OfferID
        Title => title
        Destination URL => url
        Final URL => url
        Destination name => destination
        Origin name => origin
        Image URL => picture
        Price => _FUNC_SPLIT_PRICE_ price, currencyId
        Price => _FUNC_SPLIT_PRICE_IF_ANOTHER_FIELD_EXISTS_ "Sale price", oldprice, _
        Sale price => _FUNC_SPLIT_PRICE_ price, _
        Sale price => _FUNC_SPLIT_PRICE_ sale_price, _
        product_type => "Путешествия"
        phrases => custom_phrases
        Contextual keywords => custom_phrases
    ';

    my $str_source_to_yabs = '
        Price => price => _FUNC_SPLIT_PRICE_ current, _
        Price => text => _FUNC_SPLIT_PRICE_ _, currency_iso_code
        Sale price => price => _FUNC_SPLIT_PRICE_ current, _
        Sale price => text => _FUNC_SPLIT_PRICE_ _, currency_iso_code
        Price => price => _FUNC_SPLIT_PRICE_IF_ANOTHER_FIELD_EXISTS_ "Sale price", old, _
        Title, Origin name, Destination name => text => _FUNC_FORMAT_STR_ "[Title] [Origin name] – [Destination name]", description_for_direct
    ';
    my $offer_tag = '';
    my $required_fields = '
        Destination ID
        Final URL _OR_ Destination URL
        Image URL
        Title
    ';
    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}

# маппинг для фидов формата YandexCustom
sub yandex_custom_mapping {
    my ( $self ) = @_;

    my $str_source_to_canonical = '
        URL => url
        Image => picture
        Description => description
        Title => name
        ID => OfferID
        _FUNC_FORMAT_STR_NO_EMPTY_CI_ "[ID]-[ID2]", OfferID
        Price => price
        Old Price => oldprice
        Currency => currencyId
        product_type => "Custom"
        phrases => custom_phrases
    ';

    my $str_source_to_yabs = '
        Description => text => _FUNC_CUT_HTML_EXPLICIT_TEXT_ 160, description_for_direct
        Price => price => current
        Old Price => price => old
        Currency => text => currency_iso_code
        adv_type => "custom"
    ';
    my $offer_tag = '';
    my $required_fields = '
        ID
        URL
        Image
    ';
    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}

# Маппинг для тематики clothes YandexMarket'а
#
# WARN! если хочется поменять и будем использовать НЕгруппировочные поля,
sub clothes_yabs_mapping {
    my ($self) = @_;

    my $source_to_yabs = '
        currencyId => text => currency_iso_code
        colors => text => colors
        consist => text => consist
        consist => text => consist_for_direct
        sizes => text => sizes
        sizes => text => sizes_for_direct
    ' . $self->yabs_mapping;

    return $source_to_yabs;
}


sub datacamp_yabs_mapping {
    my ( $self ) = @_;
    my $str_source_to_canonical = '
        offer:offer_id => id
        offer:offer_yabs_id => offerYabsId
        offer:original_content:available => available
        offer:original_content:url => url
        images => _FUNC_GET_FIRST_PICTURE_ picture
        offer:price:price => _FUNC_PROCESS_DATACAMP_PRICE_ price
        offer:price:currency => currencyId
        offer:original_content:shop_category_id => categoryId
        offer:original_content:shop_category_path => _FUNC_PROCESS_DATACAMP_CATEGPATH_ categpath
        offer:original_content:shop_vendor => vendor

        # need test
        offer:original_content:shop_model => model

        # need test
        offer:original_content:shop_vendor_code => vendorCode

        # need test
        offer:original_content:description => description

        # need test
        offer:original_content:type_prefix => typePrefix

        # need test
        offer:market_content:category_name => market_category
        # need test
        offer:price:old_price => _FUNC_PROCESS_DATACAMP_PRICE_ oldprice
        # need test
        offer:original_content:sales_notes => sales_notes
        # need test
        offer:original_content:name => name

        # params??? https://yandex.ru/support/partnermarket/elements/param.html
    ';

    my $str_source_to_yabs = '
        offer:original_content:description => text => _FUNC_CUT_HTML_EXPLICIT_TEXT_ 160, description_for_direct
        params_specformat => text => _FUNC_CUT_STR_BY_DELIM_ 160, _slash_semicolon_, params_for_direct
        offer:price:currency => text => currency_iso_code
    ';
    my $required_fields = '';
    my $offer_tag = '';

    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}

# маппинг для YandexMarket, https://st.yandex-team.ru/DYNSMART-301
sub yabs_mapping {
    my ($self) = @_;

    my $str_source_to_canonical = '
        age => age
        age => age_time
        size:unit => size_unit
    ';
    my $str_source_to_yabs = '
        description => text => _FUNC_CUT_HTML_EXPLICIT_TEXT_ 160, description_for_direct
        params_specformat => text => _FUNC_CUT_STR_BY_DELIM_ 160, _slash_semicolon_, params_for_direct
        currencyId => text => currency_iso_code
    ';
    my $required_fields = '';
    my $offer_tag = '';

    return ( $str_source_to_canonical, $str_source_to_yabs, $required_fields, $offer_tag );
}


sub yabs_postmapping {
    my ($self) = @_;
    my $source_to_yabs = '
        type => text => offer_type
        first_title => text => name_for_direct
    ';
    return $source_to_yabs;
}

sub yabs_postmapping_yandex_market {
    my ($self) = @_;
    my $source_to_yabs = '
        second_title => text => second_title_for_direct
    ';
    return $source_to_yabs;
}

#####
# Функция проводит дополнительный маппинг (вдобавок к существующему)
# В дополнительном маппинге могут использоваться поля, получаемые в результате работы парсера
# Вызывается от pt после генерации баннеров по нему
#
# на входе:
#   $pt  -  оффер
#   data_type =>  feed_data_source->{data_type}
sub proceed_post_mapping_for_pt {
    my ($self, $pt, %par) = @_;
    my $post_additional_data = $self->yabs_postmapping;
    if ($par{data_type} eq 'YandexMarket') {
        $post_additional_data .= "\n".$self->yabs_postmapping_yandex_market;
    }

    my $parseh = $pt->parse;
    my $pt_additional_data = join("\n",
        $self->add_values_to_map_hash( $post_additional_data, $pt ),
        $self->add_values_to_map_hash( $post_additional_data, $parseh ),
    );

    if ( $pt_additional_data ) {
        my %hres_ad_map = $self->do_smartmap( $pt_additional_data, $pt );

        $pt->{additional_data} ||= {};
        merge_hashes($pt->{additional_data},  \%hres_ad_map);
    }

    return $pt;
}

sub convert_tskvmap_to_hash {
    my ($self, $tskvmap, $required_fields) = @_;
    my $res = ();
    for my $line (grep {!/_FUNC_/} split("\n", $tskvmap)) {
        my ($left, $right) = split(/\s*\=\>\s*/, $line);
        $left =~ s/^\s+|\s+$//g if $left;
        $right =~ s/^\s+|\s+$//g if $right;
        if ($required_fields) {
            $res->{$left} = $right if $left && $right && exists($required_fields->{$right});
        }
        else {
            $res->{$left} = $right if $left && $right;
        }
    }
    return $res;
}

1;
