package BM::BannersMaker::FeedDataSource;

use utf8;
use open ':utf8';

use std;
use base qw(ObjLib::ProjPart);

use Digest::MD5 qw(md5_hex);
use Data::Dumper;
use Encode qw{ _utf8_on _utf8_off decode encode };
use Scalar::Util qw(weaken);
use Storable qw(dclone);
use Utils::Array qw(uniq_array in_array);
use Utils::Funcs qw(get_urls_from_str is_offer_id_valid is_price_valid);
use Utils::Urls qw(fix_url_scheme);
use Utils::Sys qw{split_csv_line uncompressdata file_text_sub file_result_sub make_good_utf_string};
use HTML::Entities qw(decode_entities);
use JSON qw{from_json};

########################################################
#Доступ к полям
########################################################

__PACKAGE__->mk_accessors(qw(
    page
    origin_map
    tskvmap
    additional_data
    required_fields
    feed_file_type
    feed_data_type
    feed_data_type_error
    feed_data_type_error_ru
    feed_lang
    page_text_file
    csv_delim
    srcfilesize
));

no warnings 'utf8'; # из-за варнингов ловим segfault в bmapi, глушим варнинги до решения DYNSMART-863

sub init {
    my ($self) = @_;

    #save data from Factory
    $self->feed_file_type( $self->{feed_file_type} ) if exists $self->{feed_file_type};
    $self->feed_data_type( $self->{feed_data_type} ) if exists $self->{feed_data_type};
    $self->feed_data_type_error( $self->{feed_data_type_error} ) if exists $self->{feed_data_type_error};
    $self->feed_data_type_error_ru( $self->{feed_data_type_error_ru} ) if exists $self->{feed_data_type_error_ru};
    $self->feed_data_type_error_structured( $self->{feed_data_type_error_structured} ) if exists $self->{feed_data_type_error_structured};
    $self->feed_lang( $self->{feed_lang} ) if exists $self->{feed_lang};
    $self->page_text_file( $self->{__page_text_file} ) if $self->{__page_text_file};
    $self->csv_delim( $self->{csv_delim} ) if $self->{csv_delim};
    $self->srcfilesize( $self->{srcfilesize}) if $self->{srcfilesize};

    #mapping
    $self->origin_map( $self->{origin_map} ) if $self->{origin_map};
    $self->tskvmap( $self->{tskvmap} ) if $self->{tskvmap};
    $self->additional_data( $self->{additional_data} ) if $self->{additional_data};
    $self->required_fields( $self->{required_fields} ) if $self->{required_fields};
}

our @list_cached_files = qw/_offers_tskv_light_file _categs_tskv_light_file page_text_file/;
#####
# Деструктор, удаляет временные закешированные файлы (чтобы лишний раз не мусорить, https://st.yandex-team.ru/DYNSMART-233)
# также проверяем, что они не совпадают с extfile, потому что его нельзя удалять
#
sub DESTROY {
    my ($self) = @_;
    unlink($self->{$_}) for (grep{$self->{$_} && -f $self->{$_} && (!$self->{extfile} || $self->{$_} ne $self->{extfile})} @list_cached_files);
}

sub _page_text_file_safe_copy {
    my ($self, $file_res) = @_;
    my $file = $self->page_text_file;
    open(F_IN, "< $file") or die "Cant open file $file: $@";
    open(F_OUT, "> $file_res") or die "Cant open file $file_res: $@";
    while(my $line = <F_IN>) {
        chomp $line;
        $line = make_good_utf_string($line);
        print F_OUT "$line\n";
    }
    close F_OUT;
    close F_IN;
}

#####
# Добавляем feed_data_type и feed_lang в файл
#
sub append_feed_info_to_file {
    my ($self, $file) = @_;
    return if !$self->feed_data_type && !$self->feed_lang;

    $self->proj->log('appending feed_data_type');

    my $temp_file = $self->get_tempfile('temp_feed.tmp', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
    open F_IN, "< $file";
    open F_OUT, "> $temp_file";
    while (<F_IN>) {
        my $append = '';
        if ($self->feed_data_type) {
            unless (m/(^|\t)feed_data_type(\t|$)/) {
                $append .= "feed_data_type=".$self->feed_data_type."\t";
            }
        }
        if ($self->feed_lang) {
            unless (m/(^|\t)feed_lang(\t|$)/) {
                $append .= "feed_lang=".$self->feed_lang."\t";
            }
        }
        if (!$append) {
            print F_OUT $_;
        } else {
            print F_OUT $append.$_;
        }
    }
    close F_IN;
    close F_OUT;

    $self->proj->do_sys_cmd("mv $temp_file $file");
    return;
}


sub filter_large_offers {
    my ($self, $input_file) = @_;

    my $output_file = $self->get_tempfile('offers_tskv_light_filtered', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});

    open( F_IN, "< $input_file" ) or die "Cant open file $input_file: $@";
    open( F_OUT, "> $output_file" ) or die "Cant open file $output_file: $@";
    my $line_number = -1;
    while (my $line = <F_IN>) {
        $line_number++;
        if ( length($line) > 1000000 ) {
            $self->proj->log("filtered large offer on line $line_number, length:", length($line));
            next;
        }
        print F_OUT $line;
    }
    close F_OUT;
    close F_IN;
    return $output_file;
}


#####
# Загрузка фида
# ! C препроцессингом
# ! БЕЗ маппинга
# фид записывается в файл
#
sub offers_tskv_light_file {
    my ($self, $h_par) = @_;

    return $self->{_offers_tskv_light_file} if $self->{_offers_tskv_light_file};

    my $file_res = $self->get_tempfile('offers_tskv_light', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
    $self->proj->log("no feed_file_type!") unless $self->feed_file_type;
    if ( $self->feed_file_type ) {
        if ( $self->feed_file_type eq 'csv' || $self->feed_file_type eq 'xls/xlsx' ) {
            $self->proj->log("offers_tskv_light_file csv");
            $self->csvfile2tskvfile($file_res);
        } elsif ( $self->feed_file_type =~ /^(?:yml|xml)$/ ) {
            $self->proj->log("offers_tskv_light_file xml/yml");
            $self->{offer_tag} ||= 'offer';
            my $file = $self->page_text_file;
            $self->proj->log("offers_tskv_light_file xml/yml $file -> $file_res");
            my $error_or_warning = eval { XMLParser::grep($file, $file_res, $self->{offer_tag}, $self->{is_new_feed}, $self->feed_data_type) }; 
            if ($error_or_warning->is_error()) {
                push(@{$self->{errors}}, { 
                    code => 1204,
                    message => 'XML Parser fatal error: '.$error_or_warning->message(),
                    message_ru => 'Фатальная ошибка XML-парсера: '.decode('UTF-8', $error_or_warning->message_ru(), Encode::FB_CROAK),
                });
            }
            if ($error_or_warning->is_warning()) {
                push(@{$self->{warnings}}, { 
                    code => 1204,
                    message => 'XML Parser warning: '.$error_or_warning->message(),
                    message_ru => 'Предупреждение от парсера XML: '.decode('UTF-8', $error_or_warning->message_ru(), Encode::FB_CROAK),
                });
            }
            if ($error_or_warning->is_error() && $h_par && $h_par->{save_error_feeds}) {
                my $bad_feed_file = $self->get_tempfile("bmapi_bad_feed");
                $self->proj->do_sys_cmd("cp $file $bad_feed_file");
                $self->proj->log("feed file written to '$bad_feed_file'");
            }
        } elsif ($self->feed_file_type eq 'offers_tskv' && !$self->{extfile} ) { # в Feed.pm при создании фида такой способ сейчас не используется, только через extfile
            $self->log("offers_tskv_light_file offers_tskv, (must be datacamp)");
            $file_res = $self->page_text_file;
        } elsif ($self->feed_file_type eq 'offers_tskv' && $self->{extfile} ) {
            $self->log("offers_tskv_light_file offers_tskv, extfile");
            $file_res = $self->page_text_file;
        }

        $file_res = $self->filter_large_offers($file_res);

        #добавление типа данных к фиду
        if ( $self->feed_data_type ) {
            $self->append_feed_info_to_file($file_res);
        } else {
            $self->{error} ||= "Err1213: Can't detect feed data type: ".$self->feed_data_type_error;
            push(@{$self->{errors}}, { 
                code => 1213,
                message => $self->feed_data_type_error,
                message_ru => ($self->can('feed_data_type_error_ru') ? $self->feed_data_type_error_ru : $self->feed_data_type_error),
            });
        }
        # New feed errors
        if ($self->feed_data_type_error && $self->feed_data_type_error =~ m/Err1204:/) {
            $self->{error} ||= $self->feed_data_type_error;
            push(@{$self->{errors}}, $self->feed_data_type_error_structured);
        }
    }
    unless ($self->proj->file( $file_res )->size) {
        $self->{error} ||= 'Err1205: No offers found!';
        push(@{$self->{errors}}, {
            code => 1205,
            message => 'No offers found!',
            message_ru => 'Не найдено офферов!',
        });
        $self->log($self->{error}); #вывод в лог если вдруг парсер споткнулся
    }
    $self->{_offers_tskv_light_file} = $file_res;
    return $file_res;
}

#####
# То же, что и offers_tskv_light_file + возвращает ТЕКСТ
#
sub offers_tskv_light {
    my ($self) = @_;
    return $self->proj->file($self->offers_tskv_light_file)->text;
}

sub csvfile2tskvfile {
    my ($self, $res_file) = @_;
    my @res = ();
    my $file = $self->page_text_file;
    #быстрый способ проверить, что в файле больше одной строки;
    open( F_IN, "< $file" ) or die "Cant open file $file: $@";
    open( F_OUT, "> $res_file" ) or die "Cant open file $res_file: $@";
    if ( $self->proj->file( $file )->is_multiline ) {
        my $firstline = undef;
        my @field_names;
        while (my $line = <F_IN>) {
            chomp $line;
            $line = make_good_utf_string( $line );
            $line =~ s/\N{U+FEFF}|\r//g;
            next if $line =~ m/^\s*$/;
            unless ( defined $firstline ) {
                $firstline = $line;
                @field_names = split_csv_line($firstline, $self->csv_delim);
                next;
            }
            # DYNSMART-396: если в фиде приходит какие-то неправильные и очень длинные строки то растет использование памяти
            next if length($line) > 1000000;
            my @values = split_csv_line($line, $self->csv_delim);
            if (scalar @field_names != scalar @values) {
                # Костыль, пока не сделали нормальный парсинг csv в DYNSMART-417:
                # выкидываем те строки, в которых значений меньше по счёту, чем полей
                next;
            }
            my @temp = map { $field_names[$_].'='.($values[$_] || '') } (0 .. $#field_names);
            if ( @temp ) {
                my $res_string = join( "\t", @temp );
                print F_OUT "$res_string\n";
            }
        }
    }
    close F_OUT;
    close F_IN;
}

#####
# Проверка на валидность типа данных
#
sub is_data_type_valid {
    my ($self) = @_;
    return $self->proj->feed_data_source_factory->is_data_type_valid($self->{business_type}, $self->feed_data_type);
}

sub url_to_mirror_data {
    my ($self, $url, %params) = @_;

    $params{download_limit} //= 20;

    # инициализация кэша конфигом с исключениями
    $self->{_mirror_data} //= dclone($self->proj->options->{mirror_data});
    $self->{_task_domains_data} //= {};

    my $cache = $self->{_mirror_data};
    my $task_domains = $self->{_task_domains_data};

    my $url_domain = $self->proj->page($url)->domain;
    my ($orig_domain, $main_mirror, $orig_domain_id, $main_mirror_id);

    # получаем orig_domain
    if ( exists $cache->{domain_to_orig}{$url_domain} ) {
        # есть в кэше
        $orig_domain = $cache->{domain_to_orig}{$url_domain};
    }
    elsif ($params{download_limit} &&  scalar(keys %{ $task_domains } ) > $params{download_limit} ) {
        # нет в кэше, и превышен лимит на скачивание
        # если редиректов нет, просто берем его же
        my $can_keep = 1;
        for my $cache_domain ( keys %{ $task_domains } ) {
            if ( $cache_domain ne $cache->{domain_to_orig}{$cache_domain} ) {
                $can_keep = 0;
                last;
            }
        }
        if ($can_keep){
            $orig_domain = $url_domain;
        }
        else {
            # берем самый частотный
            my %count = ();
            my $max_count = 0;
            for my $cache_orig_domain ( sort values %{ $task_domains } ) {
                $count{$cache_orig_domain}++;
                if ( $count{$cache_orig_domain} > $max_count ) {
                    $max_count = $count{$cache_orig_domain};
                    $orig_domain = $cache_orig_domain;
                }
            }
        }
    }
    else {
        # идем в интернет
        my $orig_url = $url;
        my $redir_url = $self->proj->page($orig_url)->redir_location;
        if ( $self->proj->validate_url($redir_url) ) {
            $orig_url = $redir_url;
        }
        $orig_domain = $self->proj->page($orig_url)->domain;
    }
    $orig_domain ||= $url_domain;
    $cache->{domain_to_orig}{$url_domain} ||= $orig_domain;
    $task_domains->{$url_domain} ||= $orig_domain;

    if ( !$params{only_orig_domain} ) {
        # получаем main_mirror
        $cache->{orig_to_mirror}{$orig_domain} //= $self->proj->site($orig_domain)->main_mirror;
        $cache->{orig_to_mirror}{$orig_domain} ||= $orig_domain;
        $main_mirror = $cache->{orig_to_mirror}{$orig_domain};

        # получаем IDшники доменов
        for my $iter_domain ( $orig_domain, $main_mirror ) {
            $cache->{domain_to_id}{$iter_domain} //= $self->proj->site($iter_domain)->yabs_domain_id;
        }
        $orig_domain_id = $cache->{domain_to_id}{$orig_domain};
        $main_mirror_id = $cache->{domain_to_id}{$main_mirror};
    }

    my $res = {
        orig_domain => $orig_domain,
        main_mirror => $main_mirror,
        orig_domain_id => $orig_domain_id,
        main_mirror_id => $main_mirror_id,
    };

    $self->proj->logger->debug('append main mirror: ', $res);

    return $res;
}

#####
# Получить все данные для Директа
# Возвращает хеш
#
sub get_directinf {
    my ($self, $h_params) = @_;

    $self->{warnings} ||= [];

    my $res = {};
    $res->{debug_info} = {};

    $res->{file_data} = "TTTTTT"; #Директ не может не сохранять данные и они не могут быть нулевой длины
    my $categs_tskv = $self->categs_tskv_light;
    my $categs = [  map { { map { ($_->[0] || '' ) => ($_->[1] || '') } map { $_->[0] =~ s/.*\://; $_ } map {[ split('=', $_, 2) ]} split("\t", $_ ) } }  split "\n", $categs_tskv ];

    my %category_ids = qw();
    for my $index (0..@{$categs}-1) {
        my $id = $categs->[$index]->{id};
        $category_ids{$id} = 1;
    }

    #Страшный костыль, так как Директ не умеет отображать больше 2 уровней категорий - срезаем первый уровень, если он один
    if((@$categs > 2)&&( (grep {! $_->{parentId} } @$categs) == 1 )){ #Если только один корневой элемент, то убираем его
        my @fst = grep {! $_->{parentId} } @$categs; #Выбираем первый элемент
        $categs = [grep {$_->{parentId} } @$categs]; #Удаляем корневой элемент
        delete($_->{parentId}) for grep {$_->{parentId} == $fst[0]{id} } @$categs;
    }
    $res->{categs} = $categs;

    my $offers_tskv_file = $self->offers_tskv_light_file($h_params);

    $res->{errors} ||= [];
    $res->{warnings} ||= [];

    $res->{feed_type} = $self->feed_data_type;

    foreach (@{$self->{warnings}}) {
        push(@{$res->{warnings}}, {
            code => $_->{code},
            message => $_->{message},
            message_ru => $_->{message_ru},
        });
    }
    foreach (@{$self->{errors}}) {
        push(@{$res->{errors}}, {
            code => $_->{code},
            message => $_->{message},
            message_ru => $_->{message_ru},
        });
        last; # we take only first error
    }
    if (@{$res->{errors}}) {
        return $res;
    }

    unless ( $res->{feed_type} ) {
        push( @{$res->{errors}}, { 
            code => 1213, 
            message => "Can't detect feed data type: ".$self->feed_data_type_error.".",
            message_ru => "Невозможно определить тип фида: ".($self->can('feed_data_type_error_ru') ? $self->feed_data_type_error_ru : $self->feed_data_type_error),
        } );
        return $res;
    }

    unless ( $self->is_data_type_valid ){
        push( @{$res->{errors}}, { 
            code => 1220, 
            message => 'Business type `'.$self->{business_type}.'` is not corresponding with our detected feed type.',
            message_ru => 'Тип бизнеса `'.$self->{business_type}.'` не соотносится с детектированным типом фида.',
        } );
    }

    #my $offers = [  map { { map { ($_->[0] || '' ) => ($_->[1] || '') } map { $_->[0] =~ s/^[^:]+://; $_ } map {[ split('=', $_, 2) ]} split("\t", $_ ) } }  split "\n", $offers_tskv ];
    #for my $prm (qw{vendor categoryId }){
    #    $res->{$prm}{$_->{$prm}}++ for @$offers;
    #}
    #$res->{all_elements_amount} = @$offers;
    $self->log('yml2directinf stat BEG');
    my ($tskvmap) = $self->proj->fdm->get_mapping_by_feed_data_type($res->{feed_type});

    # Имя промаппленного идентификатора оффера. У всех типов фидов - это "OfferID", кроме Яндекс.Маркета - у него "id".
    my $offer_id_field = "OfferID";
    if ($res->{feed_type} eq "YandexMarket") {
        $offer_id_field = "id";
    }

    my $canonical_required_fields = { 
        'categoryId' => 1, 
        'vendor' => 1, 
        'url' => 1, 
        $offer_id_field => 1, 
        'price' => 1, 
        'shop-sku' => 1,
        'type' => 1,
        'vendor' => 1,
        'model' => 1,
    };

    my $hash_from_tskvmap = $self->proj->fdm->convert_tskvmap_to_hash($tskvmap, $canonical_required_fields);
    my @required_fields_mapping = keys %$hash_from_tskvmap;
    # Добавляем каноничные поля, для случаев, когда для требуемых полей не нужен маппинг
    push @required_fields_mapping, keys %$canonical_required_fields;
    my $grep_re = join('|',@required_fields_mapping);

    my $valid_offers_percent_to_discard_new_checks = 0.7;
    my $valid_offers_percent_to_show_warning = 0.99; # >1% of wrong offers triggers warnings

    my $discarded_warning_offers = 0;

    my $bad_categoryId_use = '';
    my $bad_price = '';
    my $bad_categoryId = '';
    my $bad_categoryId_count = 0;
    my $bad_offerId = '';
    my $bad_categoryId_use_count = 0;
    my $bad_offerId_count = 0;
    my $bad_price_count = 0;
    my $duplicate_offerId = '';
    my $duplicate_offerId_count = 0;
    my $input_offers_count = 0;
    my $bad_shop_sku = '';
    my $bad_shop_sku_count = 0;
    my $bad_url = '';
    my $bad_url_count = 0;
    my $bad_id_google_count = 0;
    my $no_vendor_model_field = '';
    my $no_vendor_model_field_count = 0;
    my $bad_no_categoryId_offerid = '';
    my $bad_no_categoryId_count = 0;
    my %offer_ids = ();
    file_result_sub($offers_tskv_file, sub {
        s/\n//;
        # сразу после сплита по табу просеиваем грепом только нужные поля, иначе расчет сильно замедляется
        my $raw_h = { map {($_->[0] || '') => ($_->[1] // '')} map {
            $_->[0] =~ s/^[^:]+://;
            $_->[0] =~ s/^\s+|\s+$//g;
            $_->[1] =~ s/^\s+|\s+$//g;
            $_
        } map {[ split('=', $_, 2) ]} grep {$_ =~ /^[^=]*($grep_re)=/i} split("\t", $_) };
        my $h = {};
        for my $source_field (keys %$raw_h) {
            my $key_origin_name = $self->get_origin_field_name($source_field);
            my $canonical_key = $hash_from_tskvmap->{$key_origin_name} || $key_origin_name;
            $h->{$canonical_key} ||= $raw_h->{$source_field} if exists($canonical_required_fields->{$canonical_key});
        }

        $input_offers_count += 1;

        my $bad_offer = 0;
        my $bad_offer_new_check = 0;

        if ($res->{feed_type} eq "YandexMarket") {
            if (!(defined $h->{categoryId})) {
                $bad_no_categoryId_offerid = $h->{$offer_id_field};
                $bad_no_categoryId_count += 1;
                $bad_offer_new_check = 1;
            }

            if (defined $h->{categoryId} && !$category_ids{$h->{categoryId}}) {
                $bad_categoryId_use = $h->{categoryId};
                $bad_categoryId_use_count += 1;
                $bad_offer_new_check = 1;
            }

            if ($offer_ids{$h->{$offer_id_field}} && $h->{$offer_id_field} ne "") {
                $duplicate_offerId_count += 1;
                $duplicate_offerId = $h->{$offer_id_field};
                $bad_offer_new_check = 1;
            } else {
                $offer_ids{$h->{$offer_id_field}} = 1;
            }

            if (defined $h->{price} && !is_price_valid($h->{price})) {
                $bad_price = $h->{price};
                $bad_price_count += 1;
                $bad_offer_new_check = 1;
            }

            if (defined $h->{categoryId} && $h->{categoryId} !~ m/^[1-9]\d*$/) {
                $bad_categoryId = $h->{categoryId};
                $bad_categoryId_count += 1;
                $bad_offer_new_check = 1;
            }

            if (defined $h->{'shop-sku'} && !is_offer_id_valid($h->{'shop-sku'})) {
                $bad_shop_sku = $h->{'shop-sku'};
                $bad_shop_sku_count += 1;
                $bad_offer_new_check = 1;
            }

            if (defined $h->{type} && $h->{type} eq "vendor.model" && !(defined $h->{vendor} && defined $h->{model})) {
                $no_vendor_model_field = $h->{$offer_id_field};
                $no_vendor_model_field_count += 1;
                $bad_offer_new_check = 1;
            }
        }

        for my $prm (qw{vendor categoryId}){
            $res->{$prm}{$h->{$prm}}++ if $h->{$prm}; #&& $res->{$prm}{$h->{$prm}};
        }
        $h->{categoryId} =~ s/(^\s+|\s+$)//g if $h->{categoryId};
        $bad_categoryId = $h->{categoryId} if (!$bad_categoryId && $h->{categoryId} && $h->{categoryId} !~ m/^\d+$/);
        if ($h->{$offer_id_field} && $res->{feed_type} eq "YandexMarket" && !is_offer_id_valid($h->{$offer_id_field})) {
            $bad_offerId = $h->{$offer_id_field};
            $bad_offerId_count += 1;
            $bad_offer_new_check = 1;
        }
        my $dmn = '';
        $h->{url} = 'http://'.$h->{url} unless $h->{url} =~ /^https?:\/\//i;

        # Не удалось выделить домен
        if(!$h->{url} || !$self->proj->validate_url($h->{url})) {
            $bad_url = $h->{url};
            $bad_url_count += 1;
            $bad_offer = 1;
        }

        # Проверяем коректность id-шника за исключением фидов "GoogleFlights" и "GoogleTravel" (на практике - очень редкие типы фидов).
        # Исключение сделано в силу непрямой логики маппинга для этих типов фидов, в связи с чем
        # не представляется возможным выпарсить значения идентификаторов офферов так же, как и для других типов фидов.
        # На стадии же генерации (в смартах) неккорретные офферы будут отсеиваться и для этих типов фидов тоже.
        if (($res->{feed_type} ne "GoogleFlights") && ($res->{feed_type} ne "GoogleTravel")) {
            if ((!defined $h->{$offer_id_field}) || ($h->{$offer_id_field} eq "")) {
                $bad_id_google_count += 1;
                $bad_offer = 1;
            }
        }

        # count bad offers according to the new checks to show warnings
        if ($bad_offer_new_check || $bad_offer) {
            $discarded_warning_offers += 1;
        }

        # Skip offer if it has bad properties
        if ($bad_offer || ($bad_offer_new_check && $self->{is_new_feed})) {
            return;
        }

        $dmn = $self->url_to_mirror_data($h->{url}, only_orig_domain => 1)->{orig_domain};
        $res->{domain}{$dmn}++;
        $res->{all_elements_amount}++;
    }, 1, $Utils::Common::options->{dirs}{banners_generation_files});
    $self->log('yml2directinf stat END');

    $res->{debug_info}->{discarded_offers_count} = $discarded_warning_offers;
    $res->{debug_info}->{input_offers_count} = $input_offers_count;

    my $add_error = !$res->{domain} || !$res->{all_elements_amount} || ($self->{is_new_feed} && $res->{all_elements_amount} < $input_offers_count * $valid_offers_percent_to_discard_new_checks); 

    # If there is no error, we will push these details to the warnings
    my $discarded_offers_details = $res->{warnings};

    if ($add_error) {
        push(@{$res->{errors}}, { 
            code => 1205, 
            message => "No valid offers found!", 
            message_ru => "Не найдено корректных офферов!", 
        });

        # But if there is an error, then we will push these details to the errors to draw it on frontend. Right now, unfortunately, frontend can't draw warnings
        $discarded_offers_details = $res->{errors};
    }

    my $show_warning = $input_offers_count - $discarded_warning_offers < $input_offers_count * $valid_offers_percent_to_show_warning;
    if ($add_error || $show_warning) {
        if ($bad_url_count) {
            push(@{$discarded_offers_details}, { 
                code => 1205, 
                message => "Bad URL: `$bad_url`. Count of bad URLs: $bad_url_count.", 
                message_ru => "Неверный URL: `$bad_url`. Количество неправильных URL'ов: $bad_url_count.", 
            });
        }
        if ($bad_id_google_count) {
            push(@{$discarded_offers_details}, { 
                code => 1205, 
                message => "Offer id isn't found. Count of offers where offer id wasn't found: $bad_id_google_count.", 
                message_ru => "Не найден Offer Id. Количество офферов с этой ошибкой: $bad_id_google_count.", 
            });
        }
        if ($bad_no_categoryId_count) {
            push(@{$discarded_offers_details}, { 
                code => 1205, 
                message => "Offer has no categoryId. Count of wrong offers: $bad_no_categoryId_count. Example offer ID without categoryId from this feed: `$bad_no_categoryId_offerid`. Offer must use categoryId from your list at the beginning of the file.", 
                message_ru => "В оффере отсуствует поле categoryId. Количество неверных офферов: $bad_no_categoryId_count. Пример ID оффера с отсутствующим полем categoryId: `$bad_no_categoryId_offerid`. Оффер должен использовать categoryId из вашего списка категорий в начале фида.", 
            });
        }
        if ($bad_categoryId_use_count) {
            push(@{$discarded_offers_details}, { 
                code => 1205, 
                message => "Offer uses non-existent categoryId. Count of wrong offers: $bad_categoryId_use_count. Example of non-existent categoryId from this feed: `$bad_categoryId_use`. Offer must use categoryId from your list at the beginning of the file.", 
                message_ru => "Оффер использует несуществующую categoryId. Количество неверных офферов: $bad_categoryId_use_count. Пример несуществующего categoryId: `$bad_categoryId_use`. Оффер должен использовать categoryId из вашего списка категорий в начале фида.", 
            });
        }
        if ($bad_offerId_count) {
            push(@{$discarded_offers_details}, { 
                code => 1205, 
                message => "Bad Offer Id. Count of wrong ids: $bad_offerId_count. Example of wrong id from this feed: `$bad_offerId`. Offer id can contain: digits, latin letters, cyrillic letters (without 'ё'), symbols from list: '.,_-=/\\()[]{}' (without apostrophes).", 
                message_ru => "Неверный Offer Id. Количество неверных ID: $bad_offerId_count. Пример неверного Offer Id из данного фида: `$bad_offerId`. Offer Id должен состоять из: цифр, латинских букв, русских букв (без 'ё'), символов из списка: '.,_-=/\\()[]{}' (без апострофов).", 
            });
        }
        if ($duplicate_offerId_count) {
            push(@{$discarded_offers_details}, { 
                code => 1205, 
                message => "Found $duplicate_offerId_count duplicates by Offer Id, their offers are discarded. Each offer must have unique Offer Id. Example of duplicated Offer Id: `$duplicate_offerId`.", 
                message_ru => "Обнаружено $duplicate_offerId_count дубликатов по Offer Id, все офферы с ними отвергнуты. Каждый оффер должен обладать уникальным Offer Id. Пример дубликата Offer Id: `$duplicate_offerId`.", 
            });
        }
        if ($bad_price_count) {
            push(@{$discarded_offers_details}, { 
                code => 1205, 
                message => "Bad price. Count of wrong prices: $bad_price_count. Example of wrong price from this feed: `$bad_price`. Price must be non-zero number without spaces, commas or currency symbol, only dot allowed to separate fraction and integer part of price.", 
                message_ru => "Неверная цена. Количество неверных цен: $bad_price_count. Пример неверной цены из данного фида: `$bad_price`. Цена должна быть ненулевым числом без пробелов, запятых, символов валюты; для разделения целой и дробной части цены должна использоваться точка.", 
            });
        }
        if ($bad_shop_sku_count) {
            push(@{$discarded_offers_details}, { 
                code => 1205, 
                message => "Bad shop-sku. Count of wrong ids: $bad_shop_sku_count. Example of wrong shop-sku from this feed: `$bad_shop_sku`. shop-sku can contain: digits, latin letters, cyrillic letters (without 'ё'), symbols from list: '.,_-=/\\()[]{}' (without apostrophes).", 
                message_ru => "Неправильный shop-sku. Количество данных ошибок: $bad_shop_sku_count. Пример неверного shop-sku из данного фида: `$bad_shop_sku`. shop-sku должен состоять из: цифр, латинских букв, русских букв (без 'ё'), символов из списка: '.,_-=/\\()[]{}' (без апострофов).", 
            });
        }
        if ($no_vendor_model_field_count) {
            push(@{$discarded_offers_details}, { 
                code => 1205, 
                message => "No `vendor` or `model` field found for `vendor.model` offer type in $no_vendor_model_field_count offers. Example of offer ID without any of these field: `$no_vendor_model_field`.", 
                message_ru => "Не найдено полей `vendor` или `model` в $no_vendor_model_field_count офферах типа `vendor.model`. Пример ID оффера, без одного из этих полей: `$no_vendor_model_field`.", 
            });
        }
    }
    if ($bad_categoryId_count) {
        push(@{$res->{errors}}, { 
            code => 1290, 
            message => "Bad category id: `$bad_categoryId`. Count of wrong categoryID: $bad_categoryId_count. Must be positive integer without spaces and leading zeros.", 
            message_ru => "Неправильный ID категории: `$bad_categoryId`. Количество неверных ID категорий: $bad_categoryId_count. ID категории должно быть целым положительным числом без пробелов и лидирующих нулей.", 
        });
    }

    return $res;
}

#####
# Загрузить все категории фида в файл
# Для костыльных типов фида ставится костыльная категория
#

sub categs_tskv_light_file {
    my ($self) = @_;

    return $self->{_categs_tskv_light_file} if $self->{_categs_tskv_light_file};

    my $file_res = $self->get_tempfile('categs_tskv_light', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
    $self->proj->log("categs_tskv_light_file file $file_res");
    if ($self->feed_data_type and (
        ($self->feed_data_type eq 'GoogleHotels')
        or ($self->feed_data_type eq 'GoogleFlights')
        or ($self->feed_data_type eq 'TravelBooking')
        or ($self->feed_data_type eq 'AutoRu')
        or ($self->feed_data_type eq 'GoogleMerchant')
        or ($self->feed_data_type eq 'YandexRealty')
    )
        or ($self->feed_file_type and $self->feed_file_type eq 'csv') # для csv тоже всегда генерим заглушку
    ){
        $self->proj->log("categs_tskv_light_file ".($self->feed_data_type));
        open(F, "> $file_res") || warn "$!";
        print F "category\tcategory:id=1000000001\tcategory:parentId=0\tcategory=Все\n";
        close F || warn "$!";
    }elsif( $self->feed_file_type && $self->feed_file_type =~ /^(?:yml|xml)$/ ){
        $self->proj->log("categs_tskv_light_file xml/yml");
        my $file = $self->page_text_file;
        unless (-e $file) {
            $self->proj->log("file: $file doesn't exist");
        }
        unless (-e $file_res) {
            $self->proj->log("file_res: $file_res doesn't exist");
        }
        my $error_or_warning = XMLParser::grep($file, $file_res, 'category', $self->{is_new_feed}, $self->feed_data_type);
        if ($error_or_warning->is_error()) {
            push(@{$self->{errors}}, { 
                code => 1204,
                message => 'XML Parser fatal error: '.$error_or_warning->message(),
                message_ru => 'Фатальная ошибка XML-парсера: '.decode('UTF-8', $error_or_warning->message_ru(), Encode::FB_CROAK),
            });
        }
    }

    $self->categs_tskv_to_canonical($file_res);
    if ($self->feed_file_type and $self->feed_file_type eq 'yml' && $self->proj->file( $file_res )->text !~ /category\:id/) {
        $self->{error} ||= 'Err1206: No categories found!';
        push(@{$self->{errors}}, { 
            code => 1206,
            message => "No categories found!", 
            message_ru => "Не найдено списка категорий!", 
        });
    }

    $self->log($self->{error}) if $self->{error}; #вывод в лог если вдруг парсер споткнулся

    $self->{_categs_tskv_light_file} = $file_res;
    return $file_res;
}

sub categs_tskv_to_canonical {
    my ($self, $file,) = @_;
    # lower case to canonical
    my $categs_canonical_format = {
        'category:id'       => 'category:id',
        'category:parentid' => 'category:parentId',
        'category' => 'category',
    };

    my $temp_file = $self->get_tempfile('categs_tskv_light.tmp', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
    open F_IN, "< $file" or die("Can't open file $file, error is $!");
    open F_OUT, "> $temp_file" or die("Can't open file $temp_file, error is $!");
    while (my $line = <F_IN>) {
        while (my ($src_key, $dst_key) = each %$categs_canonical_format) {
            $line =~ s/(^|\t+)$src_key=/$1$dst_key=/ig;
        }
        print F_OUT $line;
    }
    close F_IN;
    close F_OUT;
    $self->proj->do_sys_cmd("mv $temp_file $file");
}

#####
# То же, что и categs_tskv_light_file + вoзвращает ТЕКСТ
#
sub categs_tskv_light {
    my ($self) = @_;
    my $file = $self->categs_tskv_light_file;
    $self->proj->log("categs_tskv_light file $file");
    return $self->proj->file($file)->text;
}

####
# Отображает строку фида в хэш
# Маппинг + форматирование
#
sub _line2h {
    my ($self, $line) = @_;
    my $h_res = {};
    my $param_name = ''; # конструкцию вида param:name=Цвет param=синий в хеш пишем как { Цвет => синий }
    my %offer_params = ();
    my %h_units = ();
    my $is_category_tskv = 0;

    my $offer_line_md5 = eval { md5_hex(Encode::encode('UTF-8', $line)) } // '';

    # бывш. html_entities_decode
    $line = decode_entities($line);
    $line = join " ", (split /\n+/, $line); # не даём побиться одной строке ПОЧТИ tskv на несколько, если мы раскодировали символ переноса строки
    $line =~ s/\&\#039;/'/g;

    $line = $self->proj->phrase($line)->replace_html_spec->text;
    $line =~ s/^$self->{offer_tag}\t//i;
    $line =~ s/(^|\t)$self->{offer_tag}\:/$1/ig;

    $is_category_tskv = 1 if $line =~ /^category\s/;

    my $fdm = $self->proj->fdm;
    my $param_rename = $fdm->param_rename;

    my @kv;
    my @images;
    my $disable_status_key;
    my $disable_status_val;
    my @breadcrumbs;
    for my $el ( split/\t+/, $line ){
        next unless $el;
        my ($key, $value) = split(/\=/,$el,2);
        $key   =~ s/^\s+|\s+$//g if $key;
        if ($value) {
            $value =~ s/^\s+|\s+$//g;
            $value =~ s/\s{2,}/ /g;
        }
        push @kv, [ $key, $value ];

        # -> это разделитель составного поля в парсере
#        $key =~ s/\-\>/\-/g;
        $key = 'name' if $key eq 'offer->name';

        if ($key =~ /^offer:disable_status/) {
            $disable_status_key = $value if ($key eq 'offer:disable_status:key');
            $disable_status_val = $value if ($key eq 'offer:disable_status:value:flag');
            if ($disable_status_key and $disable_status_val) {
                my $PUSH_PARTNER_SITE_DISABLE_STATUS_ID = 24;
                $h_res->{site_disable_status} = $disable_status_val if ($disable_status_key == $PUSH_PARTNER_SITE_DISABLE_STATUS_ID);
                $disable_status_key = undef;
                $disable_status_val = undef;
            }
            next;
        }

        if ($key eq 'offer:breadcrumbs') {
            push @breadcrumbs, $value;
            next;
        }

        if (lc($key) =~ /param:name$/ or  lc($key) =~ /original_content:params:name$/) {
            $param_name = lc $value;
            $param_name =~ s/^\W|\W$//g;
            next;
        }
        if (lc($key) eq 'age:unit') {
            $h_res->{age_unit} = uc($value);
            next;
        }
        if ($param_name and (lc($key) eq 'param:unit' || lc($key) =~ /original_content:params:unit$/)) {
            my $renamed_param = ($param_rename->{lc($param_name)} || $param_name).":unit";
            if (!$is_category_tskv) {
                $renamed_param = $self->get_origin_field_name($renamed_param);
            }
            $h_res->{$renamed_param} = $value;
            $h_units{$param_name} = $value; # и в отдельный хеш
            next;
        }
        if ($param_name and (lc($key) eq 'param' || lc($key) =~ /original_content:params:value$/)) {
            my $renamed_param = $param_rename->{lc($param_name)} || $param_name;
            if (!$is_category_tskv) {
                $renamed_param = $self->get_origin_field_name($renamed_param);
            }
            $h_res->{$renamed_param} = $value;
            $offer_params{$param_name} = $value; # и в отдельный хеш
            $param_name = '';
            next;
        }
        $key = $param_rename->{lc($key)} || $key;
        if (!$is_category_tskv) {
            $key = $self->get_origin_field_name($key);
        }

        # фиксим урлы: дописываем в начало http://, если там этого нет (DYNSMART-291), и собираем массив картинок
        if (($key =~ /url|link/i) and $value or $fdm->is_image_field($key, $value)) {
            my @fixed_urls = map {fix_url_scheme($_)} get_urls_from_str($value);
            push @images, @fixed_urls if $fdm->is_image_field($key, $value);
            $value = join ' ,', @fixed_urls;
        }

        next if defined($h_res->{$key});
        $h_res->{$key} = $value;
    }

    $h_res->{categpath} = join(' / ', @breadcrumbs) if @breadcrumbs;

    $h_res->{offer_line_md5} //= $offer_line_md5;
    $h_res->{images} = \@images;

    %offer_params = %{$self->filter_bad_offer_params(\%offer_params)};
    # в одно поле сваливаем все парамсы в формате key:value,key:value
    if ( %offer_params && !$h_res->{params}) {
        $h_res->{params} = join( ',', map { "$_:$offer_params{$_}".(($h_units{$_} and $_ !~ /size|размер/i) ? " $h_units{$_}" : "") } sort keys %offer_params );
        # SUPBL-185: не дублируем в params то, что выносится в отдельные поля: материал и размер
        $h_res->{params_specformat} = join( '/;/', map { "$_: $offer_params{$_}".($h_units{$_} ? " $h_units{$_}" : "") } grep { $_ !~ /size|размер|материал|material/i } sort keys %offer_params );
    }
    # для правил маппинга добавляем значения к названиям полей
    if ( $self->{tskvmap} && !$is_category_tskv ) {
        $h_res->{tskvmap} = $fdm->add_values_to_map_hash( $self->{tskvmap}, $h_res );
    }
    if ( $self->{additional_data} && !$is_category_tskv ) {
        $h_res->{additional_data} = $fdm->add_values_to_map_hash( $self->{additional_data}, $h_res );
    }
    return $h_res;
}

sub filter_bad_offer_params {
    my ($self, $offer_params) = @_;
    my $MAX_PARAM_LEN = 60;   # https://st.yandex-team.ru/DYNSMART-1049
    my $bad_offer_params = $self->bad_offer_params;
    my %filtered_params = map {$_ => $offer_params->{$_}}
        grep {
            !exists $bad_offer_params->{lc($_)} && length($_) + length($offer_params->{$_}) <= $MAX_PARAM_LEN
        } keys %$offer_params;

    return \%filtered_params;
}

# параметры товара, которые мы не хотим отображать
sub bad_offer_params :CACHE {
    my ($self) = @_;
    return {
        'addeddate'     => 1,
        'article'       => 1,
        'campaign_id'   => 1,
        'crossborder'   => 1,
        'ebsmstock'     => 1,
        'ismarketplace' => 1,
        'offer type'    => 1,
        'producttype'   => 1,
        'season_autumn' => 1,
        'stockquantity' => 1,
        'ин'            => 1,
    };
}

#####
# Каждую строку TSKV в хэш => все вместе в массив
# Костыльная тема(
#
sub tskv2arr { # в массив хешей
    my ($self, $tskv_text ) = @_;
    return [] unless $tskv_text;
    return [ map { $self->_line2h($_) } map { s/^.*?\t// if $_!~/^.+\=.+?\t/; $_} ( split /\n+/, $tskv_text ) ]; ##no critic
}

#####
# Получить размер строки в байтах
#
sub _bytes_size {
    my ($self, $data) = @_;
    _utf8_off($data);
    my $sz = length($data);
    _utf8_on($data);
    return $sz;
}

sub get_origin_field_name {
    my ($self, $field_name) = @_;
    if ($self->origin_map && $self->origin_map->{lc($field_name)}) {
        return $self->origin_map->{lc($field_name)};
    }
    return $field_name;
}

1;
