package BM::BannersMaker::FeedDataSourceFactory;

use utf8;
use open ':utf8';

use std;
use base qw(ObjLib::ProjPart);

use Data::Dumper;

use Encode qw{ _utf8_on _utf8_off };

use URI::Escape;
use LWP::UserAgent;
use IPC::Open2;
use File::Copy;

use MKDoc::XML::Tokenizer;

use JSON qw{from_json};

use XMLParser;

#EXCEL парсер
use Utils::XLS qw(spreadsheet2arr);

use Utils::Array;
use Utils::Sys qw{split_csv_line unzipdata uncompressdata uncompressfile make_good_utf_string file_text_sub file_result_sub read_sys_cmd};

use BM::LangRecognize qw(recognize_text_lang);

__PACKAGE__->mk_accessors(qw(

));

# для определения типа фида требуем, чтобы в фиде были все перечисленные поля (это все поля определённых типов фидов)
my %full_feed_data_types_fields = (
    'TravelBooking' => ["Country", "Destination Name", "Facilities", "Final URL", "Image URL", "Property ID", "Property Name", "Property Type", "Rating"],
);

# для определения типа фида требуем, чтобы были следующие обязательные поля
my %necessary_feed_data_types_fields = (
    'YandexRealty' => ["location:locality-name", "location", "type", "url", "sales-agent", "sales-agent:organization"],#через двоеточие указываем вложенные теги
    'AutoRu' => ["unique_id|vin", "url"],  # обязательное или vin, или unique_id
    'GoogleCustom' => ["ID", "Final URL"],
    'GoogleTravel' => ["Destination ID", "Final URL", "Title"],
    'GoogleFlights' => ["Destination ID", "Destination name", "Final URL"],
    'GoogleHotels' => ["Property ID", "Property name", "Final URL", "Destination name"],
    'DatacampYandexMarket' => ["offer:original_content:url", "offer:offer_id"],
    'YandexMarket' => ["categoryId", "url", "vendor", "model"], # существует 2 доступных спецификации типа YandexMarket
    'YandexMarket_light' => ["categoryId", "url", "name"], # # существует 2 доступных спецификации типа YandexMarket
    'GoogleMerchant' => ["g:id|id","g:image_link|image_link","g:link|link","g:title|title"],
    'YandexCustom' => ["ID", "URL"],
);

#необязательные поля, уникальные для того типа фида, в котором используются - в других типах фидов их не должно быть (хотя бы не должно быть совпадающих комбинаций с обязательными полями)
#используются, если по обязательным полям найдено несколько совпадений
my %unique_feed_data_types_fields = (
    'AutoRu' => ["folder_id", "mark_id"],
    'GoogleCustom' => ["Item title"],
    'GoogleTravel' => ["Destination address"],
    'GoogleFlights' => ["Flight price"],
    'YandexCustom' => ["Image", "Title"],
);

my %feed_data_type_translation = (
    'YandexMarket' => 'Yandex.Market',
    'YandexMarket_light' => 'Yandex.Market',
    'GoogleHotels' => 'AdWords “Hotels and rentals”',
    'AutoRu' => 'Auto.ru',
    'YandexRealty' => 'Yandex.Realty',
    'GoogleFlights' => 'AdWords “Flights”',
    'GoogleCustom' => 'AdWords “Custom”',
    'YandexCustom' => 'Universal feed',
    'GoogleTravel' => 'AdWords “Travel”',
);

my %h_business_type2feed_data_type = (
    retail => ['YandexMarket', 'GoogleMerchant'],
    realty => ['YandexRealty'],
    hotels => ['GoogleHotels', 'TravelBooking', ],
    auto => ['AutoRu'],
    flights => ['GoogleFlights'],
    other => ['YandexMarket', 'GoogleCustom', 'YandexCustom', 'GoogleTravel', 'GoogleMerchant'],
);


#поля для "прокидывания" в fds
my @fields_for_fds = qw/ offer_tag error errors offers __page_text_file
                        __next_data_pack_cache extfile srcfilesize tskvmap additional_data required_fields
			            feed_data_type feed_file_type page login pass download_timeout max_file_size max_file_size_type bad_flags business_id shop_id last_valid_feed_type datacamp_orig_feedurl/;

my @all_class_fields = (@fields_for_fds, qw/__page_raw_text_file __unpack_xlsx_default_status __unpack_xls_default_status/);

sub in_array_re {
  my ($el, $ar) = @_;

  return 0 unless defined $el;

  for my $a (@$ar) {
    my $re = '^'.$el.'$';
    return 1 if $a =~ /$re/;
  }
  return 0;
}

sub array_minus_re {
    my $a1 = shift;
    my $a2 = shift;

    return [ grep { !in_array_re($_, $a2) } @$a1 ];
}

sub array_intersection_re {
    my $a1 = shift;
    my $a2 = shift;

    return [ grep { in_array_re($_, $a2) } @$a1 ];
}

sub is_data_type_valid {
    my ($self, $business_type, $feed_data_type) = @_;

    if ( $business_type ) {
        return 0 unless ($h_business_type2feed_data_type{$business_type});
        return 1 if ($feed_data_type ~~ @{$h_business_type2feed_data_type{$business_type}});
        return 0;
    }

    return 1;
}

sub get_feed_data_type_by_fields {
    my ($self, $fields) = @_;
    $fields = [map {lc $_} grep { $_ !~ /xml/i } @$fields];
    my $sorted = [sort @$fields];
    #print STDERR "All fields:\n", Dumper($sorted);

    my %feed_data_type_missing_fields = ();
    my %feed_data_type_extra_fields = ();

    #это хэш, так как есть разные комплекты полей для одних и тех же типов фидов, и возможны повторы
    my %candidates = ();

    #ищем в специальных типах урлов по полному совпадению
    my @quoted_fields = map { quotemeta $_ } @$fields;
    for my $type (keys %full_feed_data_types_fields) {
        my $full_fields = [map {lc $_} @{$full_feed_data_types_fields{$type}}];
        my $missing = array_minus_re($full_fields, $fields);
        $feed_data_type_missing_fields{$type} = $missing;
        $feed_data_type_extra_fields{$type} = array_minus_re(\@quoted_fields, $full_fields);
        if ((scalar @$missing == 0) and (scalar @{$feed_data_type_extra_fields{$type}} < 4)) {
            $candidates{$type} = 1;
        }
    }
    return ( (keys %candidates)[0], '', '') if scalar(keys %candidates) == 1;

    #ищем в общих типах фидов по обязательным параметрам
    if ( !%candidates ) {
        for my $type (keys %necessary_feed_data_types_fields) {
            my $necessary_fields = [map {lc $_} @{$necessary_feed_data_types_fields{$type}}];
            my $found = array_intersection_re($necessary_fields, $fields);
            $feed_data_type_missing_fields{$type} = array_minus_re($necessary_fields, $fields);
            if (scalar @$found == scalar @$necessary_fields) {
                #здесь нужно переименовывать те типы, у которых есть несколько разных комплектов валидных полей
                #в исходном маппинге они написаны с суффиксами
                $type = 'YandexMarket' if ($type =~ /^YandexMarket/);
                $candidates{$type} = 1;
            }
        }
    }

    if ( scalar(keys %candidates) > 1 ) {
        #если найдено больше одного кандидата, пробуем пофильтровать их по дополнительным полям
        my @filtered_candidates = ();
        for my $type (keys %candidates) {
            my $unique_fields = [map {lc $_} @{$unique_feed_data_types_fields{$type}}];
            my $found = array_intersection_re($unique_fields, $fields);
            # Не записываем ничего в feed_data_type_missing_fields, так как это не обязательные поля, и по их отсутствию нельзя делать вывод о том, что мы кого-то не угадали
            # Не теряем те типы, у которых не прописаны уникальные поля
            if (scalar @$found || scalar @$unique_fields == 0) {
                push @filtered_candidates, $type;
            }
        }
        #если нет совпадений, не дропаем список: может отфильтроваться далее
        %candidates = map {$_ => 1} @filtered_candidates if scalar(@filtered_candidates);
    }

    if ( scalar(keys %candidates) > 1 ) {
        #если кандидатов все еще больше одного, отбираем валидные по business_type
        if ( $self->{business_type} ) {
            my @filtered_candidates = ();
            for my $type ( keys %candidates ) {
                push @filtered_candidates, $type if $self->is_data_type_valid( $self->{business_type}, $type );
            }
            #если нет совпадений, не дропаем список, чтобы отдать информативную ошибку
            %candidates = map {$_ => 1} @filtered_candidates if scalar(@filtered_candidates);
        }
    }

    return ( (keys %candidates)[0], '', '') if scalar(keys %candidates) == 1;

    if ( scalar(keys %candidates) > 1 ) {
        return ('', "Can't detect possible feed type, too many candidates: " . join(',', keys %candidates).".", "Невозможно определить возможный тип фида, слишком много кандидатов: " . join(',', keys %candidates).".");
    }

    # по количеству недостающих/лишних полей пытаемся предположить, какой это мог бы быть тип
    my $nearest_possible_type = (sort { scalar @{$feed_data_type_missing_fields{$a}} <=> scalar @{$feed_data_type_missing_fields{$b}} } keys %feed_data_type_missing_fields)[0];
    my $nearest_possible_type_distance = scalar @{$feed_data_type_missing_fields{$nearest_possible_type}};

    return ('', "Can't detect possible feed type, too many fields are missing.", "Невозможно определить возможный тип фида, отсутствует слишком много полей.") if ($nearest_possible_type_distance > 1); # считаем, что забыть можно не более одного поля

    my $msg = '';
    my $msg_ru = '';
    my @possible_types = grep { scalar @{$feed_data_type_missing_fields{$_}} < 2 } keys %feed_data_type_missing_fields;

    my @err = ();
    push @err,  map { "'".($feed_data_type_translation{$_})."' (missing ".(join ", ", map { my $r = $_; $r =~ s/\|/ or /; "'$r'" } @{$feed_data_type_missing_fields{$_}} ).")" }
                grep { scalar @{$feed_data_type_missing_fields{$_}} && $feed_data_type_translation{$_} } @possible_types;
    push @err, map { "'$_' (too many extra fields)" } map { $feed_data_type_translation{$_}} grep { $feed_data_type_extra_fields{$_} && $feed_data_type_translation{$_}  } @possible_types;

    my @err_ru = ();
    push @err_ru,  map { "'".($feed_data_type_translation{$_})."' (отсутствуют поля ".(join ", ", map { my $r = $_; $r =~ s/\|/ или /; "'$r'" } @{$feed_data_type_missing_fields{$_}} ).")" }
                grep { scalar @{$feed_data_type_missing_fields{$_}} && $feed_data_type_translation{$_} } @possible_types;
    push @err_ru, map { "'$_' (слишком много дополнительных полей)" } map { $feed_data_type_translation{$_}} grep { $feed_data_type_extra_fields{$_} && $feed_data_type_translation{$_}  } @possible_types;

    if (@err) {
        if (int(@err) == 1) {
            $msg = "Can't detect feed type, possible type: ".(join " or ", @err).".";
            $msg_ru = "Невозможно определить тип фида, возможный тип: ".(join " или ", @err_ru).".";
        } else {
            $msg = "Can't detect feed type, possible types: ".(join " or ", @err).".";
            $msg_ru = "Невозможно определить тип фида, возможные типы: ".(join " или ", @err_ru).".";
        }
    } else {
        $msg = "Can't detect possible feed type.";
        $msg_ru = "Невозможно определить возможный тип фида.";
    }

    return ('', $msg, $msg_ru);
}

sub init {
    my ($self) = @_;
}

sub clear_data {
    my ($self) = @_;
    $self->{page} = undef;

    # удаляем временный файл с сырым фидом
    unlink($self->{$_}) for (grep {$self->{$_} && -f $self->{$_} && !($self->{extfile} && ($self->{$_} eq $self->{extfile}))} qw/__page_raw_text_file/);

    delete $self->{$_} for grep{ exists $self->{$_} } ( @all_class_fields );
}

#####
# Создание нового FeedDataSourceXXX по входным параметрам
# Определяется тип и вся вспомогательная информация для фида, в тч содержимое
# Содержимое на выходе уже в нужной кодировке
#
sub new_fds {
    my ($self, $h_par) = @_;

    my $debug_mode = 0;

    #clear all data from previous fds-class
    $self->clear_data;

    #save input parameters
    $self->{$_} = $h_par->{$_} for (keys $h_par);

    #initilize page
    $self->{page} = $self->proj->page({
        url => $h_par->{datacamp_orig_feedurl} // $h_par->{url},
        assert_content_length => 1,
        name => $h_par->{name},
    });

    if (defined $self->{page} && $self->{login}) {
        $self->{page}->{$_} = $self->{$_} for qw{ login pass };
    }

    #fill all caches
    $self->prepare_all_data;

    $h_par->{feed_file_type} = $self->feed_file_type;
    $h_par->{feed_data_type} = $self->feed_data_type;
    $h_par->{feed_data_type_error} = $self->{feed_data_type_error};
    $h_par->{feed_data_type_error_ru} = $self->{feed_data_type_error_ru};
    $h_par->{srcfilesize} = $self->{srcfilesize};
    my $feed_data_type = $h_par->{feed_data_type};
    $h_par->{feed_lang} = $self->feed_lang;

    #save feed_data to hash
    $h_par->{$_} = $self->{$_} for ( grep { !$h_par->{$_} } @fields_for_fds);
    $h_par->{csv_delim} = $self->{csv_delim} if $self->{csv_delim};

    if ( $debug_mode ) {
        #logic for debug
        $h_par->{proj} = 'hidden for debug';
        my $tmp_deb1 = $h_par->{proj_proxy_ref} || '';
        my $tmp_deb2 = $h_par->{page} || '';
        $h_par->{proj_proxy_ref} = 'hidden for debug';
        $h_par->{page} = 'hidden for debug';

        print STDERR Dumper $h_par;

        $h_par->{page} = $tmp_deb2;
        $h_par->{proj_proxy_ref} = $tmp_deb1;
        #end of logic for debug
    }

    my $fdm = $self->proj->fdm;

    #define FDS (or mapping)
    #save reference for proj for bless(...) in base raw class (Object.pm)
    $h_par->{proj} = $self->proj;
    if ($feed_data_type) {
        if ( ($feed_data_type ne 'bl_inner_data') and ($feed_data_type ne 'dse_data') ) {
            ($h_par->{tskvmap}, $h_par->{additional_data}, $h_par->{required_fields}, $h_par->{offer_tag}) = $fdm->get_mapping_by_feed_data_type($feed_data_type);
        }
        $h_par->{origin_map} = $fdm->get_origin_mapping_h($feed_data_type);
    }

    #default FDS
    return BM::BannersMaker::FeedDataSource->new( $h_par );
}

#####
# Разогреть кэши и загрузить фид
#
sub prepare_all_data {
    my ($self) = @_;

    #download page
    $self->_get_raw_page_text_file;

    #fill cache
    $self->feed_file_type;
    $self->feed_data_type;
}

#####
# Получить первые или рандомные строки фида
#   lines - количество строк
#   shuf - перемешивать ли порядок строк (shuf = 0 - взять первые строки)
#   mode = режим чтения файла
#   tail - отдать конечные строки (по умолчанию отдает начальные)
sub _get_text {
    my ($self, %par) = @_;

    $par{lines} ||= 10_000;
    $par{shuf} //= 0;
    $par{tail} //= 0;
    my $max_length = 3_000_000;

    # Оптимизация, чтобы обойтись без загрузки файла в оперативную память
    my $file = $self->_get_page_text_file;
    my $text = '';

    my $tmp_bmfile;

    if ($par{shuf}) {
        my $tempfile = $self->get_tempfile('shuf_feed', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
        $self->proj->do_sys_cmd("cp $file $tempfile");
        $tmp_bmfile = $self->proj->file($tempfile)->shuf(2500000);
    }
    else {
        $tmp_bmfile = $self->proj->file($file);
    }

    if ( $tmp_bmfile ) {
        if ( $par{tail} ) {
            $text = $tmp_bmfile->tail_lines_bytelimit($par{lines}, $max_length);
        }
        else {
            $text = $tmp_bmfile->head_lines_bytelimit($par{lines}, $max_length);
        }
    }
    $text = $self->proj->detect_charset->text2utf8($text);

    return make_good_utf_string($text);
}

sub unpack_excel_file {
    my ($self, $input_file, $format) = @_;
    $self->proj->log("unpack_excel_file: $format");
    die("'input_file' param not passed") unless $input_file;
    die("wrong format '$format' passed! xls|xlsx supported") if (!$format || ($format ne 'xls' && $format ne 'xlsx'));

    #return $self->{"__unpack_${format}_default_status_${input_file}"} if defined( $self->{"__unpack_${format}_default_status_${input_file}"} );
    my $file = $input_file;
    my $res = 0;

    my $size = $self->proj->file($file)->size;
    return $res unless($size);

    my $excel_parser_res = spreadsheet2arr($file, $format);
    if (defined($excel_parser_res) && @$excel_parser_res) {
        # да, это excel-файл, теперь обновляем файл фида-источника
        open F, '>' . $self->_get_page_text_file;
        print F join("\n", @$excel_parser_res);
        close F;
        $res = 1;
    }
    # кэшируем статус вызова по умолчанию
    #$self->{"__unpack_${format}_default_status_${$input_file}"} = $res unless $input_file;

    return $res;
}

#####
# Получить тип файла фида по первым строкам файла
# output: { yml, csv, xml, offers_tskv, xls, xlsx }
#
sub feed_file_type {
    my ($self) = @_;
    return $self->{feed_file_type} if defined $self->{feed_file_type};

    my $texthead = $self->_get_text;
    my ($firstline, $secondline) = ('', '');
    ($firstline, $secondline) = ( $1, $2 ) if ( $texthead =~ /^([^\n]+?)\n([^\n]+?)(\n|$)/ );

    my $is_xls = 0;
    my $is_text = read_sys_cmd("file -b " . $self->_get_raw_page_text_file) =~ m/\btext\b/;  # да, это заявлено в `man file`: в выводимих им описаниях не-бинарных файлов будет слово text

    my $pagefile_extension = '';
    if ($self->{page}) {
        my $url = $self->{page}->url;
        if ( $url =~ /((\.[^.\s]+)+)$/ ) {
            $pagefile_extension = $1;
        }
    }

    if (!$is_text && $pagefile_extension !~ /\.(xml|yml)\.(gz|zip)$/) {
        # видимо, получили архив либо бинарник
        # пробуем распарсить его как XLS и XLS/XLSX в архиве
        if ($self->unpack_excel_file( $self->_get_raw_page_text_file, 'xls' ) ||
            $self->unpack_excel_file( $self->_get_page_text_file, 'xls' ) ||
            $self->unpack_excel_file( $self->_get_page_text_file, 'xlsx' )
        ) {
            $is_xls = 1;
        }
    }

    my $res;
    if ($texthead !~ /^<!DOCTYPE html>/) {
        if ($is_xls) {
            $res = 'xls/xlsx';
            $self->{csv_delim} = "\t";
        }
        elsif ( $texthead =~ /^.{0,300}<yml_catalog/ms ) {
            $res = 'yml';
        }
        elsif ( $texthead =~ /^.{0,300}<\?xml version=/ms
                || $texthead =~ /<rss [^>]*? version=/ms
                || $texthead =~ /<products><product>/ms # костыль для турецкого фида criteo, очень плохо (
        ){
            # проверяем, вдруг XLSX? (и мы видим его метаинформация в формате xml)
            if ($self->unpack_excel_file($self->_get_raw_page_text_file, 'xlsx')) {
                $res = 'xls/xlsx';
                $self->{csv_delim} = "\t";
            } else {
                $res = 'xml';
            }
        }
        elsif ( $texthead =~ /^.{0,3000}(?:[a-zA-Z]+\=.*?\t)/ ) {
            $res = 'offers_tskv';
        }
        else {
            # определяем разделитель, как наиболее часто встречающийся символ
            my %hdelim = ();
            for my $delim (",", ";", "\t", "|") {

                $hdelim{$delim} = scalar split_csv_line($firstline, $delim);
            }
            $self->{csv_delim} = (sort { $hdelim{$b} <=> $hdelim{$a} } keys %hdelim)[0];

            my @fl_parts = split_csv_line($firstline, $self->{csv_delim});
            my @sl_parts = split_csv_line($secondline, $self->{csv_delim});

            $res = 'csv' if ( @fl_parts && @sl_parts && scalar(@fl_parts) == scalar(@sl_parts) && scalar(@fl_parts) > 1 );
        }
    }

    unless ($res) {
        $self->{error} ||= "Err1212: Can\'t detect feed file type!";
        push(@{$self->{errors}}, {
            code => 1212,
            message => 'Can\'t detect feed file type!',
            message_ru => 'Невозможно определить тип файла фида!',
        });
    }

    $self->proj->log("feed_file_type: ".($res || 'not defined'));
    $self->{feed_file_type} = $res;
    return $self->{feed_file_type};
}

# были проблемы с вложенными  тегами, поэтому сейчас делаем следующим образом:
#1) берем на вход строку
#2) выпарсиваем теги из xml
#3) генерируем на основе массива тегов массив эвентов открытия и закрытия тега
#4) для каждого эвента:
#4а) запоминаем его вложенность в уже открытые ранее теги
#4б) считаем, сколько раз он открывался
#4в) считаем, сколько ниже него находится уникальных тегов
#5) из полученных кандидатов выбираем такой, у которого больше всех произведение частоты на
#количество уникальных детей. Физический смысл: находим тег, аккумулирующий внутри себя наибольшее
#количество информации. Для всех имеющихся в природе на данный момент фидов это будет оффер.
#6) получаем на выходе тег оффера, а также список его детей (причем с сохранением иерархии), который
#потом удобно распарсить и проверить на соответствие какому-нибудь типу фида.
sub get_tags {
    my ($self, $fields, $h_feedtype2offer_tag) = @_;
    my @events_start_end_array = (); # массив эвентов открытия и закрытия тега
    my $frequency = {};
    my @tags_array = ();
    my $begin_offer = 0;
    my $key_tag = {};
    my $kv = {};
    my @offer_tag_array = values %$h_feedtype2offer_tag; # массив возможных начальных тегов
    for my $field ( @$fields ) {
        my $strip_field = $field;
        $strip_field =~ s/(^\/|\/$)//g; #strip_field это тег с вырезанными угловыми скобочками и слешами
        if (in_array($strip_field, \@offer_tag_array)) {
            $begin_offer = 1;
        }
        #будем сохранять события только после первого встретившегося тега из @offer_tag_array
        #нужно - чтобы в стек не попадали теги из начала документа, которые, во-первых,
        #замедляют расчет, во-вторых, могут привести к тому, что в топ по критерию попадет корневой тег, а не то, что нужно
        if ($begin_offer) {
        #1 - start, 0 - end
            if ($field =~ /^\//) {
                push @events_start_end_array, [ $strip_field, 0];
            } elsif ($field =~ /\/$/) {
                push @events_start_end_array, [ $strip_field, 1];
                push @events_start_end_array, [ $strip_field, 0];
            } else {
                push @events_start_end_array, [ $strip_field, 1];
            }
        }
    }
    for my $event ( @events_start_end_array ) {
        if ( $event->[1] ) {
        # если это тег начала, то добавляем этот тег в массив
            push @tags_array, $event->[0];
            # идем по всем тегам из этого массива и сохраняем информацию о том, что было в этом массиве до этого тега и после
            for my $index_tag (0..$#tags_array-1) {
                my $befor_tag = join(':', @tags_array[0..$index_tag] );
                my $after_tag = join(':', @tags_array[$index_tag+1..$#tags_array] );
                $kv->{$befor_tag}{$after_tag} = 1;
            }
            my $freq_key = join(':',  @tags_array );
            $frequency->{$freq_key}++;
            $key_tag->{$freq_key} //= $event->[0];
        } elsif (@tags_array) {
        #если это тег закрытия, выкидываем из массива все, пока не выкинем этот тег (или не выкинем все). На случай невалидного xml, чтобы массив не вырос
            my $popped = pop @tags_array;
            while ($popped ne $event->[0] && @tags_array) {
                $popped = pop @tags_array;
            }
        }
    }

    my $information = { map { $_ => scalar(keys %{ $kv->{$_} }) } keys %$frequency };

    my $max_key = undef;
    my $max_crit = -1;

    my $non_offer_keys = {
        'promos:promo' => 1,
    };
    for my $key ( keys %$frequency ) {
        next if ($non_offer_keys->{lc($key)});
        my $crit = $frequency->{$key} * $information->{$key};
        if ($crit > $max_crit) {
            $max_key = $key;
            $max_crit = $crit;
        }
    }
    my @fields = ();
    if (defined $max_key) {
        @fields = keys %{$kv->{$max_key}};
    }
    return \@fields;
}


#####
# Получить тип фида + offer_tag по тексту
# output: {type of Product}
#
# !эвристический! и ленивый
#
sub feed_data_type {
    my ($self) = @_;

    return $self->{feed_data_type} if $self->{feed_data_type};

    ($self->{feed_data_type}, $self->{feed_data_type_error}, $self->{feed_data_type_error_ru}) = ('', "Can't define feed type.", "Невозможно определить тип фида.");
    my $file_type = $self->feed_file_type;

    unless ($file_type) {
        $self->{feed_data_type_error} = "Can't define feed_data_type because feed_file_type is undefined";
        return $self->{feed_data_type};
    }

    # 1) пытаемся распознать тип по 10000 первым строкам
    # 2) пытаемся распознать тип по 10000 последним строкам - нужно для xml/yml-фидов, у которых в начале описание магазина и категории
    # 3, 4) то же самое, что и первые две, но с увеличенным числом строк в 40000
    my %get_text_params = (
        0 => {},
        1 => { tail => 1 },
        2 => { lines => 40000 },
        3 => { lines => 40000, tail => 1 },

    );

    my %h_feedtype2offer_tag = (
                'YandexRealty'   => 'offer',
                'AutoRu'         => 'car',
                'TravelBooking'  => 'listing',
                'GoogleMerchant' => 'item',
    );
    my $xml_tags_indexes = {
        'offers'     => undef,
        'categories' => undef,
        'currencies'  => undef,
    };
    for my $step (0..3) {
        next if (($step > 0) and ($file_type ~~ /^(csv|xls)/i)); # для csv/xls/xlsx фидов бесполезно брать другое количество строк/перемешивать их

        my $text = $self->_get_text(%{$get_text_params{$step}});
        my $firstline = '';
        if ( $text =~ /^([^\n]+?)\n/ ) {
            $firstline = $1;
        } else {
            $firstline = $text;
        }
        next unless ($firstline);

        my @fields = ();
        $self->{feed_data_type} = '';
        $self->{offer_tag} = '';
        if ($file_type eq 'offers_tskv'){
            if ($firstline =~ /(^|\t)feed_data_type=([^\t]+)(\t|^)/) {
                # иногда в tskv уже лежит поле feed_data_type, просто берём его оттуда
                ($self->{feed_data_type},$self->{feed_data_type_error}) = ($2, '');
                last;
            }
            @fields = map { my @r = split "=", $_, 2; $r[0]; } split /\t/, $firstline;
        } elsif ($file_type eq 'csv' || $file_type eq 'xls/xlsx'){
            # принцип одинаковый для csv и xls/xlsx
            my $del = quotemeta($self->{csv_delim} // ',');
            $del = '\s*'.$del.'\s*';
            @fields = map { my $t = $_; $t =~ s/^\N{U+FEFF}|\N{U+FEFF}$|\r//g; $t =~ s/^"|"$//g; $t; } split(/$del/, $firstline);
        } else {
            # это xml
            #достаем токенайзером валидные теги. Метод graceful достает все теги до первого несоответствия своим внутренним проверкам.
            my @tags = map {$_->as_string} grep {defined $_->tag} @{MKDoc::XML::Tokenizer->process_data_graceful($text)};
            #убираем угловые скобки
            for my $tag (@tags) {
                $tag =~ s/(^<|>$)//g;
            }
            #фильтруем комменты, убираем параметры у тегов и пробелы по краям
            @tags = grep {$_ !~ /!/} @tags;

            if (!$get_text_params{$step}->{tail}) {
                for my $tag (keys %$xml_tags_indexes) {
                    my ($tag_index) = grep {$tags[$_] eq $tag} (0 .. scalar(@tags) - 1);
                    $xml_tags_indexes->{$tag} //= $tag_index;
                }
            }
            @fields = ();
            for my $tag (@tags){
                $tag =~ s/^(\/?).*?(\S+).*?(\/?)$/$1$2$3/;
                $tag =~ s/^\s+|\s+$|\r|\N{U+FEFF}//g;
                push @fields, $tag;
            }
            my $fields;
            $fields = $self->get_tags(\@fields, \%h_feedtype2offer_tag);
            @fields = @{$fields};
         }

        ($self->{feed_data_type}, $self->{feed_data_type_error}, $self->{feed_data_type_error_ru}) = $self->get_feed_data_type_by_fields(\@fields);
        if (($file_type eq 'offers_tskv') and (not $self->{feed_data_type})) {
            if (scalar @{array_intersection_re(['categpath', 'url', 'name'], \@fields)} == 3) {
                $self->{feed_data_type} = 'bl_inner_data';
                $self->{feed_data_type_error} = '';
            } elsif (scalar @{array_intersection_re(['bl_title', 'bl_phrases', 'product_type', 'url'], \@fields)} == 4) {
                $self->{feed_data_type} = 'dse_data';
                $self->{feed_data_type_error} = '';
            }
        }
        last if ($self->{feed_data_type});
    }
    if ($self->{feed_data_type} eq 'YandexMarket' && ($self->{feed_file_type} eq 'xml' || $self->{feed_file_type} eq 'yml') && $self->{is_new_feed}) {
        my $tag = 'categories';
        if (!$self->check_xml_tag_before_offers($xml_tags_indexes, $tag)) {
            $self->{feed_data_type_error} ||= "Err1204: Wrong order of tags: offers found before $tag";
            $self->proj->logger->error("Wrong order of tags: offers found before $tag");
            $self->{feed_data_type_error_structured} = {
                code => 1204,
                message => "Wrong order of tags: `offers` found before `$tag`.",
                message_ru => "Неправильный порядок тегов: `offers` находится раньше `$tag`.",
            };
        }
    }
    $self->proj->log("feed_data_type: ".$self->{feed_data_type});
    $self->{offer_tag} = $h_feedtype2offer_tag{$self->{feed_data_type}} || '';

    return $self->{feed_data_type};
}

sub check_xml_tag_before_offers {
    my ($self, $tags_indexes, $tag_name) = @_;
    my $offers_index = $tags_indexes->{offers} // -1;
    my $tag_index = $tags_indexes->{$tag_name} // -1;
    return ($tag_index > -1 && ($tag_index < $offers_index || $offers_index < 0));
}

sub feed_lang {
    my $self = shift;
    my $file = $self->_get_page_text_file;
    my $language = recognize_text_lang($self->_get_text( 'tail' => 1 ));
    $language ||= 'unknown';
    $self->proj->log("feed_lang: $language");
    return $language;
}

#####
# Распаковка и декодирование файла
# Делаем распаковку через файлы, так как иначе забивается память
#
# !удаляем исходный файл после распаковки!
#
sub _unpack_and_decode_file {
    my ($self, $file) = @_;

    my $file_decoded = $self->get_tempfile('next_data_pack_charset', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
    my $file_unzipped = $self->get_tempfile('next_data_pack_zip2text_unzip', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
    eval {
        uncompressfile($file, $file_unzipped);

        my @charset_candidates = $self->proj->detect_charset->detect_file_charset_candidates($file_unzipped, 'text/xml'); #text/xml влияет только на включение логики выпарсивания чарсета из начала файла

        my $log_charset = join(',', @charset_candidates) || 'undefined';
        $self->proj->log("_unpack_and_decode_file detect_charset $log_charset");

        # Не перекодируем файлы формата UTF-8, чтобы сломанные символы не кодировались как \xHHHH, чтобы эта ошибка ловилась дальше в XMLParser'е и о ней сообщалось клиенту.
        my $all_candidates_are_utf8 = 1;
        foreach my $i (@charset_candidates) {
            if (lc($i) ne "utf-8") {
                $all_candidates_are_utf8 = 0;
            }
        }

        # декодируем
        if ($all_candidates_are_utf8 || $self->{dont_del_comments} ) {
            copy($file_unzipped, $file_decoded);
        } else {
            $self->proj->log("_unpack_and_decode_file del comments beg");
            $self->proj->detect_charset->file_charsets2utf8($file_unzipped, $file_decoded, \@charset_candidates);
            $self->proj->log("_unpack_and_decode_file del comments end");
            my $filesize = -s $file_decoded;
            $self->proj->log("_unpack_and_decode_file filesize $filesize");
        }
    };
    if ( $@ ) {
        $self->proj->log("_unpack_and_decode_file error: $@");
    }
    #unlink($file);
    unlink $file_unzipped;
    return $file_decoded;
}

sub _download_from_datacamp {
    my ($self) = @_;
    require DatacampTskvDownloader;
    $self->log("_get_raw_page_text_file file from DataCamp BEG");
    $self->log("business_id=".$self->{business_id}." shop_id=".$self->{shop_id});
    my $file = $self->get_tempfile('next_data_pack_datacamp', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
    local $ENV{YT_TOKEN_PATH} ||= $self->proj->options->{yt_client_params}{params}{token_path};

    my $datacamp_download_result = DatacampTskvDownloader::Download($file, $self->{business_id}, $self->{shop_id});
    if (!$datacamp_download_result && $self->{is_new_task}) {
        die("Zero snapshot in Datacamp");
    }
    $self->log("_get_raw_page_text_file file from DataCamp END");
    return $file;
}

sub _download_from_zora {
    my ($self) = @_;
    my $file = $self->get_tempfile('next_data_pack_zip2text', UNLINK => 1, DIR => $Utils::Common::options->{dirs}{banners_generation_files});
    open(F, "> $file") || die("Cant open file '".$file."' - $@");
    binmode(F);

    if ( not $self->{page} ){
        $self->log("_get_raw_page_text_file empty");
        print F '';
    }
    else {
        # фид скачивается здесь
        $self->log("_get_raw_page_text_file file BEG");
        my $timeout = $self->{download_timeout} || 3600;
        $self->{page}->timeout($timeout);
        $self->{page}->{enable_lwp_whitelist} = 1;
        $self->{page}->{zora_big_files} = 1;
        $self->{page}->result_file($file);
        $self->{page}->tt; # записали в файл
        if ($self->{page}->{download_failed}) {
            $self->{error} ||= "Err1201: ".$self->{page}->{download_failed};
            push(@{$self->{errors}}, {
                code => 1201,
                message => $self->{page}->{download_failed},
                message_ru => $self->{page}->{download_failed_ru},
            });
        }
        $self->log("_get_raw_page_text_file file END");
    }
    close(F);
    return $file;
}

sub _use_datacamp {
    my ($self) = @_;
    return 0 unless $self->{business_id} && $self->{shop_id};
    return 1 if $self->{force_dc};
    return 0;
}


#####
# Получить сырой файл фида
#
sub _get_raw_page_text_file {
    my ($self) = @_;

    # Берём из файлового кэша
    return $self->{__page_raw_text_file} if $self->{__page_raw_text_file};
    my $file = '';
    if ( $self->{extfile} ) { # если уже есть external file
        $self->log("_get_raw_page_text_file extfile");
        $file = $self->{'extfile'};
    } elsif ($self->_use_datacamp) {
        $self->log("Download from datacamp");
        $file = $self->_download_from_datacamp;
        if (-z $file) {
            $self->log("Download from datacamp failed, fallback to zora");
            $file = $self->_download_from_zora;
        }
    } else {
        $file = $self->_download_from_zora;
    }

    if( $self->{max_file_size_type}){
        my $size = 0;
        if ( $self->{max_file_size_type} eq 'bytes' ) {
            $size = $self->proj->file($file)->size;
        }
        elsif ( $self->{max_file_size_type} eq 'file' ) {
            $size = $self->{srcfilesize};
        } else {
            $size = $self->proj->file($file)->wc_c;
        }

        if ( defined( $self->{max_file_size} ) && ($size > $self->{max_file_size}) ) {
            $self->{error} ||= "Err1266: Size of the feed file is too big ($size > ".$self->{max_file_size}.").";
            push(@{$self->{errors}}, {
                code => 1266,
                message => "Size of the feed file is too big ($size bytes > ".$self->{max_file_size}." bytes).",
                message_ru => "Фид имеет слишком большой размер ($size байт > ".$self->{max_file_size}." байт).",
            });
            $self->proj->do_sys_cmd(":> $file"); #Затираем данные, дальше их нет смысла обрабатывать, так как фатальная ошибка уже есть.
        }
    }

    $self->{__page_raw_text_file} = $file;
    return $file;
}

#####
# Получить распакованный и декодированный файл с данными фида
# Если уже есть, берется из кэша
#
sub _get_page_text_file {
    my ($self) = @_;

    # Берём из файлового кэша
    return $self->{__page_text_file} if $self->{__page_text_file};

    my $file_decoded = '';
    my $file = $self->_get_raw_page_text_file;

    # распаковываем и декодируем
    $file_decoded = $self->_unpack_and_decode_file($file);
    $self->{srcfilesize} = -s $file_decoded;
    $self->proj->log("_get_page_text filesize $self->{srcfilesize}");

    $self->{__page_text_file} = $file_decoded;
    $self->proj->log("_get_page_text ".$file_decoded);
    $self->proj->log("_get_page_text END");

    return $file_decoded;
}

#####
# Получить размер строки в байтах
#
sub _bytes_size {
    my ($self, $data) = @_;
    _utf8_off($data);
    my $sz = length($data);
    _utf8_on($data);
    return $sz;
}

#####
# Распаковать zip-файл и вернуть его содержимое
#
sub _zip2text {
    my ($self, $data) = @_;
    return $self->proj->detect_charset->text2utf8(uncompressdata($data));
}

1;
