package BM::BannersMaker::MakeBanners;

use strict;
use warnings;

use BM::BannersMaker::BannerLandProject;
use JSON::XS qw();

use ObjLib::SubLogger;
use ObjLib::Timer;

use Utils::Sys qw(log_time_hires_fmt md5int do_safely);



sub _yield_log {
    my ($self, $ns, $log) = @_;
    my %data;
    for my $fld ($self->{ID_FIELD}, 'task_id') {
        if ($self->{row} and $self->{row}{$fld}) {
            $data{$fld} = $self->{row}{$fld};
        } else {
            $data{$fld} = '';
        }
    }
    $self->{yield}->({
        %data,
        job_id => $self->{job_id},
        pid => $$,
        time => log_time_hires_fmt(),
        ns => $ns,
        log => $log,
    } => $self->{dst_tables}->{LOG_TABLE});
}

sub _yield_funnel {
    my ($self, $ctx) = @_;
    my $json_obj = $self->{json_obj};
    my $funnel_for_source = $ctx->{funnel_for_source};
    if ($funnel_for_source and keys(%$funnel_for_source)) {
        _yield_log($self, 'funnel_for_source', $json_obj->encode($funnel_for_source));
    }
}

sub _get_task_obj {
    my ($self, $proj, $task_inf) = @_;
    return $proj->get_task_obj($self->{TASK_TYPE}, $task_inf);
}



sub init_prepare_tasks_and_offers {
    my $self = shift;
    $self->{proj} = BM::BannersMaker::BannerLandProject->new({load_dicts => 1});
    $self->{json_obj} = JSON::XS->new->utf8(0);
}

sub begin_prepare_tasks_and_offers {
    my $self = shift;
    $self->{row_count} = 0;
}

sub end_prepare_tasks_and_offers {
    my $self = shift;
    delete $self->{row_count};
}

sub map_prepare_tasks_and_offers {
    my $self = shift;
    my $r = $self->{row};
    my $proj = $self->{proj};
    my $json_obj = $self->{json_obj};
    my $id_fld = $self->{ID_FIELD};

    if (!$r->{product_inf}) {
        # task without offers, just skip it
        return;
    }

    my $product_inf = eval { $json_obj->decode($r->{product_inf}) };
    if ($@) {
        die "bad product json!";
    }
    my $pt_class = $r->{product_class};
    my $pt = $proj->product_by_classname($pt_class, $product_inf);
    $pt->class_init(proj => $proj);

    my $task_inf = eval { $json_obj->decode($r->{task_inf}) };
    if ($@) {
        die "bad task json!";
    }
    my $task = _get_task_obj($self, $proj, $task_inf);

    do_safely( sub {$proj->validate_url($pt->url)},  no_die => 1,) || do {
        $pt->{url} = ''
    };
    my $product_domain = $proj->site($pt->url)->domain;
    $r->{norm_url} = $task->prepare_url_key($pt->url);
    $r->{norm_domain} = $task->get_source_key($product_domain);
    if (!$r->{$id_fld}) {
        $r->{$id_fld} = join('-', 'row', $$, $self->{job_id}, ++$self->{row_count});
    }
    
    # Сериализуем оффер, т.к. хотим чтобы в упрощенку попадал product после вызова конструктора
    $r->{product_inf} = $json_obj->encode($pt->FREEZE);

    $self->{yield}->($r => $self->{dst_tables}->{OUTPUT_TABLE});
}



sub init_process_offer_common {
    my $self = shift;

    my $logger = ObjLib::SubLogger->new({
        mapper_object => $self,
        preprocess => {
            max_length => 10 * 2**20,
        },
        out_sub => sub {
            my $logger = shift;
            my $text = shift;
            _yield_log($logger->{mapper_object}, 'proj', $text);
        },
    });

    my $proj = BM::BannersMaker::BannerLandProject->new({
        load_dicts => 1,
        load_minicategs_light => 1,
        load_languages => [ qw(ru en tr) ],
        use_comptrie_subphraser => 1,
        use_sandbox_categories_suppression_dict => 1,
        allow_lazy_dicts => 1,
        logger => $logger,
    });
    $self->{proj} = $proj;
    $self->{json_obj} = JSON::XS->new->utf8(0);
}

sub begin_process_offer_common {
    my $self = shift;
    $self->{timer} = ObjLib::Timer->new({prefix_sep => '/'});
}

sub end_process_offer_common {
    my $self = shift;
    _yield_log($self, 'timer_report', $self->{timer}->report_str);
    delete $self->{timer};
}


sub get_offer_preview {
    my $self = shift;
    my $row = $self->{row};
    my $proj = $self->{proj};
    my $timer = $self->{timer};
    my $json_obj = $self->{json_obj};
    my $id_field = $self->{ID_FIELD};

    my $product_class = $row->{product_class};

    $timer->set_prefix($product_class);
    $timer->time('get_offer_preview');

    my $product_inf = $json_obj->decode($row->{product_inf});
    my $product = $proj->product_by_classname($product_class, $product_inf);
    $product->class_init(proj => $proj);

    return if !$proj->validate_url($product->url);

    my $banners = $product->perf_banners_single;

    my $title = $banners->[0]{title} || '';
    my $offer_id = $product->{id} || $product->{OfferID} || $product->{OfferId} || '';


    if ($offer_id && $title) {

        my $jsondata = {
            ($product->{price} ?
                (price => {
                    current => $product->get_valid_price,
                    ( $product->{oldprice} ? (old => $product->get_valid_oldprice) : () ),
                })
            : ()),
            text => {
                ( $product->{currencyId} ? (currency_iso_code => $product->{currencyId}) : () ),
            }
        };
        if ($product->{additional_data} && $product->{additional_data}{text}) {
            $jsondata->{text}{$_} = $product->{additional_data}{text}{$_} for keys %{$product->{additional_data}{text}};
        }

        my $preview = {
            Title => $title,
            Url => $product->url,
            OfferID => $offer_id,
            Info => $proj->json_obj->encode($jsondata),
            BannerID => md5int(join(',', $offer_id, $title, $product->url)),
        };
        $self->{yield}->({
            %$preview,
            $id_field => $row->{$id_field},
            } => $self->{dst_tables}->{OUTPUT_TABLE});
    }
}


sub map_process_offer {
    my $self = shift;
    my $r = $self->{row};
    my $proj = $self->{proj};
    my $timer = $self->{timer};
    my $json_obj = $self->{json_obj};
    my $id_fld = $self->{ID_FIELD};

    my $pt_class = $r->{product_class};

    $timer->set_prefix($pt_class);
    $timer->time('decode');

    my $product_inf = $json_obj->decode($r->{product_inf});
    my $pt = $proj->product_by_classname($pt_class, $product_inf);
    $pt->class_init(proj => $proj);

    my $task_inf = $json_obj->decode($r->{task_inf});
    # сейчас мы экспортируем task_inf, а не $task->FREEZE, поэтому нужно проводить инициализацию
    my $task = _get_task_obj($self, $proj, $task_inf);

    do_safely( sub {$proj->validate_url($pt->url)},  no_die => 1,) || do {
        $pt->{url} = ''
    };

    if ($self->{TASK_TYPE} eq 'dyn') {
        if (!$task_inf->{_cache}) {
            _yield_log($self, 'error:no_cache', "no cache for task ".$r->{task_id});
            return;
        }
    }

    my $perl_data;
    my $ppar;  # $perl_data->{process_params}

    if (!$r->{perl_data}) {
        # first run!
        $ppar = $json_obj->decode($r->{ppar});
        $ppar->{ctx} = {
            offer_errors => {},
            timer => $timer,
            funnel_for_source => {},  # incremented in process_offer* subs
        };

        my $enrich_rows = $r->{enrich_rows} ? $json_obj->decode($r->{enrich_rows}) : {};

        # create ext_pt from enrich - group by source
        do {
            my %idx2rows;
            for my $ext_row (@{$enrich_rows->{external} // []}) {
                my $idx = delete $ext_row->{source_index};
                delete $ext_row->{$id_fld};  # used for enrich
                delete $ext_row->{_other};  # from YQL
                push @{$idx2rows{$idx}}, $ext_row;
            }

            my @ext_data;
            for my $idx (sort { $a <=> $b } keys %idx2rows) {
                my @ext_rows = @{$idx2rows{$idx}};

                # обычно @ext_rows == 1, но на всякий случай сортируем; предполагаем, что нет сложных полей
                @ext_rows = map { $_->[0] } sort { $a->[1] cmp $b->[1] } map {
                    my $ext_row = $_;
                    my $key = join("\t", map { "$_=$ext_row->{$_}" } sort keys %$ext_row);
                    [ $ext_row, $key ];
                } @ext_rows;

                @ext_rows = $task->filter_external_offers(\@ext_rows);

                my $source_info = $self->{stash}{external_sources}[$idx];

                my @ptl;
                for my $ext_row (@ext_rows) {
                    $ext_row->{product_type} = $source_info->{product_type} // '__external__';
                    my $ext_pt = $proj->product($ext_row);
                    push @ptl, $ext_pt;
                }

                push @ext_data, { source => $source_info, ptl => \@ptl };
            }
            $ppar->{ext_data} = \@ext_data;
        };

        $perl_data = { process_params => $ppar };

        my $res = $task->process_offer_init($pt, %$ppar) // {};  # modify ctx and $pt
        if (!$res->{gen_params}) {
            # штатные проблемы
            # сохраняем воронку т.к. оффер уже не доедет до process_offer_finalize
            _yield_funnel($self, $ppar->{ctx});
            _yield_log($self, 'offer_init', $json_obj->encode($ppar->{ctx}{offer_errors}));

            # save badflags in badflags namespace in logs
            if ($res->{badflags_str}) {
                _yield_log($self, 'badflags', $res->{badflags_str});
            }
            return;
        }
        $ppar->{gen_params} = $res->{gen_params};
    } else {
        $perl_data = $proj->decode($r->{perl_data});
        $ppar = $perl_data->{process_params};
        $ppar->{ctx}{timer} = $timer;

        $timer->time('eval_rpc');
        $proj->{micro_cdict} = $json_obj->decode($r->{MicroCDict} // "{}");
        eval { $proj->rpc->eval_rpc(delete $perl_data->{rpc}) };
        if ($@) {
            die "eval_rpc failed at row {$r->{$id_fld}}";
        }
    }

    my ($native_arr, $native_rpc) = eval { $task->process_offer_generate_native($pt, %$ppar) };
    # dyn uses banners_data interface: ($arr) or (undef, $rpc):
    $native_arr //= [];
    $native_rpc //= [];
    if ($@) {
        die "process_offer_generate_native failed at row {$r->{$id_fld}}: $@";
    }

    my ($external_arr, $external_rpc) = eval { $task->process_offer_generate_external($pt, %$ppar) };
    if ($@) {
        die "process_offer_generate_external failed at row {$r->{$id_fld}}: $@";
    }
    if (!@$external_rpc) {
        # больше не нужно таскать external product-ы
        $ppar->{ext_data} = [];
    }

    # кэшируем метод $pt->parse;
    # ещё, в banners_data и других методах к $pt могут дописывать поля, нужно это сохранить
    $r->{product_inf} = $json_obj->encode($pt->FREEZE);

    my $pack_bnr_counter = 0;
    my $pack_bnr = sub {
        my ($bnr, $flds) = @_;
        my %copy = %$bnr;
        my %row = map { $_ => (delete $copy{$_}) } @$flds;

        # для детерминированной сортировки
        $copy{_sort_info_} = {
            step => $self->{stash}{step},
            counter => ++$pack_bnr_counter,
        };
        $row{_other_} = $json_obj->encode(\%copy);
        return \%row;
    };
    my @title_info_fields = qw(title_source title_template title_template_type);
    my @long_title_info_fields = qw(long_title_source long_title_template long_title_template_type);
    my @pack_fields = qw(phrase title long_title template letter);
    push @pack_fields, @title_info_fields;
    push @pack_fields, @long_title_info_fields;
    for my $data (
        { arr => $native_arr,   enrich => 'banners_native' },
        { arr => $external_arr, enrich => 'banners_external' },
    ) {
        for my $bnr (@{$data->{arr}}) {
            my $packed = $pack_bnr->($bnr, \@pack_fields);
            $self->{yield}->({
                %$packed,
                $id_fld => $r->{$id_fld},
                enrich_field => $data->{enrich},
            } => $self->{dst_tables}->{ARR_TABLE});
        }
    }

    # Готовимся к сериализации
    $timer->time('encode');
    delete $ppar->{ctx}{timer};

    my @rpc = (@$native_rpc, @$external_rpc);
    if (@rpc) {
        $perl_data->{rpc} = \@rpc;
        $timer->time('yield_rpc');
        $proj->rpc->yield_rpc($perl_data->{rpc} => {
            cdict => sub {
                my $y = shift;
                $y->{$id_fld} = $r->{$id_fld};
                $self->{yield}->($y => $self->{dst_tables}->{RPC_TABLE});
            },
        });


        $r->{perl_data} = $proj->encode($perl_data);
        if (length($r->{perl_data}) > 64 * 1024 * 1024) {
            _yield_funnel($self, $ppar->{ctx});
            _yield_log($self, 'error:too_large_row', length($r->{perl_data}));
            return;
        }

        $timer->time('io');
        $self->{yield}->($r => $self->{dst_tables}->{OUTPUT_TABLE});
        return;
    }

    # offer processed!
    $self->{yield}->({
        $id_fld => $r->{$id_fld},
        task_inf => $r->{task_inf},
        task_id => $r->{task_id},
        product_class => $r->{product_class},
        product_inf => $r->{product_inf},
        ppar => $json_obj->encode($ppar),
    } => $self->{dst_tables}->{DONE_TABLE});
    return;
}



sub map_process_offer_finalize {
    my $self = shift;
    my $r = $self->{row};
    my $proj = $self->{proj};
    my $timer = $self->{timer};
    my $json_obj = $self->{json_obj};
    my $id_fld = $self->{ID_FIELD};

    $r->{$id_fld} = $r->{$id_fld};
    my $enrich_rows = $r->{enrich_rows} ? $json_obj->decode($r->{enrich_rows}) : {};

    $timer->time('decode');
    my $pt_class = $r->{product_class};

    my $product_inf = $json_obj->decode($r->{product_inf});
    my $pt = $proj->product_by_classname($pt_class, $product_inf);
    $pt->class_init(proj => $proj);

    my $task = _get_task_obj($self, $proj, $json_obj->decode($r->{task_inf}));

    my $ppar = $json_obj->decode($r->{ppar});
    $ppar->{ctx}{timer} = $timer;

    my $unpack_bnr = sub {
        my $row = shift;
        my $other = $row->{_other_} ? $json_obj->decode(delete $row->{_other_}) : {};
        return {%$row, %$other};
    };

    # сортировка для детерминированности, удаление лишних полей
    my $fix_arr = sub {
        my $arr = shift;
        @$arr = map { $_->[0] } sort {
            ($a->[1] <=> $b->[1]) || ($a->[2] <=> $b->[2])
        } map {
            my $elem = $_;
            my $sort_info = delete($elem->{_sort_info_});
            [ $elem, $sort_info->{step}, $sort_info->{counter} ];
        } @$arr;
    };

    my @native_arr = map { $unpack_bnr->($_) } @{$enrich_rows->{banners_native} // []};
    $fix_arr->(\@native_arr);

    my @external_arr = map { $unpack_bnr->($_) } @{$enrich_rows->{banners_external} // []};
    $fix_arr->(\@external_arr);

    my $arr = $task->process_offer_combine(\@native_arr, \@external_arr);

    # copy-paste from process_offer :(
    $ppar->{minuswords} = $task->process_offer_generate_minuswords($pt, %$ppar, phrases => \@native_arr);
    $ppar->{phrases} = $arr;

    my %titles;
    my %long_titles;
    for my $h (@$arr) {
        next if !$h->{title_source};
        _yield_log($self, "warning", "No title_template_type for title_source - $h->{title_source}") if !$h->{title_template_type};
        my $title_type = $h->{title_source}."_".$h->{title_template_type};
        if (!exists $titles{$title_type} or !$titles{$title_type}->{title}) {
            $titles{$title_type} = {
                title => $h->{title},
                title_source => $h->{title_source},
                title_template => $h->{title_template},
                title_template_type => $h->{title_template_type},
            };
        };
        if (!exists $long_titles{$title_type} or !$long_titles{$title_type}->{title}) {
            $long_titles{$title_type} = {
                title => $h->{long_title},
                title_source => $h->{long_title_source},
                title_template => $h->{long_title_template},
                title_template_type => $h->{long_title_template_type},
            };
        }
    }
    $ppar->{titles} = \%titles;
    $ppar->{long_titles} = \%long_titles;

    my $result = $task->process_offer_finalize($pt, %$ppar);
    my $task_id = $task->task_id;
    my $OrderID = $task->taskinf->{OrderID};
    my $TaskGroupingID = $task->taskinf->{GroupingID};

    $timer->time('io');
    for my $h (@{$result->{phrases}}) {
        $h->{MinusScore} = -$h->{Score};
        $h->{task_id} = $task_id;
        $h->{OrderID} = $OrderID;
        $h->{TaskGroupingID} = $TaskGroupingID;
        $h->{bannerphrase_md5} = $task->get_bannerphrase_md5($h);

        $h->{$id_fld} = $r->{$id_fld};  # поле не должно входить в bannerphrase_md5
        $self->{yield}->($h => $self->{dst_tables}->{DONE_TABLE});
    }
    _yield_funnel($self, $ppar->{ctx});
}

1;
