package BM::BannersMaker::Product;

use utf8;
use strict;
use warnings;

use base qw(ObjLib::ProjPart);
use Data::Dumper;
use Digest::MD5 qw(md5_hex);
use Utils::Array;
use Utils::Urls qw(fix_url_scheme);
use List::Util qw(min);
use Storable qw(dclone);
use JSON qw(to_json);
use URI;

use Utils::Sys;
use Utils::Urls qw(url_to_punycode);

use BM::PhraseParser;

use Time::HiRes qw/time/;

########################################################
#Доступ к полям
########################################################

__PACKAGE__->mk_accessors(qw(
    name
    url
    minicategs
    vendor
    model
    typePrefix
    categpath
    description
    product_type
    offer_source
    source_letter
    feed_lang
));

########################################################
#Инициализация
########################################################

our @offer_fields = qw{ name url minicategs market_category description images product_type
                       vendor model typePrefix categoryId categoryName categpath picture price
                       oldprice currencyId manufacturer_warranty available offerfilters
                       additional_data product_type params bad_flags color
                       custom_phrases
                       merge_key
                    };

sub get_common_offer_fields {
    return qw(
        canonized_url
        params_specformat
        main_mirror
        main_mirror_id
        orig_domain
        orig_domain_id
        offer_line_md5
        offer_source
        offerYabsId
        id
        OfferId
        OfferID
        source_letter
        feed_lang
        use_as_name
        use_as_body
        model
        vendor
        age_time
        age_unit
        size
        size_unit
        sex
        g:condition
        condition:type
        condition:reason
        condition-type
        condition-reason
        group_id
    );
}

# заглушка и пример оформления методов class_init для дочерних Product-ов
sub class_init :RUN_ONCE {
    my $class = shift;
    my %par = @_;
    my $proj = $par{proj};

    $proj->log("base class_init: nothing to do!");
}

sub init {
    my ($self) = @_;

    my $data = $self->{data};

    if(ref($data->[0]) eq 'HASH'){
        $self->{$_} = $data->[0]{$_} for @offer_fields, $self->get_common_offer_fields;

        # Дополнительно инициализируем все поля специфичных Product-ов, отличных от стандартного, если включена опция "init_specific_product_fields".
        # Касается только Product-ов, определение которых зависит от категоризации оффера (т.е. не относится к монотематикам, ProductDSE и т.д.).
        # Нужно как минимум в export_offers для дапма в атрибуты класса полной информации об оффере (на входе BLRT).
        # Вне зависимости от того, какой Product вернула Каталогия на железе (т.к. в BLRT оффер обрабатывается Product-ом от Мультика).
        if ($data->[0]{__init_specific_product_fields}) {
            $self->init_specific_product_fields($data);
        }

        $self->{_from_feed} = {};
        for my $field (qw(typePrefix model vendor name)) {
            $self->{_from_feed}->{$field} = $data->[0]{$field};
            $self->{_from_feed}->{$field} = '' if $data->[0]{$field} && $data->[0]{$field} !~ m/\w/;
        }
    }elsif(ref($data->[0]) eq 'ARRAY'){
        $self->{name} = $data->[0][0] || '';
        $self->{url} = $data->[0][1] || '';
        $self->{_from_feed}->{name} = $self->{name};
        $self->{_from_feed}->{name} = '' if $self->{name} !~ m/\w/;
    }else{
        $self->{name} = $data->[0];
        $self->{_from_feed}->{name} = $self->{name};
        $self->{_from_feed}->{name} = '' if $self->{name} && $self->{name} !~ m/\w/;
    }

    # отдельно выпарсиваем название магазина
    $self->{shopname} = $self->get_shopname || '';

    if ($self->{use_as_name}) {
        $self->{use_as_name} = $self->proj->make_bs_compatible_or_empty($self->{use_as_name});
    }

    # ключевые слова для омонимичных категорий
    if ( $self->minicategs ){
        my @keywords =  map { my $categ = $_; my $txt = $self->proj->categories_text_fields_dict->get_value($categ, 'dynamic_homonymy_words') || ''; $txt ? split/\s*\,\s*/,$txt : () }
                    grep { my $categ = $_; $self->proj->categs_tree->check_minicateg_flag($categ,'_dynamic_homonymy') }
                    split /\s+\/\s+/,$self->minicategs;
        $self->{dynamic_homonymy_words} = join(',',@keywords) if @keywords;
    }
    if ( $self->description ){
        $self->{description} = $self->proj->phrase( $self->description )->head(10);
    }
}

sub after_init {
    my ($self) = @_;
    delete($self->{data});
}

sub is_base_product {
    my $self = shift;
    return __PACKAGE__ eq ref $self;
}

sub ad_type {
    return 'retail';
}

sub match_type {
    return 'snorm';
}

sub reset_flag { # индикатор того, что property класса было сброшено
    return '__RESET__';
}

sub init_specific_product_fields {
    my $self = shift;
    my $data = shift;

    require BM::BannersMaker::ProductAutoAccessories;
    require BM::BannersMaker::ProductBeautyHealth;
    require BM::BannersMaker::ProductChildrensGoods;
    require BM::BannersMaker::ProductFurniture;
    require BM::BannersMaker::ProductGames;
    require BM::BannersMaker::ProductTiresDisks;
    require BM::BannersMaker::ProductWear;
    require BM::BannersMaker::ProductBooks;

    $self->{$_} = $data->[0]{$_} for uniq (
        BM::BannersMaker::ProductAutoAccessories->get_offer_fields(),
        BM::BannersMaker::ProductBeautyHealth->get_offer_fields(),
        BM::BannersMaker::ProductChildrensGoods->get_offer_fields(),
        BM::BannersMaker::ProductFurniture->get_offer_fields(),
        BM::BannersMaker::ProductGames->get_offer_fields(),
        BM::BannersMaker::ProductTiresDisks->get_offer_fields(),
        BM::BannersMaker::ProductWear->get_offer_fields(),
        BM::BannersMaker::ProductBooks->get_offer_fields(),
    );
}

sub clean_md5 :CACHE {
    my $self = shift;
    my $inf = dclone($self->FREEZE);

    # нужно удалить поля, дописываемые в $pt при генерации
    my @unsignif = qw(
        retarg _offid_ orig_domain
        first_title second_title
        checksum
        offer_line_md5
        offer_source
        minus_words
    );
    delete $inf->{$_} for @unsignif;

    # кэши тоже
    delete $inf->{$_} for grep { /^_cached_/ } keys %$inf;

    my $inf_json = to_json($inf, {canonical => 1, utf8 => 1});
    return md5_hex($inf_json);
}

########################################################
# Логгирование, кеширование, вспомогательные методы
########################################################

# киото
sub get_remotecache_id {
    my ($self) = @_;
    return 'v1019 '.$self->kyoto_cache_id;
}

sub kyoto_cache_id :CACHE {
    my ($self) = @_;
    return '' unless $self->url;
    return md5int_base64($self->url);
}

#Отдельный метод, чтобы гарантировать, что не будет параметров
# на YT не будет использоваться
sub dyn_banners_cached : KYOTOCACHE(1000002) { #Примерно 12 дней
    return $_[0]->dyn_banners;
}

sub title {
    my ($self) = @_;
    return $self->name;
}

sub get_shopname :CACHE {
    my ($self) = @_;
    my $dmn = $self->proj->page($self->url)->domain_2lvl;
    $dmn =~ s/\.\w+$//;
    return $dmn;
}

sub norm_url :CACHE {
    my ($self) = @_;
    return $self->proj->page($self->url)->norm_url;
}

sub get_minicategs :CACHE {
    my ($self) = @_;
    my @res = ();
    my $txtsource = $self->proj->make_bs_compatible_or_empty($self->name || $self->model);
    push @res, $self->proj->phrase( $txtsource )->get_minicategs if $txtsource;
    push @res, $self->proj->phrase( $self->categpath )->get_minicategs if $self->categpath;
    return @res;
}

sub get_simple_color :CACHE {
    my ($self, $color) = @_;
    return $color unless $color;

    my $h_latin_colors_map = $self->proj->fdm->latin_colors_map;
    $color = $h_latin_colors_map->{lc($color)} if $h_latin_colors_map->{lc($color)};
    $color =  $self->proj->phrase(lc($color))->norm_phr;
    $color =~ s/(темно|светло|нежно|серо|красно|сине|ярко)-//;

    my $map_color_hash = $self->proj->fdm->simple_color_map;

    if (exists $map_color_hash->{$color}) {
        $color = $map_color_hash->{$color};
    }
    if ($color =~ /[^а-яё]/i) {
        $color = ''; #цвет слоновой кости -> костя слоновый цвет
    }
    return $color;
}


########################################################
# Парсинг товарного текста
########################################################

sub _clear_tag {
    my ( $self, $text ) = @_;
    $text = $self->proj->phrase($text)->get_prefiltered_line->text;
    ( $text ) = $self->proj->phrase($text)->cut_qty;
    return $text;
}

sub txtsource :CACHE {
    my ($self) = @_;
    my $txtsource = $self->proj->make_bs_compatible_or_empty($self->name || $self->vendor . " " . $self->model);
    $txtsource =~ s/\+/ /g;
    $txtsource =~ s/\s+/ /g;
    $txtsource =~ s/(^\s+|\s+$)//g;
    return $txtsource;
}

sub get_type_from_categpath {
    my ($self, $categpath) = @_;
    my @parts = map {s/^\s+|\s+$//g; $_} split('/', $categpath); ##no critic
    my $res = undef;
    if (defined $parts[-1]) {
        $res = $self->proj->phrase($parts[-1])->get_goods if ($parts[-1] !~ m/\,|\/|\ и\ |\ или\ /i);
    }
    if (not defined $res and defined $parts[-2]) {
        $res = $self->proj->phrase($parts[-2])->get_goods if ($parts[-2] !~ m/\,|\/|\ и\ |\ или\ /i);
    }
    return $res;
}

# первый аргумент в приоритете
# if {b} 'in' a => a win else b win
sub compare_types {
    my ($self, $a, $b) = @_;
    $b =~ s/(?:^|\b)(?:аксессуары?|игрушк[аи]|мебель|материалы?|инструменты?|оборудован[иь]е|акустика)(?:\b|$)//gi;
    return (($self->proj->phrase($b) ^ $self->proj->phrase($a))->text_length == 0) ? $a : $b;
}

sub badcolors_phraselist :GLOBALCACHE {
    my $self = shift;
    my $badcolors = 'ослепительная платина,розовое золото,черный бриллиант,чёрный сапфир,серебристый титан,белый жемчуг,благородный изумруд';
    return $self->proj->phrase_list( [ split/\s*\,\s*/, $badcolors ] );
}

sub restore_capitalization {
    my $self = shift;
    my $h = shift;
    my $src = shift;
    $h->{$_} = $self->proj->phrase( $src )->restore_capitalization($h->{$_}) for @{array_intersection([keys %$h],[qw/model model_for model_weak brand brand_for type type_for quotes/])};
    # для типа товара опускаем регистр в тех словах, где все буквы, начиная со второй - строчные
    if ($h->{type}) {
        my @type_res = ();
        for my $word ($self->proj->phrase($h->{type})->words_lite) {
            push @type_res, ($word =~ /^.+\p{Uppercase}/) ? $word : lc $word;
        }
        $h->{type} = join ' ', @type_res;
    }
}

sub parse :CACHE {
    my ($self) = @_;
    my $logger = $self->proj->logger;
    $logger->debug("product parse beg");

    my $h = {};
    my $name = $self->proj->make_bs_compatible_or_empty($self->name);
    my $type = $self->proj->make_bs_compatible_or_empty($self->typePrefix);
    my $model = $self->proj->make_bs_compatible_or_empty($self->model);
    my $description = $self->proj->make_bs_compatible_or_empty($self->description);

    my $txtsource_phr = $self->proj->phrase( $self->txtsource );

    # вычищаем плохие цвета
    my $txtsource_badcolors_phl = $self->badcolors_phraselist->search_subphrases_in_phrase($txtsource_phr);
    if ( $txtsource_badcolors_phl->count ) {
        $txtsource_phr = $txtsource_phr->subtract_phl_by_norm( $txtsource_badcolors_phl );
    }

    eval { $h = {$txtsource_phr->parse(use_goods_basis => 1)}; };
    if ($model and $txtsource_phr->text and ($txtsource_phr->text eq $model) and $h->{type} and (not $h->{model})) {
        # похоже, что в поле model написан обобщённый type, а сама модель находится в typePrefix из фида, пробуем её оттуда выпарсить
        my $h_temp = {};
        my $type_for_parse_phr = $self->proj->phrase( $type );

        #!!!раньше здесь был безусловный subtract_phl_by_norm, который делал это неявно. Оставлено ради того, чтобы сошелся b2b. Возможно, можно убрать.
        my @tpwords = $type_for_parse_phr->words;
        $type_for_parse_phr = $self->proj->phrase( $type_for_parse_phr->restore_capitalization( "@tpwords" ) );
        #!!!

        my $type_badcolors_phl = $self->badcolors_phraselist->search_subphrases_in_phrase($type_for_parse_phr);
        if ( $type_badcolors_phl->count ) {
            $type_for_parse_phr = $type_for_parse_phr->subtract_phl_by_norm( $type_badcolors_phl );
        }
        eval { $h_temp = {$type_for_parse_phr->parse(use_goods_basis => 1)}; };
        for my $k (grep { $_ =~ /model/i } keys %$h_temp) {
            $h->{$k} = $h_temp->{$k};
        }
    }

    my $type_src = $h->{type} || '';

    if ( $h->{quotes} ){
        $h->{quotes} = '' if ( $h->{model} && lc($h->{quotes}) eq lc($h->{model}) );
        if ( $h->{model} && !array_intersection([split/\s+/,lc $h->{model}],[split/\s+/,lc $h->{quotes}]) && length($h->{model}) < 10 ){
            $h->{model} = join (' ', ($h->{model}, $h->{quotes}) );
            $h->{quotes} = '';
        }
    }

    $logger->debug("product parse 1", $h);

    #Накладываем на парсер данные из хэша товара
    # Более надежным является вендор, выпарсенный из товарного наименования (особенно если он есть в словаре брендов)
    my $explicit_feed_brand = $self->vendor ? $self->proj->phrase($self->vendor)->get_brand : undef;
    $h->{brand} = $explicit_feed_brand if $explicit_feed_brand;
    $h->{brand} ||= $self->vendor if $self->vendor;
    $h->{brand} ||= $self->proj->phrase( $description )->get_brand if $description;
    $h->{_model_has_underscores} = 1 if (($h->{model} and $h->{model} =~ /_/) or ($model and $model =~ /_/));
    %$h = $self->proj->phrase->filter_and_check_model( $h ); # проверка пока здесь, т.к. требуется исключить неудачные модели из quotes

    $h->{brand} = '' if $h->{brand} =~ /нет бр(е|э)нда|не определ(е|ё)н/i;
    $h->{brand} = '' if $h->{brand} =~ /\.(?:ru|com|org|net|su|ua|by|ру|en|uk|edu)$/;

    $logger->debug("product parse 2", $h);

    # добавил if, т.к. часто пишут большие и немодельные фразы в поле model в фидах
    if (    $self->model
        && !$h->{model}
        && $txtsource_phr->text ne $self->model
        && !$self->proj->phrase( $self->model )->get_goods
        && !$h->{quotes} ){
        $h->{model} = $self->proj->phrase->clear_text( $self->_clear_tag($self->model) );
    }

    # берем из typePrefix
    my $use_type_prefix_from_feed = 0;
    if ( $self->typePrefix && $self->proj->phrase($self->typePrefix)->get_goods_simple ) {
        $h->{type} = $self->typePrefix;
        $use_type_prefix_from_feed = 1;
    }
#    $h->{type} ||= $self->typePrefix if ( $self->typePrefix && $self->proj->phrase($self->typePrefix)->get_goods_simple );
    if ( $h->{model} && $h->{brand} ){
        $h->{model} = ( $self->proj->phrase($h->{model}) % $self->proj->phrase($h->{brand}) )->text;
    }

    if (not defined $h->{type} or $h->{type} eq '') {
        #из categpath
        my $type_cp = '';
        if ($self->categpath) {
            $type_cp = $self->get_type_from_categpath($self->categpath);
        }

        # из description
        my $type_ds = '';
        if ( $description && $description !~ /загрузка белья/i ){
            my $goods = $self->proj->phrase( $description )->get_goods || '';
            $type_ds = $goods if $goods;
            $type_ds = '' unless $self->proj->phrase( $description )->_is_type_correct( $type_ds );
        }
        if ($type_cp) {
            $h->{type} = $self->compare_types($type_src, $self->compare_types($type_ds, $type_cp));
        }
        if ($h->{type} eq '' and $type_src ne '') {
            $h->{type} = $self->compare_types($type_ds, $type_src);
        }
        if ($h->{type} eq '') {
            $h->{type} = $type_ds;
        }

        if ( my @ct = $self->proj->phrase($h->{type})->cut_colors ){ # цвет может оказаться в типе, взятом из dict_goods
            $h->{color} = $ct[0] if $ct[0];
            $h->{type} = $ct[1] if $ct[0];
        }
    }

    $self->remove_duplicates_from_stones($h);

    if ($h->{model}){
        $h->{model} =~ s/^(\S+\s+\S+\s+\S+\s+\S+)(\s+.*$|$)/$1/;
        $h->{model} =~ s/^(([а-яё]|на|не|для|под|над)\s+)|(\s+([а-яё]|на|не|для|под|над))$//gi;
        ( $h->{model} ) = $self->proj->phrase($h->{model})->cut_qty;
        $h->{model} = '' if length($h->{model}) < 3;
        $h->{model} = '' if lc($h->{type}) eq lc($h->{model});
        if ( $h->{model} !~ /\d/ && $h->{model} !~ /[a-z]/i ){
            $h->{type} ||= $h->{model};
            $h->{model} = '';
        }
        # в модели не должно быть топонима
        if ( $h->{toponyms} ){
            my $retoponyms = $h->{toponyms};
            $retoponyms =~ s/\:/\|/g;
            $h->{model} = '' if $h->{model} =~ /($retoponyms)/i;
        }
    }
    $h->{class} ||= $txtsource_phr->is_accessory ? 'accessory' : 'goods';
    if ( $h->{class} eq 'accessory'){
        $h->{type_for} = $h->{type};
        $h->{type} = '';
    }

    $h->{dynamic_homonymy_words} = $self->{dynamic_homonymy_words};
    $h->{minicategs} = $self->{minicategs};
    $h->{shopname} = $self->{shopname};

    $logger->debug("product parse 99", $h);

    # из типа и модели удаляем пол, возраст, топонимы, цвета и количество
    my $del_from_model_type = join "|", map { split/\:/ } ( map { $h->{$_} || '' } qw/toponyms colors qty vol sex age season/ );
    if ( $del_from_model_type ){
        for ( grep { $h->{$_} } qw/model type/ ){
            $h->{$_} =~ s/\b($del_from_model_type)\b/ /gi;
        }
    }

    # все цвета не нужны, оставляем один
    if ( $h->{colors} ){
        $h->{color} = (split/\:/,$h->{colors})[0] if $h->{colors};
        delete $h->{colors};
    }

    # ограничение по длине для model_weak
    if ( $h->{model_weak} ){
        $h->{model_weak} = '' unless length($h->{model_weak}) > 3;
    }

#KOSTYL' heuristics
# "noun для adv" => "noun adj"
# "noun adj" => harmonize
# проверяем, содержится ли такое исправление в dict_goods. Если содержится, ничего больше не гармонизуем
   if ($h->{type} and not $use_type_prefix_from_feed and not ($self->proj->phrase($h->{type})->is_fixed_in_dict_goods)) {
        $h->{type} =~ s/огражденье/ограждение/;
        my @words = split(/\s+/, $h->{type});
        my $cnt_adj = 0;
        my $cnt_for = 0;
        my $type_no_adj = '';
        for my $w(@words) {
            if ($w =~ m/(ий|ый|ой|вая|ная|мая|лая|кая|ые|ие|ое|ых|их|яя)$/) {
                $cnt_adj++;
            } else {
                $type_no_adj = $w;
            }
            $cnt_for++ if $w eq 'для';
        }
        if ($cnt_adj + $cnt_for == scalar(@words) - 1) {
            if ($cnt_for == 0 and $cnt_adj > 0) {
                $h->{type} = $self->proj->phrase($type_no_adj)->harmonize($h->{type});
            } else {
                $h->{type} =~ s/для\ //g;
            }
        }
        $h->{type} =~ s/оцинковавший/оцинкованный/;
    }

    delete($h->{$_}) for grep {!defined($h->{$_}) || ($h->{$_} eq '')} keys %$h;
    delete($h->{$_}) for qw{model_dirty};
    my $src = join (' ', grep {$_ && /\S/} ( $h->{src}, $name, $model, $description, $self->typePrefix, $self->vendor ) );
    $h->{type} =~ s/^(.)/\l$1/ if $h->{type};

    # фиксим слова, где в русских буквах встречается латиница
    $h->{type} = $self->proj->phrase($h->{type})->clean_mixed_ruwords;

    # костыль для аксессуаров
    if ($h->{type} and ($self->proj->phrase($h->{type})->is_accessory) and ($h->{minicategs} =~ /аксессуар/i)) {
        for my $f (qw(type brand model)) {
            if ($h->{$f} and not $h->{$f."_for"}) {
                $h->{$f."_for"} = $h->{$f};
                delete $h->{$f};
            }
        }
    }

    if ($self->{price} && $self->{currencyId} && ($self->{currencyId} =~ m/^(RUR|RUB)$/i)) {
        $h->{price_message} = "— " . int($self->{price}) . " руб." if int($self->{price});
    }

    $self->restore_capitalization($h, $src);

    $logger->debug("product parse end");

    return $h;
}

sub remove_duplicates_from_stones {
    my $self = shift;
    my $h = shift;

    if ( $h->{type} && $h->{brand} ){
        $h->{type} = ( $self->proj->phrase($h->{type}) % $self->proj->phrase($h->{brand}) )->text;
        $h->{brand} = '' if ( $h->{brand} =~ /[а-яА-Я]/ && $self->proj->phrase_list( [ $h->{brand}, $h->{type} ] )->intersection->count ); # тип Диван и брэнд Диван.ру
    }
    if ( $h->{model} && $h->{type} ){
        $h->{type} = ( $self->proj->phrase($h->{type}) % $self->proj->phrase($h->{model}) )->text;
    }
    if ( $h->{brand} && $h->{quotes} ){
        $h->{quotes} = ( $self->proj->phrase($h->{quotes}) % $self->proj->phrase($h->{brand}) )->text;
    }
    if ( $h->{type} && $h->{quotes} ){
        $h->{type} = ( $self->proj->phrase($h->{type}) % $self->proj->phrase($h->{quotes}) )->text;
    }
}

########################################################
# Шаблоны
########################################################

sub title_source {
    return 'native';
}

our $_MDL = 'model:_DEMULT';
our $_TYPE = 'type:_DEMULT';
our $_BRND = 'brand:_DEMULT';

our $MDL = "$_MDL/$_MDL:_SHORT_MDL";
our $MDLD = "$_MDL:_DOT_END/$_MDL:_SHORT_MDL:_DOT_END";
our $MDLF = 'model_for/model_for:_SHORT_MDL';

sub title_templ :STATIC :GLOBALCACHE("DERIVED") {
    my $res = "[$_TYPE/$_TYPE:_DEL_ADJ/$_TYPE:_MAINWORDS] $_BRND [$MDL], $_BRND [$MDL], [$_TYPE/$_TYPE:_DEL_ADJ] [$MDL], [$_MDL/$_MDL:_SHORT_MDL/$_MDL:_MULTIDOT_END], [$_TYPE/$_TYPE:_DEL_ADJ/$_TYPE:_MAINWORDS] $_BRND";#, sex type shopname:_UP_FIRST, type shopname:_UP_FIRST";
    return $res;
}

sub perf_title_templ :STATIC :GLOBALCACHE("DERIVED") {
    my $res = "[$_TYPE/$_TYPE:_DEL_ADJ/$_TYPE:_MAINWORDS] $_BRND [$MDL] (vol:_FIRST_SIZE), $_BRND [$MDL] (vol:_FIRST_SIZE), [$_TYPE/$_TYPE:_DEL_ADJ] [$MDL] (vol:_FIRST_SIZE), [$_MDL/$_MDL:_SHORT_MDL/$_MDL:_MULTIDOT_END] (vol:_FIRST_SIZE), [$_TYPE/$_TYPE:_DEL_ADJ/$_TYPE:_MAINWORDS] $_BRND (vol:_FIRST_SIZE)";
    return $res;
}

sub title_weak_templ :STATIC :GLOBALCACHE("DERIVED") {
    my $res = "[$_TYPE/$_TYPE:_DEL_ADJ/$_TYPE:_MAINWORDS] $_BRND model_weak, $_BRND model_weak, [$_TYPE/$_TYPE:_DEL_ADJ/$_TYPE:_MAINWORDS] model_weak, [$_TYPE/$_TYPE:_DEL_ADJ/$_TYPE:_MAINWORDS] $_BRND";
    return $res;
}

sub title_templ_quotes :STATIC :GLOBALCACHE("DERIVED") {
    my $res = "[$_TYPE/$_TYPE:_DEL_ADJ/$_TYPE:_MAINWORDS] $_BRND quotes, $_BRND quotes, [$_TYPE/$_TYPE:_DEL_ADJ] quotes, [quotes/quotes:_MULTIDOT_END]";
    return $res;
}

sub perf_title_templ_quotes :STATIC :GLOBALCACHE("DERIVED") {
    my $res = "[$_TYPE/$_TYPE:_DEL_ADJ/$_TYPE:_MAINWORDS] $_BRND quotes (vol:_FIRST_SIZE), $_BRND quotes (vol:_FIRST_SIZE), [$_TYPE/$_TYPE:_DEL_ADJ] quotes (vol:_FIRST_SIZE), [quotes/quotes:_MULTIDOT_END] (vol:_FIRST_SIZE)";
    return $res;
}

sub title_templ_acc :STATIC :GLOBALCACHE("DERIVED") {
    my $res = join ' ', map { s/^\s+//; $_ } grep { /\S/ } ##no critic
            split /\n/, "
                        $_BRND [$MDLD] [type_for:_UP_FIRST/type_for:_UP_FIRST:_DEL_ADJ/type_for:_UP_FIRST:_MAINWORDS_FOR] brand_for [$MDLF] (vol:_FIRST_SIZE),
                        $_BRND [$MDLD] [type_for:_UP_FIRST/type_for:_UP_FIRST:_DEL_ADJ/type_for:_UP_FIRST:_MAINWORDS_FOR] [$MDLF] (vol:_FIRST_SIZE),
                        [type_for:_UP_FIRST/type_for:_UP_FIRST:_DEL_ADJ/type_for:_UP_FIRST:_MAINWORDS_FOR] [$MDLF] (vol:_FIRST_SIZE),
                        $_BRND [$MDL] [model_for:_FOR/model_for:_SHORT_MDL:_FOR] (vol:_FIRST_SIZE),
                        type_for:_MAINWORDS $_BRND [$MDL] (vol:_FIRST_SIZE),
                        type_for:_MAINWORDS [$MDL] (vol:_FIRST_SIZE),
                        [type_for:_UP_FIRST/type_for:_UP_FIRST:_DEL_ADJ/type_for:_UP_FIRST:_MAINWORDS_FOR] $_BRND [$MDL] (vol:_FIRST_SIZE),
                        $_BRND [$MDL] (vol:_FIRST_SIZE),
                        [$_MDL/$_MDL:_SHORT_MDL/$_MDL:_MULTIDOT_END] (vol:_FIRST_SIZE),
                        $_BRND [type_for:_UP_FIRST/type_for:_UP_FIRST:_DEL_ADJ/type_for:_UP_FIRST:_MAINWORDS_FOR] brand_for (vol:_FIRST_SIZE),
                        [type_for:_UP_FIRST/type_for:_UP_FIRST:_DEL_ADJ/type_for:_UP_FIRST:_MAINWORDS_FOR] brand_for (vol:_FIRST_SIZE),
                        [type_for:_UP_FIRST/type_for:_UP_FIRST:_DEL_ADJ/type_for:_UP_FIRST:_MAINWORDS_FOR] $_BRND (vol:_FIRST_SIZE),

    ";
    return $res;
}

sub perf_title_templ_acc :STATIC :GLOBALCACHE("DERIVED") {
    my $class = shift;
    return $class->title_templ_acc;
}

sub title_templ_vol :STATIC :GLOBALCACHE("DERIVED") {
    return "[$_TYPE/$_TYPE:_DEL_ADJ] $_BRND [$MDL] vol:_FIRST_SIZE, $_BRND [$MDL] vol:_FIRST_SIZE, [$_TYPE/$_TYPE:_DEL_ADJ] [$MDL] vol:_FIRST_SIZE, [$_TYPE/$_TYPE:_DEL_ADJ] $_BRND vol:_FIRST_SIZE, [$_TYPE/$_TYPE:_DEL_ADJ] vol:_FIRST_SIZE";
}

sub perf_title_templ_vol :STATIC :GLOBALCACHE("DERIVED") {
    my $class = shift;
    return $class->title_templ_vol;
}

sub common_templates_text :STATIC :GLOBALCACHE("ARGS_HASH", "DERIVED") {
    my $class = shift;
    my %opts = @_;

    my $use_price = $opts{use_price} // 0;

    my ($title_templ, $title_templ_acc, $title_templ_quotes, $title_templ_vol);
    if ( $opts{templates_type} eq 'perf' ) {
        $title_templ = $class->perf_title_templ;
        $title_templ_acc = $class->perf_title_templ_acc;
        $title_templ_quotes = $class->perf_title_templ_quotes;
        $title_templ_vol = $class->perf_title_templ_vol;
    }
    else {
        $title_templ = $class->title_templ;
        $title_templ_acc = $class->title_templ_acc;
        $title_templ_quotes = $class->title_templ_quotes;
        $title_templ_vol = $class->title_templ_vol;
    }

    if ($use_price) {
        my $add_price = sub {
            my $header_template = shift;
            my @blocks = split(m/,\s*/, $header_template);
            my @blocks_with_price = map { "$_ price_message" } @blocks;
            return join(", ", @blocks_with_price, @blocks);
        };

        $title_templ = $add_price->($title_templ);
        $title_templ_acc = $add_price->($title_templ_acc);
        $title_templ_quotes = $add_price->($title_templ_quotes);
        $title_templ_vol = $add_price->($title_templ_vol);
    }

    my $res = "
        [type/type:_DEL_ADJ/type:_MAINWORDS] brand model => $title_templ
        [type/type:_DEL_ADJ/type:_MAINWORDS] model => $title_templ
        [type/type:_DEL_ADJ/type:_MAINWORDS] brand {___MAX_3000} => $title_templ
        brand model => $title_templ
        model => $title_templ
        [type/type:_DEL_ADJ/type:_MAINWORDS] brand quotes => $title_templ_quotes
        [type/type:_DEL_ADJ/type:_MAINWORDS] quotes => $title_templ_quotes
        brand quotes => $title_templ_quotes
        quotes => $title_templ_quotes
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] brand model => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] model => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] brand brand_for model_for => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] brand brand_for model => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] model brand_for model_for => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] brand model model_for => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] brand model_for => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] brand_for model_for => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] brand_for model => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] model_for model => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] model_for => $title_templ_acc
        brand brand_for model_for => $title_templ_acc
        brand brand_for model => $title_templ_acc
        model brand_for model_for => $title_templ_acc
        brand model model_for => $title_templ_acc
        brand_for model => $title_templ_acc
        model_for model => $title_templ_acc
        [type_for/type_for:_DEL_ADJ/type_for:_MAINWORDS] brand {___MAX_3000} => $title_templ_acc
#        [type/type:_DEL_ADJ/type:_MAINWORDS] vol:_MULT_SIZE {___MULT_PHRASE} {___MAX_10000} => $title_templ_vol
        [type/type:_DEL_ADJ/type:_MAINWORDS] vol:_FIRST_SIZE {___MAX_10000} => $title_templ_vol
#        [model/model:_SHORT_MDL] vol:_MULT_SIZE {___MULT_PHRASE} {___MAX_10000} => $title_templ_vol
        [model/model:_SHORT_MDL] vol:_FIRST_SIZE {___MAX_10000} => $title_templ_vol
    ";

    return $res;
}

sub dyn_templates_text {
    my ($class, %opts) = @_;
    $opts{templates_type} = 'dyn';
    return $class->common_templates_text(%opts);
}

sub perf_templates_text {
    my ($class, %opts) = @_;
    $opts{templates_type} = 'perf';
    return $class->common_templates_text(%opts);
}

# задаём кастомные шаблоны (например, для vip генерации)
sub set_custom_templates {
    my $self = shift;
    my ($text_templates, $title_templates) = @_;
    $self->{text_templates} = $text_templates;
    $self->{title_templates} = $title_templates;
}

# шаблоны, заданные методом set_custom_templates, имеют высший приоритет
# не кешируем, чтобы не зависеть от момента, когда были выставлены эти поля!
# (они задаются вручную и доля таких продактов мала...)
sub custom_templates_text {
    my $self = shift;

    my $text_templates = $self->{text_templates} // '';
    my $title_templates = $self->{title_templates} // '';
    return undef if !$text_templates and !$title_templates;

    my @res;
    for my $text (split /,[\r\n]+/, $text_templates) {
        $text .= ' {___MULT_PHRASE}' if ($text !~ /{___MULT_PHRASE}/);
        push @res, "$text => $title_templates";
    }
    return join "\n", @res;
}

########################################################
# Атомарные модификаторы для шаблонов
########################################################

# оставляем буквенно-цифровую последовательность слева до первого пробела, если она есть
sub _SHORT_MDL {
    my ( $self, $txt ) = @_;
    my $t = $txt;
    $t =~ s/^([-a-z]{2,}[0-9]+(?:[-a-z]+)?) .*/$1/;
    $t =~ s/^\s+|\s+$//g;
    return $t if length($t) < length ($txt);
    $t = substr($t,0,19);
    $t =~ s/\S+$//;
    $t =~ s/^\s+|\s+$//g;
    return $t if length($t) < length ($txt);
    return '';
}

sub _MULTIDOT_END {
    my ( $self, $txt ) = @_;
    return $self->proj->dse_tools->truncate_title($txt, 33) if ( length ($txt)>33 );
    return '';
}

sub _MULTIDOT_END_NON_EMPTY {
    my ( $self, $txt ) = @_;
    return $self->proj->dse_tools->truncate_title($txt, 33);
}

sub _FIRST_2_WORDS {
    my ($self, $txt) = @_;
    my @words = grep{$_} split /\s+/, $txt;
    return join(" ", @words[0..min(1, $#words)]);
}

sub _FIRST_3_WORDS {
    my ($self, $txt) = @_;
    my @words = grep{$_} split /\s+/, $txt;
    return join(" ", @words[0..min(2, $#words)]);
}

sub _FIRST_4_WORDS {
    my ($self, $txt) = @_;
    my @words = grep{$_} split /\s+/, $txt;
    return join(" ", @words[0..min(3, $#words)]);
}

sub _FIRST_5_4_3_WORDS {
    my ($self, $txt) = @_;
    my @words = grep{$_} split /\s+/, $txt;
    if (length(@words) < 3) {
        return '';
    }
    return join(" ", @words[0..min(4, $#words)]);
}

sub _TRUNCATE_TITLE {
    my ($self, $txt, $lim) = @_;
    return $self->proj->dse_tools->truncate_title($txt, $lim);
}

sub _TRUNCATE_TITLE_IGNORE_NARROW {
    my ($self, $txt, $lim) = @_;
    return $self->proj->dse_tools->truncate_title($txt, $lim, ignore_narrow => 1);
}

sub _TYPE_PREFIX {
    my ($self, $txt) = @_;
    return $self->typePrefix if ( $self->typePrefix && $self->proj->phrase($self->typePrefix)->get_goods_simple );
    return '';
}

our $unbreakable_nouns = "(гостиная|набор|машина|устройство|чаша|шкаф|средство|аппарат|комплект|панель|доска|песок|корзинка|приспособление|пузырь|система|установка|горка|круг|цепь|диск)";
sub _DEL_ADJ {
    my ( $self, $txt ) = @_;
    my $src = $txt;
    return '' unless $txt;
    return '' if $txt =~ /$unbreakable_nouns/i;
    my $adj = "(ий|ый|ые|ой|вая|ная|мая|лая|кая|ые|ие|ое|ых|их|яя)";
    my $excl = "(удобрен|свадебн)";
    my $t = join (' ', grep { /$excl/ || ! /$adj$/ } split /\s+/, $txt );

    my $preps_re = $BM::PhraseParser::preps_re;
    if ($t =~ /\ ($preps_re)$/) {
        return $src;
    }

    return "$t" if length($txt) > length($t);
    return '';
}

sub _UP_FIRST {
    my ( $self, $txt ) = @_;
    return undef unless ( $txt );
    $txt =~ s/^(.*)$/\u$1/;
    return $txt;
}

sub _DOT_END {
    my ( $self, $txt ) = @_;
    return '' unless $txt;
    return "$txt.";
}

sub _MAINWORDS {
    my ( $self, $txt ) = @_;
    return '' unless $txt;
    return '' if $txt =~ /$unbreakable_nouns/i;
    if ( $txt =~ /\ (?:для|с) /i ){
        $txt =~ s/\s+(?:для|с)\s+\S.*$//i;
        return $txt;
    }
    my $firstword = (split/\s+/, $txt)[0];
    return '' unless $firstword;
    return '' if lc($firstword) eq lc($txt);
    return $firstword if $self->proj->phrase( $firstword )->get_goods;
    return '';
}

sub _MAINWORDS_FOR {
    my ( $self, $txt ) = @_;
    return '' unless $txt =~ /\ для /i;
    $txt =~ s/^(.*\sдля)\s.*$/$1/i;
    return $txt;
}

sub _CUT_IF_NOT_TOO_SHORT {
    # обрезаем тип, только если результат получается не слишком коротким
    my ($self, $func, $txt) = @_;
    return '' unless ($func and $txt);
    if ($self->can($func)) {
        my $result = $self->$func($txt);
        return '' unless $result;
        my $words_before = (split /\s+/, $txt);
        my $words_after = (split /\s+/, $result);
        return '' if (($txt eq $result) or ($words_after <= 1));
        # разрешаем обрезать только одно слово, если больше - тип становится слишком общий
        return $result if ($words_before - $words_after >= 1);
    }
    return '';
}

sub _MAINWORDS_IF_NOT_TOO_SHORT {
    my ( $self, $txt ) = @_;
    my $res = $self->_CUT_IF_NOT_TOO_SHORT('_MAINWORDS', $txt);
    return $res;
}

sub _FOR {
    my ( $self, $txt ) = @_;
    return "для $txt";
}

sub _MULT_SIZE {
    my ( $self, $txt ) = @_;
    my @res = ();
    return $txt unless $txt =~ /\d(?:x|X|х|Х)\d/;
    my $txt1 = $txt;
    $txt1 =~ s/(?:x|X|х|Х)/х/g;
    push @res, $txt1;
    $txt1 =~ s/(?:x|X|х|Х)/x/g;
    push @res, $txt1;
    $txt1 =~ s/(?:x|X|х|Х)/ /g;
    push @res, $txt1;
    my @without_metrical = ();
    for my $part ( @res ){
        next unless $part =~ /\ /;
        my $temp = $part;
        $temp =~ s/\s+[a-zа-я]+$//i;
        push @without_metrical, $temp if ( $temp ne $part );
    }
    push @res, @without_metrical if @without_metrical;
    return "[".join(":", @res)."]";
}

sub _FIRST_SIZE {
    my ( $self, $txt ) = @_;
    $txt =~ s/\:.+?$//;
    return $txt;
}

sub _QTS {
    my ( $self, $txt ) = @_;
    return '"'.$txt.'"';
}

# модификаторы для GoogleTravel и GoogleFlights
sub _IF_ORIGIN_EMPTY {
    my ( $self, $txt ) = @_;
    return '' if ($self->{origin});
    return $txt;
}

sub _SAVE_ORDER {
    my ( $self, $txt ) = @_;
    return "[$txt] ~0";
}

sub _SAVE_ORDER_1 {
    my ( $self, $txt ) = @_;
    my @words = split ' ', $txt;
    my $s = join ' ', @words[1..scalar(@words)-1];
    return $words[0]." [$s]";
}

sub _LC {
    my ( $self, $txt ) = @_;
    return lc $txt;
}

########################################################
# Модификаторы для целой фразы, сгенерированной по шаблону
########################################################

# если есть части фразы с разделителем-мультипликатором (символ :), разваливает всю фразу по комбинациям с частями мультипликатора
# мультиплицируемая фраза должна быть в []
# результат разделяем табуляций
# разные разделители были использованы для большей наглядности и удобства отладки
# a b c:d:e => a b c\ta b d\ta b e
sub ___MULT_PHRASE {
    my ( $self, $txt ) = @_;
    my @multiplicators = ();
    @multiplicators = $txt =~ /\[(.+?)\]/g;
    return $txt unless @multiplicators;
    $txt =~ s/\[.+?\]/ /g;
    $txt = $self->proj->phrase( $txt )->pack_spaces;
    my @res = ();
    push @res, $txt;
    for my $multiplicator ( @multiplicators ){
        my @temp = ();
        for my $mword ( split/\:/, $multiplicator ){
            push @temp, "$mword $_" for @res;
        }
        @res = @temp;
    }
    return join("\t", @res);
}

sub _DEMULT {
    my ( $self, $txt ) = @_;
    if ($txt =~ /\[(.*?)\]/) {
        my $words = $1;
        my ($first, $rest) = split /:/, $words;
        $txt =~ s/\[(.*?)\]/$first/;
    }
    return $txt;
}

########################################################
# Парсинг шаблона
########################################################

# преобразует текст шаблона в массив хешей (шаблоны отдельно для фразы и тайтла)
my $cached_templates_arr = {};
sub _prepare_templates {
    my ($self, $templates_text) = @_;
    return '' unless $templates_text;

    if(!$cached_templates_arr->{$templates_text}) {
        my @arr = $self->_templates_text2arr($templates_text);
        @arr = map {{ phrase_templ => $_->[0], title_templ => $_->[1] }} map {[ split( /\s*=>\s*/, $_) ]} @arr;
        $_->{phrase_templ} = [grep {/\S/} split(/\s+/, $_->{phrase_templ})] for @arr;
        $cached_templates_arr->{$templates_text} = \@arr;
    }
    return @{$cached_templates_arr->{$templates_text}};
}

# раскрывает квадратные скобки, возвращает массив текстов шаблонов
sub _templates_text2arr {
    my ($self, $templates_text) = @_;
    my $proj = $self->proj;
    my @lines = map { s/(^\s+|\s+$)//g; $_ } grep {/\S/ && !/^\s*\#/} split "\n", ($templates_text || $self->templates_text); ##no critic

    # Размножаем правила с вариантами
    my @lines_to_expand = grep { /\[/ } @lines;
    @lines = grep { ! /\[/ } @lines;
    for my $line_to_expand ( @lines_to_expand ){
        $line_to_expand =~ s/=>(.*)//;
        my $title_template = $1;
        # раскрываем квадратные скобки в шаблоне тайтла
        $title_template =~ s/\n/ /g;
        $title_template = _expand_title_templ( $title_template ) if $title_template =~ /\[/;
        # раскрываем квадратные скобки в шаблоне фразы
        my @temp = map { $proj->phrase_list([ grep {/\S/} split /[\[\]\/]/, $_ ]) } grep {/\S/} split /(?:^| )(?=\[)|(?<=\])(?: |$)/, $line_to_expand;
        my $phl_phrases = $proj->phrase_list(['']);
        $phl_phrases *= $_ for @temp;
        push( @lines, map {"$_ => $title_template"} @$phl_phrases );
    }
    return @lines;
}

sub _expand_title_templ {
    my ($templ_text) = @_;

    my @temp = ();
    while ( $templ_text =~ /(\[|\()/ ){
        for my $templpart ( split /\s*\,\s*/, $templ_text ){
            push @temp, expand($templpart);
        }
        $templ_text = join (', ', @temp );
        @temp = ();
    }
    return $templ_text;

    sub expand {
        my ( $text ) = @_;
        if ( $text =~ /(\[.*?\])/ ){
           my $brackets = $1;
            my @parts = split /\//,substr($brackets,1,length($brackets)-2);
            $brackets = quotemeta ($brackets);
            my @temp = ();
            for my $part ( @parts ){
                my $textcopy = $text;
                $textcopy =~ s/$brackets/$part/i;
                push @temp, $textcopy;
            }
            $text = join (', ', @temp );
        }
        elsif ( $text =~ /(\(.*?\))/ ) {
           my $brackets = $1;
            my @parts = split /\//,substr($brackets,1,length($brackets)-2);
            push @parts, "";
            $brackets = quotemeta ($brackets);
            my @temp = ();
            for my $part ( @parts ){
                my $textcopy = $text;
                $textcopy =~ s/$brackets/$part/i;
                $textcopy =~ s/(^\s+|\s+$)//g;
                push @temp, $textcopy;
            }
            $text = join (', ', @temp );
        }
        return $text;
    }
}

# обычный модификатор применяется к полю
# <> статический текст, к-рый не надо разыменовывать
# {} модификатор, применяемый ко всей фразе
# Возвращает хэш:
#   { text => (результат работы), methods => (методы с rpc, которые осталось применить) }
sub _template2text { # в треугольных скобках передается статический текст, к-рый не надо разыменовывать
    my ($self, $r, $ps) = @_;

    # Если нет части полей и это не текст, то возвращаем пустую строку (модификатор поля отрезаем двоеточием)
    for my $t (@$r) {
        my $temp = $t;
        $temp =~ s/\:.*$//;
        # ищем поля в результатах парсинга или предполагаем, что это модификатор строки
        my $correct_field = ((exists $ps->{$temp}) and (defined $ps->{$temp})) || ($temp =~/^<.+>$/) || ($temp =~/^{.+}$/);
        unless ($correct_field) {
            # не нашли поля в результатах парсинга, пытаемся найти в исходных полях фида
            $correct_field = (exists $self->{_from_feed}) && (exists $self->{_from_feed}->{$temp});
            return {} unless $correct_field;
        }
    }

    my @words = ();
    my @template_modifiers = ();
    for ( @$r ){
        my ($attr, @field_modifiers) = split/\s*\:\s*/; # для каждого темплейта определяем поле и массив методов-модификаторов
        my $value;
        if (('_FROM_FEED' ~~ @field_modifiers) and (exists $self->{_from_feed})) {
            # Берём поле не из результатов парсинга ps, а из фида, то есть вызываем методы-аксессоры у self
            $value = $self->{_from_feed}->{$attr} || '';
        } else {
            $value = $ps->{$attr} || '';
        }
        if (!$value && $attr=~/^<(.+)>/){
            $value = $1;
        }
        if (!$value && $attr=~/^{(.+)}/){
            push @template_modifiers, $1;
            next;
        }
        for my $fm (@field_modifiers) {
            next if $fm eq '_FROM_FEED';
            my $mtd = $fm;
            my @args;
            if ($fm =~ /^(_TRUNCATE_TITLE\w*)_(\d+)$/) {
                $mtd = $1;
                push @args, $2;
            }
            $value = $self->$mtd($value, @args);
        }
        return {} unless $value;
        push @words, $value;
    }
    my $text = join (' ', @words );
    my @mtd;
    for my $tm ( @template_modifiers ){
        if ($tm =~ /^___MAX_(\d+)$/) {
            # не выполняем здесь, т.к. нужны RPC
            my $max_count = $1;
            push @mtd, {
                method => 'get_search_filtered',
                par => { max_count => $max_count },
            };
        } else {
            $text = join "\t", map { $self->$tm($_) } split/\t/, $text;
        }
    }
    return +{
        text => $text,
        methods => \@mtd,
    };
}

sub phrases {
    my ($self) = @_;
    my $ps = $self->parse;
    my @res = ();
    for my $tl ($self->_prepare_templates( $self->dyn_templates_text ) ){
        my $t2t = $self->_template2text($tl->{phrase_templ}, $ps);
        if ($t2t->{text}){
            push(@res, $t2t->{text});
        }
    }
    return \@res;
}

########################################################
# Генерация фраз
########################################################

sub dyn_methods_arr :GLOBALCACHE("DERIVED") {
    my ($self) = @_;
    my @res = map { s/^\s+//; $_ } grep {/\S/} grep {!/#/} ##no critic
        split /\n/, '
        modellike_product                       goods accessory P
        get_search_filtered50k                  goods accessory
        add_dynamic_homonymy_words              goods accessory P
        get_wide_filtered                       goods accessory
        postfilter_product                      goods accessory P
        snorm_phrase_list                       goods accessory
        pack_phr_lite                           goods accessory
        pack_list                               goods accessory
        get_search_filtered50k                  goods accessory
        set_exclamations_before_stops           goods accessory
        set_exclamations_before_bsstops         goods accessory
        replace_exclamations_with_pluses        goods accessory
    ';
    return @res;
}

sub perf_methods_arr :GLOBALCACHE("DERIVED") {
    my ($self) = @_;
    return $self->dyn_methods_arr;
}

# специальные методы, которые применяются в конце banners_data для всех product-ов
# могуть содержать rpc, т.ч. пока не применяем в single_banner_default
sub dyn_last_methods_arr {
    my $self = shift;
    return (
        'analyze_for_banners_data               PRODUCT',
    );
}
sub perf_last_methods_arr {
    my $self = shift;
    return (
        'store_search_count_in_minf',
    );
}

# основная работа с шаблонами: генерация фраз и заголовков
# параметры:
#   templates_text      subj
#   ps                  результат парсинга
# доп. параметры:
#   TITLE_LEN =>        из banners_data
#   LONG_TITLE_LEN =>   subj
#   ignore_narrow =>    не учитывать узкие символы при подсчёте длины
#   WORD_MAX_LEN =>     ограничение на слова тайтла
# на выходе: список хэшей с полями {phl}, {title}, {template}, ...
sub process_templates {
    my $self = shift;
    my $templates_text = shift;
    my $ps = shift;
    my %par = @_;
    my $proj = $self->proj;
    my $logger = $proj->logger;
    my $dse_tools = $self->proj->dse_tools;

    my $MIN_TITLE_LEN = 6;
    my $MIN_USE_AS_NAME_LEN = 2;
    my $WORD_MAX_LEN = $par{WORD_MAX_LEN} // 22;
    my $TITLE_LEN = $par{TITLE_LEN};
    my $LONG_TITLE_LEN = $par{LONG_TITLE_LEN};
    my $ignore_narrow = $par{ignore_narrow};
    my $need_long_title = $LONG_TITLE_LEN > 0 ? 1 : 0;
    my @templates = $self->_prepare_templates($templates_text);

    my @result;
    my %title_cache_first_level = ();
    my %title_cache_second_level = ();
    my %seen_bannerphrases = ();

    my $compatible_name = $self->proj->make_bs_compatible_or_empty($self->name);

    for my $htemplate ( @templates ){
        my $h = $self->_template2text($htemplate->{phrase_templ}, $ps);
        my $phrase_text = $h->{text} // '';
        my @mtd = ($h->{methods} and (ref $h->{methods} eq 'ARRAY')) ? @{$h->{methods}} : ();

        $logger->debug('template', $htemplate);
        $logger->debug('text:', $phrase_text, 'methods:', \@mtd);
        my $title_text   = "";
        my $second_title = "";
        my $long_title   = "";
        my $title_template = "";
        my $long_title_template = "";
        if (exists $title_cache_first_level{ $htemplate->{title_templ} } ) {
            $title_text = $title_cache_first_level{$htemplate->{title_templ}}{title_text};
            $title_template = $title_cache_first_level{$htemplate->{title_templ}}{title_result_templ};
            $second_title = $title_cache_first_level{$htemplate->{title_templ}}{second_title};
            $long_title = $title_cache_first_level{$htemplate->{title_templ}}{long_title} if $need_long_title;
            $long_title_template = $title_cache_first_level{$htemplate->{title_templ}}{long_title_result_templ} if $need_long_title;
        }
        else {
            my $is_compatible_use_as_name = '';
            if (defined $self->{use_as_name} and length($self->{use_as_name}) > $MIN_USE_AS_NAME_LEN) {
                $title_text = $dse_tools->truncate_title($self->{use_as_name}, $TITLE_LEN, ignore_narrow => $ignore_narrow);
                $long_title = $dse_tools->truncate_title($self->{use_as_name}, $LONG_TITLE_LEN, ignore_narrow => $ignore_narrow);
                $second_title = $title_text;
                $title_template = "use_as_name";
                $long_title_template = "use_as_name";
                $is_compatible_use_as_name = 1;
            }
            else {
                for my $title_templ_text ( split /\,(?![^\<]*\>)/, $htemplate->{title_templ} ){
                    my $th;
                    if ( exists $title_cache_second_level{$title_templ_text} ) {
                        $th = $title_cache_second_level{$title_templ_text};
                    }
                    else {
                        my @title_templ_arr = grep {/\S/} split /\s+/, $title_templ_text;
                        $th = $self->_template2text(\@title_templ_arr, $ps);
                        $title_cache_second_level{$title_templ_text} = $th;
                    }
                    if ($th->{methods} and @{$th->{methods}}) {
                        die "Found rpc methods in _template2text for title!\n";
                    }
                    my $title = $self->proj->make_bs_compatible_or_empty($th->{text});
                    next unless $title && length($title) > $MIN_TITLE_LEN;

                    if (!$is_compatible_use_as_name && $need_long_title && length($title) <= $LONG_TITLE_LEN &&
                        (!$long_title || length($title) > length($long_title))) {
                        $long_title = $title;
                        $long_title_template = $title_templ_text;
                    }
                    if ($title
                        and defined($TITLE_LEN) and ($dse_tools->get_length($title, ignore_narrow => $ignore_narrow) <= $TITLE_LEN)
                        and (!$WORD_MAX_LEN or get_longest_word_len($title) <= $WORD_MAX_LEN)
                    ) {
                        unless ($title_text) {
                            $title_text = $title;
                            $title_template = $title_templ_text;
                        } else {
                            $second_title ||= $title;
                            last if ($is_compatible_use_as_name || !$need_long_title);
                        }
                    }
                }
            }

            if ( !defined($self->{use_as_name}) && length($compatible_name) > $MIN_TITLE_LEN ) {
                my $name_for_title = "";
                $name_for_title = $dse_tools->truncate_title($compatible_name, $TITLE_LEN, ignore_narrow => $ignore_narrow);
                my $name_for_title_phr = $self->proj->phrase($name_for_title . " купить");
                # пробуем сматчить сгенерированный заголовок с name и, если name длиннее и содержит в себе целиком сгенерированное, заменяем на name
                # работаем только с непустыми текстами, потому что в name могут быть упячки, и вместо пустого текста его вставлять опасно
                # ignore_narrow=1 чтобы не учитывать троеточия
                # SUPBL-2057: при матчинге игнорируем слово купить
                if ( $title_text && $dse_tools->get_length($title_text, ignore_narrow => 1) < $dse_tools->get_length($name_for_title, ignore_narrow => 1) && $self->proj->phrase($title_text)->is_subphr( $name_for_title_phr ) ) {
                    $title_text = $name_for_title;
                    $title_template = 'name:_FROM_FEED';
                }
                elsif ( $second_title && $dse_tools->get_length($second_title, ignore_narrow => 1) < $dse_tools->get_length($name_for_title, ignore_narrow => 1) && $self->proj->phrase($second_title)->is_subphr( $name_for_title_phr ) ) {
                    $second_title = $name_for_title;
                }
                # для длинного заголовка - отдельная логика
                if ($need_long_title && length($compatible_name) > length($long_title) ) {
                    $long_title = $dse_tools->truncate_title($compatible_name, $LONG_TITLE_LEN);
                    $long_title_template = 'name:_FROM_FEED';
                }
            }
            my $is_need_capitalize = $title_template ne 'use_as_name';
            $title_text = fix_title($title_text, $is_need_capitalize);
            $second_title = fix_title($second_title, $is_need_capitalize);
            $long_title = fix_title($long_title, $is_need_capitalize) if $need_long_title;

            $title_cache_first_level{$htemplate->{title_templ}}{title_text} = $title_text;
            $title_cache_first_level{$htemplate->{title_templ}}{title_result_templ} = $title_template;
            $title_cache_first_level{$htemplate->{title_templ}}{second_title} = $second_title;
            $title_cache_first_level{$htemplate->{title_templ}}{long_title} = $long_title if $need_long_title;
            $title_cache_first_level{$htemplate->{title_templ}}{long_title_result_templ} = $long_title_template if $need_long_title;
            $logger->debug("cached title_text for {$htemplate->{title_templ}}: $title_text");
            $logger->debug("cached second_title for {$htemplate->{title_templ}}: $second_title");
        }
        # До этого этапа заголовок должен быть нужной длины
        if($dse_tools->get_length($title_text, ignore_narrow => $ignore_narrow) > $TITLE_LEN){
            die "Title [$title_text] length more than $TITLE_LEN!";
        }
        if($dse_tools->get_length($second_title, ignore_narrow => $ignore_narrow) > $TITLE_LEN){
            die "Title [$second_title] length more than $TITLE_LEN!";
        }

        next unless $title_text;
        $self->{first_title} //= $title_text;
        $self->{second_title} ||= $second_title;
        # дефисы не нужны и мешаются при экранировании стопслов
        $phrase_text =~ s/-/ /g;

        # если фраза была склеена мультипликатором, бьем
        my @phrases_texts = grep {$_} split/\t/, $phrase_text;
        my $phl;
        if (@phrases_texts) {
            $phl = $self->proj->phrase_list(\@phrases_texts);
            # фильтруем по количеству слов
            $phl = $phl->lgrep(sub { $_->wordcount <= $self->max_wordcount });
            #фильтруем по наличию символов, запрещенных в базе движка
            $phl = $phl->lgrep(sub { Utils::Sys::is_ucs2_compatible($_) });
        }
        else {
            $phl = $self->proj->phrase_list([]);
        }
        # текст шаблона фразы нужен для отладки и статистики
        my $phrase_template_text = join(' ', @{$htemplate->{phrase_templ}});
        $phrase_template_text = $self->ad_type.' '.$phrase_template_text if $self->ad_type;

        my $seen_str = join($;, $phl, $title_text, $need_long_title ? $long_title : "");
        next if $seen_bannerphrases{$seen_str}++;


        push @result, {
            phl => $phl,
            title => $title_text,
            title_template => $title_template,
            template => $phrase_template_text,
            methods => \@mtd,
            $need_long_title ? (long_title => $long_title): (),
            $need_long_title ? (long_title_template => $long_title_template): (),
        };
    }
    return @result;
}
sub get_longest_word_len {
    my $str = shift;
    my @words = split /\s+/, $str;
    my $max_len = 0;
    for my $word (@words) {
        $word =~ s/(^\W+|\W+$)//g;
        my $len = length($word);
        $max_len = $len if $len > $max_len;
    }
    return $max_len;
}

# основной метод генерации баннеров
# параметры, передаваемые через opts:
# methods_arr       ссылка на массив с методами для обработки
# max_count         $max: не генерировать новые пары, если их уже >= $max (по умолчанию без ограничений)
# templates_text    текст шаблонов
# ctx               контекст вычисления, запоминаем промежуточные результаты при вызове на YT
# assert_no_rpc     $flag: убеждаемся, что при обработке не используются rpc
# TITLE_LEN         максимальная длина заголовка
# LONG_TITLE_LEN    subj
# WORD_MAX_LEN      для тайтла
# ignore_narrow     не учитывать узкие символы (.,! и др) при подсчёте длины
# main_pt           "главный" продакт, передаётся из external-продактов в методы с модификатором PRODUCT
# DUMMY_PHRASE      subj
#
# возвращает либо
#   $result - список хэшей {phrase => $phrase_text, title => $title_text, template => $template_text} - результат работы метода
# либо (если есть RPC):
#   (undef,$rpc)  - список rpc запросов
#
# предполагается, что все $rpc сидят в методах $phl (methods_arr)
sub banners_data {
    my $self = shift;
    my %opts = (
        task_type => 'perf',
        @_,
    );
    my $proj = $self->proj;
    return [] unless $opts{templates_text} and $opts{methods_arr};

    $opts{TITLE_LEN} //= $self->title_len_from_type($opts{task_type});
    $opts{LONG_TITLE_LEN} //= $self->long_title_len_from_type($opts{task_type});

    my $ctx = $opts{ctx} // {};
    my $timer = $ctx->{timer} // $proj->get_new_timer;
    my $logger = $proj->logger;
    my $title_template_type = $opts{title_template_type} // 'base';
    my $title_source = $self->title_source;

    $timer->time('parse');
    if (!$ctx->{parse}) {
        # Есть шаблоны, которые не надо парсить и в которые передается готовый текст
        if ( $opts{templates_text} && ! grep {!/^\s*src:/} grep {/\S/} split /\n/, $opts{templates_text} ){
            $ctx->{parse} = { src => $self->txtsource };
        } else {
            $ctx->{parse} = $self->parse;
        }
    }
    my $ps = $ctx->{parse};

    $logger->debug('banners_data for product', $self->clean_md5, $self);
    $logger->debug('opts:', \%opts);
    $logger->debug('parse:', $ps);
    $logger->debug('ad_type:', $self->ad_type);
    if ($logger->is_debug) {
        my @caller = caller(1);
        $logger->debug('caller:', $caller[3]);
    }

    if (!$ctx->{todo}) {
        # первый вызов метода banners_data
        $timer->time('templates');
        my $templates_text = $opts{templates_text};

        my @phl_template = $self->process_templates(
            $templates_text, $ps,
            TITLE_LEN => $opts{TITLE_LEN},
            LONG_TITLE_LEN => $opts{LONG_TITLE_LEN},
            WORD_MAX_LEN => $opts{WORD_MAX_LEN},
            ignore_narrow => $opts{ignore_narrow},
        );

        my $methods_arr = $opts{methods_arr} || [];
        $methods_arr = [ grep { /\w/ && !/\#/ } @$methods_arr ];

        my @methods;
        for my $method_line (@$methods_arr) {
            my ($method, @mpar) = split /\s+/, $method_line;
            push @methods, [ $method, \@mpar ];
        }
        my @todo;
        for my $h (@phl_template) {
            # тут важно копировать, чтобы не шарить ctx между разными phl, если захотим его использовать
            my @mtd = map { +{ method => $_->[0], apply_par => $_->[1] } } @methods;
            my %long_title_h = (
                long_title => $h->{long_title},
                long_title_template => $h->{long_title_template},
                long_title_source => $title_source,
                long_title_template_type => $title_template_type,
            );
            push @todo, {
                phl => $h->{phl},
                title => $h->{title},
                title_template => $h->{title_template},
                template => $h->{template},
                methods => [ @{$h->{methods} // []}, @mtd ],  # сначала применим методы из шаблонов
                title_source => $title_source,
                title_template_type => $title_template_type,
                $opts{LONG_TITLE_LEN} ? %long_title_h: (),
            };
        }
        $ctx->{todo} = \@todo;
        if ($logger->is_debug) {
            for my $td (@todo) {
                $logger->debug(
                    'todo', "phl={$td->{phl}}", "title={$td->{title}}", "template={$td->{template}}",
                    "methods:", $td->{methods}
                );
            }
        }
    }

    my $todo = $ctx->{todo};
    my @rpc;
    TODO: for my $td (@$todo) {
        next if $td->{done};
        my $phl = $td->{phl};

        METHOD: while (@{$td->{methods}}) {
            my $h = $td->{methods}->[0];
            # в хэшрефе $h заданы:
            #   method => название метода
            #   par => хэшреф параметров, прокидывается в метод
            #   apply_par => список мета-параметров ('P','S')
            #   ctx => контекст, сейчас не используется

            my $mtd = $h->{method};
            my $apply_par = $h->{apply_par} // [];
            $timer->time("method:$mtd");

            my %par = %{$h->{par} // {}};
            if (in_array('P', $apply_par)) {
                %par = (%par, %$ps);
            }
            if (in_array('PRODUCT', $apply_par)) {
                $par{PRODUCT} = $opts{main_pt} // $self;
            }
            # сейчас $h->{ctx} не используется, кэш нужен только в одном методе (где snorm_counts, там он сделан через RPC_CACHE)
            # можно было бы его прокидывать как-то так: $par{ctx} = $h->{ctx};
            # но это нужно делать аккуратно, т.к. некоторые методы ожидают другие параметры :(
            # например: context_syns_extend и add_trade_phrases_childrens_goods

            $self->proj->rpc->flush_todo;
            my $res_phl = eval { $phl->$mtd(%par) };
            if ($@) {
                my $msg = $proj->rpc->die_msg;
                die $@ if $@ !~ /$msg/;  # нештатное исключение
                push @rpc, @{$proj->rpc->todo};
                if (@rpc and $opts{assert_no_rpc}) {
                    die "Unexpected rpc in banners_data!\n";
                }
                next TODO;
            }
            $logger->debug("apply methods: $mtd (@$apply_par): $td->{phl} ===> $res_phl") if $logger->is_debug;

            # если 'S', то добавляем результат вызова метода
            if (in_array('S', $apply_par) ){
                $phl += $res_phl;
            } else{
                $phl = $res_phl;
            }
            $td->{phl} = $phl;
            shift @{$td->{methods}};
        }

        # оставляем только информацию, нужную для дальнейшего (для ускорения серилизации/десериализации)
        $td->{done} = 1;
        $td->{phrases_data} = [ map { [ $_->text, $_->minf ] } @{$td->{phl}} ];
        delete $td->{phl};
        delete $td->{methods};
    }

    return (undef, \@rpc) if @rpc;

    my @res;
    for my $td (@{$ctx->{todo}}) {
        for my $phr_minf (@{$td->{phrases_data}}) {
            my ($phr, $minf) = @$phr_minf;
            my %long_title_h = (
                long_title => $td->{long_title},
                long_title_source => $title_source,
                long_title_template => $td->{long_title_template},
                long_title_template_type => $title_template_type,
            );
            my $bnr = {
                phrase => $phr,
                title => $td->{title},
                title_template => $td->{title_template},
                template => $td->{template},
                title_source => $title_source,
                title_template_type => $title_template_type,
                $opts{LONG_TITLE_LEN} ? %long_title_h: ()
            };
            $bnr->{minf} = dclone($minf) if keys %$minf;
            push @res, $bnr;
        }
    }

    # добавляем к каждому офферу, к которому сгенерился непустой тайтл, фразу-заглушку из заголовка (одну на оффер)

    my $dummy_res = undef;
    my $dummy_title_word_count = 0;
    for my $td (@{$ctx->{todo}}) {
        if ($td->{title}) {
            # выбираем заголовок с максимальным числом слов (и сохраняем только те, что длиннее 1 слова)
            # костыль запрошен здесь: https://st.yandex-team.ru/DYNSMART-791#1535371044000
            my $title_word_count = scalar split m/\s+/, $td->{title};
            if (
                $title_word_count > 1
                    and $title_word_count > $dummy_title_word_count
            ) {
                my $dummy_phrase = $opts{DUMMY_PHRASE} || $self->get_phrase_from_title($td->{title});
                if ($dummy_phrase) {
                    my $template = $opts{DUMMY_PHRASE} ? $td->{template}: 'phrase_from_title';
                    my %long_title_h = (
                        long_title => $td->{long_title},
                        long_title_source => $title_source,
                        long_title_template => $td->{long_title_template},
                        long_title_template_type => $title_template_type,
                    );
                    $dummy_res = {
                        phrase              => $dummy_phrase,
                        title               => $td->{title},
                        title_template      => $td->{title_template},
                        template            => $template,
                        title_source        => $title_source,
                        title_template_type => $title_template_type,
                        $opts{LONG_TITLE_LEN} ? %long_title_h: ()
                    };
                    $dummy_title_word_count = $title_word_count;
                }
            }
        }
    }
    push @res, $dummy_res if defined($dummy_res);


    # для детерминированности сортируем по хэшу
    @res = map { $_->[0] } sort { $a->[1] <=> $b->[1] } map {
        [ $_, md5int(join("\t", $_->{phrase}, $_->{title}, $_->{template})) ]
    } @res;

    # удалим дубли по (phrase, title)
    my %seen_phrase_title;
    @res = grep { !$seen_phrase_title{$_->{phrase}."\t".$_->{title}}++ } @res;

    my $max_count = $opts{max_count};
    if ($max_count and @res > $max_count) {
        $logger->debug("max count [$max_count] hit; was: ".@res);
        @res = splice(@res, 0, $max_count);
    }

    return \@res;
}

sub max_wordcount {
    my ($self) = @_;
    return 6;
}

sub get_phrase_from_title {
    my ($self, $title) = @_;
    my $match_type = $self->match_type() || 'norm';
    $title =~ s/-/ /g;
    my @words = $self->proj->phrase($title)->normalize($match_type => 1, filt_stop => 1, uniq => 1, sort => 1);
    return undef unless scalar(@words);

    if ($self->{minus_words}) {
        my $domain = lc $self->proj->page($self->url)->domain_2lvl;
        $domain =~ s/\.\w+$//;
        my $domain_translit = $self->proj->phrase($domain)->translit_simple;
        if ($self->{minus_words}->{$domain} || $self->{minus_words}->{$domain_translit}) {
            @words = grep {$_ ne $domain && $_ ne $domain_translit} @words;
        }
    }
    return undef if scalar(@words) < 2;
    my $phrase = join(' ', @words);
    if (scalar(@words) < 5) {
        $phrase .= ' ~0'
    }
    return $phrase
}

# В perf_banners/dyn_banners/banners_sinle/... методах нужно:
# - передать параметр ctx дальше в banners_data, можно передать весь @_
# - учесть, что banners_data может вернуть либо $arr, либо (undef,$rpc), поэтому возвращаем его return
# - следовательно, perf_banners/dyn_banners-методы также могут возвращать (undef,$rpc), нужно учитывать это во всех Product-классах
#
# Если мы уверены, что в banners_data не будет rpc, то можно всего этого не делать, но задать параметр (assert_no_rpc=>1)
# Считаем, что в методе single_banner_default не используются rpc, и его можно спокойно вызывать.

# TODO(malykhin): начать передавать все параметры в single_banner_default
my @SINGLE_BANNER_PAR = qw(TITLE_LEN LONG_TITLE_LEN ignore_narrow DUMMY_PHRASE);

sub dyn_banners {
    my $self = shift;
    my %par  = @_;

    # цена в заголовках - эксперимент (только динамические, только eldorado.ru, см. DYNSMART-609)
    # в течение пары месяцев после коммита должно быть либо удалено, либо переделано по-нормальному, без хардкода
    # breqwas@, 08.04.2018
    my $domain = lc $self->proj->page($self->url)->domain_2lvl;
    my $use_price = ($domain eq 'eldorado.ru');
    my @methods_arr = $self->dyn_methods_arr;
    push @methods_arr, $self->dyn_last_methods_arr;

    my $add_last_methods_arr = delete($par{add_last_methods_arr}) // [];
    push @methods_arr, @$add_last_methods_arr;

    my ($arr, $rpc) =  $self->banners_data(
        task_type => 'dyn',
        templates_text => $self->dyn_templates_text(use_price => $use_price),
        methods_arr => \@methods_arr,
        %par,
    );
    return (undef, $rpc) if !defined $arr;
    return $arr if @$arr;
    my  $res_arr = [];
    my %single_par = map { $_ => $par{$_} } grep { defined $par{$_} } @SINGLE_BANNER_PAR;
    $arr = $self->single_banner_default('dyn', %single_par);
    if ($arr and @$arr) {
        for my $banner_phrase (@$arr) {
            my $template = $par{DUMMY_PHRASE} ? $banner_phrase->{template}: 'phrase_from_title';
            my $phrase = $par{DUMMY_PHRASE} ? $par{DUMMY_PHRASE}: $self->get_phrase_from_title($banner_phrase->{title});
            if ($phrase) {
                $banner_phrase->{phrase} = $phrase;
                $banner_phrase->{template} = $template;
                push @$res_arr, $banner_phrase;
            }
        }
    }
    return $res_arr;
}

sub perf_banners {
    my $self = shift;
    my %par  = @_;

    my $templates_text = $self->custom_templates_text // $self->perf_templates_text;
    my @methods_arr = $self->perf_methods_arr;
    push @methods_arr, $self->perf_last_methods_arr;

    my $add_last_methods_arr = delete($par{add_last_methods_arr}) // [];
    push @methods_arr, @$add_last_methods_arr;

    my ($arr, $rpc) = $self->banners_data(templates_text => $templates_text, methods_arr => \@methods_arr, %par);
    return (undef, $rpc) if !defined $arr;
    return $arr if @$arr;
    my $res_arr = [];

    my %single_par = map { $_ => $par{$_} } grep { defined $par{$_} } @SINGLE_BANNER_PAR;
    $arr = $self->single_banner_default('perf', %single_par);
    if ($arr and @$arr) {
        for my $banner_phrase (@$arr) {
            my $template = $par{DUMMY_PHRASE} ? $banner_phrase->{template}: 'phrase_from_title';
            my $phrase = $par{DUMMY_PHRASE} ? $par{DUMMY_PHRASE}: $self->get_phrase_from_title($banner_phrase->{title});
            if ($phrase) {
                $banner_phrase->{phrase} = $phrase;
                $banner_phrase->{template} = $template;
                push @$res_arr, $banner_phrase;
            }
        }
    }
    return $res_arr;
}

# Метод без rpc
sub single_banner_default {
    my ($self, $type, %params) = @_;

    # DYNSMART-354: набор шаблонов для случаев, когда мы не смогли ничего выпарсить
    # шаблон с model:_MAINWORDS присутствует потому что иногда в model пишут тип
    my $title_len = $params{TITLE_LEN} || $self->title_len_from_type($type);
    my $ignore_narrow = $params{ignore_narrow};

    my @additional_methods = @{$params{'additional_methods'} // []};
    my $name = "name:_FROM_FEED";
    my $vendor = "vendor:_FROM_FEED";
    my $model = "model:_FROM_FEED";
    my $typePrefix = "typePrefix:_FROM_FEED";
    my $typePrefix_mainwords = "typePrefix:_FROM_FEED:_MAINWORDS";
    my $typePrefix_del_adj = "typePrefix:_FROM_FEED:_DEL_ADJ";
    my $TYPE = "$typePrefix/$typePrefix_del_adj/$typePrefix_mainwords";
    my @first_methods;
    my $TRUNCATE = ($ignore_narrow ? '_TRUNCATE_TITLE_IGNORE_NARROW' : '_TRUNCATE_TITLE') . "_$title_len";
    @first_methods = (
        "$name:_FIRST_4_WORDS => $name:$TRUNCATE",
        "$name:_FIRST_3_WORDS => $name:$TRUNCATE",
        "$name:_FIRST_2_WORDS => $name:$TRUNCATE",
    );
    my @templates = (
        @first_methods,
        "[$TYPE] $vendor $model {___MULT_PHRASE} => [$TYPE] $vendor $model",
        "[$TYPE] $vendor $model:_MAINWORDS {___MULT_PHRASE} => [$TYPE] $vendor $model:_MAINWORDS_IF_NOT_TOO_SHORT",
        "$vendor $model {___MULT_PHRASE} => $vendor $model",
        "[$TYPE] $vendor {___MULT_PHRASE} => $typePrefix $vendor, $typePrefix_mainwords $vendor",
        "src:_FIRST_4_WORDS => src:$TRUNCATE",
        "$vendor $model:_DEL_ADJ => $vendor $model:_DEL_ADJ",
        "$vendor $model:_MAINWORDS => $vendor [$model:_MAINWORDS_IF_NOT_TOO_SHORT/$model:_MAINWORDS]",
        "src:_FIRST_3_WORDS => src:$TRUNCATE",
        "src:_FIRST_2_WORDS => src:$TRUNCATE",
    );

    my $arr = [];
    my $methods_arr = ['pack_list'];
    push @$methods_arr, @additional_methods if @additional_methods;
    while ( not @$arr and @templates ) {
        my $templ = shift @templates;
        $arr = $self->banners_data(
            task_type           => $type,
            methods_arr         => $methods_arr,
            max_count           => 1,
            templates_text      => $templ,
            assert_no_rpc       => 1,
            title_template_type => 'fallback',
            %params,
        );
        if ($arr && @$arr && $self->is_title_has_dup_words($arr->[0]->{title})) {
            $arr = [];
        }
    }

    return @$arr ? [ $arr->[0] ] : [];
}

sub is_title_has_dup_words {
    my ($self, $title) = @_;
    $title =~ s/\s+/ /g;
    $title =~ s/(^\s+|\s+$)//g;
    my $lc_title = lc $title;
    my $title_phr = $self->proj->phrase($lc_title);
    my $brand = $title_phr->get_brand();

    if ($brand) {
        # Check brand duplicates
        my @matches = $lc_title =~ /($brand)/g;
        if (scalar @matches > 1) {
            return 1;
        }
        # Remove brands
        $lc_title =~ s/($brand)//;
    }
    my $volumes_re = '\b\d+(?:\s(?:x|х|\*|на)\s\d+){1,2}\b'; # volumes with whitespaces like (100 x 100), (5 на 5)
    my @volumes_matches = $lc_title =~ /($volumes_re)/g;
    my %volumes_h = ();
    for my $volume (@volumes_matches) {
        return 1 if ($volumes_h{$volume});
        $volumes_h{$volume} = 1;
    }

    $lc_title =~ s/$volumes_re//g;
    my @words = split /\s+/, $lc_title;
    my %words_h = ();
    for my $word (@words) {
        $word =~ s/\W//g;
        if ($word) {
            if ($words_h{$word}) {
                return 1;
            }
            $words_h{$word} = 1;
        }
    }
    return '';
}

# default title lengths
sub title_len_from_type {
    my ($self, $type) = @_;
    return 56 if $type eq 'perf';
    return 56 if $type eq 'dyn';
    die 'wrong type';
}

sub long_title_len_from_type {
    my ($self, $type) = @_;
    return 150 if $type eq 'perf';
    return 0 if $type eq 'dyn';
    die 'wrong type';
}

sub banner_single_templates_text {
    my ($self, $type) = @_;
    my $title_templ;
    if ( $type eq 'perf' ) {
        $title_templ = $self->perf_title_templ;
    }
    else {
        $title_templ = $self->title_templ;
    }
    return "model:_DEMULT => $title_templ\ntype:_DEMULT => $title_templ";
}

sub banner_single_methods_arr {
    my $self = shift;
    return ('pack_list');
}

sub banner_single {
    my $self = shift;
    my $type = shift;
    my %par  = @_;

    my $templates_text = $self->custom_templates_text // $self->banner_single_templates_text($type);
    my @methods_arr = $self->banner_single_methods_arr;
    my $add_last_methods_arr = delete($par{add_last_methods_arr}) // [];
    push @methods_arr, @$add_last_methods_arr;

    my ($arr) = $self->banners_data(
        task_type => $type,
        templates_text => $templates_text,
        methods_arr => \@methods_arr,
        max_count => 1,
        title_template_type => 'single',
        assert_no_rpc => 1,
        %par,  # ctx
    );
    return [ $arr->[0] ] if (@$arr);
    return $self->single_banner_default($type, %par);  # предполагаем, что здесь нет rpc (а parse уже был сделан выше!)
}

sub perf_banners_single {
    my $self = shift;
    return $self->banner_single('perf', @_);
}

sub dyn_banners_single {
    my $self = shift;
    return $self->banner_single('dyn', @_);
}

sub perf_banners_by_name {
    my $self = shift;
    my %par  = @_;
    my $res_arr = [];
    my $arr = $self->single_banner_default('perf', %par);
    if ($arr and @$arr) {
        for my $banner_phrase (@$arr) {
            my $template = $par{DUMMY_PHRASE} ? $banner_phrase->{template} : 'phrase_from_title';
            my $phrase = $par{DUMMY_PHRASE} ? $par{DUMMY_PHRASE} : $self->get_phrase_from_title($banner_phrase->{title});
            my $title_template = $banner_phrase->{title_template};
            if ($phrase && ($title_template eq 'use_as_name' || $title_template =~ m/^name/ )) {
                $banner_phrase->{phrase} = $phrase;
                $banner_phrase->{template} = $template;
                push @$res_arr, $banner_phrase;
            }
        }
    }
    return @$res_arr ? [ $res_arr->[0] ] : [];
}

sub minus_words {
    my ($self) = @_;
    return '';
}

sub valid_price {
    my ($self, $price) = @_;
    $price = '' if !defined($price);
    $price =~ s/\x2c/\x2e/;            # ',' -> '.'
    $price =~ s/[^\x30-\x39\x2e\-]//g; # оставляем только цифры, точку и минус
    return '' if ($price !~ m/^\d+(\.\d+)?$/);
    if ($price =~ m/^\d+\.\d+$/) {
        $price = sprintf("%0.2f", $price);
        $price =~ s/0+$//;
        $price =~ s/\.$//;
    }
    return $price . '';
}

sub get_copy_additional_data { # защищаемся от изменения additional_data (выстрелил autovivification)
    my $self = shift;
    if (ref($self->{additional_data})) {
        return dclone($self->{additional_data});
    } else {
        return {};
    }
}

sub get_valid_price : CACHE {
    my $self = shift;
    return $self->valid_price($self->{price}) unless $self->get_copy_additional_data->{price}->{current};
    return $self->valid_price($self->get_copy_additional_data->{price}->{current});
}

sub get_valid_oldprice : CACHE {
    my $self = shift;
    return $self->valid_price($self->{oldprice}) unless $self->get_copy_additional_data->{price}->{old};
    return $self->valid_price($self->get_copy_additional_data->{price}->{old});
}

sub get_parsed_params : CACHE {
    my $self = shift;
    return $self->get_copy_additional_data->{text}->{params_for_direct};
}

sub get_facilities : CACHE {
    my $self = shift;
    return $self->get_copy_additional_data->{text}->{facilities} if $self->get_copy_additional_data->{text}->{facilities};
    return $self->{facilities};
}

sub get_class : CACHE {
    my $self = shift;
    return $self->{class} ? int($self->{class}) : undef unless $self->get_copy_additional_data->{class};
    return $self->get_copy_additional_data->{class} ? int($self->get_copy_additional_data->{class}): undef;
}

sub get_location : CACHE {
    my $self = shift;
    return $self->{location} if $self->{location};
    return $self->get_copy_additional_data->{text}->{location};
}

sub get_description : CACHE {
    my $self = shift;
    return $self->{description} unless $self->get_copy_additional_data->{text}->{description_for_direct};
    return $self->get_copy_additional_data->{text}->{description_for_direct};
}

sub fix_img_url {
    my $self = shift;
    my $url = shift;

    return if !$url;
    $url = fix_url_scheme($url);
    return if !$self->proj->validate_url($url);

    # в картике требуем 80 443 порт, аватарница не умеет получать картинку по другим урлам
    my $port = URI->new($url)->port;
    return if $port != 80 and $port != 443;
    return url_to_punycode($url);
}

sub get_fixed_images :CACHE {
    my $self = shift;
    my $imgs = $self->{images} or return;

    my @src_img;
    if (ref($imgs) eq 'ARRAY') {
        @src_img = @{$imgs};
    } else {
        @src_img = split /\s*\,\s*/, $imgs;
    }

    my @img;
    for my $img (@src_img) {
        $img = $self->fix_img_url($img) or next;
        push @img, $img;
    }
    return \@img;
}

sub get_fixed_picture :CACHE {
    my $self = shift;
    return $self->fix_img_url($self->{picture});
}

my %region_inf_cache;  # глобальное кеширование
sub get_region_inf {
    my $self = shift;
    my $location = $self->{location};
    return $region_inf_cache{$location} if $region_inf_cache{$location};

    my @rgs = $self->proj->phrase($location)->get_geobase_region_ids;
    my %reginf = (
        geoids => join(',', @rgs),
        maingeoid => $rgs[0],
    );
    $region_inf_cache{$location} = \%reginf;
    %region_inf_cache = () if keys %region_inf_cache > 1e6;  # prevent overflow; better to use LRU cache

    return \%reginf;
}

# IRTDUTY-179: nullify fields if the client made mistake with CDATA encoding
sub nullify_fields_with_cdata {
    my $self = shift;
    $self->{description} = '' if $self->{description} and $self->{description} =~ /CDATA/;
    $self->{use_as_body} = '' if $self->{use_as_body} and $self->{use_as_body} =~ /CDATA/;
    $self->{use_as_name} = '' if $self->{use_as_name} and $self->{use_as_name} =~ /CDATA/;
    if ($self->{additional_data} and
            $self->{additional_data}->{text} and
            $self->{additional_data}->{text}->{description_for_direct} and
            $self->{additional_data}->{text}->{description_for_direct} =~ /CDATA/) {
        $self->{additional_data}->{text}->{description_for_direct} = '';
    }
}

sub fix_title {
    my $title_text = shift;
    my $is_need_capitalize = shift;
    return '' unless $title_text;
    $title_text =~ s/^(.*)$/\u$1/ if $title_text !~ m/^\d/ && $is_need_capitalize; # выставляем верхний регистр для первого нечислового символа
    $title_text =~ s/(\s+)(!|\.|,|\?)/$2/g; # удаляем пробелы перед знаками препинания
    $title_text =~ s/\. \b(\w)/\. \U$1/g if $is_need_capitalize; # После точки пишем с большой буквы
    $title_text =~ s/\s*[(:,]*\s*$//g; # кривые знаки припинания с конца
    return $title_text;
}

##########################################################

use overload
    '""' => sub {
            my ($self) = @_;
            return $self->{name}." => ".($self->{url} || '');
        };

1;
