use utf8;
use strict;
use warnings;
package BM::BannersMaker::ProductAutoAccessories;

use base qw(BM::BannersMaker::Product);
use Utils::Array;
use Data::Dumper;
use List::Util qw(min max);
use Utils::Sys qw(uniq);

__PACKAGE__->mk_accessors(qw(
    type
    brand
    season
));

my @offer_fields = qw(model vendorCode);

sub get_offer_fields {
    return @offer_fields;
}

sub init {
    my ($self) = @_;
    $self->SUPER::init;
    my $data = $self->{data};
    if(ref($data->[0]) eq 'HASH'){
        $self->{$_} = $data->[0]{$_} for @offer_fields;
    }
}

########################################################
#Доступ к полям
########################################################

sub ad_type {
    return 'auto_accessories';
}

sub match_type {
    return 'norm';
}


sub get_auto_brands_list {
    my $self = shift;
    return ('Nissan', 'Hyundai', 'Ford', 'Kia', 'Toyota', 'Chevrolet', 'Mercedes', 'BMW', 'Renault', 'Mitsubishi', 'Volkswagen', 'Audi', 'Honda', 'Mazda', 'Opel', 'Citroen', 'Skoda', 'Suzuki', 'Infiniti', 'Lexus', 'Subaru', 'Peugeot', 'Chery', 'Volvo', 'Daewoo', 'Geely', 'Lifan', 'VW', 'Ssang Yong', 'Acura', 'Cadillac', 'Dodge', 'Lada', 'Alfa Romeo', 'Fiat', 'Great Wall', 'Land Rover', 'Jeep');
}


sub get_types {
    my ($self, $type, $source) = @_;

    $type = $self->check_type($type, $source);
    my $res = {type => $type, types => ''};

    if ($type and scalar(split /\s+/, $type) > 2) {
        my @current_list = ('');
        foreach my $word (grep {$_} split /\s+/, $type) {
            if ($word =~ /^[а-яё]+(?:ых|их|ое|ая|ый|ий)$/i) {
                @current_list = map {($_, "$_ $word")} @current_list;
            } else {
                @current_list = map {"$_ $word"} @current_list;
            }
        }
        my @result = @current_list;
        $_ =~ s/(^\s+|\s+$)//g for @result;

        my $index = (grep {scalar(split /\s+/, $result[$_]) >= 2} 0..$#result)[0] || 0;
        if ($index < $#result) {
           $res->{type} = $result[$index];
           my @types = grep {$_} map {$result[$_]} ($index+1..$#result);
           $res->{types}  = @types > 1 ? '[' . join(':', @types) . ']' : @types == 1 ? $types[0] : '';
        }
    }
    return ($res->{type}, $res->{types});
}


sub check_type {
    my ($self, $type, $source) = @_;

    $type =~ s/[^a-zа-яё\d\-]+$//i;
    $source =~ s/\s+/ /g;

    my $typePrefix = $self->{typePrefix} || '';
    my $categpath = $self->{categpath} || '';

    if ($typePrefix and scalar(split /\s+/, $type) > 2 and $typePrefix =~ /^(.*?)\s+((?:[А-ЯЁ][а-яё]+\s*)?\(|[a-zA-Z\d]+)/) {
        $type = $1;
    }

    if ($categpath =~ /\s*(присадки)\s*\/\s*(.*?)\s*\/\s*для\s+([а-яё\-]+(?:\s+[а-яё\-]+)*)\s*$/i) {
        $self->proj->logger->debug('found in categpath:', $1, $2, $3);
        $type = "присадка для $3";
    } elsif ($categpath =~ /\/\s*(масла)[^а-яё]*$/i and $type !~ /масл/i) {
        $type = "масло";
    }

    if ($type and ($source =~ /(Супротек)\s+(\Q$type\E)/i or
                   $source =~ /(Супротек)\s+/i and $type =~ /^\s*(двигатель|редуктор|ГУР|КПП|МКПП|АКПП|ТНВД|ДВС)/i)) {
        $type = '';
    }

    my $prepositions_re = 'в|к|у|с|а|на|от|со|до|за|но|для|или|под|подо|над|без|и';
    if ($type =~ /^\s*([а-яё]+)\s+([а-яё]+)/i) {
        my ($first, $second) = ($1, $2);
        if ($source =~ /\b(\Q$first\E\s+($prepositions_re)\s+\Q$second\E)\b/i) {
            $type = $1;
        }
    }
    if (length($type) > 30 and scalar(split /\s+/, $type) > 4 and $type =~ /^((?:[а-яё\-]+\s+){2,})($prepositions_re)\b/i) {
        $type = $1;
    }elsif (length($type) < 4) {
        $type = '';
    }

    my $adjectives_re = '[а-яё]+(?:ый|ий|ой|ая|яя|ое|ее|ых|ые|ого)';
    my $abbreviations_re = join('|', map {quotemeta($_)} ('ГУР', 'КПП', 'МКПП', 'АКПП', 'ТНВД', 'ДВС'));

    $type =~ s/\s+($prepositions_re)(\s+($prepositions_re|$adjectives_re))*\s*$//i;
    $type =~ s/\s+([а-яё]+(?:ых|ого))\s*$//i;
    $type =  lc($type);
    $type =~ s/\b($abbreviations_re)\b/\U$1\E/gi;
    return $type;
}


sub get_auto_data {
    my ($self, $text) = @_;
    my $res = {};

    my @auto_brands_list = $self->get_auto_brands_list;
    my $auto_brands_re = join('|', map{quotemeta($_)} @auto_brands_list);

    if ($text =~ s/\b($auto_brands_re)\b(.*)$//i) {
        ($res->{auto_brand}, my $rest) = ($1, $2);
        $rest =~ s/\s+c\s*(\d+)/ с $1/g;

        my $auto_model_re = '[a-z\d\-]+';
        my $auto_model_rest_re = '[a-z]+[\-\d]*|\d{1,3}';
        if ($rest =~ s/^\s+($auto_model_re)((?:\s+(?:$auto_model_rest_re))*)\b/ /i and lc($1) ne lc($res->{auto_brand})) {
            ($res->{auto_model}, $res->{auto_model_rest}) = ($1, $2);
        }
        ($rest, $res->{years}) = $self->get_years($rest);
        $text .= $rest;
    }
    return ($text, $res);
}


sub get_years {
    my ($self, $text) = @_;
    my $years = [];

    my ($year_from, $year_to);
    my $year_re = '(199\d|20[0-1]\d)(?:\s*гг?\s*\.?)?';
    my $cur_year = (localtime)[5] + 1900;

    #2003- 2007г
    if ($text =~ s/\(?\s*(?:$year_re)\s*\-\s*(?:$year_re)\s*\)?/ /i) {
        ($year_from, $year_to) = ($1, $2);
    #(2009-н.в.)
    } elsif ($text =~ s/\(?\s*(?:$year_re)\s*\-\s*(?:н\.в\.)\s*\)?/ /i) {
        ($year_from, $year_to) = ($1, $cur_year);
    #с 2007 г.
    } elsif ($text =~ s/\(?\s*[c|с]\s*(?:$year_re)\s*/ /i) {
        ($year_from, $year_to) = ($1, $cur_year);
    #(2007-н., (2002-200
    } elsif ($text =~ s/\(?\s*(?:$year_re)\s*(?:\-\s*(?:\d+(?:\s*г\s*\.?)?|н[\.|в]*))?\s*\)?/ /i) {
        ($year_from, $year_to) = ($1, $1);
    }

    if ($year_from and $year_to) {
        $years = [$year_from .. $year_to];
        $self->proj->logger->debug('years:', $years);
    }
    return ($text, $years);
}


sub get_auto_brand_for_templ {
    my ($self, $auto_brand) = @_;

    if ($auto_brand and $auto_brand =~ /^\s*[a-z\s]+\s*$/i) {
        my $auto_brand_snorm = $self->proj->phrase($auto_brand)->snorm_phr;
        $auto_brand = "[$auto_brand:$auto_brand_snorm]" if lc($auto_brand_snorm) ne lc($auto_brand);
    }
    return $auto_brand;
}


sub get_auto_model_for_templ {
    my ($self, $auto_data) = @_;

    my $auto_model_res;
    if ($auto_data->{auto_model} and (scalar (split /\s+/, $auto_data->{auto_model}) <= 2)) {
        $auto_model_res = $auto_data->{auto_model};
    }

    if ($auto_data->{auto_model}) {
        my @auto_models = ($auto_data->{auto_model});
        if ($auto_data->{auto_model} =~ /^\s*[a-z]+\s*$/i) {
            my $auto_model_snorm = $self->proj->phrase($auto_data->{auto_model})->snorm_phr;
            push @auto_models, $auto_model_snorm if lc($auto_model_snorm) ne lc($auto_data->{auto_model});
        }
        my @res;
        foreach my $auto_model (@auto_models) {
            push @res, $auto_model if ($auto_model and (scalar (split /\s+/, $auto_model) <= 2));
            if ($auto_data->{years} and @{$auto_data->{years}}) {
                push @res, map {"$auto_model $_"} grep {$_} @{$auto_data->{years}};
            }
            if ($auto_data->{auto_model_rest} and my @auto_model_rest_parts = grep {$_} split /\s+/, $auto_data->{auto_model_rest}) {
                push @res, map {"$auto_model @auto_model_rest_parts[0..$_]"} (0..$#auto_model_rest_parts);
            }
        }
        $auto_model_res = '[' . join(':', @res) .']' if @res > 1;
    }
    return $auto_model_res;
}


sub parse :CACHE {
    my ($self) = @_;
    my $logger = $self->proj->logger;

    my $h = $self->SUPER::parse;
    $logger->debug('parse: SUPER::parse:', $h);
    my $model_from_parse = '';
    if (exists $self->{vendorCode}) {
        $model_from_parse = $h->{model};
        $h->{model} = $self->{vendorCode} if (exists $self->{vendorCode});
    }

    my $txtsource = $self->txtsource || '';

    if ($txtsource !~ /[а-яё]{4,}/i and $self->{description} and $self->{description} =~ /[а-яё]{4,}/i) {
        $txtsource .= " " . join(' ', grep {$_} map {$self->{$_}} qw(typePrefix description));
    }
    $logger->debug("txtsource: $txtsource");

    $h->{$_} ||= '' for qw(model type brand model_weak);

    my $rest = $txtsource;
    $rest =~ s/([а-яё]+)\s+д\/([а-яё]+)/$1 для $2/i;
    $rest =~ s/ё/е/ig;

    if ($h->{class} eq 'accessory' and $h->{type_for}) {
        $h->{type} ||= $h->{type_for};
    }
    ($h->{type}, $h->{types}) = $self->get_types($h->{type}, $rest);

    my @brands_list = ('AVS', 'SeinTex', 'Spark', 'Carav', 'HITO', 'Liqui Moly', 'Novline', 'Blackview', 'SVS', 'Defen Time', 'Hella', 'SilverStar', 'ВМП Авто', 'NORDIS', 'MTF Light', 'Killaqua', 'РМ', 'PSV', 'Amos');
    my $brands_re = join('|', map {quotemeta($_)} @brands_list);
    if (my ($brand) = map {$_ =~ /\b($brands_re)\b/i} grep {$_} ($h->{brand}, $self->{vendor}, $rest)) {
        $h->{brand} = $brand;
        $logger->debug("found brand: $brand");
    }

    $h->{model} ||= $h->{model_weak};

    #определяем detail_number, carcase_type, viscosity, material, auto_words
    if ($rest =~ s/([A-Z]*?\d+[A-Z\d]{7,}|[A-Z]+\-?\d+[A-Z\d\-]{5,}|\d+\-?[A-Z][A-Z\d\-]{5,})/ /) {
        $h->{detail_number} = $1;
    }

    my $carcase_type_re = join('|', map {quotemeta($_)} 'hatchback', 'sedan', 'седан', 'хэтчбек', 'универсал', 'хетчбек', 'хетчбэк', 'хэтчбэк', 'кроссовер', 'минивэн', 'внедорожник');
    if ($rest =~ s/\b($carcase_type_re)\b/ /i) {
        $h->{carcase_type} = $1;
    }

    if ($rest =~ s/(?:^|\s+)((\d+w)\-?(\d+))(?:\s+|$)/ /i) {
        $h->{viscosity} = "$2$3";
        $h->{model} =~ s/\b\Q$1\E\b/ /i;
    }

    my @properties_for_model;
    #H1 4300К (Цоколь: H1, Температура: 4300К)
    if ($rest =~ /\b(HB?\d{1,2})\b/ and $h->{model} !~ /\Q$1\E/) {
        push @properties_for_model, $1;
    }
    if ($rest =~ /\b((\d{4})[KК])\b/ and $h->{model} !~ /\Q$1\E/) {
        push @properties_for_model, "$2K";
    }

    my $material_re = join('|', map {quotemeta($_)} 'полиуретан', 'сталь', 'алюминий', 'кожа', 'экокожа', 'мех') . '|'
                    . join('|', map {$_ =~ /^([а-яё]+)(ый|ой)$/i ? "$1(?:ый|ой|ая|ые)" : ()} 'полиуретановый', 'резиновый', 'текстильный', 'кожаный', 'меховой');
    if ($h->{type} =~ s/(?:^|\s+)($material_re)(?:\s+|$)/ /i or $rest =~ s/(?:^|\,|\(|\s+)($material_re)(?:\s+|\,|\)|$)/ /i) {
        $h->{material} = $1;
    }

    my @auto_words = ('авто', 'машина', 'автомобиль');
    my $auto_words_re = join('|', map {quotemeta($_)} @auto_words);
    if ($h->{minicategs} =~ /($auto_words_re)/i and $h->{type} !~ /авто/i and (!$h->{types} or $h->{types} !~ /авто/i)) {
        # экранируем слова авто и автомобиль, иначе в context_syns_extend эти слова "прилипают" к другим
        $h->{auto_words} = '[' . join(':', map { my $r = $_; $r =~ s/^(авто.*)$/!$1/i; $r; }  @auto_words) . ']';
    }


    #определяем auto_brand, auto_model
    ($rest, my $auto_data) = $self->get_auto_data($rest);
    $auto_data->{auto_model} = $model_from_parse if($model_from_parse);

    $h->{auto_brand} = $self->get_auto_brand_for_templ($auto_data->{auto_brand});
    $h->{auto_model} = $self->get_auto_model_for_templ($auto_data);


    #удаляем найденные значения из brand, model, vol
    if ($h->{brand} and grep {$h->{brand} =~ /\b\Q$_\E\b/i} grep {$_} map {split /\s+/, $auto_data->{$_}} grep {$auto_data->{$_}} qw(auto_brand auto_model auto_model_rest)) {
        my $vendor = $self->vendor || '';
        $h->{brand} = (lc($h->{brand}) ne lc($vendor) and $vendor =~ /^[a-z\d\s\-]+$/i) ? $vendor : '';
    }
    if ($auto_data->{years} and @{ $auto_data->{years} }) {
        my $year = $auto_data->{years}->[0];
        if ($h->{vol}) {
            $h->{vol} = join(':', grep {$_ !~ /\Q$year\E/} split /:/, $h->{vol});
        }
        if ($h->{model}) {
            $h->{model} =~ s/\b\Q$_\E\b/ /i for @{ $auto_data->{years} };
        }
    }
    if ($h->{vol}) {
        $h->{vol} = (grep {$_ =~ /^\s*(?:\d+\s*(?:л|мл|мм|кг|вт|mm|мАч)|\d+(?:[xх]\d+){1,2})\s*$/i and $_ !~ /4[xх]4/} split /:/, $h->{vol})[0] || '';
    }

    $h->{model} =~ s/(^|\s+)\Q$_\E(\s+|$)/ /i for grep {$_} map {split /\s+/, $auto_data->{$_}}
                                                  grep {$auto_data->{$_}} qw(auto_brand auto_model auto_model_rest);
    $h->{model} =~ s/(^|\s+)\Q$_\E(\s+|$)/ /i for grep {$_} map {split /\s+/, $h->{$_}}
                                                  grep {$h->{$_}} qw(brand carcase_type viscosity type);

    if ($h->{model} =~ /^\s*([A-Za-z3-9][A-Za-z\d][A-Za-z\d\-\s]*|\d{5,}[A-Za-z\d\-\s]*|[А-ЯЁ][А-ЯЁ\d\-]+)(?:\s+|$)/) {
        my $model = $1;
        $h->{model} = (length($model) >= 4 or $model =~ /\d/ and $model =~ /[A-ZА-ЯЁ]/) ? $model : '';
    } else {
        $h->{model} = '';
    }

    if ($h->{model} and $h->{model} =~ /\S+/ or @properties_for_model) {
        $h->{model_for_title} = join(' ', grep {$_} ($h->{model}, @properties_for_model));
        my @models_list = ();
        push @models_list, $h->{model};
        $h->{model} =~ s/\-/ /g;
        my @model_parts = grep {$_} split /\s+/, $h->{model};
        push @models_list, map {join(' ', @model_parts[0..$_])} reverse (0..$#model_parts);
        push @models_list, map {join('', @model_parts[0..$_])} reverse (0..$#model_parts);

        foreach my $p (grep {$_} @properties_for_model) {
            @models_list = map {($_, "$_ $p")} @models_list;
            push @models_list, $p;
        }
        @models_list = uniq @models_list;
        $logger->debug('models_list:', \@models_list);
        $h->{model} = @models_list > 1 ? '[' . join(':', @models_list) . ']' : @models_list > 0 ? $models_list[0] : '';
    }

    foreach my $field (grep {$h->{$_}} keys %$h) {
        $h->{$field} =~ s/\s+/ /g;
        $h->{$field} =~ s/(^\s+|\s+$)//g;
    }

    $h->{detail_number} ||= ($self->{vendorCode} || '');
    $rest =~ s/\b\Q$_\E\b/ /i for grep {$_} map {split /\s+/, $h->{$_}} grep {$h->{$_}} qw(brand type vol color season qty model detail_number);

    delete $h->{$_} for grep { !$h->{$_} } keys %$h;
    $logger->debug("rest: $rest");
    $logger->debug('parse result:', $h);

    return $h;
}

sub title_templ {
    my $type = '[type/type:_MAINWORDS_IF_NOT_TOO_SHORT/type:_DEL_ADJ_IF_NOT_TOO_SHORT]';
    my $model = '[model viscosity/model/viscosity]';
    my $model_for_title = '[model_for_title:_DEMULT/model_for_title:_DEMULT:_SHORT_MDL]';

    my $auto_brand = 'auto_brand:_DEMULT:_UP_ALL_FIRST';
    my $auto_model = 'auto_model:_DEMULT:_UP_ALL_FIRST';
    my $brand = 'brand:_DEMULT:_UP_ALL_FIRST';
    my $title = '';
    foreach my $type ('type', 'type:_DEL_ADJ', 'type:_MAINWORDS') {
        $title .= "$type $brand $model_for_title, $type $auto_brand $auto_model, $type $brand, $type $auto_brand, ";
    }
    $title .= "$brand $model_for_title, [type/type:_DEL_ADJ] $model_for_title, [type/type:_DEL_ADJ]";
    return $title;
}

sub dyn_templates_text :GLOBALCACHE {
    my $self = shift;

    my $type = '[type/type:_MAINWORDS_IF_NOT_TOO_SHORT/type:_DEL_ADJ_IF_NOT_TOO_SHORT]';
    my $model = '[model viscosity/model/viscosity]';
    my $model_for_title = '[model_for_title:_DEMULT/model_for_title:_DEMULT:_SHORT_MDL]';

    my $auto_brand = 'auto_brand:_DEMULT:_UP_ALL_FIRST';
    my $auto_model = 'auto_model:_DEMULT:_UP_ALL_FIRST';
    my $brand = 'brand:_DEMULT:_UP_ALL_FIRST';
    my $title = $self->title_templ;

    my $res = "
        auto_brand $type                                            {___MULT_PHRASE} => $title
        auto_brand auto_model $type                                 {___MULT_PHRASE} => $title
        auto_model $type                                            {___MULT_PHRASE} => $title
        brand $type                                                 {___MULT_PHRASE} => $title
        auto_brand $model $type                                     {___MULT_PHRASE} => $title
        $model [type/type:_DEL_ADJ]                                 {___MULT_PHRASE} => $title
        detail_number                                               {___MULT_PHRASE} => $title
        brand $model $type                                          {___MULT_PHRASE} => $title
        auto_brand auto_model $model $type                          {___MULT_PHRASE} => $title
        auto_model $model $type                                     {___MULT_PHRASE} => $title
        auto_brand brand $type                                      {___MULT_PHRASE} => $title
        auto_model carcase_type $type                               {___MULT_PHRASE} => $title
        auto_brand auto_model carcase_type $type                    {___MULT_PHRASE} => $title
        auto_brand auto_words $type                                 {___MULT_PHRASE} => $title
        #auto_words brand                                           {___MULT_PHRASE} => $title
        auto_brand auto_model material $type                        {___MULT_PHRASE} => $title
        #auto_model brand                                           {___MULT_PHRASE} => $title
        auto_words brand $type                                      {___MULT_PHRASE} => $title
        auto_model brand $type                                      {___MULT_PHRASE} => $title
        auto_brand auto_model auto_words $type                      {___MULT_PHRASE} => $title
        detail_number [type/type:_DEL_ADJ/type:_MAINWORDS]          {___MULT_PHRASE} => $title
        auto_model material $type                                   {___MULT_PHRASE} => $title
        #brand vol                                                  {___MULT_PHRASE} => $title
        auto_brand material $type                                   {___MULT_PHRASE} => $title
        auto_brand brand $model $type                               {___MULT_PHRASE} => $title
        auto_brand auto_model $type vol                             {___MULT_PHRASE} => $title
        brand detail_number                                         {___MULT_PHRASE} => $title
        auto_brand auto_model brand $type                           {___MULT_PHRASE} => $title
        auto_model auto_words $type                                 {___MULT_PHRASE} => $title
        brand material $type                                        {___MULT_PHRASE} => $title
        brand color $model                                          {___MULT_PHRASE} => $title
        auto_brand $type vol                                        {___MULT_PHRASE} => $title
        #age auto_words $type                                       {___MULT_PHRASE} => $title
        auto_model brand $model                                     {___MULT_PHRASE} => $title
        auto_brand carcase_type $type                               {___MULT_PHRASE} => $title
        detail_number $model                                        {___MULT_PHRASE} => $title
        #auto_brand auto_model brand                                {___MULT_PHRASE} => $title
        auto_model brand $model $type                               {___MULT_PHRASE} => $title
        brand $type vol                                             {___MULT_PHRASE} => $title
        auto_model $type vol                                        {___MULT_PHRASE} => $title
        brand color $model $type                                    {___MULT_PHRASE} => $title
        brand $model vol                                            {___MULT_PHRASE} => $title
        auto_brand detail_number                                    {___MULT_PHRASE} => $title
        #$model vol                                   {___MULT_PHRASE} {___MAX_3000} => $title
        detail_number vol                                           {___MULT_PHRASE} => $title
        brand color $type                                           {___MULT_PHRASE} => $title
        #color $model                                 {___MULT_PHRASE} {___MAX_3000} => $title
        color $model $type                                          {___MULT_PHRASE} => $title
        auto_words $model $type                                     {___MULT_PHRASE} => $title
        #auto_brand auto_model material                             {___MULT_PHRASE} => $title
        auto_brand auto_model brand $model $type                    {___MULT_PHRASE} => $title
        brand detail_number $type                                   {___MULT_PHRASE} => $title
        #auto_model vol                                             {___MULT_PHRASE} => $title
        auto_brand carcase_type $model $type                        {___MULT_PHRASE} => $title
        #auto_words $model                                          {___MULT_PHRASE} => $title
        auto_brand auto_model carcase_type $model $type             {___MULT_PHRASE} => $title
        detail_number $model $type                                  {___MULT_PHRASE} => $title
        $model $type vol                                            {___MULT_PHRASE} => $title
        auto_brand auto_words $model $type                          {___MULT_PHRASE} => $title
        auto_brand auto_model carcase_type material $type           {___MULT_PHRASE} => $title
        brand detail_number $model $type                            {___MULT_PHRASE} => $title
        detail_number $type vol                                     {___MULT_PHRASE} => $title
        auto_brand auto_model material $model $type                 {___MULT_PHRASE} => $title
        auto_brand material $model $type                            {___MULT_PHRASE} => $title
        auto_model carcase_type material $type                      {___MULT_PHRASE} => $title
        #age auto_brand $type                                       {___MULT_PHRASE} => $title
        brand $model $type vol                                      {___MULT_PHRASE} => $title
        auto_words brand $model                                     {___MULT_PHRASE} => $title
        auto_brand color $type                                      {___MULT_PHRASE} => $title
        auto_model color $type                                      {___MULT_PHRASE} => $title
        #age brand $type                                            {___MULT_PHRASE} => $title
        auto_brand detail_number $type                              {___MULT_PHRASE} => $title
        auto_brand auto_model auto_words carcase_type $type         {___MULT_PHRASE} => $title
        auto_brand $model $type vol                                 {___MULT_PHRASE} => $title
        auto_model brand detail_number $model $type                 {___MULT_PHRASE} => $title
        #auto_brand material                                        {___MULT_PHRASE} => $title
        brand season $type                                          {___MULT_PHRASE} => $title
        auto_model carcase_type $model $type                        {___MULT_PHRASE} => $title
        auto_brand auto_model auto_words material $type             {___MULT_PHRASE} => $title
        auto_model detail_number $model $type                       {___MULT_PHRASE} => $title
        auto_brand color $model $type                               {___MULT_PHRASE} => $title
        auto_model auto_words carcase_type $type                    {___MULT_PHRASE} => $title
        brand $model season                                         {___MULT_PHRASE} => $title
        auto_brand auto_model auto_words $model $type               {___MULT_PHRASE} => $title
        auto_brand auto_model color $type                           {___MULT_PHRASE} => $title
        #auto_brand $model vol                                      {___MULT_PHRASE} => $title

#не из статистики
        brand $model                                                {___MULT_PHRASE} => $title

        brand types                                                 {___MULT_PHRASE} => $title
        model types                                                 {___MULT_PHRASE} => $title
        brand model types                                           {___MULT_PHRASE} => $title
        auto_brand types                                            {___MULT_PHRASE} => $title
        auto_model types                                            {___MULT_PHRASE} => $title
        auto_brand auto_model types                                 {___MULT_PHRASE} => $title
    ";
    return $res;
}


sub perf_templates_text {
    return $_[0]->dyn_templates_text;
}

sub banner_single_templates_text {
    my $self = shift;
    my $title = $self->title_templ;
    my $result = "
        brand => $title
        model => $title
        types => $title
    ";
    return $result;
}

sub dyn_methods_arr :GLOBALCACHE {
    my @res = map { s/^\s+//; $_ } grep {/\S/} grep {!/#/} ##no critic
        split /\n/, '
        modellike_product                       goods accessory P
        #snorm_phrase_list                       goods accessory
        get_search_filtered50k                  goods accessory
        add_dynamic_homonymy_words              goods accessory P
        add_trade_phrases_dynamic_auto_accessories          goods accessory S
        get_wide_filtered                       goods accessory
        filter_auto_accessories                  goods accessory
        #postfilter_product                      goods accessory P
        snorm_phrase_list                       goods accessory
        pack_phr_lite                           goods accessory
        pack_list                               goods accessory
        get_search_filtered50k                  goods accessory
        set_exclamations_before_stops           goods accessory
        set_exclamations_before_bsstops         goods accessory
        replace_exclamations_with_pluses        goods accessory
    ';
    return @res;
}

sub _UP_ALL_FIRST {
    my ( $self, $txt ) = @_;
    return '' unless $txt;
    $txt = join " ", grep { s/^(.*)$/\u$1/ } split /\s+/, $txt; ##no critic
    return $txt;
}

sub _DEL_ADJ_IF_NOT_TOO_SHORT {
    my ( $self, $txt ) = @_;
    my $res = $self->_CUT_IF_NOT_TOO_SHORT('_DEL_ADJ', $txt);
    return $res;
}

1;
