package BM::BannersMaker::ProductCars;
use strict;
use utf8;
use base qw(BM::BannersMaker::Product);
use Utils::Array;
use Data::Dumper;
use List::Util qw(min max);

my $dbg = 0;

# поля хэша == методы

__PACKAGE__->mk_accessors(qw(
    vendor
    model
    run
    owners_number
    year
    body_type
    description
    extras
    modification_id
    price
    currencyId
    color
    metallic
    additional_data
    picture
    url
    offerfilters
    name
    gear
));

# каждый класс должен имееть свою версию get_remotecache_id, которая использует родительскую версию функции
# так можно сбрасывать кеш для всех классов и для одного конкретного
sub get_remotecache_id {
    my ($self) = @_;
    my $common_product_id = $self->SUPER::get_remotecache_id;
    return 'cars000 '.$common_product_id;
}

#серии (числовые) для bmw и классы (буквенные) для mercedes
my @mercedes_classes = qw{a b c e g s cla clc cls gl gla glc gle glk x4 m ml s};
my @bmw_series = qw{1 2 3 4 5 6 7 8};

# названия двигателей
my @engines = qw{blueefficiency bluemotion cdi cdti cgi crdi d4-d dci di-d dohc fsi gdi hpi jtd mivec mpi td tdci tdi tfsi thp tsi turbo vtec vvti турбо};

my $mercedes_classes_re;
my $bmw_series_re;
my $engines_re;
my $body_type_re;

sub class_init :RUN_ONCE {
    my $class = shift;
    $mercedes_classes_re = "\\b(".join('|', @mercedes_classes).")\\b";
    $bmw_series_re = "\\b(".join('|', @bmw_series).")\\b";
    $engines_re = "\\b(".join('|', @engines).")\\b";
}

# input: цена_машины (например, 478000 )
# output:   '500000',               'за 500000',        'за 500 000',       'до 500000',        'до 500 000',
#           '500000 руб(лей)',        'за 500000 руб(лей)', 'за 500 000 руб(лей)', 'до 500000',        'за 500 000',
#           '500 тыс(яч) руб(лей)', 'за 500 тыс(яч) руб(лей)',        'за 500 000'
sub get_price_variants {
    my $price = int $_[0];

    if (($price > 1000000) or ($price < 10000)) {
        # для цен выше миллиона вариации цен в статистике не встречаются
        return [];
    }

    my ($rounded_price_thousand, @upper_prices_thousand);

    if ($price >= 100000) {
        # округляем цену до сотен тысяч
        my $hundred_thousand = POSIX::floor($price / 100000);
        $rounded_price_thousand = $hundred_thousand * 100;

        # массив цен, больших данной: 720 000 (rounded = 700 000) -> 800 000, 900 000
        @upper_prices_thousand = grep { $_ <= 1000  } map { ($rounded_price_thousand + 100 * $_) } (1 .. 2);
    }
    else {
        # округляем цену до десятков тысяч
        my $decades_thousand = POSIX::floor($price / 10000);
        $rounded_price_thousand = $decades_thousand * 10;

        # массив цен, больших данной: 72 000 (rounded = 70 000) -> 80 000, 90 000
        @upper_prices_thousand = grep { $_ <= 100  } map { ($rounded_price_thousand + 10 * $_) } (1 .. 2);
    }

    my @res;

    # точная цена, разложение по возможным вариантам

    my @price_tails = ('000', ' 000', ' тыс', ' тысяч');
    my @precise_prices = map { $rounded_price_thousand.$_ } @price_tails;

    for my $price (@precise_prices) {
        for my $rub ('', 'р', 'руб', 'рублей') {
            push @res, "!за ".(($rub ne '') ? $price." ".$rub : $price);
            push @res, (($rub ne '') ? $price." ".$rub : $price);
        }
    }

    # верхняя граница цен, разложение по возможным вариантам
    my @upper_prices;

    for my $tail (@price_tails) {
        push @upper_prices, (map { $_.$tail } @upper_prices_thousand);
    }

    for my $price (@upper_prices) {
        for my $rub ('', 'р', 'руб', 'рублей') {
            push @res, "!до ".(($rub ne '') ? $price." ".$rub : $price);
        }
    }

    return \@res;
}

my %transmission_variants = (
    cvt => '[cvt:вариатор]',
    at => '[!автомат:акпп:at]',
    mt => '[!механика:мкпп:mt]',
);

sub get_transmission {
    my $text = $_[0];
    # тип коробки передач
    if ($text) {
        if ($text =~ /\b(cvt|вариатор)\b/i) {
            return 'cvt';
        }
        elsif ($text =~ /\b(at|автомат|акпп)\b|(\bавт\.)/i) {
            return 'at';
        }
        elsif ($text =~ /\b(mt|механика|мкпп)\b|(\bручн\.)/i) {
            return 'mt';
        }
    }
    return '';
}

sub get_engine {
    my $text = $_[0];

    #  характеристики двигателя
    if ($text) {
        my @volume_ar = ('');
        my @engine_type_ar = ('');
        my @fuel_ar = ('');

        if ($text =~ /(?:\D|^)(?<volume>\d[. ]\d|[1-9]\d{0,1}00)\h*(?<fuel>бензин|дизель|гибрид.*?|d|hyb)?(?:[ .,:]|$)/i) {
            my $fuel = $+{'fuel'};
            my $volume = $+{'volume'};

            if ($fuel) {
                if ($fuel =~ /^(d|дизель)$/i) {
                    push @fuel_ar, ('дизель', 'дизельный');
                }
                elsif ($fuel =~ /^(hyb|гибрид.*?)$/i) {
                    push @fuel_ar, ('гибрид', 'гибридный');
                }
            }
            else {
                push @fuel_ar, ('бензин', 'бензиновый');
            }

            if (defined $volume) {
                my ($small_volume_dot, $small_volume_space, $big_volume);

                if ($volume > 10) {
                    $big_volume = $volume;
                    $small_volume_dot = sprintf("%.1f", ($volume / 1000));

                    $small_volume_space = $small_volume_dot;
                    $small_volume_space =~ s/\./ / if ($small_volume_space);
                }
                else {
                    if ($volume) {
                        if ($volume =~ /^\d\.\d$/) {
                            $small_volume_dot = $volume;

                            $small_volume_space = $small_volume_dot;
                            $small_volume_space =~ s/\./ / if ($small_volume_space);
                        }
                        elsif ($volume =~ /^\d\s\d$/) {
                            $small_volume_space = $volume;

                            $small_volume_dot = $small_volume_space;
                            $small_volume_dot =~ s/\s/./;
                        }
                    }

                    $big_volume = $small_volume_dot * 1000;
                }

                push @volume_ar, ($big_volume, $small_volume_dot, $small_volume_space);
            }
        }

        my $engine_type = get_engine_type($text);
        push @engine_type_ar, $engine_type if ($engine_type ne '');

        my @res = ();
        for my $eng (@engine_type_ar) {
            for my $vol (@volume_ar) {
                for my $fuel (@fuel_ar) {
                    if (($eng eq '') and ($vol eq '') and ($fuel eq '')) {
                        next;
                    }
                    my $temp_res = join ' ', ($vol, $eng, $fuel);
                    if ($temp_res) {
                        $temp_res =~ s/^\s+|\s+$//g;
                        $temp_res =~ s/\s+/ /g;
                    }
                    push @res, $temp_res;
                }
            }
        }
        return ("[".join(':', @res)."]");
    }
    return '';
}

sub get_drive {
    #привод
    my $text = $_[0];

    if ($text) {
        #привод
        if ($text =~ /^полный$/i) {
            return "[полный:полный привод]";
        }
        elsif ($text =~ /\b(4wd|awd)\b/i) {
            return "[".$1.":полный:полный привод]";
        }
        elsif ($text =~ /\bfwd\b/i) {
            return "[".$1.":передний:передний привод]";
        }
        elsif ($text=~ /\brwd\b/i) {
            return  "[".$1.":задний:задний привод]";
        }
    }
    return '';
}

sub get_horse_power {
    my $text = $_[0];

    if (($text) and ($text =~ /(?:^|\D+)(\d{2,3})\s*(?:лошадин.*?|л\.?)\s*(?:сил.*?|с\.?)/i)) {
        return $1;
    }
    return '';
}

sub horse_power_all_variants {
    my $digit = $_[0];

    if (($digit) and ($digit =~ /^\d+$/)) {
        return "[$digit лошадиные силы:$digit !л !с:$digit !л. !с.]";
    }
    return '';
}

sub get_age {
    my $text = $_[0];

    if ($text) {
        if ($text =~ /\d{1,3}(\s*\d{1,3})?\s*(?:км|километр.*?)(\h+|\p{P}|$)|бу|б(?:\h+|\p{P})у|подержан.*?|\d\h+(?:и более\h+)?владел.*?|с\h+пробег/i) {
            return "[!с пробег:!бу:!б !у:подержаный:подержанный]";
        }

        # берём первые 15 слов, чтобы слово "новая" относилось к машине, а не к запчасти
        my $small_text = join " ", (grep { $_ }  (split /\h+/, $text)[0..14]);
        if (($small_text) and ($small_text =~ /(?:(?:не\h+было|нет|0)\h+владельцев)|новый|новая/)) {
            return "[новый:!без пробег]";
        }
    }

    return '';
}

sub get_deal_type {
    my $text = $_[0];
    if (($text) and ($text =~ /(кредит|рассрочк)/gi)) {
        # ! перед "кредит" - иначе "авто кредит" слипается в "автокредит"
        return '[!кредит:!в !кредит:рассрочка:!в рассрочку]';
    }
    return '';
}

sub get_engine_type {
    my $text = $_[0];
    if (($text) and ($text =~ /$engines_re/i)) {
        return $1;
    }
    return '';
}

sub get_body_type {
    my $text = $_[0];
    return '' unless $text;

    $body_type_re ||= qr(cargo|coupe|hatchback|jeep|liftback|minivan|roadster|rodster|sedan|uneversal|universal|авобус|автбус|автобс|автобуз|автобус|автоус|автубус|атобусный|бортовой\s+карго|бортовой\s+тягач|вездеход|внедарожник|внедоржник|внедорожник|внедорожный|втобус|грузовик\s+бортовой|грузовой|грузовой-бортовой|грузовой\s+иномарка|джип|кабреолет|кабриалет|кабриолет|красовер|крассовере|кросовер|кроссовер|купе|купэ|легковой|лифтбек|матацикл|микроавтобус|миневна|миневэн|минивен|мини\s+вена|минивэн|мини\s+вэн|мини-вэн|мотоцикл|овтобус|пекап|пикап|роадстер|родстар|родстары|родстер|седан|семейный|сидан|спорткар|тягач|уневерсал|уневирсал|универсал|универсаль|унивесал|унивирсал|фургон|хатчбек|хатчбэк|хачбэк|хетчбег|хетчбек|хетчебек|хечбег|хечбек|хечбэк|хечтбек|хэтчбег|хэтчбек|хэтчбэк|хэчбек|хэчбэк|юниверсал);

    if ($text =~ $body_type_re) {
        return $1;
    }
    return '';
}

my %words_to_digits = (
    'пять' => 5,
    'пяти' => 5,
    'четырех' => 4,
    'четырёх' => 4,
    'четыре' => 4,
    'трех' => 3,
    'трёх' => 3,
    'дву' => 2,
);

sub get_doors {
    my $text = $_[0];

    return '' unless ($text);

    my $door_digit = '';

    if ($text =~ /(пяти|четырех|четырёх|трех|трёх|двух?)(?:\h+|-)?(?:дверный|двер(?:ь|ей|и))/) {
        $door_digit = $words_to_digits{$1};
    } elsif ($text =~ /(\d)(-?ти)?(?:\h+|-)?(?:дверный|двер(?:ь|ей|и))/) {
        $door_digit = $1;
    }

    if ($door_digit) {
        return "[5 дверный:5ти дверный:5дверный:5тидверный:пяти дверный:пятидверный]" if ($door_digit == 5);
        return "[4 дверный:4х дверный:4дверный:4хдверный:четырех дверный:четырехдверный]" if ($door_digit == 4);
        return "[3 дверный:3х дверный:3дверный:3хдверный:трех дверный:трехдверный]" if ($door_digit == 3);
        return "[2 дверный:2х дверный:2дверный:2хдверный:двух дверный:двухдверный]" if ($door_digit == 2);
    }
    return '';
}

sub get_price {
    my $text = $_[0];
    if ($text) {
        my @prices = ($text =~ /\b((?:\d?\h+\d{3}\h+\d{3})|(?:\d{2,3}\h+\d{3})|\d{5,7})\b/g);
        if ((scalar @prices) == 1)
        {
            #возвращаем цену, только если нашли один вариант
            return $prices[0];
        }
    }
    return '';
}

# те поля фида, которые будут инициализированы, это поля tskv как они есть
my @offer_fields = qw{bad_flags images vin unique_id vendor model run owners_number year body_type description
                     extras modification_id price currencyId color metallic additional_data picture url offerfilters
                     name gear custom_phrases merge_key engine_volume engine_power engine_type gearbox drive state};

sub init {
    my ($self) = @_;
    my $data = $self->{data};
    if(ref($data->[0]) eq 'HASH'){
        $self->{$_} = $data->[0]{$_} for @offer_fields, $self->get_common_offer_fields;
    }
    $self->class_init;
}


########################################################
#Доступ к полям
########################################################

sub ad_type {
    return 'cars';
}

# как матчится фраза с запросом

sub match_type {
    return 'snorm';
}

sub price2int {
    my $price = shift;
    $price = $1 if $price =~ /(.*?)(\.|,)/;
    $price =~ s/\s+//g;
    return $price;
}

sub parse :CACHE {
    my ($self) = @_;
    my $timer = $self->proj->get_new_timer;
    my $h = {};
    $h->{additional_data} = $self->additional_data || '';
    $h->{picture} = $self->picture || '';
    $h->{url} = $self->url || '';
    $h->{avto} = '[!авто:автомобиль:!машина:автомашина]';
    my $name = $self->name || '';

    $h->{brand_title} = $self->vendor || '';
    if ($h->{brand_title}) {
        $h->{brand_title} = ucfirst($h->{brand_title});
        #print STDERR "cars brand found in feed: ".($h->{brand_title})."\n";
    }

    my $model_full = $self->model || '';
    $model_full = ucfirst($model_full) if ($model_full);

    if ($name and (not $h->{brand_title}) and (not $model_full)) {
        # динамические по данным с сайта, всё в name
        my $phr = $self->proj->phrase( $name );
        my ($brand, $model) = $phr->parse_fast_auto;
        $h->{brand_title} = $brand;

        if (($model) and ($model =~ /\h+(20\d{2}|19\d{2})$/)) {
            # в модель может попасть год, забираем его оттуда
            $h->{year} = $1;
            $model =~ s/\h+(20\d{2}|19\d{2})$//;
        }

        if (not exists $h->{year}) {
            # не нашли год на предыдущем шаге, надо парсить из строки
            if (($name) and ($name =~ /\h+(20\d{2}|19\d{2})(?:\h+(?:г\.|года?)?|$)/)) {
                $h->{year} = $1;
            }
        }

        $h->{model_title} = $model;
        $h->{model_syn} = $h->{model_title};
        $h->{brand_syn} = $h->{brand_title};

        my ($colors, $text) = $phr->cut_colors;
        $h->{color} = (split ":", $colors)[0] if ($colors);

        $h->{transmission} = get_transmission($name);
        $h->{engine} = get_engine($name);
        $h->{drive} = get_drive($name);
        $h->{age} = get_age($name);

        # удаляем пробег, чтобы не мешало выпарсивать цену
        $name =~ s/(\d{1,6}|(\d{1,3}\h+\d{3}))\h+(км|километр.*?)//g if ($name);

        $h->{doors} = get_doors($name);
        my $price = get_price($name);
        $price =~ s/\h+//g if ($price);
        if ($price ne '') {
            $h->{price} = get_price_variants($price);
        }

        $h->{horse_pow} = get_horse_power($name);
        $h->{deal_type} = get_deal_type($name);
        $h->{body_type} = get_body_type($name);

    } else {
        # генерация по фиду
        $h->{model_title} = $model_full;
        $h->{model_title} =~ s/((?:,| \(| «| ").*$)// if ($h->{model_title});
        if ($model_full) {
            $model_full =~ s/\,/ /;
            $model_full =~ s/\s+/ /g;
        }

        # серия или класс для bmw и mercedes
        if (($h->{model_title}) and ($h->{brand_title})) {
            if (($h->{brand_title} =~ /\bbmw\b/i) and ($h->{model_title} =~ /$bmw_series_re/i)) {
                $h->{class_series} = $1." серии";
            }

            if (($h->{brand_title} =~ /\bmercedes\b/i) and ($h->{model_title} =~ /$mercedes_classes_re/i)) {
                $h->{class_series} = $1." класса";
            }
        }

        # TODO: добавить разваливание бренда и модели по синонимам за приемлемое время
        if ($h->{model_title} eq $model_full) {
            $h->{model_syn} = $h->{model_title};
        }
        else {
            $h->{model_syn} = "[".($h->{model_title}).":".($model_full)."]";
        }
        $h->{brand_syn} = $h->{brand_title};

        # новый / подержаный
        my $words_new = '[новый:!без !пробега]';
        my $words_old = '[!б !у:бу:!с !пробегом:подержаный:подержанный]';

        my $owners_number = $self->owners_number || '';
        if ($owners_number) {
            $h->{age} = ($owners_number =~ /^((не было|0) владельцев|0)$/i) ? $words_new : $words_old;
        }
        else {
            my $run = $self->run || '';
            if ($run) {
                $run =~ s/\D//g;
                $h->{age} = ($run <= 100) ? $words_new : $words_old;
            }
        }

        # год
        if ($self->year and ($self->year =~ /\s*(20\d{2}|19\d{2})\s*/)) {
            $h->{year} = $1;
        };

        # тип кузова
        my $body_type = $self->body_type || '';
        if ($body_type) {

            $body_type = lc($body_type);
            my @body_type_array = ($body_type);

            # первое слово перед пробелом
            if ($body_type =~ /^(\S*)\s+/) {
                push @body_type_array, $1;
            }

            $h->{body_type} = (scalar @body_type_array == 1) ? $body_type_array[0] : ("[".join(":", @body_type_array)."]");

            my @doors_array = ();

            # дверность: пятидверный
            if (($body_type) and ($body_type =~ /(?:\D|^)(\d)\s?дв\b/)) {
                my $doors_count = $1;
                push @doors_array, $doors_count." дверный";

                if ($doors_count == 2) {
                    push @doors_array, 'двухдверный';
                } elsif ($doors_count == 3) {
                    push @doors_array, 'трехдверный';
                } elsif ($doors_count == 4) {
                    push @doors_array, 'четырехдверный';
                } elsif ($doors_count == 5) {
                    push @doors_array, 'пятидверный';
                }
            }

            if (scalar @doors_array > 0) {
                $h->{doors} = (scalar @doors_array == 1) ? $doors_array[0] : ("[".join(":", @doors_array)."]");
            }
        }

        # цвет
        $h->{color_title} = $self->color || '';
        $h->{color_title} = lc($h->{color_title}) if ($h->{color_title} ne '');

        #TODO: сделать цвет, согласованный с полом: Белая Тойота, но Чёрный Ленд Крузер
        if ($h->{color_title} ne '') {
            my $metallic = $self->metallic || '';
            $h->{color} = ($h->{color_title}).":цвет ".($h->{color_title});
            $h->{color} = ($h->{color}).":".($h->{color_title})." !металлик:цвет ".($h->{color_title})." !металлик" if (($metallic) and ($metallic =~ /^да$/i));
            $h->{color} = "[".($h->{color})."]";
        }

        $h->{modification_for_title} = $self->modification_id || '';
        delete $h->{modification_for_title} if (length $h->{modification_for_title} > 20);

        if (($h->{modification_for_title}) and ($h->{modification_for_title} =~ /\b(?<D1>\d{1})(?<D2>\d{1})00\b/)) {
            my $temp1 = ($+{'D1'}).($+{'D2'})."00";
            my $temp2 = ($+{'D1'}).".".($+{'D2'})."л";
            $h->{modification_for_title} =~ s/$temp1/$temp2/;
        }

        $h->{modification_for_title} =~ s/\b(\d\.\d)\b/$1л/ if ($h->{modification_for_title});

        $h->{full_modification} = $self->modification_id || '';

        if ($h->{brand_title} =~ /mercedes-benz/i) {
            $h->{full_modification} =~ s/^\d{2,3} //g;
        }

        # лошадиные силы
        if ($h->{full_modification}) {
            $h->{horse_pow} = get_horse_power($h->{full_modification});

            $h->{transmission} = get_transmission($h->{full_modification});
            $h->{engine} = get_engine($h->{full_modification});

            my $gear = $self->gear || '';
            if ($gear) {
                $h->{drive} = get_drive($gear);
            }
            else {
                $h->{drive} = get_drive($h->{full_modification});
            }
        }

        # цена
        my $price = $self->price || '';
        my $currencyId = $self->currencyId || '';

        if (($currencyId) and ($currencyId =~ /^(ru[rb]?)$/i) and ($price)) {
            # цена в рублях, TODO: сделать для других валют
            $price = price2int($price);
            my @price_variants;
            @price_variants = @{get_price_variants(int($price))} if $price =~ /^\d+$/;
            $h->{price} = (scalar @price_variants == 0) ? '' : "[".join(":", @price_variants)."]";
        }

        # рассрочка /кредит
        my $description = $self->description || '';
        my $extras = $self->extras || '';

        $h->{deal_type} = get_deal_type("$description $extras");
    }

    unless (exists $h->{modification_for_title}) {
        if (($h->{engine}) and ($h->{engine} =~ /\b(\d\.\d)\b/)) {
            $h->{modification_for_title} = $1."л";
        }

        $h->{modification_for_title} .= " ".(uc $h->{transmission}) if ($h->{transmission});
        $h->{modification_for_title} .= " (".$h->{horse_pow}." л.с.)" if ($h->{horse_pow});
    }

    $h->{horse_pow} = horse_power_all_variants($h->{horse_pow}) if ($h->{horse_pow});
    $h->{transmission} = $transmission_variants{$h->{transmission}} if ($h->{transmission} and exists $transmission_variants{$h->{transmission}});

    if ($h->{modification_for_title}) {
        my @r = split /\s+/, $h->{modification_for_title};
        $h->{modification_for_title_short} = join " ", @r[0..1] if (scalar @r > 2);
    }

    my $timings = $timer->report;
    my $wstat = {};
    $wstat->{"time-$_"} += $timings->{$_} for keys %$timings;
    if ($dbg) {
        print ("$_  ".($wstat->{$_})."\n") for (sort { $wstat->{$a} <=> $wstat->{$b} } keys %$wstat);
    }

    for my $k (keys %$h) {
        unless ($h->{$k}) {
            delete $h->{$k};
            next;
        }
        $h->{$k} =~ s/(^\s+|\s+$)//g;
    }

#    print STDERR "KYOTO: '".($self->get_remotecache_id)."'\n";
#    print STDERR "CAAARS:\n", Dumper($h);
    return $h;
}

sub text_templates {
    my @res = (
        "age avto body_type {___MULT_PHRASE}",
        "age avto brand_syn {___MULT_PHRASE}",
        "age avto brand_syn model_syn {___MULT_PHRASE}",
        "age avto brand_syn model_syn year {___MULT_PHRASE}",
        "age avto brand_syn year {___MULT_PHRASE}",
        "age avto model_syn {___MULT_PHRASE}",
        "age avto model_syn year {___MULT_PHRASE}",
        "age avto price {___MULT_PHRASE}",
        "age avto price year {___MULT_PHRASE}",
        "age avto transmission {___MULT_PHRASE}",
        "age body_type {___MULT_PHRASE}",
        "age body_type brand_syn {___MULT_PHRASE}",
        "age body_type brand_syn model_syn {___MULT_PHRASE}",
        "age body_type model_syn {___MULT_PHRASE}",
        "age body_type model_syn year {___MULT_PHRASE}",
        "age brand_syn {___MULT_PHRASE}",
        "age brand_syn class_series {___MULT_PHRASE}",
        "age brand_syn class_series model_syn {___MULT_PHRASE}",
        "age brand_syn class_series model_syn year {___MULT_PHRASE}",
        "age brand_syn class_series year {___MULT_PHRASE}",
        "age brand_syn engine model_syn {___MULT_PHRASE}",
        "age brand_syn model_syn {___MULT_PHRASE}",
        "age brand_syn model_syn transmission {___MULT_PHRASE}",
        "age brand_syn model_syn year {___MULT_PHRASE}",
        "age brand_syn transmission {___MULT_PHRASE}",
        "age brand_syn year {___MULT_PHRASE}",
        "age class_series model_syn {___MULT_PHRASE}",
        "age class_series model_syn year {___MULT_PHRASE}",
        "age engine model_syn {___MULT_PHRASE}",
        "age model_syn {___MULT_PHRASE}",
        "age model_syn transmission {___MULT_PHRASE}",
        "age model_syn year {___MULT_PHRASE}",
        "avto body_type brand_syn {___MULT_PHRASE}",
        "avto body_type brand_syn model_syn {___MULT_PHRASE}",
        "avto body_type model_syn {___MULT_PHRASE}",
        "avto brand_syn class_series {___MULT_PHRASE}",
        "avto brand_syn drive model_syn {___MULT_PHRASE}",
        "avto brand_syn model_syn {___MULT_PHRASE}",
        "avto brand_syn model_syn year {___MULT_PHRASE}",
        "avto brand_syn transmission {___MULT_PHRASE}",
        "avto brand_syn year {___MULT_PHRASE}",
        "avto drive model_syn {___MULT_PHRASE}",
        "avto model_syn transmission {___MULT_PHRASE}",
        "avto model_syn year {___MULT_PHRASE}",
        "avto price {___MULT_PHRASE}",
        "body_type brand_syn class_series {___MULT_PHRASE}",
        "body_type brand_syn model_syn {___MULT_PHRASE}",
        "body_type brand_syn model_syn transmission {___MULT_PHRASE}",
        "body_type brand_syn model_syn year {___MULT_PHRASE}",
        "body_type brand_syn year {___MULT_PHRASE}",
        "body_type model_syn year {___MULT_PHRASE}",
        "brand_syn class_series {___MULT_PHRASE}",
        "brand_syn class_series year {___MULT_PHRASE}",
        "brand_syn color model_syn year {___MULT_PHRASE}",
        "brand_syn engine horse_pow model_syn {___MULT_PHRASE}",
        "brand_syn engine model_syn year {___MULT_PHRASE}",
        "brand_syn full_modification model_syn {___MULT_PHRASE}",
        "brand_syn model_syn transmission {___MULT_PHRASE}",
        "brand_syn model_syn transmission year {___MULT_PHRASE}",
        "brand_syn model_syn year {___MULT_PHRASE}",
        "brand_syn transmission {___MULT_PHRASE}",
        "class_series model_syn year {___MULT_PHRASE}",
        "engine model_syn year {___MULT_PHRASE}",
        "model_syn transmission {___MULT_PHRASE}",
        "model_syn transmission year {___MULT_PHRASE}",
        "model_syn year {___MULT_PHRASE}",
        "age avto deal_type {___MULT_PHRASE}",
        "age brand_syn deal_type model_syn {___MULT_PHRASE}",
        "age deal_type model_syn {___MULT_PHRASE}",
        "avto brand_syn deal_type {___MULT_PHRASE}",
        "avto deal_type model_syn {___MULT_PHRASE}",
        "avto deal_type year {___MULT_PHRASE}",
        "body_type brand_syn deal_type model_syn {___MULT_PHRASE}",
        "body_type deal_type model_syn {___MULT_PHRASE}",
        "brand_syn deal_type {___MULT_PHRASE}",
        "brand_syn deal_type model_syn {___MULT_PHRASE}",
        "brand_syn deal_type model_syn year {___MULT_PHRASE}",
        "brand_syn deal_type year {___MULT_PHRASE}",
        "deal_type model_syn {___MULT_PHRASE}",
        "deal_type model_syn year {___MULT_PHRASE}",
        "brand_syn model_syn doors {___MULT_PHRASE}",
        "brand_syn doors {___MULT_PHRASE}",
        "body_type doors {___MULT_PHRASE}",
    );
    return \@res;
}

sub only_perf_text_templates {
    my ($self) = @_;
    my @res = (
        "brand_syn year {___MULT_PHRASE}",
    );
    return \@res;
}

sub dyn_templates_text :STATIC :GLOBALCACHE {
    my $class = shift;
    my $title = $class->dyn_title_templ;
    my @text_templ = @{$class->text_templates};
    my $res = join "\n", (map { my $r = $_." => $title"; $r; } @text_templ);
    return $res;
}

sub perf_templates_text :STATIC :GLOBALCACHE {
    my $class = shift;
    my $title = $class->perf_title_templ;
    my @text_templ = @{$class->text_templates};
    push @text_templ, @{$class->only_perf_text_templates};
    my $res = join "\n", (map { my $r = $_." => $title"; $r; } @text_templ);
    return $res;
}

sub dyn_methods_arr :GLOBALCACHE {
    my @res = map { s/^\s+//; $_ } grep {/\S/} grep {!/#/} ##no critic
        split /\n/, '
        set_exclamations_before_stops           goods accessory
        set_exclamations_before_bsstops         goods accessory
        snorm_phrase_list                       goods accessory
        spec_pack_list                          goods accessory
        get_wide_filtered                       goods accessory
        get_search_filtered50k                  goods accessory
        pack_phr_lite                           goods accessory
        spec_pack_list                          goods accessory
        replace_exclamations_with_pluses        goods accessory
    ';
    return @res;
}

sub perf_methods_arr {
    my ($self) = @_;
    return $self->dyn_methods_arr;
}

sub dyn_title_templ {
    my @templates = (
        "brand_title model_title modification_for_title year",
        "brand_title model_title modification_for_title",
        "brand_title model_title modification_for_title_short year",
        "brand_title model_title modification_for_title_short",
        "brand_title model_title year",
        "model_title modification_for_title",
        "brand_title modification_for_title",
        "model_title modification_for_title_short",
        "brand_title modification_for_title_short",
        "brand_title model_title",
        "model_title year",
        "brand_title year",
    );

    my $res = join ", ", @templates;
    return $res;
}

sub perf_title_templ {
    my ( $self ) = @_;

    my @templates = (
        "brand_title model_title year",
        "brand_title model_title",
        "model_title year",
        "brand_title year",
    );

    my $res = join ", ", @templates;
    return $res;
}

sub perf_banners_single {
    my ($self, %params) = @_;
    my $title_templ = $self->perf_title_templ;

    my $arr = $self->banners_data(
        templates_text => "brand_title model_title year=> $title_templ\nbrand_title model_title => $title_templ\nbrand_title year => $title_templ\nmodel_title year => $title_templ",
        methods_arr => [ 'pack_list', ],
        max_count => 1,
        assert_no_rpc => 1,
        title_template_type => 'single',
        %params,
    );

    $arr = $self->single_banner_default('perf', %params) unless @$arr;
    return @$arr ? [ $arr->[0] ] : [];
}

sub minus_words :GLOBALCACHE {
    return '-+то -ремонт -запчасти -покраска -заправка -шины -замена -обслуживание -техосмотр -сервис -автосервис -шиномонтаж -стекло -аккумулятор -диагностика -чехол -колесо -масло -диски -фара -фильтр -колодки -магнитола -дверь -зеркало';
}

1;
