package BM::BannersMaker::ProductRealty;

=h
Редактировано: 7 ноября 2016
                                                Класс по недвижимости
                                              -------------------------
1. Логика генерации

 !Генерация для динамики и перфоманса РАЗЛИЧНА!
------------------------------------------------
 1.1 Динамика
<<<TODO>>>
 1.2 Перфоманс (СМАРТ)
    Специфика фидов по недвижимости заключается в
        1) стремлении к точности, а не к полноте (определено менеджерами)
        2) большом количестве вариантов и разных форм их записи для некоторых кирпичей (цена, метраж, комнатность, ЖК)
    Для решения этих кейсов было принято решение "двухсторонней" нормализации фраз: со стороны генерации и при занесении фраз в поведенческие профили.
    В результате нормализации:
        1) информация по цене и метражу приводится к определенным диапазонам
        2) выделяются "псевдокирпичи" - проблемные кирпичи в нормализованном виде (например, метраж: rltarea_10_; комнатность: rltrooms2)
    Фразы с псевдокирпичами добавляются в профиль на этапе броадматчинга.
    | Для генерации псевдокирпичи определяются ПОСЛЕ ПРЕПРОЦЕССИНГА, во время парсинга (саба parser). |
    | Фильтр по частотности на поиске ОТКЛЮЧЕН. |

 1.3 Парсинг
 Так как мы не можем показывать в тайтлах ненадежную информацию - таковой мы считаем кирпичи, полученные из name/description/typePrefix - нужно разделять эту логику.
 Поэтому сначала кирпич пытаемся получить из полей фида, и если
    1) успешная попытка, то заносим этот кирпич и в поле надежных (с префиксом $prefix_feed), и в поле "рабочих" для динамики, без префиксов
    2) неуспех - то заносим выпарсенный кирпич только в "рабочие" поля и НЕ ИСПОЛЬЗУЕМ его для тайтлов в перфомансе
--------------------------------------------------
 Имеется возможность настроить "чистоту предобработки" - отключить эвристический парсинг по descriptions, etc., по дефолту включен.
 Все исходные поля переносятся в предобработанный фид без изменений.

=cut

use base qw(BM::BannersMaker::Product);
use Utils::Array;
use Utils::Common;
use Data::Dumper;
use Utils::LibTrie;
use Utils::Words;

use utf8;

use open ":utf8";
no warnings "utf8";


__PACKAGE__->mk_accessors(qw(
    mark
    model
    name
    description
    typePrefix
));

my @stones_flat = qw{district address_street metro mortgage time_to_metro_foot time_to_metro_transport new_flat
                     price area_value area_unit decoration rooms floors floor type apartment_cmplx balcony geonames};

my $meters_symbol = 'м²';
my $prefix_flat = 'FL_';
my $prefix_feed = 'FEED_';
my $TITLE_LEN = 33;


################# Инициализация класса ####################
################# vvvvvvvvvvvvvvvvvvvv ####################
# ленивая логика

our $dict_metro_reg      = undef;
our $dict_districts      = undef;
our $dict_time_measures  = undef;
our %dict_districts_full = ();


#####
# Загрузка словарей
#
sub load_dicts {
    my ($proj) = @_;
    return if defined $dict_metro_reg;
    #load metro dictionary
    my $dict_metro_reg_txt = '(?:'.join('|',@{$proj->dict_manager->get_dict("dict_metro", '')->phrase_list->perl_array}).')';
    $dict_metro_reg = qr/$dict_metro_reg_txt/i;

    #load district dict
    %dict_districts_full = map { substr($_, 0, -2) => $_} @{$proj->dict_manager->get_dict("dict_districts", '')->phrase_list->perl_array};
    my $dict_districts_txt = '(?:'.join('|', keys %dict_districts_full).')';
    $dict_districts = qr/$dict_districts_txt/i;

    #define time measures
    $dict_time_measures = qr/мин|минут|м|км|метров|м\-ов|км\-ов/i;
}

our $geonames_trie = '';
our @geonames_terminals = ();

#####
# Загрузка трая с локациями
#
sub init_geanames_trie {
    return ($geonames_trie, @geonames_terminals) if $geonames_trie;

    # Get terminals
    my $terminal_district = $Utils::Common::options->{trie_matcher}->{districts_terminal};
    my $terminal_street   = $Utils::Common::options->{trie_matcher}->{streets_terminal};
    @geonames_terminals = ($terminal_street, $terminal_district);

    # Get streets
    my $path_to_streets   = $Utils::Common::options->{trie_matcher}->{src_streets_path};
    open (F, '<', $path_to_streets);
    my @streets_list = <F>;
    close F;

    # Get districts
    my $path_to_districts = $Utils::Common::options->{trie_matcher}->{src_districts_path};
    open (F, '<', $path_to_districts);
    my @districts_list = <F>;
    close F;

    # Construct tries
    my $streets_trie   = Utils::LibTrie::construct_trie(\@streets_list,   $terminal_street,   '\s+');
    my $districts_trie = Utils::LibTrie::construct_trie(\@districts_list, $terminal_district, '\s+');

    # Unite tries
    $geonames_trie  = Utils::LibTrie::union_tries($streets_trie, $districts_trie);

    return ($geonames_trie, @terminals);
}

sub is_separate_house {
    my ($type) = @_;
    return ($type =~ /дача|коттедж|дом|таунхаус|гараж|участок/);
}

##### init
#
#
sub init {
    my ($self) = @_;
    $self->SUPER::init;
    my $data = $self->{data};

    $self->{offer_type} = 0;

    #load everithing from tskv
    $self->{$_} = $data->[0]{$_} for keys $data->[0];
}

##################### ^^^^^^^^^^^^^ #######################
##################### Инициализация #######################


sub ad_type {
    return 'realty';
} # banner router

sub match_type {
    return 'norm';
} # matching query

sub max_wordcount {
    my ($self) = @_;
    return 10;
}


############## PSEUDO_STONES ##################
############## vvvvvvvvvvvvv ##################

sub round_int {
    my ($x) = @_;
    return POSIX::floor($x + 0.5);
}

sub normalize_price {
    my ($price) = @_;

    return '' if not defined $price or $price !~ /^\d+$/ or ($price > 1e8 or $price < 100_000);

    return round_int( $price / 100_000 ) / 10;
}

my @area_segments = ( (0, 16), (17, 22), (23, 27), (28, 33), (34, 37), (38, 50), (51, 1000) );
sub normalize_area {
    my ($area) = @_;
    $area = round_int( $area );
    return '' if ($area > 1_000 or $area < 5);
    my $res = '';
    if ($area < 45) {
        my $seg_num = 0;
        while ($area_segments[$seg_num + 1] < $area) {
            $seg_num += 2;
        }
        $seg_num /= 2;
        $res = $seg_num;
    } else {
        my $tens = round_int( $area / 10 ) * 10;
        $res = $tens;
    }
    return $res;
}

sub normalize_rooms {
    my ($rooms) = @_;
    $rooms = int( $rooms );
    return $rooms;
}

sub normalize_ac {
    my ($ac) = @_;
    $ac =~ s/^мкрн?//i;
    $ac = lc $ac;
    $ac =~ s/[^\w\d]//g;
    return ($ac ? $ac : '');
}

#ROOM normalizer
our %hash_rooms_int2str = (
    1 => 'одно',
    2 => 'двух',
    3 => 'трех',
    4 => 'четырех',
    5 => 'пяти',
    6 => 'шести',
    7 => 'семи',
    8 => 'восьми',
    9 => 'девяти',
    10 => 'десяти',
);
our %hash_rooms_str2int = (
    'одно' => 1,
    'одн' => 1,
    'двух' => 2,
    'дву' => 2,
    'трех' => 3,
    'тре' => 3,
    'четырех' => 4,
    'четыре' => 4,
    'пяти' => 5,
    'шести' => 6,
    'семи' => 7,
    'восьми' => 8,
    'девяти' => 9,
    'десяти' => 10,

);

our %supplies_en2ru = (
    'water-supply' => 'вода',
    'gas-supply' => 'газ',
    'heating-supply' => 'отопление',
    'sewerage-supply' => 'канализация',
    'electricity-supply' => 'электричество',
);

our %supplies_wieght = (
    'отопление' => 5,
    'вода' => 4,
    'газ' => 3,
    'канализация' => 2,
    'электричество' => 1,
);

sub pseudo_rooms {
    my ($rooms) = @_;
    my $tmp = normalize_rooms($rooms) || '';
    return '' unless $tmp;
    return lc('RLTROOMS'.$tmp);
}

sub denorm_pseudo_rooms {
    my ($rooms) = @_;
    my $tmp = normalize_rooms($rooms) || '';
    return '' unless $tmp;
    return "$hash_rooms_int2str{$tmp}комнатная";
}

################

sub pseudo_price {
    my ($price) = @_;
    my $temp_price = normalize_price($price) || '';

    return '' unless $temp_price;
    $temp_price =~ s/\./f/g;
    return lc('RLTPRICE'.$temp_price);
}

sub denorm_pseudo_price {
    my ($price) = @_;
    my $temp_price = normalize_price($price) || '';

    return '' unless $temp_price;
    return "$temp_price";
}


sub pseudo_area {
    my ($area) = @_;
    my $tmp = normalize_area($area) || '';

    return '' unless $tmp;
    return lc('RLTAREA'.$tmp);
}

sub denorm_pseudo_area {
    my ($area) = @_;
    my $tmp = normalize_area($area) || '';

    return '' unless $tmp;
    return $tmp;
}

sub pseudo_ac {
    my ($ac) = @_;
    my $temp_ac = normalize_ac($ac) || '';

    return '' unless $temp_ac;
    return lc('RLTAC'.$temp_ac);
}

sub denorm_pseudo_ac {
    my ($ac) = @_;
    $ac =~ s/^мкрн?//i;
    $ac = lc $ac;
    return "$ac";
}

sub parse_geonames {
    my ($trie, $phrase, @terminals) = @_;

    init_geanames_trie() unless $trie;
    return {} unless @terminals;

    $phrase = join(' ', @{ lmr->norm_words($phrase,'ru') });
    my $res = { map {$_ => []} @terminals };

    my @words = split(/\s+/, $phrase);
    for(my $i = 0; $i < scalar(@words); $i++) {
        my $geonames = Utils::LibTrie::find_in_trie($trie, \@terminals,  $i, \@words);
        push @{ $res->{$_} }, $geonames->{$_} for grep { $geonames->{$_} } @terminals;
    }

    return $res;
}

################## ^^^^^^^^^^^^^^^^^^ ######################
################## PSEUDO_STONES done #######################

################## PARSE ######################
################## vvvvv ######################
sub get_rooms_from_phrase {
    my ($phr) = @_;
    my $res = '';
    if ($phr =~ m/(одно|двух|тр[её]х|четыр[её]х|пяти|шести|семи|восьми|девяти|десяти)[а-ик-яёa-z\-\ \–\.]{0,3}(к[\.]|ком.*|ква.*|км)(?:\b|$)/i and
        $2 !~ /кварта/i)
    {
        my $rooms = lc $1;
        $rooms =~ s/ё/е/gi;
        $res = $hash_rooms_str2int{$rooms};
    } elsif ($phr =~ m/(?:\b|^)(\d{1,2})[\-\ а-яёa-z\–\.]{0,5}(к[\.]|ком.*|ква.*|км)(?:\b|$)/i and exists $hash_rooms_int2str{$1} and
        $2 !~ /кварта/i)
    {
        $res = $1;
    }

    return $res;
}

sub extract_apartment_complex {
    my $phr = shift;
    return '' unless $phr;

    $phr =~ s/[\,\.\!\?].*//;

    my @words = grep { $_ } split(/[\s\-\:\–]+/, $phr);
    my $is_prev_adj = 1;

    my @a_res = ();
    for my $word(@words) {
        my $first_l = substr($word, 0, 1);
        if ($is_prev_adj or ($first_l eq uc $first_l) or ($first_l =~ m/[a-z]/) or (length $word < 3 && $word !~ m/от|за|на|в/ ))  {
            push @a_res, $word;
            $is_prev_adj = $word =~ m/(?:ий|ый|ой|вая|ная|мая|лая|кая|ые|ие|ое|ых|их|яя)$/i;
        } else {
            last;
        }
    }
    return join(' ', @a_res);
}

sub get_apartment_complex_from_phrase {
    my $phr = shift;

    return '' if $phr =~ m/дисплей|экран/i;

    my $res = '';
    if ($phr =~ /\b(жк|ЖК|комплекс[а-яА-Я]*?)\ (?:[\«\'\"]|&laquo;)(.+?)(?:[\»\'\"]|&raquo;)/i) {
        $res = lc $2;
    }
    elsif ($phr =~ m/\b(?:ЖК|компл[ексаувы]{0,8})[\ \.\,]+(.*)/i) {
        if ($1 =~ m/[\"\'\«](.+?)[\"\'\»]/) {
            $res = lc $1;
        } else {
            $res = extract_apartment_complex( $1 );
        }
    }
    if ((not $res) && $phr =~ /^(.+) жк$/i) {
        $res = extract_apartment_complex( $1 );
    }
    $res =~ s/([\"\'\(\)\[\]]|ЖК|^\s+|\s+$)//gi;
    $res =~ s/(^\s+|[\:\!\?\.]+|\s+$)//gi;

    return $res;
}

sub get_price_from_phrase {
    my ($phr) = @_;
    my $res = '';

    if ($phr =~ m/([\d\ \,\.]+)[\ \-]{1,2}(руб|р[\.\ \,]|млн|тыс|милл)/i) {
        my ($num, $unit) = ($1, $2);
        $num =~ s/[^\d\.\,]//g;

        if (length $num > 0 && $num ne '.' && $num ne ',') {
            $num = int( format_float($num) * 1_000_000 ) if ($unit eq 'млн' || $unit eq 'милл');
            $num = int( format_float($num) * 1_000) if $unit eq 'тыс';
            $res = $num;
        }
    }
    return $res if $res;

    if (#$phr =~ m/(?:за|цена|стоимость|стоит)\ ([\d\.\,\ ]{1,15})/i or
        $phr =~ m/(\d+(?:[\.\,]\d{0,14})?)(руб|р[\.\,\ ]|млн|милл|тыс)/i)
    {
        my ($res, $unit) = ($1, $2);
        $res = int(format_float($res) * 1_000_000) if ( $unit =~ /млн|милл/ );
        $res = int(format_float($res) * 1_000) if ( $unit eq 'тыс' );
    }
    return $res;
}

sub get_district_from_phrase {
    my ($phr) = @_;
    my $res = '';

    if ($dict_districts) {
        $res = $dict_districts_full{$1} if ($phr =~ m/[\s\:\,\.\;\-](р\.|р\-?н\.?|район|мкрн\.?)/i and $phr =~ m/($dict_districts)/);
        return $res if $res;
    }

    if ($phr =~ m/(?<a>ра[ий]он|р-?о?н.?|округ|окр.?)\ (?<b>[А-ЯA-Z][\wа-яА-Я\ ]+?)(?:\/|\ |\,|\.|\!)[^А-ЯA-Z]/ or
        $phr =~ m/(?:\ |\,|\.|\!)[^А-ЯA-Z]+(?:\/|\ |\,|\.|\!)(?<b>[А-ЯA-Z][\wа-яА-Я\ ]+?)(?<a>ра[ий]он|р-?о?н.?|округ|окр.?)/)
    {
        my $addr = $+{b};
        my $tmp_res = '';
        for my $w(reverse split(/\s/, $addr)) {
            if ($w =~ /[А-ЯA-Z]/) {
                $tmp_res = $w.' '.$tmp_res;
            } else {
                last;
            }
        }
        $tmp_res =~ s/^\s+|\s+$//g;
        $res = $tmp_res if length $tmp_res > 2;
    }

    return $res;
}

sub get_metro_from_phrase {
    my ($phr) = @_;
    my $res = '';

    if ($dict_metro_reg) {
        $res = $1 if ($phr =~ m/(\ м.)|(метро\ )/i and $phr =~ m/($dict_metro_reg)/i);
    }

    return $res;
}

sub get_time_to_metro_from_phrase {
    my ($phr) = @_;
    my $res = '';

    $phr = lc $phr;
    if (($phr =~ m/\ (.+?)\ (.+?)\ (?:до|от)\ метро/ and check_metro_time($1, $2)) or
        ($phr =~ m/\ (?:до|от)\ (?:метро|м\.) (.+?)\ (.+?)\s/ and check_metro_time($1, $2)))
    {
        my ($x, $y) = ($1, $2);
        $x =~ s/^\s+|\s+$//g;
        $res = $x;  ##! .' '.$y;
    }

    return $res;
}

sub get_area_from_phrase {
    my ($phr) = @_;
    my ($res_1, $res_2) = ('', '');

    if ($phr =~ m/([\d\.\,]{2,})[\ \- \.\,]+(?:кв\bм|м\bкв|м\ ?2|м²|квад|метр)/i) {
        my $area = $1;
        $area =~ s/\,/\./g;
        $area =~ s/^\.+|\.+$//g;
        $res_1 = $area if length $area > 1;
    }

    unless ($res_1) {
        my @arr = ();
        if (@arr = $phr =~ m/\ (\d[\d\.\,]*)\ ?[^ ]*(га|соток|\&sup2|м\.?\ ?кв\.?|кв\.? ?м\.?|м2|м²)/gi) {
            $arr[0] =~ s/\,/\./g;
            $arr[2] =~ s/\,/\./g if defined $arr[2];
            if (scalar(@arr) <= 4) {
                my $q = defined $arr[1]? $arr[1]: 'м.кв.';
                $q = 'м.кв.' if $q eq '&sup2';
                $q = 'м.кв.' if $q eq 'м2';
                $q = 'и.кв.' if $q eq 'м²';
                $res_1 = $arr[0];
                $res_2  = $q;
            }
        }
    }
    if ($res_1) {
    	$res_1 =~ s/[\.\,].*//;
        $res_1 = '' if ($res_1 && substr($res_1, 0, 1) eq '-');
    	$res_1 =~ s/\D//g;
    }

    return ($res_1, $res_2);
}

sub get_floors_number_from_phrase {
    my ($phr) = @_;
    my $res = '';

    if ($phr =~ m/(\d+)[\ \-х]{0,3}этаж?/i) {
        $res = $1;
    } elsif ($phr =~ m/одно-?этаж/i) {
        $res = 1;
    } elsif ($phr =~ m/двух?-?этаж/i) {
        $res = 2;
    } elsif ($phr =~ m/тр[её]х?-этаж/i) {
        $res = 3;
    } elsif ($phr =~ m/четыр[её]х-?этаж/i) {
        $res = 4;
    } elsif ($phr =~ m/пяти-?этаж/i) {
        $res = 5;
    } elsif ($phr =~ m/шести-?этаж/i) {
        $res = 6;
    } elsif ($phr =~ m/семи-?этаж/i) {
        $res = 7;
    } elsif ($phr =~ m/восьми-?этаж/i) {
        $res = 8;
    } elsif ($phr =~ m/девяти-?этаж/i) {
        $res = 9;
    } elsif ($phr =~ m/десяти-?этаж/i) {
        $res = 10;
    }

    return $res;
}

sub get_floor_from_phrase {
    my ($phr) = @_;
    my $res = '';

    if ($phr =~ m/(\d+)\ (?:этаж|эт\.|э\.)/i or
        $phr =~ m/(?:этаж|эт\.|э\.)[\ \:]{1,2}(\d+)/i)
    {
        $res = $1;
    }

    return $res;
}

#round price
sub get_rounded_price {
    my ($price) = @_;
    my @res = ();

    if ($price < 1_000_000) {
        my $rounded_price = POSIX::floor($price / 1_000);
        push @res, $rounded_price." тыс";
    } elsif ($price < 3_000_000) {
        my $rounded_dec_price = POSIX::floor($price / 100_000) / 10;
        push @res, $rounded_dec_price." млн";
    } else {
        my $rounded_price = POSIX::floor($price / 1000_000);
        push @res, $rounded_price." млн";
    }
    return '['.join(':', @res).']';
}

#round area
sub get_rounded_area {
    my ($area) = @_;
    my $res = normalize_area($area);
    return '' unless $res;
    $res = POSIX::floor(($area_segments[$res*2] + $area_segments[$res*2+1] + 1) / 2) if $res < 50;
    return $res;
}


sub clear_metro_time {
    my ($x, $y) = @_;
    chomp $$x;
    chomp $$y;
    $$y = lc $$y;
    $$y =~ s/\.//gi;
}

sub check_metro_time {
    my ($x, $y) = @_;
    chomp $x;
    chomp $y;
    $y =~ s/\.//gi;
    return ($x =~ m/^[0-9\-\.]+$/) and ( $y =~ m/$dict_time_measures/ );
}

my @allowed_types = ('апартаменты', 'квартира', 'студия', 'гараж', 'таунхаус', 'таунхауз', 'коттедж', 'комната', 'дом', 'участок', 'дача', 'часть дома', 'cottage', 'house', 'house with lot', 'flat', 'room', 'townhouse', 'garage', 'lot');
my @type_arr = qw/апартаменты квартира студия помещение гараж таунхаус таунхауз коттедж комната дом участок дача часть cottage house with lot flat room townhouse garage/;
my %types_en2ru = (
    'cottage' => 'коттедж',
    'flat' => 'квартира',
    'house' => 'дом',
    'room' => 'комната',
    'townhouse' => 'таунхаус',
    'garage' => 'гараж',
);
sub get_type_from_phrase {
    my ($proj, $text) = @_;
    return undef unless($text);

    my $type_phrase_list = $proj->phrase_list( { phrases_arr => \@type_arr } );
    my @good_types = split(':', $type_phrase_list->search_subphrases_in_text($text)->phrases2text(':'));
    if ((join " ", sort @good_types) eq 'дом участок') {
        return 'дом с участком';
    } elsif ((join " ", sort @good_types) eq 'дом часть') {
        return 'часть дома';
    } elsif ((join " ", sort @good_types) eq 'house lot with') {
        return 'дом с участком';
    }
    for my $cand(@good_types) {
        if ($cand ~~ @type_arr) {
            return $cand;
        }
    }
    return undef;
}

#####
# Парсер оффера
# Для предобработки, подробнее 2.2 в основном комменте про класс (шапка)
#
sub parse_offer {
    my ($proj, $offer) = @_;
    delete($offer->{$_}) for grep { !defined($offer->{$_}) || $offer->{$_} eq '' } keys %$offer;
    load_dicts($proj);

    my $is_debug = 0;
    my $st_time = 0;

    my $h = ();

    my $use_feed   = 1; # turn on if feed is good
    my $use_parser = 1; # turn off if parser is too 'dirty'

    my $common_phr = '';
    $common_phr = $offer->{name}        if exists $offer->{name};
    $common_phr = $offer->{description} if exists $offer->{description};
    $common_phr =~ s/ё/е/i;

    my $common_phr_short = substr($common_phr, 0, 200);

    #STONE: address_street
    #try to parse it and extract
    # 1. District
    # 2. Street
    if ($is_debug) {
        $st_time = time;
    }

    if ($use_feed and defined $offer->{'new-flat'} and $offer->{'new-flat'} and not defined $offer->{'building-name'}) {
        # для новостроек поле 'building-name' - обязательно (SUPBL-762)
        return {};
    }

    if ($use_feed and exists $offer->{address}) {
        my @parts = split(',', $offer->{address});
        my ($street, $district) = ('', '');
        for my $part(@parts) {
            if (not $street and $part =~ m/\b(ул\.?|улиц.?|проспект.?|переул|пер\.?|пр\-?т|пр\.?|пл\.?)\b/i) {
                $part =~ s/\b$1\b|^\s+|\s+$//ig;
                $street = $part;
            }
            if (not $district and $part =~ m/\b(ра[ий]он|р-?о?н.?|округ|окр.?)\b/i) {
                $part =~ s/\b$1\b|^\s+|\s+$//ig;
                $district = $part;
            }
            my $res = $street || $district;
            if (not $res) {
                $res = $parts[ ( scalar (@parts) ) / 2];
                $res = $parts[-1] if scalar (@parts) == 2;
            }
            $res =~ s/\ к[\.\ ]?(\d+)//;
            $res =~ s/^\s+|\s+$//g;
            $res =~ s/^\s*\.\s*//;
            $h->{"${prefix_feed}address_street"} = $res if $res;
        }
    }
    if ($is_debug) {
        print STDERR "block 1 done in ".(time - $st_time)."\n";
        $st_time = time;
    }

    #STONE: apartment_cmplx
    my $clean_ac = sub {
            my $ac = shift;
            $ac =~ s/(?<=[\'\"])[^\"\']*(?!.*[\"\'])//;
            $ac =~ s/(?<=\»).*//;
            $ac =~ s/([\"\']|ЖК|^\s+|\s+$)//gi;
            $ac =~ s/(^\ |\ $)//;
            $ac =~ s/[\:\!\?\.]//;
            return $ac;
        };

    $h->{"${prefix_feed}apartment_cmplx"} = $clean_ac->( $offer->{'building-name'} ) if $use_feed and exists $offer->{'building-name'};

    if ($is_debug) {
        print STDERR "block 3 done in ".(time - $st_time)."\n";
        $st_time = time;
    }

    #STONE: metro
    $h->{"${prefix_feed}metro"} = $offer->{metro} if $use_feed and exists $offer->{metro};
    if ($use_parser and not defined $h->{"${prefix_feed}metro"}) {
        my $res = get_metro_from_phrase($common_phr);
        $h->{metro} = $res;
    }
    if ($is_debug) {
        print STDERR "block 4 done in ".(time - $st_time)."\n";
        $st_time = time;
    }

    #STONE: time_to_metro
    $h->{"${prefix_feed}time_to_metro_foot"} = $offer->{'time-on-foot'} if $use_feed and exists $offer->{'time-on-foot'};
    $h->{"${prefix_feed}time_to_metro_transport"} = $offer->{'time-on-transport' } if $use_feed and exists $offer->{'time-on-transport'};
    if ($use_parser and not exists $h->{"${prefix_feed}time_to_metro_foot"} and not exists $h->{"${prefix_feed}time_to_metro_transport"}) {
        $res = get_time_to_metro_from_phrase($common_phr_short);
        $h->{time_to_metro_foot} = $res if $res;
    }
    if ($is_debug) {
        print STDERR "block 5 done in ".(time - $st_time)."\n";
        $st_time = time;
    }


    #STONE: price
    $h->{"${prefix_feed}price"} = $offer->{'price'} if $use_feed and exists $offer->{'price'};
    if ($use_parser and not exists $h->{"${prefix_feed}FEED_price"}) {
        my $res = get_price_from_phrase($common_phr);
        $h->{price} = $res if $res;
    }
    if ($is_debug) {
        print STDERR "block 6 done in ".(time - $st_time)."\n";
        $st_time = time;
    }


    #STONE: area
    (my $area_perl_format = $offer->{'area'} || '') =~ s/\,/./;
    $h->{"${prefix_feed}area_value"} = $offer->{'area'} if $use_feed and exists $offer->{'area'} and $area_perl_format > 0;
    $h->{"${prefix_feed}area_unit"}  = $offer->{'area:unit'}  if $use_feed and exists $offer->{'area:unit'};
    if ($use_parser and not defined $h->{"${prefix_feed}area_value"}) {
        my ($res_1, $res_2) = get_area_from_phrase($common_phr);
        ($h->{area_value}, $h->{area_unit}) = ($res_1, $res_2) if $res_1;
    }
    $h->{area_value}                 =~ s/[\.\,].*// if defined $h->{area_value};
    delete $h->{area_value} unless $h->{area_value};
    $h->{"${prefix_feed}area_value"} =~ s/[\.\,].*// if defined $h->{"${prefix_feed}area_value"};
    delete $h->{"${prefix_feed}area_value"} unless $h->{"${prefix_feed}area_value"};
    if ($is_debug) {
        print STDERR "block 7 done in ".(time - $st_time)."\n";
        $st_time = time;
    }

    #STONE: number of rooms
    $h->{"${prefix_feed}rooms"} = $offer->{rooms} if $use_feed and exists $offer->{rooms};
    if ($use_parser and not exists $h->{"${prefix_feed}rooms"}) {
        my $res = get_rooms_from_phrase($common_phr);
        $h->{rooms} = $res if $res;
    }
    if ($is_debug) {
        print STDERR "block 8 done in ".(time - $st_time)."\n";
        $st_time = time;
    }


    if (defined $h->{rooms}) {
        delete $h->{rooms} if $h->{rooms} > 10;
        delete $h->{rooms} if defined $h->{type}  and $h->{type} =~ m/земельный/i;
    }
    if (defined $h->{"${prefix_feed}rooms"}) {
        delete $h->{"${prefix_feed}rooms"} if $h->{"${prefix_feed}rooms"} > 10;
        delete $h->{"${prefix_feed}rooms"} if defined $h->{type}  and $h->{type} =~ m/земельный/i;
    }

    #STONE: floor
    $h->{"${prefix_feed}floor"} = $offer->{floor} if $use_feed and defined $offer->{floor};
    if ($use_parser and not exists $h->{"${prefix_feed}floor"}) {
        my $res = get_floor_from_phrase($common_phr_short);
        $h->{floor} = $res if $res;
    }
    if ($is_debug) {
        print STDERR "block 9 done in ".(time - $st_time)."\n";
        $st_time = time;
    }


    #STONE: floors
    $h->{"${prefix_feed}floors"} = $offer->{'floors-total'} if $use_feed and exists $offer->{'floors-total'};
    if ($use_parser and not exists $h->{"${prefix_feed}floors"}) {
        my $res = get_floors_number_from_phrase($common_phr_short);
        $h->{floors} = $res if $res;
    }
    if ($is_debug) {
        print STDERR "block 10 done in ".(time - $st_time)."\n";
        $st_time = time;
    }


    #STONE: district
    $h->{"${prefix_feed}district"} = $offer->{'sub_location'} if $use_feed and exists $offer->{'sub_location'};

    if ($use_parser and not exists $offer->{"${prefix_feed}district"}) {
        my $res = get_district_from_phrase($common_phr);
        $h->{district} = $res if $res;
    }
    if ($is_debug) {
        print STDERR "block 11 done in ".(time - $st_time)."\n";
        $st_time = time;
    }

    #STONE: type
    # проблема с парсером - category пересекается с другими полями
    #    $h->{type} = $offer->{category} if $use_feed and exists $offer->{category};
    if ($use_parser and not exists $h->{type}) {
        if (($offer->{apartments}) and ($offer->{apartments} =~ /^(true|да|\+|1)$/i)) {
            $h->{type} = 'апартаменты';
        } elsif (($offer->{studio}) and ($offer->{studio} =~ /^(true|да|\+|1)$/i)) {
            $h->{type} = 'студия';
            delete $h->{$_} for grep { $_ =~ /room/i } keys %$h;
        } else {
            # берём первое предложение - в нём точно будет упомянут объект продажи
            my ($first_phrase) = ($common_phr_short =~ /(^[^\.!]+[\.!])/);
            $h->{type} = get_type_from_phrase($proj, $offer->{category}) // get_type_from_phrase($proj, $first_phrase); # квартира - костыль
        }
    }
    return {} unless ($h->{type});
    $h->{type} = $types_en2ru{$h->{type}} if (exists $types_en2ru{$h->{type}});
    return {} unless ($h->{type} ~~ @allowed_types);

    if ($is_debug) {
        print STDERR "block 12 done in ".(time - $st_time)."\n";
        $st_time = time;
    }

    #STONE: new_flat
    $h->{new_flat} = '[новостройка:от застройщика]' if $use_feed and exists $offer->{'new-flat'} and $offer->{'new-flat'} =~ m/1|\+|true|да/i;
    if ($use_parser and not exists $h->{new_flat}) {
        if ($common_phr =~ m/(новострой)|(застройщ)/gi) {
            $h->{new_flat} = '[новостройка:от застройщика]';
        }
    }

    #STONE: mortgage
    $h->{mortgage} = 'ипотека' if $use_feed and exists $offer->{mortgage} and $offer->{mortgage} =~ m/1|\+/;
    if ($use_parser and not exists $h->{mortgage}) {
        my @mortgage_arr = qw/ипотека кредит рассрочка/;
        my $mortgage_phrase_list = $proj->phrase_list( { phrases_arr => \@mortgage_arr } );
        my @arr = split(':', $mortgage_phrase_list->search_subphrases_in_text($common_phr_short)->phrases2text(':'));
        for my $cand(@mortgage_arr) {
            if ($proj->phrase($cand)->norm_phr ~~ @arr) {
                $h->{mortgage} = $cand;
                last;
            }
        }
    }
    if ($is_debug) {
        print STDERR "block 13 done in ".(time - $st_time)."\n";
        $st_time = time;
    }


    #STONE: decoration
    if ($use_parser and $common_phr =~ /отделк/i) {
        $h->{decoration} = 'с отделкой';
    }

    #STONE: balcony
    $h->{balcony} = $offer->{balcony} if $use_feed and exists $offer->{balcony};
    if ($use_parser) {
        if ($common_phr =~ /балкон/i) {
            $h->{balcony} = 'с балконом';
        } elsif($common_phr =~ /лоджи/i) {
            $h->{balcony} = 'с лоджией';
        }
    }

    #STONE: geonames for Smart
    my $phrase = join(' ', values %$h);;
    my $h_geonames = parse_geonames($geonames_trie, $phrase, @geonames_terminals) || {};
    my @geonames = ();
    for my $terminal( grep { $h_geonames->{$_} && @{ $h_geonames->{$_} } } @geonames_terminals) {
        my @variants = sort {  scalar($b =~ tr/ //) <=> scalar($a =~ tr/ //) } @{ $h_geonames->{$terminal} };
        for my $variant(@variants) {
            unless ($h->{apartment_complex} && index($h->{apartment_complex}, $variant) == -1) {
                push @geonames, _WRAP_EXCL('', $variant);
                last;
            }
        }
        last if (scalar(@geonames) == 3);
    }
    $h->{geonames} = '['.join(':', @geonames).']';

    if (grep { ( $_ =~ /type/ ) and ( $h->{$_} =~ /комната|студия/i ) } keys %$h) {
        # удаляем количество комнат для случаев, когда комната одна
        delete($h->{$_}) for (grep { /room/i } keys %$h);
    }
    if (grep { ( $_ =~ /type/ ) and ( $h->{$_} =~ /^участок$/i ) } keys %$h) {
        # "чиним" размерность площади для участка, иначе по дефолту подставятся метры
    }

    delete($h->{$_}) for grep { not defined($h->{$_}) || $h->{$_} eq '' } keys %$h;

    return $h;
}

sub format_float {
    my ($v) = @_;
    return '' unless $v;
    $v =~ s/\,/\./g;
    my $r = sprintf("%.2f", $v);
    $r = substr($r, 0, -3) if substr($r, -3) eq '.00';
    return $r;
}

#####
# Основной парсер
#
sub parse :CACHE {
    my ($self) = @_;
    my $h = ();
    my $proj = $self->proj;

    if ($self->{type} and ($self->{type} =~ /аренда/i)) {
        return {}; # DYNSMART-426, пункт 4 - ничего не генерим по офферам аренды
    }

    my $prefix = $prefix_flat;

    my %offer;
    my $parsed_offer = parse_offer($proj, $self);
    $offer{$prefix_flat.$_} = $parsed_offer->{$_} for keys %$parsed_offer;
    $offer{$prefix_flat.'type'} = lc($offer{$prefix_flat.'type'}) if $offer{$prefix_flat.'type'};
    $offer{$_} = $self->{$_} for keys %$self;

    # !важно!
    # записываем без префиксов названия кирпичей - приоритет у фидовых (с префиксом $prefix_feed) + фидовые дублируются
    # это нужно для единого формата кирпичей и для "надежности" тайтлов
    for (@stones_flat) {
        if ($offer{"${prefix}${prefix_feed}$_"}) {
            $h->{"${prefix}$_"} = $offer{"${prefix}${prefix_feed}$_"};
            $h->{"${prefix}${prefix_feed}$_"} = $offer{"${prefix}${prefix_feed}$_"};
        } elsif ($offer{"${prefix}$_"}) {
            $h->{"${prefix}$_"} = $offer{"${prefix}$_"};
        }
    }
    return {} unless ($h->{"${prefix}type"});
    $h->{"${prefix}type"} = $types_en2ru{$h->{"${prefix}type"}} if (exists $types_en2ru{$h->{"${prefix}type"}});
    return {} unless ($h->{"${prefix}type"} ~~ @allowed_types);

    #price
    if ($h->{"${prefix}${prefix_feed}price"}) {
        $h->{"${prefix}pseudo_price"}  = pseudo_price($h->{"${prefix}${prefix_feed}price"});
        $h->{"${prefix}denorm_price"}  = denorm_pseudo_price($h->{"${prefix}${prefix_feed}price"});
        $h->{"${prefix}rounded_price"} = get_rounded_price($h->{"${prefix}${prefix_feed}price"});
    }

    #time_to_metro
    if ($h->{"${prefix}time_to_metro_foot"} and $h->{"${prefix}time_to_metro_transport"}) {
        $h->{"${prefix}time_to_metro"} = '['.$h->{"${prefix}time_to_metro_foot"}.' мин пешком:'.$h->{"${prefix}time_to_metro_transport"}.' мин на транспорте]';
    } elsif (exists $h->{"${prefix}time_to_metro_transport"}) {
        $h->{"${prefix}time_to_metro"} = $h->{"${prefix}time_to_metro_transport"}.' мин на транспорте';
    } elsif (exists $h->{"${prefix}time_to_metro_foot"}) {
        $h->{"${prefix}time_to_metro"} = $h->{"${prefix}time_to_metro_foot"}.' мин пешком';
    }

    #pseudo rooms
    if ($h->{"${prefix}rooms"}) {
        $h->{"${prefix}pseudo_rooms"} = pseudo_rooms($h->{"${prefix}rooms"});
        $h->{"${prefix}denorm_rooms"} = denorm_pseudo_rooms($h->{"${prefix}rooms"});
    }

    #area
    if ($h->{"${prefix}area_value"} && int($h->{"${prefix}area_value"})) {
        $h->{"${prefix}area_unit"} = 'кв м' if not exists $h->{"${prefix}area_unit"};
        if (!$h->{"${prefix}area_unit"}  || $h->{"${prefix}area_unit"} =~ m/кв/) {
            $h->{"${prefix}area"} = $h->{"${prefix}area_value"};
            $h->{"${prefix}rounded_area"} = $self->_EXTEND_AREA_VALUE(get_rounded_area($h->{"${prefix}area_value"}));
            $h->{"${prefix}pseudo_area"} = pseudo_area($h->{"${prefix}area_value"});
            $h->{"${prefix}denorm_area"} = denorm_pseudo_area($h->{"${prefix}area_value"});
        } else {
            $h->{"${prefix}area"} = $h->{"${prefix}area_value"};
        }
        if ($h->{"${prefix}area_unit"} and ($h->{"${prefix}area_unit"} =~ /сот|га/)) {
            $h->{"${prefix}area"} = $h->{"${prefix}area_value"}." ".($h->{"${prefix}area_unit"});
        } elsif ($h->{"${prefix}${prefix_feed}area_value"}) {
            $h->{"${prefix}${prefix_feed}area"} = $h->{"${prefix}${prefix_feed}area_value"};
        }
    }

    #apartments complex
    if ($h->{"${prefix}apartment_cmplx"}) {
        $h->{"${prefix}pseudo_apartment_cmplx"} = pseudo_ac($h->{"${prefix}apartment_cmplx"});
        $h->{"${prefix}denorm_apartment_cmplx"} = denorm_pseudo_ac($h->{"${prefix}apartment_cmplx"});
    } else {
        my $denorm_ac = denorm_pseudo_ac(get_apartment_complex_from_phrase(($h->{description}||'').' '.($h->{name}||'')));
        my $pseudo_ac = pseudo_ac(get_apartment_complex_from_phrase(($h->{description}||'').' '.($h->{name}||'')));
        $h->{"${prefix}pseudo_apartment_cmplx"} = $pseudo_ac if $pseudo_ac;
        $h->{"${prefix}denorm_apartment_cmplx"} = $denorm_ac if $denorm_ac;
    }

    my @pseudo_norm_stones = ('district', 'metro'); #, 'address_street'); #?address_street
    for (@pseudo_norm_stones) {
        $h->{"${prefix}pseudo_$_"} = $proj->phrase($h->{"${prefix}$_"})->norm_phr_ordered if ($h->{"${prefix}$_"});
    }
    $h->{"${prefix}pseudo_address_street"} = $proj->phrase( $self->_CUT_STREET_TYPE($h->{"${prefix}address_street"}) )->norm_phr_ordered if ($h->{"${prefix}address_street"});

    delete($h->{$_}) for grep { !defined($h->{$_}) || $h->{$_} eq '' } keys %$h;

    if ($h->{"${prefix}type"}) {
        if (is_separate_house($h->{"${prefix}type"})) {
            if ($offer{'lot-area:value'} and $offer{'lot-area:unit'}) {
                $h->{"${prefix}lot_area"} = $offer{'lot-area:value'}." ".lc($offer{'lot-area:unit'});
                $h->{"${prefix}lot_area"} =~ s/[сc]от$/соток/i;
            }
            my @supplies = sort {$supplies_wieght{$b} <=> $supplies_wieght{$a}} map { $supplies_en2ru{$_} } grep { ($_ =~ /supply/) and ($offer{$_}) } keys %offer;
            $h->{"${prefix}supplies"} = "[".(join ":", @supplies)."]";
        } else {
            delete($h->{$_}) for grep { /floors/i } keys %$h;
        }
    }

    return $h;
}

#MODIFICATORS
sub _EXTEND_METRO_SHORT {
    my ($self, $txt) = @_;
    return "м. " . ($self->_UP_FIRST($txt));
}

sub _EXTEND_FLOORS {
    my ($self, $txt) = @_;
    # считаем, что этажей меньше 10. допиши метод, если надо больше
    my $mod = $txt % 10;
    return $txt . ' этаж' if (($mod == 1) and ($txt != 11));
    return $txt . ' этажа' if (($mod >= 2) and ($mod <= 4));
    return $txt . ' этажей';
}

sub _EXTEND_FLOOR {
    my ($self, $txt) = @_;
    $txt = $txt . ' этаж';
    return $txt;
}

sub _CUT_STREET_TYPE {
    my ($self, $txt) = @_;
    $txt =~ s/[\.\,\!\-]/\ /g;
    $txt =~ s/\b(ул|прт?|пл|пер|им|кв|а-яА-ЯA-Z|шоссе|просп(ект)?|улица|дом|д|переулок|)\b//ig;
    $txt =~ s/\d{3,}//g;
    $txt =~ s/^\W+|\W+$//g;
    return undef if ($txt =~ /^\d+$/);
    return $txt;
}

sub _EXTEND_AREA_VALUE {
    my ($self, $txt) = @_;
    return "[$txt кв м:$txt метров:$txt квадратных метров:$txt м2]";
}

sub _EXTEND_ROOMS {
    my ($self, $txt) = @_;
    my @vars = ('одно','двух','трех','четырех','пяти','шести','семи','восьми','девяти','десяти');
    my $v1 = $vars[$txt-1].'комнатная';
    return "[$v1:$txt комнаты]";
}

sub _EXTEND_ROOMS_SHORT {
    my ($self, $txt) = @_;
    return "$txt-к";
}

sub _WRAP_EXCL {
    my ($self, $txt) = @_;
    return unless ($txt);
    $txt =~ s/(\"|\')//gi;
    $txt =~ s/(^|\ |\"|\'|\«)(\w+)/$1\+$2/gi;
    return $txt;
}

sub _WRAP_AC {
    my ($self, $txt) = @_;
    my $ap_cmpl_name = $txt;
    my ($left_quote, $right_quote);
    if ($txt =~ m/(\"|\'|\«|\„|\‘)(.+)(\"|\'|\»|\”|\’)/i) {
        ($left_quote, $ap_cmpl_name, $right_quote) = ($1, $2, $3);
    }
    $ap_cmpl_name = ucfirst $ap_cmpl_name;
    return "ЖК ".$left_quote.$ap_cmpl_name.$right_quote if ($left_quote and $right_quote);
    return "ЖК \«$ap_cmpl_name\»";
}

sub _TAKE_FIRST {
    my ($self, $txt) = @_;
    $txt =~ s/\:.*$//;
    $txt =~ s/^\[//;
    return $txt;
}

sub _TAKE_AREA_TITLE {
    my ($self, $txt) = @_;
    return $txt if ($txt =~ /(га|сот[а-я]+)$/);
    $txt = $self->_TAKE_FIRST($txt);
    $txt =~ m/(\d+)([.,](\d))?/;
    $txt = $1;
    if ($3) {
        $txt = $txt . "." . $3;
    }
    return $txt." $meters_symbol";
}

sub _CUT_SUPPLIES {
    my ($self, $sup, $len) = @_;
    return undef unless ($sup);
    return $sup unless ($len);

    $sup =~ s/(^\[)|(\]$)//g;
    my @supplies = split ":", $sup;

    my $res = $supplies[0];
    my $i = 1;
    while (($i <= $#supplies) and (length($res.", ".$supplies[$i]) <= $len)) {
        $res .= ", ".$supplies[$i];
        ++$i;
    }
    return $res;
}

sub _CUT_SUPPLIES_15 {
    my ($self, $txt) = @_;
    return $self->_CUT_SUPPLIES($txt, 15);
}

sub _DENORM_LOT_AREA {
    my ($self, $txt) = @_;
    return $txt unless($txt =~ /соток$/);
    my ($num) = ($txt =~ /^(\d+)/);
    if (($num >= 5) and ($num <= 20)) {
        $txt = "$num соток";
    } elsif (($num % 10) == 1) {
        $txt = "$num сотка";
    } else {
        $txt = "$num сотки";
    }
    return $txt;
}

sub _PLURAL {
    my ($self, $txt) = @_;
    return $self->proj->phrase($txt)->set_gender_number_case({ number => "pl"});
}


#####
# Шаблоны для квартир
#
sub titles_flat {
    my $res = join ", ", (
        "rooms:_EXTEND_ROOMS_SHORT type area:_TAKE_AREA_TITLE <,> address_street:_CUT_STREET_TYPE:_UP_FIRST",
        "rooms:_EXTEND_ROOMS_SHORT type area:_TAKE_AREA_TITLE <,> apartment_cmplx:_WRAP_AC",
        "rooms:_EXTEND_ROOMS_SHORT type area:_TAKE_AREA_TITLE <,> metro:_EXTEND_METRO_SHORT",
        "rooms:_EXTEND_ROOMS_SHORT type area:_TAKE_AREA_TITLE <,> district:_UP_FIRST",
        "type ${prefix_feed}area:_TAKE_AREA_TITLE <,> floors:_EXTEND_FLOORS <,> lot_area:_DENORM_LOT_AREA",
        "type ${prefix_feed}area:_TAKE_AREA_TITLE <,> floors:_EXTEND_FLOORS <,> supplies:_CUT_SUPPLIES_15",
        "type ${prefix_feed}area:_TAKE_AREA_TITLE <,> lot_area:_DENORM_LOT_AREA <,> supplies:_CUT_SUPPLIES_15",
        "type ${prefix_feed}area:_TAKE_AREA_TITLE <,> floors:_EXTEND_FLOORS",
        "type ${prefix_feed}area:_TAKE_AREA_TITLE <,> lot_area:_DENORM_LOT_AREA",
        "type ${prefix_feed}area:_TAKE_AREA_TITLE <,> supplies:_CUT_SUPPLIES_15",
        "rooms:_EXTEND_ROOMS_SHORT type area:_TAKE_AREA_TITLE",
        "type area:_TAKE_AREA_TITLE <,> floor:_EXTEND_FLOOR",
        "type area:_TAKE_AREA_TITLE",
        "rooms:_EXTEND_ROOMS_SHORT type",
        "type lot_area:_DENORM_LOT_AREA <,> supplies:_CUT_SUPPLIES_15",
        "type lot_area:_DENORM_LOT_AREA" 
    );
    return $res;
}

#####
# Шаблоны для смарта
#
sub perf_templates_text_flat {
    my ($self) = @_;
    my $title_flat = $self->titles_flat;

    my $res = "
        denorm_price [district:_WRAP_EXCL/denorm_apartment_cmplx:_WRAP_EXCL/address_street:_WRAP_EXCL] denorm_rooms         {___MULT_PHRASE} => $title_flat
        denorm_price [district:_WRAP_EXCL/denorm_apartment_cmplx:_WRAP_EXCL/address_street:_WRAP_EXCL]                      {___MULT_PHRASE} => $title_flat
        denorm_area  [district:_WRAP_EXCL/denorm_apartment_cmplx:_WRAP_EXCL/address_street:_WRAP_EXCL]                      {___MULT_PHRASE} => $title_flat
        denorm_rooms [district:_WRAP_EXCL/denorm_apartment_cmplx:_WRAP_EXCL/address_street:_WRAP_EXCL]                      {___MULT_PHRASE} => $title_flat
        denorm_rooms denorm_area [district:_WRAP_EXCL/denorm_apartment_cmplx:_WRAP_EXCL/address_street:_WRAP_EXCL]          {___MULT_PHRASE} => $title_flat
        denorm_rooms denorm_area rounded_price denorm_apartment_cmplx:_WRAP_EXCL    {___MULT_PHRASE} => $title_flat
        denorm_rooms denorm_area rounded_price                                      {___MULT_PHRASE} => $title_flat
        denorm_apartment_cmplx:_WRAP_EXCL rounded_price                             {___MULT_PHRASE} => $title_flat
        address_street:_WRAP_EXCL                                                   {___MULT_PHRASE} => $title_flat

        geonames:_WRAP_EXCL [denorm_price/denorm_area/denorm_rooms]         {___MULT_PHRASE} => $title_flat

        pseudo_price [district/pseudo_apartment_cmplx] pseudo_rooms         {___MULT_PHRASE} => $title_flat
        pseudo_price [district/pseudo_apartment_cmplx]                      {___MULT_PHRASE} => $title_flat
        pseudo_price                                                        {___MULT_PHRASE} => $title_flat
        pseudo_area  [district/pseudo_apartment_cmplx]                      {___MULT_PHRASE} => $title_flat
        pseudo_rooms [district/pseudo_apartment_cmplx]                      {___MULT_PHRASE} => $title_flat
        pseudo_rooms pseudo_area [district/pseudo_apartment_cmplx]          {___MULT_PHRASE} => $title_flat
        pseudo_rooms pseudo_area rounded_price pseudo_apartment_cmplx       {___MULT_PHRASE} => $title_flat
        pseudo_rooms pseudo_area rounded_price                              {___MULT_PHRASE} => $title_flat
        pseudo_apartment_cmplx rounded_price                                {___MULT_PHRASE} => $title_flat
        district pseudo_rooms pseudo_area                                   {___MULT_PHRASE} => $title_flat
    ";

    $res =~ s/\btype\b(?=[^\n]*\=\>)//g;
    $res = $self->get_prefixes($res);
    return $res;
}

#####
# Шаблоны для динамики
#
sub dyn_templates_text_flat {
    my ($self) = @_;
    my $title_flat = $self->titles_flat;
    my $title_flat_template = "apartment_cmplx:_WRAP_AC <.> type:_PLURAL <от> denorm_price <млн>, rooms:_EXTEND_ROOMS_SHORT type area:_TAKE_AREA_TITLE <,> %SUB, rooms:_EXTEND_ROOMS_SHORT type <,> %SUB, type area:_TAKE_AREA_TITLE <,> %SUB, " . $title_flat;

    my $title_flat_metro = $title_flat_template;
    $title_flat_metro =~ s/\%SUB/metro:_EXTEND_METRO_SHORT/g;

    my $title_flat_address = $title_flat_template;
    $title_flat_address =~ s/\%SUB/address_street:_CUT_STREET_TYPE:_UP_FIRST/g;

    my $title_flat_district = $title_flat_template;
    $title_flat_district =~ s/\%SUB/district:_UP_FIRST/g;

    my $title_flat_apartment_cmplx = $title_flat_template;
    $title_flat_apartment_cmplx =~ s/\%SUB/apartment_cmplx:_WRAP_AC/g;

    my $res = "
        type metro:_WRAP_EXCL                           [new_flat:_TAKE_FIRST/mortgage/rooms:_EXTEND_ROOMS/decoration] {___MULT_PHRASE} {___MAX_20000} => $title_flat_metro
        type district:_WRAP_EXCL                        [new_flat:_TAKE_FIRST/mortgage/rooms:_EXTEND_ROOMS/decoration] {___MULT_PHRASE} {___MAX_20000} => $title_flat_district
        type address_street:_CUT_STREET_TYPE:_WRAP_EXCL [new_flat:_TAKE_FIRST/mortgage/rooms:_EXTEND_ROOMS/decoration] {___MULT_PHRASE} {___MAX_20000} => $title_flat_address
        type apartment_cmplx                            [new_flat:_TAKE_FIRST/mortgage/rooms:_EXTEND_ROOMS/decoration] {___MULT_PHRASE} {___MAX_20000} => $title_flat_apartment_cmplx

        type metro:_WRAP_EXCL                           {___MULT_PHRASE} {___MAX_20000} => $title_flat_metro
        type district:_WRAP_EXCL                        {___MULT_PHRASE} {___MAX_20000} => $title_flat_district
        type address_street:_CUT_STREET_TYPE:_WRAP_EXCL {___MULT_PHRASE} {___MAX_20000} => $title_flat_address
        type apartment_cmplx                            {___MULT_PHRASE} {___MAX_20000} => $title_flat_apartment_cmplx

        type metro:_WRAP_EXCL                           rooms:_EXTEND_ROOMS rounded_area {___MULT_PHRASE} {___MAX_20000} => $title_flat_metro
        type district:_WRAP_EXCL                        rooms:_EXTEND_ROOMS rounded_area {___MULT_PHRASE} {___MAX_20000} => $title_flat_district
        type address_street:_CUT_STREET_TYPE:_WRAP_EXCL rooms:_EXTEND_ROOMS rounded_area {___MULT_PHRASE} {___MAX_20000} => $title_flat_address
        type apartment_cmplx                            rooms:_EXTEND_ROOMS rounded_area {___MULT_PHRASE} {___MAX_20000} => $title_flat_apartment_cmplx

        type rounded_price {___MULT_PHRASE} {___MAX_10000} => $title_flat
        type rounded_area  {___MULT_PHRASE} {___MAX_10000} => $title_flat
        type lot_area {___MULT_PHRASE} {___MAX_10000} => $title_flat

        type metro:_WRAP_EXCL                           rounded_price [rooms/new_flat/decoration] {___MULT_PHRASE} {___MAX_20000} => $title_flat_metro
        type district:_WRAP_EXCL                        rounded_price [rooms/new_flat/decoration] {___MULT_PHRASE} {___MAX_20000} => $title_flat_district
        type address_street:_CUT_STREET_TYPE:_WRAP_EXCL rounded_price [rooms/new_flat/decoration] {___MULT_PHRASE} {___MAX_20000} => $title_flat_address
        type apartment_cmplx                            rounded_price [rooms/new_flat/decoration] {___MULT_PHRASE} {___MAX_20000} => $title_flat_apartment_cmplx

        type metro:_WRAP_EXCL                           rounded_price {___MULT_PHRASE} {___MAX_20000} => $title_flat_metro
        type district:_WRAP_EXCL                        rounded_price {___MULT_PHRASE} {___MAX_20000} => $title_flat_district
        type address_street:_CUT_STREET_TYPE:_WRAP_EXCL rounded_price {___MULT_PHRASE} {___MAX_20000} => $title_flat_address
        type apartment_cmplx                            rounded_price {___MULT_PHRASE} {___MAX_20000} => $title_flat_apartment_cmplx

        type address_street:_CUT_STREET_TYPE:_WRAP_EXCL rounded_area  {___MULT_PHRASE} {___MAX_20000} => $title_flat_address
        [rooms:_EXTEND_ROOMS/rounded_price/rounded_area] apartment_complex             {___MAX_20000} => $title_flat_apartment_cmplx
        rooms:_EXTEND_ROOMS rounded_price rounded_area apartment_complex               {___MAX_20000} => $title_flat_apartment_cmplx
   ";

    $res = $self->get_prefixes($res);
    return $res;
}


#####
# Добавляет ко всем кирпичам шаблонов префикс (FL_)
# Нужно для разделения исходных данных для ЖК и обычных квартир
#
sub get_prefixes {
    my ($self, $s) = @_;
    my $prefix = $prefix_flat;
    $s =~ s/(^|\n|\s|\[|\/|\>)(\w.+?)/$1${prefix}$2/g;
    return $s;
}

#####
# Одинокий баннер для смарта...
#
sub perf_banners_single {
    my ($self, %params) = @_;
    my $title  = $self->titles_flat;

    my $templates = "
        type district => $title,
        type metro => $title,
        type address_street => $title,
        type price {___MULT_PHRASE} => $title,
        type => $title,
    ";

    $templates  = $self->get_prefixes($templates);
    my $arr = $self->banners_data(
        templates_text => $templates,
        methods_arr => [ 'pack_list', ],
        max_count => 1,
        assert_no_rpc => 1,
        title_template_type => 'single',
        %params,
    );

    # если ничего не найдено, возвращаем первые несколько слов
    unless ( @$arr ){
        $arr = $self->banners_data(
            methods_arr => [ 'pack_list', ],
            max_count => 1,
            templates_text => "description:_FIRST_4_WORDS => $title, name:_FIRST_4_WORDS => $title",
            assert_no_rpc => 1,
            title_template_type => 'fallback',
            %params,
        );
    }

    return @$arr ? [$arr->[0] ] : [];
}

#####
# Шаблоны для смарта с псевдокирпичами
#
sub perf_templates_text {
    my ($self) = @_;
    return $self->perf_templates_text_flat;
}


#####
# Шаблоны для динамики без псевдокирпичей
#
sub dyn_templates_text {
    my ($self) = @_;
    return $self->dyn_templates_text_flat;
}

#####
# Тюнинг для смарта
# ! без get_search_filtered50k
#
sub perf_methods_arr  :GLOBALCACHE {
    my ($self) = @_;
    my @res = map { s/^\s+//; $_ } grep {/\S/} grep {!/#/} ##no critic
        split /\n/, '
        get_wide_filtered                       goods accessory
#        postfilter_base                         goods accessory P
        snorm_phrase_list                       goods accessory
        pack_phr_lite                           goods accessory
        pack_list                               goods accessory
        set_exclamations_before_stops           goods accessory
        set_exclamations_before_bsstops         goods accessory
        replace_exclamations_with_pluses        goods accessory
    ';
    return @res;
}

#####
# Заглушка для динамики
#
sub dyn_methods_arr :GLOBALCACHE {
    my ($self) = @_;
    return $self->perf_methods_arr;
}

1;
