use strict;
use warnings;
use utf8;

package BM::BannersMaker::ProductTiresDisks;

use base qw(BM::BannersMaker::Product);
use Utils::Array;
use Data::Dumper;
use List::Util qw(min max);

__PACKAGE__->mk_accessors(qw(
    type
    brand
    season
));

my @offer_fields = qw(model);

sub get_offer_fields {
    return @offer_fields;
}

sub init {
    my ($self) = @_;
    $self->SUPER::init;
    my $data = $self->{data}; 
    if(ref($data->[0]) eq 'HASH'){
        $self->{$_} = $data->[0]{$_} for @offer_fields;
    }
}

########################################################
#Доступ к полям
########################################################

sub ad_type {
    return 'tires_disks';
}

sub match_type {
    return 'norm';
}


sub parse :CACHE {
    my ($self) = @_;
    my $dbg = 0;

    my $h = $self->SUPER::parse;
    print STDERR "h: \n", Dumper($h) if $dbg;

    my $txtsource = $self->txtsource;
    print STDERR "txtsource: $txtsource\n" if $dbg;


    #определяем tires_brand disks_brand type
    my @disks_brands_list = $self->get_disks_brands_list;
    my @tires_brands_list = $self->get_tires_brands_list;

    my $brands_re = {
        disks => join('|', map {quotemeta($_)} @disks_brands_list),
        tires => join('|', map {quotemeta($_)} @tires_brands_list),
    };
    my $type;

    if (($h->{type} =~ /(шина|шины|диск|диски)/i) || ($h->{minicategs} =~ /(шины|диски)/i)) { 
        my $detected_type = lc($1) =~ /шин/ ? 'tires' : 'disks';

        if (my ($brand) = map {$_ =~ /(?:^|\s+)($brands_re->{ $detected_type })(?:\s+|$)/i} 
                         grep {$_} ($h->{brand}, $txtsource, $self->{vendor})) {
            $type = $detected_type;
            $h->{ $type . '_brand'} = $brand;
            delete $h->{brand};
            print STDERR "found brand: $brand ($type)\n" if $dbg;
        }
    }

    if ($type and $type eq 'disks' and (!$h->{type} or $h->{type} !~ /(^|\s+)(авто)?(диск|диски)(\s+|$)/i)) {
        $h->{type} = 'диски';
    } elsif ($type and $type eq 'tires' and (!$h->{type} or $h->{type} !~ /(^|\s+)(авто|мото|вело)?(шина|шины|покрышка|покрышки)(\s+|$)/i)) {
        $h->{type} = 'шины';
    }
    


    #определяем size и pcd
    $txtsource =~ s/(\d+),(\d+)/$1.$2/g; #7x18/5x114,3 ЕТ35 D67,1
    $txtsource =~ s/[\(\)]/ /g;          #96S (195 / 80 / R15)

    my ($size, $rest) = $self->cut_size($txtsource, $dbg);
    print STDERR "size: \n", Dumper($size) if $dbg;

    if ($size and scalar(keys %$size)) {
        #вырезаем размеры из модели
        if ($h->{model}) {
            my $seen = {};
            ++$seen->{lc($_)} for map {split /[\s\/\*\\]+/, $_} ($txtsource, $rest);
            #print STDERR "seen only once: ", join(":", grep {$seen->{$_} == 1} keys %$seen) . "\n" if $dbg;
            $h->{model} =~ s/(^|\s+)\Q$_\E(?=\s+|$)/ /i for grep {$_ and $seen->{$_} == 1} keys %$seen;
        }

        if ($size->{PCD} and $size->{PCD} =~ /^\s*(\d+)[хx\*](\d+)(?:\.(\d+))?\s*$/i) {
            #$h->{pcd} = "[$1x$2:$1х$2]"; #англ./рус.
            $h->{pcd} = $3 ? "[$1x$2.$3:$1 $2.$3]" : "$1x$2"; 
            $h->{pcd_title} = $3 ? "$1*$2.$3" : "$1*$2"; 
        }
        $h->{size} = $self->get_sizes_list_for_template($size);
    }



    #определяем auto_brand
    my @auto_brands_list = $self->get_auto_brands_list;
    my $auto_brands_re = join('|', map{quotemeta($_)} @auto_brands_list);

    if ($rest =~ s/(?:^|\s+)($auto_brands_re)(?:\s+|$)/ /i) {
        $h->{auto_brand} = $1;
        delete $h->{brand} if $h->{brand} && $h->{brand} =~ /\Q$h->{auto_brand}\E/i;
    }

    #определяем runflat
    if ($rest =~ s/(?:^|\s+)(runflat|run\s+flat)(?:\s+|$)/ /i) {
        $h->{runflat} = $1;
    }

    #определяем season и spike
    if ($h->{type} and $h->{type} =~ /шин/ and $h->{type} =~ s /(?:^|\s+)((?:всесезонн|зимн|летн)(?:ая|яя|ые|ие))(?:\s+|$)/ /i) {
        print STDERR "change type: $h->{type}\n" if $dbg;
        $h->{season} = $1;
    }

    my $params = $self->get_params($self->{params}, $dbg);
    print STDERR "params: ", Dumper($params) if $dbg;
    
    my $season_map = {'winter' => 'зимний', 'ice' => 'зимний', 'all season' => 'всесезонный'};
    if (!$h->{season} and $h->{type} and $h->{type} =~ /шин/) { 
        if (my ($season) = grep {$rest =~ /(?:^|\s+)\Q$_\E(?:\s+|$)/i} keys %$season_map) {
            $h->{season} = $season_map->{$season};
        } elsif ($params->{season}) {
            $h->{season} = $params->{season};
        }
    }
    $h->{spike} ||= $params->{spike}  if $params->{spike};

    if ($params->{adjective} and $h->{type} and $h->{type} !~ /[а-яё]+(ой|ый|ий|ые|ие|ая|яя)(\s+|$)/) {
        my $adjective = $self->proj->phrase($h->{type})->harmonize($params->{adjective});
        $h->{type} = "$adjective $h->{type}";
    }

    if ($h->{type} and $h->{type} =~ s/^(шин[аы])\s*(шип)$/$1/i) {
        $h->{spike} ||= 'шипованный';
    }


    #вырезаем из model найденные поля
    if ($h->{model}) {
        foreach my $field (grep {$h->{$_}} qw(auto_brand runflat disks_brand tires_brand)) {
            $h->{model} =~ s/(^|\s+)\Q$h->{$field}\E(\s+|$)/ /i 
        }
        $h->{model} = join(' ', grep {$_ !~ /^1?\d{1,2}(\.\d+)?$/} split /\s+/, $h->{model});
        $h->{model} =~ s/\s+[xх]\s*$//i;
    }
    foreach my $field (grep {$h->{$_}} qw(model type)) {
        $h->{$field} =~ s/\s+/ /g;
        $h->{$field} =~ s/(^\s+|\s+$)//g;
    }


    # согласование словоформ для заголовков
    for my $field ( grep { $h->{$_} } qw/season/ ){ #пока нет spike, тк шипованный -> шиповавшая 
        $h->{$field} = $self->proj->phrase($h->{type})->harmonize($h->{$field});
    }


    #объединяем runflat и spike в key_property
    my @key_property_fields = qw(runflat spike);
    $h->{key_property} = join(' ', grep {$_} map { $h->{$_} } @key_property_fields);
    delete $h->{$_} for @key_property_fields;

    delete $h->{$_} for grep { !$h->{$_} } keys %$h;
    

    #отладка: смотрим что осталось в rest
    my @fields = qw(brand model type size pdc tires_brand disks_brand auto_brand runflat season spike);
    $rest =~ s/\Q$_\E/ /i for grep {$_} map {$h->{$_}} @fields;
    print STDERR "h: \n", Dumper($h) if $dbg;
    print STDERR "stat: rest: $rest\n" if $dbg;
    print STDERR join('', map {"stat: $_: $h->{$_}\n"} grep {$h->{$_}} @fields) if $dbg;

    return $h;
}


sub get_disks_brands_list {
    my $self = shift;
    my $disks_brands = $self->proj->dict_manager->get_dict("brands_disks");
    return @{$disks_brands->phrase_list->perl_array};
}

sub get_tires_brands_list {
    my $self = shift;
    my $tires_brands = $self->proj->dict_manager->get_dict("brands_tires");
    return @{$tires_brands->phrase_list->perl_array};
}

sub get_auto_brands_list {
    my $self = shift;
    my $auto_brands = $self->proj->dict_manager->get_dict("brands_auto");
    return @{$auto_brands->phrase_list->perl_array};
}


sub cut_size {
    my ($self, $text, $dbg) = @_;

    my ($disks_res, $disks_text) = $self->cut_disks_size($text, $dbg);
    my ($tires_res, $tires_text) = $self->cut_tires_size($text, $dbg);

    if (scalar(keys %$disks_res) > scalar(keys %$tires_res)) {
        return ($disks_res, $disks_text);
    } else {
        return ($tires_res, $tires_text);
    }
}


#Автомобильные шины
#175/65 R14 82H

#Диаметр 14" 
#Ширина профиля 175 мм 
#Высота профиля 65 % 
#Индекс скорости H (до 210 км/ч) 
#Индекс нагрузки 82 (475 кг)  

sub cut_tires_size {
    my ($self, $text, $dbg) = @_;
    my $res = {};

    #175/65R14 
    my $diameter_re = '(?:' . join('|', 12..26) . ')(?:\.\d+)?';
    if ($text =~ s/(?:^|\s+)((\d{2,3})[\/\s](\d{2})\s?(r\s?(?:$diameter_re)c?))(?:\s+|$)/ /i) { 
        @$res{qw(tires_width tires_height diameter)} = ($2, $3, $4);
    }

    if ($text =~ s/(?:^|\s+)(\d{2}r\d{2})(?:\s+|$)/ /i) { 
        print STDERR "size: $1\n" if $dbg;
    }

    #235/50
    if ($text =~ s/(?:^|\s+)((\d{3})[\s\/\-]+(\d{2}))(?:\s+|$)/ /i) { 
        @$res{qw(tires_width tires_height)} = ($2, $3);
    }

    #R16C
    if ($text =~ s/(?:^|\s+)([rр]-?(?:$diameter_re)c?)(?:\s+|$)/ /i) {
        $res->{diameter} = $1;
    }

    #82T
    if ($text =~ s/(?:^|\s+)(1?\d{2}[a-z])(?:\s+|$)/ /i) {
        $res->{tires_index} = $1;
    }

    $text =~ s/\s+/ /g;
    $text =~ s/(^\s+|\s+$)//g;

    return ($res, $text);
}


#Колесные диски
#7J17 5*112 ET 43 DIA 57.1
#6.5x16/5x114.3 ET45 D60.1 BKF
#R15 W5.5 PCD4x114.3 ET40 DIA66

sub cut_disks_size {
    my ($self, $text, $dbg) = @_;
    my $res = {};

    #Диск литой 6.5''х15.0'' 5x100 ЕТ38
    $text =~ s/(\d+)(\.0)?\'\'/$1 /g;
    $text =~ s/(\d+)(?:\.0)(\D|$)/$1$2/g;

    #Ширина обода (J): 6 / 6.5 / 7 / 7.5 / 8 "
    #Диаметр обода (D): 15 / 16 / 17 / 18 "
    my $width_re = '(?:' . join('|', 3..15) . ')(?:\.\d+)?';
    my $diameter_re = '(?:' . join('|', 12..26) . ')(?:\.\d+)?';
   
    if ($text =~ s/(?:^|\s+)($width_re)J?\s*[Jхx]\s*($diameter_re)(?:\s+|\/|$)/ /i) { 
        $res->{disks_width} = $1;
        $res->{diameter} = $2;
    }

    #8\R18
    if ($text =~ s/(?:^|\s+)($width_re)\\(R(?:$diameter_re))(?:\s+|\/|$)/ /i) { 
        $res->{disks_width} = $1;
        $res->{diameter} = $2;
    }

    #W8.5
    if ($text =~ s/(?:^|\s+)W\s*($width_re)(?:\s+|$)/ /i) { 
        $res->{disks_width} = $1;
    }
    
    #Крепежные отверстия (NxPCD) 4x100 
    #Количество крепежных отверстий: 4 / 5 
    #Диаметр расположения отверстий: 98 / 100 / 105 / 108 / 110 / 112 / 114.3 / 115 / 120 мм
    my $PCD_num_re = '(?:' . join('|', 3..12) . ')';
    my $num_re = '\d+(?:\.\d+)?';
    my ($PCD_diameter_from, $PCD_diameter_to) = (90, 335);
    if ($text =~ /(?:^|\s+|\/)(?:PCD)?($PCD_num_re[хx\*]($num_re))(?:\s+|$)/i 
        and $2 >= $PCD_diameter_from and $2 <= $PCD_diameter_to) { 
        $res->{PCD} = $1;
        $text =~ s/(?:^|\s+|\/)(?:PCD)?(\Q$res->{PCD}\E)(?:\s+|$)/ /i;
    }

    #5x105 R17 / 7.0 38
    if ($text =~ s/(?:^|\s+)(R\s*(?:$diameter_re))\s*\/\s*($width_re)(?:\s+|$)/ /i) { 
        $res->{diameter} = $1;
        $res->{disks_width} = $2;
    }

    #R20
    if ($text =~ s/(?:^|\s+)(R\s*(?:$diameter_re))(?:\s+|$)/ /i) { 
        $res->{diameter} = $1;
    }

    #Вылет (ET): 32...52 мм
    if ($text =~ s/(?:^|\s+)(?:ET|ЕТ)\s*(\d{2}(\.\d+)?)(?:\s+|$)/ /i) { 
        $res->{ET} = $1;
    }
 
    #Диаметр центрального отверстия (DIA): 54.1...70.3 мм
    if ($text =~ s/(?:^|\s+)(?:DIA)\s*(1?\d{2}(\.\d+)?)(?:\s+|$)/ /i 
     or $text =~ s/(?:^|\s+)(?:D)\s*(1?\d{2}\.\d+)(?:\s+|$)/ /i) { 
        $res->{DIA} = $1;
    }

    #Цветовые обозначения дисков
    #BKF (BFP, BK/FP) – черный, полированный частично 
    if ($text =~ s/(?:^|\s+)(BKF|BFP|BK|FP|S|HP|SF|HS|CH|G|GF|GM|GMF|FGMF|FWF|FSF|HPB|HPL|MB|MBF|MGM|MW|W|WF)(?:\s+|$)/ /) { 
        $res->{disk_color} = $1;
    }

    $text =~ s/\s+/ /g;
    $text =~ s/(^\s+|\s+$)//g;

    return ($res, $text);
}


sub get_sizes_list_for_template {
    my ($self, $size) = @_;
    my @res;

    if ($size->{diameter}) {
        if ($size->{diameter} =~ /(\d+(?:\.\d+)?)/) {
            $size->{diameter_without_R} = $1;
        }

        if ($size->{diameter} =~ /^r\s+/ig) {
            $size->{diameter_with_space} = $size->{diameter};
            $size->{diameter} =~ s/\s+//g;
        } elsif ($size->{diameter_without_R}) {
            $size->{diameter_with_space} = "R ".($size->{diameter_without_R});
        }
    }

    my $templates = [
        {fields => [qw(diameter tires_width tires_height tires_index)], templ => '%s %s/%s %s'},
        {fields => [qw(diameter_with_space tires_width tires_height tires_index)], templ => '%s %s/%s %s'},
        {fields => [qw(diameter tires_width tires_height)], templ => '%s %s/%s'},
        {fields => [qw(diameter_with_space tires_width tires_height)], templ => '%s %s/%s'},
        {fields => [qw(tires_width tires_height)], templ => '%s/%s'},
        {fields => [qw(disks_width diameter_without_R ET)], templ => '%sx%s ET%s'},
        {fields => [qw(disks_width diameter_without_R)], templ => '%sx%s'},
        {fields => [qw(disks_width diameter_without_R ET DIA)], templ => '%sx%s ET%s DIA%s'},
        {fields => [qw(disks_width diameter_without_R ET DIA)], templ => '%sx%s ET%s D %s'},
        {fields => [qw(disks_width diameter_without_R DIA)], templ => '%sx%s %s'},
        {fields => [qw(diameter)], templ => '%s'},
        {fields => [qw(diameter_with_space)], templ => '%s'},
    ];
    foreach my $t (@$templates) {
        next if grep {!$size->{$_}} @{ $t->{fields} };
        my $text = sprintf($t->{templ}, map { $size->{$_} } @{ $t->{fields} });
        push @res, $text;
        $text =~ s/[\.\,]/ /g; # SUPBL-43
        push @res, $text;
    }
    return @res == 0 ? '' : (@res == 1 ? $res[0] : '[' . join(':', @res) . ']');
}


sub get_params {
    my ($self, $params_text, $dbg) = @_;
    return {} unless $params_text;
    my $res = {};

    #pcd:114,3,ширина:4,5,диаметр:13
    $params_text =~ s/(\d+),(\d+)/$1\.$2/g;
    print STDERR "params_text: $params_text\n" if $dbg;

    my $params = {map {split /:\s*/, $_} split /,\s*/, $params_text};

    my $season_map = {
        'зима' => 'зимний', 'зимние' => 'зимний', 
        'лето' => 'летний', 'летние' => 'летний', 
        'всесезонные' => 'всесезонный', 'всесезонная' => 'всесезонный'
    };
    foreach my $f ('сезон', 'сезонность', 'тип') {
        next unless $params->{$f};
        if (my ($season) = grep {$params->{$f} =~ /\Q$_\E/} keys %$season_map) {
            $res->{season} = $season_map->{$season};
        }
    }

    if ($params->{'шипы'} and $params->{'шипы'} eq 'да' or 
       $params->{'сезон'} and $params->{'сезон'} =~ /\(шипованные\)/) {
        $res->{spike} = 'шипованный';
    }

    if ($params->{'тип'} and $params->{'тип'} =~ /^(литые|литой|штампованные|штампованный|грузовые)$/i) { 
        $res->{'adjective'} = $1;
    }
    return $res;
}


sub dyn_templates_text {
    my ( $self ) = @_;

    my $title = "type brand model size:_DEMULT pcd_title, type brand size:_DEMULT pcd_title, brand size:_DEMULT pcd_title, type brand pcd_title, type brand model size:_DEMULT, brand model size:_DEMULT, type brand size:_DEMULT, brand size:_DEMULT, type brand model, type brand, brand model, type size:_DEMULT pcd_title, type size:_DEMULT, type, brand";

    my $tires_brand_title = "season type tires_brand model size:_DEMULT, type tires_brand model size:_DEMULT, tires_brand model size:_DEMULT, season type tires_brand size:_DEMULT, type tires_brand size:_DEMULT, tires_brand size:_DEMULT, season type tires_brand model, type tires_brand model, type tires_brand";
    my $season_tires_title = "$tires_brand_title, $title";

    my $disks_brand_title = "type disks_brand model size:_DEMULT pcd_title, type disks_brand size:_DEMULT pcd_title, disks_brand size:_DEMULT pcd_title, type disks_brand model size:_DEMULT, type disks_brand size:_DEMULT, disks_brand size:_DEMULT, type disks_brand model pcd_title, type disks_brand pcd_title, disks_brand pcd_title, type disks_brand model, type disks_brand";
    my $disks_title = "$disks_brand_title, $title";

    my $common_title = "$tires_brand_title, $disks_brand_title, $title";

    #my $type = '[type/type:_DEL_ADJ/type:_MAINWORDS]';
    my $type = '[type/type:_DEL_ADJ]';

    my $res = "
        size $type {___MULT_PHRASE} => $common_title
        auto_brand $type => $common_title
        tires_brand $type => $season_tires_title
        season size $type {___MULT_PHRASE} => $season_tires_title
        key_property $type => $season_tires_title
        season tires_brand $type => $season_tires_title
        disks_brand $type => $disks_title
        model $type => $common_title
        auto_brand size $type {___MULT_PHRASE} => $common_title
        key_property size $type {___MULT_PHRASE} => $season_tires_title
        key_property season $type => $season_tires_title
        auto_brand season $type => $season_tires_title
        size tires_brand $type {___MULT_PHRASE} => $season_tires_title
        model season $type => $season_tires_title
        model season tires_brand $type => $season_tires_title
        key_property season size $type {___MULT_PHRASE} => $season_tires_title
        model size $type {___MULT_PHRASE} => $common_title
        disks_brand size $type {___MULT_PHRASE} => $disks_title
        key_property tires_brand $type => $season_tires_title
        season size tires_brand {___MULT_PHRASE} => $season_tires_title
        key_property tires_brand => $season_tires_title
        auto_brand disks_brand model $type => $disks_title
        pcd $type {___MULT_PHRASE} => $disks_title
        auto_brand key_property $type => $season_tires_title
        season size tires_brand $type {___MULT_PHRASE} => $season_tires_title
        disks_brand model $type => $disks_title
        pcd size $type {___MULT_PHRASE} => $disks_title
        key_property season tires_brand $type => $season_tires_title
        auto_brand tires_brand => $season_tires_title
        auto_brand tires_brand $type => $season_tires_title
        model season size $type {___MULT_PHRASE} => $season_tires_title
        key_property model $type => $season_tires_title
        auto_brand model size $type {___MULT_PHRASE} => $common_title
        auto_brand key_property model $type => $season_tires_title
        key_property model season $type => $season_tires_title
        auto_brand model season $type => $season_tires_title
#для аксессуаров
        brand model $type => $common_title
        brand $type => $common_title
#нет в статистике
        pcd disks_brand $type {___MULT_PHRASE} => $disks_title
        auto_brand disks_brand $type => $disks_title
        auto_brand disks_brand size $type {___MULT_PHRASE} => $disks_title
        brand model => $common_title
        disks_brand model => $disks_title, $title
        tires_brand model => $season_tires_title, $title
        disks_brand model size {___MULT_PHRASE} => $disks_title, $title
        tires_brand model size {___MULT_PHRASE} => $season_tires_title, $title
    ";
    return $res;
}

sub perf_templates_text {
    my ( $self ) = @_;
    return $self->dyn_templates_text;
}

sub dyn_methods_arr :GLOBALCACHE {
    my ($self) = @_;
    my @res = map { s/^\s+//; $_ } grep {/\S/} grep {!/#/} ##no critic
        split /\n/, '
        modellike_product                       goods accessory P
        get_search_filtered50k                  goods accessory
        add_dynamic_homonymy_words              goods accessory P
        get_wide_filtered                       goods accessory
        #postfilter_product                      goods accessory P
        snorm_phrase_list                       goods accessory
        pack_phr_lite                           goods accessory
        pack_list                               goods accessory
        get_search_filtered50k                  goods accessory
        set_exclamations_before_stops           goods accessory
        set_exclamations_before_bsstops         goods accessory
        replace_exclamations_with_pluses        goods accessory
    ';
    return @res;
}

sub banner_single_templates_text {
    my $self = shift;
    return $self->perf_templates_text;
}

1;
