package BM::BannersMaker::ProductWear;

use strict;
use utf8;

use base qw(BM::BannersMaker::Product);
use Data::Dumper;
use List::Util qw(min);
use Utils::Array;

########################################################
#Доступ к полям
########################################################

__PACKAGE__->mk_accessors(qw(
    vendor
    model
    sex
    color
    season
    collection
    age
    material
));

my @offer_fields = qw{sex color season age material};

sub get_offer_fields {
    return @offer_fields;
}

# хеш для маппинга значений атрибутов
our %h_attr_values_map = (
    'season' => {'весна'=>'весенний', 'осень'=>'осенний', 'лето'=>'летний', 'зима'=>'зимний', 'демисезон'=>'демисезонный'},
    'sex' => {'woman'=>'женский', 'women'=>'женский', 'man'=>'мужской', 'men'=>'мужской'},
);

my $SIZE_UNITS = ['RU', 'AU', 'DE', 'EU', 'FR', 'Japan', 'INT', 'IT', 'UK', 'US', 'BRAND', 'INCH', 'Дюймы', 'Height', 'Рост', 'Round', 'Окружность'];

my $h_latin_colors_map;

sub class_init :RUN_ONCE {
    my $class = shift;
    my %par = @_;
    my $proj = $par{proj};

    # Load color mapping
    $h_latin_colors_map = $proj->fdm->latin_colors_map;
    $class->load_dict_colors_full;
}

sub init {
    my ($self) = @_;
    $self->SUPER::init;
    $self->class_init(proj => $self->proj);
    my $data = $self->{data};
    if(ref($data->[0]) eq 'HASH'){
        $self->{$_} = $data->[0]{$_} for @offer_fields;
        my $additional_data = $self->_format_additional_data(_parse_clothes_offer($data->[0]));
        for my $text_param (grep {$additional_data->{$_}} keys(%$additional_data)) {
            $self->{$text_param} //= $additional_data->{$text_param};
            $self->{additional_data}{text}{$text_param} //= $additional_data->{$text_param};
        }
    }
    # маппинг с нормализацией, но без латиницы
    $self->{sex} = ($self->proj->phrase( $self->{sex} )->cut_sex)[0] if $self->{sex};
}

sub ad_type {
    return 'clothes';
}

sub match_type {
    return 'norm';
}

sub parse :CACHE {
    my ($self) = @_;

    if ( $self->model ){
        #если похоже на код/артикул, то обнуляем, исключение - 3 и 4 цифры
        $self->{model} = '' if ( !($self->{model}=~/\w{5,}/) || $self->{model}!~/ / && $self->{model}=~/\d{2,}/ && $self->{model}!~/^\d{3,4}$/);
        # Соединяем вендора и модель (если вендор ещё не содержится в модели). Если в моделе есть тип, то пишем с маленькой буквы
        if ( $self->{vendor} && $self->{model} &&
            !($self->proj->phrase_list([$self->{vendor}])->search_subphrases_in_phrase($self->proj->phrase($self->{model})))) {
            $self->{model} = $self->vendor.' '.$self->model;
        }
    }

    # проверка фидовых значений color, age. season, material
    $self->check_attr;

    my $h = $self->SUPER::parse;

    $h->{type} = '' if $h->{type} =~ /ткань/i;
    $h->{model} = $self->proj->phrase( $h->{model} )->fix_dup_words;
    $h->{sex} ||= ($self->proj->phrase($self->{sex})->cut_sex)[0] if ( $self->{sex} && $self->{sex} ne $self->reset_flag );

    # одежная добавка
    my $additions = ['большого размера', 'для полных'];
    my $women_additions = ['для беременных','для будущих мам','для кормящих'];
    if ($h->{sex} =~ m/жен/i) {
        push(@$additions, @$women_additions);
    }
    my $phl_additions = $self->proj->phrase_list($additions);
    my $source_text = '';
    $source_text = join (' ', map { $_ || () } ( $self->model, $self->name, $self->proj->phrase( $self->description )->head(10), $self->categpath, $self->minicategs ) );
    $source_text = join (' ', $self->proj->phrase( $source_text )->words );
    my $phl_additions_found = $phl_additions->search_subphrases_in_phrase( $self->proj->phrase( $source_text ), safe_norm=>1 );

    #отфильтровываем добавки, которые есть в type
    if ( $h->{type} && $phl_additions_found && $phl_additions_found->count ) {
        my $phl_additions_found_in_type = $phl_additions->search_subphrases_in_phrase( $self->proj->phrase( $h->{type} ), safe_norm=>1 );
        if ( $phl_additions_found_in_type && $phl_additions_found_in_type->count ){
            $phl_additions_found = $phl_additions_found - $phl_additions_found_in_type;
        }
    }

    if ( $phl_additions_found && $phl_additions_found->count ){
        my @additions_found = @{$phl_additions_found->perl_array};
        my $addition = shift @additions_found;
        my $phl_additions_text = join(' ', @{$phl_additions->perl_array} );
        $h->{wear_addition} = $self->proj->phrase( $phl_additions_text )->_denorm_text( $addition );
        $h->{wear_addition} = $1 if ( $phl_additions_text =~ /(для $h->{wear_addition})/ );
    }
    # age становится одежной добаквой, если он со словом для, то есть не "детский"
    $h->{age} = '';
    my ( $age, undef ) = ( $self->proj->phrase( $source_text )->cut_age );
#    print STDERR "$age => $source_text\n";
    if ( $age ){
        if ( $age =~ /(для.+?)(?:\:|$)/ ){
            $h->{wear_addition} = $1;
            $h->{age} = '';
        } else {
            $h->{age} = (split/\:/, $age)[0];
        }
    }

    if ( $h->{brand} ){
        # если брэнд и магазин совпадают, обнуляем брэнд
        $h->{shopname} = '' if ( $h->{shopname} && lc($h->{brand}) eq $h->{shopname} );
        # поднимаем первую букву
        $h->{brand} =~ s/^(.*)$/\u$1/ if $h->{brand};
    }

    # если не распарсилось, берем из полей фида
    $h->{season} ||= ($self->proj->phrase($self->{season})->cut_season)[0] if ( $self->{season} && $self->{season} ne $self->reset_flag );
    $h->{age} ||= ($self->proj->phrase($self->{age})->cut_age)[0] if ( $self->{age} && $self->{age} ne $self->reset_flag );
    $h->{material} ||= $self->{material} if ( $self->{material} && $self->{material} ne $self->reset_flag );

    for ( qw/sex season age material/ ){
        $h->{$_} =~ s/\:.*?$// if ( $self->$_&& $h->{$_} );
    }

    if ( $self->color && !$h->{color} ){ # цвет нужно выпарсивать из поля фида из-за мусора типа средний синий, глубокий синий
        my ( $colors ) = $self->proj->phrase( $self->color )->cut_colors;
        $h->{color} = (split/\:/,$colors)[0] if $colors;
    }

    # если нет в полях фида, берем из дескрипшн
    if ( $self->description && !$h->{sex} ){
        my ( $sex ) = $self->proj->phrase($self->description)->cut_sex;
        $h->{sex} = $sex if ( $sex && $sex !~ /\:/ ); # не берем пол, если найдено несколько, чтобы не сгенерилось мужское платье
    }
    if ( $self->description && !$h->{color} ){
        my ( $colors ) = $self->proj->phrase($self->description)->cut_colors;
        $h->{color} = (split/\:/,$colors)[0] if $colors;
    }
    if ( $self->description && !$h->{season} ){
        $h->{season} = ($self->proj->phrase($self->description)->cut_season)[0];
    }
    # filter collection-like season values (see SUPBL-1935, SUPBL-2447)
    delete $h->{season} if ( $h->{season} && ( $h->{season} =~ /мульти/i || $h->{season} =~ /-|\d+/) );

    if ( $self->description && !$h->{wear_prop} ){
        my $wear_prop = '';
        $wear_prop = ($self->proj->phrase($self->description)->cut_wear_property)[0];
        $h->{wear_prop} = $wear_prop if $wear_prop;
    }

    if ($h->{type} && $h->{wear_prop}) {
        my $type_ph = $self->proj->phrase($h->{type});
        my @wear_props = grep { ($self->proj->phrase($_) ^ $type_ph) eq $self->proj->phrase($_)} split /\:/, $h->{wear_prop};
        $h->{wear_prop} = join(':', @wear_props);
    }

    # раньше ходили на лендинги за цветом, если не получилось определить
    # убрали, т.к.:
    # - долго было сломано
    # - невозможно было парсить на YT
    # - на лендинге часто парсились цвета не этого товара, а другого (внизу страницы или сбоку)

    # удаляем все лишние символы и переводим в нижний регистр
    for my $key ( grep { $h->{$_} } qw/color season age sex material/ ){
        $h->{$key} = lc $h->{$key};
        $h->{$key} =~ s/(^[^0-9a-zа-яё-]+|[^0-9a-zа-яё-]+$)//g;
    }

    # маппинг значений атрибутов
    for my $keymap ( keys %h_attr_values_map ){
        my $hmap = $h_attr_values_map{$keymap};
        $h->{$keymap} //= '';
        $h->{$keymap} = $hmap->{$h->{$keymap}} if $hmap->{$h->{$keymap}//''};
    }

    my $simple_color = $self->get_simple_color($h->{color});

    # согласование словоформ для заголовков
    for my $text ( grep { $h->{$_} } qw/color season age sex/ ){
        $h->{$text} = $self->proj->phrase($h->{type})->harmonize($h->{$text});
    }
    $simple_color = $self->proj->phrase($h->{type})->harmonize($simple_color) if $simple_color;

    # вычитаем из типа лишнее
    if ( $h->{type} && $h->{season} ){
        $h->{type} = ( $self->proj->phrase($h->{type}) ^ $self->proj->phrase($h->{season}) )->text;
    }
    if ( $h->{type} && $h->{sex} ){
        $h->{type} = ( $self->proj->phrase($h->{type}) ^ $self->proj->phrase($h->{sex}) )->text;
    }
    if ( $h->{type} && $h->{age} ){
        $h->{type} = ( $self->proj->phrase($h->{type}) ^ $self->proj->phrase($h->{age}) )->text;
    }

    # мапим цвет на латинице, только если есть тип
    # зануляем цвет, если он входит в часть после "c|со" (поло с синими вставками)
    if ( $h->{color} && $h->{type} ){
        $h->{color} = '' if ( $h->{brand} && ($self->proj->phrase( $h->{brand} )->cut_colors)[0] );
        $h->{color} = $h_latin_colors_map->{lc($h->{color})} if $h_latin_colors_map->{lc($h->{color})};
        # ищем плохую фразу, когда тип перед "с|со", а цвет после
        for my $text ( grep { $_ && /\S/ } ( $self->model, $self->name, $self->description ) ){
            my ( $before_c, $after_c ) = split/\s+(?:с|со)\s+/, $text;
            next unless defined( $after_c );
            next unless (       $self->proj->phrase_list( [$before_c, $h->{type}] )->intersection(norm=>1)->count
                            &&  $self->proj->phrase_list( [$after_c, $h->{color}] )->intersection(norm=>1)->count );
            $h->{color} = '';
        }
    }

    # взрослый и так по умолчанию, он не нужен
    if ( $h->{age} ){
        $h->{age} = '' if $h->{age} =~ /взросл/i;
    }

    # зануляем shopname, если он совпадает с брэндом
    if ( $h->{shopname} && $h->{brand} ){
        $h->{shopname} = '' if ( $h->{brand} =~ /\Q$h->{shopname}\E/i || $h->{shopname} =~ /\Q$h->{brand}\E/i );
    }

    $h->{$_}=lc($h->{$_}) for ( grep { $h->{$_} } qw/season age collection color sex wear_prop/ );

    if ($h->{color} and $simple_color and $h->{color} ne $simple_color) {
        $h->{color} = $h->{color} . ':' . $simple_color;
    }

    # заносим данные из аксессоров в итоговый хеш
    # SUPBL-454: среди полей данных из аксессоров убрали поле 'model' в связи с частыми коллизиями при некорректном
    # заполнении этого поля в фиде со стороны клиента
    my @accessor_fields = qw/vendor sex color collection age material/;
    my $reset_str = $self->reset_flag;
    for my $name(@accessor_fields) {
        $h->{"$name"} = $self->$name if !$h->{$name} && $self->$name && $self->$name !~ /$reset_str/i ;
    }

    delete $h->{$_} for ( grep {!$h->{$_}}  keys %$h);

    sub get_agreed_adj {
        my ($self, $adj, $noun) = @_;
        # check if $adj is A
        return 'унисекс' if ($adj =~ /^унисекс/i);

        my $adj_gnc = $self->proj->phrase($adj)->get_gender_number_case(add_posp => 1);
        return $adj unless $adj_gnc and ($adj_gnc->{posp}) and ($adj_gnc->{posp} eq 'A');

        my $gnc;
        # because of lemmer error
        if (lc($noun) eq 'угги' || lc($noun) eq 'дерби' || lc($noun) eq 'сандалии') {
            $gnc = { number => "pl", case => "nom" };
        } else {
            $gnc = $self->proj->phrase($noun)->get_gender_number_case;
        }
        return $self->proj->phrase($adj)->set_gender_number_case($gnc) if defined $gnc;
        return $adj;
    }

    # для полей age, color и подобных кирпичиков делаем составные поля вида '[<variant_1>:<variant_2>:...:<variant_n>]'
    if ($h->{type}) {
        my @mult_choice_fields = qw(age color sex season wear_prop);
        for my $mc_field (@mult_choice_fields) {
            $h->{$mc_field} = &get_agreed_adj($self, $h->{$mc_field}, $h->{type}) if $h->{$mc_field};
            $h->{$mc_field} = "[".$h->{$mc_field}."]" if ($h->{$mc_field} && $h->{$mc_field} =~ /:/);
        }
    }

    # Типы одёжных товаров (как минимум те, что лежат в dict_goods, должны писаться в маленьком регистре)
    if ($h->{type}) {
        my $h_type_phrase = $self->proj->phrase($h->{type});
        my $norm_h_type = $h_type_phrase->norm_phr;
        my $norm_goods = $self->proj->phrase($h_type_phrase->get_goods)->norm_phr;
        if ($norm_h_type eq $norm_goods) {
            $h->{type} = lc $h->{type};
        }
    }

    #print STDERR " =====     Кирпичи:     ========\n";
    #print STDERR Dumper($h);
    #print STDERR "================================\n\n";

    return $h;
}


sub clear_bad_colors {
    my ($self, $text) = @_;
    my @bad_colors_list = qw(яркий матовый);
    $_ =~ s/\w{2}$// for @bad_colors_list; #cut endings

    my $re = '(?:' . join('|', @bad_colors_list) . ')\w{2,3}';
    my @colors;

    if ( @colors = $text =~ /(?:^|\s)($re)(?:\s|$)/ig ){
        $text =~ s/(^|\s+)$_(\s+|$)/ / for (@colors);
        $text =~ s/\s+/ /g;
        $text =~ s/(^\s+|\s+$)//;
    }
    return ($text, join(':', @colors));
}


sub check_attr {
    my ( $self ) = @_;
    if ( $self->color ){
        $self->{color} = $h_latin_colors_map->{lc($self->{color})} if $h_latin_colors_map->{lc($self->{color})};
        $self->{color} = $self->reset_flag if ( $self->color =~ /^(?:мульти(?:.*)|вино)$/i );
    }
    $self->{age} = $self->reset_flag if ( $self->age && $self->age =~ /взросл/i );
#    $self->{sex} = $self->reset_flag if ( $self->sex && $self->sex =~ /унисекс/i );
    if ( $self->material ){
        $self->{material} = $self->reset_flag unless $self->material =~ /[a-zа-яё]/i;
        $self->{material} = $self->reset_flag if $self->material =~ /состав/i;
    }
}

my $MDL = 'model/model:_SHORT_MDL/model:_MDL_FIRSTWORD';
my $MDL_TITLE = 'model:_DEL_BRAND/model:_SHORT_MDL:_DEL_BRAND/model:_MDL_FIRSTWORD:_DEL_BRAND';#то же самое что и MDL, но добавляем модификатор удаления бренда. Необходимо, чтобы в заголовке не повторялся бренд 2 раза SUPBL-47
my $MDLC = 'model:_CHECK_MDL/model:_SHORT_MDL:_CHECK_MDL';
my $MDLC_TITLE = 'model:_CHECK_MDL:_DEL_BRAND/model:_SHORT_MDL:_CHECK_MDL:_DEL_BRAND';#то же самое что и MDLC, но добавляем модификатор удаления бренда SUPBL-47
my $COL = 'collection/collection:_3WORDS/collection:_SEASON/collection:_SECONDW';
my $COL_TITLE = 'collection:_UP_FIRST/collection:_3WORDS:_UP_FIRST/collection:_SEASON:_UP_FIRST';
my $WP = 'wear_prop';
my $END = 'model:_MULTIDOT_END';

#my $TYPE = 'type/type:_DEL_ADJ';
my $TYPE = "type wear_prop:_DEMULT/type/type:_DEL_ADJ";
my $TYPED = 'type:_DOT_END/type:_DEL_ADJ:_DOT_END';
my $TYPEWA = "type wear_prop:_DEMULT wear_addition/type wear_addition/type:_DEL_ADJ wear_addition";
my $TYPEWAD = "type wear_prop:_DEMULT wear_addition:_DOT_END/type wear_addition:_DOT_END/type:_DEL_ADJ wear_addition:_DOT_END";

sub title_templ_perf :STATIC :GLOBALCACHE("DERIVED") {
    my @res = ();
    my $txt = "
        age season [$TYPEWA] brand
        sex season [$TYPEWA] brand
        age [$TYPEWA] brand
        sex [$TYPEWA] brand
        age [$TYPEWA]
        [$TYPEWA] brand
        sex [$TYPEWA]
        [$TYPEWA]
    ";
#sex type shopname,
#        sex type

    # сначала пытаемся сделать заголовок с добавкой, потом по этому же шаблону - без
    my @temp = ();
    for my $line ( split/\n+/,$txt ){
        $line =~ s/(^\s+|\s+$)//g;
        next unless $line;
        $line .= ',';
        push @temp, $line;
    }
    for my $line ( @{[@temp]} ){
        $line =~ s/$TYPEWAD/$TYPED/;
        $line =~ s/$TYPEWA/$TYPE/;
        push @temp, $line;
    }

    # сначала пытаемся сделать заголовок с прилагательными, потом по этому же шаблону - без
    for my $line ( @temp ){
        my $line_with_adj = $line;
        $line_with_adj =~ s/\/type:_DEL_ADJ.*?(\/|\])/$1/;
        $line_with_adj =~ s/\/\//\//g;
        $line_with_adj =~ s/\/\]/\]/g;
        push @res, $line_with_adj;
#        push @res, $line;
    }
    push @res, @temp;
    return join('',@res);
}

sub title_templ_perf_old {
    my @res = ();
    my $txt = "
        sex color [$TYPE] brand [$MDLC],
        season color [$TYPE] brand [$MDLC],
        age color [$TYPE] brand [$MDLC],

        sex color [$TYPE] brand,
        season color [$TYPE] brand,
        age color [$TYPE] brand,
        sex season [$TYPE] brand,
        age season [$TYPE] brand,

        sex color [$TYPED] [$COL_TITLE],
        season color [$TYPED] [$COL_TITLE],
        age color [$TYPED] [$COL_TITLE],
        sex season [$TYPED] [$COL_TITLE],
        age season [$TYPED] [$COL_TITLE],

        sex [$TYPE] brand:_DOT_END [$COL_TITLE],
        color [$TYPE] brand:_DOT_END [$COL_TITLE],
        age [$TYPE] brand:_DOT_END [$COL_TITLE],
        season [$TYPE] brand:_DOT_END [$COL_TITLE],

        sex color [$TYPE],
        season color [$TYPE],
        age color [$TYPE],
        sex season [$TYPE],
        age season [$TYPE],

        sex [$TYPE] brand,
        color [$TYPE] brand,
        age [$TYPE] brand,
        season [$TYPE] brand,

        [$TYPE] brand:_DOT_END [$COL_TITLE],
        sex [$TYPED] [$COL_TITLE],
        color [$TYPED] [$COL_TITLE],
        age [$TYPED] [$COL_TITLE],
        season [$TYPED] [$COL_TITLE],

        [$TYPE] brand,
        sex [$TYPE],
        color [$TYPE],
        age [$TYPE],
        season [$TYPE],

        [$TYPED] [$COL_TITLE],
        ";
    for my $line ( split/\n+/,$txt ){
        $line =~ s/(^\s+|\s+$)//g;
        next unless $line;
        my @wpl = map { $_ eq "[$TYPE]" ? "[$TYPE] [$WP]" : $_ eq "[$TYPE]," ? "[$TYPE] [$WP]," : $_} split( /\s+/, $line );
        my $wp_line = join (' ', @wpl);
        push @res, $wp_line;
        push @res, $line;
    }
    return join('',@res);
}


#        color brand => color [$T0YPE] brand, color brand, [$TYPE] brand, color [$TYPE], [$TYPE], type:_MULTIDOT_END
#        age brand => age [$TYPE] brand, age brand, [$TYPE] brand, age [$TYPE], [$TYPE], type:_MULTIDOT_END
#        wear_addition brand => [$TYPE] brand wear_addition, [$TYPE] brand, [$TYPEWA], [$TYPE], type:_MULTIDOT_END
#        wear_addition brand sex => sex [$TYPE] brand wear_addition, [$TYPE] brand wear_addition, [$TYPE] brand, [$TYPEWA], [$TYPE], type:_MULTIDOT_END


#   [$TYPE] sex => sex [$TYPE], [$TYPE], type:_MULTIDOT_END
sub dyn_templates_text :STATIC :GLOBALCACHE("DERIVED") {
    my $class = shift;
    my $res = "
        [$TYPE] color [$MDL] => color [$TYPE] brand [$MDL_TITLE], [$TYPE] brand [$MDL_TITLE], color [$TYPE], color brand [$MDL_TITLE], brand [$MDL_TITLE], $END
        [$TYPE] color => color [$TYPE] brand [$MDL_TITLE], [$TYPE] brand [$MDL_TITLE], color [$TYPE], color brand [$MDL_TITLE], brand [$MDL_TITLE], [$TYPE] shopname, $END
        [$TYPE] material => sex [$TYPE] brand [$MDL_TITLE], color [$TYPE] brand [$MDL_TITLE], [$TYPE] brand [$MDL_TITLE], color [$TYPE], color brand [$MDL_TITLE], brand [$MDL_TITLE], [$TYPE] shopname, $END
        [$TYPE] sex [$MDL] => sex [$TYPE] brand [$MDL_TITLE], [$TYPE] brand [$MDL_TITLE], sex [$TYPE], sex brand [$MDL_TITLE], brand [$MDL_TITLE], $END
        [$TYPE] brand [$MDL] => type brand [$MDL_TITLE], brand [$MDL_TITLE], $END
        [$TYPE] age brand [$MDL] => age type brand [$MDL_TITLE], age brand [$MDL_TITLE], brand [$MDL_TITLE], $END
        [$TYPE] brand => [$TYPE] brand, [$TYPE], type:_MULTIDOT_END
        [$TYPE] color brand => color [$TYPE] brand, [$TYPE] brand, color [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] sex brand => sex [$TYPE] brand, [$TYPE] brand, sex [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] sex brand year => sex [$TYPE] brand year, sex [$TYPE] brand, [$TYPE] brand, sex [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] season brand => season [$TYPE] brand, [$TYPE] brand, season [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] color sex brand => color sex [$TYPE] brand, sex [$TYPE] brand, color [$TYPE] brand, [$TYPE] brand, sex [$TYPE], color [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] sex season brand => season sex [$TYPE] brand, sex [$TYPE] brand, season [$TYPE] brand, [$TYPE] brand, sex [$TYPE], season [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] color sex => color sex [$TYPE], sex [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] color season => color season [$TYPE], season [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] color age => age color [$TYPE], age [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] sex season => season sex [$TYPE], sex [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] wear_addition => [$TYPEWA], [$TYPE], type:_MULTIDOT_END
        [$TYPE] sex wear_addition => sex [$TYPEWA], sex [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] sex wear_addition brand => sex [$TYPE] brand wear_addition, sex [$TYPE] brand, sex [$TYPEWA], sex [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] color brand [$MDL] => color type brand [$MDL_TITLE], color brand [$MDL_TITLE], brand [$MDL_TITLE], $END
        [$TYPE] season wear_addition => season [$TYPE] brand wear_addition, season [$TYPEWA], season [$TYPE], [$TYPE] wear_addition, [$TYPE], type:_MULTIDOT_END
        [$TYPE] age => age [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] brand year => [$TYPE] brand year, [$TYPE] brand, [$TYPE], type:_MULTIDOT_END
#       [$TYPE] [shopname/sex] {___MAX_20000} => [$TYPE] shopname, [$TYPE] brand
        [$TYPE] {___MAX_1000} => type brand [$MDL_TITLE], color [$TYPE] brand [$MDL_TITLE], color [$TYPE] brand, [$TYPE] brand], [$TYPE] shopname, type:_MULTIDOT_END
        [$TYPE] [shopname/sex] {___MAX_20000} => sex [$TYPE] brand, [$TYPE] brand, sex [$TYPE], [$TYPE], type:_MULTIDOT_END
        [$TYPE] brand season year => [$TYPE] brand season year, [$TYPE] brand season, [$TYPE] brand, [$TYPE], type:_MULTIDOT_END
        [$TYPE] brand wear_addition => [$TYPE] brand wear_addition, [$TYPE] brand, [$TYPE], type:_MULTIDOT_END
        sex [$MDLC] => sex [$TYPE] brand [$MDLC_TITLE], sex type [$MDLC_TITLE], sex brand [$MDLC_TITLE], sex [$MDLC_TITLE], [$MDLC_TITLE], $END
        season [$MDLC] => season [$TYPE] brand [$MDLC_TITLE], season type [$MDLC_TITLE], season brand [$MDLC_TITLE], season [$MDLC_TITLE], [$MDLC_TITLE], $END
        age [$MDLC] => age [$TYPE] brand [$MDLC_TITLE], age type [$MDLC_TITLE], age brand [$MDLC_TITLE], age [$MDLC_TITLE], [$MDLC_TITLE], $END
        shopname [$MDLC] => sex [$TYPE] brand [$MDLC_TITLE], sex type [$MDLC_TITLE], sex brand [$MDLC_TITLE], sex [$MDLC_TITLE], [$MDLC_TITLE], $END
        color brand [$MDL] => color brand [$MDL_TITLE], brand [$MDL_TITLE], [$MDL_TITLE], $END
        age brand season => age [$TYPE] brand season, age [$TYPE] brand, age season brand, [$TYPE] brand, age [$TYPE], [$TYPE], type:_MULTIDOT_END
        age brand year => age [$TYPE] brand year, age [$TYPE] brand, age season brand, [$TYPE] brand, age [$TYPE], [$TYPE], type:_MULTIDOT_END
        shopname brand => [$TYPE] brand, [$TYPE], type:_MULTIDOT_END
        shopname wear_addition type => [$TYPE] brand wear_addition, [$TYPE] brand, [$TYPEWA], [$TYPE], type:_MULTIDOT_END
    ";
    return $class->add_modifiers_to_templates( $res );
}

sub add_modifiers_to_templates :STATIC {
    my ($class, $text) = @_;
    my @resarr = ();
    for my $line ( map { s/(^\s+|\s+$)//g; $_ || () } split( /\n/, $text) ){ ##no critic
        my ($phrase, $title) = split /\s*=>\s*/, $line;
        # к цетам добавляем мультипликатор для упрощения оттенков
        if ($phrase =~ /(^|\s+)color(\s+|$)/) {
            $phrase .= "{___MULT_PHRASE} " if $phrase !~ /{___MULT_PHRASE}/;
        }
        # в заголовках мультипликатор удаляем
        if ($title =~ /(^|\s+)color(\s+|$)/) {
            $title =~ s/(^|\s+)color(\s+|$)/$1color:_DEMULT$2/g;
        }
        $line = join(' => ', $phrase, $title);
        push @resarr, $line;
    }
    return join ("\n", @resarr );
}

sub perf_templates_text :STATIC :GLOBALCACHE("DERIVED") {
    my $class = shift;
    my @res = ();
    my $title_templ_perf = $class->title_templ_perf;
    for my $line (split /\n/, $class->dyn_templates_text) {
        my ( $phrt, $tt ) = split /\s*=>\s*/, $line;
        push @res, "$phrt => $title_templ_perf";

    }
    return join("\n", @res);
}

sub banner_single_templates_text :STATIC :GLOBALCACHE("DERIVED") {
    my $class = shift;
    my @res = ();
    my $title = $class->title_templ_perf;
    my $result = "
        [$TYPE] => $title
        brand => $title
        [$MDL] => $title
    ";
    return $result;
}

# Модификатор для удаления брэнда из текста. Делали для удаления бренда из модели SUPBL-47
sub _DEL_BRAND {
    my ( $self, $txt ) = @_;
    return $txt unless ($self->vendor);
    my $brand = quotemeta($self->vendor);
    $txt =~ s/^$brand\s+//;
    return $txt;
}

sub _CHECK_MDL {
    my ( $self, $txt ) = @_;
    return '' unless ( $txt =~ / / && length($txt)>4 );
    return $txt;
}

sub _SHORT_MDL {
    my ( $self, $txt ) = @_;
    my @words = grep{$_} split /\s+/, $txt;
    return join(" ", @words[0..min(1, $#words)]);
}

sub _MDL_FIRSTWORD {
    my ( $self, $txt ) = @_;
    my @words = grep{$_} split /\s+/, $txt;
    return '' unless @words;
    return $words[0] if length($words[0])>4;
    return '';
}

sub _SEASON {
    my ( $self, $txt ) = @_;
    if ( $txt =~ /(?:^|\s)([а-яё]+[\/-][а-яё]+)(?:\s|$)/i ){
        return $1 unless (lc($1) eq lc($txt));
    }
    return '';
}

sub _SECONDW {
    my ( $self, $txt ) = @_;
    my $season = $self->_SEASON($txt);
    return '' unless $season;
    return (split/ /, $season)[1];
}

sub _3WORDS {
    my ( $self, $txt ) = @_;
    my $season = $self->_SEASON($txt);
    return '' unless $season;
    $season =~ s/[\/-]/ /g;
    return '' unless $season =~ / {2,}/;
    return $self->proj->phrase($season)->head(3);
}

sub _ADD_COMMA {
    my ( $self, $txt ) = @_;
    return "$txt," if ($txt);
    return "";
}

sub _denorm_type {
    my ( $self, $txt ) = @_;
    my $res = $txt;
    for my $source_text ( grep { $_ } ($self->name, $self->model, $self->typePrefix, $self->proj->phrase($self->description)->head(10)) ){
        $res = $self->proj->phrase($source_text)->_denorm_text($txt);
#        print STDERR Dumper ([$source_text, $txt, $res]);
        $res = $self->proj->phrase($source_text)->restore_capitalization( $res );
        if ( $res ){
            my $gnc = $self->proj->phrase( $res )->get_gender_number_case;
            $gnc->{case} = 'nom';
            $res = $self->proj->phrase( $res )->set_gender_number_case( $gnc );
            last;
        }

#        print Dumper ( [$res, $gnc] );
#        last if $res;
    }
#    print STDERR Dumper ([$self->name, $self->model, $self->typePrefix, $source_text,$txt,$res]);
    return $res;
}

sub dyn_methods_arr :STATIC :GLOBALCACHE("DERIVED") {
    my ($self) = @_;
    my @res = map { s/^\s+//; $_ } grep {/\S/} grep {!/#/} ##no critic
        split /\n/, '
        add_dynamic_homonymy_words              goods accessory P
        modellike_product                       goods accessory P
        get_wide_filtered                       goods accessory
        add_trade_phrases_dynamic_wear          goods accessory S
        get_search_filtered50k                  goods accessory
        pack_phr                                goods accessory
        postfilter_base                         goods accessory
        spec_pack_list                          goods accessory
        get_search_filtered                     goods accessory
        set_exclamations_before_stops           goods accessory
        set_exclamations_before_bsstops         goods accessory
        replace_exclamations_with_pluses        goods accessory
    ';
    return @res;
}

sub perf_banners_single {
    my ($self, %params) = @_;

    my ($arr) = $self->banners_data(
        templates_text => $self->banner_single_templates_text,
        methods_arr => [ 'pack_phr goods accessory', 'pack_list goods accessory', ],
        max_count => 1,
        title_template_type => 'single',
        assert_no_rpc => 1,
        %params,  # ctx
    );

    unless ( @$arr ){
        $arr = $self->banners_data(
            methods_arr => [ 'pack_phr goods accessory', 'pack_list goods accessory', ],
            max_count => 1,
            templates_text => "[$TYPE] => [$TYPE]",
            assert_no_rpc => 1,
            title_template_type => 'single',
            %params,
        );
    }

    $arr = $self->single_banner_default('perf', %params) unless @$arr;

    return @$arr ? [ $arr->[0] ] : [];
}

our $dict_colors_full = undef;
our $dict_colors_full_norm = undef;
sub load_dict_colors_full {
    return 1 if $dict_colors_full;
    my ($self) = @_;

    $dict_colors_full = {};
    $dict_colors_full_norm = {};
    open F, "<".$self->proj->{options}{dicts}."/dict_colors_full";
    while(my $line = <F>) {
        chomp $line;
        my %h = map { split('=', $_, 2) } split("\t", $line);
        $dict_colors_full->{ lc $h{name} } = \%h;
        $dict_colors_full->{ $h{name_norm} } = \%h;
    }
    close F;

    return 1;
}

sub _parse_clothes_offer {
    my ($h_offer) = @_;
    my %h_res = ();
    my %params = map { my ($k, $v) = split(':', $_, 2); (lc $k) => $v } map { s/^\s+|\s+$//g; $_ } ##no critic
                 split(/(?:^|,)(?=[ а-яА-Яa-zA-ZёЁ]+?:)/, $h_offer->{params}) if $h_offer->{params};
    # единицы измерения размера (быват ещё длины или другого параметра)
    my $valid_size_unit = _validate_clothes_size_unit($h_offer->{'size:unit'});
    $h_res{unit} = $valid_size_unit || '';
    # материал
    $h_res{consist} = $params{material} || $params{'материал'} || '';
    # размеры
    $h_res{sizes} = [];
    if (($h_offer->{'size:unit'} && $valid_size_unit) || !$h_offer->{'size:unit'}) {
        for (qw/размер size/) {
            my $size = _validate_clothes_size($params{$_});
            push @{ $h_res{sizes} }, $size if $size;
        }
    }
    # цвета
    $h_res{colors} = [];
    for (qw/цвет color colour/) {
        push @{ $h_res{colors} }, $params{$_} if $params{$_};
    }
    return \%h_res;
}

sub get_size_units_mapping :CACHE {
    return +{ map { lc($_) => $_ } @$SIZE_UNITS };
}

sub _validate_clothes_size_unit {
    my $size_unit = shift;
    my $size_units_mapping = get_size_units_mapping();
    return $size_units_mapping->{lc($size_unit)};
}

sub _validate_clothes_size {
    my $size_value = shift;
    return unless $size_value;
    my ($first_value, undef) = split(/,\s+/, $size_value, 2);
    if ($first_value && $first_value =~ m/\w+/ && $first_value =~ m/^[\w\-\/\s\.,\(\)*;]+$/) {
        return $first_value;
    }
    return;
}

sub _format_sizes {
    my ($sizes, $unit)  = @_;
    return undef unless ($sizes and (ref $sizes eq 'ARRAY') and (@$sizes));
    $sizes = [map {split /\ *[.,\/\\;]+\ +/} uniq_array (@$sizes)];
    if ( scalar(@$sizes) == scalar(grep { /^\d+([.,\/]{1}\d+)?$/ } @$sizes) ) {
        # считаем, что размеры числовые (примеры: 48; 37,5; 37,5; 30/100)
        my $sorting_func = sub {
            my ($aa, $bb) = ($a, $b);
            $_ =~ s/[^\d.]/./g for ($aa, $bb); # чтобы убрать ворнинг из-за числового сравнения не-чисел
            return $aa <=> $bb;
        };
        $sizes = [sort $sorting_func @$sizes];
    } elsif ( scalar(@$sizes) == scalar(grep { /^(x*(?:S|L)|M)$/i } @$sizes) ) {
        # интернациональные размеры XS, S, M, L, XL...
        $sizes = [ map { uc($_) } @$sizes ];
        my $i = 0;
        my %int_sizes_order = map { $_ => $i++ } qw/XXXS XXS XS S M L XL XXL XXXL/;
        $sizes = [sort { $int_sizes_order{$a} <=> $int_sizes_order{$b} } @$sizes];
    } else {
        # сортируем по алфавиту
        $sizes = [sort {$a cmp $b} @$sizes];
    }
    return ($unit ? $unit.": " : '').join(';', @$sizes);
}

sub _format_additional_data {
    my ($self, $h_data) = @_;
    my $unit = $h_data->{unit} || '';
    if ($unit) {
        $unit =~ s/^\s+|\s+$//g;
    }
    if ( $h_data->{sizes} ) {
        $h_data->{sizes} = _format_sizes($h_data->{sizes}, $unit);
        $h_data->{sizes_for_direct} = $h_data->{sizes};
    }; # DYNSMART-517
    if ( $h_data->{consist} ) {
        $h_data->{consist} = join(';',
            split /\ *[.,\/\\;]+\ */,
            $h_data->{consist}
        );
        $h_data->{consist_for_direct} = $h_data->{consist};
    };
    if ( $h_data->{colors} ) {
        $h_data->{colors} = join(';',
            uniq_array (
                grep { $_ }
                #отображение цвета в HEX
                map { $dict_colors_full->{$_}->{hex} || $dict_colors_full_norm->{$self->proj->phrase($_)->norm_phr}->{hex} || '' }
                map {split /\ *[.,\/\\;]+\ */}
                uniq_array(@{ $h_data->{colors} })
            )
        );
    };
    return $h_data;
};

1;
