package BM::BannersMaker::Tasks::BasePerfTask;

use utf8;
use open ':utf8';

use std;
use base qw(BM::BannersMaker::Tasks::Task);

use Storable qw(dclone);

use Utils::Sys qw(
    do_safely
    md5int
    merge_hashes
);
use Utils::Urls;
use Utils::Array qw/uniq_array/;
use Utils::Funcs qw/is_offer_id_valid is_price_valid/;
use Time::HiRes qw/time/;
use Utils::CompileTime;
use BM::BannersMaker::Tasks::TaskQueue;
use Utils::Regions qw/get_countries/;

sub banners_method_name {
    return 'perf_banners';
}


sub get_pt_checksum {
    my $self = shift;
    my $pt = shift;
    return (md5int($pt->{id} // $pt->{OfferID} // $pt->{OfferId}) & 0xffffffff) >> 1;
}

# BannerHash соответствующий БК-шному:
# https://st.yandex-team.ru/DYNSMART-912
# https://a.yandex-team.ru/arc/trunk/arcadia/yabs/server/cs/libs/utils/banner_land/misc.cpp?rev=4341578#L22-27
sub get_BannerHash {
    my $self = shift;
    my $bp = shift;  # banner-phrase
    my $Info = $self->proj->json_obj->decode($bp->{Info});

    my $bnr_str = join('_',
        $bp->{TitleMediaSmart}, $bp->{OfferID}, $bp->{CanonizedUrl}, $Info->{image}, $self->taskinf->{OrderID}, $self->GroupExportID,
    );
    return md5int($bnr_str);
}

# BannerID, соответствующий БК-шному:
# https://st.yandex-team.ru/DYNSMART-912
# https://a.yandex-team.ru/arc/trunk/arcadia/yabs/server/cs/libs/utils/banner_land/misc.cpp?rev=4341578#L36
sub get_BannerID {
    my $self = shift;
    my $bp = shift;
    my $BannerHash = $self->get_BannerHash($bp);
    my $high_byte_in_banner_id = 2;  # было хост-опцией
    my $BannerID = ($BannerHash & ((1 << 56) - 1)) + $high_byte_in_banner_id * (1 << 56);
    return $BannerID;
}


#Домен таска
sub domain :CACHE {
    my ($self) = @_;
    return $self->feed_or_datacamp_domain;
}

# файл, в котором хранится хэш с информацией о генерации, поля:
#   task_md5    =>  md5 от таски
#   feed_md5    =>  md5 от фида (mpd-файла)
#   timestamp   =>  subj
sub info_file_gen {
    my $self = shift;
    return $self->dir.'/info-gen.json';
}

my %ad_type2min_native_length = (
    'cars' => 9,
);

# Выбираем заголовки, приоритет: 1) use_as_name 2) нативная генерация 3) dse
# параметры:
#   $pt -  product
#   $titles -  хэш {type=> {
#               title =>
#               title_source =>
#               title_template =>
#               title_template_type =>
#           }
#    }
# доп. парметры (в хэше):
#   is_vip =>  флаг вип генерации
#       для vip-генерации нужно брать нативные тайтлы т.к. они по кастомным шаблонам
#       https://st.yandex-team.ru/DYNSMART-812
sub choose_title {
    my ($self, $pt, $src_titles, %par) = @_;
    return {} unless %$src_titles;
    #берем только совместимые с движком
    my $titles = { map {$_ => $src_titles->{$_}} grep {
        $self->check_banned_title_source($src_titles->{$_}->{title_source}, $pt->{main_mirror}) &&
        $self->proj->is_bs_compatible($src_titles->{$_}->{title})
    } keys %$src_titles };

    my @title_types = qw(native_base native_single native_fallback);

    for my $title_type ( @title_types ) {
        if ($titles->{$title_type} and $titles->{$title_type}{title_template} eq 'use_as_name') {
            return $titles->{$title_type};
        }
    }

    my $native_title = {};
    for my $title_type ( @title_types ) {
        $native_title = $titles->{$title_type} if !%$native_title and $titles->{$title_type} and $titles->{$title_type}->{title};
    }
    if ($native_title and $titles->{dse_base} and $titles->{dse_base}->{title}) {
        return $native_title if $par{is_vip};
        my $dse_tools = $self->proj->dse_tools;
        my $ad_type = $pt->ad_type;
        my $min_native_length = $ad_type2min_native_length{$ad_type} // 16;

        if ((length($native_title->{title}) < $min_native_length) and
            ($dse_tools->get_length($native_title->{title}, ignore_narrow => 1) < $dse_tools->get_length($titles->{dse_base}->{title}, ignore_narrow => 1))
        ) {
            return $titles->{dse_base};
        } else {
            return $native_title;
        }
    }
    return $native_title if ($native_title and $native_title->{title});
    return {} if $par{is_vip};
    return $titles->{dse_base} if ($titles->{dse_base} and $titles->{dse_base}->{title});
    return {};
}

sub get_market_data {
    my ($self, $id) = @_;
    return {} unless $id;
    my $h = $self->proj->market_subphraser->get_market_data($id);

    # нужны названия, соотв. таблице MarketIDCtgVendorModel
    my %mapping = (
        model_id    => 'ID',
        category    => 'Category',
        vendor      => 'Vendor',
        model       => 'Model',
        price_min   => 'PriceMin',
        price_avg   => 'PriceAvg',
        price_max   => 'PriceMax',
        offer_count => 'OfferCount',
        rating      => 'Rating',
        rating_count => 'RatingCount',
        is_new      => 'IsNew',
    );
    return +{
        map { $mapping{$_} => $h->{$_} }
        grep { defined $mapping{$_} }
        keys %$h
    };
}

sub get_feed {
    my ($self) = @_;
    my $proj = $self->proj;
    my $feedparams = $self->get_feedparams;
    return $proj->feed($feedparams);
}

sub get_feed_domain :CACHE {
    my ($self) = @_;
    my $proj = $self->proj;
    my $fd = $self->get_feed; #Так как фид не кэшируется, то это другой объект
    $fd->iter_init; #Сбрасываем цикл обхода, если фид уже обходился
    my $ptl = $fd->next_ptl_pack(2);
    my $domain = '';
    if($ptl){
        my @offs = @$ptl;
        my $s = $proj->site($offs[0]->url);
        $domain = $s->domain,
    }
    return $domain;
}

sub get_geo_countries :CACHE {
    my ($self) = @_;
    my $task = $self->taskinf;
    my $geo = $task->{Resource}{Geo};
    return $geo ? [get_countries($geo)] : [];
}

sub worker {
    my ($self) = @_;
    do_safely(
        sub { $self->worker_safe(); },
        timeout => 48 * 3600,
        die_if_timeout => 1,
    );
}

sub get_sim_distance_by_letter {
    my ($self, $letter, $text) = @_;
    return $self->proj->options->{PerfSources}->{sim_distance}->{$letter};
}

# Определяем тип генерации (нативная / dse ) по business_type, DYNSMART-633
# окончательное решение по включению натики будет принимается на основе product
sub get_generation_flags_for_business_type {
    my ($business_type) = @_;
    my ($ignore_native, $ignore_external) = (0, 0);
    $ignore_native = 1 if ($business_type and ($business_type eq 'other'));
    return ($ignore_native, $ignore_external);
}

sub worker_safe {
    my ($self) = @_;

    $self->check_deprecated_options;

    $self->worker_prepare  or do {
        $self->log("can't prepare task");
        return;
    };

    my $feedurl = $self->feedurl;
    $self->stagelog("=== begin generate $feedurl ===");
    my $result = $self->worker_generate;
    $self->stagelog("=== end generate $feedurl ===");

    return $self->worker_finalize($result);
}

sub worker_prepare {
    my $self = shift;

    my $proj = $self->proj;

    $proj->log("perf worker_safe");
    $proj->log(Utils::CompileTime::status_str());

    my $prevbeg = $self->get_param('begin');
    my $curtime = time;

    if($prevbeg && ($curtime - $proj->dates->trdate('db_time', 'sec', $prevbeg ) < 3600 * 4) ){
        #Не перегенерим, если прошлый обход был недавно
        $proj->log("Regen timeout: $prevbeg => $curtime - ".$proj->dates->trdate('db_time', 'sec', $prevbeg )." < ".(3600 * 4));
    }

    my $task = $self->taskinf;

    #Персональный костыль для проблемного клиента
    if($self->taskinf->{Resource}{FeedUrl} && ($self->taskinf->{Resource}{FeedUrl} =~ /yoox\.com/)){
        if($prevbeg && ($curtime - $proj->dates->trdate('db_time', 'sec', $prevbeg ) < 3600 * 24) ){
            #Не перегенерим, если прошлый обход был недавно
            $proj->log("Regen timeout: $prevbeg => $curtime - ".$proj->dates->trdate('db_time', 'sec', $prevbeg )." < ".(3600 * 4));
            return;
        }
    }

    $proj->log("update db inf BEG");

    $self->{_begin_time} = time;

    $self->set_params({
        cronlight_id  => $proj->{cronlight_id} || '',
        begin         => $proj->dates->trdate('sec', 'db_time', $self->{_begin_time}),
        end           => '',
        crontime      => $proj->{cronlight_crontime} || '',
        prev_begin    => $self->get_param('begin'),
        prev_end      => $self->get_param('end'),
        prev_crontime => $self->get_param('crontime'),
        FeedUrl       => $self->taskinf->{Resource}{FeedUrl},
        Geo           => (ref($self->taskinf->{Resource}{Geo}) eq 'ARRAY' ? join(' ',@{$self->taskinf->{Resource}{Geo}}) : $self->taskinf->{Resource}{Geo}),
        Body          => $self->taskinf->{Resource}{Body},
        host          => $proj->host_info->{host},
        ParentExportIDs => join(' ', sort @{$task->{ParentExportIDs} || []}),
        BannerIDs => join(' ', sort @{$task->{BannerIDs} || []}),
        GroupExportIDs => join(' ', sort @{$task->{GroupExportIDs} || []}),
        GenerationID  => $self->taskinf->{GenerationID} || '',
        BSTaskID      => $self->taskinf->{BSTaskID} || '',
    });

    $proj->log("update db inf END");

    $self->update_monitor_state("Start");

    my $feedurl = $self->feedurl;

    $self->globallog("BEGIN OrderID=".$task->{OrderID}." feedurl=$feedurl");

    #Если нет урла - выходим
    unless($feedurl){
        print STDERR "Bad url name: '$feedurl'\n";
        my $funnel = "task_deleted_no_feedurl";
        $self->set_params({ FunnelInfo => $funnel });
        $self->export_offers_from_empty_feed(funnel => {$funnel => 1});
        return;
    }

    unless ($self->get_param('domain')) {
        my $domain = $self->feed_or_datacamp_domain;
        $self->set_params({
            domain => $proj->site($domain)->main_mirror,
        });
    }

    # не обрабатываем таск если он в списке заблокированных
    if ($self->is_blocked_task) {
        $self->stagelog("=== task_id was blocked, do nothing ===");
        $self->set_params({ FunnelInfo => "task_blocked", });
        return;
    }

    return 1;
}

sub worker_generate {
    my $self = shift;
    my %opts = @_;

    $self->check_deprecated_options;

    my $proj = $self->proj;
    my $task = $self->taskinf;

    my $filefilterinf = $self->fileformat("filterinf");
    my $filetaskjson  = $self->fileformat("taskjson");

    $self->load_custom_phrases_settings;

    my $feedparams = $self->get_feedparams;
    $proj->log('proj->feed params dumper:');
    $proj->log_dump($feedparams);

    $proj->log("filterinf beg");
    open(my $fhinf, "> ${filefilterinf}_tmp") or die("File '${filefilterinf}_tmp' error: $@");
    print $fhinf $proj->dump_lite($feedparams);
    print $fhinf "Task dump:\n";
    print $fhinf $proj->dump_lite($task);
    close($fhinf);
    $proj->log("filterinf end");
    $self->_tmp2file($filefilterinf);

    #Сохряняем json таски
    my $task_json = $self->{source_taskjson};
    open(my $fhtj, "> $filetaskjson"."_tmp");
    print $fhtj "$task_json\n";
    close($fhtj);
    $self->_tmp2file($filetaskjson);

    # скачиваем фид, если это необходимо
    my $prepare_feeddata_timings = {'start_time' => $proj->dates->cur_date('timings')};
    $self->prepare_feeddata;

    $prepare_feeddata_timings->{end_time} = $proj->dates->cur_date('timings');
    $prepare_feeddata_timings->{duration} = $proj->dates->delta_time($prepare_feeddata_timings->{start_time}, $prepare_feeddata_timings->{end_time}, 'timings', 'seconds');
    $self->{_timings}->{prepare_feeddata} = $prepare_feeddata_timings;

    my $last_feed_info = $self->load_feed_info;
    my $filetskv_mpd_feeddata = $last_feed_info->{tskv_mpd};
    unless ( $filetskv_mpd_feeddata && (-e $filetskv_mpd_feeddata)){
        my $error = 'no_tskv_mpd';
        $proj->log("ERROR: $error");
        return +{
            error => $error,
        };
    }

    # создаем новый фид по tskv
    $proj->log("creating feed by file $filetskv_mpd_feeddata");
    unless ($last_feed_info->{feed_data_type}){
        my $error = 'no_feed_data_type';
        $proj->log("ERROR: $error");
        return +{
            error => $error,
        };
    }

    my $fd = $self->get_feed_by_tskv_mpd( $filetskv_mpd_feeddata );

    $task->{Resource}{UseAsName} =~ s/(^\s+|\s+$)//g if ($task->{Resource}{UseAsName});
    $task->{Resource}{UseAsBody} =~ s/(^\s+|\s+$)//g if ($task->{Resource}{UseAsBody});

    $fd->{use_as_name} = $self->fill_use_as_field($fd, $task->{Resource}{UseAsName}) if ($task->{Resource}{UseAsName});
    $fd->{use_as_body} = $self->fill_use_as_field($fd, $task->{Resource}{UseAsBody}) if ($task->{Resource}{UseAsBody});

    $fd->fds->{business_type} = $self->get_business_type if $self->get_business_type;

    # DEPRECATED: TODO: удалить, когда начнёт работать новый код (#1)
    unless ($self->use_product_page_visit) {
        while (my ($phid, $hf) = each %{$self->filters_params}) {
            if ($hf->{target_funnel} eq 'product_page_visit') {
                $hf->{target_funnel} = 'same_products';
                $hf->{_was_product_page_visit} = 1;
            }
        }
    }

    # готовим export_offers
    my %eo_par = (
        check_required_fields => 1,
        %{$opts{export_offers_par} // {}},
    );
    return $self->do_export_offers(feeds => [$fd], %eo_par) // {};
}

sub worker_finalize {
    my $self = shift;
    my $result = shift;

    $self->stagelog("=== end task ===");
    my $resparams;
    if ($result->{export_info}) {
        # обновили export_offers, пишем в железную воронку данные
        $resparams = $self->get_export_offers_resparams($result->{export_info});
    } elsif ($result->{error}) {
        # при ошибке удаляем данные, но только если прошло достаточно времени
        my $start = 1558904400;  # 2019-05-27; may remove later

        my $max_wait_time = 3600 * 24 * 21;  # 3 weeks
        my $last_export_time = $self->get_export_offers_info->{timestamp} // $start;
        my $error = $result->{error};
        my $funnel;
        if (time() - $last_export_time > $max_wait_time) {
            $funnel = "task_deleted_$error";
            $self->export_offers_from_empty_feed(funnel => {$funnel => 1});
        } else {
            $funnel = "task_skipped_$error";
        }
        $resparams = { FunnelInfo => $funnel };
    } else {
        # всё штатно, просто не переэкспортировали
    }
    $self->set_params($resparams) if $resparams;
}

# по offer-у получить список фраз
# на входе:
#   $pt - offer
#   ctx => контекст генерации, изменяемый in-place хэш с полями
#       timer - объект-таймер; offer_errors, funnel_for_source -  счётчики
#   feed => хэш с данными по фиду, не объект
#   ext_data => список дополнительных офферов (dse) для process_offer_generate_external
#   ...
#
# на выходе - хэш с полями:
#   phrases => список хэшей с инфо о фразах
#   spec_count => дополнительные счётчики
#   badflags_str => информация о флагах, требующих предупреждений
#   error => выставляется при ошибке
#       FunnelInfo => что пишется в funnel_info при ошибке
#
# DEPRECATED  этот метод останется только для отладки на железе, на yt работают под-методы process_offer_*
#
sub process_offer {
    my $self = shift;
    my $pt   = shift;
    my %par  = @_;
    my $ctx  = $par{ctx};

    my $init = $self->process_offer_init($pt, %par);
    return $init if !$init or !$init->{gen_params};  # some errors

    $par{gen_params} = $init->{gen_params};
    my ($native_arr, undef) = $self->process_offer_generate_native($pt, %par);

    my ($ext_arr, undef) = $self->process_offer_generate_external($pt, %par);
    my $arr = $self->process_offer_combine($native_arr, $ext_arr);

    my $titles = {};
    my $long_titles = {};
    for my $h (@$arr) {
        next if !$h->{title_source};
        $self->proj->log("Warning: No title_template_type for title_source - $h->{title_source}") if !$h->{title_template_type};
        my $title_type = $h->{title_source}."_".$h->{title_template_type};

        if (!exists $titles->{$title_type} or !$titles->{$title_type}->{title}) {
            $titles->{$title_type} = {
                title               => $h->{title},
                title_source        => $h->{title_source},
                title_template      => $h->{title_template},
                title_template_type => $h->{title_template_type},
            };
        };
        if (!exists $long_titles->{$title_type} or !$long_titles->{$title_type}->{title}) {
            $long_titles->{$title_type} = {
                title               => $h->{long_title},
                title_source        => $h->{title_source},
                title_template      => $h->{title_template},
                title_template_type => $h->{title_template_type},
            };
        }
    }
    $par{titles} = $titles;
    $par{long_titles} = $long_titles;

    if (!@$arr) { # Не удалось выделить фраз
        $ctx->{offer_errors}{err}{nophrases}++;
        return;
    }
    $self->get_funnel_for_pt($pt, $ctx)->{"offers_phrases_generated"}++;

    $par{phrases} = $arr;
    $par{minuswords} = $self->process_offer_generate_minuswords($pt, %par);
    my $result = $self->process_offer_finalize($pt, %par);

    return $result;
}



# Простая пред-обработка оффера, получение параметров и т.п.
sub process_offer_init {

    my $self = shift;
    my $pt   = shift;
    my %par  = @_;
    my $ctx  = $par{ctx};

    my $proj = $self->proj;
    my $task = $self->taskinf;
    my $filters_params = $self->filters_params;

    my ($oe, $timer) = ($ctx->{offer_errors}, $ctx->{timer});
    my $fcount = $self->get_funnel_for_pt($pt, $ctx);

    $pt->nullify_fields_with_cdata();

    # Проверки, что можно обрабатывать оффер
    $timer->time('init:check');

    #Специальный временный фильтр для фидов недвижимости, так как с ней какие-то проблемы ( https://st.yandex-team.ru/IRT-990 )
    if(($task->{OrderID} == 11955143) || ($task->{OrderID} == 11954595)){
        my $buildingname = $pt->{'FL_FEED_apartment_cmplx'} // $pt->{'AC_FEED_apartment_cmplx'};
        my $flt = {};
        $flt = { map {$_=>1} ('Видный город') } if $task->{OrderID} == 11955143;
        $flt = { map {$_=>1} ('Опалиха О3') } if $task->{OrderID} == 11954595;
        unless( $flt->{$buildingname} ){
            $oe->{err}{buildingname}++;
            return;
        }
    }

    my $retarg = 0; # флаг генерации по офферному ретаргтингу: генерим одну баннерофразу, в ней фраза - offerid_<offerid>
    my $gen_phrases = 0; # флаг генерации человекочитаемых фраз
    my $phraseless = 0; # флаг того, что мы затираем фразы

    #Получаем обобщённые параметры фильтров
    for my $phid ( grep { defined($_) } split ',', $pt->{offerfilters} ){
        my $hf = $filters_params->{$phid};

        #Собираем типы генерации, которые есть у всех фильтров, которые сработали для данного товара
        if ($hf->{goal_context_id}) {
            $retarg = 1;
        }
        if ($hf->{target_funnel}) {
            my $target_funnel = $self->get_effective_target_funnel($hf->{target_funnel});
            if ($target_funnel eq 'product_page_visit') {
                $retarg = 1;
            }
            if ($target_funnel eq 'new_auditory') {
                $gen_phrases = 1;
            }
            if ($target_funnel eq 'same_products') {
                $retarg = 1;
                unless ($hf->{goal_context_id}) {
                    $gen_phrases = 1;
                }
            }
        }
    }

    $pt->{retarg} = $retarg; #Прокидываем параметр, так как это влияет на генерацию
    $oe->{retarg}++ if $retarg;

    $pt->{_offid_} = $pt->{'id'} // $pt->{'OfferID'} // $pt->{'OfferId'};

    if ($pt->{offer_source} && $pt->{offer_source} eq "site") {
        $pt->{AutogeneratedOfferID} = $pt->{_offid_};
        if ($pt->{WatchLogOfferID}) {
            $pt->{_offid_} = $pt->{WatchLogOfferID};
        }
    }

    if ((!defined $pt->{_offid_}) || ($pt->{_offid_} eq "")) {
        $oe->{err}{offer_id}++;
        return;
    }
    $fcount->{"offers_ids_ok"}++;

    #Не обрабатываем офферы, у которых нет картинки или она не валидна, в рекоме данная проверка игнорируется
    unless ($self->is_valid_pt_picture($pt)) {
        $oe->{err}{picture}++;
        return;
    }
    $fcount->{"offers_picture_url_ok"}++;

    #Не обрабатываем офферы, у которых нет урла
    unless( $pt->{url} ){
        $oe->{err}{url}++;
        return;
    }
    #Не обрабатываем офферы, у которых битые урлы
    if( $pt->url =~ /[\\"]/ ){
        $oe->{err}{url}++;
        return;
    }
    if (!$self->get_pt_fixed_url($pt)) {
        $oe->{err}{fixed_url}++;
        return;
    }

    $fcount->{"offers_url_ok"}++;

    #Не обрабатываем офферы, у которых нет валюты
    if ((exists $pt->{price}) and (defined $pt->{price})){ # см тикет SUPBL-8
        if (!$pt->{currencyId}) {
            my $currency_id = $self->get_region_currency;
            if (!$currency_id) {
                $oe->{err}{no_currencyId}++;
                return;
            } else {
                $pt->{currencyId} = $currency_id;
            }
        }
    }
    $fcount->{"offers_currency_ok"}++;

    if (!$pt->{orig_domain}) {
        my $s = $proj->site($pt->url);
        $s = $proj->site('https://www.airfrance.ru') if ($task->{OrderID} == 15959129); # Костыль! по-нормальному сделать в DYNSMART-539
        $s = $proj->site('https://www.bonprix.ru') if ($task->{OrderID} == 9268131); # Костыль! по-нормальному сделать в DYNSMART-539
        $pt->{orig_domain} = $s->domain;
    }
    #Не обрабатываем офферы, у которых проблемы с доменами
    unless( $pt->{orig_domain} && $pt->{orig_domain_id}){
        $oe->{err}{domain}++;
        return;
    }
    unless( $pt->{main_mirror} && $pt->{main_mirror_id}){  # main_mirror выставлен перед process_offer
        $oe->{err}{main_mirror}++;
        return;
    }

    if (
        $self->proj->options->{banned_domains}->{get_sec_level_domain($pt->{main_mirror})} ||
        $self->proj->options->{banned_domains}->{get_sec_level_domain($pt->{orig_domain})} ||
        $self->proj->options->{banned_domains}->{$pt->{main_mirror}} ||
        $self->proj->options->{banned_domains}->{$pt->{orig_domain}}
    ) {
        $oe->{err}{domain}++;
        $oe->{err}{main_mirror}++;
        return;
    }

    $fcount->{"offers_domains_ok"}++;

    my $gentype = 'perf';

    my ($custom_text_templates, $custom_title_templates) = $self->get_custom_templates($pt->{main_mirror}, $pt->ad_type);
    if ($custom_text_templates and $custom_title_templates) {
        $pt->set_custom_templates($custom_text_templates, $custom_title_templates);
        $gentype = 'perf_vip';  # используется при выборе тайтлов!
    }

    return +{
        gen_params => {
            retarg => $retarg,
            gen_phrases => $gen_phrases,
            phraseless => $phraseless,
            gentype => $gentype,
        }
    }
}

sub is_valid_pt_picture {
    my $self = shift;
    my $pt = shift;

    my $imgs = $pt->get_fixed_images;
    unless (defined($imgs) and @$imgs) {
        return;
    }
    unless($pt->get_fixed_picture) {
        return;
    }
    return 1;
}

# генерация баннеров через вызов perf_banners-методов; на YT будет вызываться несколько раз из-за RPC
sub process_offer_generate_native {
    my $self = shift;
    my $pt   = shift;
    my %par  = @_;

    my $ctx  = $par{ctx};
    my $gen_params = $par{gen_params};

    my $proj = $self->proj;
    my $logger = $proj->logger;
    my $task = $self->taskinf;
    $self->load_custom_phrases_settings;

    my (@arr, @rpc);

    my ($oe, $timer) = ($ctx->{offer_errors}, $ctx->{timer});
    my $fcount = $self->get_funnel_for_pt($pt, $ctx);

    my $offid = $pt->{_offid_};
    my $gen_ctx = ($ctx->{generate_native} //= {});

    # вычисляем фразы из соответствующего метода $pt
    # кэшируем результат в $ctx, чтобы не перевычислять, запросы добавляем в @rpc
    my $get_phrases = sub {
        my ($key, $method, %opts) = @_;
        my $h = ($gen_ctx->{$key} //= {});
        return if $h->{done};

        $h->{ctx} //= {};

        # таймер передаём только на время выполнения метода, чтобы не сериализовывать его
        $logger->debug("get_phrases $key");
        $timer->append_prefix("generate:$key");
        $h->{ctx}{timer} = $timer;

        my ($arr, $rpc) = $pt->$method(
            ctx => $h->{ctx},
            %{$self->get_banners_method_par($pt)},
            $opts{DUMMY_PHRASE} ? (DUMMY_PHRASE => $opts{DUMMY_PHRASE}): (),
        );
        delete $h->{ctx}{timer};
        $timer->pop_prefix;

        if (!$arr) {
            push @rpc, @$rpc;
            return;
        }
        delete $h->{ctx};  # уже не нужен
        $h->{done} = 1;
        return $arr;
    };

    if ($gen_params->{retarg}) {
        #Для ретаргетинга не генерируем множество фраз, а оставляем одну
        my $arr = $get_phrases->('retarg', 'perf_banners_single', DUMMY_PHRASE => "offerid_$offid",);
        if ($arr and @$arr) {
            $_->{phrase} = "offerid_$offid" for @$arr;
            $_->{template} = 'offer' for @$arr; # https://st.yandex-team.ru/DYNSMART-543
            $_->{letter} = 's' for @$arr;
            push @arr, @$arr;
        }
    }

    if ($gen_params->{gen_phrases}) {
        if ($self->{custom_phrases_allowed} and $pt->{custom_phrases}) {
            if (!$gen_ctx->{custom_phrases}++) {
                # DYNSMART-491: дописываем кастомные фразы из фида и не генерим наши
                my @custom_phrases = split ",", $pt->{custom_phrases};

                if (scalar @custom_phrases > $self->{custom_phrases_max_count_per_offer}) {
                    @custom_phrases = @custom_phrases[0..$self->{custom_phrases_max_count_per_offer}-1];
                }
                for my $ph (@custom_phrases) {
                    push @arr, { phrase => $ph, title => '', template => 'custom', letter => 'q'}; # https://st.yandex-team.ru/DYNSMART-543
                }
            }
        }
        my $arr = [];
        my $business_type = $self->get_business_type;
        #  Для business_type other и базового $pt нативная генерация только по use_as_name
        my ($ignore_native_by_business_type) = get_generation_flags_for_business_type($business_type);
        if ($ignore_native_by_business_type && $pt->is_base_product) {
            $arr = $get_phrases->('name_only', 'perf_banners_by_name');
        }
        else {
            $arr = $get_phrases->('main', 'perf_banners');
        }
        if ($arr and @$arr) {
            push @arr, @$arr;
        }
    }

    if ($gen_params->{phraseless}){
        my $arr = $get_phrases->('phraseless', 'perf_banners_single', DUMMY_PHRASE => "___PHRASE_LESS___",);
        if ($arr and @$arr) {
            $_->{phrase} = '' for @$arr;
            $_->{template} = 'no_phrase' for @$arr; # https://st.yandex-team.ru/DYNSMART-543
            push @arr, @$arr;
        }
    }

    return (\@arr, \@rpc);
}

sub process_offer_generate_external {
    my $self = shift;
    my $pt   = shift;
    my %ppar  = @_;

    return ([], []) unless ($ppar{gen_params}{gen_phrases} or $ppar{gen_params}{retarg});
    return $self->SUPER::process_offer_generate_external($pt, %ppar);
}

sub postprocess_external {
    my $self = shift;
    my $pt = shift;
    my $arr = shift;
    my $source_info = shift;
    my %ppar = @_;

    $arr = $self->SUPER::postprocess_external($pt, $arr, $source_info, %ppar);
    my @arr;
    if ($ppar{gen_params}{gen_phrases}) {
        push @arr, @$arr;
    }
    if ($ppar{gen_params}{retarg} and @$arr) {
        my $retarg_banner = dclone($arr->[0]);
        $retarg_banner->{phrase} = "offerid_".$pt->{_offid_};
        $retarg_banner->{letter} = 's';
        push @arr, $retarg_banner;
    }
    return \@arr;
}

sub process_offer_generate_minuswords {
    my $self = shift;
    my $pt   = shift;
    my %par  = @_;

    my $ctx  = $par{ctx};
    my $timer = $ctx->{timer};

    my $proj = $self->proj;
    my $task = $self->taskinf;

    $timer->time('minuswords');
    if ($par{gen_params}{phraseless}) {
        # не дописывать минус-слова для пустых фраз DYNSMART-418
        return '';
    }

    my $readable_phrases = [grep { $_->{phrase} !~ /^offerid_/i } @{$par{phrases}}];
    if (!@$readable_phrases) {
        # nothing to do
        return '';
    }

    return $pt->minus_words;
}

sub process_offer_combine {
    my $self = shift;
    my ($native_arr, $external_arr) = @_;
    if (grep { $_->{phrase} =~ /^offerid_/ } @$native_arr) {
        $external_arr = [ grep { $_->{phrase} !~ /^offerid_/ } @$external_arr ];
    }
    return [ @$native_arr, @$external_arr ];
}

# обработка данных, полученных из perf_banners; перевод в нужный для БК формат
sub process_offer_finalize {
    my $self = shift;
    my $pt   = shift;
    my %par  = @_;

    my $ctx  = $par{ctx};
    my ($oe, $timer) = ($ctx->{offer_errors}, $ctx->{timer});
    my $fcount = $self->get_funnel_for_pt($pt, $ctx);

    my $proj = $self->proj;
    my $task = $self->taskinf;
    my $filters_params = $self->filters_params;
    my $logger = $proj->logger;
    $self->load_custom_phrases_settings;
    $logger->debug("process_offer_finalize ...");
    $logger->debug("product_md5: ".$pt->clean_md5);
    $logger->debug("filters_params:", $filters_params);

    my $arr = $par{phrases};
    my $gen_params = $par{gen_params};

    my %result = (
        spec_count => {},
        phrases => [],
    );

    my $bl_banner_details = {
        title_source        => '',
        title_template      => '',
        title_template_type => '',
        offer_source        => $pt->offer_source,
        exp_gen             => 0,
    };

    # выбираем общий для всех привязок title
    # следовательно, по одному офферу генерится один баннер, используем это далее!
    my ($title, $long_title);
    my $is_vip = ($gen_params->{gentype} eq 'perf_vip') ? 1 : 0;
    my $title_res = $self->choose_title($pt, $par{titles}, is_vip => $is_vip); # DYNSMART-633
    return \%result unless (%$title_res and $title_res->{title});  # не можем вернуть баннеры без заголовка, если он обязателен
    my $long_title_res = $self->choose_title($pt, $par{long_titles}, is_vip => $is_vip);
    if (%$long_title_res && !$self->is_valid_long_title_length($long_title_res->{title}, $title_res->{title})) {
        $long_title_res = {};
    }
    $pt->{first_title} = $title_res->{title}; # записываем выбранный заголовок в product, чтобы сработал пост-маппинга на поле заголовка
    $title = $title_res->{title};
    $bl_banner_details->{title_source} = $title_res->{title_source};
    $bl_banner_details->{title_template} = $title_res->{title_template};
    $bl_banner_details->{title_template_type} = $title_res->{title_template_type};

    $long_title = %$long_title_res ? $long_title_res->{title} : '';
    if (%$long_title_res) {
        $bl_banner_details->{long_title_source} = $long_title_res->{title_source} // '';
        $bl_banner_details->{long_title_template} = $long_title_res->{title_template} // '';
        $bl_banner_details->{long_title_template_type} = $long_title_res->{title_template_type} // '';
    }
    $logger->debug("title: $title, long_title: $long_title");

    $fcount->{"banners_with_title"}++;
    # https://st.yandex-team.ru/DYNSMART-440
    # дополнительный маппинг не должен использовать настроек исходного фида, кроме тех, что даны в $par
    $timer->time('finalize:post_mapping');
    $self->proj->fdm->proceed_post_mapping_for_pt($pt, data_type => $self->taskinf->{Resource}{LastValidFeedType});

    # Подготовка поля Info ($jsondata)
    $timer->time('finalize:info');
    #my $PhraseID = $task->{Resource}{Targets}[0]{DynamicConditionID} ; #Номер условия нацеливания
    my $PhraseID = $pt->{offerfilters}; #Номер условия нацеливания
    my $Direct2BSData = '{"OrderID": '.$task->{OrderID}.', "GroupExportIDs": [ '. join(',', @{$task->{GroupExportIDs} || []} ) .' ]}';

    my $pttext = $pt->name || join(' ', grep {defined($_)} $pt->typePrefix, $pt->model, $pt->vendor, $pt->{categpath} );

    my $ph = $proj->phrase($pttext);
    my $ctg_directids = join(',', $ph->get_minicategs_directids);
    my $ctgnames = join('/', $ph->get_minicategs);
    $logger->debug("categories from pttext {$pttext}: $ctgnames");

    my $business_type = $self->get_business_type;
    my $is_body_need = ($task->{Resource}{UseAsBody} && $task->{Resource}{UseAsBody} eq $self->proj->options->{'use_as_body_text_disable_flag'}) ? 0 : 1;
    my $body_builder = $self->proj->body_builder();
    my $body_builder_res = $body_builder->get_pcode_data($pt, $business_type, $is_body_need);

    #DYNSMART-697
    #дополняем логику категоризации баннера:
    #если категоризация не удалась - просто берём категорию оффера и подставляем её в категорию баннера.
    if ($pt->{minicategs}) {
        # категорийные флаги берем всегда
        my @categs = split(m/\//, $pt->{minicategs});
        if ( !$ctgnames ) {
            $ctg_directids = join (',',  map{$self->proj->categs_tree()->get_minicateg_directid($_)} @categs);
            $ctgnames = $pt->{minicategs};
            $logger->debug("categories from \$pt->{minicategs}: $pt->{minicategs}");
        }
    }

    my $picture = $pt->get_fixed_picture;
    my $images  = $pt->get_fixed_images;

    my $currencyId = $pt->{currencyId} || '';
    $currencyId = 'TRY' if $currencyId =~ /^tl$/i;

    #Данные по карточке из Маркета
    my $market_text = $pt->{name} || join(' ', grep { $_ } ($pt->{typePrefix}, $pt->{vendor}, $pt->{model}));
    my $market_response = $proj->market_subphraser->get_market_ids([$market_text])->{$market_text};  # [model_id,categ_id]
    my $market_data = $self->get_market_data($market_response->[0]);

    #Данные для Крипты
    my $attrh = {};
    my $parseh = $pt->parse;
    $attrh->{$_} = ''.$parseh->{$_} for grep {$self->proj->is_bs_compatible( ''.$parseh->{$_} )} grep { $parseh->{$_} }  qw{ model brand type };
    $attrh->{$_} = $pt->{$_} for qw{ categoryId };
    $attrh->{market} = $market_data;

    my $jsondata = {
        counter_id => $task->{CounterID},
        text => {
            client_id => $task->{ClientID},
             ($currencyId ? (currency_iso_code => $currencyId) : ()),
             ($business_type ? (business_type => $business_type): ()), # DYNSMART-301
         },
         image => $picture,
         images => $images,
         ($pt->{price} ?
             (price => {
                 current => $pt->valid_price( $pt->{price} ),
                 ( $pt->{oldprice} ? (old => $pt->valid_price( $pt->{oldprice} )) : () ),
             })
             : ()),
         attribute => $attrh,
         adv_type => $pt->ad_type,
         smart_tgo => 1,
    };

    if(defined($pt->{additional_data})&&(ref($pt->{additional_data}) eq 'HASH')){
        # merge_hashes будет при совпадении полей брать то, что во втором хэше. Это нормально, так как информация в $pt->{additional_data} более приоритетная
        merge_hashes( $jsondata, $pt->{additional_data} );
        if( $pt->{additional_data}{facility} ){ #Временный костыль для туризма?
            my $fctd = $self->proj->fdm->facilities_dict;
            my $text = $pt->{additional_data}{facility};
            $jsondata->{text}{facility} = $text; #Исходные данные для вёрстки
            $text = join(',', grep {$_} map { $fctd->{$_} } split(';', lc($text)));
            $jsondata->{facility} = $text; #Исправленные данные для машинного обучения
        }
        if( $pt->{additional_data}{class} ){ #Временный костыль для туризма?
            $jsondata->{text}{class} = $pt->{additional_data}{class};
        }
    }
    if ($jsondata->{attribute}->{market}) {
        my $market_attrs = $jsondata->{attribute}->{market};
        $market_attrs->{$_} = $self->proj->make_bs_compatible_or_empty($market_attrs->{$_}) for keys $market_attrs;
    }

    if($task->{search_body}){
        $jsondata->{text}{body} = $self->proj->make_bs_compatible_or_empty($task->{search_body}) || $task->{search_body};
    }

    $timer->time('finalize:info:location');
    if(defined($pt->{location}) && ($pt->ad_type || '') ne 'realty'){ #Временный костыль для туризма?
        my $lctn = $pt->{location};
        if($lctn){
            $jsondata->{text}{location} = $lctn;
            my $reginf = $pt->get_region_inf();
            if($reginf->{geoids}){ #Если смогли определить регион
                $jsondata->{location} = $reginf->{maingeoid};
            }else{
                # не смогли определить location, по следам https://st.yandex-team.ru/DYNSMART-241
                delete $jsondata->{location} if (exists $jsondata->{location});
                # $jsondata->{location} = -1;
            }
        }
    }

    $timer->time('finalize:info');

    # форматируем price (могли поменять)
    if ($jsondata->{price}) {
            $jsondata->{price}{$_} = $pt->valid_price($jsondata->{price}{$_}) for grep { defined $jsondata->{price}{$_} } qw/ current old /;
    }
    if ($business_type) {
        $jsondata->{text} ||= {};
        $jsondata->{text}{business_type} = $business_type;
    }

    $jsondata->{body_for_direct} = $body_builder_res->{body};
    $jsondata->{callouts_list} = $body_builder_res->{callouts};
    $jsondata->{long_body} = $body_builder_res->{long_body} if $body_builder_res->{long_body};

    for my $key ( keys %{$jsondata->{text}} ) {
        $jsondata->{text}{$key} = $self->proj->make_bs_compatible_or_empty($jsondata->{text}{$key}) || $jsondata->{text}{$key};
    }

    #хэш для проверки на совместимость с движком. Из него нужно вырезать урлы, например, урлы картинок
    my $jsondata_text_fields = dclone($jsondata);
    delete $jsondata_text_fields->{image};
    delete $jsondata_text_fields->{images};

    # this is title for TGA smart; it goes also to Info['text']['name_for_direct'] in postmapping
    # as now (13.05.2020) equals title for classic smart (TitleMediaSmart), but it may change
    my $title_tga = $pt->{first_title} // '';

    my $title_ext = '';
    if ($self->taskinf->{Resource}{LastValidFeedType} eq 'YandexMarket') {
        $title_ext = $pt->{second_title} // '';
    }

    if ( ! $proj->is_bs_compatible_recursive( $jsondata_text_fields ) ) {
        return \%result;
    }
    my @bs_check_fields = ($title, $title_tga, $title_ext);
    if (grep { !$proj->is_bs_compatible($_) } @bs_check_fields) {
        return \%result;
    }
    $fcount->{"banners_with_bs_compatible_text_fields"}++;

    # временный фикс для IRT-267
    my $url = $self->get_pt_fixed_url($pt);

    my @phids = grep {defined($_)} split ',', $PhraseID;
    s/\:\d+// for @phids;

    # создаём временную копию $filters_params, из которой будем вычёркивать те фильтры, которые требуют только одной строки
    my %current_offer_filters_params = %$filters_params;

    # скор привязок - по нему отбирается топ (чем лучше привязка, тем выше скор)
    # скор должен зависеть от баннера, т.к. движок ставит noload по кол-ву баннеров и геноцидить нужно соответственно
    # считаем в миллионных долях, т.е. Score - число из {0,1,2,...,999999}
    if ($pt->{orig_domain} =~ /^(www\.)?booking\.com$/) {
        # в букинге можно ранжировать по get-параметру rank из урла:
        # "http://booking.com/hotel/me/villa-mira-tivat.ru.html?dsa=1&rank=100"
        # TODO
    }

    $timer->time('finalize:phrases');

    $jsondata->{long_title} = $long_title if $long_title;
    $jsondata->{display_href} = $self->get_green_url($pt, $arr->[0]);  # TODO: более умно выбирать зелёный урл

    my %banner = (
        Title               => $title_tga,  # тайтл ТГО: https://st.yandex-team.ru/BSSERVER-12614#5e7b1881f9e9d713d14c2d32
        TitleMediaSmart     => $title,
        TitleExtension      => $title_ext,
        Body                => $body_builder_res->{body},
        Url                 => $url,
        CanonizedUrl        => $pt->{canonized_url} // $url,
        Info                => $proj->json_obj->encode($jsondata),
        OfferID             => $pt->{_offid_},
        AutogeneratedOfferID => $pt->{AutogeneratedOfferID},
        ClientID            => $self->taskinf->{ClientID},
        ParentExportID      => $task->{ParentBannerId},
        Categories          => $ctg_directids,
        CategoriesNames     => $ctgnames,
        Site                => Utils::Urls::safe_punycode_decode($pt->{orig_domain}),
        TargetDomain        => $pt->{main_mirror},  # склейщик васи брядова
        TargetDomainID      => $pt->{main_mirror_id},
        SiteFilter          => $pt->{orig_domain},  # из урла
        SiteFilterID        => $pt->{orig_domain_id},
        Lang                => 'ru',
        MarketModelID       => $market_response->[0],
        MarketCategoryID    => $market_response->[1],
        BSData              => $Direct2BSData, # хеш для БК
        DynamicBannerID     => $self->get_pt_checksum($pt), # контрольная сумма чанки всегда считается на нашей стороне и отправляется в БК, это md5int от OfferiD
        categoryId          => ($pt->{categoryId}||''),
        BLBannerDetails     => $proj->json_obj->encode($bl_banner_details),
        BannerlandBeginTime => $self->taskinf->{BannerlandBeginTime},
        OrderTags           => $self->taskinf->{OrderTags} // '[]',
        OfferFilters        => $pt->{offerfilters},
    );

    $banner{BusinessId} = $self->datacamp_business_id;
    $banner{ShopId} = $self->datacamp_shop_id;
    $banner{OfferYabsId} = $pt->{offerYabsId};

    # not in FIELDS, but will be used in full_state:
    $banner{GroupExportID} = $self->GroupExportID;

    $fcount->{"banners_after_throttle"}++;

    my $banned_simdistances = $self->get_banned_simdistances();

    $_->{letter} = $_->{letter} || 'n' for @$arr;

    my $sorted_arr = $self->sort_phrases_by_priority_order($arr);

    for my $el (@$sorted_arr) {
        my $letter = $el->{letter};

        my $phtext = $el->{phrase};
        my $is_retarg_phrase = ($phtext =~ /^offerid_/);
        my $is_dse_phrase = ($el->{template} eq 'dse');
        my $is_phrase_from_title = ($el->{template} eq 'phrase_from_title');
        my $is_custom_phrase = ($el->{template} eq 'custom');

        next if !$is_custom_phrase && $self->{OnlyClientPhrases};
        next if $proj->phrase($phtext)->is_porno_phrase;

        #Минус-слова
        if ($par{minuswords} and !$is_retarg_phrase and !$is_dse_phrase and !$is_phrase_from_title
            and !($self->{custom_phrases_allowed} and $pt->{phrases})
            and $phtext !~ /~0\s*$/)
        {
            $phtext .= " ".$par{minuswords};
        }

        PHRASEID: for my $phid (grep { $current_offer_filters_params{$_} } @phids){
            $logger->debug("try phrase {$el->{phrase}} with PhraseID $phid ...");
            #Определяем тип фразы
            my $match_type = $pt->match_type eq 'norm' ? 'norm' : 'snorm';

            #Учёт точечных настроек фильтров, если фраза не подходит под настройки фильтра, не выводим её в файл
            my $fsettings = $filters_params->{$phid};
            my $orig_target_funnel = $fsettings->{target_funnel};

            # DEPRECATED: TODO: удалить, когда все старые product-ы будут обработаны (#2)
            if ($fsettings->{_was_product_page_visit}) {
                $orig_target_funnel = 'product_page_visit';
            }

            my $target_funnel = $self->get_effective_target_funnel($orig_target_funnel);
            my $curphtext = $phtext;
            if ($fsettings->{goal_context_id}) {
                $match_type = 'goal-context';
                $curphtext = $fsettings->{goal_context_id};
                # но в Info{"targets_params"} не добавляем, т.к. Info на баннер
            } elsif ($target_funnel) {
                if ($target_funnel eq 'product_page_visit') {
                    # для product_page_visit печатаем только фразы ретаргетинга
                    if (!$is_retarg_phrase) {
                        $logger->debug("skip: product_page_visit: only retarg");
                        next PHRASEID;
                    }
                } elsif ($target_funnel eq 'new_auditory') {
                    # для new_auditory печатаем только человекочитаемые фразы
                    if ($is_retarg_phrase) {
                        $logger->debug("skip: new_auditory: only readable");
                        next PHRASEID;
                    }
                }
            }

            my $search_count = $el->{minf}{search_count} || 0;  # поставили при генерации

            if ($target_funnel) {
                if ($orig_target_funnel eq 'product_page_visit') {
                    $result{spec_count}{text_phrases_before_search_filtering}++ if !$is_retarg_phrase;

                    # частотность считается и кэшируется без учёта минус-слов
                    # dse не фильтруем -- уже профильтрованы!
                    # phrase_from_title не фильтруем, это резервная фраза!
                    if (!$is_dse_phrase and !$is_retarg_phrase and !$is_phrase_from_title
                        and defined $search_count
                        and $search_count > $proj->options->{wide_phrase_search_count_limit})
                    {
                        $result{spec_count}{text_phrases_deleted_in_search_filtering}++;
                        $logger->debug("skip: is_phrase_wide");
                        next PHRASEID;
                    }
                }
            }

            my $sim_distance = $self->get_sim_distance_by_letter($letter, $curphtext);
            next if $banned_simdistances->{$sim_distance}; # бан по симдистансу для отдельных клиентов
            my $bl_phrase_details = {
                bl_phrase_template      => ($el->{template} // ''),
                bl_phrase_template_type => $letter,
                search_count            => $search_count,
            };

            my $h = {
                %banner,
                Type               => $match_type,
                Text               => $curphtext,
                PhraseID           => $phid, # номер фильтра
                CTR                => 0, # предсказание блока снизу
                PCTR               => 0, # предсказание спецразмещения
                SpecPlaceFlag      => 1, # показывать в спецразмещении?
                OnlyYandexFlag     => 0, # показывать только на внутренних сервисах Яндекс (не показывать на партнерских)
                SimDistance        => $sim_distance,
                APCRatio           => 1000000, # это единица, т.к. в миллионных долях, сюда можно записывать в зависимости от категорий исправляющую ставку
                searchcount        => 0,  # deprecated
                OnlyRetargetingPhrase  => int($is_retarg_phrase),
                LMPhraseFlag       => 0,  # для EXTPHRASE было $lmphrase = 1, сейчас их нет

                BLPhraseDetails    => $proj->json_obj->encode($bl_phrase_details),
                bl_phrase_template_type => $letter,  # not in FIELDS, but required for filter_bp_duplicates on YT and for BLPhraseTemplateID
                Score => 0,  # not in FIELDS, but required for limit_banners_count on YT!

                # expt_add_fields:
                product_md5        => $pt->clean_md5,
            };

            if ($el->{letter} and $el->{letter} eq 'e') {
                $h->{LMPhraseFlag} = 1;
                $h->{Type} = 'snorm';
            }

            # temp field for sorting
            my $phrase_score = 0;
            if ($is_retarg_phrase) {
                $phrase_score = 1.0;
            } elsif ($search_count) {
                $phrase_score = 0.9 * $search_count / (1 + $search_count);
            }
            $h->{phrase_score} = $phrase_score;

            $self->check_banner_phrase_fields($h);
            $self->cast_result_values($h);
            push @{$result{phrases}}, $h;
            $logger->debug("phrase added!");
            # для (same_products + goal_context_id) и product_page_visit нужна только одна фраза
            delete $current_offer_filters_params{$phid} if (($fsettings->{goal_context_id})
                    or ($target_funnel eq 'product_page_visit'));

            next if ($fsettings->{goal_context_id}); # одной фразы достаточно в этом случае
        } # end loop on @phids
    } # end loop on @$arr

    $result{phrases} = $self->filter_phrase_duplicates($result{phrases});

    my $max_phrases_per_banner = $self->get_max_phrases_per_banner;
    if (defined $max_phrases_per_banner) {
        $result{phrases} = $self->get_top_phrases(
            $result{phrases},
            limit => $max_phrases_per_banner,
            sort_by => 'phrase_score',
            group_by => 'BannerID',
        );
    }

    $fcount->{"banners_with_phrases"}++ if @{$result{phrases}};

    return \%result;
}

# раньше в task_id входила чанка, но после перехода на YT от неё избавились
sub task_id {
    my $self = shift;
    my $task = $self->taskinf;
    return join("_", $task->{OrderID}, @{$task->{GroupExportIDs} || []});
}

sub is_valid_long_title_length {
    my ( $self, $long_title, $title) = @_;
    my $min_length_difference = 1.2;
    return $long_title && $title && length($long_title) / length($title) >= $min_length_difference;
}

sub get_all_tasks_from_yt_table {
    my $self = shift;
    my $table = shift // $self->get_default_input_tasks_yt_table();

    my $tasks = $self->SUPER::get_all_tasks_from_yt_table($table);

    #Исправление неправильного формата фильтра
    $_->{Resource}{Targets} = { map {%$_} @{$_->{Resource}{Targets}} } for grep { ref( $_->{Resource}{Targets} ) eq 'ARRAY' } @$tasks;

    #Фильтрация тестовых заказов из Директа, так как они забивают всю очередь
    $tasks = [ grep { !defined($_->{Resource}{FeedUrl}) || $_->{Resource}{FeedUrl} !~ /yandex\.st/ } @$tasks ]; # фильтруем таски директа

    # Хардкодим подмену фидурла
    my $kostyl_order2feedurl = $self->proj->options->{bannerland_perf_kostyl_order2feedurl};
    $_->{Resource}{FeedUrl} = $kostyl_order2feedurl->{$_->{OrderID}} for grep { $kostyl_order2feedurl->{$_->{OrderID}} } @$tasks;

    # дописываем домен
    $_->{domain} = $self->proj->site($_->{Resource}{FeedUrl} || "")->domain for @$tasks;

    return $tasks;
}

sub filters_params :CACHE {
    my ($self) = @_;
    my $task = $self->taskinf;
    my $targdata = $task->{Resource}{TargetsParams}; #Фильтры для фида
    unless( $targdata ){ #Нет данных по фильтрам
        print STDERR "WARN: Empty TargetsParams\n";
        return {};
    }
    return $targdata;
}

sub update_monitor_state {
    my $self = shift;
    my $msg = shift;
    my $queue = BM::BannersMaker::Tasks::TaskQueue->new({ proj => $self->proj, type => $self->get_task_type() });
    $queue->update_monitor_state($self, $msg);
}

sub get_optional_fields {
    my $self = shift;
    my $result_fields = $self->proj->options->{'perf_result_columns'};
    my $optional_fields = { map {$_->{'name'} => 1} grep {$_->{'optional'}} @$result_fields };
    return $optional_fields;
}

sub get_result_fields {
    my $self = shift;
    return $self->proj->options->{'perf_result_columns'};
}

1;
