package BM::BannersMaker::Tasks::DynGrpTask;

use utf8;
use open ':utf8';

use std;
use base qw(ObjLib::ProjPart);

use Data::Dumper;

use Encode;
use Encode qw{_utf8_on _utf8_off};

use JSON;
use URI::Escape;
use LWP::UserAgent;
use IPC::Open2;

use List::Util qw(min max);
use Scalar::Util qw(weaken);

use Digest::MD5 qw(md5_hex);
use File::Basename qw (basename);
use XML::Parser;
use POSIX qw(strftime);
use XMLParser;
use Utils::Sys qw(do_safely mem_usage);
use Utils::Hosts qw(get_curr_host);
use BM::BannersMaker::Tasks::Task;
use BM::BannersMaker::Tasks::TaskQueue;

########################################################
#Доступ к полям
########################################################

__PACKAGE__->mk_accessors(qw(
    domain
    tasks
));

########################################################
# Интерфейс
########################################################
#   make_task                       главный метод, итерируется по групповой таске и запускает атомарные таски
#   make_merged_tskv                генерит tskv по сайту и доп. источникам из yt
#   sort_inner_tasks                сортирует атомарные таски внутри групповой
#   first_task                      возвращает ссылку на первую атомарную таску в групповой
#
#   grlog_file
#   open_grlog 
#   close_grlog
#   grlog
#   update_monitor_state
#   get_tskv_filters


########################################################
# Инициализация
########################################################


########################################################
# Методы
########################################################


sub init {
    my $self = shift;
    $self->check_deprecated_options();
}

sub check_deprecated_options {
    my $self = shift;

    # DynGrpTask не наследуется от Task
    # возможно это надо поправить, а пока вызываем метод так
    return BM::BannersMaker::Tasks::Task::check_deprecated_options($self);
}

#Первый таск
sub first_task :CACHE {
    my ($self) = @_;
    my $first_task = $self->proj->dyntask($self->tasks->[0]);

    # задаем директорию для tskv
    my $tskvdir = $first_task->basedir."/".$first_task->filedmn ."/tskv";
    $first_task->{extdir} = $tskvdir;

    return $first_task;
}

sub grlog_file :CACHE {
    my ($self) = @_;
    my $first_task = $self->first_task;
    #Логи для текущей пакетной обработки
    my $grlog_dir = $self->{grlog_dir} || "grplog";
    my $logfile = $first_task->basedir."/".$first_task->filedmn ."/$grlog_dir";

    (-d $_ || mkdir $_) for  ( $first_task->basedir."/".$first_task->filedmn, $logfile );
    $logfile .= "/log";
    return $logfile;
}

sub open_grlog {
    my ($self) = @_;
    open(my $grlog, ">> ".$self->grlog_file) or warn "Can't open grlog: $!\n";
    $self->{grlog} = $grlog;
}

sub close_grlog {
    my ($self) = @_;
    close($self->{grlog});
}

sub grlog {
    my ($self, $text) = @_;
    my $fh = $self->{grlog};
    print $fh $self->proj->curtime." [".$self->{domain}."] ".$text."\n";
}

sub update_monitor_state {
    my ($self, $state, %opts) = @_;
    my $dbtable = $self->proj->dbtable("DynGrpTasksMonitorEntries", undef, "bannerland_dbh");
    my $entry = {
        Domain => $self->{domain},
        Host => get_curr_host(),
        State => $state
    };
    if ($opts{DebugInfo}) {
        my $debug_info = $opts{DebugInfo};
        $debug_info = JSON::to_json($debug_info) if ref($debug_info);
        $entry->{DebugInfo} = $debug_info;
    }
    $dbtable->Add($entry);
}

sub sort_inner_tasks {
    my ($self) = @_;

    # Задаем порядок задач:
    # задачи из одной группы располагаем рядом;
    # сортируем группы по времени выполнения - в начале будут те , в которых есть задачи,
    # которые выполнялись очень давно или еще не выполнялись.

    my $groups = {};
    for my $task (@{$self->tasks}) {
        my $task_obj = $self->proj->dyntask($task);
        my $task_id = $task_obj->task_id;
        my $time = $task_obj->get_last_exec_time;
        my $group = $task_obj->group_export_id;
        unless (exists $groups->{$group}) {
            $groups->{$group} = {tasks => [$task], task_ids => [$task_id], min_time => $time};
        } else {
            push @{$groups->{$group}->{tasks}}, $task;
            push @{$groups->{$group}->{task_ids}}, $task_id;
            $groups->{$group}->{min_time} = $time if $time < $groups->{$group}->{min_time};
        }
    }

    my @keys = sort { $groups->{$a}->{min_time} <=> $groups->{$b}->{min_time} } keys(%$groups);
    my @vals = @{$groups}{@keys};

    $self->{tasks} = [map{@{$_->{tasks}}} @vals];

    my $group_cnt = 0;
    for my $group_id (@keys) {
        $group_cnt++;
        my $group_info = $groups->{$group_id};
        my ($task_ids, $sort_key) = ($group_info->{task_ids}, $group_info->{min_time});
        $self->{task_sort_key}{$_} = $sort_key for @$task_ids;
        $self->grlog("sort_inner_tasks: group #$group_cnt: $group_id (sort_key=$sort_key) tasks: ".join(', ', @$task_ids));
    }
}

# есть в атомарных тасках фильтры не по спецурлам?
sub has_tskv_filters {
    my ($self) = @_;

    my @tskv_filters = map {
            ref($_->{Targets}) eq 'HASH'
            ?  (  keys %{$_->{Targets}} )
            : (  ref($_->{Targets}) eq 'ARRAY'
                 ? ( grep {! ( grep { defined($_->{kind}) && $_->{kind} eq 'equals' } ( ref($_->{Condition}) eq 'ARRAY' ? @{$_->{Condition}} : $_->{Condition} ) ) } grep { ref($_) eq 'HASH' } @{$_->{Targets}} )
                 : () )
        }
        map { ref($_->{Resource}) eq 'HASH' ?  $_->{Resource} : {} } grep {!$_->{Resource}{FeedUrl}} @{$self->tasks};
    
    $self->grlog(Dumper(['tskv_filters_data:', \@tskv_filters, [map { $_->{Resource} } @{$self->tasks}] ]));

    return 1 if @tskv_filters;
    return 0;
}

sub make_merged_tskv {
    my ($self) = @_;

    my $queue = BM::BannersMaker::Tasks::TaskQueue->new({ proj => $self->proj, type => "dyn" });

    # обходчик будет запускаться только по первой атомарной таске
    my $first_task = $self->first_task;

    $self->grlog("Tasks: ". (0+@{$self->tasks}));
    my $filename_tskv_gen;

    my $tskv_err = 0;
    if ($self->has_tskv_filters) { # попытаемся получить tskv по сайту и доп. источникам
        my $last_tskv_time = $first_task->get_domain_param("tskv_time");
        my $today = strftime('%Y-%m-%d', localtime(time));

        if ($last_tskv_time and $last_tskv_time ge $today) { # достаточно ли времени прошло с предыдущего обхода?
            $filename_tskv_gen = $first_task->find_last_format_file('tskv_gen');
            $self->grlog("last_tskv_time in db='$last_tskv_time', today='$today'");

            # проверяем, что последний tskv_gen существует и был сгенерирован сегодня
            if (!$filename_tskv_gen) {
                $self->grlog("old tskv_gen not found on host; Need Renew tskv");
            } elsif (substr(basename($filename_tskv_gen),0,11) gt strftime('%Y%m%d_%H', localtime(time-3600*25))) {
                # был сгенерирован сегодня
                $self->grlog("use old tskv_gen: $filename_tskv_gen");
                return $filename_tskv_gen;
            } else {
                $self->grlog("File $filename_tskv_gen is too old; Need Renew tskv");
            }
        }

        $queue->update_monitor_state($self, "Start", TaskID => "tskv");
        my $tskv_success = 0;
        do_safely(
            sub {
                $self->grlog("Renew tskv: Beg");
                $first_task->globallog("Renew ".$self->{domain}." tskv: Begin");
                $first_task->action_before_task; #Чтобы правильно перенаправить логи
                # $first_task->site->delete_parse_cache; #Удаляем данные кэша
                $first_task->{debug_info} = {};
                unless ($first_task->renew_tskv) {
                    $tskv_err = 1;
                }
                $first_task->action_after_task;  #Возвращаем перенаправление логов обратно
                $first_task->globallog("Renew ".$self->{domain}." tskv: End");
                $self->grlog("Renew tskv: End");
                $queue->update_monitor_state($self, "Finish", TaskID => "tskv",
                    DebugInfo => JSON::to_json($first_task->get_debug_info_hash() || {}));

                my $curtime = $first_task->curtime;
                my $tskv_size = $self->proj->file($first_task->filenames->{sitetskv_pkd})->wc_l;
                my $tree_size = $self->proj->file($first_task->filenames->{sitetree})->wc_l;
                $first_task->set_domain_params({
                    tskv_time => $curtime,
                    tskv_size => $tskv_size,
                    tree_size => $tree_size,
                });

                my $prev_host = $first_task->get_domain_param('host');
                my $cur_host = $self->proj->host_info->{host};
                if( $prev_host ne $cur_host ) {
                    $first_task->set_domain_params({
                        host => $cur_host,
                        prev_host => $prev_host,
                        host_changed_time => $curtime,
                    });
                }
                $tskv_success = 1;
                $filename_tskv_gen = $first_task->filenames->{tskv_gen};
            },
            no_die => 1,
            timeout => 45 * 3600,
            timeout_handler => sub {
                $queue->update_monitor_state($self, "Timeout", TaskID => "tskv");
                $tskv_success = 0;
            },
            die_handler => sub {
                $queue->update_monitor_state($self, "Fail", TaskID => "tskv");
                $tskv_success = 0;
            },
        );
        $first_task->set_debug_info('memory_usage_mb', mem_usage() / (1 << 20));
        return undef unless $tskv_success;
    } else {
        $self->grlog("Renew tskv: The tskv filter list is empty.");
        $self->grlog(Dumper([ map { $_->{Resource} } @{$self->tasks} ]));
    }

    if ($tskv_err) {
        $self->grlog("Renew tskv: error.");
        return undef;
    }

    my $reslines = 0;
    $reslines = $self->proj->file(($filename_tskv_gen))->wc_l if -e $filename_tskv_gen;
    $self->grlog("tskv_gen lines: ".$reslines);

    return $filename_tskv_gen;
}

sub make_task {
    my ($self) = @_;

    $self->check_deprecated_options;

    $self->update_monitor_state("Start");
    my $begin_time = time;
    my $proj = $self->proj;
    my %debug_info;

    # для мониторинга статусов
    my $queue = BM::BannersMaker::Tasks::TaskQueue->new({ proj => $proj, type => "dyn" });

    # лог групповой таски
    $self->open_grlog;

    # сортируем атомарные таски
    $self->sort_inner_tasks();

    # строим tskv по обходу сайта и доп. источников
    my $ext_tskv_gen = $self->make_merged_tskv;
    if (!$ext_tskv_gen and $self->has_tskv_filters) {
        $self->grlog("Fail make_merged_tskv; Exit");
        $self->update_monitor_state("Fail");
        $self->close_grlog;
        return;
    };

    my $cc = 0;
    my $total = @{$self->tasks};
    $debug_info{tasks_count} = $total;

    my $s = undef;
    my $grp_hash = {};

    my $grp_timeout = $self->get_grp_timeout_hours * 3600;

    # итерируемся по таскам
    for my $task (@{$self->tasks}){
        $cc++;
        my $curr_task = $proj->dyntask($task);
        my $timeout_hours = $self->get_task_timeout_hours($curr_task);
        $self->grlog("Task: ".$curr_task->task_id.", timeout: $timeout_hours hours");
        do_safely(
            sub {
                $s = $curr_task->site unless defined $s;
                $curr_task->{ext_site} = $s;
                $queue->update_monitor_state($curr_task, "Start");
                $self->grlog("Task [".$cc.'/'.$total."] ".$curr_task->task_log_id." BEG");
                $self->grlog(Dumper($task));
                $curr_task->{ext_tskv_gen} = $ext_tskv_gen;
                $curr_task->{debug_info} = {};
                $curr_task->{timeout_hours} = $timeout_hours;
                my $geid = $curr_task->group_export_id;
                my $tid = $curr_task->task_id;
                # для тасок с одинаковым GroupExportID передаем сигнатуру, чтобы копировать результат генерации
                if ($geid) {
                    if ($grp_hash->{$geid}) {
                        my $copy_tid = $grp_hash->{$geid}->{tid};
                        $self->grlog("Task $copy_tid copy to $tid");
                        $curr_task->{copy_from} = $copy_tid;
                        my $copy_debug_info = $grp_hash->{$geid}->{debug_info};
                        for (keys %$copy_debug_info) {
                            $curr_task->{debug_info}->{$_} = $copy_debug_info->{$_};
                        }
                    } else {
                        $grp_hash->{$geid} = {tid => $tid, debug_info => $curr_task->{debug_info}};
                    }
                }
                # запускаем атомарную таску
                $curr_task->make_task;
                $self->grlog("Task [".$cc.'/'.$total."] ".$curr_task->task_log_id." ".$curr_task->{result_stat}." END");

                my $state = $curr_task->{FalseStart} ? "FalseStart" : "Finish";
                $curr_task->set_debug_info('memory_usage_mb', int(mem_usage() / (1 << 20))) unless ($state eq "FalseStart");
                $queue->update_monitor_state($curr_task, $state, DebugInfo => JSON::to_json($curr_task->get_debug_info_hash() || {}));
                $debug_info{last_processed_task_sort_key} = $self->{task_sort_key}{$tid};
                $debug_info{processed_tasks_count}++;
            },
            no_die => 1,
            timeout => ($timeout_hours + 4) * 3600,  # с запасом, т.к при жёстком таймауте не экспортируются данные; он не должен срабатывать
            timeout_handler => sub {
                $queue->update_monitor_state($curr_task, "Timeout");
            },
            die_handler => sub {
                $queue->update_monitor_state($curr_task, "Fail");
            },
            verbose => 1,
        );
        last if time - $begin_time > $grp_timeout;
    }
    my $grp_state = ($cc < $total) ? "Timeout" : "Finish";
    $self->update_monitor_state($grp_state, DebugInfo => \%debug_info);
    $self->close_grlog;
}

sub get_grp_timeout_hours {
    my $self = shift;
    return 14 * 24;
}

sub get_task_timeout_hours {
    my $self = shift;
    my $task = shift;

    my $order_id = $task->taskinf->{OrderID};
    my $order_timeout = $self->proj->options->{dyn_timeout_hours_by_order_id}{$order_id};
    return $order_timeout if defined $order_timeout;

    if (!$task->get_param("begin")) {
        return 12;  # new task, start shows asap
    }

    my $n_tasks = @{$self->tasks};
    my $hours = int($self->get_grp_timeout_hours / $n_tasks);
    $hours = min($hours, 72);
    $hours = max($hours, 24);

    return $hours;
}

1;
