package BM::BannersMaker::Tasks::TaskQueue;

use utf8;
use open ':utf8';

use std;
use base qw(ObjLib::ProjPart);

use Data::Dumper;
use File::Copy;
use POSIX qw(strftime);

use Encode;
use JSON::XS;
use Utils::Sys qw(md5int get_file_lock release_file_lock _get_lockname);
use Utils::Hosts qw(get_curr_host);
use BM::BannersMaker::Tasks::TaskQueueSettings;

########################################################
#Доступ к полям
########################################################

__PACKAGE__->mk_accessors(qw(
    tasks_dir
    type
    source_table
));

########################################################
# Инициализация
########################################################

sub init {
    my ($self) = @_;
}

########################################################
# Интерфейс
########################################################
#
#   refresh_tasks                   удаляет старые таски из директории и добавляет новые
#   add_task                        добавляет новый файл таски в директорию
#   get_next_task                   следующую таску из очереди переводит в состояние work и ставит блокировку на таску
#   invoke_worker                   отдает таску воркеру и ставит блокировку на домен
#
#   get_settings                    возвращает список настроек для очередей 
#   tasks_name                      возвоащает название метода, который выкачивает таски для хоста
#   worker_name                     возвращает название метода воркера
#   worker                          возвращает ссылку на метод воркера
#   get_task_type                   вовзращает тип таски (dyn, perf)
#   get_task_lock_name              возвращает название блокировки для таски
#   read_task_from_file             читает таску из файла в json-объект
#   free_last_task                  удаляет текщую таску из очереди
#   tasks_dir                       возвращает название директории, в которой лежит очередь
#   get_task_count                  возвращает число тасков в очереди, не взятых в работу
#   get_task_files                  возвращает список файлов с тасками, не взятыми в работу    
#   get_work_files                  возвращает список файлов с тасками, взятыми в работу
#   is_task_running                 проверяет, в работе ли данный файл с таской
#   queue_table                     возвращает ссылку на таблицу с очередью
#   monitor_table                   возвращает ссылку на таблицу мониторинга очереди
#   update_monitor_state            добавляет запись в таблицу мониторинга очереди


sub get_settings :CACHE {
    my ($self) = @_;
    my $settings = $BM::BannersMaker::Tasks::TaskQueueSettings::types->{$self->type};

    $self->log("ERROR: unknown type " . $self->type) if !$settings;

    return $settings;
}

sub tasks_name :CACHE {
    my ($self) = @_;

    return $self->get_settings->{tasks_name};
}

sub worker_name :CACHE {
    my ($self) = @_;

    return $self->get_settings->{worker_name};
}

sub worker {
    my ($self) = @_;
    my $worker = $self->proj->cronlight_workers->{$self->worker_name};
    return $worker;
}

sub queue_table {
    my ($self) = @_;
    my $name = $self->get_settings->{queue_table_name};

    return undef if !$name;

    my $is_table_exist = $self->proj->bannerland_dbh->List_SQL("SELECT *  FROM information_schema.tables WHERE table_name = '$name' LIMIT 1;");
    die "db table $name not exists" unless (@$is_table_exist);

    return $self->proj->dbtable($name, undef, "bannerland_dbh");
}

sub monitor_table {
    my ($self) = @_;
    my $name = $self->get_settings->{monitor_table_name};

    return undef if !$name;

    my $is_table_exist = $self->proj->bannerland_dbh->List_SQL("SELECT *  FROM information_schema.tables WHERE table_name = '$name' LIMIT 1;");
    die "db table $name not exists" unless (@$is_table_exist);

    return $self->proj->dbtable($name, undef, 'bannerland_dbh');
}

# Возвращает уникальный идентификатор от task_json
# Однозначно задает путь для генерации
# Требуется проверять, если изменяется путь для генерации/способ генерации пути
sub _get_task_name_int {
    my ($self, $task_json) = @_;

    my $task_id = '';
    my $sep = '_';
    my @fields = ();

    for my $key (grep { $task_json->{$_} } qw/domain BannerIDs OrderID GroupExportIDs chunk_count chunk_remainder distribution_key lock_value/) {
        if (not ref $task_json->{$key}) {
            push @fields, $task_json->{$key};
        }
        elsif (ref($task_json->{$key}) eq 'ARRAY' and @{$task_json->{$key}}) {
            push @fields, (join $sep, @{$task_json->{$key}});
        }
    }

    $task_id = join($sep, @fields);
    return 0 unless $task_id;

    return md5int($task_id);
}

# добавить таск в очередь
sub add_task {
    my ($self, $task) = @_;
    my $json_text = JSON::XS->new->pretty(0)->encode($task);

    my $name = $self->_get_task_name_int($task);
    unless ($name) {
        $self->log("task name is empty because task_json doesn't have required fields while generating name for new task");
        return;
    }
    my $final_file = $self->tasks_dir . "/task_$name";

    if(-e $final_file) {
        $self->log("task $name already exists");
        return;
    }

    my $temp_file = $self->tasks_dir . "/tmp_$name";
    open F, ">$temp_file" or die($!);
    print F $json_text;
    close F;

    $self->do_sys_cmd("mv $temp_file $final_file");
}

# обновить очередь
sub refresh_tasks {
    my ($self) = @_;

    # сначала удаляем старые таски, DYNSMART-295
    $self->log("remove task_* files");
    $self->do_sys_cmd("rm " . $self->tasks_dir . "/task_* -v", no_die=>1);

    # выкачиваем очередь для хоста
    my $tasks = $self->proj->cronlight_tasks->{$self->tasks_name}->($self->proj, $self->source_table);

    # добавляем файлы тасок в директорию
    $self->log("new tasks: " . scalar(@$tasks));
    for my $task (@$tasks) {
        $self->add_task($task);
    }
}

# получить из очереди очередной таск
sub get_next_task {
    my ($self) = @_;
    my @files = $self->get_task_files;

    for my $task_file (@files) {
        my $task = $self->read_task_from_file($task_file);

        next if !$task;

        my $lock_name = $self->get_task_lock_name($task_file);
        if(!get_file_lock($lock_name)) {
            $self->log("$task_file is already being processed");
            next;
        }

        $self->log("processing task " . $task_file);

        my $new_file_name = $task_file;
        $new_file_name =~ s/\/task_([^\/]+)/\/work_$1/;
        $self->do_sys_cmd("mv $task_file $new_file_name");

        $self->{last_task} = $new_file_name;
        $self->{last_task_lock} = $lock_name;

        return $task;
    }

    return undef;
}

# вызвать обработчик для таска
sub invoke_worker {
    my ($self, $task) = @_;

    # делаем лок, если нужно
    my $lock_file = "";
    if($task->{lock_value}) {
        $lock_file = join("_", "task", $self->tasks_name, md5int($task->{lock_value}));

        $self->log("locking " . $task->{lock_value} . "...");
        if(!get_file_lock($lock_file)) {
            $self->log($task->{lock_value} . " is already being processed");
            return;
        } else {
            $self->log($task->{lock_value} . " is locked");
        }
    }

    $self->worker->($self->proj, $task);

    # отпускаем лок
    release_file_lock($lock_file) if $lock_file;
}

sub get_task_lock_name {
    my ($self, $task_file) = @_;
    return join "_", "task", $self->get_task_type, md5int($task_file);
}

sub read_task_from_file {
    my ($self, $task_file) = @_;
    my $handle;

    if(!open($handle, $task_file)) {
        return undef;
    }

    my $lines = join(" ", map{chomp; $_} <$handle>);
    my $json = JSON::XS->new->pretty(0);
    my $task;
    eval {
        $task = $json->decode($lines);
    };
    if($@){
        #Что-то неправильное с таской
        $self->proj->log("Bad task file '$task_file': $@");
        my $newfile = $task_file;
        $newfile =~ s/task_(\d+)$/badtask_$1/;
        my $cmd = "mv $task_file $newfile";
        $self->proj->log("move cmd: $cmd");
        system($cmd);
        #move( $task_file, $newfile);
        die("Bad task was moved to '$newfile'"); #Умираем, так как не можем отдать нормальный таск
    }
    close $handle;
    return $task;
}

# удалить из очереди последний таск, полученный из очереди
sub free_last_task {
    my ($self) = @_;

    return if !$self->{last_task};

    if(!unlink($self->{last_task})) {
        $self->log("ERROR: can't remove " . $self->{last_task});
    }

    release_file_lock($self->{last_task_lock});
    unlink $self->{last_task_lock};

    $self->log("/ processing task " . $self->{last_task});

    $self->{last_task} = undef;
}

# размер очереди
sub get_task_count {
    my ($self) = @_;
    my @files = $self->get_task_files;

    return scalar(@files);
}

# список файлов с тасками
sub get_task_files {
    my ($self) = @_;
    my @files = sort glob($self->tasks_dir . "/task_*");

    return @files;
}

# таски, которые находятся в обработке
sub get_work_files {
    my ($self) = @_;
    my @files = sort glob($self->tasks_dir . "/work_*");

    return @files;
}

sub is_task_running {
    my ($self, $work_file) = @_;
    my $proj = $self->proj;
    my $orig_task_file = $work_file;
    $orig_task_file =~ s/work_([^\/]+)$/task_$1/g;

    my $lock_name = $self->get_task_lock_name($orig_task_file);
    my $lock_file = _get_lockname($lock_name);

    $proj->log("work_file: '$work_file'");
    $proj->log("orig_task_file: '$orig_task_file'");
    $proj->log("lock_file: '$lock_file'");

    if(!(-e $lock_file)) {
        $proj->log("lock_file does not exist");
        return 0;
    }

    if(open(F, "lsof $lock_file |")) {
        my @pid_lines = grep{$_} map{chomp; $_} <F>;
        close F;

        $proj->log("lsof $lock_file");
        $proj->log("$_") for @pid_lines;
        $proj->log("/ lsof $lock_file");

        return 1 if @pid_lines;
    } else {
        $proj->log("can't lsof $lock_file");
    }

    return 0;
}

sub update_monitor_state {
    my ($self, $task, $state, %opts) = @_;
    my $tbl = $self->monitor_table;

    return if !$tbl;

    my $dmn = $task->domain;
    $dmn =~ s/^www.//;

    my $entry = {
        Time        => strftime("%F %T", localtime),
        Domain      => $dmn || "",
        Host        => get_curr_host(),
        TaskID      => $opts{TaskID} // $task->task_id,
        DebugInfo   => $opts{DebugInfo} || "",
        State       => $state,
    };

    my $error_mail = $self->get_settings->{error_mail};
    if($error_mail && ($state eq "Timeout" || $state eq "Fail")) {
        my $bad_thing = $state eq "Timeout" ? "Таймаут" : "Ошибка";
        if(!$self->proj->SendMail({
            from => 'no_reply@yandex-team.ru',
            body => "State: $state\nDomain: " . $entry->{Domain} . "\nTask: " . $entry->{TaskID} . "\nHost: " . $entry->{Host} . "\n",
            %$error_mail
        })) {
            $self->log("ERROR: SendMail failed");
        }
    }

    $tbl->Add($entry);
}

# директория, в которой создаются таски
sub tasks_dir :CACHE {
    my ($self) = @_;
    my $dir = $self->proj->temp_dir . "/tasks_" . $self->get_task_type;

    if(!(-d $dir)) {
        $self->do_sys_cmd("mkdir -p $dir");
    }

    return $dir;
}

sub get_task_type :CACHE {
    my ($self) = @_;

    return $self->{task_type} || $self->{type} || "noname";
}

1;
