package BM::Bootstrap;

use strict;
use warnings;

use File::Temp;
use Cwd qw/getcwd/;
use Data::Dumper;

use Utils::Common;
use Utils::CompileTime;
use Project;
use BM::Resources qw(get_resource);

require Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(get_bootstrap);

my $arcadia_root = $Utils::Common::options->{dirs}{arcadia};
my $root = $Utils::Common::options->{dirs}->{root};
my $target_root = $Utils::Common::options->{dirs}->{temp} . "/bootstrap";
my $target_store = "$target_root/store";
my $fake_archive_for_local_yt = "FAKE_ARCHIVE_FOR_LOCAL_YT";

my @gen_dicts_resources = qw(generated_dicts models marketdata_id_category_vendor_model marketdata_subphraser);

my %BOOTSTRAPS = (
    catalogia_publish => {
        'bin' => 1,
        'bin-dicts' => {
            'categorization'    => 1,
            'geo'               => 1,
            'normalize' => [qw/lemmer-test-debian-64-2014.01/],
        },
        'configs' => 1,
        'dicts' => 1,               # get all files from dicts/
        'gen-dicts' => {            # get all but listed files from gen-dicts/
            '' => 1,
            map { $_ => 0 } qw{
                forecast
                dict_harmonization_addition_heads
                dict_harmonization_addition_tails
            }
        },
        'scripts' => {
            '' => 1,
            map { $_ => 0 } qw{users mapreduce tests yt dyn-smart-banners bender}
        },
        '.' => {
            rsync_resources => [@gen_dicts_resources],
            arcadia_files => {
                'rt-research/bannerland/bin/avatars/avatars' => 'bin/avatars',
                'rt-research/monitoring/solomon/agent/prod_config' => 'configs/solomon_agent_prod_config',
                'rt-research/monitoring/atoms/calculate_atoms/calculate_atoms' => 'bin/calculate_atoms',
            },
        },
    },
    
    # данные, нужные для запуска скриптов бродматча баннеров
    bm_dicts => {
        'bin-dicts' => {
            '' => 0,
            'syns' => 0, # used only for dicts generation; not broadmatch
            'categorization' => 1,
            'geo' => 1,
            'normalize' => [qw/lemmer-test-debian-64-2014.01/],
        },
        'dicts' => 1,
        'gen-dicts' => {
            '' => 1,
            'models' => {
                '' => 0,
                'wide_models' => 1,
            },
            (map { $_ => 0 } qw(
                forecast
                gen-dicts
                dict_harmonization_addition_heads
                dict_harmonization_addition_tails
            )),
        },
        'work' => {
            '' => 0,
            'forecast/models' => 1,
            'ContextFiltering/models_clusters' => 1,
        },
        '.' => {
            rsync_resources => [@gen_dicts_resources],
        },
    },
    bm_scripts => {
        'scripts' => {
            lib => 1,
            cpan => 1,
        },
    },

    qloud_gen_dicts => {
        'gen-dicts' => { '' => 1 },
        '.' => {
            rsync_resources => [@gen_dicts_resources],
            target_name => "gen-dicts.tar.gz",
        },
    },
);


if (!-d $target_store) {
    xsystem("mkdir -p $target_store");
}

# copy: $src/$what -> $tgt/$to_what
sub _do_copy {
    my ($src, $tgt, $what, $to_what) = @_;

    die "$tgt/$to_what already exists" if -e "$tgt/$to_what";
    if (-d "$src/$what") {
        xsystem("mkdir -p $tgt");
        xsystem("ln -s $src/$what $tgt/$to_what 2> /dev/null || cp -r $src/$what $tgt/$to_what\n");
    } else {
        xsystem("mkdir -p $tgt");
        my $arg = $ENV{$fake_archive_for_local_yt} ? '-s' : '';
        xsystem("ln $arg $src/$what $tgt/$to_what 2> /dev/null || cp $src/$what $tgt/$to_what\n");
    }
}

sub _deep_copy {
    my ($source, $target, $what, $config, $root) = @_;

    if (!ref $config) {
        if ($config) {
            _do_copy($source, $target, $what, $what);
        }
        return;
    }
    if (ref $config eq 'ARRAY') {
        $config = { ('' => 0), map { $_ => 1 } @$config };
    }
    if (ref $config ne 'HASH') {
        die "Unknown config: " . Dumper $config;
    }

    my $settings = delete $config->{'.'} || {};
    my $default = delete $config->{''} || $settings->{default} || 0;
    if ($settings->{target}) {
        $target = "$root/$settings->{target}";
    }
    my $to_what = $settings->{rename} || $what;

    my @ks = keys %$config;
    for my $k (@ks) {
        if ($k =~ m{^([^/]+)/+(.+)}) {
            die "You reconfigured path $what / $1" if $config->{$1};
            $config->{$1} = { '' => $default, $2 => $config->{$k} };
            delete $config->{$k};
        }
    }
    @ks = undef;

    if (-d "$source/$what") {
        for my $f (glob "$source/$what/*") {
            $f =~ s{^$source/$what/}{};
            if (exists $config->{$f}) {
                _deep_copy("$source/$what", "$target/$to_what", $f, $config->{$f}, $root);
            } else {
                _do_copy("$source/$what", "$target/$to_what", "$f", "$f") if $default;
            }
        }
    } else {
        _do_copy($source, $target, "$what", "$to_what") if $default;
    }
}

sub get_bootstrap {
    my ($name, %par) = @_;

    my $config = $BOOTSTRAPS{$name} or die "Unknown bootstrap config: $name";
    my $target_tmp = File::Temp->newdir("temp_${name}_XXXX", DIR => $target_root);
    my $extension = $par{uncompressed} ? 'tar' : 'tgz';

    my $config_settings = delete $config->{'.'} || {};
    my $target_name = $config_settings->{target_name} || "$name.$extension";
    $target_name = "$target_store/$target_name"; # now - fullname

    if ($config_settings->{rsync_resources} and !$par{dont_get_rsync_resources}) {
        for my $res (@{$config_settings->{rsync_resources}}) {
            get_resource($res);
        }
    }
    _deep_copy($root, $target_tmp, '.', $config, $target_tmp);

    # хэш: src_path_in_arcadia  =>  dst_path_in_broadmatching
    if ($config_settings->{arcadia_files}) {
        while (my ($arc_path, $dst_path) = each %{$config_settings->{arcadia_files}}) {
            _do_copy($arcadia_root, $target_tmp, $arc_path, $dst_path) if !-f "$target_tmp/$dst_path";  # check because of symlink
        }
    }

    Utils::CompileTime::save_revision_to_file(bm_path => $target_tmp);
    
    my $dereference = $ENV{$fake_archive_for_local_yt} ? '' : '--dereference';
    my $z = $par{uncompressed} ? '' : 'z';

    my @filelist = map { my $fn = $_; $fn =~ s{^$target_tmp/?}{}; $fn } glob("$target_tmp/*");
    xsystem(join " && ",
        "cd $target_tmp",
        "echo '.' >> _mr_target_dir",
        "tar $dereference --exclude-vcs --ignore-failed-read -c${z}f $target_tmp/.tmp.$name.$extension _mr_target_dir @filelist",
        "mv $target_tmp/.tmp.$name.$extension $target_name"
    );

    return "$target_name";
}

sub describe_bootstrap {
    my ($name) = @_;
    return $BOOTSTRAPS{$name};
}

sub list_bootstraps {
    my ($name) = @_;
    return keys %BOOTSTRAPS;
}

{
    my $pkg = __PACKAGE__;
    my $pkg_s = $pkg;
    $pkg_s =~ s/.+::([^:]+)$/$1/;
    my $log_debug = $ENV{LOG_DEBUG} && ($ENV{LOG_DEBUG} eq '1' || $ENV{LOG_DEBUG} =~ /$pkg|$pkg_s/);

    sub xsystem {
        my $cmd = shift;
        warn "xsystem: $cmd\n" if $log_debug;

        $! = 0;
        system($cmd) and die("Failed [$cmd] @ " . getcwd() . ": $!: $?");
        warn $! if $!;
    }

    sub xqx {
        my $cmd = shift;
        warn "xqx: $cmd\n" if $log_debug;

        $! = 0;
        my $res = qx($cmd);
        die("Failed [$cmd] @ " . getcwd() . ": $!: $?") if $?;
        warn $! if $!;
        return $res;
    }

}

1;
