package BM::ContextSyns::ContextLinks;
use strict;
use utf8;
use Data::Dumper;

# базовый класс для работы с контекстными синонимами

use open ':utf8';

use base qw(ObjLib::ProjPart);

use Storable qw(dclone);
use Utils::Sys qw(
    uniq
    staticmap_value
);
use List::Util qw(max);

__PACKAGE__->mk_accessors(qw(
    source_phl
    links
    file
    dicts
    init_phl
    lang
    language
    dict_dir
    params
    categs_mapping
    parse_params
    search_params
    staticmap_files
));

# Модуль для работы со словарями контекстных синонимов
# основной метод: extend_phl - расширение фраз синонимами
# загрузка словарей производится при первом вызове
# вся работа происходит по snorm; в частности, словари при загрузке снормализуются
# (также см. ниже про точные словоформы)
#   
#
# Словари бывают двух типов:
# 1) обычные, формат:
# synonyms @ context_type: context
#   ИЛИ
# source => assocs @ context_type: context
# т.е. могут быть односторонние или двусторонние замены, можно задать контекст
# такие словари загружаются в память
# матчинг по точным словоформам в src не поддерживается!
#
# 2) быстрые, формат:
# src_word <TAB> assocs
# т.е. замены только односторонние, контекст не задаётся
# словари работают через staticmap
# поддерживается матчинг по точным словоформам


# Данные, хранящиеся в объекте (для обычных словарей):
# source_phl: список всех фраз-источников, которые заменяются на синонимы (для построения индекса)
# links: хэш { snorm_phr => \@links },
#   где @links - список троек [источник, ассоциации, контекст]
#   контекст: хэш { type => $type, .... }
# categs_mapping: хэш с маппингами категорий:
#   { data => { $id => { $cat => $gencat } }, id => { $name => $id } }
# в контекстах храним $id маппинга

my $cat_path_cache = {};

sub init {
    my $self = shift;

    $self->links( {} );

    my $lang = $self->{lang} // $self->proj->default_lang;  # язык задается явно в конфиге!
    $self->lang($lang);
    $self->language($self->proj->get_language($lang));

    $self->dict_dir( $self->proj->options->{dirs}{dicts}.'/csyns' );

    $self->params( $self->proj->options->{ContextSyns_params} );

    $self->categs_mapping({});
}

#
# Методы для расширения фраз
#

# сгенерировать фразы, полученные разваливанием слов на контекстные синонимы
# удалить повторы старого
# на входе:
#   $phl  -  список фраз
#   $context  -  контекст расширяемых фраз (хэш категорий баннера)
# доп. параметры  -  передаются в метод extend_phr
sub extend_phl {
    my $self = shift;
    my $phl = shift;
    my $context = shift;
    my %par = @_;

    my %seen = map { $_->snorm_phr => 1 } $phl->phrases;
    my @result;
    for my $phr ($phl->phrases) {
        my $ext_phl = $self->extend_phr($phr, $context, %par);
        $ext_phl = $ext_phl->lgrep(sub { !$seen{$_->snorm_phr}++ });
        push @result, @$ext_phl;
    }

    return $self->proj->phrase_list(\@result);
}

# разваливание отдельной фразы
# на входе:
#   $phr  -  фраза
#   $context  -  контекст 
# доп. параметры  -  передаются в метод parse_phr
sub extend_phr {
    my $self = shift;
    my $phr = shift;
    my $context = shift;
    my %par = @_;

    my %seen;
    my @result;
    my $repl_phl = $self->parse_phr($phr, context => $context, %par);
    my @words = $phr->words;
    for my $repl_phr ($repl_phl->phrases) {
        my %pos = map { $_ => 1 } @{$repl_phr->inf};
        my @tail_words = map { $words[$_] } grep { !$pos{$_} } 0 .. $#words;
        my $new_text = join(' ', $repl_phr->text, @tail_words);
        my $new_phr = $self->proj->phrase($new_text);
        next if $seen{$new_phr->snorm_phr}++;

        $new_phr->minf(dclone($repl_phr->minf));
        push @result, $new_phr;
    }

    return $self->proj->phrase_list(\@result);
}

# DEPRECATED
sub extend_phraselist {
    my $self = shift;
    my $context = shift;
    my $phl = shift;
    return $self->extend_phl($phl, $context, @_);
}

# выделить слова для контекстной замены во фразе
# обяз. параметры:
#   $phr  -  фраза
# доп. параметры:
#   context  -  контекст фразы
#   strict  -  учитывать порядок слов
#   ret_words - вместо словопозиций возвращать слова
#   no_check  -  не проверять контекст
#   no_replace -  не заменять слова, выдавать список заматченных исходных фраз
# на выходе: список фраз для замены, в $phr->inf список словопозиций исходных слов (соотв. $phr->words)
sub parse_phr {
    my $self = shift;
    my $phr  = shift;
    my %par  = (
        %{$self->parse_params // {}},
        @_,
    );;

    my $proj = $self->proj;
    $self->_load;

    my $repl1 = $self->_get_repl($phr, %par);
    my $repl2 = $self->_get_repl_fast($phr, %par);

    my @new_phr;
    my %match;
    for my $r (@$repl1, @$repl2) {
        my ($src_text, $assoc_text, $dict_id) = ($r->{src_text}, $r->{assoc_text}, $r->{dict_id});
        if ($par{no_replace}) {
            $match{$src_text} = $r->{context_check};
            next;
        }

        my ($word_pos, $context) = ($r->{word_pos}, $r->{context});
        my $new_phr = $proj->phrase($assoc_text);
        if ($par{ret_words}) {
            my @orig = $new_phr->words;
            $new_phr->inf([ map { $orig[$_] } sort { $a <=> $b } @$word_pos ]);
        } else {
            $new_phr->inf([ @$word_pos ]);
        } 

        $new_phr->{matcher_inf}{repl} = join("=>", $src_text, $assoc_text);
        #$new_phr->{matcher_inf}{src_phr} = $phr;
        #$new_phr->{matcher_inf}{src_phr_text} = $phr->text;
        $new_phr->{matcher_inf}{context} = $context->{type} if $context;
        $new_phr->{matcher_inf}{src_name} = $self->{dict_id2name}{$dict_id}   if $self->{store_dict_names} and $dict_id;
        #$self->log("repl: $src_text, $assoc_text, $dict_id, " . $new_phr->{matcher_inf}{src_name})  if $self->{store_dict_names};   # TODO;

        push @new_phr, $new_phr;
    }
    return \%match if $par{no_replace};

    return $proj->phrase_list(\@new_phr);
}

# получить список замен (по обычному словарю)
# на входе:
#   $phr  -  subj
#   context => subj
#   strict => 0|1  -  выкидываем фразы, которые не являются максимальными
# результат: список хэшей :
#   word_pos => список словопозиций, assoc_text => замена (сырая?),  плюс отладочная инфа:
#   src_text => src из списка линков (снормализована),
#   context => контекст из линка; dict_id => subj
sub _get_repl {
    my $self = shift;
    my $phr = shift;
    my %par = @_;

    return [] if !$self->source_phl->count;

    my $proj = $self->proj;
    my @result;
    my %search_par = (
        use_xs => 1,
        use_existing_phrases => 1,
        %{$self->search_params // {}},
    );
    my $subphl = $self->source_phl->search_subphrases_in_phrase($phr, %search_par);
    my %subsub;  # выкидываем фразы, которые не являются максимальными
    if ($par{strict}) {
        for my $subphr ($subphl->phrases) {
            my $subsubphl = $subphl->search_subphrases_in_phrase($subphr, %search_par);
            $subsub{$_->snorm_phr} = 1 for grep { $_->snorm_phr ne $subphr->snorm_phr } $subsubphl->phrases;
        }
    }

    my @replaces;
    for my $subphr ($subphl->phrases) {
        next if $subsub{$subphr->snorm_phr};
        my $word_pos = $self->_get_word_pos($subphr, $phr) or next;
        for my $link (@{$self->links->{$subphr->snorm_phr}}) {
            my ($src_text, $assocs_ref, $context) = @$link;

            my $context_check = $self->_check_context($context, $phr, $par{context});
            next if !$par{no_check} and !$context_check;
            next if $par{strict} and !$self->_check_strict($proj->phrase($src_text), $phr);

            if ($par{no_replace}) {
                push @replaces, {
                    src_text => $src_text,
                    context_check => $context_check,
                };
                next;
            }

            my ($assocs_str, $dict_id) = split /:/, $$assocs_ref;
            for my $assoc_text (split /,/, $assocs_str) {
                next if $src_text eq $assoc_text;  # не заменяем сами на себя
                push @replaces, {
                    src_text => $src_text,
                    assoc_text => $assoc_text,
                    context => $context,
                    word_pos => $word_pos,
                    dict_id => $dict_id,
                };
            }
        }
    }
    return \@replaces;
}

# получить список замен (по быстрому словарю)
# на входе:
#   $phr  -  subj
#   no_replace => не поддерживается
# результат: см. _get_repl
sub _get_repl_fast {
    my $self = shift;
    my $phr = shift;
    my %par = @_;

    return [] if $par{no_replace};  # not supported

    my $dicts = $self->staticmap_files;
    return [] if !$dicts or !@$dicts;

    my @w = $phr->words;
    my @sw = $phr->snormwords_with_stops;
    return [] if @w != @sw;  # smth strange

    my @src;  # список пар [ словопозиция, слово ]
    my @all;  # замены - пары [ словопозиция, замена ]
    my @main;  # главные замены - их будем комбинировать!
    my $bp = $self->bad_pairs;

    for my $i (0 .. $#sw) {
        # каждое слово берём в снормализованном виде, и в сыром виде с "!"
        push @src, [$i, $sw[$i]];
        push @src, [$i, "!$w[$i]"] if $w[$i] !~ /^\!/;
    }

    for my $data (@src) {
        my ($i, $w) = @$data;
        for my $dict (@$dicts) {
            my $assoc_str = staticmap_value($dict, $w) or next;
            my $aphl = $self->proj->phrase_list({ phrases_text => $assoc_str });

            # проверка на badpairs; пользуемся тем, что $w уже снормализовано
            $aphl = $aphl->lgrep(sub { my $s = $_->snorm_phr; !$bp->{$w}{$s} and !$bp->{$s}{$w} });
            next if !$aphl->count;  # всё зафильтровали

            my @r = map { [ $i, $_->text ] } @$aphl;
            push @all, @r;
            push @main, $r[0];
        }
    }

    # обычные замены: заменяем во фразе только одно из слово на любую ассоциацию
    my @repl;
    push @repl, map { { src_text => $sw[$_->[0]], assoc_text => $_->[1], word_pos => [ $_->[0] ] } } @all;

    if ($par{expand_main}) {
        # для "главных" замен берем также пары и тройки
        for (my $i=0; $i<@main; ++$i) {
            my $mi = $main[$i];
            for (my $j=$i+1; $j<@main; ++$j) {
                my $mj = $main[$j];
                push @repl, {
                    src_text => join(' ', @sw[$mi->[0], $mj->[0]]),
                    assoc_text => $mi->[1].' '.$mj->[1],
                    word_pos => [ $mi->[0], $mj->[0] ],
                };
                for (my $k=$j+1; $k<@main; ++$k) {
                    my $mk = $main[$k];
                    push @repl, {
                        src_text => join(' ', @sw[$mi->[0], $mj->[0], $mk->[0]]),
                        assoc_text => join(' ', $mi->[1], $mj->[1], $mk->[1]),
                        word_pos => [ $mi->[0], $mj->[0], $mk->[0] ],
                    };
                }
            }
        }
    }

    return \@repl;
}


# проверка контекста
# на входе:
#   $syn_context  -  контекст замены
#   $phr  -  фраза
#   $phr_context  -  контекст фразы (напр., хэш категорий соотв. баннера)
sub _check_context {
    my $self = shift;
    my ($syn_context, $phr, $phr_context) = @_;
    return 1 if !$syn_context;
    my $type = $syn_context->{type};

    if ($type =~ /categ/ and $syn_context->{cmap_id}) {
        my $cmap = $self->categs_mapping->{data}{$syn_context->{cmap_id}};
        $phr_context = { map { ($cmap->{$_} // $_) => 1 } keys %$phr_context };
    }
    if ($type eq 'word') {
        my @w = grep { $phr->snormwordshash->{$_} } @{$syn_context->{words}};
        return 0 if !@w;
        return { words => \@w };
    } elsif ($type eq 'minusword') {
        return 0 if grep { $phr->snormwordshash->{$_} } @{$syn_context->{words}};
        return 1;
    } elsif ($type eq 'categ') {
        my @ctg = grep { $phr_context->{$_} } @{$syn_context->{categs}}; 
        return 0 if !@ctg;
        return { categs => \@ctg };
    } elsif ($type eq 'nocateg') {
        return 0 if grep { $phr_context->{$_} } @{$syn_context->{categs}}; 
        return 1;
    } elsif ($type eq 'categ_subtree') {
        my $syn_ctgs = $syn_context->{categs_hash};
        for my $ctg (keys %{$phr_context}) {
            return 1 if grep { $syn_ctgs->{$_} } $ctg, $self->_cache_cat_path($ctg);
        }
        return 0;
    }
}

# слова подфразы идут подряд (стоп-слова между не разрешены), в том же порядке
sub _check_strict {
    my $self = shift;
    my $subphr = shift;
    my $phr = shift;

    my $sub_pos = $self->_get_word_pos($subphr, $phr)
        or return 0;
    return 0 if grep { $sub_pos->[$_] != $sub_pos->[0] + $_ } 1 .. $#$sub_pos;
    return 1;
}

# список словопозиций для слов подфразы в фразе (матчинг по snorm, позиции соотв. $phr->words)
# для каждого слова - первое его появление
# если хотя бы одно слово не найдено - возвращает undef
sub _get_word_pos {
    my $self = shift;
    my $subphr = shift;
    my $phr = shift;

    my $language = $phr->language;
    my @sw = $phr->snormwords_with_stops;

    my @orig = $phr->words;
    my %first_pos = map { $sw[$_] => $_ } reverse(0 .. $#sw);

    my @sub_sw = $subphr->snormwords;
    my @sub_pos = @first_pos{@sub_sw};

    return undef if grep { !defined } @sub_pos;
    return \@sub_pos;
}



#
# Методы для загрузки словарей
#

# проверить, что данные загружены
sub _load {
    my $self = shift;
    return if $self->{'source_phl_loaded'} and not $self->{reload_always};
    $self->load;
    my $phl = $self->language->phrase_list([ sort keys %{$self->links} ]);
    $self->source_phl( $phl->snorm_pack_list );
    $self->{'source_phl_loaded'} = 1;
}

# Загружаем из файлов ($self->file - может быть один файл или ссылка на массив файлов;  $self->dicts - ссылка на массив словарей, лежащих в директории ContextSyns_params{dir_dicts})
#     или/и из phraselist ($self->init_phl - phraselist)
sub load {
    my $self = shift;
    $self->clear_links;
    
    my $file = shift // $self->file;
    my @files = ref($file) ? @$file : ($file || ());
    push @files, map { $self->params->{dir_dicts} . "/" . $_ } @{$self->dicts || []};

    $self->load_from_file($_) for grep {$_} @files;

    if ($self->init_phl) {
        $self->load_from_phl($_) for ($self->init_phl);
    }
    $self->log("loading all data: done!");
}

sub load_from_file {
    my $self = shift;
    my $file = shift;

    my %add_par;
    if ($self->{cmap_file}) {
        my $cmap_id = $self->add_categs_mapping($self->{cmap_file});
        $add_par{cmap_id} = $cmap_id;
    }

    if ($self->{store_dict_names}) {
        my $dict_name = dict_filename2dict_name($file);
        my @ids = values %{$self->{dict_name2id} // {}};
        $self->{dict_name2id}{$dict_name} //= max(@ids, 0) + 1;
        $self->{dict_id2name}{ $self->{dict_name2id}{$dict_name} } = $dict_name;
        #$self->proj->dd([ dict_id2name => $self->{dict_id2name}, dict_name2id => $self->{dict_name2id} ]);
        $add_par{dict_id} = $self->{dict_name2id}{$dict_name};
    }

    $self->log("loading context synonyms from file '$file' ...");
    open my $fh, '<', $file
        or $self->log("ERROR: can't open '$file': $!") and return;
    while(<$fh>) {
        chomp;
        if (/^\s*CATEGS_MAPPING\s*=\s*([.\w-]+)\s*$/) {
            my $name = $1;
            $self->log("using categs_mapping: $name");
            my $file = $self->dict_dir.'/'.$name;
            $add_par{cmap_id} = $self->add_categs_mapping($file);
            next;
        }
        $self->add_from_line($_, %add_par)  or
            $self->log("ERROR: wrong format in '$file' line $.");
    }
    close $fh;
    $self->log("loading done!");

    return 1;
}

sub load_from_phl {
    my $self = shift;
    my $phl = shift;

    $self->log("loading context synonyms from phl ...");
    for my $phr ($phl->phrases) {
        my $data = $self->_parse_line($phr->text)
            or do { $self->log("ERROR: wrong format in phl: ".$phr->text); next };
        $self->add_links($data->{sources_phl}, $data->{assocs_phl}, $data->{context});
    }
    $self->log("loading done!");

    return 1;
}

# загрузка строки словаря в стандартном формате:
# source => assocs @ context
#   ИЛИ
# synonyms @ context
# (в этом случае source = assocs = synonyms)
#
# контекст задается в виде type:value
# типы контекстов:
#   * - пустой контекст; применимо всегда (пустой контекст вообще можно не указывать)
#   categ - наличие в баннере категории (хотя бы одной из списка), список через "/"
#   categ_subtree - наличие в баннере категорий или их подкатегорий, список через "/"
#   nocateg - отсутствие в баннере категорий, список через "/"
#   word - наличие во фразе слова (хотя бы одного из списка), список через пробел или запятую
#   minusword - отсутствие во фразе слов из списка
#
# source, assocs - списки фраз, разделенных запятой (не пробелом, т.к. во фразе может быть более одного слова!)
#
# смысл: любую фразу из source можно заменить на любую фразу из assocs внутри контекста context
sub add_from_line {
    my $self = shift;
    my $line = shift;
    my %par  = @_;

    my $data = $self->_parse_line($line)
        or do { $self->log("ERROR: wrong format"); return };

    return 1 if $data->{type} eq 'comment';

    my $context = $data->{context};
    if ($context and $context->{type} =~ /categ/ and $par{cmap_id}) {
        $context->{cmap_id} = $par{cmap_id};
    }

    $self->add_links($data->{sources_phl}, $data->{assocs_phl}, $context, dict_id => $par{dict_id});
    return 1;
}

sub _parse_line {
    my $self = shift;
    my $line = shift;

    $line =~ s/\s*#.*//;
    return { type => 'comment' } if $line =~ /^\s*$/;

    my ($syns, $context_str) = split /\s*\@\s*/, $line, 2;
    my ($source, $assocs) = split /\s*=>\s*/, $syns, 2;
    $assocs = $source if !defined $assocs;

    my $context = $self->_parse_context($context_str);
    return if !defined $context;
    return +{
        type => 'data',
        sources_phl => $self->_parse_phrs($source),
        assocs_phl => $self->_parse_phrs($assocs),
        context => $context,
    };
}

sub _parse_context {
    my $self = shift;
    my $context_str = shift;

    my ($type, $value);

    if (!defined $context_str or $context_str =~ /^\s*\*/) {
        return 0;  # пустой контекст
    } elsif ($context_str =~ /^\s*(categ|categ_subtree|nocateg|word|minusword)\s*:(.*)/) {
        ($type, $value) = ($1, $2);
        $value =~ s/^\s*//;
        $value =~ s/\s*$//;
    } else {
        return undef;
    }

    my $context = { type => $type };
    if ($type =~ /categ/) {
        my @ctg = split '/', $value;
        if ($self->lang ne 'ru') {
            @ctg = map { $self->language->category_to_ru($_) } @ctg;
        }
        if ($type eq 'categ_subtree') {
            $context->{categs_hash} = { map { $_ => 1 } @ctg };
        } else {
            $context->{categs} = \@ctg;
        }
    } elsif ($type eq 'word') {
        $context->{words} = [ $self->language->phrase($value)->snormwords ];
    }
    return $context;
}

sub clear_links {
    my $self = shift;

    $self->links( {} );
    $self->source_phl('');
}

# Возвращает хэш плохих замен: { src => { assoc1 => 1, assoc2 => 1, ... } }
sub bad_pairs {
    my $self = shift;
    unless ($self->{bad_pairs}) {
        $self->log("Loading bad_pairs ...");
        my $bad_pairs = {};
        for my $file (@{ $self->params->{dicts_badpairs} // [] }) {
            $self->log("bad_pairs dict: $file");
            open my $fh_in, '<', $file or do {
                my $msg = "Could not load dicts_badpairs: $file";
                $self->log("ERROR: $msg");
                die $msg;
            };
            while (<$fh_in>) {
                chomp;
                s/#.*//;
                next unless /[^\s]/;
                my @phs = split /\s*,\s*/;
                my @snorms = uniq( map { $self->proj->phrase($_)->snorm_phr } @phs );
                for my $sn (@snorms) {
                    $bad_pairs->{$sn}{$_} = 1   for grep { $_ ne $sn } @snorms;
                }
            };
            close $fh_in;
        };
        $self->{bad_pairs} = $bad_pairs;
        $self->log("Loading bad_pairs done");
    }
    return $self->{bad_pairs};
}

# old version, deprecated
sub add_links_text {
    my $self = shift;
    my $sources_arr = shift;
    my $assocs_arr = shift;

    my $sources_phl = $self->language->phrase_list($sources_arr);
    my $assocs_phl = $self->language->phrase_list($assocs_arr);

    return $self->add_links($sources_phl, $assocs_phl, @_);
}

sub add_links {
    my $self = shift;
    my ($sources_phl, $assocs_phl, $context, %par) = @_;
    return if !$assocs_phl->count or !$sources_phl->count;

    my $assoc_phrs2assoc_str = sub {
        my $assocs_phrs = shift;
        return join(',', map { $_->text } @$assocs_phrs) . ":" . ($par{dict_id} // '');
    };

    my $assocs_str_all = $assoc_phrs2assoc_str->( [$assocs_phl->phrases] );  # храним в строке для экономии памяти
    my $bad_pairs = $self->bad_pairs || {};
    for my $src_phr ($sources_phl->phrases) {
        my $key = $src_phr->snorm_phr;

        my $assocs_str;
        if ($bad_pairs->{$key}) {
            my @assocs_phrases_curr = grep { not $bad_pairs->{$key}{ $_->snorm_phr } }  $assocs_phl->phrases;     # TODO
            $assocs_str = $assoc_phrs2assoc_str->([ grep { $_->snorm_phr ne $key }  @assocs_phrases_curr ]);
            next if not $assocs_str;
        }

        my @data = ($src_phr->text,  $assocs_str ? \$assocs_str : \$assocs_str_all);
        push @data, $context if $context;
        push @{ $self->links->{$key} }, \@data;
    }
    $self->source_phl('');  # нужно будет обновить source_phl
}

# загрузить мапинг категорий, вернуть (внутренний) id мапинга
sub add_categs_mapping {
    my $self = shift;
    my $file = shift;

    my $cmap = $self->categs_mapping;
    if ($cmap->{id}{$file}) {
        return $cmap->{id}{$file};
    }

    my $map_id = 1 + keys %{$cmap->{data}};

    $self->log("loading categs_mapping from file `$file' ...");
    $cmap->{data}{$map_id} = Utils::Sys::load_map($file);
    $cmap->{id}{$file} = $map_id;
    return $map_id;
}

# не должно быть повтора snorm-слов; слова фразы нормализуем, но не снормализуем и не сортируем
# экранируем стоп-слова
# оставляем не более N фраз
sub _parse_phrs {
    my $self = shift;
    my $phrs_text = shift;

    my @arr;
    my $raw_phl = $self->language->phrase_list({phrases_inf_text => $phrs_text});
    if ($phrs_text =~ /\[/) {
        $raw_phl = $raw_phl->atoms2list_phl;
    }

    my @res;
    my %seen;
    for my $phr (@$raw_phl) {
        next unless $phr->text;
        $phr = $phr->set_pluses_before_stops;
        my @snorm = $phr->snormwords;
        next unless @snorm;
        next if @snorm != uniq(@snorm);
        push @res, $phr if !$seen{$phr->snorm_phr}++;
    }

    my $max_phrs = 500;
    @res = splice(@res, 0, $max_phrs);

    return $self->language->phrase_list(\@res);
}

sub _cache_cat_path {
    my $self = shift;
    my $ctg = shift;
    if (!$cat_path_cache->{$ctg}) {
        my @path = $self->proj->categs_tree->get_cat_path($ctg);
        $cat_path_cache->{$ctg} = \@path;
    }
    return @{$cat_path_cache->{$ctg}};
}

# Из имени файла словаря или пути к файлу словаря получить название словаря, которое будет записываться в matcherlog
sub dict_filename2dict_name {
    my $s = shift;
    $s =~ s!.*/!!;
    $s =~ s/^dict_csyns_/dict_/; # Оставляем "dict_", чтобы отличать, например, словарь dict_csyns_medium_exp от источника medium_exp.
    return $s;
}

1;
