package BM::DetectCharset;

use std;
use base qw(ObjLib::ProjPart);

no warnings 'utf8';

use utf8;

use Encode;
use Encode qw/_utf8_on _utf8_off find_encoding/;
use PerlIO::encoding;
#libencode-detect-perl
use Encode::Detect::Detector;
use Utils::Sys qw(get_encoder);
use Utils::Funcs qw(content_type_is_xml content_type_is_html);
use File::Copy;

my $optns = {
    pair_size => 2,
    min_diff => 1.5,
    charsets => [qw(CP1251 KOI8-R ISO-8859-5 CP866)],
};

sub init {
    my $self = shift;
    #статистика пар для поиска русского языка
    $self->{sym_table} = {};
#    my $symbfile = '/opt/broadmatching/dicts/Russian_chst'; # имя файла таблицы символов
    my $symbfile = $self->{symbfile}; # имя файла таблицы символов

    open(my $fh, "<:utf8", $symbfile);
    while (<$fh>) {
        $self->{sym_table}{$1} = $2 if /(\w+)\s(\d+)/;
    }
    close($fh);
    $self->{octets_table} = {};
    for my $charset ( @{$optns->{charsets}} ) {
        my $encoder = get_encoder($charset);
        for my $bigram ( keys %{$self->{sym_table}} ) {
            my $octets = $encoder->encode( $bigram );
            $self->{octets_table}{$charset}{$octets} = $self->{sym_table}{$bigram};
        }
    }
    my $good_bigrams = {};
    for my $charset (@{$optns->{charsets}}) {
        my $octets_table = $self->{octets_table}{$charset};
        $good_bigrams->{$_} = 1 foreach keys %$octets_table;
    }
    $self->{bigrams_regex} = join('|', sort keys %$good_bigrams);

}

sub detect_text {
    my ( $self, $text ) = @_;

    $text =~ tr/\r\n//;
    my $allres = $self->detect_old($text);
    return $self->process_old_results($allres);
}

sub detect_old {
    my ($self, $text) = @_;
    my $result = {};
    Encode::_utf8_off($text);
    my $bigram_stat = {};
    my $pair_size = $optns->{pair_size};
    my $good_bigrams = {};
    my $bigrams_regex = $self->{bigrams_regex};
    $bigram_stat->{$_}++ foreach $text =~ m{ (?= ($bigrams_regex) ) }xmsg;
    for my $charset (@{$optns->{charsets}}) {
        my $wght = 0;
        my $octets_table = $self->{octets_table}{$charset};
        for my $bigram (keys %$bigram_stat) {
            $wght += ($octets_table->{ $bigram } // 0)*$bigram_stat->{$bigram};
        }
        $result->{$charset} = $wght;

    }
    return $result;
};

sub process_old_results {
    my ($self, $allres) = @_;
    my @res = sort { $allres->{$b} <=> $allres->{$a} } grep {$allres->{$_}} keys %$allres;

    #кодировка не определилась
    return -1 if( !defined $res[0] || $allres->{$res[0]} < 1 );
    #определяемая кодировка должна иметь вес в полтора раза больше, чем следующая за ней
    return 0 if defined $res[1] && $allres->{$res[0]} < $allres->{$res[1]} * $optns->{min_diff};

    return $res[0];

}

#Определяем кодировку
sub detect_charset {
    my ( $self, $text ) = @_;
    my $charset = '';
    eval {
        $charset = Encode::Detect::Detector::detect($text);
#        print "===$charset===\n";
        # фигурный костыль, когда скорее всего неправильно определяется x-mac-cyrillic, пробуем еще раз через старый метод
        if ( $charset && ($charset eq 'x-mac-cyrillic')){
            my $try_charset_another = $self->detect_text($text);
            $charset = 'windows-1251' if $try_charset_another eq 'CP1251';
        }
        $charset = 'windows-1251' if defined($charset) and $charset eq 'windows-1252'; #Бывают ложные срабатывания, опорная технология
#        return $charset;
    };
    return $charset;
}

sub text2utf8 {
    my ( $self, $text ) = @_;

    eval {
        my $charset = $self->detect_charset($text);
        if($charset){ #Убираем лишние варнинги
            Encode::_utf8_off($text);
            $text = get_encoder($charset)->decode( $text );
        }
    };
    return $text;
}

sub get_signature_charset {
    my ($self, $text, $content_type) = @_;
    # определение кодировок в XML, HTML и JSON "позаимствовано" из HTTP::Message
    my $signature_charset;
    if (content_type_is_xml($content_type)) {
        # http://www.w3.org/TR/2006/REC-xml-20060816/#sec-guessing
        # XML entity not accompanied by external encoding information and not
        # in UTF-8 or UTF-16 encoding must begin with an XML encoding declaration,
        # in which the first characters must be '<?xml'
        for ($text) {
            $signature_charset //= "UTF-32BE" if /^\x00\x00\x00</;
            $signature_charset //= "UTF-32LE" if /^<\x00\x00\x00/;
            $signature_charset //= "UTF-16BE" if /^(?:\x00\s)*\x00</;
            $signature_charset //= "UTF-16LE" if /^(?:\s\x00)*<\x00/;
            if (/^\s*(<\?xml[^\x00]*?\?>)/) {
                if ($1 =~ /\sencoding\s*=\s*(["'])(.*?)\1/) {
                    my $enc = $2;
                    $enc =~ s/^\s+//; $enc =~ s/\s+\z//;
                    $signature_charset //= $enc if $enc;
                }
            }
        }
    }
    elsif (content_type_is_html($content_type)) {
        # look for <META charset="..."> or <META content="...">
        # http://dev.w3.org/html5/spec/Overview.html#determining-the-character-encoding
        require IO::HTML;
        # Use relaxed search to match previous versions of HTTP::Message:
        my $encoding = IO::HTML::find_charset_in($text, { encoding    => 1,
                                                           need_pragma => 0 });
        $signature_charset //= $encoding->mime_name if $encoding;
    }
    elsif (($content_type) && ($content_type eq "application/json")) {
        for ($text) {
            # RFC 4627, ch 3
            $signature_charset //= "UTF-32BE" if /^\x00\x00\x00./s;
            $signature_charset //= "UTF-32LE" if /^.\x00\x00\x00/s;
            $signature_charset //= "UTF-16BE" if /^\x00.\x00./s;
            $signature_charset //= "UTF-16LE" if /^.\x00.\x00/s;
            $signature_charset //= "UTF-8";
        }
    }
    return $signature_charset;
}

sub detect_file_charset_candidates {
    my ($self, $filename, $content_type) = @_;

    my $detector = new Encode::Detect::Detector;
    my $old_result = {};

    my ($signature_charset, $old_charset, $new_charset);

    open my $fh, '<', $filename or die $!;
    binmode $fh;
    my $first = 1;
    while (1) {
        my $buffer;
        my $bytes_read = read $fh, $buffer, 100000;
        die $! if !defined $bytes_read;
        last if !$bytes_read;

        if ($first) {
            $first = 0;
            $signature_charset = $self->get_signature_charset($buffer, $content_type);
        }

        $detector->handle($buffer);
        my $temp_detector_result = $detector->getresult;
        if ( !defined $temp_detector_result || $temp_detector_result =~ /^x-/ || $temp_detector_result eq 'windows-1252' ) {
            # это практически всегда ошибка, поэтому запускаем дополнительно свой детектор по биграммам
            my $buffer_old_result = $self->detect_old($buffer);
            $old_result->{$_} += $buffer_old_result->{$_} foreach keys %{$buffer_old_result};
        }
    }
    close $fh;

    $detector->eof;
    $new_charset = $detector->getresult;
    $old_charset = $self->process_old_results($old_result);
    $new_charset //= $old_charset;
    if ( defined($new_charset) && (($new_charset eq 'x-mac-cyrillic' && $old_charset eq 'CP1251') || ($new_charset eq 'windows-1252')) ) {
        # характерные ошибки определения кодировок
        $new_charset = 'windows-1251';
    }

    my @candidates = ($signature_charset, $new_charset);
    @candidates = grep {/[a-zA-Z]/} grep {$_} @candidates;

    return @candidates;
}

sub file_charsets2utf8 {
    my ($self, $input_file, $output_file, $candidates) = @_;
    my $success = 0;
    my @candidates = grep { $_ && find_encoding($_) } @$candidates;
    my %candidates_bytes_read = ();
    for my $charset ( @candidates ) {
        #сначала пробуем декодирование в строгом режиме, которое будет отваливаться при ошибках
        ($success, $candidates_bytes_read{$charset} ) = _try_file_charset2utf8($charset, $input_file, $output_file, {strict => 1});
        last if $success;
    }
    if ( !$success && scalar(@candidates)) {
        #сломалось на середине. Попробуем нестрогое декодирование с самой успешной кодировкой
        #если никакая не успешная, берем первую
        my $charset = $candidates[0];
        if ( scalar(keys %candidates_bytes_read) ) {
            ($charset) = sort { $candidates_bytes_read{$b} <=> $candidates_bytes_read{$a}} keys %candidates_bytes_read;
        }
        ($success) = _try_file_charset2utf8($charset, $input_file, $output_file, {strict => 0});
    }
    if ( !$success ) {
        #все совсем плохо, просто копируем файл
        File::Copy::copy($input_file, $output_file);
    }
}

sub _try_file_charset2utf8 {
    my ($charset, $input_file, $output_file, $params) = @_;
    $params //= {};
    my $success = 0;
    my $total_bytes_read = 0;
    no warnings 'utf8';
    eval {
        # Модуль Encode кидает ворнинги вне зависимости от pragma warnings, а WARN: utf8 могут забить логи миллионами записей,
        # поэтому здесь приходится приглушить все ворнинги разом. см: DYNSMART-1001
        local $SIG{__WARN__} = sub {};
        local $PerlIO::encoding::fallback = $params->{strict} ? Encode::FB_CROAK|Encode::STOP_AT_PARTIAL : $PerlIO::encoding::fallback;
        open my $in_fh, "<:encoding($charset)", $input_file or die $!;
        open my $out_fh, ">:encoding(utf-8)", $output_file or die $!;
        my $first = 1;
        while (1) {
            my $buffer;
            #читаем маленькими блоками, запоминая, сколько для каждой кодировки мы прочитали.
            #если посередине файла будет ошибка, значит, кодировка в целом угадана правильно, но внутри файла есть битые символы
            #в таком случае возьмем кодировку, которая сломалась позже всех, и прогоним с ней нестрогое декодирование
            my $bytes_read = read $in_fh, $buffer, 1000;
            $total_bytes_read += $bytes_read // 0;
            die $! if !defined $bytes_read;
            last if !$bytes_read;
            if ($first) {
                #если это xml, надо поменять кодировку в начале файла
                $buffer =~ s/encoding\s*=\s*(["'])(.*?)\1/encoding="UTF-8"/m;
                $first = 0;
            }
            print $out_fh $buffer;
        }
        close $in_fh;
        close $out_fh;
    };
    $success = 1 unless $@;
    return ($success, $total_bytes_read);
}

1;
