package BM::Filter;

# системные модели
use Data::Dumper;
use Utils::Words;
use Utils::Sys qw(url_decode_safe);

use URI::_idna;

# внутренние модули
use base qw(ObjLib::ProjPart);

use strict;

no warnings 'utf8';

sub ext2bm {
    my ( $bs_conditions ) = @_;
    my $raw_result = {};
    if ( !ref($bs_conditions) or ref($bs_conditions) ne 'ARRAY' ) {
        print STDERR "ERROR: bad bs_conditions: " . Dumper($bs_conditions);
        return {};
    }
    for my $element ( @$bs_conditions ) {

        # условие на пустое поле
        next if ( exists( $element->{type}) && $element->{type} eq 'any' );

        # специальные фильтры
        my $key   = $element->{type};
        my $not   = ( $element->{kind} =~ /^(not_exact|not_match|not_equals)$/ ) ? " NOT" : "";
        my $type  = $element->{type} // "";
        my $kind  = $element->{kind} // "";
        my $value = $element->{value} // "";
        my $condition = "";

        if ( $kind =~ /exact/ && $type =~ /^domain$/i ) {
            push @{$raw_result->{"url" . $not . " domainlike"}}, $value;
            next;
        }

        if ( $kind =~ /exact/ && $type =~ /^url$/i ) {
            #Для некоторых кириллических доменов фильтры указывают в punycode 
            $value =~ s/(^https?:\/\/)([^\/\?\:]+\.[^\/\?\:\.]+)/$1.URI::_idna::decode($2)/e; 
            push @{$raw_result->{"url" . $not . " ilike"}}, $value;
            next;
        }

# ЕСЛИ 
#  { 'kind' => 'excat', value => 'Квартира Истра', type => 'content' },
# ТО
#  { "url contains" => "Квартира Истра" }
        if ( $kind =~ /exact/ && $type =~ /^content$/i ) {
            push @{$raw_result->{"url" . $not . " contains"}}, $value;
            next;
        }

# ЕСЛИ 
#  { 'kind' => 'excat', value => 'Квартира Истра', type => 'title' },
# ТО
#  { "url titlecontains" => "Квартира Истра" }
        if ( $kind =~ /exact/ && $type =~ /^title$/i ) {
            push @{$raw_result->{"url" . $not . " titlecontains"}}, $value;
            next;
        }

# /особенные фильтры

        # валидность остальных фильтров
        if ( !exists($element->{kind}) || !exists($element->{value}) || !exists($element->{type}) ) {
            print STDERR "ERROR: bad element in bs_conditions:" . Dumper($element);
            next;
        };
        if ( $element->{kind} !~ /^(exact|not_exact|match|not_match|equals|not_equals)$/ ) {
            print STDERR "ERROR: bad kind in element:" . Dumper($element);
            next;
        };
        if ( $element->{type} !~ /^(title|content|domain|URL|URL_prodlist|url|url_prodlist)$/ ) {
            print STDERR "ERROR: bad type in element:" . Dumper($element);
            next;
        };
        if ( $element->{value} =~ /^\s*$/ ) {
            print STDERR "ERROR: bad value in element:" . Dumper($element);
            next;
        };

        $condition = " ilike"     if $element->{kind} =~ /^(exact|not_exact)$/;
        $condition = " normlike" if $element->{kind} =~ /^(match|not_match)$/;
        push @{$raw_result->{$key . $not . $condition}}, $element->{value};
    }

    # ситуации вида [a, b, [c, d]] в пределах одного ключа должны обрабатываться как a && b && ( c || d )
    # Поэтому для каждого value делаем отдельный ключ, размножая $raw_result
    my $result = {};
    for my $key ( keys %$raw_result ) {
        my $index = 0;
        for my $value ( @{$raw_result->{$key}} ) {
            my $result_key = $key;
            if ($index) {
                $result_key .= " $index";
            }
            push @{$result->{$result_key}}, ( ref($value) eq 'ARRAY' ? @$value : $value );
            $index++;
        }
    }
    # В итоге получаем хэш из условий, обрабатываемых через AND, а value внутри условия обрабатываются через OR

    return $result;
}

# фильтрация контента типа TSKV из формата ksamelyuk
# вход scalar, выход хэш строк
# средняя скорость 23K rps
sub filter_tskv_to_hash_strings {
    my ( $self, $content ) = @_;
    my @array = ();
    my %result = ();

    # случайный ключ, в котором хранится исходная строка
    my $random_key = time . int(rand()*1000);

    my $linescount = 0;
    for ( split/\n/, $content ) {
        chomp; 

        # tskv2hashedarray
        my %vars = map { /^(\w+)\=(.+)/ } split /\t/;
        $vars{$random_key} = $_; # саму строчку тоже в хэш
        push @array, \%vars;

        # фильтрация
        if ( ++$linescount % 10_000 == 0 ) {
            my $filtered_list = $self->filter(\@array);
            if ( scalar(@$filtered_list) > 0 ) {
                map { $result{$_->{$random_key} } = 1 } @$filtered_list;
            }
            @array = ();
        }

    }
    # last pack
    {
        my $filtered_list = $self->filter(\@array);
        if ( scalar(@$filtered_list) > 0 ) {
            map { $result{$_->{$random_key} } = 1 } @$filtered_list;
        }
        @array = ();
    }
    return \%result;
}

# grep objects that satisfy filter
sub filter {
    my ( $self, $all_list ) = @_;

    my @current_list = @$all_list;
    my $chunk_size = 2000;
    my @result;
    while ( @current_list ) {
        my @pack = splice(@current_list, 0, $chunk_size);
        my $filt = $self->filter_by_simple_conditions(\@pack);
        $filt = $self->filter_by_extended_conditions($filt);
        $filt = $self->filter_by_zora_conditions($filt);
        push @result, @$filt;
    }
    return \@result;
}

# сгенерированный код будет eval-иться в соответствующих функциях
# код может обращаться к локальным переменным:
#   $self - оттуда берётся proj
#   $list - в нём хранится список объектов для фильтрации
#   $filtered_list - в нём хранится текущий список объектов, удовлетворяющих условиям

# наложение простых фильтров
sub filter_by_simple_conditions {
    my ( $self, $list ) = @_;
    return $list if !@$list or !$self->{code};

    my $filtered_list = [];
    local $SIG{'__WARN__'} =  sub {};  # варнинги в этом месте подавляем, иначе они быстро сожрут диск  
    eval $self->{code};

    return [] if $@;
    return $filtered_list;
}

# наложение сложных фильтров
sub filter_by_extended_conditions {
    my ($self, $list) = @_;
    return $list if !@$list or !$self->{extended_code};

    my $filtered_list = $list;  # extended_code ожидает на входе $filtered_list

    eval $self->{extended_code};
    return [] if $@;
    return $filtered_list;
}

# наложение контентных фильтров
sub filter_by_zora_conditions {
    my ($self, $list) = @_;
    return $list if !@$list or !$self->has_zora_conditions;

    # для оптимизации проверки контентных фильтров нужно делать сразу для множества фильтров,
    # поэтому они реализованы в Filters.pm
    # создаём фейковый filters и входной список в нужном формате
    my $filters = $self->self_filters;
    my $id = (keys %{$filters->{filters}})[0];
    my @list_with_fids = map { [ $_, [$id] ] } @$list;
    my $filt = $filters->filter_by_zora_conditions(\@list_with_fids);
    return [ map { $_->[0] } grep { scalar(@{$_->[1]}) } @$filt ];
}

# из одного фильтра - $self
sub self_filters :CACHE {
    my $self = shift;
    my $filter_id = 1;
    return $self->proj->filters({$filter_id => $self->{filter}});
}

sub init {
    my ( $self )  = @_;
    $self->init_filter;
}

sub init_filter {
    my ( $self )  = @_;
    my $filter = $self->{filter};

    # kostyl science. Превращаем массив массивов просто в массив
    for my $filter_key ( keys %$filter ) {
        my $filter_value = $filter->{$filter_key};
        next if ref($filter_value) ne 'ARRAY';
     
        my $new_filter_value = [];
        for my $item ( @$filter_value ) {
            push @$new_filter_value, ref($item) eq 'ARRAY' ? @$item : $item;
        }
        $filter->{$filter_key} = $new_filter_value;
    }
    # /kostyl science

    # пройдем по ключам фильтра и во-первых выпарсим точные значения
    my @equal_conditions   = $self->_get_equal_conditions( $filter );   # NOT included
    my @inarray_conditions = $self->_get_inarray_conditions( $filter ); # NOT included
    my @compare_conditions = $self->_get_compare_conditions( $filter ); # NOT included
    my @exists_conditions  = $self->_get_exists_conditions( $filter );  # NOT included


    $self->{conditions} = [ @equal_conditions, @inarray_conditions, @compare_conditions, @exists_conditions ];
    my $all_conditions_string = join(' && ', 1, @{ $self->{conditions} } );
    $self->{code} = '$filtered_list = [ grep { local $_ = _preprocess_elem($_);' . $all_conditions_string . ' } @$list ]';

#   # дополнительный кусок кода, связанный с нормализацией итд
    $self->{extended_code} = $self->_get_extended_code( $filter );

    # вычисляем что мы будем делать, если есть опция "contains" 
    $self->get_zora_conditions;
    return $self;
}

# отбираем условия про контент страниц и переводим в удобный для применения формат
# пример:
# bm format: {
#   'url NOT contains' => ['нет в наличии','Сообщить о поступлении','последняя цена'],
#   'url NOT titlecontains' => ['официальный дилер','официального дилера']
# }
# return format: (
#   {'field' => 'url','type' => 'contains','negation' => 1,'values' => ['нет в наличии','сообщить о поступлении','последняя цена']}
#   {'field' => 'url','type' => 'titlecontains','negation' => 1,'values' => ['официальный дилер','официального дилера']}
# )
sub get_zora_conditions :CACHE {
    my $self = shift;
    my @zora_conds;
    while (my ($full_key, $value) = each %{$self->{filter}}) {
        my $not = 0;
        if ($full_key =~ /\s+NOT\s+/i) {
            $full_key =~ s/\s+NOT\s+/\ /i;
            $not = 1;
        }

        my ($field, $type) = split /\ +/, $full_key;
        next if !$type;
        next if $type !~ /^(contains|titlecontains)$/;

        my @values;
        if (!ref($value)) {
            @values = ($value);
        } elsif (ref($value) eq 'ARRAY') {
            @values = @$value;
        } else {
            print STDERR "ERROR: bad value in filter: " . Dumper($self->{filter});
            return ();
        }

        # lowercase!
        @values = map { lc } @values;

        push @zora_conds, {field => $field, values => \@values, type => $type, negation => $not};
    }
    return @zora_conds;
}

sub has_zora_conditions {
    my $self = shift;
    my @conds = $self->get_zora_conditions;
    return @conds ? 1 : 0;
}

sub remove_name_normlike {
    my ( $self ) = @_;
    delete $self->{filter}->{'name normlike'};
    $self->init_filter;
}

# фильтр, который использует внутренние возможности BM, категоризацию, нормализацию итд
sub _get_extended_code {
    my ( $self, $filter ) = @_;
    my @result = ();
    my $flags_key = {};
    for my $full_key ( keys %$filter ) {
        my $value = $filter->{$full_key};
        # NOT CASE
        my $not_key = "";
        if ( $full_key =~ /\s+NOT\s+/i ) {
            $full_key =~ s/\s+NOT\s+/\ /i;
	    $not_key = '!';
        }

        my ( $key, $compare_symbol ) = split /\ +/, $full_key;

        next if ( !defined($compare_symbol) || !$compare_symbol );
        next if ( $compare_symbol !~ /^(normlike|domainlike)$/ );

        # скачали код стрницы через zora - для каждого ключа $key мы скачиваем это ОДИН раз
        my $contain_code = "";

        # инициализация хэша нормализованных слов
        if ( !exists($flags_key->{$key}) ) {
            $contain_code .= q~ my $norm_words_%KEY% = {}; ~;
        }
        $flags_key->{$key}++;

        # список нормализованных слов

        # хэш нормальзованных слов. "ключ" => "нормализованные слова"
        my $hname = $compare_symbol . '%KEY%' . int( rand() * 10_000);

        $contain_code .= q~ 
	     $norm_words_%KEY% = {
                    map { $_->{ %KEY% } =>  {
                            map { $_ => 1 }
                            @{ Utils::Words::lmr->norm_words( $_->{ %KEY% }  , 'ru') } 
                        
                    } } @$filtered_list
             };
             my $~ . $hname . q~ = {};
        ~;

        # список значений
        my @values = ref($value) eq 'ARRAY' ? @$value : ( $value );
        for my $value_item ( @values ) {
            $value_item =~ s/\'/\\'/g;
            $value_item = lc($value_item);
            if( $compare_symbol eq 'normlike' ) {   # все нормализованные слова значения есть в нормализованных словах контента
                $contain_code .= q~
		    map {
                        my $val = 1;
                        my $item = $_;
                        for ( @{Utils::Words::lmr->norm_words('%VALUE%', 'ru')} ) {
                            if ( !exists( $norm_words_%KEY%->{$item->{%KEY%}}{$_} ) ) {
                                $val = 0;
                                last;
                            }
                        };
                        if ( ~ . $not_key . q~$val ) {
                            $~ . $hname . q~->{ $item->{%KEY%} } = 1;
                        }
                    } @$filtered_list;~ . "\n";
            } elsif ( $compare_symbol eq 'domainlike' ) {
                $contain_code .= q~ map { 
                        my $item = $_;
                        if ( ~ . $not_key . q~ (index($self->proj->site($item->{%KEY%})->domain,'%VALUE%') > -1) ) {
                            $~ . $hname . q~->{ $item->{%KEY%} } = 1;
                        } 
                    } @$filtered_list; ~ . "\n";
            }
	    $contain_code =~ s/\%VALUE\%/$value_item/g;
        }

        $contain_code .= q~ $filtered_list = [ grep { exists($~ . $hname . q~->{ $_->{%KEY%} } ) } @$filtered_list ]; ~;

        $contain_code =~ s/\%KEY\%/$key/g;
        push @result, $contain_code;
    }
    return join("\n",@result);
}

sub _gen_compare_filter_code {
    my ($key, $compare_symbol, $not_key, $value) = @_;
    $value = _preprocess_value($key, $value);
    $value =~ s/\'/\\'/g;
    if ( $compare_symbol eq 'like' or $compare_symbol eq 'ilike' ) {
        return $not_key . '(index(lc($_->{\'' . $key . '\'}),' . "'". lc($value) ."'". ') > -1)';
    } elsif ( $compare_symbol eq '<->' ) {
            my ( $value_from, $value_to ) = split /\-/, $value;
            if ( !defined($value_from) || !$value_from && $value_from ne '0' && defined($value_to) ) {
                return $not_key . '($_->{\'' . $key . '\'} <= ' . ($value_to+0) ." )";
            } elsif ( !defined($value_to) || !$value_to && $value_to ne '0' && defined($value_from) ) {
                return $not_key . '($_->{\'' . $key . '\'} >= ' . ($value_from+0) ." )";
            } else {
                $value_from //= 0; $value_to //= 0;
                return $not_key . '($_->{\'' . $key . '\'} >= ' . ($value_from+0) . ' && $_->{\'' . $key . '\'} <= ' . ($value_to+0) ." )";
            }
    } elsif ( $compare_symbol =~ /^\s*(>|<|==|>=|<=)\s*$/ ) {
        return  $not_key . '($_->{\'' . $key . '\'} ' . $compare_symbol . " " . $value . ")";
    } elsif ( $compare_symbol =~ /^\s*(lt|le|gt|ge|eq|ne)\s*$/ ) {
        return $not_key . '(lc($_->{\'' . $key . '\'}) ' . $compare_symbol . " '" . lc($value) . "')";
    }
    return '';
};

sub _get_compare_conditions {
    my ( $self, $filter ) = @_;
    my @result = ();
    for my $full_key ( keys %$filter ) {

        my $value = $filter->{$full_key};       

        # NOT CASE
        my $not_key = "";
        if ( $full_key =~ /\s+NOT\s+/i ) {
            $full_key =~ s/\s+NOT\s+/\ /i;
	    $not_key = '!';
        }

        my ( $key, $compare_symbol ) = split /\ +/, $full_key;
        next if ( !defined($compare_symbol) || !$compare_symbol );
        if ( !ref($value) ) { # это означает строку в условии типа 'value1 >=' => 'ddd';
            my $code = _gen_compare_filter_code($key, $compare_symbol, $not_key, $value);
            push @result, $code if $code;
        } elsif ( ref($value) eq 'ARRAY' ) {
            my @small_result = ();
            for my $value_item ( @$value ) {
                my $code = _gen_compare_filter_code($key, $compare_symbol, $not_key, $value_item);
                push @small_result, $code if $code;
            }
            # array means OR, если не отрицание
            if ( @small_result ) {
                push @result, "(" . join(($not_key ? ' && ' :  ' || '), @small_result ) . ")"
            }
        }
    }
    return @result;
}

# получает точные соответствия хэша
# массив [ value1 eq '5' ]
sub _get_equal_conditions {
    my ( $self, $filter ) = @_;
    my @result = ();
    for my $key ( keys %$filter ) {
        my $value = $filter->{$key};
        my $not_key = "";
        if ( $key =~ /\s+NOT\s*$/i ) {
            $key =~ s/\s+NOT\s*//gi;
            $not_key = '!';
        }
        next if $key =~ /\ /;
        if ( !ref($value) ) { # это означает строку в условии типа value1 => 'ddd'
            $value =~ s/\'/\\'/g;
            push @result, $not_key . '(lc($_->{\'' . $key . '\'}) eq ' . "'" . lc($value // '') . "')";
        }
    }
    return @result;
}

# проверка на наличие поля
sub _get_exists_conditions {
    my ( $self, $filter ) = @_;
    my @result = ();
    for my $key ( keys %$filter ) {
        next if ( $key !~ /\ exists$/ );
        my $value = $filter->{$key};
        my $not_key = "";
        if ( $key =~ /\s+NOT\s+exists$/i ) {
            $not_key = '!';
        }
        my ( $name_key ) = ( $key =~ /^([^\ ]+)(\ |$)/ );
        next if !$name_key;
        if ( !ref($value) ) { # это означает строку в условии типа value1 => 'ddd'
            $value =~ s/\'/\\'/g;
            push @result, $not_key . 'exists($_->{\'' . $name_key . '\'})';
        }
    }
    return @result;
}

# условия на вхождение элемента в массив. каждый новый массив инкрементируется в объекте
sub _get_inarray_conditions {
    my ( $self, $filter ) = @_;
    my @result = ();
    for my $key ( keys %$filter ) {
        my $value = $filter->{$key};

        # не обрабатываем вложенные массивы
        my $flag_multiple = 0;
        if ( ref($value) eq 'ARRAY' ) {
            for my $item ( @$value ) {
                if ( ref($item) ) {
                    $flag_multiple = 1;
                    last;
                }
            }
        }
        next if $flag_multiple;
        # /не обрабатываем вложенные массивы

        my $not_key = "";
        if ( $key =~ /\s+NOT\s*$/i ) {
            $key =~ s/\s+NOT\s*//gi;
            $not_key = '!';
        }
        next if $key =~ /\ /;
        if ( ref($value) eq 'ARRAY' ) { # это может ознчать сложный массив типа => ['a','b',['c','d']], то есть A && B && (C || D)
            # отдельный вариант, когда массив из одного элемента - это просто условие типа equal:
            if ( scalar(@$value) == 1 ) {
                push @result, $not_key . '($_->{\'' . $key . '\'} eq ' . "'" . ($value->[0] // '') . "')";
            } else { # в условии несколко элементов, поэтому делаем через присутствие в хэше

                # формируем этот хэш
                if ( !exists( $self->{hashcount} ) ) {
                    $self->{hashcount} = 0;
                    $self->{arraycond} = [];
                } else {
                    $self->{hashcount}++
                };

                # заполняем хэш из массива $value
                $self->{arraycond}[ $self->{hashcount} ] = { map { lc($_) => 1 } @$value };

                # пишем соответствующее условие в строку result:
                push @result, $not_key . '(exists($self->{arraycond}[' . $self->{hashcount} . ']{lc($_->{\'' . $key . '\'})}))';
            }
        }
    }
    return @result;
}

sub _preprocess_elem {
    my %elem = %$_;
    $elem{$_} = _preprocess_value($_, $elem{$_}) foreach keys %elem;
    return \%elem;
}

sub _preprocess_value {
    my ($key, $value) = @_;
    if ( $key =~ /^url/i ) {
        $value = url_decode_safe($value);
    }
    return $value;
}

1;
