package BM::Geo;
use strict;
use utf8;

use base qw(ObjLib::ProjPart);

my %regtree;
my %dict2id;

my %geoname2id;

#
# Работа с нашими регионами (словари .region *)
#

sub get_regtree {
    my $self = shift;
    $self->init_regions_tree;
    return \%regtree;
}

sub dict2reginf {
    my $self = shift;
    my $dict = shift;
    $self->init_regions_tree;
    my $id = $dict2id{$dict} // return;
    return $regtree{$id};
}
sub reg2reginf {
    my $self = shift;
    my $reg  = shift;
    my $dict = ".region $reg";
    $self->init_regions_tree;
    my $id = $dict2id{$dict} // return;
    return $regtree{$id};
}

sub is_region {
    my $self = shift;
    my $reg  = shift;
    my $reginf = $self->reg2reginf($reg) // return 0;
    return $reginf->{is_region};
}

sub reg2geoid {
    my $self = shift;
    my $reg = shift;

    if (!keys %geoname2id) {
        require geobaselite;
        while (my ($id, $inf) = each %geobaselite::Region) {
            $geoname2id{$inf->{name}} = $id;
        }
    }

    return $geoname2id{$reg};
}

sub is_world_region {
    my $self = shift;
    my $reg  = shift;
    my $reginf = $self->reg2reginf($reg) // return 0;
    return $reginf->{is_world_region};
}


sub init_regions_tree {
    my $self = shift;
    return if keys %regtree;

    $self->log('init_regions_tree ...');
    my $ctree = $self->proj->categs_tree;
    my $root_dict = '.regions';
    my @dicts = ($root_dict);
    my @all_dicts = @dicts;

    my %parent;
    while (@dicts) {
        my $dict = shift @dicts;
        my @children = $ctree->get_minicateg_children($dict);
        $parent{$_} = $dict for @children;
        push @dicts, @children;
        push @all_dicts, @children;
    }

    # создаём ноды
    my $max_id = 0;
    for my $dict (@all_dicts) {
        my $id = ++$max_id;
        my $name;
        if ($dict =~ /^\.region\s+(.*)$/) {
            $name = $1;
        } elsif ($dict =~ /\.regions_world/) {
            $name = 'ROOT_WORLD';
        } elsif ($dict =~ /\.regions_rus/) {
            $name = 'ROOT_RUS';
        } elsif ($dict eq '.regions') {
            $name = 'ROOT';
        } else {
            warn "Wrong name format in regions tree: `$dict'";
        }

        $name //= $dict;
        my $node = {
            dict => $dict,
            name => $name,
            is_region => ($dict =~ /^\.regions/) ? 0 : 1,
        };
        $regtree{$id} = $node;
        $dict2id{$dict} = $id;
    }

    # родители
    while (my ($id, $node) = each %regtree) {
        my $parent_dict = $parent{$node->{dict}};
        $node->{parent} = $dict2id{$parent_dict} if $parent_dict;
    }

    # путь наверх
    while (my ($id, $node) = each %regtree) {
        my @path;
        my $curr = $regtree{$id}{parent};
        while ($curr) {
            push @path, $curr;
            $curr = $regtree{$curr}{parent}; 
        }
        $node->{path} = [ reverse @path ];
    }

    # определение is_world - по пути наверх
    while (my ($id, $node) = each %regtree) {
        my $path = $node->{path};
        $node->{is_world_region} = (grep { $regtree{$_}{name} eq 'ROOT_WORLD' } @$path) ? 1 : 0;
    }


    # поддерево
    my %sub;
    while (my ($id, $node) = each %regtree) {
        my @parents = @{$node->{path} || []};
        push @parents, $id;  # для единообразия -- subtree включает саму вершину
        $sub{$_}{$id} = 1 for @parents;
    }
    while (my ($id, $subh) = each %sub) {
        $regtree{$id}{subtree} = [ sort keys %$subh ];
    }

    $self->log('init_regions_tree done!');
}


#
# Переход от регионов геобазы к нашим регионам
#

# по списку плюс и минус-регионов (id из геобазы) получить список наших регионов
sub geoids2regions {
    my $self = shift;
    my $ids = shift;

    $self->init_regions_tree;
    my (%plus, %minus);
    for my $id (@$ids) {
        my @regs = $self->_geoid2regions(abs($id));
        if ($id > 0) {
            $plus{$_} = 1 for @regs;
        } else {
            $minus{$_} = 1 for @regs;
        }
    }

    # если нет минус-регионов, то отдаём список плюс
    return sort keys %plus if !keys %minus;

    # нужно брать поддеревья (только листья!) 
    # TODO через API!
    my @plus_subtree = grep { @{$regtree{$_}{subtree}} == 1 } map { @{$regtree{$_}{subtree}} } map { $dict2id{".region $_"} } keys %plus;
    my @minus_subtree = grep { @{$regtree{$_}{subtree}} == 1 } map { @{$regtree{$_}{subtree}} } map { $dict2id{".region $_"} } keys %minus;
    my %minus_subtree = map { $_ => 1 } @minus_subtree;
    return map { $regtree{$_}{name} } grep { !$minus_subtree{$_} } sort @plus_subtree;
}


sub _geoid2regions {
    my $self = shift;
    my $geo_id = shift;

    require geobaselite;
    my $Region = \%geobaselite::Region;

    my $reginf = $Region->{$geo_id} or return ();

    # пробуем сам регион
    if ($self->is_region($reginf->{name})) {
        return ($reginf->{name});
    }

    # пробуем пойти вниз
    my @regions;
    my @children = @{$reginf->{chld} || []};
    while (@children) {
        my $id = shift @children;
        my $reg = $Region->{$id};
        my $name = $reg->{name};
        if ($self->is_region($name)) {
            push @regions, $name;
        } else {
            push @children, @{$reg->{chld} || []};
        }
    }
    return @regions if @regions;

    # пробуем пойти наверх
    for my $id (reverse @{$reginf->{path} || []}) {
        my $name = $Region->{$id}{name};
        return ($name) if $self->is_region($name);
    }

    return ();
}

sub geobase_norm_dict {
    my ( $self ) = @_;
    if ( exists($self->{geobase_norm_dict}) ) { # словарь: нормализованный регион => [массив id из geobase];
        return $self->{geobase_norm_dict};
    };
    require geobaselite;
    $self->log("loading geonorm");
    my $Region = \%geobaselite::Region;

    my $result = {};
    for my $region_id ( keys %$Region ) {
        my $normalized_region_name = $self->proj->phrase( $Region->{$region_id}->{name} )->norm_phr();
        if ( $normalized_region_name ) {
            push @{$result->{$normalized_region_name}}, $region_id;
        }
    }
    $self->{geobase_norm_dict} = $result;
    $self->log("/ loading geonorm");
    return $self->{geobase_norm_dict};
}

sub _get_tr_region_ids_re : GLOBALCACHE {
    my ($self) = @_;
    require geobaselite;
    my $Region = \%geobaselite::Region;
    # id Турции
    my $id_tr = (grep { ($Region->{$_}{name} // '') eq 'Турция' } keys %$Region)[0]
        or die "Could not get Turkey region id!";
    my %tr_region_ids = ($id_tr => 1);
    for my $id (keys %$Region) {
        my $path = $Region->{$id}{path};
        if (grep { $_ == $id_tr } @$path) {
            $tr_region_ids{$id} = 1;
        }
    }
    my $tr_region_ids_str = join("|", map {'\b'.$_.'\b'} sort keys %tr_region_ids);
    my $tr_region_ids_re = qr/$tr_region_ids_str/;
    #print "tr_region_ids_str: $tr_region_ids_str\n", "tr_region_ids_re:  $tr_region_ids_re\n";
    return $tr_region_ids_str;
}

# На входе - строка id регионов, через запятую
# На выходе - 1, если есть регион, среди родителей которого есть Турция;  иначе - 0
sub has_tr_region_id {
    my ($self, $ids_str) = @_;
    return 1  if $ids_str =~ $self->_get_tr_region_ids_re();
    return 0;
}

1;
