package BM::GraphiteClient;

use strict;

use IO::Socket;
use Data::Dumper;

use Utils::Sys qw[ print_err ];
use Utils::Hosts qw[ get_curr_host get_readable_hostname get_host_role get_host_datacenter ];

sub new {
    _new_instance(@_);
}

sub _new_instance {
    my ($class, %options) = @_;

    my $socket = IO::Socket::INET->new("$options{host}:$options{port}") 
        or die "couldn't connect to $options{host}:$options{port}: $@\n";
    my $host = get_curr_host;
    $host =~ s/[^0-9a-zA-Z-]/_/g;
    die "ERROR!!! couldn't define host!!!\n" unless $host;

    return bless {
        socket          => $socket, 
        host            => $host,
        prefix          => ($options{log_prefix} // 'one_min').'.'.$host,
        period          => $options{period} // 60,
        debug_mode      => $options{debug_mode},
    }, $class;
};

sub send_value {
    my ($self, $path_array, $value, %prm) = @_;
    my $time = $prm{time} // time;
    my $hostname_fqdn = $prm{host} // get_curr_host();
    my $hostname_readable = $prm{host} // get_readable_hostname();
    my $period = $prm{period} // 60;
    my $is_virtual_host = $prm{is_virtual_host} // 0;
    
    my @errors;
    if (grep { ref($_) } (@$path_array, $value)) {
        push @errors, "Bad input data types";
    }
    my $path_str = join(".", @$path_array);
    if (!$path_str or $path_str =~ m/[^a-zA-Z0-9_\-\.]/ or $path_str =~ m/(^\.|\.\.|\.$)/) {
        push @errors, "Bad path_str";
    }
    if (not defined $value or $value !~ m/^[0-9\.]+$/) {
        push @errors, "Bad value";
    }
    if (@errors) {
        print_err("ERROR: GraphiteClient::send_value: @errors: " . join(" ", map { "{" . ($_ // 'UNDEF') . "}" } $path_str, $value));
        return;
    }

    my @prefix_arr = ("one_min");
    unless ($is_virtual_host) {
        my $host_role = get_host_role($hostname_fqdn) || 'unknown';
        my $datacenter = get_host_datacenter($hostname_fqdn) || 'unknown';
        push @prefix_arr, "one_min.bm.$host_role.$datacenter";
    }

    for my $prefix (@prefix_arr) {
        my $host_str = $hostname_readable;
        $host_str =~ s/[^0-9a-zA-Z-]/_/g;

        my $out = join(" ",
            join('.', $prefix, $host_str, $path_str),
            $value,
            int($time / $self->{period}) * $self->{period},
        );

        if ($self->{debug_mode}) {
            print_err("GraphiteClient::send_value $out (not really, debug_mode is on)");
        } else {
            print_err("GraphiteClient::send_value $out");
            my $socket = $self->{socket};
            print $socket $out."\n"  or die "Could not write to socket!";
        }        
    }
    return 1;
};

1
