package BM::Matching::Graph;
use strict;

use utf8;
use open ':utf8';

use std;
use base qw(ObjLib::ProjPart);

use List::Util qw(sum min max);
use Utils::Sys;

# Это на самом деле симпграф, а не граф!

########################################################
# Интерфейс
########################################################

#   load_graph                 загрузка графа в память
#   get_history                по списку (баннер, фраза) восстановить связки в симпграфе
#   get_phl_phrases            по списку фраз (PhraseList) возвращает привязки (PhraseList)
#   get_bnr_phrases            по баннеру возвращает привязки к его родным фразам

########################################################
# Инициализация
########################################################

sub init {
    my ($self) = @_;
}

sub load_graph {
    my $self = shift;
    my %par  = @_;
    return if $self->{graph} and !$par{force};

    my $proj = $self->proj;
    my $h = {};
    $self->log("load graph ".$self->{'source'});
    open(my $F, "<", $self->{'source'})
        or $self->log("ERROR: can't load graph '$self->{source}'")
        and return undef;
    while(defined( my $s = <$F> )){
        my @a = split("\t", $s);
        unless ($self->{dont_use_synonyms}) {
            $h->{$proj->phrase($a[0])->snorm_phr} = $a[1];
        } else {
            $h->{$proj->phrase($a[0])->norm_phr} = $a[1];
        }
    }
    $self->{'graph'} = $h;
    $self->log("/ load graph ".$self->{'source'});
    return 1;
}

sub get_all_phrases {
    my $self = shift;
    my $phl  = shift;

    my $proj = $self->proj;
    $self->load_graph;
    my $graph = $self->{graph};

    my @result;
    for my $src ($phl->phrases) {
        my $key = $src->snorm_phr;
        next if !$graph->{$key};
        my $assoc_phl = $proj->phrase_list({ phrases_inf_text => $graph->{$key} });
        $_->{'matcher_inf'}{'srcphr'} = $key for $assoc_phl->phrases;
        push @result, [ $key, $assoc_phl ];
    }
    return \@result;
}


# $self->get_phl_phrases($phl, %par)
# параметры:
#   smpg_top  -  размер TOP-а внутри симпграфа (по умолчанию - берем всё)
#   smpg_top_ratio  -  доля отбираемого TOP-а (по умолчанию - берем всё)
#   (если заданы оба параметра, берется минимум)
#
#   smpg_fml  -  формула для ранжирования (по умолчанию: default)
#       default  -  основная формула продакшена
#       sum  -  простая сумма весов
#       avg  -  среднее арифметическое весов
#       max  -  максимум
#       none  -  не ранжировать вообще! все веса пишутся через ';'
#   verbose => в srcphr пишем все src
#   smpg_min_weight  -  минимальный вес в симпграфе (до применения функций)
# по списку фраз получает привязки от сервера,
# дописывает к каждой фразе:
#   $phr->{matcher_inf}{srcphr} - текст исходной фразы (если несколько, берется фраза с макс. весом)
#   $phr->{matcher_inf}{smpg_weight} - суммарный вес связей (по формуле)
# дописывает в итоговый PhraseList:
#   $phl->{matcher_inf}{smpg_uncut}  -  кол-во привязок до обрезания
sub get_phl_phrases {
    my $self = shift;
    my $phl  = shift;
    my %par  = (smpg_fml => 'default', @_);

    my $proj = $self->proj;
    my $empty = $proj->phrase_list({phrases_arr => []});

    my %old = map { $_->snorm_phr => 1 } $phl->phrases;  # не расширяем исходными фразами
    my $src_phl = $phl;
    $src_phl = $src_phl->snorm_pack_list;
    $src_phl->cache_is_good_phrase unless $ENV{MR_BROADMATCH};
    $src_phl = $src_phl->good_phrases_list;

    my (%wei, %phr);  # все веса и объект для фразы-ассоциации
    my (%maxwei, %src);  # фраза-источник с макс. весом -- для отладки
    my %fullinf;
    my $assoc_arr = $self->get_all_phrases($src_phl);
    for my $h (@$assoc_arr) {
        my ($src, $assoc_phl) = @$h;
        for my $phr ($assoc_phl->phrases) {
            next if $par{check_phrase} && !$par{check_phrase}->($phr);
            my $snorm = $phr->snorm_phr;
            next if $old{$snorm};
            my $w = $phr->inf->[0];

            if (defined $par{smpg_min_weight}) {
                next if $w < $par{smpg_min_weight};
            }

            push @{$wei{$snorm}}, $w;
            $phr{$snorm} //= $phr;

            if ($par{verbose}) {
                push @{$fullinf{$snorm}}, [ $src, $w ];
            }

            if (!defined $maxwei{$snorm} or $w > $maxwei{$snorm}) {
                $src{$snorm} = $src;
                $maxwei{$snorm} = $w;
            }
        }
    }

    # выставим srcphr
    for my $text (sort keys %phr) {
        my $phr = $phr{$text};
        if ($par{verbose}) {
            $phr->{matcher_inf}{srcphr} = join(';', map { $_->[0].':'.$_->[1] } sort { $b->[1] <=> $a->[1] } @{$fullinf{$text}});
        } else {
            $phr->minf->{srcphr} = $src{$text};
        }
    }


    if ($par{smpg_fml} eq 'none') {
        my @res;
        for my $text (sort keys %phr) {
            my $phr = $phr{$text};
            $phr->{matcher_inf}{smpg_weight} = join(';', @{$wei{$text}});
            push @res, $phr;
        }
        my $mphl = $proj->phrase_list({phrases_list => \@res});
        $mphl->{matcher_inf}{smpg_uncut} = @res;
        return $mphl;
    }

    # ранжирование
    my $score_func;
    my $fml = $par{smpg_fml};
    if ($fml eq 'sum') {
        $score_func = sub { sum(@_) };
    } elsif ($fml eq 'avg') {
        $score_func = sub { sum(@_) / @_ };
    } elsif ($fml eq 'rand') {
        $score_func = sub { rand() };
    } elsif ($fml eq 'max') {
        $score_func = sub { max(@_) };
    } else {
        # основная формула!
        $score_func = sub { sum(@_) + 0.3 * @_ };
    }

    my %score;
    for my $snorm (keys %wei) {
        $score{$snorm} = $score_func->(@{$wei{$snorm}});
    }

    # специальные формулы, переранжируем
    if ($fml eq 'md5') {
        for my $snorm (keys %wei) {
            $score{$snorm} = (md5int($snorm) % 1000) / 1000;
        }
    } elsif ($fml eq 'avg_index') {
        my %avg;
        for my $snorm (keys %wei) {
            my @w = @{$wei{$snorm}};
            $avg{$snorm} = sum(@w) / @w;
        }
        my @snorm = sort { $avg{$b} <=> $avg{$a} } keys %avg;
        my %index = map { $snorm[$_] => ($_ + 1) } 0 .. $#snorm;
        for my $snorm (keys %wei) {
            $score{$snorm} = (@snorm + 21) / ($index{$snorm} + 20);
        }
    } elsif ($fml eq 'special_schepin') {
        my %avg;
        for my $snorm (keys %wei) {
            my @w = @{$wei{$snorm}};
            $avg{$snorm} = sum(@w) / @w;
        }
        my @snorm = sort { $avg{$a} <=> $avg{$b} } keys %avg;
        my %index = map { $snorm[$_] => ($_ + 1) } 0 .. $#snorm;
        # фильтруем в зависимости от симпграфа
        my $grname = $self->{simpgrname};
        my %thr = (
            Banners => 0.2,
            QueryLog => 0.25,
            HitLog => 0.25,
            Bindings => 0.75,
            Tags => 0.9,
            Slotstat => 0.0,
        );
        my $thr = $thr{$grname} // 0.5;
        for my $snorm (keys %wei) {
            $score{$snorm} = $index{$snorm} / (@snorm + 1) - $thr;
        }
    }

    # определяем размер TOP-а
    my $uncut_top = keys %score;
    my $top = $uncut_top;
    $top = min($top, $par{smpg_top}) if defined $par{smpg_top};
    $top = min($top, int($uncut_top * $par{smpg_top_ratio})) if defined $par{smpg_top_ratio};

    my @texts = sort { $score{$b} <=> $score{$a} } keys %score;

    if ($fml eq 'special_schepin') {
        @texts = grep { $score{$_} >= 0 } @texts;
        $top = min($top, scalar @texts);  # !!
    }

    $#texts = $top - 1;
    my @res;
    for my $text (@texts) {
        my $phr = $phr{$text};
        $phr->{matcher_inf}{smpg_weight} = $score{$text};
        push @res, $phr;
    }
    my $mphl = $proj->phrase_list({phrases_list => \@res});
    $mphl->{matcher_inf}{smpg_uncut} = $uncut_top;
    return $mphl;
}


1;
