package BM::Monitor::Logs;
use strict;
use utf8;

# модуль для мониторинга ошибок в логах

use open ':utf8';

use Data::Dumper;
use List::Util qw(min max maxstr minstr);
use IO::Socket;
use LWP::UserAgent;
use POSIX qw(strftime);
use Time::Local qw(timelocal);
use File::Find;
use Time::Piece;

use Utils::Sys qw(
    file_bytes
    mtime
    dir_files
    time_unix2db
    time_log2unix
    load_json
    save_json
    print_err
    handle_errors
    uniq
);
use Utils::Common;
use Utils::Hosts qw(get_hosts get_host_role get_host_info get_curr_host);
use Utils::DB qw(get_tbl_update_time);
use BM::Monitor::Utils;
use BM::GraphiteClient;
use BM::SolomonClient;
use Encode;

use base qw(Exporter);
our @EXPORT = qw(
    %logs_errors_file
    get_logs_errors
    send_logs_errors_diff
    save_logs_errors
    print_logs_errors
    send_logs_errors_metrics
);

our @EXPORT_OK = qw(
    error2str
    get_project
    get_subscriptions
    convert_keys_format
);

my $options = $Utils::Common::options;
my $mon_dir = $options->{dirs}{monitor_info};

our %logs_errors_file = (
    current => "$mon_dir/logs_errors-current.json",   # Файл со списком ошибок в логах. Хранится на каждом хосте; используется для базы ошибок в логах на мастере bmfront-а
    sent    => "$mon_dir/logs_errors-sent.json",      # Вспомогательный файл для send_logs_errors_diff
    sent_to_subscribers     => "$mon_dir/logs_errors-sent_to_subscribers.json",         # Вспомогательный файл для send_events_to_subscribers (на мастере bmfront-а)
);

# Дополнительные директории логов (кроме $Utils::Common::options->{dirs}{log})
# Роль хоста => [ директории, в которых проверяем логи ]
my %hostrole2logdirs = (
    bannerland => [
        {log_dir_type => 'dyn', dir => $Utils::Common::options->{dyn_banners_dirs}{temp_dir}, },
        {log_dir_type => 'perf', dir => $Utils::Common::options->{perf_banners_dirs}{temp_dir}, }
    ],

    # для тестирования
    #bmfront => [
    #    # TODO
    #    $Utils::Common::options->{dirs}{root} . "/temp/log_test1",
    #    $Utils::Common::options->{dirs}{root} . "/temp/log_test_A",
    #],
);         #TODO


sub get_host_roles_dirs {
    my $host_role = get_host_role();
    my $specific_roles_dirs = $hostrole2logdirs{$host_role} // [];
    my @host_roles_dirs = (
        {log_dir_type => 'common', dir => $Utils::Common::options->{dirs}{log},},
        @$specific_roles_dirs
    );
    return \@host_roles_dirs;
}

sub get_logs_errors {
    my $hours = shift;

    my $max_errors = 500;
    my $max_badfmt = 10000;
    my $max_lines = 10_000_000;
    my $max_file_processing_duration = 3*60;  # Может не хватить для чтения $max_lines. TODO - увеличить max_file_processing_duration или уменьшить max_lines ?

    my $max_error_len = 1000;
    my $max_line_length = 10_000_000;

    # Ошибки (для строк не в стандартном формате)
    #   Perl:     syntax error|BEGIN failed
    #   Perl, забыли проверить результат open:  readline() on closed filehandle
    #   MySQL:    WARN: DBD::mysql
    #   FCGI:     bind/listen: Address already in use
    my $tmpl_err = qr/(\b|_)error(\b|_)|BEGIN failed|WARN: DBD::mysql|(bash|sh):.* No such file or directory|sh:.* not found|Out of memory|Segmentation fault|bind\/listen: Address already in use|readline\(\) on closed filehandle/i;

    # Исключения (ложные срабатывания $tmpl_err)
    #   '  DBH ERROR   ',  '  SQL ERROR   ',  '  Do_SQL error  '   исключаем из ошибок, т.к. после них делается die
    #   'content reader error', 'Content read error' - Zora output
    #   '2016-09-09	12:11:02,883 WARNING'  -  YT
    #   ConnectTimeout, ConnectionError   -  YT
    my $tmpl_noerr_all = qr/^(rsync error|\@ERROR: max connections .* reached|  DBH ERROR   |  SQL ERROR   |  Do_SQL error  |content reader error$|Content read error$|\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2},\d{3}\s*(WARNING|INFO)|ConnectTimeout: |ConnectionError: |raise ConnectionError|\s*\'(AdGroupMinusWords|CampaignMinusWords)\')/;
    # Исключения для отдельных файлов
    my %tmpl_noerr = (
        #
        "yt_read_ignore.err"                    => qr/./,
        "get-BM-Quintile-AgeCount-AgeBillCost-data_ignore.err"          => qr/./,
        "bm-age-quantiles_ignore.err"           => qr/./,

        #""                                      => qr//,
        "check-logs.log"                        => qr/^(no errors|found some errors, sending|no new errors|cut: write error: Broken pipe|cut: write error)/,
        "check-logs.err"                        => qr/^(cut: write error: Broken pipe|cut: write error)/,
        "compute-indicators.err"                => qr/^(tac: write error|tac: write error: Broken pipe)$/,
        "fcgi-http-server.err"                  => qr/^opening error log\.\.\./,
        "cdict-start-server.log"                => qr/^(COMMAND: getnorm\t|COMMAND: get\t)/i,
        "prefprojsrv-restart-server.err"        => qr/^phrase_list \(ru\):/,
        "test-db-encode_bmfront.err"            => qr/^\s*SQL ERROR   (DataSource::deferreddbi::List_SQL <= Project |DataSource::Elem::List_SQL <= DataSource::deferreddbi |DataSource::Elem::_sth_execute <= DataSource::Elem |\(eval\) <= main |Project::List_SQL <= main )/,
        "test-db-encode_catalogia_media.err"    => qr/^\s*SQL ERROR   (DataSource::deferreddbi::List_SQL <= Project |DataSource::Elem::List_SQL <= DataSource::deferreddbi |DataSource::Elem::_sth_execute <= DataSource::Elem |\(eval\) <= main |Project::List_SQL <= main )/,
        "red-button.log"                        => qr!^(
             [A-Z]\s+(/opt/broadmatching|arcadia|rt-research)/[a-zA-Z0-9\-_\./]+|\[host:
            |^[A-Z] +[a-zA-Z0-9\-_\./]+
            |\|[ 0-9\.]+\%\| +\[[A-Z]+\] \$\([A-Z]+\)+[a-zA-Z0-9\-_\./ ,\{\}]+
        )$!x,
        "get-resources.err"                     => qr/./, # т.к. проверяем индикаторами
        "update-db-shortterm.err"               => qr!^\@ERROR: invalid gid nogroup$!,
        "collect-indicators.err"                => qr!^(\@ERROR: invalid gid nogroup|\@ERROR: Unknown module 'bmexport')$!,
        "collect-logs.err"                      => qr!^(\@ERROR: invalid gid nogroup|\@ERROR: Unknown module 'bmexport')$!,
        "arcadia-setup.log"                     => qr/^A    .*/,
        "yt_external_monitor_broad_match.err"   => qr/./,
        "yt_external_monitor_filt_logs.err"     => qr/./,
    );

    # Блоки строк: для обработки логов типа  local $SIG{'__DIE__'} = sub { print STDERR $proj->stack_trace('  ERROR   '); die('  ERROR TEXT: '.join('', @_)); };
    #   Обрабатываем блок строк нестандартного формата =~ $tmpl_block{$current_block_type}{line},
    #   после которого есть строка стандартного формата =~ $tmpl_block{$current_block_type}{start}
    my %tmpl_block = (
        stack_trace => {
            start => qr/\tERROR:STRACE/,
            line  => qr/^ *STRACE_ERR +/,
        },
    );

    # Регулярка для Warning-ов
    my $tmpl_warn = qr/\b(WARN:|WARNING:)/i;

    my $block_separator = "\n";   # При обработке блока строк будем склеивать их в одну ошибку в базе. Склеиваем через $block_separator
    my $max_block_lines = 500;  # Максимальное количество строк в блоке

    my $curr_time = time;
    my $min_err_time = $curr_time - ($hours * 3600 + 300);

    my @collected_errors_list;
    my %err_warn_counter = ();

    my $err_sum = 0;
    my $warn_sum = 0;

    # Подфункция для обработки очередного логирующего файла
    my $log_file_handler = sub {

        my $log_dir_type = shift;

        my $file_err_sum = 0;
        my $file_warn_sum = 0;

        my $file_path = $File::Find::name;
        return if !-f $file_path;
        #return if $file_path =~ m/\.(\d+)(\.gz)?$/;  # старые логи
        return unless $file_path =~ m/\.(err|log)(|\.1)$/;
        return if mtime($file_path) < $min_err_time;  # на случай, если последние строки плохого формата

        my $file = File::Spec->abs2rel($file_path, $Utils::Common::options->{dirs}{log});
        print_err("file: $file");

        my ($lines_count, $badfmt_count, $block_lines_count) = (0, 0, 0);
        my $current_block_type = '';
        my $last_time = mtime($file_path);
        my $has_too_long_line = 0;
        my %err_indicator_hash = ();

        # Подфункция для обработки очередной записи об ошибке или warning-е:
        #   - добавляем соотвествующее значение в счётчик ошибок $err_warn_counter;
        #   - добавляем элемент в массив ошибок, если их количество не стало больше допустимого (а именно $max_errors)
        my $handle_new_error_item = sub {
            my ($err_text, $err_type, $err_time, $err_pid) = @_;

            if (!$err_time) {
                $err_time = time;
            }
            my $hour_unixtime = (int $err_time / 3600) * 3600;
            $err_warn_counter{$log_dir_type}{$err_type}{$hour_unixtime}++;

            # warning-и пока не заносим в массив ошибок, т.к. на текущий момент их слишком много (см. DYNSMART-887)
            if ($err_type eq "warning") {
                $file_warn_sum++;
                return;
            }

            $file_err_sum++;

            # если записей об ошибках накопилось слишком много, не добавляем новый элемент в массив
            if ($err_indicator_hash{max_errors}) {
                return;
            }

            $err_pid = $err_pid // '';
            $err_text = substr($err_text, 0, $max_error_len);
            my %err =  (pid => $err_pid, file => $file, error => $err_text, time => $err_time, type => $err_type);
            push (@collected_errors_list, \%err);
        };


        my $start_file_processing_time = time;
        # Используем timeout, т.к. tac ... | head может работать некорректно (см. https://a.yandex-team.ru/commit/2678841)
        # Делаем cut -c, чтобы не падать на длинных строках

        open my $fh_log, "timeout --kill-after 30s $max_file_processing_duration tac $file_path | cut -c 1-$max_line_length |"  or do {
            $handle_new_error_item->("ERROR: Could not open log file ($!) !", "system");
            return;
        };

        binmode $fh_log, ':bytes';  #   Avoid the 'Malformed UTF-8 character (fatal)' error

        while (<$fh_log>) {

            my $line_bytes = $_;
            chomp $line_bytes;
            if (length($line_bytes) == $max_line_length) {
                $has_too_long_line = 1; # Не делаем здесь  push @collected_errors_list, {...},  т.к. можем находиться в середине блока ($current_block_type)
            }
            my $line = Encode::decode('UTF-8', $line_bytes);
            $lines_count++;
            print_err("lines_count: $lines_count") unless $lines_count % 100_000;

            if ($lines_count >= $max_lines) {
                $handle_new_error_item->("ERROR: Too many lines in the log file!", "system");
                last;
            }
            if (($badfmt_count >= $max_badfmt) && (!$err_indicator_hash{badfmt}))  {
                $handle_new_error_item->("ERROR: Too many bad lines in the log file!", "system");
                $err_indicator_hash{badfmt} = 1;
            }
            if (($file_err_sum >= $max_errors) && (!$err_indicator_hash{max_errors})) {
                #  если начала фиксироваться ошибка блока, удаляем её из массива
                if ($current_block_type) {
                    pop @collected_errors_list;
                }
                $handle_new_error_item->("ERROR: Too many errors in the log file!", "system");
                $err_indicator_hash{max_errors} = 1;
            }
            $line = Encode::decode('UTF-8', Encode::encode('UTF-8', $line));

            next if ($line =~ /^\[DUMP\]\t/); # игнорируем блоки дампа.

            if ($current_block_type) {
                if ($tmpl_block{$current_block_type}{line}  and  $line =~ $tmpl_block{$current_block_type}{line}) {
                    # Если обрабатываем блок строк
                    $block_lines_count++;
                    if ($block_lines_count >= $max_block_lines) {
                        $handle_new_error_item->("ERROR: Too many lines in block($current_block_type) in the log file!", "system");
                        $current_block_type = '';
                        $block_lines_count = 0;
                    } else {
                        if (!$err_indicator_hash{max_errors}) {
                            $collected_errors_list[-1]{error} .= $block_separator . substr($line, 0, $max_error_len);  # "Приклеиваем" строку к предыдущей ошибке
                        }
                        next;   # и прекращаем обработку этой строки
                    }
                } else {
                    $current_block_type = ''   if $current_block_type;  # Блок закончился
                    $block_lines_count = 0;
                }
            }

            my @fld = split /\t/, $line, 3;     #  (в сообщении разрешается \t, но не \n)
            my $unixtime;
            my $tm_str = $fld[0];
            if (defined $tm_str) {
                $tm_str =~ s/\.\d{6}$//;   # datetime.now() в python пишет в формате  2016-10-20 06:26:34.383370
                $unixtime = time_log2unix($tm_str);
            };

            if (@fld < 3  or  !defined $unixtime) {
                # Строка не в стандартном формате
                $badfmt_count++;
                if (     $line =~ $tmpl_err
                    and  $line !~ $tmpl_noerr_all
                    and  not ($tmpl_noerr{$file}  and  $line =~ $tmpl_noerr{$file})
                ) {
                    $handle_new_error_item->($line, "exterior", $last_time); # Если в строке не записано время в правильном формате, то ей присваивается время следующей за ней ошибки (или mtime, если таких ошибок нет).  TODO - корректно обрабатывать это в базе ошибок
                }
            } else {
                # Стандартный формат строки лога
                last if $unixtime < $min_err_time;
                $last_time = $unixtime;
                $badfmt_count = 0;

                $current_block_type = (grep { $tmpl_block{$_}{start}  and  $line =~ $tmpl_block{$_}{start} } keys %tmpl_block)[0] // ''; # Определяем, является ли эта строка началом блока
                my $pid = $fld[1];
                $pid =~ s/[\[\]]//g;
                if ( $current_block_type    # Если строка является началом блока, нужно сохранить ее в @collected_errors_list
                    or  $fld[2] =~ /(^|\t)ERROR/    # В строке может быть несколько полей, разделяемых табуляцией; ERROR может быть в начале одного из полей
                        and $fld[2] !~ /^WARN/      # Если это стандартная строка, начинающаяся с "WARN", игнорируем ее (для строк типа "WARNING: SOAP failed: 2015-04-15 09:40:26 [30920] ERROR: died: 403 Forbidden ...")
                ) {
                    $handle_new_error_item->($fld[2], "own", $unixtime, $pid);
                } elsif ($fld[2] =~ $tmpl_warn) {
                    $handle_new_error_item->($fld[2], "warning", $unixtime, $pid);
                }
            }
        }


        print_err("close $file_path ...");
        close $fh_log or do {
            print_err("WARN: close failed $file_path ($!)");
        };
        print_err("close $file_path done");
        my $duration = time - $start_file_processing_time;
        print_err("$file_path processing time: $duration seconds.");
        if ($duration >= $max_file_processing_duration) {
            $handle_new_error_item->("ERROR: file takes too long time to process. Too large file?", "system");
            print_err("ERROR: Too long time to process $file_path ($start_file_processing_time, $duration >= $max_file_processing_duration)");
        }

        if ($has_too_long_line) {
            $handle_new_error_item->("ERROR: Too long lines in the log file!", "system");
        }

        $err_sum += $file_err_sum;
        $warn_sum += $file_warn_sum;
    };

    # Subroutine for log directory
    my $handle_dir  = sub {
        my ($log_dir, $log_dir_type) = (@_);

        print_err("log_dir: $log_dir");

        # follow:   Causes symbolic links to be followed.
        File::Find::find( {
                wanted => sub {$log_file_handler->($log_dir_type)},
                follow => 0,
                no_chdir => 1, # Т.к. могли запускать из директории, в которую не сможем сделать cd. Например, sudo -H -u bmclient bash -c '/opt/broadmatching/scripts/monitors/check-logs.pl'  из директории с правами  drwx------ 8 emurav 4096 Jan 10 20:51 /home/emurav/
            },
            $log_dir,
        );
    };

    my $host_roles_and_dirs = get_host_roles_dirs();
    for my $host_role_info (@$host_roles_and_dirs) {
        my $log_dir = $host_role_info->{dir} // "";
        my $log_dir_type = $host_role_info->{log_dir_type} // "";
        if ($log_dir && $log_dir_type) {
            $handle_dir->($log_dir, $log_dir_type);
        }
    }

    print_err("get_logs_errors done");
    return (\@collected_errors_list, \%err_warn_counter, $err_sum, $warn_sum);
}


sub save_logs_errors {
    my $errors = shift;  # array of hash

    my $file_errors = $logs_errors_file{current};

    my $errors_all = load_json($file_errors);
    if (not ref($errors_all) eq 'ARRAY') {
        print_err("WARN: Could not get previous errors from file ($file_errors)");
        $errors_all = [];
    }

    push @$errors_all, reverse @$errors;

    my $min_err_time = time - 24 * 3600;
    @$errors_all = grep { $_->{time} >= $min_err_time }  @$errors_all;

    my %seen;
    @$errors_all = grep { !$seen{ join("\t", $_->{type}, $_->{time}, $_->{file}, $_->{error}) }++ }  @$errors_all;  # uniq @$errors_all

    my $res = save_json($errors_all, $file_errors);
    if (!$res) {
        print_err ("ERROR: save_json failed ($file_errors)");
    }

    # Делаем аналогично compute-indicators.pl
    # проставим симлинк - временное решение!
    my $dirs = $Utils::Common::options->{dirs};
    my $link = $file_errors;
    $link =~ s/.*\///;
    $link = $dirs->{export}.'/'.$link;
    if (! -e $link) {
        symlink $file_errors, $link;
    }
}

# Параметры:
#   reduce_host     1/0 - Заменять хост на его роль и группу
#   keep_str        1/0 - не уникализовать строку
sub error2str {
    my ($err, %prm) = @_;

    # TODO: change fields names to DB fields names?
    my $str_host = $err->{Host} // $err->{host} // '';
    my $str_err = $err->{ErrorText} // $err->{error};
    my $str_file = $err->{File} // $err->{file};

    unless ($prm{keep_str}) {
        if ( $str_err =~ m/^(ERROR: died: dbh error \d+: \[.*\]; SQL: .* values ).*/ ) {
            # ERROR: died: dbh error 2013: [Lost connection to MySQL server during query]; SQL: replace into BannerMatchHistory (bid,InfoKey,InfoValue) values (1120744812,'cmid','13843926-20151029192313-7/15-6565@host.yandex.ru'),(1120744812,'lang','ru'),(1120744812,'cost_filtration','0'),(1120744812,'cost_forecast','0'),(1120744812,'categs',''),(1120746178,'cmid','13843926-20151029192313-7/15-6565@host05h.yandex.ru'),(1120746178,'lang','ru'),(1120746178,'cost_filtration','0'),(1120746178,'cost_forecast','0'),
            $str_err = $1;
        } elsif ( $str_err =~ m/(^.*\bERROR: '?can't connect to prefprojsrv \(host='[^']+', last_error='[^']+', str='DoJson\s+\["[^"]+","[^"]+",").+/) {
            # ERROR: cmd processing failed (edit_phrase_list): ERROR: can't connect to prefprojsrv (host='127.0.0.1', last_error='prefprojsrv can_read timeout', str='DoJson    ["phrase_list_lang_ru","get_catalogia_flags_hash","Алмазный инструмент для камня Сегменты сверла фрезы корпуса диски канат для
            # ERROR: can't connect to prefprojsrv (host='127.0.0.1', last_error='prefprojsrv can_read timeout', str='DoJson   ["phrase_list_lang_ru","get_catalogia_flags_hash","Алмазный инструмент для камня Сегменты  сверла  фрезы  корпуса  диски  канат для обработки камня  мрамора,Станки и оборудование
            # ERROR: 'can't connect to prefprojsrv (host='catalogia-media-front01i.yandex.ru', last_error='prefprojsrv can_read timeout', str='DoJson   ["phrase_lang_tr","add_subphraser_category","kuru vakum makinalar","Клининговое профессиональное оборудование"]') at /home/broadmatching/scripts/broadmatching-server/../lib/BM/BMClient/PrefProjSrvClient.pm line 102.
            $str_err = $1;
        } elsif ( $str_err =~ m!^(Can't download url )[^\s]+(.*?)(at [^\s]+Project.pm line \d+.( <[^\s]+> line \d+\.)?)?$! ) {
            # Can't download url http://www.rcvostok.ru/dacha_sad_ogorod/tovary_dlya_rassady/podvjazki_klipsy_dlja_rastenij?&utm_source=yadirect&utm_medium=cpc&utm_term=mainlink--{keyword}--{position_type}--{position}&utm_content={ad_id}&utm_campaign=dacha_sad_ekb_site => 2016-04-14 21:22:38    [27629] ERROR: died: system cmd 'timeout -k10 6s /usr/local/bin/zoracl fetch --format=document --max-redirects=9 --priority=0 -E --ignore-low-priority=yes --redirect-mode=last --freshness=36000 --source=drf --send-quota=1000 --input=/opt/broadmatching/temp/Zora/temp_urls1460658151.27629 --output=/opt/broadmatching/temp/Zora/temp_urls_result1460658151.27629 --timeout=6' failed:( Inappropriate ioctl for device ) Duration: 6.115618 at /opt/broadmatching/scripts/controller/../lib/Project.pm line 2316.
            # Can't download url http://mebelmagnat.ru/82/shkaf/?utm_medium=cpc&utm_source=yandex.{source_type}&utm_campaign=Шкафы_Купе_Остальные_1&utm_content={ad_id}&utm_term={keyword}&rs=direct1_{source_type}_{banner_id}_{keyword} => 2016-04-18 15:00:01    [3655]  ERROR: died: system cmd 'timeout -k10 6s /usr/local/bin/zoracl fetch --format=document --max-redirects=9 --priority=0 -E --ignore-low-priority=yes --redirect-mode=last --freshness=36000 --source=drf --send-quota=1000 --input=/opt/broadmatching/temp/Zora/temp_urls1460980795.3655 --output=/opt/broadmatching/temp/Zora/temp_urls_result1460980795.3655 --timeout=6' failed:( Inappropriate ioctl for device ) Duration: 6.126096 at /opt/broadmatching/scripts/controller/../lib/Project.pm line 2317, <GEN7> line 858317.
            if ($3) {
                $str_err = "$1$2$3";
            } else {
                # Строка была обрезана по длине
                $str_err = "$1";
            }
        } elsif ( $str_err =~ m!(^.*\bERROR: Too large cartesian_product_size:)! ) {
            $str_err = $1;
        }

        $str_err =~ s/\.tmp\.\d+\.[0-9a-zA-Z_]{4}\b/.tmp.XXXX/g;   # Файлы из get_tempfile c pid новый формат
        $str_err =~     s/\.temp\.[0-9a-zA-Z_]{4}\b/.temp.XXXX/g;   # Файлы из get_tempfile для совместимости с кодом до 02.10.2018
        $str_err =~      s/\.tmp\.[0-9a-zA-Z_]{4}\b/.tmp.XXXX/g;    # Файлы из get_tempfile для совместимости с кодом до 02.10.2018

        $str_err =~ s/\d+(, *\d+)*/X/g;   # Несколько чисел, перечисленные через запятую (с пробелами или без)
        $str_err =~ s/\?, \?, \?, \?(, \?)/\?, \?, \?, \?/g;   # Для ошибок MySQL
        $str_err =~ s/[а-яё]+( [а-яё]+)*/Ы/ig;   # Слова и фразы из русских букв
        #$str_err =~ s/([^A-Z0-9\s\.\-~\`!@#\$%\^&\*\(\)_\-\+\=\[\]\{\}\|\\;:'"\<\>,\.\?\/]+[\s,]*)+/Z/ig;
        $str_err =~ s/(malformed or illegal unicode character in string \[)[^\]]*/$1/ig;

        $str_file =~ s/\d+/X/g;
    }
    if ($prm{reduce_host}) {
        $str_host = join(";", map {$_ // ''} (
                $err->{host_role} // get_host_role($str_host),
                $err->{host_group} // (get_host_info($str_host) || {})->{group},
        ));
    }
    return join("\t",  map {$_ // ''} ( $str_host, $err->{ErrorInfo} // $err->{type}, $str_file, $str_err));
}


sub send_logs_errors_diff {
    my $errors = shift;  # array of hash

    my $file_errors = $logs_errors_file{sent};
    my $errors_times = load_json($file_errors);
    if (not ref($errors_times) eq 'HASH') {
        print_err("WARN: Could not get previous errors from file ($file_errors)");
        $errors_times = {};
    }

    my @errors_new;
    my $min_err_time = time - 24 * 3600;

    for my $err (reverse @$errors) {
        if ( !$errors_times->{error2str($err)}  or  $errors_times->{error2str($err)} < $min_err_time) {
            push @errors_new, $err;
        }
        $errors_times->{error2str($err)} = max(($errors_times->{error2str($err)} || 0), $err->{time});
    }

    for (keys %{$errors_times}) {
        delete $errors_times->{$_}  if $errors_times->{$_} < $min_err_time;
    };

    if (@errors_new) {
        @errors_new = sort {
            $a->{file} cmp $b->{file} or
            $a->{time} cmp $b->{time} or
            $a->{pid} cmp $b->{pid} or
            $a->{error} cmp $b->{error} or
            0
        } @errors_new;

        my @out;
        for my $i (0 .. $#errors_new) {
            push @out, "\n".$errors_new[$i]->{file}.":"   if ($i == 0  or  $errors_new[$i - 1]->{file} ne $errors_new[$i]->{file});
            my $err = $errors_new[$i];
            push @out, sprintf("    %s  [%s]\t%s", time_unix2db($err->{time}), $err->{pid} // 0, $err->{error});
        }

        my @files = map { $_->{file} } @errors_new;
        $_ =~ s/\.(log|err)$//  for @files;
        @files = uniq(@files);

        BM::Monitor::Utils::send_error(["New errors in log files [" . join(",", @files) . "]", '', @out]);
    } else {
        print_err("No new errors");
    }

    save_json($errors_times, $file_errors);
}


sub print_logs_errors {
    my $errors = shift;  # array of hash

    if (ref($errors) ne 'ARRAY') {
        die "Internal ERROR: Bad errors list in print_logs_errors!";
    }

    print "======== print_logs_errors: " . (scalar (@$errors)) . " errors ========", "\n";

    for my $err (reverse @$errors) {
        if (ref($err) ne 'HASH') {
            print "Internal ERROR: Bad error in list in print_logs_errors!";
            next;
        }
        print join("\t",
            @{$err}{qw[ file type ]},
            time_unix2db($err->{time}),
            "[" . ($err->{pid} // '') . "]",
            @{$err}{qw[ error ]},
        ), "\n";
    }
    print "==============================================", "\n";
}


# Отправка в Графит значений (почасовых) из счётчика ошибок и warning-ов в заданном часовом диапазоне
sub send_logs_errors_metrics {

    my $err_warn_counter = shift;
    my $start_time = shift;
    my $end_time = shift;

    # хешик, отображающий имена рассматриваемых ошибок в функции get_logs_errors на имена метрик в Графите
    my %err_type2graphite_name = (
        "warning"  => "warning",
        "own"      => "error_own",
        "system"   => "error_system",
        "exterior" => "error_exterior",
    );

    my $graphite_client = BM::GraphiteClient->new( %{$Utils::Common::options->{GraphiteClient_params}} );
    my $solomon_client = BM::SolomonClient->new();
    my $host_roles_and_dirs = get_host_roles_dirs();
    my %log_dir_types = map {$_->{log_dir_type} => 1} (@$host_roles_and_dirs);
    for my $log_dir_type (keys %log_dir_types) {
        for my $err_type (keys %err_type2graphite_name) {
            for (my $hour_unixtime = $start_time; $hour_unixtime <= $end_time; $hour_unixtime += 3600) {
                my $value = $err_warn_counter->{$log_dir_type}{$err_type}{$hour_unixtime} // 0;
                $graphite_client->send_value(["monitoring.logs\.$log_dir_type\.$err_type2graphite_name{$err_type}"],
                                              $value, time => $hour_unixtime);
                $solomon_client->push_single_sensor({
                    cluster     => "host_info",
                    service     => "scripts",
                    sensor      => "logs_count",
                    labels      => {
                        host => get_curr_host(),
                        dir_type => $log_dir_type,
                        err_type => $err_type2graphite_name{$err_type},
                    },
                    value       => $value,
                    ts_datetime => gmtime($hour_unixtime)->datetime(),
                });
            }
        }
    }
}

# Возвращает project для ошибки
# На входе - ошибка в виде хэша  { host => ..., file => ..., ... }
sub get_project {
    my ($err) = @_;

    my $project = $err->{project} // Utils::Hosts::get_host_info($err->{host})->{project};
    return $project;
}

sub convert_keys_format {
    my ($data, $format) = @_;

    my %convert = (
        host => 'Host',  file => 'File',  time => 'ErrorTime',  pid => 'PID',  error => 'ErrorText',  type => 'ErrorInfo',
        host_role => 'HostRole', host_group => 'HostGroup',  host_short => 'HostShort',
    );
    if ($format eq 'scripts') {
        %convert = reverse %convert;
    } elsif ($format eq 'db') {
        ;
    } else {
        die "Bad format ($format)";
    }

    if (ref($data) eq 'HASH') {
        for my $key (keys %$data) {
            convert_keys_format($data->{$key}, $format);
            if ($convert{$key}) {
                $data->{ $convert{$key} } = delete $data->{$key};
            }
        }
    } elsif (ref($data) eq 'ARRAY') {
        convert_keys_format($_, $format)   for @$data;
    }
    return $data;
}

# Возвращает список "подписок" - хэшей вида { title => ..., users = [], filter => []  }
# users => список логинов на @yandex-team
sub get_subscriptions {
    my @subscr;

    push @subscr, {
        users => [qw[ emurav ]],
        title => 'project_BM_4emurav',
        filter => [
            { project => [qw[ BM ]], },
        ],
        exclude_filter => [
            { host_role => [qw[ bmdictdb ]], },
            { host_role => [qw[ bmfront ]], file => [qw[
                deploy-tests.log deploy-tests.err
            ]] },
            { script_name => [qw[
                    bm_corpora_write_active_banners_raw
                    bm_corpora_normalize_banners
                    bm_corpora_count_tokens
                    bm_corpora_generate_resources
            ]], },  # scripts_serkh
            { file => [qw[ update_yt_logs.err ]], },  # serkh@
        ],
    };

    push @subscr, {
        users => [qw[ serkh ]],
        title => 'scripts_serkh',
        filter => [
            {
                script_name => [qw[
                    bm_corpora_write_active_banners_raw
                    bm_corpora_normalize_banners
                    bm_corpora_count_tokens
                    bm_corpora_generate_resources
                    update_yt_logs
                ]],
            },
        ],
    };

    push @subscr, {
        users => [qw[ breqwas emurav ]],
        title => 'bm-dev',
        filter => [
            { host_role => [qw[ bm-dev ]], },
        ]
    };

    return \@subscr;
}

1;
