
package BM::Monitor::Utils;
use strict;
use utf8;

# объединяющий модуль для системы мониторинга

use open ':utf8';

use Utils::Sys qw(
    uniq
    do_safely
    print_err
);
use Utils::Funcs qw(
    prm_list_filter
);
use Utils::Common;
use Utils::Hosts qw(get_host_role get_host_group get_curr_host get_host_datacenter get_short_hostname);

use base qw(Exporter);
our @EXPORT_OK = qw(
    send_error
    %indicators_file
    add_aux_fields_to_events_list
    filter_list
    get_subscriptions_filters
    dates_obj
    check_cache_status
);

my $options = $Utils::Common::options;
my $mon_dir = $options->{dirs}{monitor_info};
my $zbx_dir = $mon_dir;


# файлы, куда следует писать значения индикаторов
our %indicators_file = (
    archive => "$mon_dir/indicators-archive.log",
    current => "$mon_dir/indicators-current.log",
);

###############################################
# Вспомогательные функции
###############################################

# strf = String Format
# strp = String Parse
# %indc - данные об индикаторе,
#    (group => 'simgpraphs', object => 'QueryLog', property => 'n_lines', host => 'somehost');
# %indcline - данные об записи в логе со значением индикатора
#    ('time' => $unixtime, pid => $pid, value => $val, indc => \%indc);


# параметры:
# prefix =>     префикс для Subject (по умолчанию - вычисляется по имени хоста)
sub send_error {  # via sendmail
    my $err_arr = shift;
    my %par = @_;

    my $host = get_curr_host();
    my $prefix = $par{prefix};
    if (!defined $prefix) {
        my $role = get_host_role($host) || $host;
        $prefix = "[$role]";
    }

    my $sendmail = "/usr/sbin/sendmail -t";
    my $reply_to = "From: SendError\n";
    my $script = $0;
    $script =~ s/.*\///;
    my $subject = "Subject: $prefix $script at $host\n";
    my $send_to = 'To: '.$Utils::Common::options->{admin_mail}."\n";

    my $content = "";
    $content .= scalar(localtime)."\n";
    $content .= join("\n", @$err_arr)."\n";

    if (!$Utils::Common::options->{is_devel}) {
        open(SEND, "|$sendmail") or die "Cannot open $sendmail: $!";
    } else {
        open(SEND, ">&STDERR");
    }
    print SEND $reply_to;
    print SEND $subject;
    print SEND $send_to;
    print SEND "Content-type: text/plain; charset=UTF-8\n\n";
    print SEND $content;
    close(SEND);
}

# На входе:
#   Список хэшей - индикаторов или ошибок из логов
#   event_type => 'error', 'indicator'
# На выходе - тот же список хэшей, в которые дописаны поля project, host_role, host_short, ...
#
sub add_aux_fields_to_events_list {
    my ($list, %prm) = @_;
    #print STDERR "add_aux_fields_to_events_list ... " . `date`;

    for my $el (@$list) {
        $el->{host} //= $el->{Host};
        $el->{file} //= $el->{File};
    }

    my %host2project = map {
        $_ => ( Utils::Hosts::get_host_info($_) // {} )->{project}
    }  uniq  map {$_->{host} // ()}  @$list;
    my %host2role = map {
        $_ => get_host_role($_)
    }  uniq  map {$_->{host} // ()}  @$list;
    my %host2group = map {
        $_ => get_host_group($_)
    }  uniq  map {$_->{host} // ()}  @$list;

    for my $el (@$list) {

        #my $event_type = $el->{event_type} // $prm{event_type} // '';
        #if ($event_type eq 'error') {
        #    $el->{project} //= BM::Monitor::Logs::get_project($el);
        #} elsif ($event_type eq 'indicator') {
        #    $el->{project} //= BM::Monitor::Indicators::get_project($el);
        #}
        #$el->{host_role} //= get_host_role($el->{host});
        # Не используем функции BM::Monitor::Logs::get_project($el) и get_host_role($el->{host}), чтобы ускорить обработку больших массивов
        $el->{project} //=  $el->{host}  ?  $host2project{ $el->{host} }  :  "";
        $el->{host_role} //=  $el->{host}  ?  $host2role{ $el->{host} }  :  "";
        $el->{host_group} //=  $el->{host}  ?  $host2group{ $el->{host} }  :  "";

        $el->{datacenter} //= get_host_datacenter($el->{host}) // '';
        $el->{host_short} //= get_short_hostname($el->{host}) // '';
        $el->{file_template} //= do { my $s = $el->{file} // '';  $s =~ s/\d+/X/g;  $s };
        $el->{script_name} //= do { my $s = $el->{file} // ''; $s = ($s =~ /^(.*)\.(err|log)$/)[0] // $s;  $s =~ s/[_\-0-9]*$//;  $s };    # TODO
    }

    #print STDERR "add_aux_fields_to_events_list done " . `date`;
    return $list;
}

sub hash2str {
    my ($h, %prm) = @_;
    my $delimiter = "\n";  #TODO
    return join($delimiter, map { $_, ($h->{$_} // '') }  ($prm{keys} ? @{$prm{keys}} : (sort keys %$h)) );
};

# Фильтрует список фильтром, заданным в параметре filter; исключает элементы, подходящие под фильтр, заданный в параметре exclude_filter
# Параметры:
#   filter          =>
#   exclude_filter  =>
# TODO перенести в scripts/lib/Utils/Funcs.pm аналогично prm_list_filter
sub filter_list_old {
    my ($list, $filters) = @_;

    # Фильтрует список фильтром
    # Параметры:
    #   filter      => фильтр
    #   exclude     => 1/0  -  вернуть объекты, не подходящие под фильтр
    # TODO перенести в scripts/lib/Utils/Funcs.pm аналогично prm_list_filter
    sub _filter_list_old {
        my ($list, $filter, %prm) = @_;
        my $list_new = [];
        
        if ($prm{exclude}) {
            my $list_exclude = _filter_list_old($list, $filter);
            my %exclude = map { hash2str($_) => 1 } @$list_exclude;
            $list_new = [ grep { not $exclude{ hash2str($_) } } @$list ];
        } else {
            if (ref($filter) eq 'HASH') {
                $list_new = prm_list_filter($list, { filter => $filter });
            } elsif (ref($filter) eq 'ARRAY') {
                for my $flt (@$filter) {
                    push @$list_new, @{ prm_list_filter($list, { filter => $flt }) || [] };    # элементы в list_new могут дублироваться
                }
                my %new = map { hash2str($_) => 1 } @$list_new;
                $list_new = [ grep { $new{ hash2str($_) } } @$list ];
            } else {
            }
        }
        #print_err("_filter_list_old done");
        return $list_new;
    };

    my $list_new = [];
    for my $flt (@{ $filters || [] }) {
        my $list_new_part = $list;
        $list_new_part = _filter_list_old($list_new_part, $flt->{filter});
        $list_new_part = _filter_list_old($list_new_part, $flt->{exclude_filter}, exclude => 1);
        push @$list_new, @$list_new_part;
    }
    my %new = map { hash2str($_) => 1 } @$list_new;
    $list_new = [ grep { $new{ hash2str($_) } } @$list ];

    return $list_new;
}

# Фильтрует список фильтром, заданным в параметре filter; исключает элементы, подходящие под фильтр, заданный в параметре exclude_filter
# Параметры:
#   filter          =>
#   exclude_filter  =>
# TODO перенести в scripts/lib/Utils/Funcs.pm аналогично prm_list_filter
sub filter_list {
    my ($list_data, $filters, %prm) = @_;
    print_err("filter_list ... ".(scalar @$list_data));

    my $key_fld = '__key_filter_list';
    print_err("add key_fld ... " . join(",", @{$prm{keys} || []}) );
    $_->{$key_fld} = hash2str($_, keys => $prm{keys})   for @$list_data;
    print_err("key2el ... ");
    my $key2el = { map { $_->{$key_fld} => $_ }  @$list_data };
    print_err("list ...");
    my $keys_hash = { map { $_->{$key_fld} => 1 }  @$list_data };
    print_err("list done");
    my $keys_out = {};

    # $keys_hash    { ключ => 1 }
    # Параметры:
    #   filter      => фильтр
    #   exclude     => 1/0  -  вернуть объекты, не подходящие под фильтр
    # TODO перенести в scripts/lib/Utils/Funcs.pm аналогично prm_list_filter
    sub _filter_list {
        my ($keys_hash, $key2el, $filter, %prm) = @_;
        my $keys_hash_new = {};
        print_err("_filter_list ... ");

        if ($prm{exclude}) {
            my $keys_hash_exclude = _filter_list($keys_hash, $key2el, $filter);
            $keys_hash_new = {};
            $keys_hash_new->{$_} = 1   for grep { not $keys_hash_exclude->{$_} }  keys %$keys_hash;
        } else {
            if (ref($filter) eq 'HASH') {
                $keys_hash_new = { %$keys_hash };
                for my $key (keys %$filter) {
                    my $val = $filter->{$key};
                    if (ref($val) eq '') {
                        delete $keys_hash_new->{$_}  for grep { ($key2el->{$_}{$key} // '') ne $val }  keys %$keys_hash_new;
                    } elsif (ref($val) eq 'ARRAY') {
                        my %val = map { $_ => 1 } @$val;
                        delete $keys_hash_new->{$_}  for grep { defined $key2el->{$_}{$key}  and  not $val{ $key2el->{$_}{$key} } }  keys %$keys_hash_new;
                    }
                }
            } elsif (ref($filter) eq 'ARRAY') {
                $keys_hash_new = {};
                for my $flt (@$filter) {
                    $keys_hash_new->{$_} = 1   for keys %{ _filter_list($keys_hash, $key2el, $flt ) || {} };
                }
            } else {
            }
        }
        print_err("_filter_list done  " . (scalar keys %$keys_hash) . " -> " . (scalar keys %$keys_hash_new));
        return $keys_hash_new;
    };

    my $keys_hash_new = {};
    for my $flt (@{ $filters || [] }) {
        my $keys_hash_new_part = _filter_list($keys_hash, $key2el, $flt->{filter});
        $keys_hash_new_part = _filter_list($keys_hash_new_part, $key2el, $flt->{exclude_filter}, exclude => 1);
        $keys_hash_new->{$_} = 1  for keys %$keys_hash_new_part;
    }

    my $list_data_new = [grep { $keys_hash_new->{ $_->{$key_fld} } } @$list_data ];
    delete $_->{$key_fld}   for @$list_data_new;
    print_err("filter_list done ".(scalar @$list_data) . " -> " . (scalar @$list_data_new));
    return $list_data_new;
}

# На входе:
#   $proj
#   Список подписок
#   user => пользователь    или   title => название подписки
# Возвращает список фильтров для заданного пользователя или названия подписки
sub get_subscriptions_filters {
    my ($proj, $subscriptions, %prm) = @_;

    my $subscriptions_selected = $subscriptions;
    if ($prm{user}) {
        $subscriptions_selected = [grep { my @users = @{$_->{users}};  grep { $_ eq $prm{user}} @users }  @$subscriptions_selected];
    }
    if ($prm{title}) {
        $subscriptions_selected = [grep { $_->{title} eq $prm{title} }  @$subscriptions_selected];
    }

    my $filters = [];
    if ($subscriptions_selected) {
        $filters = [ map {{ filter => $_->{filter},  exclude_filter => $_->{exclude_filter} }} @$subscriptions_selected ];
    }

    return $filters;
}

# Посылает список событий одному пользователю
sub send_events_to_user {
    my ($proj, $events_selected, $user, %prm) = @_;
    my $send_opts = $prm{send_opts} // {};
    my $fields = $prm{fields};
    my $fields_title = $prm{fields_title};
    my $fields_hierarchy = $prm{fields_hierarchy};
    my $title = $prm{title} // "";

    my $text = '';
    for my $fld (@{$fields_title // []}) {
        $text .= "  [" . join(", ", uniq sort (map {$_->{$fld} // ""} @$events_selected)) . "]\n";
    }
    $text .= scalar(localtime) . "\n";
    $text .= "\n";

    sub cmp_by_fields {
        my ($el1, $el2, $fields) = @_;
        for my $f (@{ $fields // [] }) {
            my $res = $el1->{$f} cmp $el2->{$f};
            return $res if $res;
        }
    }

    for my $host (uniq sort map { $_->{host} // ''} @$events_selected) {
        my $role = get_host_role($host) || $host;
        $text .= "====== Host: $host, Role: $role ======\n\n";
        my $events_selected_host = [ (grep { ($_->{host} // '') eq $host }  @$events_selected) ];

        $events_selected_host = [ sort {
                    cmp_by_fields($a, $b, [ map { @{ $_ // [] } } ($fields_hierarchy, $fields) ])
            } @$events_selected_host ];

        my $str_title_curr = '';
        for my $event (@$events_selected_host) {
            my $str_title = $fields_hierarchy  ?
                       "__ " . join(", ", map {"$_: " . ($event->{$_} // '')} @{$fields_hierarchy}) . " __"
                    :  "";
            if ($str_title ne $str_title_curr) {
                $text .= "$str_title\n\n";
                $str_title_curr = $str_title;
            }

            if (ref($fields) eq 'ARRAY') {
                $text .= join("\t", map {$event->{$_} // ''} @$fields) . "\n\n";
            } else {
                $text .= "  $_:\t" . ($event->{$_} // '') . "\n"    for  @{ $fields // [sort keys %$event]};
                $text .= "\n";
            }
        };
    };

    #$text .= "\n\n================================================\n\n";
    #$text .= "You have received this message because of this subscription settings:\n  " . to_json($filters);

    my $email_to = $user . '@yandex-team.ru';
    $proj->SendMail({
            to => $email_to,
            subject => "Monitoring for $user ($title)",
            body => $text,
            from => 'SendError@bmfront.bm.yandex-team.ru',
            %$send_opts,
    }) || $proj->log("ERROR: SendMail failed for $user ($title)");
}

# Посылает список событий подписчикам (в соответствии с фильтрами из списка подписок)
# На входе:
#   $proj
#   Список "событий" (например, ошибок из логов)
#   Список подписок
#   fields_hierarchy =>
#   fields_title =>
#   fields =>
sub send_events_to_subscribers {
    my ($proj, $events, $subscriptions, %prm) = @_;

    my @errors;     # Ошибки при отправке

    my %sent_events;
    for my $user (uniq sort map { @{ $_->{users} // [] } } @$subscriptions) {
        $proj->log("Processing user $user ...");
        my $filters = get_subscriptions_filters($proj, $subscriptions, user => $user);

        my $events_selected = filter_list_old($events, $filters);   # TODO use sub filter_list
        if (not @$events_selected) {
            $proj->log("No events for $user");
            next;
        }

        do_safely( sub {
                send_events_to_user($proj, $events_selected, $user,
                    (map {$_ => $prm{$_}} qw[ title fields fields_title fields_hierarchy ]),
                     send_opts => {},
                );
                $sent_events{ hash2str($_) } = $_  for @$events_selected;
                return 1;
            }, no_die => 1,
        ) || do {
            $proj->log("ERROR: Sending failed for $user");
            push @errors, "User:$user";
            next;
        };
    }

    # События, которые никому не были отправлены
    my $events_unsent = [grep {not $sent_events{ hash2str($_) }} @$events];
    if (@$events_unsent) {
        $proj->log("Processing unsent events ...");
        do_safely( sub {
                send_events_to_user($proj, $events_unsent, 'bm-dev',
                    (map {$_ => $prm{$_}} qw[ title fields fields_title fields_hierarchy ]),
                    send_opts => {
                        subject => "Monitoring: Uncared events (" . ($prm{title} // '') . ")",
                    },
                );
            }, no_die => 1,
        ) || do {
            $proj->log("ERROR: events_unsent sending failed");
            push @errors, "Uncared_events";
        };
    } else {
        $proj->log("All events were sent!");
    }

    $proj->log("send_events_to_subscribers done." . (@errors  ?  " Errors were at [ ".join(", ", @errors) . "]"  :  ""));
    return (@errors  ?  0  :  1);
}

my $graphite_client;        
sub graphite_client {       
    return $graphite_client if $graphite_client;        
    require BM::GraphiteClient;     
    $graphite_client //= BM::GraphiteClient->new(       
        %{ $Utils::Common::options->{GraphiteClient_params} }       
    );      
    return $graphite_client;        
}

# Проверить set/get кэша
# На входе:
#   $client - объект, у которого есть методы set($key, $value, $expiration_secs) и get($key)
sub check_cache_status {
    my $client = shift;
    my %par = @_;

    my $time = time;
    my $host = get_curr_host();
    my $key = "_check_cache_status_${time}_${host}_$$";
    my $value = "value_${time}_${host}_$$";

    my $set_result = $client->set($key, $value, 60);
    return 0 if !$set_result;
    sleep 1;
    my $get_result = $client->get($key);
    return 0 if $get_result ne $value;
    return 1
}

1;
