package BM::Pages::Page;

use utf8;
use open ':utf8';

use std;
use base qw(ObjLib::ProjPart BM::MCached BM::Pages::PageText BM::Pages::PageSnippet);

use Data::Dumper;

use Encode;
use URI::_punycode;
use URI::_idna;

use URI::Escape;
use LWP::UserAgent;
use IPC::Open2;
use Digest::MD5 qw(md5_hex);
use FileHandle;
use Utils::Common;
use MIME::Base64;

use ObjLib::FileLogger;
use Utils::Sys;
use Utils::Urls qw(get_url_parts);

use Encode;

use JSON qw(to_json from_json);

use Scalar::Util qw(weaken);

use open ":utf8";
no warnings 'utf8';

########################################################
#Доступ к полям
########################################################

__PACKAGE__->mk_accessors(qw(
    name
    url
    url_filter
    inf
    timeout
));
#    site_proxy_ref


########################################################
# Интерфейс
########################################################

#  Работа с урлом

#    url                               урл страницы
#    domain                            домен, получаемый из урла
#    domain_2lvl                       домен второго уровня, получаемый из урла
#    domain_dir                        домен и директория
#    domain_path                       домен и путь до параметров (до [?#])

#    norm_url                          нормализованный урл (нужно для сравнений)
#    digits_norm_url                   нормализованный урл с заменой цифр на _digits_
#    fixed_url                         урл с частичным удалением незначащих параметров

#    is_bad_url_format                 Урл не является ссылкой на другую страницу (картинка, js и т.д.)
#    is_main_url                       Ссылка на главную страницу
#    is_search_url                     Ссылка на поиск
#    is_https_url
#    is_file_url

#    is_metalinks_url                  Служебный урл

#  / Работа с урлом

#    is_metalinks_text                 Служебный или широкий текст

#    norm_name                         нормализованный текст урла

#    pagetext                          текст страницы

#    split_url_for_tmpl                разбивает урл на массив элементов для шаблонов
#    compare_with_tmpl                 сравниваем урл с шаблоном

#    chname($name)                     вернуть объект с новым именем урла
#    churl($url)                       вернуть объект с новым урлом


#    language                          текущий язык, определяется настройками

#    get_minicategs                    получить категории страницы

#    is_brand                          ссылка ли это на страницу бренда
#    is_model                          ссылка ли это на страницу модели

# хандлеры сегнотатора
my $segin;
my $segout;

########################################################
# Инициализация
########################################################

our $badre = '';
our $badrearr = [];

# on these domains toponyms are ok - they are part of goods names
our %DOMAIN_WITH_OK_TOPONYMS = map { $_ => 1 } (
    'leonardo-stone.ru',  # SUPBL-2217
);


sub init {
    my ($self) = @_;

    $self->{url} ||= "";
    $self->{url} =~ s/(^\s+|\s+$)//g if $self->{url}; # убираем начальные и конечные пробелы
    $self->{url} = "http://".$self->{url} if $self->{url} !~ /^(?:https?:|ftp:)/i && ! $self->_metaprefix_url_check(lc($self->{url}));

    $self->load_wide_url_names unless $badre;

    $self->{'url_filter'} = [] unless $self->{'url_filter'};
}

########################################################
#Вспомогательные методы
########################################################

sub get_remotecache_id {
    my $self = shift;
    return md5int_base64( 'pagefunction' . $self->{url} . '_' . $self->{text});
}

sub cache_id :CACHE {
    my $self = shift;
    return md5int_base64( 'pagefunction' . $self->{url} . '_' . ($self->{text} // ''));
}

sub load_wide_url_names :GLOBALCACHE {
    my ($self) = @_;

    unless($badre){
        if($self->{opts} && $self->{opts}{wide_url_names}){
            open(F, $self->{opts}{wide_url_names});
            my @bad_urls = <F>;
            close(F);
            s/ =>.*$// for @bad_urls; #Удаляем перевод
#            $badre = join '|', grep {/\S/} map { s/^\s+|\s+$//g; $_ } @bad_urls;
#            $badre = join '|', grep {/\S/} map { s/^\s+|\s+$//g; $_ } @bad_urls;
            @bad_urls = grep {/\S/} map { s/^\s+|\s+$//g; $_ } @bad_urls; ##no critic
            $badrearr = [  map {lc($_)} @bad_urls ];
            my @beg_end = grep { /^\^.*\$$/ } @bad_urls;
            $badre = join '|', grep {! /^\^.*\$$/} @bad_urls;
            s/^\^|\$$//g for @beg_end;
            $badre .= ($badre ? '|' : '').'^(?:'.join('|',@beg_end).')$';
            $badre = lc($badre);
            $badre = qr/$badre/;

#print $self->_badre_reason('sdfsdfsd', 'http://yandex.ru')."<<<bad\n";
#print $self->_badre_reason('любая страна', 'http://yandex.ru')."<<<bad\n";
#print $self->_badre_reason(' любая страна', 'http://yandex.ru')."<<<bad\n";
#print $self->_badre_reason(' условия акции', 'http://yandex.ru')."<<<bad\n";
#print $self->_badre_reason('условия акции', 'http://yandex.ru')."<<<bad\n";

#print $badre."\n";
#exit;
        }
    }
    return 1;
}

sub get_badre :GLOBALCACHE {
    my ($self) = @_;
    $self->load_wide_url_names;
    return $badre;
}

sub color_regexp_old :GLOBALCACHE {
    my $psdcolors = 'мультиколор';
    my $modif = 'зелено|красно|черно|бело|желто|оранжево|малиново|сине|светло|темно|кирпично|розово|лимонно|лазурно|джинсовый|потертый|нежно|потерто|серебристо|кобальтово|коричнево|ярко|яко|дымчато|серо|шоколадный|ежевичный|кремовый|клетка|клетка сине|отделка|отд';
    my $colors = 'аква|белый|белая|белые|бежевые|бежевый|бирюзовые|бирюзовая|бирюзовое|бирюзовый|блестящие|бордовый|бордовые|бордовая|бордовое|бронзовые|бронзовый|бронзовая|бронзовое|ванильный|вишневый|голубые|голубая|голубая набивка|голубое|голубой|горчичный|горчично-желтый|графитный|желтый|желтое|желтые|золотистый|золотой|золотые|зеленые|зеленая|зеленый|зеленое|земляничный|изумрудный|индиго|камуфляжный|кремово-белый|коричневое|коричневые|коричневая|коричневый|коралловый|королевский синий|красные|красный|красные|красный.молочный|кремовый|кукурузный желтый|лавандовая набивка|\&quot;леопардовый\&quot;|леопардовые|леопардовая|леопардовый|лиловый|лимонный|лососевый|малиновый|меланжевый|металлик|молочно-белый|мятный|небесно-голубой|неоновый лососевый|оливковый|оранжевый|оранжевые|оттенки коричневого|оттенки кофейного|охра|песочный|песочные|песочная|песочное|пудровый|пурпурный|разноцветный|разноцветная|разноцветные|розовые|розовый|розовый неоновый|рыжие|рыжее|рыжая|рыжий|салатовый|салатовая|салатовое|салатовые|серебристый|серебряные|серебряный|серебряное|серебряная|серный|серые|серый|серая|серый меланжевый|серые|сиреневый|сиреневые|сиреневая|сиреневое|синие|синее|синяя|синий|синий &quot;потертый&quot;|синий потертый|сливовый|слоновая кость|терракотовый|телесный|фуксия|фисташковый|фиолетовое|фиолетовый|фиолетовые|хаки|хромовый|цвет белой шерсти|цвет бронзы|цвет голубого льда|цвет лайма|цвет мальвы|цвет морской волны|цвет розового золота|цвет слоновой кости|цвет [а-я]+|цветной|цикламен|черное|черные|черный|черный \&quot;леопардовый\&quot;|черный леопардовый|экрю|ягодный|яркие';
    my $postfix = 'в горох|в горошек|в клетку|в клеточку|в полоску|в полосочку|с цветочным рисунком|в цветочек|с цветами|с рисунком|с принтом|комбинированный|нефритовый|однотонный|\&quot;потертый\&quot;|&quot;варенка&quot;|варенка|с золотистой звездой|деним|кремовый|цветной|шоколадный|&quot;состаренный&quot;|состаренный|&quot;стираный&quot;|стираный|с &quot;леопардовым&quot; принтом|с матовым покрытием|с [^ ]+ рисунком|с рюшами|со стразами|с кружевом|на выпускной|с короткими рукавами|с рукавом летучая мышь|с вышивкой|на бретелях|с длинными рукавами|с баской|на молнии|с рукавом фонарик|с бантом|с тонкими бретельками|со складками|с рукавом реглан|с рукавами 3 4|с камнями|без бретелек|с завышенной талией|с воротником стойка|без рукавов|с карманами|кружевные|без подкладки|Несколько цветов|клетка серо-черная|клетка серо-синяя|синий горошек|(?:горошек|журавли|цветы)\s+на\s+(?:белом|сиреневом|сером|синем)';
    my $reclr = '(?:\d[xх]\s+|[Цц]вет\s*[-:]\s*)?(?:(?:(?:'.$modif.')[-/ ])*?(?:'.$colors.'|\/\/   дальше не цвета   \/\/|'.$postfix.'|'.$psdcolors.')\s*(?:[,\+\/\-и]\s*)?)+(?:\(\d+\))?';
    return $reclr;
}

sub wide_color_regexp_old :GLOBALCACHE {
    my ($self) = @_;
    my $re = $self->color_regexp;
    $re = '^(?:'.$re.')$';
    $re = qr/$re/;
    return $re;
}

sub wide_color_regexp :GLOBALCACHE {
    my ($self) = @_;
    return $self->proj->phrase->wide_color_regexp;
}

sub check_badre_text {
    my ($self, $text) = @_;
    $text ||= ''; #Убираем ворнинги на undef
    my $re = $self->get_badre;
    $text = lc($text) if $text =~ /[A-ZА-ЯЁ]/;
    return $1 if $text =~ /($re)/;
    my $clre = $self->wide_color_regexp;
    return $1 if $text =~ /($clre)/i; #Соответствие шаблону цветов
    return 0;
}

sub check_badre_text_debug {
    my ($self, $text) = @_;
    $text ||= ''; #Убираем ворнинги на undef
    my $re = $self->get_badre;
    $text = lc($text) if $text =~ /[A-ZА-ЯЁ]/;
    for my $sre (@$badrearr){
        #print "sre:'$sre'\n";
        #print Dumper([$text, $sre, ($text =~ /($sre)/)]);
        return "[ $1 => $sre ]" if $text =~ /($sre)/;
    }
    my $clre = $self->wide_color_regexp;
    return $1 if $text =~ /($clre)/i; #Соответствие шаблону цветов
    return '';
}

sub good_site_subsections :GLOBALCACHE {
    my ($self) = @_;
    my $file = $self->proj->options->{'site_subsections'};
    my $h = {};
    open(F, "<$file");
    while(<F>){
        chomp;
        $h->{$_}++;
    }
    close(F);
    return $h;
}

sub good_site_subsections_re :GLOBALCACHE {
    my ($self) = @_;
    my $file = $self->proj->options->{'site_subsections_re'};
    my $re = '';
    open(F, "<$file");
    while(<F>){
        chomp;
        $re .= '|' if $re;
        $re .= $_;
    }
    close(F);
    return $re;
}

sub good_site_subsections_re_check {
    my ($self, $name) = @_;
    my $re = $self->good_site_subsections_re;
    return $name =~ /$re/i ? 1 : 0;
}

#Фразы, которые могут быть как товарами, так и категориями
sub homonymy_names :GLOBALCACHE {
    my ($self) = @_;
    my $file = $self->proj->options->{'site_homonymy_names'};
    my $h = {};
    open(F, "<$file");
    while(<F>){
        chomp;
        $h->{$_}++;
    }
    close(F);
    return $h;
}

our $page_cache = {};
#Создание нового объекта страницы
sub page {
    my ($self, $url, $name) = @_;
    my $key = $url."\t".$name;

    # храним в кэше по урл + имя, но сравниваем также site_proxy_ref и user_agent
    if (defined $page_cache->{$key}) {
        my $cached = $page_cache->{$key};
        if (are_strings_equal($cached->{user_agent}, $self->{user_agent}) &&
                are_refs_same($cached->{site_proxy_ref}, $self->{site_proxy_ref})) {
            return $cached;
        }
    }

    my $p = $self->proj->page({
        url         => $url,
        name        => $name,
        user_agent  => $self->{user_agent},
        site_proxy_ref => $self->{site_proxy_ref},
    });
    # если страница уже есть в кэше с другим site_proxy_ref
    # или user_agent, то оставляем первую
    unless (defined $page_cache->{$key}) {
        $page_cache->{$key} = $p;
        weaken($page_cache->{$key});
    }

    return $p;
}

sub page_list {
    my $self = shift;
    my $pgl = $self->proj->page_list(@_);
    $pgl->site_proxy_ref($self->site_proxy_ref);
    return $pgl;
}

sub chname {
    my ($self, $name) = @_;
    return $self->page($self->url, $name);
}

sub churl {
    my ($self, $url) = @_;
    return $self->page($url, $self->name);
}

sub dbg {
    my ($self) = @_;
    return $self->{dbg} || ($self->site && $self->site->dbg);
}

#проверяет, не является ли первый из урлов вторым с добавленным хвостиком
sub _check_suburl {
    my $self = shift;
    my ($url1, $url2) = @_;
    return 1 if (substr($url2, 0, length($url1)) eq $url1);
    return 0;
}

sub _page_md5 :CACHE {
    my ($self) = @_;
    return md5_hex('_menu_arr'.encode_utf8("$self"));
}

########################################################
#Методы
########################################################

#Ссылка на объект Site
sub site_proxy_ref {
    my ($self, $val) = @_;
    if( defined $val ){
        $self->{'site_proxy_ref'} = $val;
        return $val;
    }
    return '' unless $self->{'site_proxy_ref'};
    return $self->{'p_subsites_cached'} if $self->{'p_subsites_cached'}; #если уже выделяли подсайт
    if(${$self->{'site_proxy_ref'}}->subsites){
        my $sbsite = ${$self->{'site_proxy_ref'}}->subsites->{$self->domain};
        if( $sbsite ){
#            print "spr ".$sbsite->domain."  ".$self->domain." ".$self->url."\n";
#            print $self->proj->stack_trace;
            $self->{'p_subsites_cached'} = $sbsite ? $sbsite->proxy_ref : $self->{'site_proxy_ref'};
            return $sbsite->proxy_ref;
        }
    }
    return $self->{'site_proxy_ref'};
}

#Сайт страницы, если указан
sub site {
    my ($self) = shift;
    my $sr = $self->site_proxy_ref;
    return $$sr if $sr;
    return '';
}

sub _metaprefix_url_check {
    my ($self, $url) = @_;
    return ($url =~ /^(?:["'])?(?:mailto|callto|tel|skype|javascript|tg|viber|whatsapp):/ ? 1 : 0);
}

sub is_metaprefix_url :CACHE {
    my ($self) = @_;
    return $self->_metaprefix_url_check($self->lc_url);
}

sub is_image_url :CACHE {
    my ($self) = @_;
    my $lc_url = $self->lc_url;
    return ($lc_url =~ /\.(?:jpg|gif|png|jpeg)(?:$|\?)/ ? 1 : 0);
}

sub is_file_url :CACHE {
    my ($self) = @_;
    my $lc_url = $self->lc_url;
    return ($lc_url =~ /\.(?:pdf|xls|xlsx|ppt|zip|rar|mp3|doc|docx|swf|odt)(?:$|\?)/ ? 1 : 0);
}

sub is_https_url :CACHE {
    my ($self) = @_;
    return  $self->url =~ /^https:\/\// ? 1 : 0;
}

sub protocol :CACHE {
    my ($self) = @_;
    if( $self->url =~ /^(https?)/ ){
        return $1;
    }
    return '';
}

sub domain :CACHE {
    my ($self) = @_;
    if( $self->url =~ /^(?:https?|ftp):\/\/([^@\/\?\#]*@)?([^\/\?\:\#]+\.[^\/\?\:\.\#]+)/ ){ #Удалось выделить домен
        my $domain = $2;
        return lc($domain);
    }
    return '';
}

sub domain_encoded :CACHE {
    my ($self) = @_;
    my $result = $self->domain;
    eval { $result = URI::_idna::encode($self->domain); };
    if($@) {
        $self->log("URI::_idna::encode: $@");
    }
    return $result;
}

sub uri :CACHE {
    my ($self) = @_;
    my $uri =  $self->url;
    $uri =~ s/^https?:\/\/([^\/\?\:]+\.[^\/\?\:\.]+)//; #Удаляем домен
    return $uri;
}

sub norm_uri :CACHE {
    my ($self) = @_;
    my $uri =  $self->norm_url;
    $uri =~ s/^https?:\/([^\/\?\:]+\.[^\/\?\:\.]+)//; #Удаляем домен
    return $uri;
}

sub regions_domains :GLOBALCACHE {
    my ($self) = @_;
    my $proj = $self->proj;
    my @arr = $proj->file($proj->options->{'site_region_domains'})->lines;
    return \@arr;
}

sub regions_domains_re :GLOBALCACHE {
    my ($self) = @_;
    my $arr = $self->regions_domains;
    my $lst = join('|', @$arr);
    $lst =~ s/\./\\\./g;
    return $lst;
}

sub domain_2lvl :CACHE {
    my ($self) = @_;
    my $domain = $self->domain;
    #my @arr = split /\./, $domain;
    #$domain = $arr[-2].'.'.$arr[-1] if @arr > 2; #Собираем домены 3-го уровня в одной директории
    my $re = $self->regions_domains_re;
    $domain = "$1.$2" if $domain =~ /([^\.]+)\.([^\.]{2,4}|$re)$/; #TODO нужно сделать правильный учёт технических доменов третьего уровня
    return $domain;
}

sub domain_path :CACHE {
    my ($self) = @_;
    if( $self->url =~ /^(https?:\/\/[^\?\#]+)/ ){ #Удалось выделить домен
        my $domain_path = $1;
        return $domain_path;
    }
    return '';
}

sub domain_dir :CACHE {
    my ($self) = @_;
    if( $self->url =~ /^(https?:\/\/[^\?\#]+)/ ){ #Удалось выделить домен
        my $domain_dir = $1;
        $domain_dir =~ s/\/[^\/]+$// if $domain_dir =~ /[^\/]\/[^\/]/;
        return $domain_dir;
    }
    return '';
}

sub lc_url :CACHE {
    my ($self) = @_;
    return lc($self->url);
}

#Нормализованный урл
#Может не открываться и не работать, используется для сравнения урлов
sub nrmurl {  #устаревшее
    my ($self) = @_;
    return $self->norm_url;
}

sub norm_url :CACHE {
    my ($self) = @_;
    my $url = $self->url;
    $url =~ s/^https:/http:/;
    if( $url =~ /^http\:\/\/(?:www\.)ozon\.ru/ ){
        $url =~ s/(?:store|sort)=[^=\&]+(?:\&|$)//g;
    }elsif($url =~ /^http\:\/\/(?:www\.)dom220v\.ru/){
        $url =~ s/\/catalog(\d+).htm$/\/category\/$1/;
    }elsif($url =~ /^https?\:\/\/(?:www\.)?yandex\.(?:ru|ua|kz|com\.tr)/){
        $url =~ s/\?clid=[-_0-9]+(?:&win=[-0-9])?$//;
    }
    $url =~ s/href=.+?(\&|$)//g; #Многие через многократное добавление параметра пишуть путь назад
    if($url =~ /([^\?]+)\?(.+)/){
        my $dmn = $1;
        my $prms = $2;
        $prms = join( '&', sort split( /\&/, $prms));
        $url = $dmn.'?'.$prms;
    }

    $url =~ s/\&{2,}/&/g; #Иногда получаются повторы

    #$url =~ s/((j?session.?id|sid|osCsid)=[A-Fa-f0-9]{10,})//g; #сессионные части ссылок
    $url =~ s/((j?session.?id|sid)=[A-Za-z0-9]{10,})//g; #сессионные части ссылок, используются все буквы, а не только A-F
    $url =~ s/i/y/g; #фикс для сайтов, где бывают расхождения в транслитерации
    $url =~ s/kh/h/g; #фикс для сайтов, где бывают расхождения в транслитерации
    $url =~ s/yu/u/g; #фикс для сайтов, где бывают расхождения в транслитерации
    $url =~ s/^http\:\/\///;
    $url =~ s/www\.//;
    $url =~ s/\#.*$//;
    $url =~ s/\/+/\//g; #Несколько слэшей превращаем в один
    $url =~ s/[\/\?]+$//;
    return $url;
}

sub digits_norm_url :CACHE {
    my ($self) = @_;
    my $t = $self->norm_url;
    $t =~ s/\d+/_digits_/g;
    return $t;
}

sub fixed_url :CACHE {
    my ($self) = @_;
    my $url = $self->url;
    $url =~ s/utm_(?:[_a-z0-9]+)=[^\&]+\&?//gi; #удаление меток
    $url =~ s/((j?session.?id|sid)=[A-Za-z0-9]{10,})//g; #сессионные части ссылок, используются все буквы, а не только A-F
    $url =~ s/[\?\&]$//;
    return $url;
}

sub is_bad_url_format :CACHE { #Урл не является ссылкой на другую страницу
    my ($self) = @_;
    my $url = $self->lc_url;
    #return 'bad01' if $url =~ /\.(jpg|gif|png|jpeg)$/; #Удаляем ссылки на картинки
    return 'image' if $self->is_image_url; #Удаляем ссылки на картинки
    return 'baddomain' if $url =~ /http:\/\/[\[\]\{\}\(\)'"]/;
#    return 'bad02: '."[$1]" if $text =~ /($badre)/i; #Выкидываем служебные урлы
    return 'css' if $url =~ /\.css(\?|$)/; #Убираем служебные урлы
    #return 'bad_mailto' if $url =~ /^mailto\:/; #Убираем служебные урлы
    #return 'bad_tel' if $url =~ /^tel\:/; #Убираем служебные урлы
    #return 'bad05' if $url =~ /javascript\:/; #Убираем служебные урлы
    return 'metaprefix' if $self->is_metaprefix_url;
    return 'levelup' if $url =~ /\.\.\//; #Возвраты на более верхний уровень категории - всегда обратно по иерархии
    return 'file' if $self->is_file_url;
    return 'backslash' if $url =~ /\\ /x;
    #return 'champagne' if ($text =~ /^champagne$/) && ($url =~ /^http:\/\/(?:www\.)?ozon\.ru/); #костыль для озона - это цвет
    return '';
}

sub is_main_url :CACHE {
    my ($self) = @_;
    my $dmn = $self->domain_2lvl;
    $dmn =~ s/[\[\]\{\}\(\)'"*]//g;
    my $mainre = '^https?:\/\/(www\.)?'.$dmn.'\/?$'; #Только для доменов второго уровня, так как для сайтов часто появляются поддомены
    return 'firstpage' if $self->url =~ /$mainre/;
    return '';
}

our @searchrules = (
     [ url_search    => qr/(search\.(php|aspx|asp|pl|cgi|py)|internalsearch)/ ],
     [ url_search2   => qr/\/(search)(\/|\?|$)/ ],
     [ url_search3   => qr/(http:\/\/search\.)/ ],
     [ url_search4   => qr/context=search\&/ ],
     [ url_search5   => qr/search_result/ ],
     [ url_search6   => qr/\?search_for=[^\&]+$/ ],
     [ url_search_tr => qr/\/(arama)(\/|\?|$)/ ],  #турецкий поиск
);
sub is_search_url {
    my ($self) = @_;
    my $url = $self->lc_url;
#    return 'recursion' if $url eq $cururl;
#    return 'exit' if $url =~ /---exit/;
#    my $mainre = '^http:\/\/(www\.)?'.$self->domain.'\/?$';
#    return 'firstpage' if $url =~ /$mainre/;
    for my $d (@searchrules){
        my ($nm, $re) = @$d;
        return "$nm: $1" if $url =~ /$re/;
    }
    return '';
}

sub is_metalinks_url :CACHE { #текст и урл должны быть приведены к нижнему регистру (оптимизация производительности)
    my ($self) = @_;
    my $url = $self->url;
    return "url_brand" if $url =~ /\?(?:firms?|vendors?)\%5B0\%5D=\d+/;
    return "mobile_site" if $url =~ /^https?:\/\/m\./;
    return "url_help" if $url =~ /^https?:\/\/help\./;
    return "mvideoreff" if ($url =~ /mvideo\./) && $url =~ /reff=menu_main$/;
    return "url1: $1" if $url =~ /\/(informacija|registration|forum|get_?file|sales|obzor|questions|producers|otzyvi|otzyvy|add|novosti|(\w+[-_]?)?news)(\?|\/|$)/;
    return "url2: $1" if $url =~ /\/(?<!videos\/)(all)\/(?!by_)/;
    return "url_logout: $1" if $url =~ /(logout\.(php|aspx|asp|pl|cgi|py))/;
    return "url_delivery" if $url =~ /\/delivery\//;
    return "url_login" if $url =~ /\/login\/(?:google|yandex|ok|facebook)\//;
    return "url_login2" if $url =~ /^https?:\/\/login\./;
    return "url_mail" if $url =~ /^https?:\/\/mail\./;
    return "url_contacts" if $url =~ /\/contacts\//;
    return "url_technology" if $url =~ /\/technology\/$/; #Обычно ссылки на описания технологий
    return "url_producer" if $url =~ /\/producers?(?:\-\d+)?\//; #Производитель
    return "url_photos" if $url =~ /\/photos\//;
    return "url_basket" if $url =~ /\/basket\.[a-z]{2,4}$/;
    return "url_rating" if $url =~ /[\?\&]rating=\d/;
    return "url_photos" if $url =~ /\/giftsMain\/?$/;
    return "url_faq" if $url =~ /\/faq\//;
    return "url_faqhtml" if $url =~ /[^A-Za-z]faq\.html?$/;
    return "url_vopros" if $url =~ /\/vopros\d*\//;
    return "url_shopinfo" if $url =~ /\/shopinfo\//;
    return "url_about" if $url =~ /\/about\/?$/;
    return "url_about" if $url =~ /\/about_us\/?$/;
    return "url_about" if $url =~ /\/o-kompanii\/?$/;
    return "url_badmedia: $1" if $url =~ /\/(author|actor)(\/|$)/;
    return "url_gittigidiyor: $1" if $url =~ /((\?|\/|\&)IS\d+_\d+=)/;
    return "url_gittigidiyor2: $1" if $url =~ /(ch_mod_list\.php)/;
    return "url_inceleme: $1" if $url =~ /^(http:\/\/inceleme\.)/; #турецкие обзоры
    return "url_profile: $1" if $url =~ /^(http:\/\/profile?\.)/;
    return "url_profile2" if $url =~ /\/Profile\//i;
    return "url_corp: $1" if $url =~ /^(http:\/\/corp?\.)/;
    return "url_discount: $1" if $url =~ /^(http:\/\/discount\.)/;
    return "url_properties: $1" if $url =~ /\?show\=properties$/;
    return "url_blog: $1" if $url =~ /^(http:\/\/blog\.)/;
    return "url_forum: $1" if $url =~ /^(http:\/\/(?:messageboards|(?:[a-z]{1,6})?forum)\.)/;
    return "url_journal" if $url =~ /\/journal\//i;
    return "url_news: $1" if $url =~ /(\/news\.(php|html)\.)/;
    #return "url_hepsiburada: $1" if $url =~ /(department\.aspx\?q\=)/ && $text !~ /^\d+$/;
    return "url_reviews: $1" if $url =~ /(ReviewsProduct\.aspx|productFullReview|\/reviews?\/)/;
    return "url_reviews: $1" if $url =~ /((?:Show)?UserReviews?)/i;
    return "url_reviews: $1" if $url =~ /((?:Show)?Attraction_Review?)/i;
    return "url_reviews" if $url =~ /\/reviews\//i;
    return "url_reviews" if $url =~ /iherb\.com\/r\//i;
    return "url_goodszone: $1" if $url =~ /(_z_z_z_z\.html)/;
    return "url_lngs: $1" if $url =~ /^(http:\/\/espanol\.)/;
    return "url_sale" if ($url =~ /-sale\//)&&($url !~ /technopoint\.ru/) ;
    return "url_userprofile: $1" if $url =~ /(user(review)?profile)/;
    return "url_wikimart: $1" if $url =~ /(wikimart\.ru.*\/(tag|brand|user)\/)/;
    return "url_mvideo" if $url =~ /^http:\/\/mvideo.ru\/.+\?reff=menu_main$/;
    return "url_brands: $1" if $url =~ /(\/(brands?|sellers)(\/|\.|$))/i;
    return "url_brands2: $1" if $url =~ /(\/(firms?|goods-producer)\/)/;
    return "url_brands3: $1" if $url =~ /[\&\?](kollektsiya|material|brend)\[/;
    return "url_brands4: $1" if $url =~ /\/(all_?brands?)\//;
    return "url_brands5" if $url =~ /[?&]brand=\d+$/;
    return "url_brands6" if $url =~ /(?<!velo-shop\.ru)\/brendy\//; # DYNAMICADS-236
    return "url_manufacturer_id" if $url =~ /[?&]manufacturer_id=\d+/;
    return "url_e5: $1" if $url =~ /(ref=p_bottom_near_nm)/;
    return "url_map: $1" if $url =~ /\/(map)\//;
    #return "url_sessions: $1" if $url =~ /((j?session.?id|sid)=[A-Fa-f0-9]{10,})/; #сессионные ссылки
    return "url_makemeheal: $1" if $url =~ /(alldepartments)/; #Вывод полного списка, что ломает иерархию
    return "url_makemeheal2: $1" if $url =~ /(makemeheal\.com\/(pictures|classifieds|videos|directory|u))(\/|$)/; #Картинки болезней, доска объявлений
    return "url_makemeheal2_: $1" if $url =~ /(makemeheal\.com\/.+\/(pictures|education))(\/|$)/;
    return "url_makemeheal3: $1" if $url =~ /(\/answers)(\/|$)/; #Ответы на вопросы, что-то вроде форума
    return "url_makemeheal4: $1" if $url =~ /(education\.makemeheal\.com)/; #Картинки болезней
    return "url_makemeheal5: $1" if $url =~ /(\/whats.?new)(\/|$)/; #Новинки
    return "url_vatanbilgisayar: $1" if $url =~ /((?:\&|\?)qs=\|)/;
    return "url_akakce: $1" if $url =~ /\/tum.magazalar/; #Все магазины
    #return "url_https" if $url =~ /^https/; #считаем урлы с https закрытимы
    return "url_goodsmatrix" if $url =~ /\/(goods-video|goods-commerce|goods-analogue|subject-directory|GMMap\.aspx)\//;
    return "url_goodsmatrix2" if $url =~ /goodsmatrix\.ru\/users\//;
    return "url_articles: $1" if $url =~ /(\/(articles|internet.?articles|useful.?information|our.?publications|internet.?digest)\/)/;
    return "url_photogallery: $1" if $url =~ /(\/(photo.?gallery)\/)/;
    return "url_ajax" if $url =~ /ajax/;
    return "url_glossary: $1" if $url =~ /(\/(glossary)\/)/;
    return "url_download" if $url =~ /ftpgetfile/;
    return "url_rozetka: $1" if $url =~ /\/c\d+\/.*\d+=\d+\/$/;
    return "url_iviaway" if $url =~ /away\/\?url\=http/;
    return "url_support" if $url =~ /\/support\//;
    return "url_support" if $url =~ /^http:\/\/(support)\.[-a-z0-9]+\.(?:ru|com)/;
    return "url_getGoodComments" if $url =~ /getGoodComments/;
    return "url_bs" if $url =~ /\/bs-click\//;
    return "url_iviperson" if $url =~ /ivi\.ru(?:\/temporary)?\/person\//;
    return "url_service_center" if $url =~ /\/service_center\//;
    return "url_forgot_password" if $url =~ /forgot_password|remind-password/;
    return "url_profile" if $url =~ /\/profile/;
    return "url_sharelink" if $url =~ /sharelink=(?:print|email|digg|vkontakte|bobrdobr|memori|twitter|facebook|delicious)/;
    return "url_mvideo:" if $url =~ /\/(?:grid-view|list-view)$/;
    return "url_help" if $url =~ /\/help\//;
    return "url_holodilnik_ch:" if $url =~ /^http:\/\/(?:[a-z]+\.)?holodilnik\.ru\/ch\//;
    return "url_holodilnik_subdomains:" if $url =~ /^http:\/\/(?!www\.)(?:[a-z]+\.)holodilnik\.ru\//; #Подавляем все поддомены для холодильника.ру
    return "url_holodilnik_ru_action:" if $url =~ /\/action\//;
    return "url_search" if $url =~ /[\?\&]query=[^\&]/;
    return "url_kak_vybrat" if $url =~ /\/kak_vybrat\//;
    return "url_archive" if $url =~ /\/archive\//;
    return "url_date" if $url =~ /y=20\d\d\&m=\d+\&gg=\d+/;
    return "url_questions" if $url =~ /[\?\&]\/questions/;
    return "url_filter"  if $url =~ /\/filter=\d+[,;:]\d+/;
    return "url_filter2" if $url =~ /COLLECTION[^\=\&\?]{0,5}\d+_\d+/;
    return "url_filter3" if $url =~ /numinpack=/;
    return "url_filter4" if $url =~ /features_hash=/;
    return "url_filter5" if $url =~ /100sporta\.ru/ &&  $url =~ /[a-z]\+[a-z]/; #Фильтры, которые сложно подавить алгоритмически
    return "url_filter6" if $url =~ /features_hash\[\]=/;
    return "url_filter7" if $url =~ /\?filters(\[\]|%5B%5D)?=/;  #mediamarkt.ru
    return "url_filter8" if $url =~ /mediamarkt\.ru/ && $url =~ /\/f_\d+_\d+/;
    return "url_perfekto" if $url =~ /perfekto\.ru\/catalog\/[^\/]+\/[^-\/]+\//;
    return "url_ulmart_mxp" if $url =~ /ulmart\.ru\/mxp\/|mxp\.ulmart\.ru/;
    return "url_brand" if $url =~ /division=.+&producer=\d+/; #Фильтр по брендам (deoshop.ru)
    return "url_term" if $url =~ /__term_\d+/; #Справочная информация (deoshop.ru)
    if( $url =~ /eldorado\.ru/ ){
        return "url_eldorado_filters" if $url =~ /\/cat\/\d+\/[-_A-Za-z0-9\.\|\'\`]+\// && $url !~ /\/page\//;
        return "url_eldorado_static" if $url =~ /static\.eldorado\.ru/;
        return "url_eldorado_static" if $url =~ /\?filter_/;
    }
    return 'url_yoox_brand' if $url =~ /^http:\/\/(?:www\.)?yoox\.com/ && $url =~ /_d$/;
    return "url_coolfilter" if $url =~ /coolfilter=/; #http://limpopo.com.ru
    if( $url =~ /ulmart\.ru/ ){
        return "url_ulmart_gifts" if $url =~ /\/gifts\//;
    }
    return 'url_roznica_filter' if $url =~ /roznica\.com\.ua.*\/[0-9_\-]+\/.+/;
    return 'url_alibaba_filter' if $url =~ /alibaba\.com.*-----(SIZE|ATTR|AREA|CNTRY)/;
    return 'url_blog' if $url =~ /\/blog\//;
    return 'url_sort' if $url =~ /(sort|orderBy)\=/;
    return 'url_tag' if $url =~ /product\-tag/;
    return 'url_tag2' if $url =~ /\/tag\//;
    return 'url_promotion' if $url =~ /\/promotions?\//;
    return 'url_promotion' if $url =~ /\/besplatnaya-dostavka/;

    return 'url_user' if $url =~ /\/[uU]sers?([^a-zA-Z]|$)/;
    return 'url_question' if $url =~ /\/[qQ]uestions?([^a-zA-Z]|$)/;
    return 'url_login' if $url =~ /\/[lL]ogins?([^a-zA-Z]|$)/;
    return 'url_contacts' if $url =~ /\/[kK]ontakty([^a-zA-Z]|$)/;
    return 'url_policy' if $url =~ /\/[pP]olitika([^a-zA-Z]|$)/;
    return 'url_sitemap' if $url =~ /\/[sS]itemap([^a-zA-Z]|$)/;
    return 'url_policy' if $url =~ /\/[pP]olicy([^a-zA-Z]|$)/;
    return 'url_policy' if $url =~ /\/[pP]rivacy[pP]?([^a-zA-Z]|$)/;
    return 'url_review' if $url =~ /\/[oO]bzor[iy]?([^a-zA-Z]|$)/;
    return 'url_review' if $url =~ /\/[oO]tz[iy]v[iy]?([^a-zA-Z]|$)/;
    return 'url_question' if $url =~ /\/[vV]opros[iy]?([^a-zA-Z]|$)/;

    # geo должен быть последний, чтобы при обходе гео-доменов (например moscow.gtshina.ru)
    # правильно распознавать metalinks
    return "url_geo: $1" if $url =~ /^https?:\/\/(shakhty|ivanovo|kaluga|kazan|kolomna|kostroma|krasnodar|lipetsk|michurinsk|morshansk|moscow|nnovgorod|novomoskovsk|novosibirsk|obninsk|rostovnd|russia|ryazan|sankt-peterburg|tambov|tula|tver|ufa|vladimir|vnovgorod|volgograd|voskresensk|yaroslavl|spb|saint-petersburg|belgorodskaya-obl|chechenskaya-resp|rostovskaya-obl|permskiy-kray|murmanskaya-obl|krasnoyarskiy-kray|primorskiy-kray|pskovskaya-obl|zabaykalskiy-kray|nsk|prm|ufa|tomsk|salavat|sochi|omsk|perm|yugra-ao|novgorodskaya-obl|kabardino-balkarskaya-resp|penzenskaya-obl|rostovskaya-obl|saratovskaya-obl|saha-yakutiya-resp|sahalinskaya-obl|ryazanskaya-obl|vladimirskaya-obl|volgogradskaya-obl|vologodskaya-obl|yamalo-nenetskiy-ao|sverdlovskaya-obl|mordoviya-resp|krasnodarskiy-kray|kostromskaya-obl)\.[-a-z0-9]+\.(?:ru|com)/;
    return "url_geo: $1" if $url =~ /^http:\/\/(strl\.koleso\.)/; #Фильтрация неявных регионов
    return '';
}

sub _split_url_for_tmpl {
    my ($self, $url) = @_;
    $url =~ s/-[a-z]+\d+-\d+$/\/_specgoodssfx/;
    return map { s/^(\D|[-A-Za-z0-9]+=)?\d+$/($1 || '').'_digits'/e; $_ } split( /[\/\?\&]/, $url ); ##no critic
}

sub split_url_for_tmpl {
    my ($self) = @_;
    return $self->_split_url_for_tmpl($self->norm_url);
}

#Возвращаем массив для  шаблона, если есть бренд, заменяя бренды на обозначение
#Если нет, то возвращает пустой список !!!
our $brndfxd = [qw{catalog}];
$brndfxd = { map {$_=>1} @$brndfxd};
sub _get_split_url_for_tmpl_with_brands {
    my ($self) = @_;
    my $brflt = $self->proj->phrase->_get_brands_hash;
    my @lst = $self->split_url_for_tmpl;
    if(grep { $brflt->{$_} } @lst){
        my @nlst = map { ($brflt->{$_} && ! $brndfxd->{$_}) ? '_brand_' : $_ } @lst;
#        my @nlst = map { $brflt->{$_} ? '_words' : $_ } @lst;
        return \@nlst;
    }
    return ();
}

sub compare_with_tmpl { #Добавляем кэширование для проверок соответствия шаблону
    my ($self, $tmpl) = @_;
    $self->{ compare_with_tmpl_cache } ||= {};
    my $tmplcache = $self->{ compare_with_tmpl_cache };
    return $tmplcache->{$tmpl} if defined($tmplcache->{$tmpl});
    $tmplcache->{$tmpl} = $self->_compare_with_tmpl($tmpl);
    return $tmplcache->{$tmpl};
}

sub _compare_with_tmpl {
    my ($self, $tmpl) = @_;
    my @a1 = $self->_split_url_for_tmpl($tmpl);
    my @a2 = $self->split_url_for_tmpl;
    return 0 if @a1 != @a2;
    for my $i ( 0 .. @a1 - 1){
        next if ($a1[$i] eq '_digits') && $a2[$i] =~ /^\d+$/;
        next if ($a1[$i] eq '_words') && $a2[$i] =~ /^[-_A-Za-zА-Яа-яёЁ0-9]+$/;
        next if ($a1[$i] eq '_brand') && $a2[$i] =~ /^[-_A-Za-zА-Яа-яёЁ0-9]+$/;
        return 0 if $a1[$i] ne $a2[$i];
    }
    return 1;
}

#######################################################################
# / Работа с урлом
#######################################################################

our $url_name_types_table = ''; #url_name_types
sub web_urlname_filter :GLOBALTIMECACHE(10) { #Регулярно обновляемые данные из таблицы в базе
    my ($self) = @_;
    return {} unless $url_name_types_table;
    return { map { $_->{text} => $_->{vtype} } grep {$_->{vtype}} map{@$_} $self->proj->dbtable($url_name_types_table)->List };
}

sub is_web_wideurlname :CACHE {
    my ($self) = @_;
    return $self->web_urlname_filter->{$self->name};
}

sub is_metalinks_text :CACHE {
    my ($self) = @_;
    my $text = $self->name;
    #return "url_favorites" if $text =~ /^закладки/ && $url =~ /favorites/;
    #return "bad: $1" if $text =~ /($badre)/;
    return $self->is_web_wideurlname if $self->is_web_wideurlname;
    return $self->check_badre_text($text);
}

sub is_metalinks_text_debug {
    my ($self) = @_;
    return $self->is_web_wideurlname if $self->is_web_wideurlname;
    return $self->check_badre_text_debug($self->name);
}

#является ли страница листалкой
sub is_a_pager :CACHE {
    my ($self) = @_;
    return 0 if $self->name =~ /^\d{5,}$/; #крупные номера могут быть артикулами
    return 0 if $self->name =~ /^\d{3}[1-9]$/; #Крупные номера, не заканчивающиеся на 0
    return 0 if $self->url =~ /perpage/;
    return 0 if $self->url =~ /list_num=/; # eldorado.ru
    return 1 if $self->name eq '_pagerurl_';
    return 1 if $self->name =~ /^\s*\d+,\s*$/; # 1, 2, 3, 4
    return 1 if $self->name =~ /^(\s*\[\s*)?\d+(\s*\]\s*)?$/;
    return 1 if $self->name =~ /^(\s*\[\s*)?([A-Z]|[А-Я])(\s*\]\s*)?$/; # заглавная латинская или русская буква
    return 1 if $self->name =~ /^(\s*\[\s*)?\d+\-\d+(\s*\]\s*)?$/; # 61-68 на некоторых сайтах листалки диапазонами
    return 1 if $self->name =~ /^(?:Страница)\s*\d+$/i; #
    return 1 if $self->name =~ /^Следующий$/i && $self->url =~ /page=/; #
    return 1 if $self->name =~ /^Следующие \d+ страниц$/i; #
    return 1 if $self->name =~ /^Перейти на станицу \d+$/i; #
    return 1 if $self->name =~ /^(?:ВПЕРЕД|НАЗАД)$/i; #
    return 1 if $self->name =~ /^(?:\&lt;\&lt;|\&gt;\&gt;)$/i; #
    #Tümü  - турецкое "все"
    return 1 if $self->name =~ /^\s*(отобразить|показать|вывести|посмотреть|смотреть)\s+(вс[её]|ещ[её]|все модели)\W*(\&gt\;\&gt\;)?$/i
                && $self->url !~ /\/category\//i #Убираем явную ссылку на другой раздел
                && $self->url !~ /\/\/[^\/]+(?:\/[-a-z]+)?$/i #Убираем явную ссылку на другой раздел
          ;
    return 0;
}


# язык страницы
sub language :CACHE {
    my ($self) = @_;

    return $self->proj->get_language($self->{lang}) if $self->{lang};
    return $self->{language} if $self->{language};
    return $self->proj->default_language;
}


sub norm_name :CACHE {
    my ($self) = @_;
    my $text = lc($self->name);
    $text =~ s/^\s+|\s+$//g;
    $text =~ s/\s{2,}/ /g;
    return $text;
}

sub is_brand :CACHE {
    my ($self) = @_;
    return 1 if $self->is_brand_exactly;
    return 0;
}

sub is_brand_exactly :CACHE {
    my ($self) = @_;
    my $text = $self->name;
    $text =~ s/\(\d+\)\s*$//;
    return $self->proj->phrase($text)->is_brand;
    return 0;
}

#Является ли эта страница страницей модели
sub is_model :CACHE {
    my ($self) = @_;
    return 'url_wikimart' if $self->url =~ /(wikimart\.ru.*\/(model|seller-model)\/)/i;
    return 'url_gittigidiyor' if $self->url =~ /^http:\/\/urun\.gittigidiyor\.com/;
    return 'url_hepsiburada' if $self->url =~ /productDetails/i;
    return 'url_makemeheal' if $self->url =~ /\/product\.do\?/;
    return 'url_goodsmatrix' if $self->url =~ /goodsmatrix.ru\/goods\/(.\/)?\d+\.html/i;
#    return 1 if $self->proj->phrase($self->name)->get_brand;
#    return 1 if $self->get_model_like_phrases;
    return 1 if $self->proj->phrase($self->name)->has_model;
    return 0;
}

sub get_model_like_phrases :CACHE {
    my ($self) = @_;
    my @arr = ();
#    my $title = $self->title;
    my $name = $self->name;

    my $ph = $self->proj->phrase($name);

    return ($ph->_get_model_like_phrases,);
}

####################################################################################
# Категории страницы
# Так как есть вероятность не скачать страницу,
# критичный к этому код может использовать get_minicategs_with_retries
# filecache один общий на три метода
####################################################################################

sub get_minicategs {
    my ($self) = @_;
    return @{$self->get_minicategs_with_status->{categs}};
}

sub get_minicategs_with_status :FILECACHE(864000) {
    my ($self) = @_;
    return {
        categs => [$self->language->pages_categories->categorize_page($self)],
        success => $self->one_line_text ? 1 : 0,
    };
}

#метод отключает чтение из filecache и делает несколько ретраев, если страницу не удалось скачать. Если удалось - выводит сохраненное в filecache
sub get_minicategs_with_retries {
    my ($self) = @_;
    my $data = $self->get_minicategs_with_status;
    my $retries = 3;
    while ( !$data->{success} && $retries-- > 0 ) {
        #сбрасываем локальный кэш, создавая новый объект
        my $temp_page = $self->proj->page($self->url, $self->name);
        $temp_page->{dont_read_from_filecache} = 1;
        $data = $temp_page->get_minicategs_with_status;
    }
    return $data;
}

sub get_minicategs_without_hrefs :CACHE {
    my ($self) = @_;
    return $self->language->pages_categories->categorize_page($self, dont_use_hrefs => 1, );
}

sub get_minicategs_debug_inf {
    my ($self) = @_;
    return $self->language->pages_categories->categorize_page_debug_inf($self);
}


sub get_minicategs_log :CACHE {
    my ($self) = @_;
    my $res = Dumper([$self->language->pages_categories->categorize_page($self)]);
    $res .= 'lang:'.$self->lang."\n";
    return $res;
}

sub get_categs_subphrases :CACHE {
    my ($self) = @_;
    return $self->language->pages_categories->get_page_categs_subphrases($self->one_line_text);
}

# параметры $par:
# no_hier = 1: не дописывать иерархию категорий в результат, по умолчанию no_hier = 0

# пример:
# no_hier = 1:
# { 'детский лагерь' => {
#                           'categs' => {
#                               'Детские лагеря' => 1
#                           } ...
# no_hier = 0:
# { 'детский лагерь' => {
#                           'categs' => {
#                               'Отдых' => 1,
#                               'Детский отдых' => 1,
#                               'Услуги' => 1,
#                               'Детские лагеря' => 1
#                           } ...
sub get_categs_subphrases_with_param {
    my ($self, $par) = @_;
    if (($par) and (ref $par eq 'HASH')) {
        return $self->language->pages_categories->get_page_categs_subphrases($self->one_line_text, %$par);
    }
    else {
        return $self->language->pages_categories->get_page_categs_subphrases($self->one_line_text);
    }
}

####################################################################################
# / Категории страницы
####################################################################################

sub is_catalog_url :CACHE {
    my ($self) = @_;
    #print Dumper([$self->url, $self->uri, ($self->uri =~ /^\/?catalog(?:ue)?\/?$/)]);
    return 1 if $self->name =~ /^(?:каталог(\s+товаров)?|Полный каталог)$/i;
    return 1 if $self->uri =~ /^\/?cat(?:alog(?:ue)?)?\/?$/; #Главная страница каталога
    return 0;
}

sub get_type :CACHE {
    my ($self) = @_;
    return 'catalog' if $self->is_catalog_url; #Главная страница каталога
    return 'pager' if $self->is_a_pager;
    return 'brand' if $self->is_metalinks_url =~ /url_brand/;
    my $text = $self->name;
    my $cltext = $text;
    $cltext =~ s/\s*(?:\[\d+\]|\(\d+\))\s*$//;
    $cltext =~ s/^\s+|\s+$//g;
    $cltext =~ s/\s+/ /g;
    return '' if $self->good_site_subsections->{$cltext}; #Прокидываем названия из разрешённого списка
    my $url = $self->url;
    $text =~ s/\s*\(\d+\)\s*$//;
    $text =~ s/Артикул ?: ?\w+//;
    #Словарь исключений для брендов, которые часто идут однословными текстами
    return 'service' if $text =~ /^(?:услуги|установка|настройка|обслуживание|Подписка на|Новая сервисная услуга)(?:\s|$)/i; #Услуги
    return 'category' if $url =~ /category\.[^\/]+$/i;
    my $widebrands = 'lg|FAIRY|hq|nb|BON|Energy|FINISH|Фея|DON|Thermos|EXE|LED|iPad|CRYSTAL|ЗЕБРА|SMART|СТАРТ|РЫЖИЙ КОТ|df|DELUXE|АТЛАНТ|SN-T|GF|Park|TITAN|4GO|OZ|NZ|RS|WD|Twelve|SBS|НИКА|Jet|BIBER|KERAMIKA|ТАЙФУН|ТОП ХАУС|ФБ|UDI|AVS|GRANIT|CLASSIC|ТРАФФИК|КОНСТРУКТ|НИКОЛЬ|A4|Flat|White Diamonds|GALAXY|PILOTAGE|TURBO|ADA|МАРТ|ЧАЙКА|[A-Za-z]+\&(?:amp\;)?[A-Za-z]+|Тек А Тек|Беларус|ШТОК|КОСМОС|Cicle|Legioner|Testo|PALISAD|3М|SPARTA|АТАКА|НПП|Вепрь|КВТ|ДУГА|TDM|Watts|STP|OT|Tytan|Global|LUX|Master|ME|MS|P|РОС|Спец|Вымпел|Мега|Тропик|НИЗ|СИЛА|Угра|МЕГА|GAV|Калибр|Texas|Princess|Worth|Заря|Crema|Combat|OZONE|GOK|Борино|Печкин|ЖМЗ|Garage|Качок|Тор|ТРАКТ|КОБАЛЬТ|SUIT|Freequent|CAT|OK Baby|Cam|ИНВЕНТ|СОКОЛ|Алмаз|Гонец|SHTOK|МАСТАК|Sata|Technical|Allen|Dallas|БИЗОН|Lotos|Стэк|Stream|Яркий луч|Penguin|КРЕП|Fly|ASFALTO|MOUSTACHE|DOMESTIC|PowerPlant|VW|GENERAL|GP|Atlet|Expert|BQ|Weekday';
    return 'brand_exactly' if $text =~ /^(?:$widebrands)$/i;
    return 'model' if $text =~ /№\s*\d+/;
    return 'model' if $text =~ /^(?:$widebrands)\s+[^ \(\[]/i;
    return 'model' if $url =~ /__m\d+\.html/;
    return 'model' if $url =~ /\/product\//;
    return 'brand' if $url =~ /[\?\&]brandID=\d+/i;
    return 'brand' if $url =~ /lamoda\.ru\/c?b\//;
    return 'lang'  if ($url =~ /\/\/(fr|ru|nl|he|ja|pt|id|ko|it|us|de|ar|tr|es|vi|th)\./) && $text =~ /^(?:Тайский|Вьетнамский|Испанский|Русский|Иврит|Японский|Нидерландский|Турецкий|Арабский|Немецкий|Французский|Португальский|Итальянский|Корейский|Индонезийский|Russian|French|German|Italian|Portuguese|Spanish|Turkish|Arabic|Japanese|Korea|Thai|Vietnam|Brasil|Español|Français|Deutsch|Italiano|日本語|한국어|Nederlands|اللغة العربي|עברית|tiếng Việt|ภาษาไทย|Türk|Polska)(?:\s+язык)?$/i ;
    if ($self->site) {
        my $specre = $self->site->special_goods_url_filter_re;
        return 'model' if $specre && $url =~ /$specre/;
    }
    my $proj = $self->proj;
    #После фразы про мебель слово с большой буквы
    return 'model' if $text =~ /^(?:Автокресла|Бра|Гостиная|Гостиная модульная система|Диван(?:(?:\s+|\-)(?:прямой|кожаный|книжка|тканевый|кровать))*|Журнальный столик(?:\s+(?:приставной|журнальный))*|Зеркало|Каркасная кровать|Комод(?:\s+(?:в прихожую|для кухни|в спальню|в коридор|\d+ ящ\.|\d+ ящиков))*|Конференц-кресло|Кресло(?:\s+(?:кожаное|тканевое))*|Кровать(?:\s+(?:односпальная|2-х ярусная|\d+х\d+))*|Кровать-чердак|Кресло для оператора|Кресло для руководителя|Кухонный комплект|Кухонный гарнитур|Кухня|Люстра|Матрас|(Анатомический|Детский) матрас|Мебель для гостиной|Подвесная люстра|Пуф|Полка(?:\s+(?:навесная|со стеклом))*|Мягкая кровать|Пенал|Садовые качели|Светильник|Спальня|Стеллаж(?:\s+(?:с крючками|с полками|торцевой))*|Стенка|Стол(?:ик)?(?:(?:\s+|\s?\-\s?)(?:трансформер(?:\s+\d)?|обеденный|раздвижной|приставной|со стеклом|книжка(?:\s*\d)?|круглый|с хромированными ножками|туалетный|журнальный|письменный))*|Стул(?:\s+(?:на металлокаркасе))*|Спот|Табурет|Топпер|Тумба для обуви|Тумба прикроватная|Тумба|Угловой стеллаж(?:\s+(?:с крючками|с полками))*|Угловой диван|Шкаф(?:\s+(?:угловой|левый|правый|с зеркалом))*|Шкаф-витрина|Шкаф[- ]купе(?:\s+(?:\d-х дв(?:ерный|\.)?|с зеркалом|в прихожую))*)(?:\s+(?:большой|большое|большая|в прихожую|для прихожей|в спальню|для спальни|в коридор|в кухню|на кухню|для кухни|складной|из дсп))*\.?(?:\s*\-\s*\d+)?\s+(?:(?:\&laquo;|["«'])?[А-ЯA-Z])/i;
    #После фразы про товары для дома и с большой буквы
    return 'model' if $text =~ /^(?:Бязь набивная|Постельное белье|Семейный комплект постельного белья|Подушка|Евро одеяло|Двуспальное одеяло|Полутороспальное одеяло)*\.?(?:\s*\-\s*\d+)?\s+(?:(?:\&laquo;|["«'])?[А-ЯA-Z])/i;
    #После фразы про торговое оборудования и с большой буквы
    return 'model' if $text =~ /^(?:Платежный терминал|Мотобуксировщик|Весы с принтером этикеток|Платёжный терминал|Платформенная тележка|Детектор банкнот|Упаковщик банкнот|Сортировщик банкнот|Счетчик банкнот|Темпо-касса|Тележка для баллонов|Тележка|Грузовая тележка|Драйвера|Весы-рокла|Кассовый аппарат|Кассовая система|Фронт-система|Контейнер сетчатый|Контейнер трубчатый|Тележка хозяйственная|Бочкокантователь штабелер|Сканер штрих-кодов|Принтер чеков|Дисплей покупателя|Денежный ящик)*\.?(?:\s*\-\s*\d+)?\s+(?:(?:\&laquo;|["«'])?[А-ЯA-Z])/i;
    #После фразы про ремонт и слово с большой буквы
    return 'model' if $text =~ /^(?:Обои ?(?:флизелиновые|виниловые))*\.?(?:\s*\-\s*\d+)?\s+(?:(?:\&laquo;|["«'])?[А-ЯA-Z])/i;
    #После фразы про игрушки и слово с большой буквы
    return 'model' if $text =~ /^(?:Сани|Шлем|Велочоппер|Велосипед(\s+двухколесный|\s+трехколесный)?|Электровелосипед|Экомобилик|Электроскутер|Электроскейт|Стендбайк|Комплект Мотор-колесо|Электромобиль|Мопед|Гироскутер|Мотоцикл|Гольф-кар|Игровой домик|Динозавр|Детеныш|Мозаика|Фигурка|Стол-стул для кормления|Кроватка|Манеж|Манеж-кровать|Санки|Санки-коляска)*\.?(?:\s*\-\s*\d+)?\s+(?:(?:\&laquo;|["«'])?[А-ЯA-Z])/i;
    #Для продуктовых сайтов
    return 'model' if $text =~ /\D[\d\.]{2,4}x[\d\.]{2,4}x[\d\.]{2,4}(\D|$)/;
    return 'model' if $text =~ /\D[\d\.]{2,4}х[\d\.]{2,4}х[\d\.]{2,4}(\D|$)/;
    return 'model' if $text =~ /\D[\d\.]{2,4}\*[\d\.]{2,4}\*[\d\.]{2,4}(\D|$)/;
    return 'model' if $text =~ /.{7}\d\s*(л|шт\.?\s*упаковка|шт\/уп|пакетиков|шт|гр|кг|см|cм|литров|мм|мкм)(?:[ \.\,\/\)]|$)/i; #Единицы измерения - хороший признак конкретного товара, а не категории
    return 'model' if $text =~ /^([a-z]+\s+)?(?:чехол|чехол-клавиатура|Сумка|Аксессуар|Кронштейн|Освежитель|Стиральная машина|Сумка-холодильник|Планшет|Подставка|Коврик|Настенное крепление|Разъем для|Умные часы и браслет|фляжка|Патронташ|Набор|Рюкзак|Рюкзак-трансформер)\s/i; #Признак модели по числу
    return 'model' if $text =~ /^(?:велосипед)\s*(?:[a-z]+|\d{4})/i; #Для велосипедов часто указывают только модель
    my $ph = $proj->phrase($text);
    my %h = $ph->parse;
    return 'model' if $h{model}; #Содержит модель
    return 'model' if $h{model_for}; #Что-то для конкретной модели
    return 'model' if $h{brand} && $text =~ /[a-z]{2}\d{2}/i; #Временное решение, чтобы дождаться исправления в парсере !!!
    return 'model' if ($h{brand} || $text =~ /[«"]/ || $ph->has_color) && $text =~ /^(?:бандана|бейсболка|блузка|боди|ботфорты|брюки|бриджи|бюстгальтер|варежки|ветровка|водолазка|гольфы|джемпер|джинсы|джинсовая куртка|дождевие|домашний комплект|жакет|капри|кардиган|кеды|кимоно|комбинезон|комплект носков|конверт|корсаж|костюм|кофта|колготки|комбинация|комплект белья|комплект одежды|комплект верхней одежды|кроссовки|купальник|куртка|Леггинсы|лонгслив|майка|носки|пальто|пиджак|пинетка|платье|плавки|плащ|ползунки|поло|пояс|пуловер|пуховик|полукомбинезон|распашонка|ремень|рубашка|сарафан|(?:полу)?сапоги|(?:полу)?ботинки|ботильоны|свитер|свитшот|Спортивный костюм|термоноски|термобельё|термобелье|толстовка|топ|трусы|туника|туфли|футболка|худи|чепчик|чулки|шапка|шарф|шорты|штанишки|юбка|портупея|корсаж)(?: |$)/i; #Одёжные товары
    # Детский товар + цвет
    return 'model' if $ph->has_color && $text =~ /детск(?:ий|ая|ое|ие)/i;
    my $gnc = $ph->get_gender_number_case;
    my $number = $gnc->{number};
    return 'model' if $h{brand} && $number && $number eq 'sg' && $ph->is_subcategory("Кухонные посуда и принадлежности");
    return 'model' if $h{brand} && $number && $number eq 'sg' && $ph->is_subcategory("Техника для кухни");

    my $case = $gnc->{case};
    # продается конкретный товар, если есть бренд, при **строгом** парсе найден type, и текст в именительном падеже и единственном числе
    if ( $h{brand} && $number && $number eq 'sg' && $case && $case eq 'nom' ) {
        my %strict_h = $ph->parse(use_goods_basis => 0);
        return 'model' if $strict_h{type};
    }

    return 'brand_exactly' if $h{brand} && ($ph->norm_phr eq $proj->phrase($h{brand})->norm_phr);
    return 'model' if $h{brand} && $text =~ /\d{3} (?:р\.|уб)/; #Содержит бренд и цену
    return 'model' if $h{brand_for} && $text =~ /\d{3} (?:р\.|уб)/; #Содержит бренд и цену
    return 'brand' if $h{brand}; #Содержит бренд
    return 'brand' if $h{brand_for}; #Содержит бренд
    return 'qstn'  if $self->name =~ /\?\s*$/;
    return 'personal_ads' if $text =~ /^(?:продам|куплю|ищу)/i;

    my $probably_toponym = $h{toponyms} ||
                           $text =~ /^(?:Россия|РФ|RUSSIAN FEDERATION|KAZAKHSTAN|UKRAINE|SWITZERLAND|QATAR|ALGERIA|CROATIA|DOMINICAN REPUBLIC|PARAGUAY|CZECH REPUBLIC|PHILIPPINES|AZERBAIJAN|NORWAY|SURINAME|CYPRUS|LITHUANIA|POLAND|COLOMBIA|ICELAND|TURKEY|LIBERIA|BOSNIA AND HERZEGOVINA|MACAU|THAILAND|PERU|LEBANON|OMAN|DENMARK|ROMANIA|MALTA|HONG KONG|NEW ZEALAND|SINGAPORE|TUNISIA|CHILE|PORTUGAL|ALBANIA|SOUTH AFRICA|UNITED STATES|SOUTH KOREA|UNITED ARAB EMIRATES|UNITED KINGDOM)$/i ||
                           $text =~ /^\S+\s+область|Областя$/i ||
                           $text =~ /^Крим$/i ||
                           $url =~ /region\?city=/i ||
                           $url =~ /action=changeCity/i;
    # это топонимный урл, только если не категоризуется во что-то нормальное
    if ( $probably_toponym && !$ph->is_subcategory('Товары') && !$ph->is_subcategory('Услуги') && !$ph->is_subcategory('Оборудование') 
        && !$DOMAIN_WITH_OK_TOPONYMS{$self->domain_2lvl}
    ) {
        return 'toponyms';
    }
    return '';
}

sub links_to_himself {
    my ( $self ) = @_;
    my $pgl = $self->get_subpages;
    $pgl = $pgl->lgrep(sub { $_->norm_url eq $self->norm_url });
    $pgl = $pgl->lgrep(sub {! $_->is_metalinks_url });
    $pgl = $pgl->lgrep(sub { $_->name !~ /^\d+$/ }); #Удаляем ссылки листалок
    return $pgl;
}

#Пытаемся получить название ссылки по ссылкам на самой странице
sub get_pagename : CACHE {
    my ( $self ) = @_;
    my $pagename = $self->name;
    unless($pagename){ #пробуем получить имя ссылки на страницу
        #my $allpgl = $self->get_subpages_pgl->pgrep(sub {! $_->bad_page_reason  }); #хорошие, но без фильтрации по самому урлу
        #$allpgl = $allpgl->pgrep(sub { $_->name !~ /^\d+$/ }); #Удаляем ссылки листалок
        my $nmpgs = $self->links_to_himself;  #$allpgl->pgrep(sub { $_->norm_url eq $self->norm_url });
        unless($nmpgs->size){
            #$allpgl = $self->get_internal_subpages_pgl->rand_items(30);
            my $allpgl = $self->get_internal_subpages;
            #ограничиваем число страниц
            $allpgl = $allpgl->nameurl_pack_urls->lsort(sub {$_->name})->distributed_elems(30);
#print "rand:\n$allpgl\n====================\n";
#print "url: $_\n".$_->get_internal_subpages_pgl->pgrep(sub { $_->norm_url eq $self->norm_url })."\n====\n\n"  for @$allpgl;
            $allpgl = $allpgl->lmap(sub { @{ $_->get_internal_subpages->pgrep(sub { $_->norm_url eq $self->norm_url }) } });
#            $allpgl = $allpgl->pgrep(sub {! $_->bad_page_reason  });
#            $nmpgs = $allpgl->pgrep(sub { $_->norm_url eq $self->norm_url });
            $nmpgs = $allpgl;
#            $nmpgs = $nmpgs->nameurl_pack_urls;
            my %hh = ();
            $hh{$_->name}++ for @$nmpgs;
            my %flt = map { @$_ } h2top(\%hh, 0.6, 0.4);
            $nmpgs = $nmpgs->pgrep(sub { $flt{$_->name} });
#            print Dumper(h2top(\%hh, 0.6, 0.4));
            $nmpgs = $nmpgs->nameurl_pack_urls;
        }
#print "$_\n".$_->norm_url."\n" for @$allpgl;
#print "$_\n".$_->norm_url."\n" for @$nmpgs;
        $nmpgs = $nmpgs->pack_urls;
        $pagename = $nmpgs->[0]->name if @$nmpgs;
#        $pagename ||= 'PRTNNAME';
    }
    return $pagename;
}

sub cleared_text {
    my ( $self ) = @_;
    my $proj = $self->proj;
    my $text = $self->text;
    my $mnpg = $proj->page('http://'.$self->domain);
    if($mnpg->norm_url ne $self->norm_url){ #Очищаем повторяющиеся строки с первой страницы из обрабатываемого текста
        my $mptext = $proj->page('http://'.$self->domain)->text; #Текст первой страницы
        my $mpflt = { map {$_=>1} split("\n", $mptext) };
        $text = join("\n", grep { ! $mpflt->{$_} } split("\n", $text) );
    }
    $text =~ s/\<(head|style|script|noscript)[^\>]*\>.*?\<\/\1\>//ig;
    $text =~ s/\<\/?[a-z]+[^\>]*\>/ /g;
    $text =~ s/\s+/ /g;
    return $text;
}

sub page_phrases {
    my ( $self ) = @_;
    my $proj = $self->proj;
    my $text = $self->cleared_text;
    my $tph = $proj->phrase( $text );

    my $phl = $tph->subphrases_list;
    $phl = $phl->good_phrases_list;
    $phl->cache_bnr_counts->cache_search_query_count->cache_search_count->cache_cdict_minicategs;
    my $ttlph = $proj->phrase($self->title);
    my $flt = { map {$_ => 1} $ttlph->notwidewords };
    my $arr = [ map {"$_"}
        grep { grep { $flt->{$_} } $proj->phrase( $_.' '.join(' ', $_->get_minicategs))->snormwords }
        grep { $_->get_search_query_count / $_->get_search_count > 0.001 }
        grep { $_->get_bnr_count > 10 }
        grep { $_->get_search_query_count > 100 }
        grep { $_->get_search_count > 100 } @$phl ];

    return $arr;
}

sub page_phrases2 {
    my ( $self ) = @_;
    my $proj = $self->proj;

    my $text = $self->cleared_text;
    my $tph = $proj->phrase( $text );
    my $bgflt = { map { $_=>1 } $tph->not_wide_words_with_biwords };

    my $phl = $tph->subphrases_list;
    $phl = $phl->good_phrases_list;
    my $pgflt = { map { $_->snorm_phr => 1 } @$phl }; #Хэш фраз, которые встречались на странице
    $phl->cache_bnr_counts->cache_search_query_count->cache_search_count->cache_cdict_minicategs;
    my $ttlph = $proj->phrase($self->title)->get_banner_prefiltered_phrase;
    my $flt = { map {$_ => 1} $ttlph->not_wide_words_with_biwords };
    $phl = $phl->lgrep( sub {
           $_->get_search_count > 100
       &&  $_->get_search_query_count > 100
       &&  $_->get_bnr_count > 3
       &&  $_->get_search_query_count / $_->get_search_count > 0.001
    } );
    my @ctgs = keys %{{ map { $_=>1 } grep { ! /^\./ } map { $_->get_minicategs } @$phl }};
    my $ctgpsfx = {};
    $ctgpsfx->{ $_ } = [ $proj->phrase(join(' ',  $_, $ttlph->get_cat_path($_) ))->not_wide_words_with_biwords ] for @ctgs;
    my $arr = [ map {"$_"}
        grep { grep { $flt->{$_} } $proj->phrase( $_ )->not_wide_words_with_biwords, map {@{$ctgpsfx->{$_}||[]}} $_->get_minicategs }
        @$phl ];

    my $newarr = [];

    for my $t (@$arr){
        #$ph $proj->dd( $ph->get_search_tail );
          my @tails = keys %{$proj->phrase($t)->get_search_tail};
          @tails = grep { $bgflt->{$_} } @tails;
          @tails = $proj->phrase(join(' ',@tails))->notwidewords;
          push(@$newarr, map { "$t $_" } @tails);
    }
    return $arr;
}

sub page_phrases_groups {
    my ( $self ) = @_;
    return $self->proj->phrase_list($self->page_phrases)->subphrases_groups_format;
}

sub get_normalized_url :CACHE {
    my ( $self ) = @_;
    return undef unless ($self->url);
    my $gemini_url = 'http://gemini.yandex.net:8080';
    my $page = $self->proj->page($gemini_url);
    $page->{additional_headers} = { "Content-type" => 'application/json;charset=utf-8', };
    $page->{timeout} = 600;
    my $resinf = $page->tt_post(to_json({ 'url' => $self->url, 'type' => 'weak', }));
    my $res = from_json($resinf);
    if ($res and ($res->{Response}) and ($res->{Response}->{MainUrl})) {
        return @{$res->{Response}->{MainUrl}}[0];
    }
    return $self->url;
}

use overload
    '""' => sub {
            my ($self) = @_;
            my $name = $self->name // '';
            my $url = $self->url // '';
            return $name." =-> ".$url;
        };

1;
