package BM::Phrase;

use utf8;
use open ':utf8';

use std;
use base qw(ObjLib::ProjPart BM::MCached BM::PhraseLongText BM::PhraseProduct);

use Utils::Dicts;
use Utils::Words;
use Utils::Sys;
use Utils::Regions;
use Utils::Funcs qw(encode_quotes is_translit);

use URI::Escape;
use Data::Dumper;
use Encode;
use List::Util qw(min max sum first);
use LWP::UserAgent;
use File::Basename qw(dirname);

use Utils::Sys qw(md5int h2sa);
use Utils::Array;

use Encode qw{_utf8_on};

use Utils::Common;
use StaticMap;

use URI::Escape qw(uri_unescape uri_escape uri_escape_utf8);

use BM::LangRecognize qw(recognize_text_lang);

########################################################
# Интерфейс
########################################################

#    text_id                   числовой идентификатор текста
#    lctext               =>   фраза в нижнем регистре
#    do_ucfirst                переводит первую букву в верхний регистр
#    words                =>   массив слов фразы
#    minuswords           =>   список минус-слов
#    split_by_minus_words =>   быстрое разделение строки на две части по первому ' -', возвращает массив
#    pluswords_length          количество символов в части плюс-слов
#    norm_minuswords      =>   нормированные минус-слова
#    words_without_stops  =>   удаляет стоп-слова, но не нормализует

#    normwords            =>   массив нормализованных слов фразы
#    norm_phr_ordered     =>   нормализация фразы с учетом порядка слов, возвращает текст
#    pnorm_phr_ordered    =>   нормализация фразы с учетом порядка слов, возвращает phrase
#    normwords_with_stops =>   массив нормализованных слов фразы с сохранением стоп-слов
#    snormwords_with_stops =>  массив снормализованных слов фразы с сохранением стоп-слов
#    normwords_pairs      =>   массив пар изначальной словоформы и нормализованного слова (третим элементом - стоп-слово или нет)
#    snormwords           =>   массив приведённых к синонимам слов
#    snorm_phr_ordered    =>   cнормализация фразы с учетом порядка слов, возвращает текст
#    psnorm_phr_ordered   =>   cнормализация фразы с учетом порядка слов, возвращает phrase
#    pluswords            =>   Массив плюс-слов
#    pluswords_phr        =>   Фраза из массива плюс-слов
#    delete_minus_words   =>   Удаляем минус-слова из фразы
#    uniqnormwords        =>   Уникальные слова
#    uniqsnormwords       =>   Уникальные слова с учётом синонимов
#    snormminuswords      =>   массив приведённых к синонимам минус-слов
#    normwordshash        =>   Хэш нормальных форм слов
#    normwordsorighash    =>   Хэш преобразования нормальных форм в исходные
#    words_positions      =>   Хеш массивов словопозиций

#    notwidewords         =>   только неширокие слова фразы
#    orig_subphrase       =>   По phl возвращает список фраз в оригинальном написании
#    pack_phr             =>   удаление дублей и стоп-слов без нормализации
#    pack_spaces          =>   возвращает текст без начальных и конечных пробелов и сжимает повторяющиеся пробелы до одного
#    norm_phr             =>   нормализация фразы, возвращает текст
#    pnorm_phr            =>   нормализация фразы, возвращает phrase
#    norm_phr_safe        =>   нормализация фразы без потери слов, возвращает текст
#    pnorm_phr_safe       =>   нормализация фразы без потери слов, возвращает phrase
#    snorm_phr            =>   нормализованная с учётом синонимов фраза, возвращает текст
#    psnorm_phr           =>  нормализованная с учётом синонимов фраза, возвращает phrase
#    norm_phr_without_hyphens => нормализованная фраза без дефисов
#    is_wide_phrase       =>   проверка, широкая ли это фраза
#    is_porno_phrase      =>   проверка на порнушность фразы
#    bad_words_in_phrase  =>   есть ли плохие слова во фразе
#    minwordcount         =>   частота самого редкого слова из фразы
#    number_of_words      =>   количество слов во фразе
#    virthits             =>   виртуальное число хитов во фразе
#    badphrsreason        =>   Причина, по которой фраза плохая
#    is_good_phrase       =>   Итоговая проверка по всем фильтрам
#    getmodelsphraseslist =>   Возвращает объект списка фраз, полученный с помощью разваливания по моделям
#    subphrases_list      =>   Возвращает PhraseList из 1, 2 и 3 словных нормализованных подфраз с учетом словосочетаний
#    vendors              =>   Список вендеров из фразы
#    get_bnr_count        =>   Частотность фразы в баннерах
#    get_modellike_modifications => Возвращает объект списка фраз, полученный с помощью переклейки похожих на модели слов
#    has_celebrities      =>   В тексте упоминаются селебретис
#    get_phrase_without_hyphens => Получить объект фразы без дефисов
#    set_exclamations_before_stops => Нормализует слова, проставляет воскл. знаки перед стоп-словами
#    set_exclamations_before_bsstops => Не нормализует слова, проставляет воскл. знаки перед стоп-словами БК, которые не являются нашими стоп-словами

#    text_without_cities_words удалить названия городов из фразы, фраза нормализуется

#    cuttext                   обрезаем текст до указанной длины, дефолтная 80
#    splitlongtext             разрезаем длинный текст на части

#    get_yandex_phraselist     получить заголовки первых 10 документов из Яндекса

#    replace_html_spec  =>  Заменяет спецсимволы html-кодировки

#    get_search_count            => частота на поиске, минус-слова не учитываются
#    get_search_query_count      частота на поиске как точной фразы

#    get_matched_bannerlist     => извлекает все баннеры, которые категоризуются по фразе
#    contains_wordsbag          => входит или нет мешок слов во фразу?
#    eq_by_inclusion            => одинаковы ли фразы по включению? (есть ли включение первой фразы во вторую, или второй в первую?)

#    get_permutations           =>  возвращает фрейзлист строк из всех комбинаций слов в исходной фразе

#    delete_bad_subphrases      Вычищает тексты от плохих подфраз

#    direct_format_phrase       Удаляем из фразы лишнее в соответствии с ограничениями Директа

#    all_subphrases_list
#    all_subphrases_list_arrayref
#    all_snorm_subphrases_list_arrayref

#    get_dict_phl               Возвращает список фраз из словаря
#    split_by_dict              Возвращает фразу без найденных слов и список фраз из словаря ($ph, $phl)
#    split_by_tmpl_dict         По словарю и шаблону возвращает модифицированный шаблон со вставкой найденной фразы из словаря. Если нашлось больше 1 фразы - ничего не делаем

#    has_subphrase              Проверяет, является ли указанная фраза подвразой текущей
#    delete_subphrase_words     Удаляет все слова указанной фразы у текущей

#    clear_utf8                  Исправляет битый utf8

#    delete_useless_minus_words  Удаление минус-слов без частот

#    delete_trade_words
#    delete_town_words

#    delete_pluses               Удаляет из фразы все символы +

#    poor_layout                 Вариант записи фразы в другой раскладке

#    delete_double_words         Удаляет совпадающие в написании повторяющиеся слова

#    smart_cut_text              Обрезание текста до указанной длины с учётом слов

#    substitute_words            Подставввляет во фразу переданные через хеш замены слов, с сохранением порядка слов, регистра и пунктуации

#    minf           =>  хэш с информацией о привязке (для BroadMatch)

#    hotels         => хэш отелей
#    get_geobase_region_ids => IN:"найти валенки в москве и санкт-петербурге", OUT: (213,2); # возвращает id-шники регионов по geobase

########################################################
# Инициализация класса
########################################################

my $porn = {};
my $not_wide = {};

my $localopts = {};

our (%grammar1f, %grammar2f);

our $RU = "А-ЯЁа-яё";
our $LA = "A-Za-z";
our $bit_masks;

sub class_init :RUN_ONCE {
    my ($class, $opt) = @_;
    my $log = $opt->{logger};
    my $proj = $opt->{'proj'};

    $log->log("/wide_phrases");

    $log->log("porno");
    $porn      = load_words_dict($opt->{'porn_words'}, { snorm => 1, only_words => 1 });
    $log->log('/ porno');

    open(FC, $opt->{'dict_not_wide'});
    while(my $str = <FC>){
        chomp $str;
        $not_wide->{word2snorm $str}++;
    }
    close(FC);

    if ($proj->{load_grammar_dict}) {
        $proj->log('loading grammar dict ...');
        open my $fh, '<', $opt->{grammar_dict};
        while (<$fh>) {
            chomp;
            my @fld = split /\t/;
            push @{$grammar1f{$fld[0]}}, $_;
            push @{$grammar2f{$fld[1]}}, $_;
        }
        close $fh;
        #print STDERR Dumper ( \%grammar2f );
    }
}


########################################################
#Методы
########################################################

sub language :CACHE {
    my ($self) = @_;
    return $self->{language} || $self->proj->languages->{$self->lang} || $self->proj->default_language;
}


# get-set accessor
sub inf {
    my $self = shift;
    if (@_) {
        $self->{inf} = $_[0];
    }
    return $self->{inf};
}

# hinf во всём подобен inf, только h (hash);
sub hinf {
    my $self = shift;
    if (@_) {
        $self->{hinf} = $_[0];
    }
    return $self->{hinf};
}

# доступ к полю matcher_inf (хэш с информацией о привязке)
sub minf {
    my $self = shift;
    if (@_) {
        $self->{matcher_inf} = $_[0];
    }
    return ($self->{matcher_inf} //= {});
}


# информация для родных фраз
sub bnr_inf {
    my $self = shift;
    my $inf = $self->inf || [0, 0, 0, 0, 0];
    my $shows = $inf->[1] + $inf->[3];
    my $clicks = $inf->[2] + $inf->[4];
    my %inf = (
        bid => $inf->[0],
        shows => $shows, clicks => $clicks,  # суммарные показы/клики
        pshows => $inf->[3], pclicks => $inf->[4],  # спецразмещение
        active_flag => $inf->[5],
        id  => $inf->[6],  # БК-шный id фразы, может быть нулём, если пока не отправили в БК
    );
    return \%inf;
}

sub text {
    return $_[0]{'text'};
}

sub text_length :CACHE {
    return length($_[0]{'text'});
}

sub is_quoted :CACHE{
    my $self = shift;
    return ($self->external_text =~ /\s\~0\s*$/) ? 1 : 0;
}

#Удаляем повторяющиеся пробелы и точки при подсчёте длины
sub flt_text_length :CACHE {
    my $t = $_[0]{'text'};
    $t =~ s/[-\. ,]{2,}/ /g;
    return length($t);
}

sub lctext {
    return lc $_[0]{'text'};
}


sub do_ucfirst {
    my ($self) = @_;
    my $text = $self->{'text'};
    $text =~ s/^\s*([a-zа-я])/uc($1)/e;
    return $self->proj->phrase($text);
}

# Заменить слова на капитализацию
sub casecorrection {
    my ($self) = @_;
    my $lang = $self->{lang};
    my $text = $self->{'text'};
    my $punctuation_sep = $self->proj->_bannerland_punctuation_not_valid_for_naming();
    my @words = split /(\s+|[\Q.$punctuation_sep\E])/, $text;
    my $cpfx = $self->capitalization->{$lang};
    my @result;

    sub capitalize_word {
        my ($cpfx, $lang, $word) = @_;
        if($cpfx->{$word}) {
            $word = $cpfx->{$word};
        } else {
            my $norm = word2norm($word, $lang);
            my $fixed = $cpfx->{$norm};

            # пытаемся преобразовать капитализацию нормализованной формы, если она существует
            if($fixed) {
                my @word_parts = split "-", $word;
                my @fixed_parts = split "-", $fixed;
                my @result_parts;

                if(scalar(@word_parts) != scalar(@fixed_parts)) {
                    @word_parts = ($word);
                    @fixed_parts = ($fixed);
                }

                for my $i (0..$#word_parts) {
                    my $first_char = substr($word_parts[$i], 0, 1);
                    my $fixed_char = substr($fixed_parts[$i], 0, 1);
                    $first_char = uc($first_char) if $fixed_char eq uc($fixed_char);
                    push @result_parts, $first_char . substr($word_parts[$i], 1)
                }

                $word = join "-", @result_parts;
            }
            elsif ( scalar(split '-', $word) > 1 ) {
                #капитализируем отдельно части слова, если оно разделено дефисом
                $word = join '-', map { capitalize_word($cpfx, $lang, $_) } split '-', $word;
            }
        }
        return $word;
    };

    my $in_quotes = 0;
    for my $i (0..$#words) {
        $in_quotes = 1 if ($words[$i] =~ /^[«'"\„\‘]/);
        if ($in_quotes or ($words[$i] eq join "-", map { ucfirst lc $_ } split "-", $words[$i])) {
            # не меняем капитализацию слова в двух случаях: 1) если оно в кавычках 2) если в нём первая буква большая, а остальные маленькие
            push @result, $words[$i];
        } else  {
            push @result, capitalize_word($cpfx, $lang, lc($words[$i]));
        }
        $in_quotes = 0 if ($words[$i] =~ /[»'"\”\’]$/);
    }
    $text = join("", @result);
    return $self->language->phrase($text)->_case_dot_transf($self->{'text'});
}

#Оставляем исходный вариант регистра, если слово идёт после точки
sub _case_dot_transf {
    my ($self, $prevtext) = @_;
    my @arr = $prevtext =~ /(\. \w)/g;
    return $self unless @arr;
    my $trsf = { map { @$_ } grep { $_->[0] ne $_->[1] } map { [ lc($_) => $_ ] } @arr };
    my $text = $self->text;
    $text =~ s/(\. \w)/$trsf->{$1} || $1/ge;
    return $self->language->phrase($text);
}

# Возвращает хэш { слово => наиболее частый вариант капитализации }  Если наиболее частый вариант - в нижнем регистре, то слова в хэше нет
sub capitalization : GLOBALCACHE {
    my ($self) = @_;
    my $capitalization_data = {};
    for my $lang ( @{$self->proj->{load_languages}}, 'ru') {
        my $file = $Utils::Common::options->{Capitalization_params}{"file_$lang"};
        next if not $file;
        open(F, $file)  or do { next };
        while (my $line = <F>) {
            chomp $line;
            my ($word, $word_cap) = split /\t/, $line;
            $capitalization_data->{$lang}{$word} = $word_cap;
        }
        close(F);
    }
    return $capitalization_data;
}

sub words {
    my $self = $_[0];
    my $lang = $self->lang;
    text2words($self->{'text'}, $lang);  # приведение к нижнему регистру зависит от языка!
}

sub words_lite {
    my $self = $_[0];
    my $lang = $self->lang;
    text2words_nolc($self->{'text'}, $lang);  # без приведения к нижнему регистру
}

# Удаляет стоп-слова, но не нормализует
sub words_without_stops { $_[0]->normalize(filt_stop => 1) }

sub text_add_pluses_to_stop_words {
    my $self = shift;
    return join (' ', map { stop4norm($_, $self->lang) ? '+'.$_ : $_ } text2words($self->{'text'}, $self->lang) );
}

sub text_delete_pluses_except_stop_words {
    my $self = shift;
    return join (' ', map { my $pure = $_; $pure =~ s/^\+//; stop4norm($pure, $self->lang) ? $_ : $pure } text2words($self->{'text'}, $self->lang) );
}

sub text_without_stops {
    my $self = shift;
    return join(" ", $self->words_without_stops);
}

sub pack_phr :CACHE {
    my $self = shift;
    #my @w = sort values %{{ map { word2norm($_) => $_ } $self->words_without_stops }};
    #return join(" ", @w);
    return join(" ", sort keys %{{ map {$_=>1} split / /, $self->norm_phr }});
}

# вариант без нормализации
sub pack_phr_lite :CACHE {
    my $self = shift;
    return join(" ", sort keys %{{ map {$_=>1} split /\s+/, $self->text }});
}

sub pack_spaces :CACHE {
    my $self = shift;
    my $t = $self->text;
    #$t =~ s/^\s+|\s+$//g;
    $t =~ s/\s+$//;
    $t =~ s/^\s+//;
    $t =~ s/\s\s+/ /g;
    return $t;
}

sub pack_spaces_phr :CACHE {
    my $self = shift;
    my $t = $self->pack_spaces;
    return $self->proj->phrase($t);
}

sub pack_minuswords : LANG {
    my $self = shift;
    return $self if $self->text =~ /^\s*"/;
    my @mw = grep { substr($_, 0, 1) eq '-' } $self->words;
    #@mw = sort keys %{{ map {$_=>1}  @mw}};
    @mw = sort grep{!$self->normwordshash->{substr($_, 1)}} keys %{{ map {$_=>1}  @mw}};
    return $self->proj->phrase(join(" ", $self->pluswords, @mw));
}

sub pluswords {
    grep { substr($_, 0, 1) ne '-' } $_[0]->words;
}

sub pluswords_phr {
    my $self = shift;
    my $txt = join(' ', $self->pluswords);
    return $self->proj->phrase( $txt );
}

sub delete_minus_words {
    join ' ', grep { substr($_, 0, 1) ne '-' } $_[0]->words_lite;
}

sub split_by_minus_words {
    return split( /(?= -[^\s])/, $_[0]->{text}, 2 );
}

sub plus_text :CACHE {
    my ($self) = @_;
    my ($pw, $mw) = $self->split_by_minus_words;
    return $pw;
}

sub minus_text :CACHE {
    my ($self) = @_;
    my ($pw, $mw) = $self->split_by_minus_words;
    return $mw;
}
#@returns BM::Phrase
sub minus_phr :CACHE {
    my $self = shift;
    $self->language->phrase(join(' ', $self->minuswords));
}

sub pluswords_length {
    my ($self) = @_;
    my ($pw, $mw) = $self->split_by_minus_words;
    return length($pw);
}

sub minuswords :CACHE {
    my $self = shift;
    my @arr = grep { /^-/ } $self->words;
    $_ =~ s/^-// for @arr;
    return @arr;
}

# простой двуязычный транслит, транслитерирует язык фразы в противоположный, если не указан параметр tolang
# tolang имеет два значения: (tolang=>'ru') или (tolang=>'en')
    my $eng_to_ru = { a=>'а', b=>'б', c=>'ц', d=>'д', e=>'е', f=>'ф',
               g=>'г', h=>'х', i=>'и', j=>'дж',k=>'к', l=>'л',
               m=>'м', n=>'н', o=>'о', p=>'п', q=>'ку',r=>'р',
               s=>'с', t=>'т', u=>'у', v=>'в', w=>'в', x=>'кс',
               y=>'и', z=>'з',
               A=>'А', B=>'Б', C=>'Ц', D=>'Д', E=>'Е', F=>'Ф',
               G=>'Г', H=>'Х', I=>'И', J=>'ДЖ',K=>'К', L=>'Л',
               M=>'М', N=>'Н', O=>'О', P=>'П', Q=>'КУ',R=>'Р',
               S=>'С', T=>'Т', U=>'У', V=>'В', W=>'В', X=>'КС',
               Y=>'И', Z=>'З'};

    my $ru_to_eng = { 'а'=>'a',  'б'=>'b',  'в'=>'v', 'г'=>'g',  'д'=>'d',  'е'=>'e',
               'ё'=>'yo', 'ж'=>'j',  'з'=>'z', 'и'=>'i' , 'й'=>'y',  'к'=>'c',
               'л'=>'l',  'м'=>'m',  'н'=>'n', 'о'=>'o',  'п'=>'p',  'р'=>'r',
               'с'=>'s',  'т'=>'t',  'у'=>'u', 'ф'=>'f',  'х'=>'h',  'ц'=>'c',
               'ч'=>'ch', 'ш'=>'sh', 'щ'=>'sch','ь'=>'',   'ы'=>'y',  'ъ'=>'',
               'э'=>'a',  'ю'=>'yu', 'я'=>'ya',
               'А'=>'A',  'Б'=>'B',  'В'=>'V', 'Г'=>'G',  'Д'=>'D',  'Е'=>'E',
               'Ё'=>'YO', 'Ж'=>'J',  'З'=>'Z', 'И'=>'I' , 'Й'=>'Y',  'К'=>'K',
               'Л'=>'L',  'М'=>'M',  'Н'=>'N', 'О'=>'O',  'П'=>'P',  'Р'=>'R',
               'С'=>'S',  'Т'=>'T',  'У'=>'Y', 'Ф'=>'F',  'Х'=>'X',  'Ц'=>'C',
               'Ч'=>'CH', 'Ш'=>'SH', 'Щ'=>'SCH', 'Ь'=>'', 'Ы'=>'Y',  'Ъ'=>'',
               'Э'=>'A',  'Ш'=>'SH', 'Я'=>'YA' };

sub translit_simple {
    my ( $self, %par ) = @_;
    #print STDERR Dumper \%par;
    my $text = $self->text;
    my $lang;
    $lang = 'en' if $text =~ /[a-z]/i;
    $lang = 'ru' if $text =~ /[а-яё]/i;
    #print STDERR "$lang => ", $par{tolang}, "\n";
    return $text if ( !$lang || exists($par{tolang}) && $par{tolang} ne 'ru' && $par{tolang} ne 'en' );

    my @text_letters = split //, $text;

    my $h;
    if ( exists($par{tolang}) ) {
        $h = $par{tolang} eq 'en' ? $ru_to_eng : $eng_to_ru;
    }
    else {
        $h = $lang eq 'ru' ? $ru_to_eng : $eng_to_ru;
    }

    my @ares = map { $h->{$_} // $_ } @text_letters;

    return join '', @ares;
}

sub translit_en2ru {
    my ($self) = @_;
    my $dict_translit = $self->proj->dict_manager->get_dict("translit_en2ru");
    my $trie = $dict_translit->{trie};
    my $prefix2node = $dict_translit->{prefix2node};
    my ($word2count, $restore_signs) = map{$self->proj->dict_manager->get_dict("wordcount")->{$_}} qw(word2count restore_signs);
    my @new_words = ();
    my $re = "([" . join("", keys %$trie) . "]+)";
    my @words = (lc($self->text) =~ /$re/g);

    for my $word (@words) {
        my $best_word = $word;
        if(length($word) < 20) {
            my @chars = split //, $word;
            my $prefix2versions = { "" => [ "" ] };
            for my $char_pos (0..$#chars) {
                my $char = $chars[$char_pos];
                my $next_prefix2versions = {};

                while(my ($prefix, $versions) = each %$prefix2versions) {
                    my $node = $prefix2node->{$prefix};
                    next if !$node->{$char};

                    if($node->{$char}{" "}) {
                        for my $version_add (@{$node->{$char}{" "}}) {
                            for my $version (@$versions) {
                                push @{$next_prefix2versions->{""} ||= []}, $version . $version_add;
                            }
                        }
                    }

                    if($char_pos < $#chars && $node->{$char}{$chars[$char_pos + 1]}) {
                        for my $version (@$versions) {
                            push @{$next_prefix2versions->{$prefix . $char}}, $version;
                        }
                    }
                }

                $prefix2versions = $next_prefix2versions;
            }

            my $best_count = $word2count->{$best_word} || 0;
            for my $new_word (@{$prefix2versions->{""} || []}) {
                my $new_count = $word2count->{word2norm($new_word)} || 0;

                # пытаемся восстановить [ьъ]
                my $restored = $restore_signs->{$new_word};
                if($restored && $word2count->{$restored} > $new_count) {
                    $new_word = $restored;
                    $new_count = $word2count->{$restored};
                }

                if($new_count > $best_count) {
                    $best_count = $new_count;
                    $best_word = $new_word;
                }
            }
        }

        push @new_words, $best_word;
    }

    return $self->proj->phrase(join " ", @new_words);
}

sub text_without_cities_words {
    my $self = shift;
    my @wds = $self->normwords;
    my $cities_words = $self->language->get_dict("cities")->snorm_hash;
    return join ' ', grep { ! $cities_words->{ norm2snorm($_, $self->{'lang'})} } @wds;
}

# Нормализация - основные методы
# используется API из Utils::Words

sub normwords :CACHE {
    my $self = shift;
    return split ' ', $self->text if $self->{normed};
    return text2normwords($self->{'text'}, $self->{'lang'});
}

# нормализованная фраза с учетом порядка слов
sub norm_phr_ordered :CACHE { $_[0]->normalize(norm => 1, filt_stop => 1, join => 1) }
sub norm_phr_saving_order  { $_[0]->norm_phr_ordered }  # alias

sub pnorm_phr_ordered {
    my $self = shift;
    my $text = $self->norm_phr_ordered;
    return $self->proj->phrase( $text );
}

sub snorm_phr_ordered :CACHE { $_[0]->normalize(snorm => 1, filt_stop => 1, join => 1) }
sub snorm_phr_saving_order  { $_[0]->snorm_phr_ordered }  # alias

# cнормализованная фраза с учетом порядка слов
sub psnorm_phr_ordered {
    my $self = shift;
    my $text = $self->snorm_phr_ordered;
    return $self->proj->phrase( $text );
}

sub norm_phr_without_hyphens :CACHE {
    my $self = shift;
    return $self->get_phrase_without_hyphens->norm_phr_quote;
}

#Обвязка для выполнения методов бинарного нормализатора
sub lmr_mthd {
    my $self = shift;
    my $mthd = shift;
    my $lm = lmr;
    return $lm->$mthd($self->text, @_);
}

sub norm_phr_quote :CACHE {
    my $self = shift;
    return $self->{'text'} if $self->{'normed'};
    return $self->norm_phr unless $self->{'text'} =~ /"/;
    my $t = join ' ', sort( keys %{{ map { word2norm($_) => 1 } text2words($self->{'text'}, $self->lang) }} ); #Получаем список стоп-слов
    return qq{"$t"};
#    return join ' ', sort( $ppp->text2normwords($self->{'text'}, $self->{'lang'}));
    #return join ' ', text2normwords($self->{'text'}, $self->{'lang'});
    #text2normwords($self->{'text'}, $self->{'lang'});
    #return 'cccc';
}


sub norm_phr_uniq :CACHE {
    my ($self) = @_;
    return join " ", $self->uniqnormwords;
}

sub norm_phr_safe :CACHE {
    my ($self) = @_;
    return '' unless $self->text;
    my @res = ();
    for my $word ( split/\s+/, $self->text ){
        my $norm = $self->proj->phrase( $word )->norm_phr || '';
        push @res, ( !$norm || $norm =~ / / ) ? $word : $norm;
    }
    return join(' ', @res);
}

sub pnorm_phr_safe {
    my ($self) = @_;
    my $text = $self->norm_phr_safe;
    return $self->proj->phrase( $text );
}

# нормализация с учетом синтаксиса рекламных фраз (сохраняем кавычки)
sub norm_adphr :SCACHE {
    my $self = shift;
    my $n = $self->norm_phr;
    return $self->is_quoted ? qq("$n") : $n;
}
sub snorm_adphr :SCACHE {
    my $self = shift;
    my $n = $self->snorm_phr;
    return $self->is_quoted ? qq("$n") : $n;
}

sub snormwords :CACHE {
    my $self = shift;
    return split ' ', $self->{'text'} if $self->{snormed};
    return text2snormwords_fast($self->{'text'}, $self->{'lang'});
}

sub snorm_phr :SCACHE {
    my $self = shift;
    return $self->{'text'} if $self->{snormed};
    my $str = $Utils::Words::snlm->GetNorm($self->{text}, ($self->{'lang'} || 'ru') );
    _utf8_on($str);
    return $str;
}

# снормализация, возвращает фразу
sub psnorm_phr {
    my $self = shift;
    my $snormtext = $self->snorm_phr || '';
    return $self->proj->phrase( $snormtext );
}

sub snorm_phr_uniq {
    my ($self) = @_;
    return join " ", $self->uniqsnormwords;
}

# должен быть эквивалентен $self->normalize(norm => 1, filt_stop => 1, sort => 1, join => 1)
sub norm_phr :SCACHE {
    my $self = shift;
    return $self->{'text'} if $self->{normed};
    return '' unless defined( $self->{'text'} );
    my $str = $Utils::Words::lm->GetNorm($self->{text}, ($self->{'lang'} || 'ru') );
    _utf8_on($str);
    return $str;
}

# нормализация, возвращает фразу
sub pnorm_phr {
    my $self = shift;
    my $normtext = $self->norm_phr || '';
    return $self->proj->phrase( $normtext );
}

# хеш массивов словопозиций слов
sub words_positions {
    my ($self) = @_;
    my %hres = ();
    my @words = $self->words_lite;
    for my $i (0..@words-1){
        my $word = $words[$i];
        $hres{$word} ||= [];
        push @{$hres{$word}}, $i;
    }
    return %hres;
}

sub correct_phr :CACHE {
    my $self = shift;
    my $lang = $self->{lang};
    return join(' ', map { norm2good($_, $lang) } $self->normwords);
}

sub is_misspell :CACHE {
    my ($self) = @_;
    my $lang = $self->{lang};
    for my $nw ($self->normwords) {
        if ($nw ne (norm2good($nw, $lang))) {
            return 1;
        }
    }
    return 0;
}

sub cdict_command {
    my ($self, $cmd) = @_;

    return $cmd . "_" . $self->lang if $self->lang ne "ru";
    return $cmd;
}

sub cdict_preprocess :CACHE {
    my ($self) = @_;
    my $text = $self->text;
    $text =~ s/\~\d/ /g;
    $text = $self->language->phrase($text)->norm_phr;
    $text =~ s/\// /g;
    $text =~ s/\-/ /g;
    $text =~ s/[\!\+]//g;
    return $self->language->phrase($text);
}

sub cdict_preprocess_norm :CACHE {
    my ($self) = @_;
    my $text = $self->text;
    $text =~ s/\~\d/ /g;
    $text = join " ", $self->language->phrase($text)->pluswords;
    $text =~ s/[\+\!\/\t\n]/ /g;
    return $self->proj->phrase($text);
}

sub cdict_key_norm :SCACHE {
    my ($self) = @_;
    return $self->cdict_preprocess->norm_phr_uniq;
}
sub cdict_key_snorm :SCACHE {
    my ($self) = @_;
    return $self->cdict_preprocess->snorm_phr_uniq;
}

# общий метод нормализации фразы
# по возможности используем функции Utils::Words только здесь
# флаги (по умолчанию отключены):
#   use_minus => 1  брать минус-слова вместо плюс слов
#   norm => 1  нормализовать слова
#   snorm => 1  син-нормализовать слова
#   filt_stop => 1  удалить стоп-слова
#   uniq => 1  оставить только уникальные
#   sort => 1  сортировать полученный список
#   join => 1  объединить список в строку
sub normalize {
    my $self = shift;
    my %par = @_;
    my $lang = $self->lang;

    return map { [ $_, word2norm(lc($_), $lang), stop4norm(word2norm($_, $lang), $lang) ] } text2origwords($self->{'text'}) if $par{normpairs}; #Возвращает не слова, а массив указателей на пары ['word', 'normword']

    my @words;
    @words = $par{use_minus} ? $self->minuswords : $self->pluswords;
    if ($par{norm}) {
        @words = map { word2norm($_, $lang) } @words;
    } elsif ($par{snorm}) {
        @words = map { word2snorm($_, $lang) } @words;
    }
    if ($par{filt_stop}) {
        if ($par{norm} or $par{snorm}) {
            @words = grep { !stop4norm($_, $self->lang) } @words;
        } else {
            @words = grep { !stop4norm(word2norm($_, $lang), $lang) } @words;
        }
    }
    @words = uniq(@words) if $par{uniq};
    @words = sort(@words) if $par{sort};
    return join(' ', @words) if $par{join};
    return @words;
}

# возвращает список хэшей с информацией о словах
sub analyze_words :CACHE {
    my $self = shift;
    my $lang = $self->lang;
    my @res;
    my $pos = 0;
    for my $w ($self->words) {
        my %inf;
        my $n = word2norm($w, $lang);
        $inf{pos}  = $pos++;
        $inf{word} = $w;
        $inf{norm} = $n;
        $inf{snorm} = word2snorm($w, $lang);
        $inf{stop} = (stop4norm($n, $lang) ? 1 : 0);
        $inf{bsstop} = (bsstop4word($w) ? 1 : 0);
        push @res, \%inf;
    }
    return @res;
}

sub is_good_digits {
    my ($self, $d) = @_;
    return 0 if $d =~ /\D/;
    $d =~ s/000+$/0/; #Сжимаем последние нули, если много цифр
    my $dcnt= keys %{{ map {$_=>1} split //, $d }};
    return 1 if $d =~ /\d{5}/ && ($dcnt > 3);
    return 1 if $d =~ /\d{6}/;
    return 1 if $d =~ /^(666|585|999)$/;
    return 1 if $d =~ /^(206|21(0[4789]|31|20|13|15))$/;
    return 1 if $d =~ /^(8600|6300|6110|7390|6290|3307|3309|9100|2752|2347|7511|8130|3613|2217|3501|3308|4361|3205|8092|8306|3720|8061|8734|2080|6271|7562|7506|3725|6516|1682|4320|6288|1589|6267|8110|4308|8310|1795|8102|6129|2428|6210|6520|2710|2630|2612|6501|1812|2515|8734|2101|8732|65115|22171|65116|10880|77100|54115|20880|21214|6802|5871|3182|7669|2206|7505|2595|88358|8204|4513|9446|3505|11200|7201|8421|6270|7224|8504|8725|7223|4713|6007|7994|6180|50560|2711|4088|3961|56686|7197|5008|5427|784|55944|53453|3716|8405|3438|5022|9860|6304|6387|66062|8812|81611|8601|2274|6317|2793|6702|7438|7857|8972|11554|2641|2825|6467|1665|4151|1652|56996|3964|9730|43114|54452|21721|30130|4403|1612|6022|9262|8162|7660|4706|22270|33314|4014|2414|4061|6461|9780|70002|6337|5460|4556|2523|8140|8389|60260|6500|6303|2717|2123|2512|2106|2126|5080)$/;
    return 1 if $d =~ /^(59|926|903|499|86|814|428|808)$/;
    return 1 if $d =~ /\d{4}/ && ($d > 2030) && $d =~ /[2-9]/ && ($dcnt == 2);
    return 0;
}

# Нормализация - вспомогательные методы
# используются другие методы BM::Phrase

sub normwords_with_stops :CACHE {
    my $self = shift;
    return $self->normalize(norm => 1);
}

sub normwords_without_bsstops :CACHE {
    my ($self) = @_;

    return grep{!bsstop4word($_)} $self->normwords;
}

sub snormwords_with_stops :CACHE {
    my $self = shift;
    my $w = $Utils::Words::snlm->norm_words($self->{text}, $self->{lang} // 'ru');
    return @$w;
}

sub normwords_pairs :CACHE {
    my $self = shift;
    return $self->normalize(normpairs => 1);
}

sub normminuswords :CACHE {
    my $self = shift;
    return $self->normalize(use_minus => 1, norm => 1, filt_stop => 1);
}
sub norm_minuswords { $_[0]->normminuswords }  # alias

sub number_of_words :CACHE {
    my $self = shift;
    my @w = $self->normwords;
#    print STDERR Dumper \@w;
    my $c = @w;
    return $c;
}

#Синоним number_of_words
sub wordcount :CACHE { return $_[0]->number_of_words }


sub uniqnormwords {
    my $self = shift;
    return sort { $a cmp $b } keys %{{map {$_=>1} $self->normwords}};
}

sub uniqsnormwords :CACHE {
    my $self = shift;
    return sort {$a cmp $b} keys %{{map {$_=>1} $self->snormwords}};
}

sub snormminuswords :CACHE {
    my $self = shift;
    return $self->normalize(use_minus => 1, snorm => 1, filt_stop => 1);
}

sub notwidewords :CACHE {
    my $self = shift;
    return grep {!$self->language->is_snorm_wide($_) || $not_wide->{$_}} $self->snormwords;
}

sub not_wide_words_with_biwords :CACHE {
    my $self = shift;
    return grep {!$self->language->is_snorm_wide($_) || $not_wide->{$_}} map { word2snorm($_) } split /\s+/, $self->norm_with_multiwords;
}

sub norm_phr_with_stops :CACHE {
    my $self = shift;
    return $self->normalize(norm => 1, join => 1);
}


# /Нормализация

sub orig_subphrase {
    my ($self, $phl, $first) = @_;
    my @nw = $self->normwords; #Список нормализованных слов исходной фразы
    my $h = { map { $_ => 1, } @nw }; #Хэш слов исходной фразы
    my @res = ();
    for my $ph (@$phl){ #Перебираем фразы, в которых ищем подфразу
        my @ww = $ph->normwords_pairs; #Пары исходных и нормализованных слов
        #for my $i (0 .. @ww-1){ #Перебираем слова
        my $cur = 0;
        while($cur < @ww){ #Перебираем слова
            last if $first && ($cur > 0); #Если стоит флаг, рассматриваем только случаи от начала фразы
            my @arr = (); #Массив совпадений от текущего слова
            $cur++, next if $ww[$cur][2] && !@arr && ($ww[$cur][0] ne 'для'); #Не начитаем совпадение со стоп-слова
            my @stp = ();
            while( $ww[$cur] && ($ww[$cur][2] || $h->{$ww[$cur][1]}) ){
                if($ww[$cur][2]){
                    push(@stp, $ww[$cur]); #Ведём отдельный массив стоп-слов
                }else{
                    push(@arr, @stp, $ww[$cur]);
                    @stp = (); #доклеиваем и сбрасываем стоп-слова
                }
                $cur++;
            }
            $cur++;

            #Фильтруем повторяющиеся слова
            if(@arr){
                my %flt = %$h;
                #$self->proj->dd(\@arr,\%flt);
                @arr = grep {
                    ( ! $flt{$_->[1]} ) #Слово, которого не было в исходной выборке (стоп-слова)
                   || ( ++$flt{$_->[1]} == 2 )    #Слово, которое встретилось более 1 раза
                } @arr;
#$self->proj->dd('EE');
#exit;
                #$self->proj->dd(\@arr);
                while( $arr[@arr-1][2] ){ pop @arr } #Срезаем конечные стоп-слова, если они появились после удаления дублирующихся слов
            }
            #/Фильтруем повторяющиеся слова

            next if @nw > grep { ! $_->[2] } @arr; #Найдено меньше слов, чем в искомой фразе
            #print Dumper(["$ph", \@arr]);
            my $curh = { map {$_->[1]=>1} @arr }; #Хэш совпадения
            next if grep { ! $curh->{$_} } @nw; #В совпадении есть все слова исходной фразы
            #Добавляем найденную фразу
            push( @res, join(' ', map {$_->[0]} @arr) );
        }
    }
    return $self->proj->phrase_list({ phrases_arr => \@res });
}

sub widephr :CACHE {
    my $self = shift;
    return $self->language->is_phrase_wide($self);
}

sub is_wide_phrase :CACHE {
    my $self = shift;
    return $self->{cdict_is_wide} if defined($self->{cdict_is_wide});

    # предобработка текста фразы
    my @clean_words;
    for my $word ($self->normwords) {
        next if bsstop4word($word);    # не учитываем слова, которые являются стоп-словами в БК
        $word =~ s/[\+\-\!]/ /g;         # убираем спецсимволы
        push @clean_words, $word;
    }
    my $text = join " ", @clean_words;
    my $clean_phr = $self->language->phrase($text);

    my @nww = $clean_phr->notwidewords;
    if( $text =~ /\b[A-Za-z]{3}[A-Za-z]*\b/ ){
        return 0 if $text =~ /\b([A-Za-z]+\d+|\d+[A-Za-z]+)\b/;
        return 0 if $text =~ /\b[A-Za-z]{1,2}\b/ && $text =~ /\d/;
        my @dgtw = grep {/\d/} $self->words;
        return 0 if @dgtw > 1;
        my @engw = grep {/\d/} $self->words;
        return 0 if (@engw > 1) && @dgtw;
    }elsif( $text =~ /\b([A-Za-z]+\d+|\d+[A-Za-z]+)\b/ ){
        return 0 if $text =~ /\d{3}/;
    }elsif( $text =~ /\d/ ){
        my @dgtw = grep {/^\d+$/} $self->words;
        for my $d (@dgtw){
            return 0 if $self->is_good_digits($d);
        }
    }
    return 0 if $self->language->is_phrase_not_wide($self);
    return 1 if $text =~ /^\s*\S{1,2}\s+\S{1,2}\s*$/; #Убираем частые двухбуквенные сочетания
    return 1 if @nww == 0;
    return 1 if $clean_phr->widephr;
    return 0;
}

sub porno_words :CACHE {
    my ($self) = @_;
    return grep { $porn->{$_} } map {my $w = $_; $w =~ s/^[+!]//; $w eq 'как' ? '' : word2snorm($w)} $self->snormwords;
}

sub is_porno_phrase :CACHE {
    my ($self) = @_;
    return 1 if $self->porno_words;
    return 0;
}

sub is_wide_spam_phrase :CACHE {
    my ($self) = @_;
    return $self->language->is_phrase_wide_spam($self);
}

sub has_stopwords :CACHE {
    my ($self) = @_;
    for( $self->normwords_pairs ){
        return 1 if $_->[2]; # stop4norm
    }
    return 0;
}

sub bad_words_in_phrase :CACHE {
    my ($self) = @_;
    return 1 if grep { bad4snorm($_) } $self->snormwords;
    return 0;
}

# минимальное значение wordcount слов фразы (лучше использовать get_search_count!)
sub minwordcount :CACHE {
    my $self = shift;
    my @phrs = map { $self->language->phrase($_) } $self->normwords;
    my $phl = $self->proj->phrase_list(\@phrs);
    $phl->cache_search_count;
    return min(map { $_->get_search_count } @$phl) // 0;
}


sub has_celebrities :CACHE {
    my ($self) = @_;
    my $phclbs = $self->proj->dict_manager->get_dict("celebrities", $self->lang)->phrase_list->search_subphrases_in_phrase($self);
    my @a = $phclbs->phrases;
    return @a ? 1 : 0;
}

sub is_directmod_banned :CACHE {
    my $self = shift;
    my $dict = $self->proj->dict_manager->get_dict('directmod');
    my $subphl = $dict->phrase_list->search_subphrases_in_phrase($self, only_norm => 1);
    return $subphl->count ? 1 : 0;
}

sub badphrsreason :CACHE {
    my ($self) = @_;

    my $reason = '';
    $reason .= '_badwords'      if $self->bad_words_in_phrase; #Выкидываем фразу, если есть плохие слова
    $reason .= '_porno'         if $self->is_porno_phrase; #Фильтруем порно и плохие фразы
    $reason .= '_banned'        if $self->is_directmod_banned;  # забанено модерацией Директа
    $reason .= '_wide'          if $self->is_wide_phrase;

    $reason .= '_shotword' if ($self->number_of_words == 1) && (length(($self->normwords)[0]) < 3); #выбрасываем короткие одиночные слова

    #Удаляем повторы одного и того же слова
    my @uniqwords = keys %{{ map { $_ => 1} $self->snormwords }};
    if(@uniqwords < $self->number_of_words){
        $reason .= '_doubleword';
    }

    $reason .= '_commonLemmas' if $self->is_collapsing_2words;

    return $reason;
}

sub is_good_phrase :CACHE {
    my ($self) = @_;
    return $self->{cdict_is_good_phrase} if defined $self->{cdict_is_good_phrase};
    return ! $self->badphrsreason;
}

# список синонимов через bender
sub get_bender_syn :CACHE {
    my ($self) = @_;
    my @groups;

    for my $w ($self->uniqsnormwords) {
        push @groups, join("|", snorm2goodsyns($w));
    }

    my $top = $self->proj->bender_client->exec_command("top\t20\t1\t".join(" ", map{"($_)"} @groups));
    my %data = split "\t", $top;

    return $self->proj->phrase_list({ phrases_arr => [ keys %data ] });
}

sub normwordshash :CACHE {
    my ($self) = @_;
    return { map { $_ => 1 } $self->normwords };
}

#Хэш преобразования нормальных форм в исходные
sub normwordsorighash :CACHE {
    my ($self) = @_;
    return { map { ([text2normwords($_, $self->{'lang'})]->[0] || '') => $_ } $self->pluswords };
}

sub snormwordshash :CACHE {
    my ($self) = @_;
    return { map { $_ => 1 } $self->snormwords };
}

# $self->is_subphr($phr, $flag)
# является ли $self подфразой $phr
# $flag = 'norm' или 'snorm'
sub is_subphr {
    my $self = shift;
    my $phr = shift;
    my $type = shift || 'norm';
    my ($h, @w);
    if ($type eq 'norm') {
        $h = $phr->normwordshash;
        @w = $self->normwords;
    } elsif ($type eq 'snorm') {
        $h = $phr->snormwordshash;
        @w = $self->snormwords;
    } else {
        return undef;
    }
    return 0 if grep { !$h->{$_} } @w;
    return 1;
}

# нормализовать слова, предварительно склеив мультиворды через '+'; порядок слов сохраняется!
sub norm_with_multiwords {
    my $self = shift;
    my @words = $self->normwords;

    # разбиваем на группы
    my @mulinf = $self->get_multiwords_inf;
    my %pos2grp;
    for my $d (@mulinf) {
        my ($dict, $start, $stop) = @$d;
        next if grep { $pos2grp{$_} } ($start .. $stop);  # кого-то уже определили в группу
        $pos2grp{$_} = $d for ($start .. $stop);
    }

    my @v;
    for (my $i=0; $i<@words; ++$i) {
        my $grp = $pos2grp{$i};
        if ($grp) {
            my ($dict, $start, $stop) = @$grp;
            push @v, join('+', @words[$start .. $stop]);
            $i = $stop;
        } else {
            push @v, $words[$i];
        }
    }
    return join(' ', @v);
}




# подфразы: 1,2,3-словные; слова идут рядом или через одно; с учетом multiwords (+ исходная фраза)
# параметры задаются в хэше:
# normtexts      => возвращать только norm_phr, а не PhraseList (по умолчанию: 0)
# preprocessed   => фраза уже нормализована и multiwords склеены через '+', см. norm_with_multiwords (по умолчанию: 0)
sub subphrases_list {
    my ($self, %par) = @_;

    # "виртуальные слова" с учетом biwords
    my @v;
    if ($par{preprocessed}) {
        @v = split ' ', $self->text;
    } else {
        @v = split ' ', $self->norm_with_multiwords;
    }
    $_ =~ tr/+/ / for @v;

    # однословные подфразы
    my @res = @v;

    # добавляем двусловные
    push @res, map {
        join(' ', @v[$_, $_ + 1]),
        join(' ', @v[$_, $_ + 2]),
    } 0 .. ($#v - 2);
    push @res, join(' ', @v[-2, -1]) if @v >= 2;

    # трехсловные
    push @res, map {
        join(' ', @v[$_, $_+1, $_+2]),
        join(' ', @v[$_, $_+1, $_+3]),
        join(' ', @v[$_, $_+2, $_+3]),
    } 0 .. ($#v - 3);
    push @res, join(' ', @v[-3, -2, -1]) if @v >= 3;

    push @res, $self->norm_phr;

    my @texts = uniq(map { join(' ', sort split) } @res);
    return \@texts if $par{normtexts};

    my $proj = $self->proj;
    my @phrases = map { $proj->normed_phrase($_) } @texts;
    return $proj->phrase_list({ phrases_list => \@phrases});
}


sub all_subphrases_list_arrayref {
    my ($self) = @_;
    my @arr = ();
    my @words = sort $self->normwords;
    if((! $self->{max_phrase_words}) || (@words <= $self->{max_phrase_words})) {
        my $max_mask = 1 << scalar(@words);
        for(my $mask = 1; $mask < $max_mask; $mask++) {
            my $subphtext = join " ", map{$words[$_]} grep{$mask & (1 << $_)} (0 .. $#words);
            push(@arr, $subphtext);
        }
    }
    return \@arr;
}

sub all_snorm_subphrases_list_arrayref {
    my ($self, $wordslimit) = @_;
    my @arr = ();
    my @words = sort $self->snormwords;
    @words = @words[0 .. $wordslimit-1] if $wordslimit; #Ограничиваем количество слов, чтобы избежать огромного числа комбинаций
    if((! $self->{max_phrase_words}) || (@words <= $self->{max_phrase_words})) {
        my $max_mask = 1 << scalar(@words);
        for(my $mask = 1; $mask < $max_mask; $mask++) {
            my $subphtext = join " ", map{$words[$_]} grep{$mask & (1 << $_)} (0 .. $#words);
            push(@arr, $subphtext);
        }
    }
    return \@arr;
}

my $init_gap_done = 0;
my %sbhfl = ();
my %rdct = ();
sub init_gap {
    my %sbh = ();
    my $max_cnt = 10;
    my @gaplist = 0 .. (2**$max_cnt-1);
    my %dct = ();

    return if $init_gap_done;
    for my $m (@gaplist){
        my $msk = [ reverse split //, sprintf('%b', $m)];
        my $k = @$msk;
        $dct{$m} = [ map { $msk->[$_] ? $_ : () } 0 .. $k-1 ];
        $rdct{ join(' ',@{$dct{$m}}) } = $m;
    }
    for my $m (@gaplist){
        $sbh{$m} = [ map { $_ ? $dct{$_} : undef } map { (($m & $_) == $m)&&($m != $_) ? $_ : undef } @gaplist ];
        $sbhfl{$m}[$_] = [grep {$_} @{$sbh{$m}}[0 .. 2**$_ - 2 ]] for 1 .. $max_cnt;
    }
    $init_gap_done = 1;
}

sub get_gap {
    my ($self, $ph) = @_;
    init_gap;
    my @w1 = $self->normwords;
    my $k1 = @w1;
    my @w2 = $ph->normwords;
    my $k2 = @w2;
    return [] if $k1 < $k2 + 2; #вторая длиннее - нет смысла проверять
    my @curmsk = ();
    my $i2 = 0;
    for(my $i=0;$i<$k1;$i++){
        my $elcmp = defined($w2[$i2]) ? ($w1[$i] cmp $w2[$i2]) : -1;
        return [] if $elcmp > 0;
        push(@curmsk, $i), $i2++ if $elcmp == 0;
    }
    return [] unless $i2; #Не совпало ни одного слова
    my $mskkey = $rdct{join(' ', @curmsk)};
    return [ map { join ' ', @w1[ @$_ ] } @{$sbhfl{$mskkey}[$k1]}];
}

#возвращает уже нормализованные фразы
sub all_subphrases_list {
    my ($self) = @_;
    return $self->proj->phrase_list($self->all_subphrases_list_arrayref)->mark_phrs_normed;
}

sub vendors :CACHE {
    my ($self) = @_;
    my $vndrs = $self->proj->models->vendors;
    return grep {$vndrs->{$_}} $self->snormwords;
}

sub getmodelsphraseslist {
    my $self = shift;
    my $models = $self->proj->models;
    my @phrs = ( $models->get_phr_models($self)->phrases );
    return $self->proj->phrase_list( { phrases_list => \@phrs } );
}

sub has_latwords :CACHE {
    my $self = shift;
    return first { /^[-A-Za-z]{1,4}$/ } (split /\s+/, $self->text);
}

sub has_ruswords :CACHE {
    my $self = shift;
    return first { /^[-А-ЯЁа-яёA-Za-z]{1,4}$/ } (split /\s+/, $self->text);
}

sub has_digwords :CACHE {
    my $self = shift;
    return first { /^[\d.-]+$/ } (split /\s+/, $self->text);
}

sub look_like_a_model :CACHE {
    my $self = shift;
    return (
        $self->norm_phr =~ /([A-Za-zА-ЯЁа-яё][0-9]|[0-9][A-Za-zА-ЯЁа-яё])/ || (
            $self->has_digwords && (
                $self->has_ruswords || $self->has_latwords
            )
        )
    ) ? 1 : 0;
}

sub verjoin {
    my ($self, $arr) = @_;
    return @$arr unless @$arr > 1;
    my @source = split /(?=\s)|(?<=\s)/, join(' ', @$arr);
    my @space_indexes = map { $_ % 2 ? $_ : () } (0..$#source);

    # нельзя склеивать два слова вместе, если буквы склеиваются с буквами, или цифры с цифрами
    sub can_join {
        return 0 unless @_;
        my ( $arr ) = @_;
        return 1 if @$arr == 1;
        my $prev = shift @$arr;
        my $res = 1;
        # для каждой соседней пары из переданных
        for ( @$arr ) {
            next if !$_;
            # ...но можно, если склеиваются две одинокие буквы
            if ($_ =~ /^\d/ && $prev =~ /\d$/ ||  $_ =~ /^[$RU$LA]/ && $prev =~ /[$RU$LA]$/ && (length($_)>1 || length($prev)>1)) {
                $res = 0;
                last;
            }
            $prev = $_;
        }
        return $res;
    }

    my $space_positions = array_subsets_ordered( \@space_indexes ); # все комбинации словопозиций пробелов (массив строк)

    my @res = ();
    push @res, join('', @source);
    my %can_join_cache = ();
 L: for ( @$space_positions ){
        my @sp_line = split / /;
        for ( @sp_line ){
            if ( !exists $can_join_cache{$_} ) {
                $can_join_cache{$_} = can_join( [ $source[$_-1], $source[$_+1] ] );
            }
            next L unless $can_join_cache{$_}; # нельзя что-то склеить - переходим к другой строке
        }
        my @temp = @source;
        delete $temp[$_] for ( @sp_line ); # удаляем пробел, склеивая части
        @temp = grep { $_ } @temp;
        push @res, join( '', @temp );
    }
    return @res;
}

# без кеширования
# без фильтрации фраз по счетчику поиска
# есть предохранение от выкидывания значимой части модели справа
sub get_modellike_modifications_lite {
    my ($self, %par) = @_;
    my $mdl = $par{model} || '';
    return $self->proj->phrase_list([]) unless $self->look_like_a_model;

    my $text = $self->text;

    return $self->proj->phrase_list([$text]) if $self->wordcount > 8; # очень большие фразы не разваливаем

    unless ( $mdl ) {
        if ($par{fast}) {
            $mdl = $self->proj->phrase($text)->parse_model;
        } else {
            my %pres = $self->proj->phrase($text)->parse;
            $mdl = $pres{model} if $pres{model};
        }
    }
    return $self->proj->phrase_list( { phrases_arr => [$text] } ) unless $mdl;

    my $mdl_orig = $mdl;

    # заменяем конструкции вида 5x7x9.6
    $mdl = join ' ', map { s/(x|х)/___$1___/g if /^\d[\d.,]*((?:x|х)\d[\d.,]*){1,}$/i; s/(^|$)/___/g if /___/; $_ } split( /\s+/, $mdl ); ##no critic
    my $mdl_nosize = join ' ', grep { !/___/ } split( /\s+/, $mdl );

    my @arr = ();
    for my $mdl ( uniq_array(($mdl,$mdl_nosize)) ){ # второй вариант - с выкинутым размером
        # ищем первую часть слева, в которой есть цифры, она уменьшаться не будет
        my @stopper_parts = ();
        for my $part ( split /\s+/, $mdl ){
            push @stopper_parts, $part;
            last if ( $part =~ /\d/ && $part =~ /[a-z]/i );
        }
        my $stopper = join (' ', @stopper_parts );
        $stopper = $mdl if $par{no_shrink};

        # отделяем пробелами буквы и цифры
        $mdl =~ s/(([a-zа-яё])(?=\d)|(\d)(?=[a-zа-яё]))/$1 /gi;

        #заменяем модельные разделители пробелами
        $mdl =~ s/[-]/ /g;
        $mdl =~ s/\./ /g unless $mdl =~ /___/;
        my @pt = grep {$_} split /\s+/, $mdl;
        next unless @pt > 1;

        @pt = @pt[0..9] if @pt > 10;

        # разваливаем по моделям
        push @arr, $mdl_orig;
        push @arr, "@pt";
        while ( @pt && num_of_dw("@pt")>=num_of_dw($stopper) ){
            push @arr, $self->verjoin(\@pt);
            pop @pt;
        }
    }
    @arr = uniq_array( @arr );
    if (defined $par{max_mlike} and @arr > $par{max_mlike}) {
        # оставляем варианты с меньшим числом разбиений
        @arr = map { $_->[0] } sort { $a->[1] <=> $b->[1] } map { [ $_, scalar(split / /, $_) ] } @arr;
        @arr = splice(@arr, 0, $par{max_mlike});
    }

    $_=~s/___//g for @arr;
    # разваленные модели подставляем в исходную фразу
    my @res = ();
    for my $mlike (@arr) {
        my $phr = $text;
        my $mdl_orig_quoted = quotemeta($mdl_orig);
        eval { $phr =~ s/$mdl_orig_quoted/$mlike/i if $mdl_orig_quoted; };
        push @res, $phr;
    }
    @res = uniq_array(@res); #сильно ускоряет следующую строку
    my $resphl = $self->proj->phrase_list([@res])->pack_list;
    return $resphl unless $par{save_stops};
    return $resphl->set_exclamations_before_stops->set_exclamations_before_bsstops;

    sub num_of_dw {
        my ($txt) = @_;
        return scalar( @{[($txt=~/[a-zа-яё0-9]/gi)]} );
    }
}

# параметры см. в методе PhraseList
sub get_modellike_modifications {
    my $self = shift;
    my %par  = @_;
    $par{save_stops} //= 0;  # save old behaviour
    return $self->self_phl->get_modellike_modifications(%par);
}


# не парсит модель, в отличие от get_modellike_modifications
sub get_modellike_modifications_without_model_parsing {
    my ($self) = @_;
    my $static_part = '';
    my $changeable_part = '';
    my $brand = $self->get_brand;
    $brand = '' if ( $brand && $self->text =~ /(?<=[a-z])\s+$brand/i );
    # предполагается, что могут быть брэнд, русская часть (товар), остальное - модель
    if ( $brand ){
#        print "brand: $brand\n";
        ( $static_part, $changeable_part ) = split /(?:^|\s+)$brand(?:\s+|$)/i, $self->text;
        $static_part = $static_part ? "$static_part $brand" : $brand;
    } elsif ( $self->text !~ /[а-яёА-ЯЁ]/ ){
        $changeable_part = $self->text;
    } else {
        ( $static_part, $changeable_part ) = _split_4modellike($self->text);
    }
    $changeable_part = '' if ( $changeable_part !~ /\d/ || $changeable_part !~ /[a-zA-Z]/ );
#    print Dumper ( [ $static_part, $changeable_part  ] );
    return $self->proj->phrase_list( [ $self->text ] ) unless $changeable_part;

    # пробуем расширить число моделей, отбрасывая слева части, в которых нет цифр
    my @temp = split /\s+/,$changeable_part;
    my @changeable_models = ();
    push @changeable_models, $changeable_part;
    while ( @temp ){
        my $word = shift @temp;
        last if $word =~ /[0-9]/;
        push @changeable_models, "@temp";
    }

    my $phl = $self->proj->phrase_list;
    for my $model ( @changeable_models ){
        $phl += $self->proj->phrase($model)->get_modellike_modifications( advq_threshold=>0, use_cache=>0, no_filter_by_search_count=>1, model=>$model );
    }
#   print Dumper ( $phl->perl_array );
    $phl = $phl->lmap(sub { "$static_part $_" }) if $static_part;
    $phl = $phl->lgrep(sub { !$_->is_brand_exactly && !$_->is_wide_phrase && $_->text !~ /^[a-z]+$/i && !(length($_->text)<3) });
    return $phl;
}

sub _split_4modellike {
    my ( $txt ) = @_;
    my $rus_part = '';
    my $lat_part = '';
    my @temp = split /\s+(?=(?:[-a-zA-Z0-9 ]+|[А-ЯЁ]+))/, $txt;
    for my $i (0..$#temp){
        if ( $temp[$i] !~ /^(?:[А-ЯЁ-]+|[а-яё-]+)$/ ){
            $rus_part = join (' ', @temp[0..$i-1]);
            $lat_part = join (' ', @temp[$i..$#temp]);
            last;
        }
    }
    $lat_part = $txt unless ( $rus_part && $lat_part );
    return ( $rus_part, $lat_part );
}

sub modellike_product {
    my ($self, %hparsed) = @_;
#    print Dumper ( [$self->text, \%hparsed] );
    my $phlres = $self->proj->phrase_list([$self->text]);
    for my $key ( qw/model model_for/ ){
        next unless defined $hparsed{$key};
        my $model_src = $hparsed{$key};

        my @models_arr = ($model_src);
        # Models format: 'model' or '[model1:model2]' or even '[model]'
        if ($model_src =~ /^\[(.*)\]$/) {
            @models_arr = split(/:/, $1) if $1;
        }
        for my $model_src (@models_arr) {
            next unless $model_src;
#           print STDERR Dumper (['before shr', $self->proj->phrase_list([$model_src])->perl_array]);
            my $keep_all_words = $hparsed{_model_has_underscores} || 0;
            my $phlshr = $self->proj->phrase_list([$model_src])->left_shrinking_models_extend(keep_all_words => $keep_all_words);
#           print Dumper (['after shr', $phlshr->perl_array]);
            for my $phrms ( @$phlshr ) {
                my $phrtext = $self->text;
                my $model_shrinked = $phrms->text;
                my $model_src_reg = quotemeta $model_src;
                my $model_shrinked_reg = quotemeta $model_shrinked;
                $phrtext =~ s/$model_src_reg/$model_shrinked_reg/i unless $model_shrinked eq $model_src;
#               print Dumper ( [$phrtext, $model_shrinked] );
                $phlres += $self->proj->phrase($phrtext)->get_modellike_modifications_lite( model=>$model_shrinked );
#               print Dumper ( $phlres->perl_array );
            }
        }
    }
#    $phlres = $phlres->phgrep( sub { length($_->norm_phr) ne length($_->text) } )->lgrep( sub { length($_)>3 } );
    $phlres = $phlres->lgrep( sub { length($_->text)>2 && $_->wordcount < 5 && length($_->norm_phr) eq length($_->text) } );
    return $phlres;
}

# возвращает ссылку на массив хешей, содержащих текст, норм и снорм бивордов
sub get_biwords_dict_with_norm_and_snorm :GLOBALCACHE {
    my $self = shift;

    my $dict = $self->proj->dict_manager->get_dict("biwords");
    my @arr = ();
    for my $phr (@{$dict->phrase_list}) {
        my %res = ();
        my $lang = $phr->lang;
        $res{text} = $phr->text;
        $res{norm} = word2norm($res{text}, $lang);
        $res{snorm} = word2snorm($res{text}, $lang);
        push @arr, \%res;
    }

    return \@arr;
}

# возвращает информацию о мультисловах: список [dict,start_pos,stop_pos]
# словопозиции соотв. методу $phr->normwords, т.е. стопы не считаем:
# "узнать курс доллара и посмотреть курс евро" => (['biwords',1,2],['biwords',4,5])
sub get_multiwords_inf :CACHE {
    my $self = shift;

    my $proj = $self->proj;
    my $dm = $proj->dict_manager;

    my $trie = $dm->get_dict('multiwords')->{trie};
    my $biwords = $dm->get_dict('biwords')->{pairs};
    my $biwords_strict = $dm->get_dict('biwords_strict')->{pairs};
    my $lang = $self->lang;

    my @mulpos;

    # strict - normwords
    my @nw = $self->normwords;
    for my $i (0 .. ($#nw-1)) {
        my @pair = sort ($nw[$i], $nw[$i+1]);
        push @mulpos, [ 'biwords_strict', $i, $i+1 ] if $biwords_strict->{$pair[0], $pair[1]};
    }

    # snormwords
    my @w = $self->snormwords;

    # traverse hash
    for my $i (0 .. ($#w-1)) {
        my @pair = sort ($w[$i], $w[$i+1]);  # сейчас такая логика -- в бивордах не учитываем порядок!
        push @mulpos, [ 'biwords', $i, $i+1 ] if $biwords->{$pair[0], $pair[1]};
    }

    # traverse trie (via khromov)
    my @todo = map { [ $_, $_, $trie ] } 0 .. $#w;
    while (@todo) {
        my ($start, $pos, $node) = @{shift @todo};
        my $word = $w[$pos];
        my $child = $node->{$word} or next;
        push @mulpos, [ 'multiwords', $start, $pos ] if $child->{" "};  # нашли
        push @todo, [ $start, $pos + 1, $child ] if $pos < $#w;
    }

    return @mulpos;
}

# то же, что get_multiwords_inf, но с оригинальными словопозициями
# вместо start,stop выдаём полный список словопозиций, чтобы исключить стоп-слова
sub get_multiwords_orig_pos :CACHE {
    my $self = shift;

    my @mulpos = $self->get_multiwords_inf;
    return () if !@mulpos;

    my @pos = map { $_->{pos} } grep { !$_->{stop} } $self->analyze_words;
    return map { [ $_->[0], [ @pos[($_->[1] .. $_->[2])]]] } @mulpos;
}


sub get_multiwords_hashes :CACHE {
    my $self = shift;
    my @sw = $self->snormwords;
    return map { +{ map { $sw[$_] => 1 } ($_->[1] .. $_->[2]) } } $self->get_multiwords_inf;
}

sub _get_negative_words_with_pluses :GLOBALCACHE {
    my $self = shift;
    return { map {$_ => 1} map { $self->proj->phrase($_)->text_add_pluses_to_stop_words } ('не','без','никакой','никакая','никакое','никакие','никакого','никакой','никаких','никакому','никаким','нет','нельзя','невозможно') };
}

sub get_negations_arr {
    my $self = shift;
    my $neg_hash = $self->_get_negative_words_with_pluses;
    my $mark = 0;
    my @result;
    for my $word ($self->words) {
        if ($mark) {
            push @result, $self->proj->phrase($word)->snorm_phr;
            $mark = 0;
        }
        $mark = 1 if exists $neg_hash->{$word};
    }
    return map {'__no_'.$_} grep {$_} @result;
}

#Анализ содержимого текста
sub text_content_type :CACHE {
    my ($self) = @_;
    my $text = $self->text;
    my @res = ();
    push( @res, 'ru' ) if $text =~ /[А-Яа-яёЁ]/;
    push( @res, 'en' ) if $text =~ /[A-Za-z]/;
    push( @res, 'dg' ) if $text =~ /[0-9]/;
    push( @res, 'en_dg' ) if $text =~ /[A-Za-z][0-9]|[0-9][A-Za-z]/;
    push( @res, 'ru_dg' ) if $text =~ /[А-Яа-яёЁ][0-9]|[0-9][А-Яа-яёЁ]/;
    return join('  ', @res);
}

sub word_is_not_escaped {
    my $word = shift;
    return ($word !~ /^[-!+]/);
}


#Замена букв в словах
our @rr = (
        'AaOoEePpYyTKkXxCcBbMmnH',
        'АаОоЕеРрУуТКкХхСсВьМмпН',
);
our @a1 = split //, $rr[0];
our @a2 = split //, $rr[1];
our %h_er = ();
our %h_re = ();
$h_er{$a1[$_]} = $a2[$_] for 0 .. @a1-1;
$h_re{$a2[$_]} = $a1[$_] for 0 .. @a1-1;
#/ Замена букв в словах
sub _fix_bad_words {
    my $t = shift;
    my $t1 = $t;
#    $t1 =~ s/([$rr[0]])/$h_er{$1}/ge; #пробуем заменить в одну сторону
    $t1 =~ s/([$rr[0]])/$h_er{$1}/ge; #пробуем заменить в одну сторону
    return $t1 if $t1 !~ /[A-Za-z]/; #заменили все английские буквы
    my $t2 = $t;
    $t1 =~ s/([$rr[1]])/$h_re{$1}/ge; #пробуем заменить в одну сторону
#    $t2 =~ tr/$rr[1]/$rr[0]/; #пробуем заменить в другую сторону
    return $t2 if $t2 !~ /[А-Яа-я]/; #заменили все русские буквы
    return $t;
}
sub _fix_bad_text {
    my $t = shift;
    $t =~ s/([А-Яа-я]*([A-Za-z]+[А-Яа-я]+)+)/_fix_bad_words($1)/eg;
    return $t;
}

sub _is_word_mixed {
    my $text = shift;
    return 1 if ( $text =~ /[A-Za-z]/i && $text =~ /[А-ЯЁа-яё]/i && $text !~ /\d/ );
    return 0;
};

our @sp_prefilter_phrases = (
    'из рук в руки',
    'на диване',
    'мой мир',
    'под ключ',
    'сам себе',
    'своими руками',
    'для чайников',
    'для душа и души',
    'в добрые руки',
    'по для',
    'от а до я',
    'магазин на диване',
);
our $sp_pref_phrs_re = join '|', @sp_prefilter_phrases;
sub rplsp { my $t = shift; $t =~ s/\s/_/g; return $t; }

sub fltrd_text {
    my ($self) = @_;
    my $s = $self->text;
    $s =~ s/\b($sp_pref_phrs_re)\b/rplsp($1)/ieg;
    return $s;
}

sub get_banner_prefiltered_phrase {
    my ($self, %parameters) = @_;
    return $self->language->prefilter->get_prefiltered_phrase($self->text, %parameters);
}

sub get_banner_prefiltered_text {
    my ($self) = @_;

    return {
        text    => $self->get_banner_prefiltered_phrase(debug => 1)->text,
        debug   => $self->language->prefilter->{debug_info},
    };
}

sub get_xml_search {
    require XML::Simple;
    my ($self, %opts) = @_;
    my $ua = LWP::UserAgent->new(
        agent => "Yacobot/1.0",
    );

    my ($query, $mns) = $self->split_by_minus_words;
    $query = join(' & ', map {"+$_"} map { s/\./ /g; $_ } split( /\s+/, $query)); ##no critic
#print "SS: $query\n";
    $query = uri_escape_utf8($query);

    my $params = join("&", (map{"$_=$opts{$_}"} keys %opts), "query=".$query);
    my $qurl = "http://yandex.ru/search/xml?$params";
    my $resp = $ua->get($qurl);
    #print "$qurl\n";

    if(!$resp->is_success) {
        $self->log("ERROR: xmlsearch failed ".$resp->status_line);
        return ();
    }

    my $text = $resp->decoded_content;
    $text =~ s/<\/?hlword>//g; # удаляем ненужные тэги

    my $xml;
    eval {
        no warnings "all";
        $xml = XML::Simple::XMLin($text);
    };

    my $groups = ((($xml->{response} || {})->{results} || {})->{grouping} || {})->{group} || [];
#$self->proj->dd($qurl, $xml->{response}, ($xml->{response} || {})->{results}, (($xml->{response} || {})->{results} || {})->{grouping}, $groups);
    my @result;
    if(ref($groups) eq 'ARRAY'){
        for my $group (@$groups) {
            my $doc = $group->{doc} || {};
            push @result, $doc;
        }
    }else{
        @result = ();
    }

    return @result;
}

sub get_yandex_phraselist {
    my ($self, %opts) = @_;
    $opts{'groupby'} //= 'groups-on-page%3D100'; #100 документов на страницу
    my @docs = $self->get_xml_search(%opts);

    return $self->proj->phrase_list({phrases_arr => [ map{$_->{title}} @docs]})->tmap(sub { s/,/ /g; $_ });
}

sub get_yandex_domains {
    my ($self) = @_;
    my @docs = $self->get_xml_search;

    return map { $_->{'domain'} } @docs;
}

sub get_yandex_urls {
    my ($self) = @_;
    my @docs = $self->get_xml_search;

    return map { $_->{'url'} } @docs;
}

########################################################################
# Работа с парадигмами слов
########################################################################

# быстрая версия, с проверкой по буквам (как в analyze_text_lang в Директе)
sub lang_recognize {
    my $self = shift;
    my $text = $self->text;

    return recognize_text_lang($text);
}

sub classic_recognize {
    my $self = shift;
    return recognize_text_lang($self->text, {use_regex => 0});
}

sub get_remotecache_id {
    my ($self) = @_;
    return $self->lang.':'.$self->norm_phr;
}

sub get_phrase_without_hyphens {
    my ($self) = @_;
    my $text = join " ", map { split /\-/, $_ } $self->pluswords;
    return $self->proj->phrase($text);
}

sub dumper_text_lite {
    my ($self) = @_;
    $_[0] = \"phrase: $self";
    return @_;
}

sub spec_text : REMOTECACHE {
    my ($self) = @_;
    print "run spec_text:$self\n";
    return "spec_text:".$self->text;
}

sub get_harmonized_text :CACHE {
    my ($self) = @_;

    if(!defined($self->{cdict_harm})) {
        my $res = $self->get_harmonized_text_from_cdict // $self->norm_phr;
        $self->{cdict_harm} = $res;
    }

    return $self->{cdict_harm};
}

#гармонизация из cdict на основе запросов
sub get_harmonized_text_from_cdict :CACHE {
    my ($self) = @_;

    my %res = $self->proj->cdict_client->get_harmonized_hash($self);
    return $res{$self->norm_phr};
}

sub get_search_query_count {
    my ($self) = @_;
    if(!defined($self->{search_query_count})) {
        $self->self_phl->cache_search_query_count(force => 1);
    }
    return $self->{search_query_count};
}

sub get_mobile_count {
    my ($self) = @_;
    if(!defined $self->{cdict_countm}) {
        $self->self_phl->cache_mobile_counts(force => 1);
    }
    return $self->{cdict_countm};
}

# хэш { $tail => $count }
sub tail2count {
    my $self = shift;
    if (!defined $self->{tail2count}) {
        $self->self_phl->cache_cdict_tail_categs(force => 1);
    }
    return $self->{tail2count};
}

# хэш { $tail => [ $categ_ids ] }
sub tail2categs {
    my $self = shift;
    if (!defined $self->{tail2categs}) {
        $self->self_phl->cache_cdict_tail_categs(force => 1);
    }
    return $self->{tail2categs};
}

# фрейзлист из самой фразы, для вызова методов из PhraseList*
# кэшировать через :CACHE нельзя во избежание циклических ссылок
sub self_phl {
    my $self = shift;
    $self->proj->phrase_list([$self]);
}

# частота фразы на поиске
# на входе:
#   $region_ids =  ссылка на массив плюс и минус регионов
# доп. параметры (если задан $region_ids):
#   is_subtree => 0/1 - регионы уже преобразованы в поддерево
sub get_search_count {
    my $self = shift;
    my $region_ids = shift;
    my %par = @_;

    if (defined $region_ids and ref($region_ids) ne 'ARRAY') {
        # deprecated - задан один регион вместо списка
        $region_ids = [ $region_ids ];
    }

    # если регионы не указаны, возвращаем общую частоту
    if(!$region_ids || !@$region_ids) {
        if(!defined($self->{search_count})) {
            $self->self_phl->cache_search_count(force => 1);
        }
        return $self->{search_count};
    }

    # частота по указанным регионам
    my @subtree;
    if ($par{is_subtree}) {
        @subtree = @$region_ids;
    } else {
        @subtree = Utils::Regions::geobase_subtree($region_ids);
    }

    my $reg_cnt = $self->get_regions_count;
    return sum(0, map{ $reg_cnt->{$_} // 0 } @subtree);
}

# частота для снорма
sub get_snorm_search_count {
    my ($self) = @_;
    if(!defined($self->{snorm_search_count})) {
        my $snorm2count = $self->self_phl->get_snorm_search_counts;
        $self->{snorm_search_count} = $snorm2count->{$self->snorm_phr};
    }
    return $self->{snorm_search_count};
}

sub get_regions_count {
    my ($self) = @_;
    if(!defined $self->{regions_count}) {
        $self->self_phl->cache_regions_count(force => 1);
    }
    return $self->{regions_count};
}

sub remove_porn_minuswords {
    my ($self) = @_;
    return join(" ", $self->plus_text, map{"-$_"} grep{!$porn->{word2norm(norm2snorm($_))}} $self->minuswords);
}

#Оставляем только те минус-слова, у которых есть частоты
sub delete_useless_minus_words {
    my ($self, $trh, $regids) = @_;
    $trh //= 0;
    my $proj = $self->proj;

    #Для слов в кавычках упрощённая логика
    return $proj->phrase($1) if $self->text =~ /^\s*("[^"]+")\s+-/;

    my ($t, $tmn) = $self->split_by_minus_words;
    my @mns = $self->minuswords;
    my $h = {}; #Хэш соответствия слов и фраз с ним
    my @arr = (); #Массив фраз с добавленным словом
    for my $mw (@mns){
        my $ph = $proj->phrase("$t $mw");
        $h->{$mw} = $ph;
        push(@arr, $ph);
    }
    #$proj->phrase_list(\@arr)->cache_regions_count; #Кэшируем частоты, чтобы не ходить много раз
     if(!$regids || !@$regids) {
         $proj->phrase_list(\@arr)->cache_search_count;  #Кэшируем частоты, чтобы не ходить много раз
     } else {
         $proj->phrase_list(\@arr)->cache_regions_count; #Кэшируем частоты, чтобы не ходить много раз
     }
    #$proj->dd( $self, [ map { "$_ => ".$h->{$_}->get_search_count($regids) } @mns ], $regids );
    @mns = grep { $h->{$_}->get_search_count($regids) > $trh } @mns;
    return $proj->phrase($t.join("", map {" -$_"} @mns));
}

sub get_search_syns {
    my ($self) = @_;
    if (!defined $self->{search_syns}) {
        $self->self_phl->cache_search_syns(force => 1);
    }
    return $self->{search_syns};
}

sub get_search_syns_keeping_minuswords :CACHE {
    my ($self) = @_;
    my $minus = $self->minus_text // '';
    $minus = " $minus" if $minus =~ /^[^\s]/;  # непусто + начинается не с пробела
    return $self->get_search_syns->lmap(sub { $_->text.$minus });
}

sub get_search_tail {
    my ($self) = @_;
    if (!defined $self->{search_tail}) {
        $self->self_phl->cache_search_tail(force => 1);
    }
    return $self->{search_tail};
}

sub get_search_tail_phl {
    my ($self, %par) = @_;
    my $thre = $par{threshold} || 0;
    my $top_size = $par{top_size};
    #my $text = $self->text;
    my ($pls, $mns) = $self->split_by_minus_words;
    $mns //= '';
    my $flt = { map { $_ => 1 } $self->norm_minuswords }; #Фильтр по существующим минус-словам
    my $h = $self->get_search_tail;

    # отбираем топ, если указан размер
    if($top_size) {
        my @top = sort{$h->{$b} <=> $h->{$a}} keys %$h;
        $h = { map{$_ => $h->{$_}} @top[0..min($#top, $top_size - 1)]};
    }

    #print Dumper( [ keys %$h ] );
    return $self->proj->phrase_list([ map { "$pls $_ $mns" } grep { ! $flt->{$_} } grep {$h->{$_} > $thre } keys %$h ]);
}

# отобрать топ запросов по фразе с учетом минус-слов
# возвращает список фраз в кавычках
sub get_matched_queries :CACHE {
    my $self = shift;
    return $self->self_phl->get_matched_queries->{$self->text} // $self->proj->phrase_list([]);
}

#Разбивает текст по запятым, нормализует фразы и возвращает массив текстов
sub phrase_list_text_norm_arr {
    my ($self) = @_;
    my @arr = split ',', $self->text;
    s/ -.*$// for @arr; #удаляем минус-слова - ускоряет дальнейшую обработку
    return phrases2norm( \@arr );
}

sub phrase_list_text_norm {
    my ($self) = @_;
    return join ',', $self->phrase_list_text_norm_arr;
}

#возвращает общий префикс двух фраз
sub get_common_prefix {
    my ($self, $ph) = @_;
    my $pref = '';
    my @w1 = $self->words;
    my @w2 = $ph->words;
    return $pref unless ( @w1 and @w2 );
    for( my $i=0; $i < @w1; $i++ ){
        last if ( !$w2[$i] or ($w1[$i] ne $w2[$i]) );
        $pref .= $pref ? ' '.$w1[$i] : $w1[$i];
    }
    return $self->proj->phrase($pref);
}

#Обрезание текста до определённой длины
sub cuttext {
    my ($self, $lngth) = @_;
    $lngth ||=80;
    my $text = $self->text;
    return substr( $text, 0, $lngth-3).'...' if length($text) > $lngth;
    return $text;
}

#Разрезаем длинный текст на части
sub splitlongtext {
    my ($self, $lngth) = @_;
    $lngth ||=80;
    my $text = $self->text;
    #$self->proj->dd([$text, length($text)]);
    if( length($text) > $lngth ){
        $text =~ /^(.{0,$lngth} )(.{0,1000}(?:\s|$))?(.{0,10000}(?:\s|$))?(.*)/   #Разбиваем строку по пробелам
        || $text =~ /^(.{0,$lngth})(.{0,1000})?(.{0,10000})?(.*)/;                #Если нет пробелов (непрерывная строка), то дробим просто по символам
        my @arr = grep {$_} ( $1, $2, $3, $4 );
        #$self->proj->dd(['INS', \@arr]);
        return \@arr;
    }
    return [$text];
}

# числовой идентификатор текста
# используется в модерации Каталогии
sub text_id :CACHE {
    my $self = shift;
    my $text = $self->text;
    chomp($text);
    return md5int($text);
}

# айдишник, выставляемый БК
sub bs_id :CACHE {
    my $self = shift;
    return md5int($self->external_text);
}

# айдишник, уникальный в рамках привязки к баннеру
sub minf_id :CACHE {
    my $self = shift;
    return $self->minf->{SimDistance}.':'.$self->bs_id;
}

# айдишник для снормализованной фразы
# можно посолить
sub snorm_id {
    my $self = shift;
    my $salt = shift // '';
    return md5int($self->snorm_phr . $salt);
}
# айдишник для нормализованной фразы
# можно посолить
sub norm_id {
    my $self = shift;
    my $salt = shift // '';
    return md5int($self->norm_phr . $salt);
}

# экранируем плюсами мультиворды (строгие)
sub escape_multiwords :CACHE {
    my $self = shift;

    my @mulinf = $self->get_multiwords_orig_pos;
    @mulinf = grep { $_->[0] eq 'biwords_strict' } @mulinf;
    return $self if !@mulinf;

    my %seen = map { $_ => 1 } map { @{$_->[1]} } @mulinf;

    my @res;
    my @words = $self->words;
    for my $i (0 .. $#words) {
        my $word = $words[$i];
        if ($seen{$i}) {
            push @res, '+'.$word;
        } else {
            push @res, $word;
        }
    }
    return $self->proj->phrase(join(' ', @res));
}

# экранируем стоп-слова
# параметры:
#   $op - оператор, которым экранируем
sub escape_stop_words {
    my $self = shift;
    my $op = shift;
    my @words;
    for my $r ($self->analyze_words) {
        if ($r->{stop}) {
            push @words, $op.$r->{word};
        } else {
            push @words, $r->{word};
        }
    }
    return $self->language->phrase(join(' ', @words));
}

sub set_pluses_before_stops { return $_[0]->escape_stop_words('+') }
sub set_exclamations_before_stops { return $_[0]->escape_stop_words('!') }

# проставляет восклицательнве знаки перед стоп-словами без потерь нормализации
# не теряет квадратные скобки в фразе, которые фиксируют порядок слов
sub set_exclamations_before_stops_safe {
    use Utils::Words;
    my $self = shift;
    my @res = ();
    for my $word (split /\s+/, $self) {
        $word =~ s/^([\[\"]*)//;
        my $l = $1;
        $word =~ s/([\]\"]*)$//;
        my $r = $1;
        $word = $self->proj->phrase($word)->norm_phr_safe;
        $word = '!'.$word if stop4norm($word, $self->lang);
        push @res, $l.$word.$r;
    }
    return $self->proj->phrase("@res");
}

sub set_exclamations_before_bsstops {
    use Utils::Words;
    my $self = shift;
    my @res = ();
    for my $word (split /\s+/, $self) {
        $word =~ s/^([\[\"]*)//;
        my $l = $1;
        $word =~ s/([\]\"]*)$//;
        my $r = $1;
        $word = $self->proj->phrase($word)->norm_phr_safe;
        $word = '!'.$word if bsstop4word($word) && !stop4norm($word, $self->lang);
        push @res, $l.$word.$r;
    }
    return $self->proj->phrase("@res");
}

sub replace_exclamations_with_pluses {
    use Utils::Words;
    my $self = shift;
    $self =~ s/\!([\w\d])/\+$1/g;
    return $self;
}

# возвращает phrase list, где во фразу подставлены ее контекстные синонимы
sub get_csyns {
    my ( $self ) = @_;
    my @res = ();

    my $source = $self;
    my $csyn = $self->proj->context_syns->{synonyms};
    my $source_csyn = $csyn->parse_phr($self, strict=>1, ret_words=>1);

    #print STDERR "source phrase: [",$self->text,"]\n";

    for my $syn_phr ( $source_csyn->phrases ) {
        my $text = $self->text;
        #print STDERR "synonym: [".$syn_phr->text."]\n";
        my $replaced = join ' ', @{$syn_phr->inf};
        #print STDERR "replaced: [$replaced]\n";
        my $harmonized_syn = $self->proj->phrase($replaced)->harmonize( $syn_phr->text );
        #print STDERR "harmonized syn: [$harmonized_syn]\n";
        next unless $harmonized_syn;
        $harmonized_syn =~ s/((?<=\s)|(?<=^))\!(?=\w)//g;
        $replaced =~ s/ /\\s\+/g;
        next if ( lc($replaced) eq lc($harmonized_syn) );
        while ( $text =~ /(?:^|\s+|[^а-я0-9-`])($replaced)(?:\s+|[^а-я0-9-`]|$)/i ) { # заменяем все вхождения с сохранением регистра каждого
            my $matched = $1;
            #print STDERR "replacement matching: [$matched]\n";
            $harmonized_syn =~ s/^(.*)$/\U$1/ if $matched =~ /^([ A-ZА-ЯЁ0-9-])+$/;
            $harmonized_syn =~ s/^(.)/\U$1/ if $matched =~ /^[A-ZА-ЯЁ][a-zа-яё]/;
            $text =~ s/$replaced/```$harmonized_syn```/i;
        }

        $text =~ s/```//g;
        #print STDERR "pushed into array: [$text]\n";
        push @res, $text;
    }

    #print STDERR Dumper (\@res);
    return $self->proj->phrase_list( { phrases_arr => \@res } );
}

sub get_context_arr :REMOTECACHE {
    my ($self) = @_;
    #print $ph->get_yandex_domains;
    my $phl = $self->proj->phrase_list([$self, $self->get_search_syns->phrases]);
    my $srchphl = $phl->get_yandex_phraselist;
#print $phl."<<<\n";
#exit;
#    $phl = $phl->add_search_tail;
    $phl += $srchphl;
#    $phl = $phl->lgrep(sub {$_->text =~ /холодильный/});
#    print $phl;
#exit;
    my %f = map {$_=>1} $self->snormwords;
#print Dumper(\%f);
    my %h = ();
    $h{$_}++ for grep {!$f{$_}} map {$_->snormwords} @$phl;
#print join( ', ',  map {$_->snormwords} @$phl)."<<<\n";
#print Dumper(\%f, \%h);
#exit;
    my @res = h2top(\%h, 0.01, 0.01);
    return \@res;
    #my $dt = [ map { $_->[0] } @res ];
    #return $dt;
    #return "DDD";
    #print $phl->get_yandex_phraselist;
}

sub get_context_tail_arr :REMOTECACHE {
    my ($self) = @_;
    my $phl = $self->proj->phrase_list([$self, $self->get_search_syns->phrases]);
    my $srchphl = $self->get_search_tail_phl;
    $phl += $srchphl;
    my %f = map {$_=>1} $self->snormwords;
    my %h = ();
    $h{$_}++ for grep {!$f{$_}} map {$_->snormwords} @$phl;
    my @res = h2top(\%h, 0.01, 0.01);
    return \@res;
}

sub get_tail_context {
    my ($self) = @_;
#    return [ map { $_->[0] } @{$self->get_context_arr} ];
    return [ map { $_->[0] } @{$self->get_context_tail_arr} ];
}

sub get_search_context {
    my ($self) = @_;
    return [ map { $_->[0] } @{$self->get_context_arr} ];
}

sub get_context {
    my ($self) = @_;
#    return [ map { $_->[0] } @{$self->get_context_arr} ];
    return [ map { $_->[0] } @{$self->get_context_tail_arr} ];
}

our $tcnt = {
    spec => [qw{ работа кухня }],
    prod => [qw{ купить интернет-магазин магазин каталог товар цена сеть скидка бу объявление закупка сколько кредит маркет стоять }],
    discount => [qw{ акция дискаунт скидка }],
    grp => [qw{ блок система }],
    mgr => [qw{ дюйм см л литр объем }],
    rent => [qw{ снимать }],
    gift => [qw{ подарок }],
    chld => [qw{ детский ребенок }],
    access => [qw{ подставка крепление дверца панель полка стекло насадка кронштейн стена }],
    sell => [qw{ продавать }],
    free => [qw{ бесплатный }],
    material => [qw{ дерево металл пластиковый железный материал покрытие }],
    patriot => [qw{ русский }],
    color => [qw{ цвет черный белый красный розовый }],
    food => [qw{ рецепт приготавливать запекать готовить приготовление }],
    food2 => [qw{ сэндвич мед плов котлета сосиска запеканка кефир вкусный грудка пирог фарш фаршированный рыба свинина яйцо молоко гречка яблоко куриный крылышко курица хлеб шампиньон сыр сухарь скумбрия творог шашлык выпечка блин свекла пельмень рис горшочек мундир горбуша окорочок гриб согревать печеный испечь свиной голубец жареный гриль жир пирожок пицца лимон торт сухарик фри безе ножка блюдо макароны говядина креветка омлет шарлотка капуста филе картошка семга тыква каша мясо кекс тесто овощ пожарить тушеный майонез сливки }],
    bad => [qw{ со г стр the где под год через м куда два д }],
    brands => [qw{ lg samsung panasonic ariston braun aeg аег bosch бош whirlpool керхер philips electrolux gorenje siemens сименс beko sharp bork jvc rolsen vitek гефест rowenta haier дэу горение vestel hotpoint indesit zanussi веко whirlpool вирпул атлант sony viewsonic tefal цептер elenberg delonghi лж лджи кайзер supra макита toshiba }],
    house => [qw{ бытовой техника дом }],
    srv => [qw{ установка встраивать ремонт подключать ленремонт обслуживание мастер экспресс-сервис чистка поменять заменять запчасть работать замена течь проблема сервис сломать монтаж почистить повесить накипь ошибка код греться модуль }],
    class => [qw{ промышленный портативный домашний настенный встроенный класс тип электрический настольный напольный 3d 3д цифровой универсальный электронный }],
    prms => [qw{ характеристика выбор совет форум какой рекомендация отзыв сравнивать совместимость ли узнавать обзор эксплуатация против критерий технический статья тест производитель габарит подбирать производство модель режим вид рейтинг размер мощность функция марка отличаться качество }],
    prms2 => [qw{ низкий высокий специализированный большой сверхвысокий быстрый широкий мощный мини маленький ширина высота верхний легкий узкий глубина форма вертикальный вес }],
    trns => [qw{ самовывоз доставка опт заказ круглосуточный склад курьер поставка поставлять }],
    org => [qw{ ооо оао компания }],
    tps  => [qw{ новый выгодный дешевый хороший правильный целесообразный должный интересный оптимизация старый дорогой простой красивый профессиональный надежный люкс pro }],
    cites => [qw{ город регион санкт-петербург москва спб уфа минск петербург красноярск тула самара екатеринбург краснодар донецк одесса днепропетровск украина новосибирск челябинск ростов новгород воронеж саратов пермь харьков киев казань нижний волгоград омск тверь санкт ижевск тюмень алматы запорожье владивосток }],
    countries => [qw{ германия немецкий китайский китай }],
    sites => [qw{ techport калинка mtonline мт-онлайн эльдорадо видео м-видео рука авито polaris поларис витька }],
    internet => [qw{ internet интернет онлайн ru by ру www html сайт яндекс mail }],
    dngr => [qw{ вредный опасный вред здоровье }],
    srvinf => [qw{ инструкция настойка настройка библиотека драйвер скачать документация ответ официальный руководство проверять можно программа разборка схема включать нельзя нужный почему делать сбор применение прошивка управление работа находить показывать отключать разъем отмывать обновление }],
    srvinf2 => [qw{ картинка рисунок фото смотреть видеть прикол рисовать реклама презентация фотка книга торрент мультик }],
    games => [qw{ игра minecraft майнкрафт игрушка }],
    inf => [qw{ реферат википедия bestreferat курсовой пособие викизнание принцип устройство помощь миф история действие факт тема термин словарь }],
    made => [qw{ самодельный сделать }],
};
our $tcntflt = { map {$_=>1} map {@$_} values %$tcnt };
sub get_df_contexts : CACHE {
    my ($self) = @_;
    my $arr = [ grep {! $tcntflt->{$_} } grep {!/^\d+$/} @{$self->get_context} ];
    return $self->proj->phrase_list($arr);
}

sub get_bnr_count {
    my ($self) = @_;
    return $self->{cdict_bnr_count} // ($self->proj->cdict_client->get_bnr_count( $self ))[0];
}

our $spec_cities_vrs = {
    'спб' => 'спб',
    'орел' => 'орловский',
    'махачкала' => 'махачкалинский',
};
sub get_adjective {
    my ($self) = @_;
    my $w = $self->text;
    return $spec_cities_vrs->{$w} if $spec_cities_vrs->{$w};
    return $w."ий" if $w =~ /ск$/;
    return $w."ий" if $w =~ /цк$/;
    return $w if  $w =~ /(кий|кая|кие)$/;
    return $w if  $w =~ s/ава$/авский/;
    return $w if  $w =~ s/ва$/овский/;
    return $w if  $w =~ s/ха$/хинский/;
    return $w if  $w =~ s/л$/льский/;
    return $w if  $w =~ s/ц$/цкий/;
    return $w if  $w =~ s/ный$/ненский/;
    return $w if  $w =~ s/вка$/вский/;
    return $w if  $w =~ s/на$/ненский/;
    return $w if  $w =~ s/ла$/льский/;
    return $w if  $w =~ s/гда$/годский/;
    return $w if  $w =~ s/ма$/мской/;
    return $w if  $w =~ s/([нв])о$/${1}ский/;
    return $w if  $w =~ s/га$/жский/;
    return $w if  $w =~ s/та$/тинский/;
    return $w if  $w =~ s/ра$/рский/;
    return $w if  $w =~ s/ры$/рский/;
    return $w if  $w =~ s/фа$/финский/;
    return $w if  $w =~ s/и$/инский/;
    return $w if  $w =~ s/([рснм])ь$/${1}ский/;
    $w .= 'ский';
    return $w;
}

our $hpr = {
    amp => '&',
    lt => '<',
    gt => '>',
    apos => "'",
    quot => '"',
    copy => '©',
    reg => '®',
    trade => '™',
    euro => '',
    pound => '',
    bdquo => '„',
    ldquo => '“',
    laquo => '«',
    raquo => '»',
    ge    => '≥',
    le    => '≤',
    asymp => '≈',
    ne    => '≠',
    equiv => '≡',
    sect  => '§',
    infin => '∞',
    nbsp  => ' ',
    ndash => '—',
    mdash => '–'
};
our $hprkeys = join('|', %$hpr);

sub replace_html_spec {
    my ( $self, %par ) = @_;
    my $text = $self->text;
    my @arr = split(/\n/, $text);
    s/\&($hprkeys)\;/$hpr->{$1}/g for @arr;
    $text = join("\n", @arr);
    return $text if $par{ret_text};
    return $self->proj->phrase( $text );
}

# для двусловных фраз: возвращает кратчайший путь между словами-синонимами
sub analyze_synonyms_path {
    my ($self) = @_;
    my @words = $self->normwords;
    my $norm = $self->proj->generated_dicts->{norm};
    my $path = $norm->analyze_synonyms_path($words[0], $words[1], $self->{lang});

    return [] if !$path;
    return [ map{join ":", sort($_->[0], $_->[1])} @$path ];
}

# возвращает список хэшей с информацией о регионах { name => , town => , is_world => ... }

sub get_regions :CACHE {
    my $self = shift;
    my $atoms = $self->{cdict_regions_phrases} || $self->search_atoms_snorm;
    my $layer = $self->language->layer_atoms_named;
    my @reg;
    while (my ($subphr, $dicts) = each %$atoms) {
        for my $dict (keys %$dicts) {
            my $reg_inf = $self->proj->geo->dict2reginf($dict) or next;
            my %inf = (
                name => $reg_inf->{name},
                town => $layer->get_orig_phrase($dict, $subphr),
                is_world => $reg_inf->{is_world_region},
            );
            push @reg, \%inf;
        }
    }
    return @reg;
}

#возвращает список регионов без поля town. Работает быстрее, так как нет неободимости вызывать get_orig_phrase
sub get_regions_fast :CACHE {
    my $self = shift;
    my $atoms = $self->{cdict_regions_phrases} || $self->search_atoms_snorm;
    my @reg;
    my %dicts = map {$_ => 1} map {keys %{$_}} values %$atoms;
    my $geo = $self->proj->geo;
    for my $dict (keys %dicts) {
        my $reg_inf = $geo->dict2reginf($dict) or next;
        my %inf = (
            name => $reg_inf->{name},
            is_world => $reg_inf->{is_world_region},
        );
        push @reg, \%inf;
    }
    return @reg;
}

sub contains_wordsbag {
    my ( $self, $wordsbag, %par ) = @_;
    %par = () unless %par;
    my $srcphl = $self->proj->phrase_list( { phrases_text => $wordsbag } );
    my $resphl = $srcphl->search_subphrases_in_phrase($self, %par);
    return $resphl ? 1 : 0;
}

sub eq_by_inclusion {
    my ( $self, $other ) = @_;
    return 1 if ( $self->contains_wordsbag($other->text, only_norm=>1) || $other->contains_wordsbag($self->text, only_norm=>1) );
    return 0;
}

sub check_subphrase_distance {
    my ( $self, $subphr_text, $distance ) = @_;
    my @subphr_words = $self->proj->phrase( $subphr_text )->words_lite;
    return 0 unless @subphr_words;
    return 0 unless @{array_intersection( \@subphr_words, [$self->words_lite] )} == @subphr_words;
    $distance = 0 unless defined($distance);
    my %hpos = $self->words_positions;
#    print Dumper ( \%hpos );
    my %hsubphr = map { my $word=$_; my @wpos = @{$hpos{$word}}; map { $_ => $word } @wpos } @subphr_words;
    my @subphr_words_indexes = sort { $a <=> $b } keys %hsubphr;
#    print Dumper ( \%hsubphr );
#    print Dumper ( \@subphr_words_indexes );
    O: for my $i ( 0..$#subphr_words_indexes ){
#        print "START INDEX: $i\n";
        my @chunk = ();
        my @chunk_indexes = ();
        for my $j ( $i..$#subphr_words_indexes ){
            push @chunk_indexes, $subphr_words_indexes[$j];
            push @chunk, $hsubphr{$subphr_words_indexes[$j]};
            last if ( scalar(uniq_array(@chunk)) == scalar(uniq_array(@subphr_words)) );
        }
#        print Dumper ( \@chunk );
        last if ( scalar(uniq_array(@chunk)) < scalar(uniq_array(@subphr_words)) );
        for ( 0..@chunk_indexes-2 ){
#            print Dumper ( [$chunk_indexes[$_+1], $chunk_indexes[$_]] );
            next O if ( $chunk_indexes[$_+1] - $chunk_indexes[$_] > ( $distance + 1 ) );
        }
#        print "Phrase $subphr_text is OK!\n";
        return 1;
    }
#    print "Phrase $subphr_text is BAD!\n";
    return 0;
}

sub delete_bad_subphrases {
    my ($self) = @_;
    my $text = $self->text;
    $text =~ s/\([^)]+\)/ /g;
    $text =~ s/\[[^\]]+\]/ /g;
    $text =~ s/\{[^\}]+\}/ /g;
    $text =~ s/^\d+\t/ /g;
    $text =~ s/
           (^|\s)
           (
             ((с|от)\s*)?
             (\d+\s*[\*xх]\s*)?
             \d+
             (\s*\-\s*\d+)?
             (\.\d+)?
             (-?(х))?
             (\s*до\s*\d+)?
             \s*
             (л|мл|кг|гр?|м|мес|коробок|кор|месяцев|шт|штук|месяца|лет|грамм?|год|лет|уп|упак|\%|г\/мл)
             \.?\s*
           )+
           (\s|$)/ /igx;
    $text =~ s/(^|\s)(уп|упак|с рожд|с рождения|б\/сах)\.?(\s|$)/ /g;
    return $text;
}

sub direct_format_text {
    my ($self) = @_;
    my ($pls, $mns) = $self->insert_hyphens->split_by_minus_words;
    $mns ||= '';
    my $t = $pls;
    $t =~ s/\s+\-\s+/ /g; # тире
    $t =~ s/(.+)\s+\~0\s*(.*)/"$1 $2"/; #Решаем проблему формата фраз с тильдой
    $t =~ s/[~`*;:'@%^&\(\)\[\],\#?«»−]/ /g; #Удаляем странные символы
    $t =~ s/[-!+]( |$)/ /g; #Удаляем специальные символы, если после них пробел
    $t =~ s/(?<=[^- \t"])[!+]/ /g; #Удаляем специальные символы, если перед ними не пробел
    $t =~ s/^\./ /g; #Удаляем специальные символы, если перед ними не пробел
    $t =~ s/\s*[\/\\]\s*/ /g; #Удаляем слэши и бэкслэши
    $t =~ s/ \./ /g; #Удаляем лишние точки
    $t =~ s/\s+/ /g; #удаляем повторы пробелов
    $t =~ s/^\s*"\s*|\s*"\s*$/"/g; #Удаляем пробелы около кавычек
    $t =~ s/^\s+|\s+$//g; #Удаляем начальные и конечные пробелы
    $t =~ s/(?:^|(?<=\s))лучш([а-я]{2})(?:(?=\s)|$)/хорош$1/ig; #Заменяем "лучший" на "хороший"
    return $t.$mns;
}

sub direct_format_phrase {
    my ($self) = @_;
    return $self->proj->phrase($self->direct_format_text);
}

sub add_chars_log {
    my ($self) = @_;
    return $self->text.' => '.join('-', map { "$_(".ord($_).")" } split //, $self->text);
    #return $self->text.' => '.md5int($self->text).' '.length($self->text);
}

our %QTAIL_WORDS_CONFIG  = (
    min_shows => 50,
    k => 8, #во сколько раз минусслово должно ухудшать кликабильность
    min_good_ctr => 0.005,
    great_ctr => 0.02,
    max_bad_ctr => 0.007
);

#Вернуть список подфраз из указанного словаря
sub get_dict_phl {
    my ($self, $dict) = @_;
    #$self->proj->dd($self->search_dict_phrases($dict),$dict, "$self");
    return $self->proj->phrase_list({ phrases_arr=> [$self->search_dict_phrases($dict)]});
}

#Возвращает фразу без найденный слов и список фраз из словаря ($ph, $phl)
sub split_by_dict {
    my ($self, $dict) = @_;
    my $phl = $self->get_dict_phl($dict);
    return ($self, $phl) unless $phl->count;
    my $flt = $phl->normwordshash;
    my $newtxt = join(" ", map {$_->[0]} grep { ! $flt->{$_->[1]} } $self->normwords_pairs );
    return ( $self->proj->phrase($newtxt), $phl );
}

#Проверяет, является ли фраза подфразой текущей
sub has_subphrase {
    my ($self, $ph, %par) = @_;
    my $h = { map { $_=>1 } $par{snorm} ? $self->snormwords : $self->normwords };
    return 0 if grep {! $h->{$_}} $par{snorm} ? $ph->snormwords : $ph->normwords;
    return 1;
}

sub insert_hyphens {
    my ($self) = @_;
    my @words = $self->pluswords;
    my @norms = map{word2norm($_, $self->{'lang'})} @words;
    my @output;
    my $last_insert = -1;
    my $biwords = $self->language->{complex_words};

    for my $i (0..$#words) {
        next if $last_insert >= $i;
        if($i < $#words && ($biwords->{$norms[$i]} || {})->{$norms[$i + 1]}) {
            my $complex = $words[$i] . "-" . $words[$i + 1];
            push @output, $complex;
            $last_insert = $i + 1;
        } else {
            push @output, $words[$i];
        }
    }

    if($last_insert > -1) {
        return $self->language->phrase(join(" ", @output, map{"-$_"} $self->minuswords));
    }

    return $self;
}

#Удаляем слова другой фразы
sub delete_subphrase_words {
    my ($self, $ph) = @_;
    my $h = { map { $_=>1 } $ph->normwords };
    my $newtxt = join(" ", map {$_->[0]} grep { ! $h->{$_->[1]} } $self->normwords_pairs );
    return $self->proj->phrase($newtxt);
}

sub delete_pluses {
    my $self = shift;
    my $t = $self->text;
    return $self unless $t =~ /\+/;
    $t =~ s/\+/ /g;
    return $self->proj->phrase($t);
}

# удаление оператора ("+" или "!") с учётом стоп-слов
# "+" удаляется и в минус-словах!
# сохраняем закавыченность
# параметры:
#   $op  -  оператор "+" или "!"
sub delete_op_safe {
    my $self = shift;
    my $op = shift;

    my $lang = $self->lang;
    my $is_stop = sub { bsstop4word($_[0]) or stop4norm(word2norm($_[0], $lang), $lang) };

    my $opreg = quotemeta($op);
    my (@ws, $changed);
    for my $tok ($self->words) {
        if ($op eq '+' and $tok =~ /^\-\+/) {
            my $w = substr($tok, 2);
            if ($is_stop->($w)) {
                push @ws, $tok;
            } else {
                push @ws, "-$w";
                $changed = 1;
            }
        } elsif ($tok =~ /^$opreg/) {
            my $w = substr($tok, 1);
            if ($is_stop->($w)) {
                push @ws, $tok;
            } else {
                push @ws, $w;
                $changed = 1;
            }
        } else {
            push @ws, $tok;
        }
    }
    return $self if !$changed;

    my $text = join(' ', @ws);
    $text = '"'.$text.'"' if $self->is_quoted;
    return $self->language->phrase($text);
}

sub add_plus_words {
    my ($self, @ws) = @_;
    return $self unless @ws;
    my $wt = join(" ", @ws);
    my $t = $self->text;
    if( $t =~ /^"/ ){ #Отдельно обрабатываем фразы в кавычках
        $t =~ s/^"/"$wt /;
        return $self->proj->phrase($t);
    }
    return $self->proj->phrase("$wt $t");
}

# восстанавливаем +слова
# на входе $phl
# возвращает новый $phl
sub restore_spec_words {
    my $self = shift;
    my $phl = shift;

    my @sw = map { s/^\+//; $_ } grep { /^\+/ } $self->normwords; ##no critic
    my %sw = map { $_ => 1 } @sw;
    return $phl if !@sw;

    my @phrs;
    for my $phr (@$phl) {
        # либо потеряли '+' у слова, либо потеряли слово совсем
        my $fixed = 0;
        my @nw;
        for my $nw ($phr->normwords) {
            if ($sw{$nw}) {  # спец.слово без плюса
                push @nw, '+'.$nw;
                $fixed = 1;
            } else {
                push @nw, $nw;
            }
        }
        my %nw = map { $_ => 1 } @nw;
        my @lost = grep { !$nw{$_} } map { '+'.$_ } @sw;
        if (!$fixed and !@lost) {
            push @phrs, $phr;
        } else {
            my $text = join(' ', @nw, @lost);
            push @phrs, $self->language->phrase($text);
        }
    }

    return $self->language->phrase_list(\@phrs);
}

our %en_to_ru = (
    a => 'ф', b => 'и', c => 'с', d => 'в', e => 'у', f => 'а',
    g => 'п', h => 'р', i => 'ш', j => 'о', k => 'л', l => 'д',
    m => 'ь', n => 'т', o => 'щ', p => 'з', q => 'й', r => 'к',
    s => 'ы', t => 'е', u => 'г', v => 'м', w => 'ц', x => 'ч',
    y => 'н', z => 'я', ';' => 'ж', "'" => 'э', '[' => 'х', ']' => 'ъ',
    ',' => 'б', '.' => 'ю', '`' => 'ё',
);
our %ru_to_en = reverse(%en_to_ru);
sub poor_layout {
    my $self = shift;
    return join "", (map{$en_to_ru{$_} || $_} (split //, $self->text));
}

sub has_common_lemmas {
    my ($self) = @_;

    my $common_lemmas_data = $self->common_lemmas_data->{$self->lang};
    return   if not $common_lemmas_data;

    my @can_collapse = grep { $common_lemmas_data->{$_} } (
        (map { $self->language->phrase($_)->norm_phr }  map { split /-/ }  grep { /-/ }  uniq $self->normwords),
        (grep {not /-/ }  uniq $self->normwords),
    );
    if ( (scalar @can_collapse) >= 2 ) {
        my %can_collapse = map {$_ => 1} @can_collapse;
        for my $w (@can_collapse) {
            my @wc = @{ $common_lemmas_data->{$w} };
            return 1   if @can_collapse{ @wc };
        }
    };
    return 0;
}

sub is_collapsing_2words {
    my ($self) = @_;
    my $res = 0;

    my @words =
        grep {not bsstop4word($_)}
        map {split /-/}
        grep {not stop4norm($_, $self->lang)}
        uniq $self->normwords ;

    #$self->proj->dd([words => \@words]);
    return 1 if (@words == 2  and  $self->language->phrase(join(" ", @words))->has_common_lemmas);
    return 0;
}

sub common_lemmas_data : GLOBALCACHE {
    my ($self) = @_;
    my $common_lemmas_data = {};
    for my $lang ( @{$self->proj->{load_languages}}, 'ru') {
        my $file = $Utils::Common::options->{Collapsed_words_params}{"file_$lang"};
        #$self->proj->dd([$lang, $file]);
        next if not $file;
        open(F, $file)  or do {
            $self->proj->log("ERROR: Could not open file ($file)");
            next
        };
        my @groups;
        while (my $line = <F>) {
            chomp $line;
            next if $line =~ /^\s*#/;
            my @words = split /\s*,\s*/, $line || next;
            push @groups, [uniq sort (map { $self->language->phrase($_)->norm_phr } @words)];
        }
        close(F);
        my $data = {};
        for my $gr (@groups) {
            next if @$gr < 2;
            for my $w (@$gr) {
                $data->{$w}{$_} = 1   for grep {$_ ne $w} @{$gr};   # Для каждого слова сохраняем список тех слов, с которыми оно может "схлопнуться"
            }
        }
        $data->{$_} = [sort keys %{$data->{$_}}]  for keys %$data;
        $common_lemmas_data->{$lang} = $data;
    }
    return $common_lemmas_data;
}

sub smart_split_text {
    my ($self, $lnth) = @_;
    my $text = $self->text;
    my $l = length($text);
    return ($text, '') if $l <= $lnth;
    my $lmt = $lnth - 1;
    return ($1, $2) if $text =~ /^(.{,$lmt})[ \/\.,](.*)$/; #Если есть пробел
    return (substr($text, 0, $lnth-1), substr($text, $lnth-1));
}

sub substitute_words {
    my ($self, $h) = @_;
    my %hnorm = map { $self->proj->phrase($_)->norm_phr => $h->{$_} } keys %$h;
    return map { my $wn = $self->proj->phrase($_)->norm_phr; exists( $hnorm{$wn} ) ? $hnorm{$wn} || () : $_ } split( /\s+/, $self->text);
}

sub restore_capitalization {
    my ( $self, $text ) = @_;
    return $text unless $self->text;
    my @res = ();
    my $source_text = $self->text;
    for my $word ( split /\s+/, $text ){
        my $word_re = '(?:\b|[-_\/\\.,])('.quotemeta($word).')'; # граница слова только для начала слова, чтобы отловить случаи, когда оно в каком-то падеже: найти "Ашан" во фразе "товары из Ашана"
        $word_re .= '\b' if (length $word < 3); # фиксируем конец слова для случая падежей
        if ( $source_text =~ /$word_re/i ){
            push @res, $1;
        } else {
            push @res, $word;
        }
    }
    return join(' ', @res);
}

sub dict_trade_words :GLOBALCACHE {
    my ( $self ) = @_;
    my %h = map { $_ => 1 } $self->proj->phrase( "купить,магазин,цена,интернет-магазин,распродажа,скидки,заказ" )->snormwords;
    return \%h;
}

sub delete_trade_words {
    my ( $self ) = @_;
    my $flt = $self->dict_trade_words;
    my @words = $self->pluswords;
    @words = grep { ! $flt->{[text2snormwords_fast($_, $self->{'lang'})]->[0] || ''} } @words;
    return join(' ', @words, map {-$_} $self->minuswords );
}

sub delete_town_words {
    my ( $self ) = @_;
    my $towntexts = join(' ', map { $_->{town} } $self->get_regions );
    my $towntexts2 = $towntexts;
    $towntexts2 =~ s/-/ /g;
    my $flt = { map { $_ => 1 } $self->proj->phrase("$towntexts $towntexts2")->normwords };
    my @words = $self->pluswords;
    @words = grep { ! $flt->{[text2normwords($_, $self->{'lang'})]->[0] || ''} } @words;
    return join(' ', @words, map {-$_} $self->minuswords );
}


# текст для отправки базы в БК, сохранения в базу и т.п.
sub external_text :CACHE {
    my $self = shift;
    return encode_quotes($self->text);
}

sub retrieve_atoms {
    my $self = shift;

    my @result = ();
    for my $atoms_text ($self->text =~ /\[([^\]]+)\]/g) {
        my @chunk = map {s/^\s*|\s*$//g; $_} split /\//, "$atoms_text "; ##no critic
        push @result, \@chunk if @chunk;
    }

    return \@result;
}

sub erase_atoms {
    my $self = shift;

    my $text = $self->text;
    $text =~ s/\[[^\]]*\]//g;

    return $text;
}

sub erase_named_atoms {
    my $self = shift;

    my @atoms_info = @{$self->retrieve_atoms};
    my $text = $self->text;
    $text =~ s/\[[^\]]*\]//g;
    for my $atoms_bucket (@atoms_info) {
        my @anonymous_atoms = grep {$_ !~ /^\./} @$atoms_bucket;
        $text .= ' [' . join('/', @anonymous_atoms) . ']' if @anonymous_atoms;
    }

    return $text;
}

sub head {
    my ($self, $numwords) = @_;
    return '' unless $self->text;
    return $self->text unless $numwords;
    my @arr = split /\s+/, $self->text;
    return $self->text if @arr <= $numwords;
    return join (' ', @arr[0..$numwords-1] );
}

sub check_brackets_syntax {
    my $self = shift;

    my $matches_count = 0;
    my $text = $self->text;
    # curly brackets
    $text =~ s/\{[^\[\]<>{}]+\}//g;
    $matches_count += $text =~ /[{}]/;
    # angle brackets
    $text =~ s/<[^\[\]<>{}]+>//g;
    $matches_count += $text =~ /[<>]/;
    # square brackets
    $text =~ s/\[[^\[\]]+\]//g;
    $matches_count += $text =~ /[\[\]]/;

    return $matches_count ? 0 : 1;
}

# IN:"найти валенки в москве и санкт-петербурге"
# OUT: (213,2);
sub get_geobase_region_ids {
    my ( $self ) = shift;
    my @result = ();
    my $geobase_norm_dict = $self->proj->geo->geobase_norm_dict();
    for my $reg ( $self->get_regions ) {
        if ( exists($reg->{town}) ) {
            my $normtown = $self->proj->phrase( $reg->{town} )->norm_phr();
            if ( exists($geobase_norm_dict->{$normtown}) ) {
                push @result, @{$geobase_norm_dict->{$normtown}};
            }
        }
    }
    return @result;
}

sub sentences {
    my $self = shift;

    my @split_patterns = ();
    # split по не-точкам: любое количество идущих подряд разделителей
    push @split_patterns, '[\!\?\|]+\s*';

    # split по точкам
    # если больше одной точки подряд
    push @split_patterns, '\.\.+\s+';

    # если после точки есть пробел, но за ним не строчная буква (иначе считаем сокращением)
    # исключения:
    #   1. одна заглавная буква перед точкой (инициалы)
    #   2. 'им.', 'г.', 'м.', ...
    my @exceptions = (
        '^\p{L}', # любая единичная буква в начале
        '\W\p{Uppercase}', # одна заглавная буква
        '\Wг', '\Wм', '\Wо', 'им',  # сокращения
    );
    # ни одно исключение не должно сработать
    my $all_exceptions = join("", map {'(?<!'.$_.')'} @exceptions);
    push @split_patterns, $all_exceptions.'\.\s+(?!\p{Lowercase})';

    my $split_pattern = join("|", @split_patterns);

    my @sentences = split /$split_pattern/, $self->text;
    return $self->proj->phrase_list(\@sentences);
}

# max_sentences => N  -  кол-во предложений (берём первые N)
sub get_keywords_from_title {
    my $self = shift;
    my %par  = @_;

    my $phl = $self->sentences;
    if (defined $par{max_sentences}) {
        $phl = $phl->lhead($par{max_sentences});
    }

    my @sep = ('|', ': ', '!', '?', ' - ', ' – ', ' / ', '   ');
    my $sep_regexp = join('|', map { '(?:'.quotemeta($_).')' } @sep);
    $phl = $phl->tmap(sub { split qr/$sep_regexp/, $_ });

    $phl = $phl->lgrep(sub { $_->number_of_words >= 2 and $_->number_of_words <= 5 });
    $phl = $phl->lgrep(sub { $_->text !~ /,.*,/ });  # перечисления, убираем пока что

    return $phl;
}


sub clarifications {
    my $self = shift;

    my $h = $self->get_search_tail;
    my @all_tails = sort {$h->{$b} <=> $h->{$a}} keys $h;
    my @clarifs;
    my %all_contexts;
    my $add_tails = 1;
    for my $tail (@all_tails) {
        my $tail_phr = $self->proj->phrase($tail);
        my $double_tails = $tail_phr->get_search_tail_phl(top_size => 1);
        my %last_words = map {$_ => 1} map {my @w = $_->words; pop @w} @$double_tails;
        for my $w (keys %last_words) {
            $add_tails = 0 if defined $all_contexts{$w};
            $all_contexts{$w} = 1;
        }
        last unless $add_tails;
        push @clarifs, $tail;
    }

    return $self->proj->phrase_list(\@clarifs);
}

sub word_analyze {
    my $self = shift;

    my %result = ();
    my $word = $self->text;
    my $lang = $self->language->name;
    $result{'word'} = $word;
    my $norm = word2norm($word, $lang);
    my $snorm = norm2snorm($norm, $lang);
    $result{'norm'} = $norm;
    $result{'snorm'} = $snorm;
    my @flags;
    push @flags, 'STOP' if stop4norm($norm, $lang);
    push @flags, 'WIDE' if $self->language->is_snorm_wide($snorm);
    push @flags, 'BAD' if bad4snorm($snorm);    # TODO  $lang
    $result{'flags'} = \@flags;
    $result{'goodsyns'} = [ grep { $_ ne $norm } snorm2goodsyns($snorm, $lang) ];
    $result{'misprints'} = [ norm2misprints($norm, $lang) ];

    return \%result;
}

sub cut_subphrases_by_dict {
    my ($self, $hdict, $is_by_norm) = @_;
    $is_by_norm = 0 unless defined $is_by_norm;
    my $text = $self->text;
    my @subs_to_cut = $self->match_itemdict( $hdict, by_norm=>$is_by_norm, item_maxsize=>5 );
    my $phl_subs_to_cut = $self->proj->phrase_list( \@subs_to_cut )->filter_subphrases;
    # вырезаем найденное
    for my $phr ( @$phl_subs_to_cut ){
        my $sub = $phr->text;
        $sub = quotemeta( $sub );
        $text =~ s/(?:^|\s+)$sub(?:\s+|$)/ /gi;
    }
    # режем мусор на границах
    @subs_to_cut = map { $self->proj->phrase($_)->clear_edges } @subs_to_cut;
    my @res = ( $self->proj->phrase( $text )->pack_spaces, \@subs_to_cut );
#    print Dumper( [$self->text, \@res] );
    return @res;
}

# очищает мусор на границах слов фразы
sub clear_edges {
    my ($self) = @_;
    my $text = $self->pack_spaces;
    return Utils::Words::clear_edges_fast($text);
}


sub move_minuswords_to_end {
    my ( $self ) = @_;
    my @words = ();
    my @minuswords = ();
    for my $text ( split /\s+/, $self->text ){
        if ( $text =~ /^-[^-]/ ){
            push @minuswords, $text;
        } else {
            push @words, $text;
        }
    }
    push @words, @minuswords if @minuswords;
    return join(' ', @words);
}

sub replace_strange_symbols_by_space {
    my ( $self ) = @_;
    my $text = $self->text;
    $text =~ s/[!"№;%@#\$\%\^&\*\(\),\.\/\?\:"\'\|~`]/ /ig;
    return $text;
}

our $flight_minus_words = "-концерт -театр -опера -электрички -автобус -футбол -стадион -футбольный -спортивный -спорт -скачать -ресторан -кафе -стол -столик -меню -книга";
sub add_minus_words_flights {
    my ( $self ) = @_;
    return $self->text if $self->text =~ /\~0$/;
    return "$self $flight_minus_words";
}

our $travel_minus_words = "-концерт -театр -опера -футбол -стадион -футбольный -спортивный -спорт -скачать";
sub add_minus_words_travel {
    my ( $self ) = @_;
    return $self->text if $self->text =~ /\~0$/;
    return "$self $travel_minus_words";
}

sub subtract_phl_by_norm {
    my ($self, $phl) = @_;
    my $phr = $self;
    $phr  = $phr ^ $_ for @$phl;
    return $phr;
}

sub clean_mixed_ruwords {
    my ($self) = @_;
    my @res = ();
    for my $word ( $self->words ){
        $word = ( _fix_bad_words($word) || $word ) if _is_word_mixed($word);
        push @res, $word;
    }
    return join (' ', @res);
}

sub wflt :CACHE {
    my ($self) = @_;
    my %h = map {$_=>1} $self->normwords;
    return \%h;
}

our $JSON_KEY = '__j_BM_Phrase__';   # TODO: shorter key for compaction?
our $JSON_CLASS = 'BM::PhraseModif';

use Scalar::Util qw/refaddr/;

sub fixuri {
    my ($self) = @_;
    my ($dmn, $uri) = split /\?/, $self->text;
    return $self->text unless $uri;
    $uri = join '&', map { defined($_->[1]) ? $_->[0].'='.uri_escape_utf8($_->[1]) : $_->[0] } map {[ split /=/, $_ ]} split /\&/, $uri;
    return $dmn."?".$uri;
}

sub FREEZE {
    my ($self) = @_;
    my %inf;
    $inf{text} = $self->text;
    $inf{lang} = $self->lang;
    if ($self->inf) {
        $inf{inf} = $self->inf;
    }
    if ($self->hinf) {
        $inf{hinf} = $self->hinf;
    }
    if (keys %{$self->minf}) {
        $inf{matcher_inf} = $self->minf;
    }
    my @cache = map { "_cached_$_" } qw(cdict_key_norm cdict_key_snorm get_minicategs);
    my @fld = qw(search_count);
    for my $key (@cache, @fld) {
        my $val = $self->{$key};
        $inf{$key} = $val if defined $val;
    }
    return \%inf;
}

sub TO_JSON {
    my $self = shift;

    my $j_addr = refaddr($self);
    my %inf = ( __j_addr => $j_addr );

    # Try cache
    my $j_key = $BM::Phrase::JSON_KEY . '@' . $j_addr;
    if ($self->proj->smart_json_encode_cache($j_key)) {
        # already encoded once. Will only return stub
        return { $BM::Phrase::JSON_KEY => \%inf };
    }

    # Not in cache. Need to build full representation and return it
    $inf{text} = $self->text;
    $inf{lang} = $self->lang;
    if ($self->inf) {
        $inf{inf} = $self->inf;
    }
    if (keys %{$self->minf}) {
        $inf{matcher_inf} = $self->minf;
    }

    # remembere we have encoded this phrase once
    $self->proj->smart_json_encode_cache($j_key, 1);

    return { $BM::Phrase::JSON_KEY => \%inf };
}

sub FROM_JSON {
    my ($class, $proj, $proto_obj, $hash, $smart_cache) = @_;

    my $j_addr = delete $hash->{__j_addr};
    my $has_data = scalar %{$hash};

    my $obj;

    if ($j_addr) {
        my $j_key = $BM::Phrase::JSON_KEY . '@' . $j_addr;
        my $dummy_j_key = "dummy/" . $j_key;

        if ($has_data) {    # Hash with data - can create real object and/or fullfill existing lite object
            if (exists $smart_cache->{$dummy_j_key}) { # there is lite object cached
                $obj = delete $smart_cache->{$dummy_j_key};
                for my $key (keys %$hash) {
                    $obj->{$key} = $hash->{$key};
                    $obj->{proj} = $proj;
                }
                $obj = $proto_obj->new($obj);
            } elsif (exists $smart_cache->{$j_key}) {
                $obj = $smart_cache->{$j_key};
                #warn "Phrase redefined in json. Should not happen.";
            } else {
                $hash->{proj} = $proj;
                $obj = $proto_obj->new($hash);
            }
            $smart_cache->{$j_key} = $obj;
        } else {            # Hash has only j_addr - use cached object or use lite object until real object'll be created
            if (exists $smart_cache->{$dummy_j_key}) {
                $obj = $smart_cache->{$dummy_j_key};
                # somebody already waiting for data
            } elsif (exists $smart_cache->{$j_key}) {
                $obj = $smart_cache->{$j_key};
                # somebody already created real object
            } else {
                $obj = $proto_obj->new_lite({});
                $smart_cache->{$dummy_j_key} = $obj;
            }
        }
    } else {    # No smartness
        $hash->{proj} = $proj;
        $obj = $proto_obj->new($hash);
    }
    return $obj;
}

use overload
    '""' => sub {
            my ($self) = @_;
            return $self->text;
        },
    'eq' => sub {
            my ($self, $other) = @_;
            return $self->text eq $other->text;
        },
    '==' => sub {
            my ($self, $other) = @_;
            return $self eq $other;
        },
    '~~' => sub { # вычитает из первой фразы ее пересечение со второй, возвращает норм
            my ($self, $other) = @_;
            my @slf = split /\s+/, $self->norm_phr;
            my @othr = split /\s+/, $other->norm_phr;
            my @res = grep { ! in_array($_, \@othr) } @slf;
            return $self->proj->phrase('') unless @res;
            return $self->proj->phrase("@res");
        },
    '^' => sub { # вычитает из первой фразы ее пересечение со второй по норму, возвращает ненормализованный вариант
            my ($self, $other) = @_;
            my @slf = $self->words;
            my @othr = split /\s+/, $other->norm_phr_safe;
            my @res = grep { ! in_array(($self->proj->phrase($_)->norm_phr_safe || $_), \@othr) } @slf;
            return $self->proj->phrase('') unless @res;
            my $res_cap = $self->restore_capitalization("@res");
            return $self->proj->phrase($res_cap);
        },
    '%' => sub { # вычитает из первой фразы ее пересечение со второй, без нормализации
            my ($self, $other) = @_;
            my @slf = map { $_=lc $_; $self->proj->phrase($_)->clear_edges } split/\s+/, $self->text; ##no critic
            my @othr = map { $_=lc $_; $self->proj->phrase($_)->clear_edges } split/\s+/, $other->text; ##no critic
            my @res = grep { ! in_array($_, \@othr) } @slf;
            return $self->proj->phrase('') unless @res;
            return $self->proj->phrase("@res");
        },
    '+' => sub {
            my ($self, $other) = @_;
            return $self->proj->phrase( $self->text." ".$other->text );
        };
1;
