package BM::PhraseModif;

use utf8;
use open ':utf8';

use std;
use base qw(BM::PhraseParser);

use BM::Phrase;
use BM::LemmerTest;
use Utils::Array;
use Data::Dumper;
use List::Util qw(first);

########################################################
# Интерфейс
########################################################
#    harmonize                              соглаует переданную фразу по данной
#    get_gender_number_case                 возвращает род-число-падеж фразы - суммирует грамматические характеристики по каждому слову фразы и выбирает наиболее частотные, для однословников также                                                возвращает также часть речи
#    get_gender_number_case_lite            возвращает в упрощенном формате ie
#    set_gender_number_case                 ставит фразу (слово) в нужные род-число-падеж
#    _mark_phrase                           делит фразу на изменяемую и неизменяемую части (например, клюшка для гольфа, 'клюшка' - изменяемая, 'для гольфа' - нет)
#    _get_word_lemmer_inf                   парсит выдачу леммера в более удобно читаемую структуру

#    change_text_number(n)                  Заменяет одно число на другое с сохранением формата фразы

sub lmr :GLOBALCACHE {
    my ($self) = @_;
    return BM::LemmerTest->new({proj => $self->proj, args => [ '-p' ]});
}

sub _get_word_lemmer_inf {
    my ($self, $word) = @_;
    my %dict = ();

    my $str = $self->lmr->analyze($word);
    my @lemmas = split("\n",$str);
    for (my $j = 2; $j < @lemmas; $j++) {
        (my $lemm )= $lemmas[$j] =~ /^(\S+)/;
        $lemmas[$j] =~ /(Good|Bastard|Sob\\|Prefixoid|Sob|Good\\|BadRequest)(\|LLL)?\s?([A-Z]+)\,?(\S+)?\s(\S+)?/;
        my $PartSp = $3 || 'empty';
        my $PermFtrs = $4 || 'empty';
        (my $prdgm) = $lemmas[$j] =~ /(\[.+)/;
        my @WordFs;
        @WordFs = split ("}", $prdgm) if defined($prdgm);
        for my $k (@WordFs) {
            $k =~ s/^\,//;
            $k =~ s/\[|\]|\}//g;
            $k =~ s/\{/ /;
            (my $sf) = $k =~ /(^\S+)/;
            (my $gr) = $k =~ /\s(.*)/;
            my @gramm = split ("\\|", $gr);
            push (@{$dict{$sf}{$lemm.",".$PartSp}},"invar") if (@gramm == 0) && !(grep {/^invar$/} @{$dict{$sf}{$lemm.",".$PartSp}});
            for my $m (@gramm) {
                next if (($m =~ /ger/) || ($m =~ /obsol/));
                $m = $PermFtrs.",".$m if ($PermFtrs);
                $m =~ s/(^|\W)(nom|gen|dat|acc|ins|abl)(\W|$)/${1}case=$2$3/g;
                $m =~ s/(^|\W)(m|f|n|mf)(\W|$)/${1}gender=$2$3/g;
                $m =~ s/(^|\W)(pl|sg)(\W|$)/${1}number=$2$3/g;
                $m =~ s/(^|\W)(supr|comp)(\W|$)/${1}degree=$2$3/g;
                $m =~ s/(^|\W)(praet|praes|inpraes)(\W|$)/${1}tense=$2$3/g;
                $m =~ s/(^|\W)(pf|ipf)(\W|$)/${1}aspect=$2$3/g;
                $m =~ s/(^|\W)(imper|indic)(\W|$)/${1}mood=$2$3/g;
                $m =~ s/(^|\W)(act|pass)(\W|$)/${1}voice=$2$3/g;
                $m =~ s/(^|\W)(brev|plen)(\W|$)/${1}form=$2$3/g;
                $m =~ s/(^|\W)(persn|famn|geo)(\W|$)/${1}title=$2$3/g;
                $m =~ s/(^|\W)(anim|inan)(\W|$)/${1}anim=$2$3/g;
                $m =~ s/(^|\W)(1p|2p|3p)(\W|$)/${1}person=$2$3/g;
                $m =~ s/(^|\W)(tran|intr)(\W|$)/${1}tran=$2$3/g;
                if ($m =~ /partcp/) {
                    $m =~ s/partcp//;
                    push (@{$dict{$sf}{$lemm.","."Partcp"}},$m) unless (grep {/^$m$/} @{$dict{$sf}{$lemm.","."Partcp"}});
                }
                else {
                    next if (grep {/^$m$/} @{$dict{$sf}{$lemm.",".$PartSp}});
                    push (@{$dict{$sf}{$lemm.",".$PartSp}},$m);
                }
            }
        }
    }
    my (%h1f, %h2f) = ();
    for my $i (sort keys %dict) {
        for my $j (sort keys %{$dict{$i}}) {
            my $tt = $i."\t".$j."\t".join("//",@{$dict{$i}{$j}})."\n";
            $h1f{$i} ||= [];
            push(@{$h1f{$i}}, $tt);
            $h2f{$j} ||= [];
            push(@{$h2f{$j}}, $tt);
        }
    }
    return \%h1f, \%h2f;
}
# грамм. словарь по словоформе
sub _get_grammar1f {
    my ($self, $w) = @_;
    return $BM::Phrase::grammar1f{$w} if $BM::Phrase::grammar1f{$w};
    my ($h1f, $h2f) = $self->_get_word_lemmer_inf($w);
    my $result = $h1f->{$w} || [];
    $BM::Phrase::grammar1f{$w} = $result;
    return $result;
}
# грамм. словарь по лемме
sub _get_grammar2f {
    my ($self, $w) = @_;
    return $BM::Phrase::grammar2f{$w} if $BM::Phrase::grammar2f{$w};
    my $word = $w;
    $word =~ s/\,.*$//;
    my ($h1f, $h2f) = $self->_get_word_lemmer_inf($word);
    my $result = $h2f->{$w} || [];
    $BM::Phrase::grammar2f{$w} = $result;  # корректно, т.к. word это функция от w
    return $result;
}

# возвращает хеш с родом-числом-падежом фразы
sub get_gender_number_case {
    my ($self, %par) = @_;
    my $add_posp = $par{add_posp};
    my %res = ();
    my %summary = ();
    my %after_subject = ();
    my $case_found;
    #print STDERR "[".$self->text."]\n";
    return {} unless $self->text;
    my $text = lc $self->get_prefiltered_line->text;

    if ( 0 && $self->norm_phr =~ /^[a-z ]+$/i ){
        $res{gender} = 'm';
        $res{number} = 'sg';
        $res{case} = 'nom';
        return \%res;
    }

    # итерация 0: группируем и суммируем значения рода, числа и падежа по каждому слову для всех форм парадигмы,
    #             затем по каждому показателю из всех полученных вариантов отбираем наиболее частотный
    # итерация 1: пытаемся более точно определить падеж: суммируем значения падежа только в тех формах,
    #             в которых совпадают значения рода и числа из итерации 0; если падеж не определился таким образом,
    #             оставляем результат нулевой итерации

    for my $i (0..1) {

        my %subjects = ();

     L: for my $word (split /\s+/, $text) {
            $word =~ s/-.*$//;
#            next if $word !~ /^[а-яё]+$/i;
            my $posp_lines = $self->_get_grammar1f($word); # получаем парадигму, разбитую по частям речи

#            print STDERR "paradigm of [$word]:\n";
#            print STDERR Dumper ($posp_lines);

            my $posp_lines_filtered = $par{no_filter_paradigm} ? $posp_lines : filter_posp_lines( $posp_lines, %par);
#            print STDERR "paradigm filtered of [$word]:\n";
#            print STDERR Dumper ($posp_lines_filtered);

            # если словоформа совпадает с леммой, то остальные строки не рассматриваем
            my $temp = [ grep { /^(.+)\s+\1\,/ } @$posp_lines_filtered ];
            $posp_lines_filtered = $temp if @$temp;

            for my $posp_line ( @$posp_lines_filtered ) { # для каждой части речи
                #print STDERR "$posp_line\n";
                my @posp_line = split /\s+/, $posp_line; # ее строка состоит из 3-х частей
                my @posp = split /\,/, $posp_line[1]; # саму часть речи извлекаем из 2-й части

                # упрощаем картину
                $posp[1] = 'A' if $posp[1] eq 'APRO'; # местоименное прилагательное
                $posp[1] = 'A' if $posp[1] eq 'Partcp'; # прич.
                $posp[1] = 'A' if $posp[1] eq 'ANUM'; # нумерующее прилагательное
                $posp[1] = 'S' if $posp[1] eq 'SPRO'; # местоименное существительное

                $summary{posp}{$posp[1]}++ if ( $self->text !~ /\s/ and $add_posp ); # для однословников определяем еще часть речи
                next if $posp[1] eq 'V'; # глаголы не учитываем

                if ( $posp[1] eq 'PR' ) { # просматриваем до предлога или до второго существительного
                    last L;
                } elsif ( $posp[1] eq 'S') {
                    $subjects{$word}++ if ( $posp[1] eq 'S' );
                    #print STDERR Dumper (\%subjects);
                    last L if keys %subjects > 1;
                }

                my @possible_forms;
                @possible_forms = split /\/\//, $posp_line[2] if $posp_line[2];

                for my $form ( @possible_forms ) {
                    #print STDERR "form = $form\n";
                    next if ( $i and (!$res{gender} or !$res{number} or $form !~ /\=$res{gender}/ or $form !~ /\=$res{number}/) ); # на первой итерации ищем формы по роду и числу
                    for my $grammar_unit ( split /\,/, $form ) {
                        my ( $name, $value ) = split /\=/, $grammar_unit;
                        $summary{$name}{$value}++ if ( (!%subjects or $posp[1] eq 'S') and $name and $value ); # слова c начала фразы по первое существительное
                        $after_subject{$name}{$value}++ if ( %subjects and ($posp[1] ne 'S') and $name and $value ); # слова после первого существительного и до второго
                    }
                }
            }
        }
        #print STDERR Dumper(\%summary);
        #print STDERR Dumper(\%after_subject);

        # объединяем хеши
        %summary = %{_sum_hashes(\%summary, \%after_subject)} if ( %after_subject and keys %subjects < 2 ); # если 2 существительных, то слова после первого не учитываем *синяя бутылка жидкого клея*
        #print STDERR Dumper(\%summary);

        # веса на случай совпадения счетчиков
        my %weights = (
            sg => 2,
            pl => 1,
            m => 40,
            f => 30,
            mf => 20,
            n => 10,
            nom => 900,
            gen => 800,
            ins => 700,
            abl => 600,
            acc => 500,
            dat => 400,
            PR => 9000,
            ADV => 8001,
            A => 8000,
            S => 7000,
            ART => 6999,
            CONJ => 6998,
            V=>6000
        );

        # сравниваем частоты и берем наибольшую, если совпадают - взвешиваем
        for my $key ( qw {gender number case posp} ) {
            my $unit = $summary{$key};
            #print STDERR "--$key\n";
            #print STDERR Dumper ($unit);
            my @sorted_units = sort {
                    return 0 if ( not defined $$unit{$a} or not defined $$unit{$b} );
                    if ( $$unit{$b} eq $$unit{$a} ) {
                        return 0 if ( not defined $weights{$a} or not defined $weights{$b} );
                        $weights{$b} <=> $weights{$a}
                    } else {
                        $$unit{$b} <=> $$unit{$a}
                    }
                } keys %$unit;

            for ( @sorted_units ) {
                $res{$key} = $_ unless $i;
                $case_found = $_ if ( $i and ($key eq 'case') );
                last;
                print STDERR "\t".$_."->".$$unit{$_}."\n";
            }
        }
    }
    # если на итерации 1 был найден падеж, то берем его
    $res{case} = $case_found if $case_found;

#    print STDERR Dumper (\%res);
    return \%res;
}

# подавление лишнего
sub filter_posp_lines {
    my ( $posp_lines, %par ) = @_;
    my $res = [()];

    my $gender ||= $par{gender};
    my $number ||= $par{number};
    my $case ||= $par{case};

#    print STDERR Dumper ( $posp_lines );

    # подавляем формы-производные от имен, если кроме них есть еще другие формы (мАшина-машИна)
    $res = [ map { s/title=famn.*?(\/\/|$)//gr } @$posp_lines ]; #filter
    $res = $posp_lines unless grep { /\=/ } @$res; #restore if empty

    # если были прилагательные, то это точно прилагательное (другие формы отбрасываем)
    # убираем эту логику для кратких форм прилагательных
    my $has_adj_line = first {  /(?:ый|ая|ое|ее|ий|ой|яя|),A\s/i } grep {!/form=brev/} @$res;
    $res = [ grep {  /,A\s/ } @$res ] if $has_adj_line;
    $res = $posp_lines unless $res;

    # подавляем превосходную степень и краткую форму прилагательного
    $res = [ grep { ! /(?:degree\=supr|form=brev)/ } @$res ];

    $res = [ grep { /gender\=$gender/ } @$res ] if $gender;
    $res = [ grep { /number\=$number/ } @$res ] if $number;
    $res = [ grep { /case\=$case/ } @$res ] if $case;

    return $res;
}

# объединяет два хеша, при совпадении ключей суммирует их значения
sub _sum_hashes {
    my ($h1, $h2) = @_;
    #print STDERR Dumper ($h1, $h2);
    my %res = ();
    my $common = { %$h1 };
    $common->{$_}++ for (keys %$h2);
    #print STDERR Dumper ($common);
    for my $unit ( keys %$common ) {
        my $common = {};
        $common = { %{$h1->{$unit}} } if $h1->{$unit};
        $common->{$_}++ for ( keys %{$h2->{$unit}} );
        $res{$unit} = { map { $_ => ($h1->{$unit}{$_} || 0) + ($h2->{$unit}{$_} || 0) } keys %$common };
    }
    return \%res;
}

# проверяет равенство хешей
sub _eq_hashes {
    my ($h1, $h2) = @_;
    return !grep { $h2->{$_} and ($h1->{$_} ne $h2->{$_}) } keys %$h1;
}

sub get_first_noun {
    my ( $self ) = @_;
    for my $word ( split/\s+/, $self->text ){
        next unless ( $word =~ /^[а-яё][-а-яё]+$/i && $self->proj->phrase($word)->norm_phr );
        my $gnc = $self->proj->phrase($word)->get_gender_number_case(add_posp=>1);
        return $word if ( $gnc->{posp} && $gnc->{posp} eq 'S' );
    }
    return '';
}

sub set_to_single_nom {
    my ( $self ) = @_;
    return $self->set_gender_number_case( { case => 'nom', number => 'sg', }, force_set=>1 );
}

sub harmonize_by_first_noun {
    my ( $self ) = @_;
    my $first_noun = $self->get_first_noun;
    return $self->proj->phrase($first_noun)->harmonize($self->text, force_set=>1);
}

sub harmonize_to_single_nom {
    my ( $self ) = @_;
    my $harmonized = $self->proj->phrase( $self->text )->harmonize_by_first_noun;
    return $self->proj->phrase( $harmonized )->set_to_single_nom;
}

sub harmonize_to_nom {
    my ( $self ) = @_;
    my $harmonized = $self->proj->phrase( $self->text )->harmonize_by_first_noun;
    return $self->proj->phrase( $harmonized )->set_gender_number_case( { case => 'nom', }, force_set=>1 );
}

sub harmonize_by_first_rusword {
    my ( $self, $text ) = @_;
    my $first_rusword = '';
    my @before_words = ();
    my @after_words = split /\s+/, $self->text;
    while ( my $word = shift @after_words ){
        if ( $word =~ /^[а-яё][-а-яё]+$/i && $self->proj->phrase($word)->norm_phr ){
            $first_rusword = $word;
            last;
        }
        push @before_words, $word;
    }
    return $text unless $first_rusword;
    my ($before, $after) = ( join(' ', @before_words), join(' ', @after_words) );
    $after = $self->proj->phrase($first_rusword)->harmonize($after);
    return join(' ', grep { $_ }($before, $first_rusword, $after) );
}

sub harmonize {
    my ( $self, $text, %par ) = @_;
    my $src = $self->text;
    return $text unless ( $src && $text );
    # леммер плохо работает с ё
    $src =~ s/ё/е/gi;
    $text =~ s/ё/е/gi;
    my $gnc = $self->proj->phrase( $src )->get_gender_number_case;
    return $text unless $gnc;
    return $self->proj->phrase( $text )->set_gender_number_case( $gnc, %par );
}

our $frmh = {
   i => { case => 'nom', number => 'pl', },
   r => { case => 'gen', number => 'pl', },
   d => { case => 'dat', number => 'pl', },
   v => { case => 'acc', number => 'pl', },
   t => { case => 'ins', number => 'pl', },
   p => { case => 'abl', number => 'pl', },
   ie => { case => 'nom', number => 'sg', },
   re => { case => 'gen', number => 'sg', },
   de => { case => 'dat', number => 'sg', },
   ve => { case => 'acc', number => 'sg', },
   te => { case => 'ins', number => 'sg', },
   pe => { case => 'abl', number => 'sg', },
   is => { case => 'nom', }, #Сохраняем число
   rs => { case => 'gen', },
   ds => { case => 'dat', },
   vs => { case => 'acc', },
   ts => { case => 'ins', },
   ps => { case => 'abl', },
};
our $unfrmh = {};
for(keys %$frmh){
    next unless $frmh->{$_}{number};
    $unfrmh->{ $frmh->{$_}{case}.' '.$frmh->{$_}{number} } = $_;
}

sub set_gender_number_case {
    my ( $self, $target_gnc, %par ) = @_;
    my $proj = $self->proj;

    if( $target_gnc && ! ref( $target_gnc ) ){ #Преобразуем из текстового названия формата в хэш
        return $self->text unless $frmh->{$target_gnc};
        $target_gnc = $frmh->{$target_gnc};
    }
    return $self->text unless ( $target_gnc && ($target_gnc->{gender} || $target_gnc->{number} || $target_gnc->{case}) );

    my $source_gnc = $self->get_gender_number_case;

    my $gender = $target_gnc->{gender} || $source_gnc->{gender} || '';
    my $number = $target_gnc->{number} || $source_gnc->{number};
    my $case = $target_gnc->{case} || $source_gnc->{case};

    # если передан параметр force_set - всегда делаем set
    my $is_source_gnc_ne_target = 0;
    if ( $par{force_set} ){
        $is_source_gnc_ne_target = 1;
    # если нет - то только при неравенстве gnc-хешей
    } else {
        for my $key ( keys %$target_gnc ){
            if ( (not defined $source_gnc->{$key}) || ($target_gnc->{$key} ne $source_gnc->{$key}) ){
                $is_source_gnc_ne_target = 1;
                last;
            }
        }
    }
    return $self->text unless $is_source_gnc_ne_target;

    my ( $changeable, $res ) = $self->_mark_phrase;

    my @gender = ();
    push @gender, $gender;
    push @gender, $_ for ( qw /m f mf n empty/ ); # если не определился gender гармонизируемой фразы, или с эти гендером не найдем словоформу

    # для каждого изменяемого слова ищем нужный вариант словоформы
    # TODO: если число множественное и результат гармонизации не найден, то нужно искать pluralia tantum (дома - жильё)
 L: for my $word (sort keys %$changeable ) {
        my $ufrst = $word =~ /^[А-ЯA-Z]/; #Начиналось ли слово с заглавной буквы
        $word = lc($word);
        # для слов с дефисами гармонизируем каждое слово, а затем снова ставим дефисы
        if ( $word =~/\-/ ) {
            my @arr = split /\-/, $word;
            @arr = map { $self->proj->phrase( $_)->set_gender_number_case( $target_gnc ) } @arr;
            my $harmonized_word = "@arr";
            $harmonized_word =~ s/ /-/g;
            $res->{$harmonized_word} = $changeable->{$word};
            next L;
        }

        my $word_lemms = $self->_get_lemms($word);

        my $posp_lines_all = [()];
        for my $lemm ( @$word_lemms ) {
            my $lemm_posp_lines = $self->_get_grammar2f($lemm);
            push @$posp_lines_all, $_ for ( @$lemm_posp_lines );
        }
        my $posp_lines_filtered = filter_posp_lines( $posp_lines_all );
        $res->{$word} = $changeable->{$word} unless @$posp_lines_filtered;

        for my $gender ( @gender ) {
            for my $posp_line ( @$posp_lines_filtered ) {
                my @posp_line = split /\s+/, $posp_line; # состоит из 3-х частей
                my @posp = split /\,/, $posp_line[1]; # саму часть речи извлекаем из 2-й части
                my @possible_forms;
                @possible_forms = split /\/\//, $posp_line[2] if $posp_line[2];
                for my $form ( @possible_forms ) {
                    for my $grammar_unit ( split /\,/, $form ) {
                        if (    $gender ne 'empty' && $form =~ /gender=$gender/ &&
                                                      ( !$number || $form =~ /number=$number(\,|$)/ ) &&
                                                      ( !$case || $form =~ /case=$case(\,|$)/ )

                            ||  $gender eq 'empty' && ( !$number || $form =~ /number=$number(\,|$)/ ) &&
                                                      ( !$case || $form =~ /case=$case(\,|$)/ )
                           ){
                            my $newword = $posp_line[0];
                            $newword =~ s/^(.)/\u$1/ if $ufrst;
                            $res->{$newword} = $changeable->{$word};
                            next L;
                        }
                    }
                }
            }
        }
        $res->{$word} = $changeable->{$word};
    }

    $res->{$_} //= 0 for keys %$res; #Убираем варнинги при сортировке
    my @res = sort { $res->{$a} <=> $res->{$b} } keys %$res;
    return "@res";
}

sub _mark_phrase {
    my ( $self ) = @_;

    my @words = split /\s+/, $self->text;

    my %changeable = (); # хеш изменяемых слов и их порядковых номеров
    my %unchangeable = (); # хеш неизменяемых слов и их порядковых номеров
    my %rest = ();
    my %rest_gnc = ();

    my %subj1 = (); # первое существительное и его порядковый номер
    my %subj2 = (); # второе существитeльное и его порядковый номер

    my %subj2_gnc = (); # gnc второго существительного

    # первый проход: определяем части речи, заполняем %subj1, %subj2 и по возможности заполняем результирующий и изменяемый хеш
    # все кроме существительных, предлогов и глаголов складываем в rest
    for my $i ( 0..$#words ) {
        my $word = $words[$i];
        if ( $word =~ /^\!/ || $word !~ /^[-а-яё]+$/i ) { # слова с воскл.знаками и нерусские слова не трогаем
            $unchangeable{$word} = $i;
            next;
        }
        my $gnc = $self->proj->phrase($word)->get_gender_number_case(add_posp=>1);
        $gnc->{posp} ||= ''; #Убираем варнинг
        if ( $gnc->{posp} eq 'S' ) {
            delete $gnc->{posp}; # часть речи удаляем
            # первое существительное - изменяемая часть
            if ( !%subj1 ) {
                $subj1{$word} = $i;
                $changeable{$word} = $i;
            # второе существительное - неизменяемая часть
            } elsif ( %subj1 && !%subj2_gnc) {
                %subj2_gnc = %$gnc;
                $subj2{$word} = $i;
                $unchangeable{$word} = $i;
            # последующие существительные - тоже неизменяемая часть
            } else {
                $unchangeable{$word} = $i;
            }
        # предлог - неизменяемая часть
        } elsif ( $gnc->{posp} eq 'PR' ) {
            $unchangeable{$word} = $i;
        # глагол - неизменяемая часть
        } elsif ( $gnc->{posp} eq 'V' ) {
            $unchangeable{$word} = $i;
        # остальное - в rest
        } else {
            $rest{$word} = $i;
            delete $gnc->{posp};
            $rest_gnc{$word} = $gnc;
        }
    }

    # второй проход - по rest, делим на изменяемые и неизменяемые
    my $source_gnc = $self->get_gender_number_case;
    for my $word ( keys %rest ) {
        my $gnc = $rest_gnc{$word};
        my $wpos = $rest{$word};
        #print STDERR "[$word]", Dumper ($source_gnc,$gnc);
        # если нет второго существительного, все слова изменяемые
        if ( !%subj2 ) {
            $changeable{$word} = $rest{$word};
        # 2 существительных, если gnc слова совпадает с gnc гармонизируемой фразы *изгиб гитары желтый*
        } elsif ( _eq_hashes($gnc, $source_gnc) ) {
            # и слово находится между 1 и 2 существительным, не изменяем его (считаем, что оно принадлещит 2-му) *ремонт лампы дневного света*
            if ( _eq_hashes($source_gnc, \%subj2_gnc) && $wpos > (values %subj1)[0] && $wpos < (values %subj2)[0] ) {
                $unchangeable{$word} = $rest{$word};
            # если нет - изменяем
            } else {
                $changeable{$word} = $rest{$word};
            }
        # если отличается - неизменяемое *изгиб гитары желтой*
        } else {
            $unchangeable{$word} = $rest{$word};
        }
    }
    return ( \%changeable, \%unchangeable );
}

sub _get_lemms {
    my ($self, $w) = @_;
    return () unless $w;
    my $h1f = $self->_get_grammar1f($w);
    my $flt = [ grep {!/form=brev/} @$h1f ];
    $h1f = $flt if @$flt;
    my @lemms = ();
    for my $line ( @$h1f ) {
        push ( @lemms, [split ("\t",$line)]->[1] );
    }
    return \@lemms;
}

sub get_gender_number_case_lite {
    my $res = get_gender_number_case(@_);
    return $unfrmh->{$res->{case}.' '.$res->{number}};
}

sub get_case_lite {
    my $res = get_gender_number_case(@_);
    return $unfrmh->{$res->{case}.' '.$res->{number}};
}

#Возвращает тип хвостиков для числа
sub nmb_type {
    my ($self, $nmb) = @_;
    my $t = 2;
    my $dg = $nmb % 10;
    my $dgl = $nmb % 100;
    if( ($dgl > 10 && $dgl < 21) || ($dg > 4) || ($dg == 0) ){
        $t = 3;
    }elsif( $dg == 1 ){
        $t = 1;
    }
    return $t;
}

#Заменяет одно число на другое с сохранением формата фразы
sub change_text_number {
    my ($self, $newnmb, $dbg) = @_;
    if($self->text =~ /(?:^|\s)(\d+)\s+(((?:\S+(?:ая|ое|ий|ие|ые|ый|ых|их|ой)\s+)+)?(\w+))/ ){
        my $nmb = $1; #число
        my $prl = $3; #прилагательные
        my $txt = $4; #существительное
        my $tail = $2;
        my $ph = $self->proj->phrase($txt);
        my $t1 = $self->nmb_type($nmb);
        my $t2 = $self->nmb_type($newnmb);
        if( $t1 == $t2 ){
            my $res = $self->text;
            $res =~ s/$nmb /$newnmb /g;
            return $res;
        }
        my $restxt = "$newnmb ";
        my $more = 0;
        $more = 1 if $self->text =~ /(?:^|\s)([Бб]олее|[Мм]енее|[Оо]коло|от|до)(\s$nmb\s+$tail)/;

        #Ожидаемые параметры, нужны для снятия омонимия для случаев типа "коробка - коробок"
        my $def_param = {};

        $def_param->{'number'} = 'pl';
        $def_param->{'number'} = 'sg' if $t1 == 1;
        $def_param->{'case'} = 'nom' if $t1 == 1;
        $def_param->{'case'} = 'gen' if $more;

        my $gnd = $ph->get_gender_number_case(%$def_param)->{gender};
        $gnd = $ph->get_gender_number_case->{gender} unless $gnd;

        if( $t1 == 2 ){
             if( $gnd eq 'f' ){
                 $def_param->{'case'} = 'nom';
                 $def_param->{'number'} = 'pl';
             }else{
                 $def_param->{'case'} = 'gen';
                 $def_param->{'number'} = 'sg';
            }
        }

        my $c1 = $ph->get_case_lite( %$def_param );
        $c1 = $ph->get_case_lite unless $c1;
        my $newgnc = '';
        if($t1 == 1){
            $c1 .= 'e' unless $c1 =~ /e$/;
            if($c1 eq 'ie'){
                #$newgnc = $t2 == 2 ? 're' : 'r';
                $newgnc = $t2 == 2 ? ($gnd eq 'f' ? 'i' : 're') : 'r';
            }elsif($c1 eq 're'){
                $newgnc = $t2 == 2 ? 'r' : 'r';
            }
        }elsif($t1 == 2){
            $c1 =~ s/e//;
            if($c1 eq 're'){
                $newgnc = $t2 == 1 ? 'ie' : 'r';
            }elsif($c1 eq 'r'){
                $newgnc = $t2 == 1 ? ($more ? 're' : 'ie') : 'r';
            }elsif($c1 eq 'i'){
                $newgnc = $t2 == 1 ? 'ie' : 'r';
            }
        }elsif($t1 == 3){
            $c1 =~ s/e//;
            if($more){
                $newgnc = $t2 == 1 ? 're' : 'r';
            }else{
                $newgnc = $t2 == 1 ? 'ie' : ($gnd eq 'f' ? 'i' : 're');
            }
        }
        if( $newgnc ){
            if( $prl ){
                for my $pr ( grep {$_} split /\s+/, $prl ){
                    my %pr_params = (%{ $frmh->{$newgnc} }, gender =>  $gnd);
                    $pr_params{number} = $t2 == 1 ? 'sg' : 'pl'; #Для $t2 == 2 число прилагательного не согласуется с мужским и средним родами (где sg) и всегда pl
                    $restxt .= $self->proj->phrase($pr)->set_gender_number_case(\%pr_params).' ';
                }
            }
            $restxt .= $ph->set_gender_number_case({%{ $frmh->{$newgnc} }, gender =>  $gnd});
        }
        my $res = $self->text;
        my $lstl = '';
        $lstl = 'а' if $gnd eq 'f';
        $lstl = 'о' if $gnd eq 'n';
        if( $t2 == 1 ){
            $res =~ s/ны(\s+$nmb\s+$tail)/н$lstl$1/g;
        }elsif( $t1 == 1 ){
            $res =~ s/н$lstl(\s+$nmb\s+$tail)/ны$1/g;
        }
        $res =~ s/$nmb\s+$tail/$restxt/g;
        $res .= " => $gnd $t1 $c1 -> $t2 $newgnc " if $dbg;
        return $res;
    }
    return "$self";
}

sub _ends_of_banners_hash :GLOBALCACHE {
    my ($self) = @_;
    my $file = $self->proj->options->{ends_of_banners};
    return {} unless -e $file;
    my $h = {};
    open(F, "<$file") or die("ERROR: ends_of_banners => $!\n");
    while(defined( my $l = <F>)){
        if($l =~ /^\s*(\S+)/){
            $h->{$1}++;
        }
    }
    close(F);
    return $h;
}

sub ends_of_banners_filter {
    my ($self, $log) = @_;
    my $text = $self->text;
    my $h = $self->_ends_of_banners_hash;
    my $re_punc = "[\\.\\,\\!]";
    my $re_word = "(?<=$re_punc)\\s*[А-Яа-я][а-я]+";
    my $re1 = "(($re_word${re_punc}{1,3}\\s*){0,2}(?:$re_word${re_punc}{0,3}\\s*))(domain_\\S*)?\$";
    if( $text =~ /$re1/ ){
        my @arr = map { lc($_) } grep {/\S/} split(/[ \t,\.\!]+/, $1);
        my @farr = grep { $h->{$_} } @arr;
        if(( @farr > 1 )||( @farr == @arr )){
            my $dm = $3 // '';
            $text =~ s/$re1/ $dm/g;
        }
        if( $log ){
            $text .= ' => '.join(' ', map {"[$_]"} @arr)." => ".join(' ', map {"[$_]"} @farr);
        }
    }
    return $text;
}

sub ends_of_banners_filter_log {
    my ($self) = @_;
    return $self->ends_of_banners_filter(1);
}

sub word_grammar_info {
    my $self = shift;
    my $w = shift;

    my $grammar_inf = $self->_get_grammar1f($w);
    my @word_info;
    for my $gi (@$grammar_inf) {
        my ($init_form, $lm, $infos) = split /\s+/, $gi;
        my $part = pop [split ",", $lm] || 'empty';
        for my $info (split /\/\//, $infos) {
            my %dict = map {my @kv = split /=/; $kv[0] => $kv[1]} split ",", $info;
            my $tot_info = join("_", $part, $dict{case} // 'empty');
            push @word_info, $tot_info;
        }
    }
    return join("|", sort keys {map {$_ => 1} @word_info});
}

sub get_changeable_nouns_count {
    my $self = shift;
    my $text = $self->text;
    my @words = split /\s+/, $text;
    my $subj_count = 0;
    for my $i (0 .. $#words) {
        my $word = $words[$i];
        next if ($word =~ /^\!/ || $word !~ /^[-а-яё]+$/i);
        my $gnc = $self->proj->phrase($word)->get_gender_number_case(add_posp => 1);
        $gnc->{posp} ||= ''; #Убираем варнинг
        if ($gnc->{posp} eq 'S' and $gnc->{case} ne 'nom' ) {
            $subj_count++;
        }
    }
    return $subj_count;
};

1;
