package BM::PhraseParser;

use utf8;
use open ':utf8';
use feature 'state';

use std;
use base qw(BM::PhraseCategs);
use Utils::Array;
use Utils::Words;
use List::Util qw(min max sum minstr);

use Time::HiRes qw/time/;

our $brandre = '';
our $brandre_ru = '';
our $brandre_en = '';
our $widebrandre = '';
our $noisere = '';
our $accbrandre = '';
our $dict_metricalre = '';
our $goodsre = '';
our $wear_propre_shorten = '';
our $wear_propre = '';
our $use_goods_basis = 1; # флаг для источников links  см. тикет DYNSMART-398

our %dict_brands = ();
our %dict_auto_brands = ();
our %dict_brands_phl_by_lang = ();  # lang => phrase_list(dict_brands)
our %dict_context_brands = ();
our %dict_acc_brands = ();
our %dict_noises = ();
our %dict_metrical = ();
our %dict_goods = ();
our %dict_surnames_ru = ();
our %dict_surnames_en = ();
our %dict_surnames_wide_ru = ();
our %dict_names_ru = ();
our %dict_wordforms = ();
our %dict_banks = ();
our %dict_notmodels = ();
our $phl_notmodels;
our %dict_geo = ();
our %dict_toponyms = ();
our %hcsyns = ();
our %dict_wear_properties = ();

our %lazy_dicts_filenames = ();

our $adj = "(ий|ый|ой|вая|ная|мая|лая|кая|ые|ие|ое|ых|их)";
our $RU = "А-ЯЁа-яё";
our $LA = "A-Za-z";
our $SUBPHR_LEN = 4;
our @homonymized_goods = ('аккумулятор', 'зарядное устройство', 'аккумуляторная батарея', 'батарея', 'акб', 'чехол'); 

our @local_wides = qw/sport series ports space led titan turbo super mega ultra pro ultra_pro online active classic classica power vision energy plus multi standard master phone system action energy de_luxe trade_in trade-in extreme series style matrix светильники million sport outdoor mini super мини супер sms смс/;

our @model_wides = qw/мтс galaxy mp3 usb sd 2g pro 3g 4g lte full hd hdtv 720p 3d blu-ray dvd cd hdd wireless flash wifi wi-fi ru microusb li-ion/;

our $preps_re = "в|к|у|с|c|а|на|от|со|до|за|но|для|или|под|подо|над|без|и|по";

my $phl_toponyms = '';

our $acc_preps_re = "для|под";

sub class_init :RUN_ONCE {
    my ($class, $opt) = @_;
    my $log = $opt->{logger};
    my $proj = $opt->{'proj'};

    $log->log('PhraseParser::class_init ...'); 

    unless($widebrandre){
        if($opt->{'dict_context_brands'}){
            open(F, $opt->{dict_context_brands}) or die "open failed ($!)";
            my @bad_urls = <F>;
            close(F) or die "close failed ($!)";
            s/ =>.*$// for @bad_urls; #Удаляем перевод
            $widebrandre = join '|', grep {/\S/} map { s/^\s+|\s+$//g; $_ } @bad_urls; ##no critic
        }

    }
    unless($noisere){
        if($opt->{'dict_noises'}){
            open(F, $opt->{dict_noises}) or die "open failed ($!)";
            my @noises = <F>;
            close(F) or die "close failed ($!)";
            s/ =>.*$// for @noises; #Удаляем перевод
            $noisere = join '|', grep {/\S/} map { s/^\s+|\s+$//g; $_ } @noises; ##no critic
        }
    }
    
    unless($accbrandre){
        if($opt->{'dict_acc_brands'}){
            open(F, $opt->{dict_acc_brands}) or die "open failed ($!)";
            my @acc_brands = <F>;
            close(F) or die "close failed ($!)";
            s/ =>.*$// for @acc_brands; #Удаляем перевод
            $accbrandre = join '|', grep {/\S/} map { s/^\s+|\s+$//g; $_ } @acc_brands; ##no critic
        }
    }

    unless($dict_metricalre){
        if($opt->{'dict_metrical'}){
            open(F, $opt->{dict_metrical}) or die "open failed ($!)";
            my @metrical = <F>;
            close(F) or die "close failed ($!)";
            s/ =>.*$// for @metrical; #Удаляем перевод
            $dict_metricalre = join '|', grep {/\S/} map { s/^\s+|\s+$//g; $_ } sort { length $b <=> length $a } @metrical; ##no critic
        }
    }

    unless($goodsre){
        if($opt->{'dict_goods'}){
            open(F, $opt->{dict_goods}) or die "open failed ($!)";
            my @goods = <F>;
            close(F) or die "close failed ($!)";
            s/ =>.*$// for @goods; #Удаляем перевод
            $goodsre = join '|', grep {/\S/} map { s/^\s+|\s+$//g; $_ } @goods; ##no critic
        }
    }

    unless(%dict_brands) {
        if($opt->{'dict_brands'}) {
            open(F, $opt->{dict_brands}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                s/\s*\@.*$//;
                for my $word (split /\s*,\s*/, $_) {
                    $dict_brands{lc($word)} = 1;
                }
            }
            close(F) or die "close failed ($!)";
            my @good_brands = sort grep { /\S/ } keys %dict_brands;
            $brandre = join '|', @good_brands;
            $brandre_ru = join '|', grep { /[А-Яа-я]/} @good_brands;
            $brandre_en = join '|', grep {!/[А-Яа-я]/} @good_brands;
        }
    }

    unless(%dict_acc_brands) {
        if($opt->{'dict_acc_brands'}) {
            open(F, $opt->{dict_acc_brands}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                $dict_acc_brands{lc($_)}++;
            }
            close(F) or die "close failed ($!)";
        }
    }

    unless(%dict_context_brands) {
        if($opt->{'dict_context_brands'}) {
            open(F, $opt->{dict_context_brands}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                my @arr_line = split /\s*@\s*/;
                my ( $left, $right ) = ( lc($arr_line[0]), lc($arr_line[1]) );
                my @arr_left = split /\s*\,\s*/, $left;
                $dict_context_brands{$_}{$right}++ for ( @arr_left );
            }
            close(F) or die "close failed ($!)";
            #print STDERR Dumper (\%dict_context_brands);
        }
    }

    unless(%dict_noises) {
        if($opt->{'dict_noises'}) {
            open(F, $opt->{dict_noises}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                $dict_noises{lc($_)}++;
            }
            close(F) or die "close failed ($!)";
        }
    }

    unless(%dict_metrical) {
        if($opt->{'dict_metrical'}) {
            open(F, $opt->{dict_metrical}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                $dict_metrical{$_}++;
            }
            close(F) or die "close failed ($!)";
        }
    }

    unless(%dict_goods) {
        if($opt->{'dict_goods'}) {
            open(F, $opt->{dict_goods}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                my $line = $_;
                next unless $line;
                my ( $txt, $is_accessory ) = split/\s*\:\s*/, $line;
                $txt =~ s/\s+/ /g;
                $txt =~ s/(^\s+|\s+$)//g;
                ( $txt, my $denorm, my $denorm_plural ) = split/\//, $txt;
                $txt = lc $txt;
                # если встретился дубль и с, и без accessory, считаем, что это accessory
                $dict_goods{$txt}{is_accessory} ||= $is_accessory;
                $dict_goods{$txt}{denorm} = $denorm;
                $dict_goods{$txt}{denorm_plural} = $denorm_plural;
            }
            close(F) or die "close failed ($!)";
            $dict_goods{__INIT__}=0;
        }
#        print Dumper ( \%dict_goods );
    }

    unless(%dict_surnames_ru) {
        if($opt->{'dict_surnames_ru'}) {
            open(F, $opt->{dict_surnames_ru}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                $dict_surnames_ru{lc($_)}++;
            }
            close(F) or die "close failed ($!)";
        }
    }

    unless(%dict_surnames_en) {
        if($opt->{'dict_surnames_en'}) {
            open(F, $opt->{dict_surnames_en}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                $dict_surnames_en{lc($_)}++;
            }
            close(F) or die "close failed ($!)";
        }
    }

    unless(%dict_surnames_wide_ru) {
        if($opt->{'dict_surnames_wide_ru'}) {
            open(F, $opt->{dict_surnames_wide_ru}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                $dict_surnames_wide_ru{lc($_)}++;
            }
            close(F) or die "close failed ($!)";
        }
    }

    unless(%dict_names_ru) {
        if($opt->{'dict_names_ru'}) {
            open(F, $opt->{dict_names_ru}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                $dict_names_ru{lc($_)}++;
            }
            close(F) or die "close failed ($!)";
        }
    }

    unless(%dict_banks) {
        if($opt->{'dict_banks'}) {
            open(F, $opt->{dict_banks}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                $dict_banks{lc($_)}++;
            }
            close(F) or die "close failed ($!)";
        }
    }

    $lazy_dicts_filenames{dict_notmodels}  = $opt->{dict_notmodels};
    unless ($opt->{allow_lazy_dicts}) {
        dict_notmodels()
    }

    unless ( %dict_wordforms ) {
        #print STDERR "inside!\n";
        my $filename = $Utils::Common::options->{DictGrammar}{file};
        if ( -e $filename && -s _ ) {
            open F, $filename or die "open failed ($!)";
            while ( <F> ) {
                chomp;
                my ($word, $gi) = split /\t/;
                my ($parts, $cases) = split /\,/, $gi;
                $parts =~ s/^parts\://;
                $cases =~ s/^cases\://;
                #print STDERR "$_\n";
                $dict_wordforms{$word}{parts}{$_}++ for ( split / /, $parts );
                $dict_wordforms{$word}{cases}{$_}++ for ( split / /, $cases );
            }
            close(F) or die "close failed ($!)";
        }
    }

    unless ( %dict_geo ) {
        if ($opt->{'dict_geo'}) {
            open(F, $opt->{dict_geo}) or die "open failed ($!)";
            while ( <F> ) {
                chomp;
                my ($region, $cities) = split /\t/;
                my @cities = split /\,/, $cities;
                $dict_geo{lc $region}++;
                $dict_geo{lc $_}++ for @cities;
            }
            close(F) or die "close failed ($!)";
        }
    }

    unless( %dict_wear_properties ){
        if($opt->{'dict_wear_properties'}) {
            open(F, $opt->{dict_wear_properties}) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                $dict_wear_properties{lc($_)}++;;
            }
            close(F) or die "close failed ($!)";
            
            if ($wear_propre eq '')
            {
                # если размер словаря больше, чем 7 тысяч слов, заменить использование регулярки использованием match_itemdict, иначе всё будет работать очень медленно
                $wear_propre_shorten = join '|', (grep {/\S/} map { s/^\s+|\s+$//g; $_ } sort keys %dict_wear_properties); ##no critic
            }
        }
    }
    $log->log('PhraseParser::class_init ... done!'); 
}

sub dict_brands {
    return \%dict_brands;
}

sub dict_brands_phl {
    my ($self) = @_;
    my $lang = $self->lang // 'ru';
    my @brands = sort keys %{$self->dict_brands};
    $dict_brands_phl_by_lang{$lang} //= $self->proj->get_language($lang)->phrase_list(\@brands);
    return $dict_brands_phl_by_lang{$lang};
}

sub dict_notmodels {
    unless(%dict_notmodels) {
        if (my $filename = $lazy_dicts_filenames{'dict_notmodels'}) {
            open(F, $filename) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                $dict_notmodels{lc($_)}++;
            }
            close(F) or die "close failed ($!)";
        }
    }
    return \%dict_notmodels;
}

# быстрый метод извлечения строк словаря из фразы 
# $dict - ссылка на хеш словаря
# дальше идет хеш параметров
# item_maxsize - максимальное число слов строки словаря, по которому будет происходить поиск совпадений (1 по умолчанию)
# first_only - искать до первого совпадения
#
# если используется контекстный поиск, то в хеше должны быть заданы следующие параметры
# dict_context - ссылка на хеш контекстного словаря, представленного в формате 'слово' => 'categ: categ1/categ2/...' или подобном
# func_match_context - ссылка на функцию, возвращающую 1, если текущая подфраза найдена в контекстном словаре и соответствует контексту и 0 в противном случае
#   функция принимает параметры: ссылка на хеш контекстного словаря и подфраза
sub match_itemdict {
    my ($self, $dict, %par) = @_;
    return () unless ( %{$dict} && $self->text );
    my $item_maxsize = $par{item_maxsize} || 1;
    $item_maxsize = 1 unless $item_maxsize > 0;

    my @res = ();
    my %hsaved_wordforms = ();
    my @text = ();
    @text = split /\s+/, $self->text;
    # если сравнение по норму, сохраняем словоформы и нормализуем исходный текст с сохранением порядка слов
    if ( $par{by_norm} ){
        $dict = $self->_norm_hash_keys( $dict );
        %hsaved_wordforms = map { lc($self->proj->phrase($_)->norm_phr_safe) => $_ } @text;
        @text = split /\s+/, $self->proj->phrase($self->text)->norm_phr_safe;
#        print Dumper ( [\%hsaved_wordforms, \@text, $dict] );
    }
    my $item_size = $item_maxsize;
    my $break = 0;

    my $dict_context = $par{dict_context};
    my $func_match_context = $par{func_match_context};

    while ( $item_size && !$break ) {
        for my $i (0..$#text) {
            last if ($i+$item_size-1 > $#text); # если подфраза выходит за границы фразы
            my @word = ();
            @word = @text[$i..$i+$item_size-1];

            my $word = clear_edges_fast("@word");
#            $word =~ s/(^[^$RU$LA^0-9]+(?=[$RU$LA^0-9])|(?<=[$RU$LA^0-9])[^$RU$LA^0-9]+$)//;
            if ( $dict_context && exists $$dict_context{lc($word)} && $func_match_context ) {
                my $is_context_found = $self->$func_match_context($dict_context, $word);
                push @res, $word if $is_context_found;
                if ( $par{first_only} ) {
                    $break = 1;
                    last;
                }
            } elsif ( $$dict{lc($word)} ) {
                push @res, $word;
                if ($par{first_only}) {
                    $break = 1;
                    last;
                }
            }
        }
        $item_size--;
    }
    # восстанавливаем исходные словоформы для сравнений по норму
    @res = map { $hsaved_wordforms{$_} || $_ } @res if $par{by_norm};
    return @res;
}

sub _norm_hash_keys {
    state %cache;
    my ($self, $hash) = @_;
    if (not exists $cache{$hash}) {
        my %res = ();
        $res{ lc($self->proj->phrase($_)->norm_phr_safe) }++ for ( keys %$hash );
        $cache{$hash} = \%res;
    }
    return $cache{$hash};
}

# функция поиска в контекстном словаре брэндов
sub match_context_brand {
    my ( $self, $dict_context, $word ) = @_;
    my $res = 0;
    my %context = ();
    for my $line ( keys %{$$dict_context{lc($word)}} ) {
        my @arr = split /\s*\:\s*/, $line;
        $context{$arr[0]} = $arr[1];
        #print STDERR "$line\n";
    }
    #print Dumper (\%context);
    if ( exists $context{word} ) {
        #print "inside context\n";
        my @arr_words = split /\s*\,\s*/, $context{word};
        for my $context_word ( @arr_words ) {
            $res = 1 if ( $self->text =~ /\b$context_word\b/i );
        }
    }
    if ( exists $context{categ} && !$res ) {
        #print "inside_categ\n";
        my $context_categs = $context{categ};
        my @arr_categs = $self->get_minicategs;
        for my $categ ( @arr_categs ) {
            $res = 1 if ( $context_categs =~ /$categ/i );
        }
    }
    return $res;
}

sub match_itemdict_context {
    my ($self, $dict_name ) = @_;
#    my $ctgs = { map { $_ => 1 } $self->get_minicategs };
    my $ctgs = {}; # временно оторвали категории, они пока не нужны, потому что контекты определяем по ключевым словам
    $hcsyns{$dict_name} ||= BM::ContextSyns::ContextLinks->new({proj => $self->proj, ( file => $self->proj->{options}->{$dict_name} ) }); # categs_growscale_mapping_10 пока оторвали
    my $parsed = $hcsyns{$dict_name}->parse_phr($self, context => $ctgs, strict => 1, no_replace => 1);
    return keys %$parsed;
}

sub test_mid {
    my ( $self ) = @_;
    my @abrand = $self->match_itemdict(\%dict_brands, func_match_context=>\&match_context_brand, dict_context=>\%dict_context_brands, item_maxsize=>4, first_only=>1);
    #print Dumper (\@abrand);
    return @abrand;
}

sub get_brand {
    my ( $self, %par ) = @_;
    my @cbrands = ();
    @cbrands = $self->_get_context_brands if exists($par{use_context});
    return $cbrands[0] if @cbrands;
    return $self->_get_brand;
}

sub _get_context_brands :CACHE {
    my $self = shift;
    return $self->match_itemdict_context('dict_context_brands');
}

sub _get_brand :CACHE{
    my $self = shift;
    my @brands = ();
    @brands = $self->match_itemdict(\%dict_brands, item_maxsize=>5, first_only=>1);
    return '' unless @brands;
    my $res = $brands[0];
    return '' if ( $res =~ /^(фея|николь|рос|sparta|россия)$/i && $self->text !~ /$res\s+\d+/ );
    return $res;
    return '';
}

sub get_brands {
    my ( $self, %par ) = @_;
    my @cbrands = ();
    @cbrands = $self->_get_context_brands if exists($par{use_context});
    my @brands = ();
    @brands = $self->match_itemdict(\%dict_brands, item_maxsize=>5, first_only=>0);
    push @brands, @cbrands if @cbrands;
    return @brands;
}

sub get_brands_snorm_phl {
    my ( $self, %par ) = @_;

    my $brands_phl = $self->dict_brands_phl->search_subphrases_in_phrase($self);
    #print "brands_phl: ", $brands_phl, "\n";
    return $brands_phl;
}

sub get_acc_brand {
    my ( $self ) = @_;
    my @acc_brands = $self->match_itemdict(\%dict_acc_brands, item_maxsize=>4, first_only=>1);
    return $acc_brands[0] if @acc_brands;
    return '';
}

sub get_bank {
    my ( $self ) = @_;
    my @abanks = $self->match_itemdict(\%dict_banks, item_maxsize=>4, first_only=>1);
    #print Dumper (\@abrand);
    return $abanks[0] if @abanks;
    return '';
}

sub get_toponyms {
    my ( $self ) = @_;
    my $res = '';
    unless ( %dict_toponyms ){
        my $cattree = $self->proj->categs_tree;
        %dict_toponyms = map { $_ = lc $_;  s/.region (?:республика )?//; $_ => 1 } $cattree->get_minicategs_subtrees('.regions'); ##no critic
    }
    my @toponyms = $self->match_itemdict(\%dict_toponyms, item_maxsize=>3);
    my @reg = $self->get_regions;
    push @toponyms, $_->{town} for ( @reg );
    $res = join (':', @toponyms);
    return $res;
}

sub get_goods {
    my ( $self, %par ) = @_;

    my @result = $self->_get_goods;
    return @result if $par{all_fields};
    return $result[0];
}

sub phl_dict_goods :GLOBALCACHE {
    my $self = shift;
    return $self->proj->phrase_list([sort keys %dict_goods]);
}

sub dict_goods_words :GLOBALCACHE {
    my $self = shift;
    my %words_freq = ();
    for my $phr ( @{$self->phl_dict_goods} ) {
        for my $word ( $phr->uniqnormwords ) {
            $words_freq{$word}++;
        }
    }
    my %words = ();
    for my $phr ( @{$self->phl_dict_goods} ) {
        my $max_freq_word;
        my $max_freq = 0;
        for my $word ( $phr->uniqnormwords ) {
            if ( $words_freq{$word} > $max_freq ) {
                $max_freq = $words_freq{$word};
                $max_freq_word = $word;
            }
        }
        $words{$max_freq_word} = 1 if defined $max_freq_word;
    }
    return \%words;
}

# быстрая проверка:
# 1 - во фразе, возможно, есть товары из dict_goods
# 0 - во фразе точно нет товаров из dict_goods
sub dict_goods_fast_check :CACHE {
    my $self = shift;
    my $goods_words = $self->dict_goods_words;
    for my $word ($self->uniqnormwords) {
        return 1 if $goods_words->{$word};
    }
    return 0;
}

sub _get_goods :CACHE {
    my $self = shift;

    return '' unless $self->dict_goods_fast_check;

    my $phl_found = $self->phl_dict_goods->search_subphrases_in_phrase( $self, only_norm=>1 )->filter_subphrases;

    return '' unless $phl_found;
    my $res_norm = '';
    if ( $phl_found->count > 1 ){ # из нескольких найденных выбираем ту, что ближе к началу фразы
        my %hpos_of_first_word = ();
        my %hcompactness = ();
        my @src_norm = map { $self->proj->phrase($_)->norm_phr || $_ } $self->words;
        my %src_ordered = ();

        $src_ordered{$src_norm[$_]}=$_ for ( 0..$#src_norm );
        $self->proj->logger->debug('in _get_goods src_ordered=', \%src_ordered);
        # для каждой найденной фразы вычисляем словопозицию первого слова
        for my $phrase_found ( @$phl_found ){
            my @wpos = map { $src_ordered{$_} // () } $phrase_found->words;
            my $fpos = min(@wpos);
            $hpos_of_first_word{$phrase_found->text} = $fpos;
            my $lpos = max(@wpos);
            $hcompactness{$phrase_found->text} = ($lpos-$fpos)/$self->wordcount;
            # чтобы в словаре не дублировать многие типы для слова аксессуар, такое правило: если слово аксессуар конкурирует с другими, ему дается низший приоритет (Аксессуар чехол-книжка ...)
            $hcompactness{$phrase_found->text} = 9999999 if $phrase_found->text =~ /^аксессуар$/i;
        }
        $res_norm = (sort { $hpos_of_first_word{$a} <=> $hpos_of_first_word{$b} || $hcompactness{$a} <=> $hcompactness{$b} } @{$phl_found->perl_array})[0];
    } else {
        $res_norm = $phl_found->perl_array->[0];
    }

    return '' unless $self->_is_type_correct( $res_norm );

    my $res = '';
    $res = $dict_goods{$res_norm}{denorm} if $dict_goods{$res_norm}{denorm};

    $res ||= $self->_denorm_text( $res_norm );

    my $is_accessory = $dict_goods{$res_norm}{is_accessory};
    return ( $res, $is_accessory );
}

sub get_goods_simple {
    my ( $self ) = @_;
    my $phl_found = $self->phl_dict_goods->search_subphrases_in_phrase( $self, only_norm=>1 )->filter_subphrases;
    return '' unless $phl_found->count;
    return minstr(@$phl_found);
}

#TODO: встроить в парсер, как и get_goods
sub _check_type {
    my ( $self, $type, $model, $brand ) = @_;
    return 0 unless $type;
    my %hh_pos = ();
    my @atype = map { $self->proj->phrase($_)->norm_phr || $_ } split /\s+/, $type;
    my $type_first = shift @atype;
    my $type_last = '';
    $type_last = pop @atype if @atype;
    my $text = $self->text;
    $text =~ s/(\!|\.\:\;)/ $1 /g;
    my @atext = map { $_ =~ /^\.|\!$/ ? $_ : $self->proj->phrase($_)->norm_phr || $_ } split /\s+/,$text;
#    print STDERR "+++ [@atype] => @atext => $type_first\n";
    return 0 if "@atext" =~ /(^|\s)для\s$type_first/;
    return 1 unless ( $model || $brand );
    for my $i ( 0..$#atext ){
        $hh_pos{$atext[$i]}{$i}++;
    }
    for my $line ($model, $brand){
        next unless $line;
        my @words = split /\s+/,$line;
        my $line_first = shift @words;
        my $line_last = '';
        $line_last = pop @words if @words;
        for my $typeword ( $type_first, $type_last ){
            my @wpos_typeword = keys %{$hh_pos{$typeword}};
            for my $lineword ( $line_first, $line_last ){
                my @wpos_lineword = keys %{$hh_pos{$lineword}};
                for my $wptw ( @wpos_typeword ){
                    for my $wplw ( @wpos_lineword ){
                        return 1 if ( abs($wptw-$wplw) == 1 );
                    }                  
                }
            }
        }
    }
    return 0;
}

# насколько похожи две фразы посимвольно в долях от 1
sub compare_by_symb {
    my ( $self, $txt ) = @_;
    return 0 unless ( $self->text && $txt );
    return 1 if $self->text eq $txt;
    my @arr1 = split //, $self->text;
    my @arr2 = split //, $txt;
    my $cnt = 0;
    for my $i ( 0..$#arr1 ){
        last if not defined $arr2[$i];
        last if $arr1[$i] ne $arr2[$i];
        $cnt++;
    }
    return $cnt/length($self->text);
}

#Возвращаем ссылку на хэш брендов
sub _get_brands_hash {
    my ($self) = @_;
    return \%dict_brands; 
}

sub is_brand_exactly :CACHE {
    my ($self) = @_;
    my $brand = $self->get_brand; 
    return 1 if ( $self->proj->phrase($brand)->norm_phr eq $self->norm_phr );
    return 0;
}

sub is_brand :CACHE {
    my ($self) = @_;
    #    return 1 if $self->name =~ /^\s*\(?(($brandre)(\s*\(\s*\d+\s*\))?(\s*,)?)\)?\s*$/i;
    #    return 1 if $self->name =~ /^\s*\(?(($widebrandre)(\s*\(\s*\d+\s*\))?(\s*,)?)\)?\s*$/i;
    # return 1 if $self->text =~ /^(?:TM|ТМ)?\s*\(?["']?(($brandre|$widebrandre)(\s*\(\s*\d+\s*\))?(\s*,)?)["']?\)?\s*$/i;
    my ($brand, $model) = $self->parse_fast;
    return 1 if $brand && (!$model) && ($self->proj->phrase($brand)->norm_phr eq $self->norm_phr); 
    return 0;
}

sub has_model {
    my ($self) = @_;
    my %h = $self->_parse_brand_model;
    return 1 if $h{model}; 
} 

###############################################
#Работа с выделением фраз из текстов
###############################################

our $re_rusw = '\b[-А-Яа-я]+\b'; #русское слово
our $re_engw = '\b[-A-Za-z]+\b'; #английское слово
our $re_rusengw = '\b[-А-Яа-яA-Za-z]+\b'; # смешаное слово
our $re_mrus = '\b(?:[А-Яа-я][-\.А-Яа-я]*\d[-\.А-Яа-я0-9]*|\d+[-\.А-Яа-я0-9]*[А-Яа-я][-\.А-Яа-я0-9]*)\b'; #русские буквы и цифры
our $re_meng = '\b(?:[A-Za-z][-\.A-Za-z]*\d[-\.A-Za-z0-9]*|\d+[-\.A-Za-z0-9]*[A-Za-z][-\.A-Za-z0-9]*)\b'; #английские буквы и цифры
our $re_mruseng = '\b(?:[А-Яа-яA-Za-z][-\.А-Яа-яA-Za-z]*\d[-\.А-Яа-яA-Za-z0-9]*|\d+[-\.А-Яа-яA-Za-z0-9]*[А-Яа-яA-Za-z][-\.А-Яа-яA-Za-z0-9]*)\b'; #русские и английские буквы и цифры
our $re_mmeng = '\b[-A-Za-z0-9]*[A-Za-z][-A-Za-z0-9]*\b'; #английские буквы с допустимыми цифрами

#Общие типы моделей - ПОРЯДОК ВАЖЕН!
our $modeltypes = [
    "\\d+\"\\s+[A-Za-z]\\d+", # 7" T7
    "$re_engw\\s+\\d+\\.\\d+(?:\\s+[A-Za-z]{1,3}){1,2}", # Timberk TOR 21.2009 MG I 
    "$re_engw\\s+$re_engw\\s+$re_meng", # Samsung Galaxy Tab SM-153x 
    "$re_engw\\s+$re_engw\\s+\\d+", # NOIROT Spot E-III 1500 
    "$re_meng\\s+\\d+(?:\\s+[A-Za-z]{1,3})?", #Nobo C4F 20 XSC
    "$re_meng\\s+[A-Za-z]{1,3}\\s+\\d+\\s+[A-Za-z]{1,3}", # Timberk TEC.PS1 LE 1500 IN
    "$re_meng\\s+$re_meng", 
#    "(?:(?:$re_engw){1,2}\\s+)?[A-Za-z]+\\-\\d+(?:\\-[A-Za-z]+)?", 
    "(?:$re_meng|$re_engw|\\d+(\\.\\d+)?)\\s+(?:$re_meng|$re_engw|\\d+(\\.\\d+)?)",
    qq{(?:['"]?(?:$re_engw|$re_meng)['"]?)?\\s+(?:арт\\.\\s*)?(?:[\.a-z0-9]+(?:\\-[\\.a-z0-9]+)+)(?:\\[-_][a-z0-9]+)?},
    "$re_meng",
    "\\d{3,}(?:\\.\\d+)?", # 306
];

#Типы моделей только для русских вендеров - ПОРЯДОК ВАЖЕН!
our $modeltypes_ru = [
    "$re_mrus", 
    "(?:(?:$re_rusw){1,2}\\s+)?[A-Za-z]+\\-\\d+(?:\\-[A-Za-z]+)?", 
##    "(?:$re_rusw|$re_mrus|\\d+)\\s+(?:$re_rusw|$re_mrus|[\\d-]+)",
    "(?:\\d+\\s+)?(?:$re_mruseng|$re_rusengw|$re_mrus|$re_rusw|[\\d-]+)",
##    qq{(?:['"]?(?:$re_rusw|$re_mrus)['"]?)?\\s+(?:арт\\.\\s*)?(?:[a-zа-я0-9]+(?:\\-[a-zа-я0-9]+)+)(?:\\[-_][a-zа-я0-9]+)?},
    "([-А-Я]{3,}|[А-Я]{2,})", # АБ, КПЧ
];

our $modeltypes_ab = [
    "\b(?:[А-Я][-а-яё]+)\b",
];

our $postfixre = '(?:([\.]|\s+)(?:\d*[A-Z]{1,}|[A-Z][a-z]+(?:\s+[A-Z][a-z]+)*)(?=\s|$))?';

our $modeltypesre = join('|', @$modeltypes);
our $modeltypesre_ru = join('|', @$modeltypes_ru);

our $rpls = {
   '\bд\/д\b' => ['детей ', 'детские '],
   '\bдет(с|ск)?\.\b' => ['детей ', 'детские '], 
   '\bвлажн?\.\b' => ['влажные '],
   '\bстир\. ' => ['стиральный '],
   '\bдев\. ' => ['девочек '],
   '\bроз\. ' => ['розовый '],
};

our @widemodelwords = qw{ \d+Gb \d+Mb };
our $wmdlflt = '(^|\s)('.join('|', @widemodelwords).')(\s|$)';

#Очищаем текст от различных сокращений
sub _cleared_text {
    my ($self) = @_;
    my $text = $self->text;
    
    #Убираем количественные слова
    $text =~ s/\b\d+\s*(мл|шт|кг)\b\.?//g;
    $text =~ s/\(\s*\)//g;

    $text =~ s/\bд\/($re_rusw)/для $1/;

    $text =~ s/\://g;

    my @res = ($text);
    for my $re (sort keys %$rpls){
        for my $txt (@res){
            if($txt =~ /$re/){
                @res = grep { $_ ne $txt } @res;
                for my $nt (@{$rpls->{$re}}){
                    my $ctxt = $txt;
                    $ctxt =~ s/$re/ $nt /g;
                    push(@res,$ctxt);                    
                }
            }
        }
    } 
    return @res;
}
our $getbrandre = '';
sub getbrandre { $getbrandre ||= qr{(?:^|\s)($brandre)(?:\s|$)}i; return $getbrandre;}
sub get_brand_old {
    my ($self) = @_;
    my $text = $self->text;
    my $re = getbrandre;
    return $1 if $text =~ /$re/;
    return '';
}

our $post_prefilter_bannerre = '';
sub get_post_prefilter_bannerre {
    $post_prefilter_bannerre ||= qr{(?:\s|^|[,])(?:quot|за \d+|pink|\d{4} год(а)?|(?:c|до)\s+\d\d?\.\d\d?(?:\.\d\d\d{2}?)?|от производителя|от \d+|trade-in|trade in|планируете|хотите|в кредит|опт и розн|быстро и легко|где|предлагает|найти|поиск|беспокоит|хотите|нуж(?:на|ны|ный|)|нужен|ище(?:те|шь|м)|узна(?:й|йте|ть)|недорого|где можно|продать|купи(?:те|шь|ть)?|купля|прода(?:жа|ются|ется)|доставка|получ(?:ить|ение)|заказ(?:ать)?|(?:(?:до|на|свыше|более)\s+)\d*_percent|официальный дилер|белый_ветер|бытовая техника|бытовой техники|(?:dmn_)?холодильник.ру|доставка по россии|(?:[а-я]{2}\s+)?(?:предпраздничн\w+\s+)?(?:скидк|ставк)\w+(?:\s+[а-я]{2}?)|выгодный|выгодная(?: доставка)?(?: в ваш регион)?|дешев\w+|специальное_предложение|(спец\.?\s*)?предложение( на)?|качественный_сервис|(?:(?:в|от)\s+)?(?:dmn_\w+|[\w\.]+(?:ru|ua|com|org|net|by))|интернет_магазин|белый_ветер|220_instrument|под_ключ|сотмаркет|(?:в )?евросет(?:ь|и)|самовывоз|24_часа|в_подарок|ситилинк|здесь|sale(?: на)?)(?=(\s|[.,!:?]|$))};
    return $post_prefilter_bannerre;
}


sub get_model_inf {
    my ($self) = @_;
    my $text = $self->text;
    if( $text =~ s/\b($brandre)\b/=====/i ){
        my $vender = $1;
        my $badphrs = 'в|на|для|по|и|покупают|у|оптом|со?|in|for|and';
        if($vender =~ /[А-Яа-я]/){ #Русские модели
            if( $text =~ s/=====(?!\s+(?:$badphrs)\b)\s+((?:$modeltypesre|$modeltypesre_ru)$postfixre)// ){
                my $model = $1;
                return ($text, $vender, $model);
             }
        }else{ #Английские модели
            if( $text =~ s/=====(?!\s+(?:$badphrs)\b)\s+((?:$modeltypesre)$postfixre)// ){
                my $model = $1;
                return ($text, $vender, $model);
             }
        }
        $text =~ s/=====\s*//;
        return ($text, $vender); 
    }
#    return ("$1 $4", $2, $3) if $text =~ /(^|.+?\s)((?i)$brandre)\s+($modeltypesre)(\s.+?|$)/;
#    return ("$1 $4", $2, $3) if $text =~ /(^|.+?\s)((?i:$brandre))(\s.+?|$)/;
    return ($self->text);
}

#Фразы, очень сильно похожие на модели
#Используется для остановки обхода иерархии, важно чтобы было достаточно чисто по качеству
#Используется ещё и при категоризации
sub _get_model_like_phrases : CACHE {
    my ($self) = @_;
    my @arr = ();
    my $text = $self->text;
    $text =~ s/\([^\)\(]*\)//g; #Удаляем всё в круглых скобках
    $text =~ s/\([^\)\(]*\)//g; #Удаляем всё в круглых скобках, если были вложенности
    $text =~ s/([^\/ ]+\/)([^\/ ]+\/)+[^\/ ]+//g; #удаляем детализации конфигураций, что часто встречается при описаниях моделей
    my $ph = $self->proj->phrase($text);
    $text = join " ", $ph->words_without_stops;

    $text =~ s/$wmdlflt/ /ig;
 
    $text =~ s/[\/\+\&]/ /g;
    $text =~ s/\&\w+;//g; #удаляем символы html

#print "(($brandre)(\s+$re_meng|(\s+($re_engw|$re_meng|\d+)){2}))";
#    push(@arr, $1) if $text =~ /(($brandre)(\s+$re_meng|(\s+($re_engw|$re_meng|\d+)){2})?)/i;
#    print "$text\n";
#print "$brandre\n";

#my $tt = '';
#$tt = $1 if $text =~ /(($brandre)(\s+$re_meng|(\s+($re_engw|$re_meng|\d+)){2}))/i;
#my $ph = $self->proj->phrase($tt);
#print $ph->badphrsreason."<<\n"; 


    if( $text =~ /[A-Za-z]/ ){ #Для случая английского или смеси русского и английского текста вытаскиваем английский
        push(@arr, $1) if $text =~ /(($re_engw\s+){1}$re_meng)/;   #английская модель и слово перед ним
        push(@arr, $1) if $text =~ /(($re_meng\s+){1}$re_meng)/;   #два модельных слова подряд
        push(@arr, $1) if $text =~ /(($re_engw\s+){2}\d\d+)(\s|$)/;      #два английских слова и число
        push(@arr, $1) if $text =~ /((^|$re_engw\s+)($re_engw\s+){2}\d)(\s|$)/;      #три английских слова (два, если от начала строки) и цифра
        push(@arr, $1) if $text =~ /($re_engw\s+\d\d+$re_engw)/;   #английское слова, число, английское слова
        push(@arr, $1) if $text =~ /(([-A-Za-z]+\s+){2,3}II+)/;    #английские фразы, заканчивающиеся на слово II
        push(@arr, $2) if $text =~ /(^|\s)(($re_mmeng\s+){2,3}(II+|I?V|VI+|IX))(\s|$)/;    #английские фразы, заканчивающиеся на слово II
        push(@arr, "$1$3") if $text =~ /($re_engw\s+)($re_engw\s+)+($re_meng)/; #доклеиваем первое англ. слово (надеясь, что это вендер) и модель
        push(@arr, $1) if $text =~ /(($re_engw\s+){2}[A-Z]+)(\s|$)/; #два английских слова и слово из заглавных букв
        push(@arr, $1) if $text =~ /(($re_engw\s+){2}\d+\.\d+)(\s|$)/; #два английских слова и число с точкой
        push(@arr, $1) if $text =~ /$re_rusw\s+($re_engw\s+\d+\s+$re_engw)/; #вариант выделения модели после русского слова
    }
    if((!@arr) && ($text =~ /[А-Яа-я]/)){ #есть русские слова
        if( $text =~ /(($re_rusw\s+){1}($re_mrus))/ ){  #русская модель и слово перед ним
            my $mtxt = $1;
            my $mm = $3;
            unless($mm =~ /^\d+\-?(х|ой|ие|ая|ый|ое|е|й)$/){
                push(@arr, $mtxt);   
            }
        }
    }

    #удаляем повторы
    @arr = keys %{{ map { $_ => 1 } @arr }}; 

    @arr = map {$_->text} $self->proj->phrase_list({phrases_arr => \@arr })->pack_list->good_phrases_list->phrases;

    push(@arr, $1) if $text =~ /(($brandre)(\s+$re_meng|\s+\d{3,}|(\s+($re_engw|$re_meng|\d+)){2}))/i;
    push(@arr, $1) if $text =~ /(($brandre_ru)(\s+$re_mrus|\s+\d{3,}|(\s+($re_rusw|$re_mrus|\d+)){2}))/i;

    #Вендеры электродеталей более сложные
    push(@arr, $1) if $text =~ /(($brandre)(\s+['"]?($re_rusw|$re_engw|$re_mrus|$re_meng)['"]?)?\s+(арт\.\s*)?([a-zа-я0-9]+(\-[a-zа-я0-9]+)+)(\_[a-zа-я0-9]+)?)/i;

    return @arr;
}

sub _get_good_phrases : CACHE {
    my ($self) = @_;

    my @arr = ();

    my $origtext = $self->text;
    my $text = $self->text;

    $text =~ s/[\(\)]/ /g if $text =~ /^$re_rusw(\s+$re_rusw)?\s+\(i($re_rusw(\s+$re_rusw)+)\)$/; #Если большая часть текста в скобках - удаляем сами скобки, а не текст

    $text =~ s/\([^\)\(]*\)//g; #Удаляем всё в круглых скобках
    $text =~ s/\([^\)\(]*\)//g; #Удаляем всё в круглых скобках, если были вложенности
    $text =~ s/([^\/ ]+\/)([^\/ ]+\/)+[^\/ ]+//g; #удаляем детализации конфигураций, что часто встречается при описаниях моделей
    $text =~ s/([-0-9\.]+\/)([-0-9\. ]+\/)+[-0-9\. ]+//g; #удаляем наборы цифр через слэш, включая пробелы
    $text =~ s/(\/[^\/]+)(\/[^\/]+)+//g; #удаляем детализации конфигураций, что часто встречается при описаниях моделей
    $text =~ s/\b(ПО|СО)\b/!$2/g; #заменяем болезненные абравиатуры, совпадающие со стоп-словами
    #удаляем количественные подфразы
    $text =~ s/\s([a-z]=)?((\d+([\.\,]\d+)?)([-\/\*xXхХ]\d+([\.\,]\d+)?)*)\s*(штук|шт|таблеток|таб|мл|кг|г|gb|mb|гб|мб|см|м|мм|кв|ден|короб(ка|ок))(\b|[\.:,]|$)(\s+в\s+(упаковке|уп\.?|пачке|коробке|кор\.?))?(\s|$)/ /g; 

    my $ph = $self->proj->phrase($text); #отфильтрованная фраза

    my @wrds = $ph->words_without_stops;
    @wrds = grep {! /^\d+(мл|г|кг|gb|mb|гб|мб)$/} @wrds;

    $text = join(" ",  @wrds);

    #короткие фразы (до 3 слов включительно) не меняем
    if(@wrds <= 3){
        push(@arr, join(" ", @wrds));
    }

    push(@arr, $1) if $text =~ /(($re_mmeng\s+)+$re_mmeng)/; #последовательности английских слов
    push(@arr, $1) if $text =~ /(($re_rusw\s+){1,2}$re_engw)/;  #переход от русского текста к английскому

    #отдельно отрабатываем "для"
    my $prefix = '';
    if($self->text =~ /^(\S+(\s\S+)*)\sдля\s/){
        $prefix = $1;
    }

    if($prefix){
        @arr = map {"$prefix $_"} @arr; #доклеиваем ко всем фразам префикс

        my $shortprefix = '';
        $shortprefix = $1 if $prefix =~ /(\S+)\s*$/;

        #смелее выделяем подфразы
        push(@arr, "$prefix $1") if $text =~ /($re_engw \d\d+)/;
        push(@arr, "$prefix $1") if $text =~ /($re_engw(\s+$re_engw)+)/;
        push(@arr, "$prefix $1") if $text =~ /($re_mrus)/;
        push(@arr, "$prefix $1") if $text =~ /($re_meng)/;
        push(@arr, "$shortprefix $1") if $text =~ /($re_engw(\s+$re_engw)+)/;

        push(@arr, "$1 $3") if $self->text =~ /^(\S+(\s\S+){2})\sдля\s(\S+)/; #если перед для 3 слова
        push(@arr, "$1 $3") if $self->text =~ /^(\S+(\s\S+){1})\sдля\s(\S+)/; #если перед для 2 слова
        push(@arr, "$1 $2") if $self->text =~ /^(\S+)\sдля\s(\S+\s\S+)/; #если перед для 1 слова
        push(@arr, "$1 $2", "$1 $3") if $self->text =~ /^(\S+)\sдля\s+(\S+),\s+(\S+)/; #разворачиваем запятые после для
        push(@arr, "$1 $2", "$1 $3", "$1 $4") if $self->text =~ /^(\S+)\sдля\s+(\S+),\s+(\S+),\s+(\S+)/; #разворачиваем запятые после для
        push(@arr, (map {"$1 $_"} split(",",$2)), "$1 $5", "$1 $6" ) if $self->text =~ /\sдля\s+(лечения)((\s+$re_rusw(\s+$re_rusw)?,)+)\s+($re_rusw)\s+и\s+($re_rusw(\s+$re_rusw))/; #обработка части перечислений после для
    }

    push( @arr, $ph->_get_model_like_phrases); #выделяем моделеподобные фразы


    push(@arr, $1, $2) if $self->text =~ /^(\S+)\s-\s(\S+\s\S+)/; #слово перед тире
    push(@arr, $1) if $self->text =~ /(($re_rusw\s+){1,2}"+$re_rusw"+)/;  #обработка однословных названий в кавычках
    push(@arr, $1) if $self->text =~ /"+((\S+\s+){3,5}\S+)("+|\()/;  #обработка текстов в кавычках
    push(@arr, $1) if $self->text =~ /""+((\S+\s+){3,7}\S+)""+/;  #обработка текстов в кавычках
    push(@arr, $2, $4) if $self->text =~ /""+((($re_rusw\s+){2,7})\((($re_rusw\s+){2,6}$re_rusw)\))""+/;  #обработка текстов в кавычкаe
    push(@arr, $2, $4) if $self->text =~ /""+((($re_engw\s+){2,7})\((($re_rusw\s+){2,6}$re_rusw)\))""+/;  #обработка текстов в кавычкаe
    push(@arr, $2, $4) if $self->text =~ /""+((($re_rusw\s+){2,7})\((($re_engw\s+){2,6}$re_engw)\))""+/;  #обработка текстов в кавычкаe
    push(@arr, $1) if $self->text =~ /""+((([A-Za-z]{1,3}\s+)*$re_engw\s+){4})/;  #первые 4 слова длинной фразы в кавычках
    push(@arr, $1) if $self->text =~ /Как (($re_rusw\s+){2,3}$re_rusw)/i; #фразы со словом как
    push(@arr, $1) if $self->text =~ /[А-Яа-я]: (($re_rusw\s+){2,3}$re_rusw)/i; #фразы после :
    push(@arr, $1) if $text =~ /(($re_rusw\s+){3})($re_engw|$re_meng)/i; #три русских слова перед английским
    push(@arr, $1) if $text =~ /^([А-Яа-я]+\-[А-Яа-я]+\s+\S+)/; #слова с дефисом в начале предложения добавлять безопасно
    push(@arr, $1) if $text =~ /(($re_rusw\s+){2}[А-Яа-я]+\-[А-Яа-я]+)/; #слово с дефисом и два слова перед ним
    push(@arr, $1) if $text =~ /^"?($re_engw(\s+$re_rusw){2})/; #добавляем фразы, начинающиеся с англ.слова от начала предложения

    push(@arr, $1, "$1 $3") if $text =~ /(([^ \.]+\s+){2,3})(\d+\s+класс)/; #Отдельное правило для учебников
    push(@arr, $1) if $origtext =~ /(($re_rusw\s+){2}языка?)/; #Отдельное правило для языков
    push(@arr, $1) if $origtext =~ /($re_rusw\s+языка?\s+по\s+$re_rusw)/; #Отдельное правило для языков
    push(@arr, "$1 $2") if $origtext =~ /($re_rusw\s+языка?)[^\(]+\(([^\(]+)\)/; #Отдельное правило для языков

    unless(@arr){ #если не получается ничего путного
        for(my $i=0; $i < @wrds-3; $i++){
            push(@arr, join(" ", @wrds[$i .. $i+3]));
        }
    }

    push(@arr, join(" ",@wrds[0 .. 3])) if @wrds > 3; #всегда доклеиваем первые 4 слова

    #подправляем формат
    @arr = map {lc($_)} map { ##no critic
         s/[,"':?!«»]/ /g; 
         s/\-(\s|$)/ /g; 
         s/[\/]/ /g; 
         s/\.+( |$)/ /g; 
         s/\s\s+/ /g; 
         s/( ?)\(/$1/g; 
         s/\)( ?)/$1/g;
         s/([а-я])[\.\+]([а-я])/$1 $2/gi; 
         $_
       } @arr;
    @arr = map { $self->proj->phrase($_)->pack_phr } @arr; #Удаляем повторы слов

    return @arr;
}

sub get_good_subphrases : CACHE {
    my ($self) = @_;
    my @arr = $self->_get_good_phrases;
#    my $phl = $self->proj->phrase_list({phrases_arr => \@arr })->good_phrases_list->pack_list;
    my $phl = $self->proj->phrase_list({phrases_arr => \@arr })->pack_list;
    $self->proj->cdict_client->cache_wide_phrases($phl->phrases);

    $phl = $self->proj->phrase_list({phrases_list => [ grep { ! $_->is_wide_phrase } $phl->phrases ] });
    return $phl;
}

sub get_text_subphrases : CACHE {
    my ($self) = @_;

    my @arr = $self->_cleared_text;
    my @farr = ();
    for my $t (@arr){
        push( @farr, join( ' ', @{[ grep {$_ !~ /\bдля\b/} $self->proj->phrase($t)->normwords ]}[0 .. 2] ));
    }
    #Удаляем однословные английские слова, так как это могут быть вендеры
    @farr = grep { ! /^\s*[A-Za-z]+\s*$/ } @farr;

    my @marr = map { $self->proj->phrase($_)->_get_good_phrases } @arr;
    @arr = (@farr, @marr);
    
#    return map {$_->text} $self->proj->phrase_list({phrases_arr => \@arr })->pack_list->phrases;
    my $phl = $self->proj->phrase_list({phrases_arr => \@arr })->pack_list;
    $phl = $phl->good_phrases_list;
    return $phl;
}

sub get_prefiltered_line {
    my ($self) = @_;
    my $text = $self->text;
    return $self->proj->phrase('') unless $text;
    $text =~ s/\&quot\;/\"/g;
    $text =~ s/quot\;/\"/g;
    $text =~ s/\&nbsp\;//g;
    $text =~ s/nbsp\;//g;
    $text =~ s/\&amp\;/\&/g;
    $text =~ s/amp\;/\&/g;
    $text =~ s/(\d(?:x|х))(\/|\\)(\d)/$1$3/g; # 5x/18 на 5x18
    $text =~ s/[\/()\[\]\+\;\:\%\…\?\|\!\®\™\®\=\*\\]/ /g; # заменяем пробелами скобки, слэши и спецсимволы
    $text =~ s/([\d])\,([\d])/$1\.$2/g; # в числах заменяем запятую точкой
    $text =~ s/([-$RU$LA\d"]+)[.]( |"|$)/$1$2/g; # точку в конце слова заменяем пробелом
    $text =~ s/([$RU]+)[.]([$RU]+)/$1 $2/g; # точку между русскими буквами заменяем пробелом
    $text =~ s/[,_<>#]/ /g; # заменяем пробелами запятые и символы подчеркивания, знаки больше и меньше
    $text =~ s/([a-zа-яёA-ZА-ЯЁ])(\'|\`|\’)([a-zа-яё])/$1$3/g; # апострофы между буквами удаляем l'oreal -> loreal
    $text =~ s/[«»`]/"/g;
    $text =~ s/ [^\d"'&$RU$LA]+/ /g; # удаляем последовательности из мусорных символов, пока не трогаем &, кавычки и апострофы
    $text =~ s/\\n/ /g; # иногда встречается в текстовом виде символ перевода строки
    $text =~ s/->/-/g;
    $text =~ s/(-+[\s]|[\s]-+|-+$|^-+)/ /g; # удалить тире перед пробелом и после него, а также в начале и в конце строки
    $text =~ s/(-|\*)+(-|\*)+/ /g; # последовательность из нескольких тире заменяем пробелом
    $text =~ s/( [A-Z][a-zA-Z]+) & ([A-Z][a-zA-Z]+ )/$1&$2/g; # Procter & Gamble на Procter&Gamble   
    $text =~ s/(&+[\s]|[\s]&+|&+$|^&+)/ /g; # удалить & перед пробелом и после него, а также в начале и в конце строки
    $text =~ s/\s+/ /g; # несколько пробелов подряд заменяем одним
    $text =~ s/^\s+|\s+$//; # обрезаем начальные и конечные пробелы
        
    return $self->proj->phrase($text); #phrase instead of text
}

our $getnoisere = '';
sub getnoisere { $getnoisere ||= qr{((?<=^)|(?<=\s))($noisere)(?=(\s|$))}i; return $getnoisere;}

our $getaccbrandre = '';
sub getaccbrandre { $getaccbrandre ||= qr{(?:^|\s)($accbrandre)(?:\s|$)}i; return $getaccbrandre;}

our $getmetricalre = '';
sub getmetricalre { $getmetricalre ||= qr{(?:^|\s|\d+)($dict_metricalre)(?:\.|\s|\!|$)}; return $getmetricalre;}

our $getgoodsre = '';
sub getgoodsre { $getgoodsre ||= qr{(?:^|\s|\d+)($goodsre)(?:\s|$)}i; return $getgoodsre;}

our $getmodelsre = '';
sub getmodelsre { $getmodelsre ||= qr{\b($modeltypesre|$modeltypesre_ru)\b}; return $getmodelsre;}
our $getmodelsre_ru = '';
sub getmodelsre_ru { $getmodelsre_ru ||= qr{\b($modeltypesre_ru)\b}; return $getmodelsre_ru;}
#sub getmodelsre { $getmodelsre ||= qr{\b($modeltypesre)\b}; return $getmodelsre;}

# на входе хэш с параметрами
# pres - ссылка на хеш с распарсенной фразой (если не нужен парсинг)
# brand - только подфразы по брэнду
# model - только подфразы по модели
# modellike - с разваливанием по похожим моделям, если указан, то параметр по подфразам обязателен
# при вызове без параметров выдаются все подфразы без разваливания по моделям
sub get_subphrases {
    my ($self, %par) = @_;
#    print STDERR Dumper ( \%par );
    my %pres = ();
    %pres = %{$par{pres}} if $par{pres};
    my $subs_by_brand = $par{brand};
    my $subs_by_model = $par{model};
    my $all_subs = ($subs_by_brand or $subs_by_model) ? 0 : 1;
    my $modellike = $par{modellike} || 1;
    my $all_not_models = $par{all_not_models};

    my $text = $self->text;
    if ( $text && $text !~ / / ) {
        my $res = $self->proj->phrase( $text )->get_modellike_modifications( model => $text );
        return $res if $res;
        return $self->proj->phrase_list( { phrases_inf_text => lc($text) } );
    }
    my @subs = (); # массив подфраз
    my @subs_not_models = (); # для немоделей отдельный массив, чтобы отфильтровать их по частотам

    %pres = $self->proj->phrase($text)->parse unless %pres;
    #print STDERR Dumper (['pres:', \%pres]);

    if ($pres{class} eq "author-media") {
        my $surnames = $pres{surnames};
        my $quotes = $pres{quotes};
        my @surnames = $surnames ? (split /\s+/, $surnames) : ();
        # выкидываем из названия то, что меньше двух символов или предлог
        my @quotes = $quotes ? (grep { length($_) > 2 && $_!~/^для|как|над|под|изо|это|или|нет$/i } split /\s+/, $quotes) : ();
        # берем фамилий меньше на 1, чем длина фразы
        my @surnames_part = @surnames;
        @surnames_part = @surnames[0..$SUBPHR_LEN-1-1] if @surnames > ($SUBPHR_LEN-1-1);
        # для списка фамилий выкидываем по одному слову на каждой итерации
        while ( @surnames_part ) {
            # вычисляем длину остатка (название) так, чтобы он не превысил длину фразы
            my $quotes_len = $SUBPHR_LEN - @surnames_part;
            my @quotes_part = @quotes[0..$quotes_len-1];
            # из кавычек тоже выкидываем по одному слову
            while ( @quotes_part ) { 
                push @subs, make_subphr (\@surnames_part, \@quotes_part);
                pop @quotes_part;
            }
            pop @surnames_part;
        }
        # просто название без автора
        while (@quotes) {
            push @subs, make_subphr (\@quotes) if @quotes > 1;
            pop @quotes;
        }
    }    

    if ($pres{class} ne "author-media") {
        my ($type, $core, $model, $quotes, $brand) = ($pres{type}, $pres{core}, $pres{model}, $pres{quotes}, $pres{brand});
        $brand = "" if ($core eq $brand || $type eq $brand);
        if ( $pres{class} eq "undefined" && !$pres{model} && $pres{src} ){
            $model = $pres{src};
            $model =~ s/$pres{brand}// if $pres{brand};
            my @model = split /\s+/, $model;
            @model = @model[0..3] if @model > 4;
            $model = "@model";
        }

        # всегда кладем тип и ядро (по просьбе Маркета)
        if ($all_subs || !%par) {
            #push @subs_not_models, $core if $core;
            #push @subs_not_models, $type if ($type =~ /\s/ && $type ne $core);
            push @subs_not_models, $type if ( $type =~ /\s/ );
        }
        # разваливаем модель на много похожих моделей, если требуется
        my @modellikes = (); # массив с моделями
        if ( $model && $modellike ) {
            my @modellike_phrases = $self->proj->phrase($model)->get_modellike_modifications( model=>$model )->phrases;
            for my $phr (@modellike_phrases) {
                push @modellikes, $phr->text; # каждую модель добавляем в массив
            }
            #print STDERR Dumper ( \@modellikes );
        }

        $model = "*" unless $model; # чтобы произошло выделение подфраз, нужно попасть в цикл по моделям, * отбросится при проверках

        push @modellikes, $model; # основную модель добавляем всегда

        my @type = split " ", $type; 
        my @core = split " ", $core; 
        my @quotes = split " ", $quotes;
        my @brand = split " ", $brand;

        if (    $pres{class} eq "accessory" 
            &&  ($all_subs || !%par)
            &&  $pres{type}
            &&  ($pres{brand_for} || $pres{model_for})
           ) {
            my $cbm = $pres{type};
            $cbm .= ' '.$pres{brand_for}.' '.$pres{model_for} if ( scalar( @{[split /\s+/, $pres{type}]} ) <=2 && $pres{brand_for} && $pres{model_for});
            my @cbm = split /\s+/, $cbm;
            push @subs, "@cbm" if @cbm;

            if ($pres{brand_for}) {
                push @subs, $pres{type}.' '.$pres{brand_for};
            }

            if ($pres{model_for}) {
                push @subs, $pres{type}.' '.$pres{model_for};
            }
        }

        #print STDERR Dumper (\@subs);
        #print STDERR Dumper (\@modellikes);

        for my $ml ( @modellikes ) {
            my @ml = split /\s+/, $ml;
            while (@ml) {
                 my $model = join " ", @ml;
                 my $num = $self->proj->phrase($model)->get_model_wc; # определяем словность модели
                
                 my @temp = ();

                 # это сочетание включаем всегда
                 my @cbm = ();
                 for ( ($type, $brand, $model) ) {
                     push @cbm, $_ if $_;
                 }
                 push @subs, "@cbm" if (@cbm && "@cbm" =~ /\s/);

                 #print STDERR Dumper (\@cbm);

                 if ($num > 1) { # неоднословные модели сразу в @subs
                     push @subs, $model if ( $model =~ /\d/ and ($subs_by_model || $all_subs || !%par) );
                 } else { # смешиваем различные массивы и кладем в @subs
                     push @subs, $model if ( $model =~ /\d/ && $model =~ /\w/ && length($model) > 4 );
                     if ($subs_by_brand || $all_subs || !%par) {
                        push @subs_not_models,
                            make_subphr(\@type, \@brand),
                     }
                     if ($subs_by_model || $all_subs || !%par) {
                        push @subs,
                            make_subphr(\@brand, \@ml),             
                            make_subphr(\@brand, \@quotes, \@ml),
                            make_subphr(\@type, \@ml),
                            make_subphr(\@type, \@quotes, \@ml),
                        push @subs,
                            make_subphr(\@quotes, \@ml) if "@ml" ne '*'; # отдельно кавычки нельзя делать подфразой, теряется смысл
                     }
                     if ($all_subs || !%par) {
                        push @subs_not_models,
                            make_subphr(\@type, \@brand, \@quotes),
                            make_subphr(\@type, \@quotes),
                     }
                 }
                 pop @ml; # выкидываем слово, уменьшаем модель
            }
        }
    }
=h
    print STDERR "..................\n";
    print STDERR Dumper (\@subs, \@subs_not_models);
    print STDERR "..................\n";
=cut
    # если фраза без модели не слишком частотная, берем ее, частотные - отбрасываем
    my $phl_not_models = $self->proj->phrase_list( { phrases_arr=>\@subs_not_models } )->cache_search_count;
    for ( $phl_not_models->phrases ) {
        my $phr = $_->text;
        $phr =~ s/-/ /g; # cdict не работает с дефисами
        my $sc = $self->proj->phrase($phr)->get_search_count;
        #print STDERR $_->text, ' : ', $sc, "\n";
        push @subs, $_->text if ( $sc && $sc < 10000 );
    }        
    #print STDERR Dumper (\@subs);

    my $subs_brushed = brush_subs_array ($self, \@subs, \%pres); 
#=h
#    print STDERR $"=" / "; print STDERR "@subs\n"; $"=" ";    
#    print STDERR $self->text,"\n";
#    print STDERR Dumper(\%pres);
#    print STDERR Dumper ($subs_brushed);
#    print STDERR "-----------------------------------------\n";
#=cut
    return $self->proj->phrase_list( { phrases_arr=>$subs_brushed } );
}

sub get_subphrases_with_attributes {
    my ($self, %par) = @_;
    my %pres = ();
    %pres = %{$par{pres}} if $par{pres};
    my $modellike = $par{modellike};
    my $attributes = $par{attributes};
    return $self->proj->phrase_list( { phrases_arr=>[()] } ) unless $attributes;

    my $text = $self->text;

    if ( $text && $text !~ / / ) {
        return $self->proj->phrase_list( { phrases_inf_text => lc($text) } );
    }

    %pres = $self->proj->phrase($text)->parse unless %pres;

    my ($type, $model, $quotes, $brand) = ($pres{type}, $pres{model}, $pres{quotes}, $pres{brand});
    $brand = "" if ($type eq $brand);

    my @modellikes = (); # массив с моделями
    if ( $model && $modellike ) { # разваливаем модель на много похожих моделей, если требуется
        my @modellike_phrases = $self->proj->phrase($model)->get_modellike_modifications( model=>$model )->phrases;
        for my $phr (@modellike_phrases) {
            push @modellikes, $phr->text; # каждую модель добавляем в массив
        }
    }

    $model = "*" unless $model; # чтобы произошло выделение подфраз, нужно попасть в цикл по моделям, * отбросится при проверках
    push @modellikes, $model; # основную модель добавляем всегда

    my @type = split " ", $type; 
    my @quotes = split " ", $quotes;
    my @brand = split " ", $brand;

    # конструируем подфразы для признаков
    my @subs_attributes = ();
    my $subs_attributes_brushed = (); # причесанные подфразы

    for my $ml ( @modellikes ) {
        my @ml = split " ", $ml;
        next if $ml =~ /^(\d{1,3}|\w{,2})$/i;
        while ( @ml ) {
            last if "@ml"=~ /^(\d{1,3}|\w{,2})$/i;
            #print STDERR ">@ml\n";
            for my $attribute ( @$attributes ) {
               my @attribute = split /\s+/, lc( $attribute );
               #print STDERR Dumper (\@subs_attributes);
               push @subs_attributes,
                   make_subphr({invert=>1, remove_neigh_dups=>1}, \@brand, \@ml, \@attribute),             
                   make_subphr({invert=>1, remove_neigh_dups=>1}, \@quotes, \@ml,\@attribute),
                   make_subphr({invert=>1, remove_neigh_dups=>1}, \@type, \@ml, \@attribute),
                   make_subphr({invert=>1, remove_neigh_dups=>1}, \@ml, \@attribute);
                #print STDERR "<$attribute>\n";
                #print STDERR Dumper (\@subs_attributes);
                #print STDERR "***\n";
                # нужно взять те, у которых перед атрибутом либо 4 цифры, либо не менее четырех символов (между которыми могут быть пробелы) но обязятельно есть хотя бы одна буква
                @subs_attributes = grep { ##no critic
                                          my $res = 1;
                                          if ( s/\s+$attribute(\s+|$)// ) {
                                            my @t = $_=~ /[^\s]/g; 
                                            my $res = $_=~ /^\d{4,}$/ || @t > 2 && /[a-z]/i; 
                                            $_ .= " $attribute";
                                          }
                                          $res;  
                                        } @subs_attributes;
                #print STDERR Dumper (\@subs_attributes);
                #print STDERR "...\n";
            }
            pop @ml;
        }
    }

    @subs_attributes = grep { ! /^(\d{1,3}|\w)\s+/i } @subs_attributes;
    $subs_attributes_brushed = brush_subs_array ($self, \@subs_attributes, \%pres);

    return $self->proj->phrase_list( { phrases_arr => $subs_attributes_brushed } );
}

sub brush_subs_array {
    my ($self, $arr, $pres) = @_;
    my @subs = ();

    # удаляем звездочки, если есть
    @subs = map { $_ =~ s/ \*$//r } @$arr;

    # мусор на границах фразы (TODO) по идее его не должно быть
    @subs = map { $_ =~  s/(^[^$RU$LA^0-9]+(?=[$RU$LA^0-9])|(?<=[$RU$LA^0-9])[^$RU$LA^0-9]+$)//; $_ } @subs; ##no critic

    # удаляем дубли из фразы
    @subs = map { $_ ? $self->proj->phrase($_)->fix_dup_words : () } @subs;

    # удаляем дубликаты из массива фраз
    my %c;
    @subs = grep { ! $c{$_}++; } @subs;

    #print STDERR Dumper (\@subs);

    # TODO: разобраться, нужен ли этот код
    # отбрасываем низкочастотные фразы -- увеличивает скорость, но уменьшает кол-во получаемых фраз
#    my $phl = $self->proj->phrase_list({ phrases_arr => \@subs});
#    $phl->cache_search_count;
#    @subs = map{$_->text} grep { $_->get_search_count } @$phl;

    # отбрасываем широкие фразы
    my @curr_phrases = map{$self->proj->phrase($_)} @subs;
    $self->proj->cdict_client->cache_wide_phrases(@curr_phrases);
    @subs = map{$_->text} grep { ! $_->is_wide_phrase } @curr_phrases;

    #print STDERR Dumper (\@subs);

    # отбрасываем то, что испортит нормализатор
    @subs = grep {
        my @sub = split " ", $_;
        my @nsub = $self->proj->phrase($_)->normwords;
        @sub <= @nsub || @sub > 2; # если это двухсловник и н-р съест слово, такую фразу не берем
    } @subs;

    # если часть подфразы после брэнда широкая, отбрасываем
    my @to_check = ();
    for my $phr (@subs) {
        #my $brand = ( $self->proj->phrase($phr)->match_itemdict(\%dict_brands, item_maxsize=>4, first_only=>1) )[0] || "";
        my $brand = $self->get_brand;
        my $bank = $self->get_bank;
        my $without = $phr;
        $without =~ s/($brand | $brand$)// if $brand;
        $without =~ s/($bank | $bank$)// if $bank;
        push @to_check, [$phr, $brand, $self->proj->phrase($without)];
    }
    $self->proj->cdict_client->cache_wide_phrases(map{$_->[2]} @to_check);
    @subs = map{$_->[0]} grep{!$_->[1] || !$_->[2]->is_wide_phrase} @to_check;

    # то же, но без cdict:
#    @subs = grep {
#        my $phr = $_;
#        #my $brand = $self->proj->phrase($phr)->get_brand;
#        my $brand = ( $self->proj->phrase($phr)->match_itemdict(\%dict_brands, item_maxsize=>4, first_only=>1) )[0] || "";
#        $phr =~ s/($brand | $brand$)// if $brand;
#        !$brand || !$self->proj->phrase($phr)->is_wide_phrase;
#    } @subs;

    #print STDERR Dumper (\@subs);

    # если висяшие символы после после core или type, отбрасываем (электрообогреватель t, тепловентилятор 15)
    @subs = grep {
        my $res = 1;
        my @sub = split " ", $_;
        if ($$pres{core}) {
            my $phr = $_;
            $phr =~ s/$$pres{core}//i;
            $res = 0 unless ( @sub > 2 || (length ($phr) > 2  && $phr !~ /^[\d]+$/) || !$phr );
        }    
        if ($$pres{type}) {
            my $phr = $_;
            $phr =~ s/$$pres{type}//i;
            $res = 0 unless ( @sub > 2 || (length ($phr) > 2  && $phr !~ /^[\d]+$/) || !$phr);
        }    
        $res;
    } @subs;  

    #print STDERR Dumper (\@subs);
    return \@subs;
}

# склеивает массивы в подфразу не длиннее 4 слов
# необязательный параметр {invert=>1} передается первым и служит для того, чтобы при обрезании по длине подфразы лишние слова отбрасывались с начала, а не с конца
# параметр нужен при генерации подфраз с признаками, где признак доклеивается в конец, но обрезаться при этом не должен
sub make_subphr {
    my $par = {};
    if (ref($_[0]) eq 'HASH') {
        $par = shift @_;
    }
    my $res = '';
    my @temp = ();
    my %temp = ();

    #print STDERR Dumper(\@_);
    my $has_empty = grep !@{$_}, @_;
    return if $has_empty; # выходим, если по ссылке передан хотя бы 1 пустой массив
    push @temp, @{$_} for (@_); # все слова в один массив
    # убираем дубликаты
    unless ( $par->{remove_neigh_dups} ) {
        my %cnt;
        @temp = grep { defined($_) && !$cnt{$_}++; } @temp;
    }
    else { 
        my $prev = ''; # только соседние
        @temp = grep { $res = $_ ne $prev; $prev = $_; $res } @temp;
    }

    @temp = reverse @temp if $par->{invert}; # переворачиваем, чтобы конец оказался в начале
    @temp = @temp[0..$SUBPHR_LEN-1] if @temp > $SUBPHR_LEN; # обрезаем по длине подфразы
    @temp = reverse @temp if $par->{invert}; # переворачиваем обратно
    $res = join " ", @temp if @temp; # строку помещаем в подфразы
    my $last_word = pop @temp;
    $res = "@temp" if stop4norm( $last_word ); 
    return $res;
}


# оставляем только буквы, цифры, слеши, тире и кавычки
sub clear_text {
    my ( $self, $txt ) = @_;
    $txt =~ s/\s(for|для)$//i;
    my $valid_symbols = $self->proj->_bannerland_allowed_naming_chars;
    my @incorrect_parts = $txt =~ /([^$valid_symbols]+)/gi;
    for ( @incorrect_parts ){
        $_ = quotemeta($_);
        $txt =~ s/$_/ /i;
    }
    $txt =~ s/ +/ /g; # несколько пробелов подряд заменяем одним
    $txt =~ s/^ +| +$//g; # обрезаем начальные и конечные пробелы
    return $txt;
}


sub filter_and_check_model {
    my ( $self, $parsed ) = @_;
    my @local_wides = map { s/_/ /g; $_ } @local_wides; ##no critic

    my %res = %$parsed;
    return %res unless $self->text;

    for (0..1) {
        my $brand = $_ ? 'brand_for': 'brand';
        my $model = $_ ? 'model_for': 'model';
        next unless $res{$model};
        my $wc = scalar @{[ split /\s+/, $res{$model} ]} || 0;
        my $dc = scalar @{[ $res{$model} =~ /[\d]/g ]} || 0;
        my $lc = scalar @{[ $res{$model} =~ /[a-zа-яё]/gi]} || 0;

        my @model = grep { $_ } split /\s+/, $res{$model};
        @model = @model[0..3] if @model > 4;
        @model = grep { $_ && ( $_ !~ /(?:^|\s)[0-9]{1,2}-(?:х|x)(?:\s|$)/ ) } map { s/(?:^[^a-zа-яё0-9]+|[^a-zа-яё0-9]+$)//gi; $_ || ()} @model; ##no critic
        @model = grep { $_ && ( $_ !~ /(?:^|\s)[0-9]{1,2}-(?:х|x)(?:\s|$)/ ) } map { s/(?:^[^a-zа-яё0-9]+|[^a-zа-яё0-9]+$)//gi; $_ || ()} @model; ##no critic
        $res{$model} = join (' ', @model);

        my $valid = 0;
        $valid = 1 if $res{$model} =~ /(^|\s)iphone([^a-z]|$)/i;
        $valid = 1 if ( $res{$brand} && $dc && $lc && ($dc+$lc-$wc)>1 );
        $valid = 1 if ( $res{$brand} && defined($res{$model}) && $res{$model} =~ /^[А-ЯЁ]+[-а-яё]+|[A-Z]+[-a-z]+/ && $self->text =~ /$res{$brand}\s+$res{$model}/i );
        $valid = 1 if ( $res{brand} && $res{$model} =~ /^[1-9]{3,}$/ );
        # если не похоже на модель, удаляем ее
        $res{$model} = '' if (  (   !$res{$brand} && (    $res{$model} !~ /[a-zа-яё]/i 
                                                            || @{[$res{$model} =~ /[0-9]/g]} < 2 && $wc == 1 && $res{$model} !~ /\-/
                                                            || $res{$model} !~ /[0-9]/ && $wc > 1
                                                            || $res{$model} =~ /^(\d+\s+[А-ЯЁ]?[-а-яё]+|[А-ЯЁ]?[-а-яё]+\s+\d+)$/ 
                                                            || $res{$model} =~ /^(\d+[-.:]?[A-ZА-ЯЁ]?[-a-zа-яё]+)(\s+\d+[-.:]?[A-ZА-ЯЁ]?[-a-zа-яё]+)?$/
                                                            #|| $res{$model} =~ /^\d?\d\.\d\d(\s*\-\s*\d?\d\.\d\d)?$/ 
                                                            ) 
                                     || $res{$brand} && $res{$model} =~ /[а-яё]/i && $res{$model} !~ /[0-9]/
                                     || $res{$model} =~ /^\d+(?:x|х)\d+$/
                                     || length($model) < 2 
                                     || $res{$model} =~ /^[\d.,-]+$/ && $dc < 3
                                     || $res{$model} =~ /\.(ru|su|info|com)(\s|$)/i ) 
                                && !$valid
                            ); 
        $res{$model} = '' if defined($res{$model}) && ( not grep { ! in_array($_, \@local_wides)  } split/\s+/, lc $res{$model} );
    }
    # вырезаем цвета
#    ( $res{colors}, $res{model} ) = $self->proj->phrase($res{model})->cut_colors if $res{model};
    return %res;
}

sub parse_sentences {
    my ( $self ) = @_;
    return () unless $self->text;
#    print STDERR ">>>".$self->text."\n";
    my @sentences = split /(?:\.|\!|\?)(?: |\.|\!|\?)*/, $self->text;
    #print Dumper (\@sentences);

    my %sentence_parse_hash = ();

    #парсим, ищем модель
    for my $sentence ( @sentences ) {
        my %h = ();
        eval { %h = $self->proj->phrase( $sentence )->parse; };
        return %h if (  $h{model} or ($h{class} and $h{class} eq 'accessory' and $h{model_for}) );
        $sentence_parse_hash{$sentence} = \%h;
    }

    #если при парсинге не нашли модель, пытаемся отдать бренд
    for my $sentence ( @sentences ) {
        my %h = %{$sentence_parse_hash{$sentence}};
        return %h if ( $h{brand} or ($h{class} and $h{class} eq 'accessory' and $h{brand_for}) );
    }

    return ();
}
        

sub hasnt_model {
    my ($self) = @_;
    my $dict_notmodels = $self->proj->phrase->dict_notmodels;
    $phl_notmodels = $self->proj->phrase_list({phrases_arr=>[keys %$dict_notmodels]}) unless $phl_notmodels;
    my $phl_notmodels_found = $phl_notmodels->search_subphrases_in_phrase($self, only_norm => 1);
#    print Dumper ( $phl_notmodels_found->perl_array );
    return 1 if $phl_notmodels_found;
    return 0;
}

# для фразы, содержащей товарное наименование, возвращает хеш с наименованием товара
# core, однословное наименование
# type, многословное наименование
# type2, длинное многословное наименование
# prep, предлог
# clean=>1 - c предварительной очисткой фразы
sub get_goods_basis {
    my ($self, %p) = @_;
    my $text = $p{clean} ? $self->get_prefiltered_line->text : $self->text;
    return '' unless $text;
    $text = lc($text);
    my @keys = qw /core type type2/;
    my %res = ();
    $res{$_} = "" for @keys;
    my $prep = "";
    if ( $text =~ /(?:^|[^\w])($acc_preps_re)(?:[^\w]|$)/ ) {
        $prep = $1;
    }
    $res{prep} = $prep;
    return %res unless $text =~ /[$RU]{3,}/;

    my @before_prep = ();
    my $after_prep;

    #my %gi = $self->proj->phrase($text)->get_grammar_inf;
    #print STDERR "$text\n";
    #print STDERR Dumper (%gi);
    my @phr = split /\s+/, $text;
    my $prep_found = 0;

    my %gi_hash = ();

    for my $i ( 0..$#phr ) {
        my $word = lc($phr[$i]);
        next if ( $word !~ /[$RU]/ || $word =~ /[0-9]{2,}/ ); # отбрасываем нерусские слова и если больше 2-х цифр

        $gi_hash{$word} = { $self->proj->phrase($word)->get_grammar_inf };
    }

    for my $i ( 0..$#phr ) {
        my $word = lc($phr[$i]);
        next if ( $word !~ /[$RU]/ || $word =~ /[0-9]{2,}/ ); # отбрасываем нерусские слова и если больше 2-х цифр

        if ( $word eq $prep ) { # ищем предлог
            $prep_found = 1;
            next;
        }

        push @before_prep, $word if ( $prep && !$prep_found ); # собираем русские слова до предлога

        my %par = (); # внутренний хэш параметров, чтобы не плодить кучу переменных
        my %gi = %{ $gi_hash{$word} };
        my @posp = keys %{$gi{$word}{parts}};
        my @case = keys %{$gi{$word}{cases}};
        $par{is_noun} = !grep { $_ ne 'S'} @posp if @posp; # ставим признак существительного
        #$par{is_noun} = grep { $_ eq 'S'} @posp if @posp; # ставим признак существительного
        $par{is_nom} =  grep { $_ eq 'nom'} @case if @case; # ставим признак именительного падежа
        #print STDERR "$word\n"; print STDERR Dumper (\@posp); print STDERR Dumper(\@case);

        # ищем core - до предлога, первое существительное в именительном падеже
        if ( ($prep && !$prep_found || !$prep) && !$res{core} && $par{is_noun} && $par{is_nom} ) {
            $res{core} = $word;
            $res{type} = $word;
            # пытаемся приклеить префикс, если нужно (LED телевизор)
            my $pre = $i ? $phr[$i-1] : "";
            $pre = "" unless ($pre && $pre !~ /[$RU]/ && $pre !~ /\d.*\d/ && length($pre) > 1);
            $pre = "" if ( $dict_brands{lc($pre)} || $dict_banks{lc($pre)} ); # префиском не могут быть брэнды
            if ($pre) {
                $res{core} = $pre.' '.$word; # итоговое ядро
                $before_prep[$#before_prep] = $res{core} if $prep; # не забываем проапдейтить массив before_prep, в нем нет иностранных слов
            }
            # пытаемся найти прилагательное до ядра
            $par{adj1} = $phr[$i-1] if ($i>=1 && $phr[$i-1] =~ /$adj$/ && !$pre); # предыдущее слово, если нет префикса
            $par{adj1} = $phr[$i-2] if ($i>=2 && $phr[$i-2] =~ /$adj$/ && $pre); # через одно слово, если есть префикс
            $res{type} = $res{core}.' '.$par{adj1} if $par{adj1};
            # пытаемся найти прилагательное после ядра
            my $wordnext;
            $wordnext = $phr[$i+1] if ( $i < $#phr && !($phr[$i+1] !~ /[$RU]/ || $phr[$i+1] =~ /[0-9]{2,}/) );
            #print STDERR ">>$wordnext\n";
            $par{adj2} = $wordnext if ($wordnext && $wordnext =~ /$adj$/);
            $res{type} = $res{type}.' '.$par{adj2} if $par{adj2};

            # если не найдено прилагательное после, ищем существительное в род.падеже (блок питания)
            if ($wordnext && !$par{adj2}) {                    
                my %gi = %{ $gi_hash{$wordnext} };
                my @posp = keys %{$gi{$wordnext}{parts}};
                my @case = keys %{$gi{$wordnext}{cases}};
                $par{next_is_noun} = !grep { $_ ne 'S'} @posp if @posp;
                $par{next_is_gen} =  grep { $_ eq 'gen'} @case if @case;

                $par{noun_gen} = $wordnext if ($par{next_is_noun} && $par{next_is_gen} && ($wordnext ne $prep));
            }

            # если найдено прилагательное после, ищем существительное в род.падеже (фотографии циркулярной пилы)
            my $wordnextnext;
            $wordnextnext = $phr[$i+2] if ( $i < $#phr-1 && !($phr[$i+2] !~ /[$RU]/ || $phr[$i+2] =~ /[0-9]{2,}/) );
            if ($wordnextnext && $par{adj2}) {
                my %gi = %{ $gi_hash{$wordnextnext} };
                my @posp = keys %{$gi{$wordnextnext}{parts}};
                my @case = keys %{$gi{$wordnextnext}{cases}};
                $par{nextnext_is_noun} = !grep { $_ ne 'S'} @posp if @posp;
                $par{nextnext_is_gen} =  grep { $_ eq 'gen'} @case if @case;

                $par{noun_gen} = $wordnextnext if ($par{nextnext_is_noun} && $par{nextnext_is_gen} && ($wordnextnext ne $prep));
            }
            $res{type} = $res{type}.' '.$par{noun_gen} if $par{noun_gen};
        }

        # берем существительное в род. падеже после предлога (смеситель для ванной)
        $par{is_gen} =  grep { $_ eq 'gen'} @case if @case;
        #print STDERR "$word\n", Dumper (\@case), Dumper (\@posp);
        $after_prep = $word if (@before_prep && !$after_prep && $prep_found && $par{is_noun} && $par{is_gen});
        #print STDERR Dumper(\%par);
    }
    # большой многословный тип
    $res{type2}  = "@before_prep" if @before_prep;
    $res{type2} = "@before_prep $prep ".$after_prep if $after_prep;

    $res{type} = "" if ( $res{type} && length($res{type}) < 3 );
    $res{type2} = "" if ( $res{type2} && length($res{type2}) < 3 );

    $res{$_} = lc($res{$_}) for (keys %res);
    #print STDERR Dumper(\%res);
    return %res;
}

sub get_grammar_inf {
    my ( $self ) = @_;
    return '' unless $self->text;
    my $phr = $self->text;
    my %res = ();
    for my $word ( split / /, $phr ) {
        # если слово с дефисом, берем грамматику второй части
        my @word_arr = split /-/, $word;
        my $true_part = $word_arr[1] || $word_arr[0];
        #print STDERR $word,'--', Dumper ($dict_wordforms{$word}), "\n";
        $res{$word} = $dict_wordforms{$true_part} if exists $dict_wordforms{$true_part};
        unless ( $res{$word} ) {
            my %li = $self->proj->phrase($true_part)->get_lemmer_inf;
            #print STDERR $word, '==', Dumper(\%li), "\n";
            $res{$word} = $li{$true_part} if exists $li{$true_part};
        }
    }
    return %res;
}

# выделяет брэнд и модель
sub parse_fast : CACHE {
    my ($self) = @_;
    #my $text = $self->text;
    my $text = $self->text;
    my $re = getmodelsre;

    my $brand = ($self->match_itemdict(\%dict_brands, item_maxsize=>3, first_only=>1))[0] || "";
    my $model = "";
    if ( $brand && ($text =~ /$brand(?:\s+\([^\)]+\))?\s+($re)/)) {
        $model = $1;
    }
    $text =~ s/\b$brand\b// if $brand;
    #print STDERR "---$model\n";
    # если не похоже на модель, возвращаем пустоту
    $model = "" unless $model =~ /[A-ZА-ЯЁ0-9]/;
    return ($brand, $model);
}

# выделяет брэнд и модель для автомобилей
sub parse_fast_auto : CACHE {
    my ($self) = @_;
    my $re = getmodelsre;

    # загрузка словаря автомобильных брендов при первом вызове функции
    unless(%dict_auto_brands) {
        my $dict_filename = $Utils::Common::options->{dict_auto_brands};
        if($dict_filename) {
            open(F, $dict_filename) or die "open failed ($!)";
            while (<F>) {
                chomp;
                s/\s+/ /g;
                s/(^\s+|\s+$)//g;
                s/\s*\@.*$//;
                $dict_auto_brands{lc($_)}++ for ( split /\s*\,\s*/ );
            }
            close(F) or die "close failed ($!)";
        }
    }

    my $brand = ($self->match_itemdict(\%dict_auto_brands, item_maxsize=>3, first_only=>1))[0] || "";
    my $model = "";
    my $text = $self->text;
    $model = $1 if ( $brand && ($text =~ /$brand(?:\s+\([^\)]+\))?\s+($re)/));
    $model = "" unless $model =~ /[A-ZА-ЯЁ0-9]/;
    return ($brand, $model);
}

# выделяет брэнд аксессуара из строки
sub old_get_acc_brand {
    my ($self) = @_;
    my $text = $self->text;
    my $re = getaccbrandre;
    return $1 if $text =~ /$re/i;
    return '';
}

# выделяет единицы измерения из строки
sub get_metrical {
    my ($self) = @_;
    my @res = ();
    my $text = $self->text;
    my $re = getmetricalre;
    @res = $text =~ /$re/gi;
    return @res;
}

# возвращает массив содерживого кавычек
# случай 1 в случае вложенных кавычек возвращает только первый уровень
# случай 2 в случае нескольких последовательностей с кавычками возвращает их массив
sub get_quotes {
    my ($self, $QUO) = @_;
    $QUO = '"' unless $QUO; # по умолчанию - двойные кавычки

    my $text = $self->text; 
    $text =~ s/\'\'|\`\`/\"/g; # двойные апострофы заменяем кавычками
    $text =~ s/[«»”'`“]/\"/g; # все остальные апострофы заменяем кавычками
    $text =~ s/(?<=\")[,.;:]/ /g; # удаляем знаки препинания после кавычек
    $text =~ s/\.\"/\"/g; # удаляем точку перед кавычками
    $text =~ s/([a-zа-яё])(\")([a-zа-яё])/$1\'$3/g; # кавычки между буквами заменяем апострофом
    #print STDERR "---->", $self->text, "\n";
    my @a = split //, $text; # строка
    my @b = (); # содержимое кавычек
    my @r = (); # остаток фразы без кавычек
    my $n = 0; # признак сохранения содержимого
    my $c = 0; # счетчик кавычек
    my @res = ();
    for my $i (0..$#a) {
        if ($a[$i] eq $QUO) {
            $c++;
            if ($c == 1) { # начало содержимого
                $n = 1;
            } elsif ($c % 2 == 0) {
                if ($a[$i-1] !~ /[ (]/ && @b > 0) { # конец содержимого
                    #print join("", @b), "\n";
                    push @res, join("", @b);
                    @b = ();
                    $n = 0;
                    $c = 0;
                } else {
                    push @b, $QUO;
                }
            } else {
                push @b, $QUO;
            }
        } else {
            push @b, $a[$i] if $n == 1;
            push @r, $a[$i] if $n == 0;
        }
    }
#    print STDERR Dumper (\@res);
    my $res_quotes = join ":", @res;
    $" = '';
    my $res_rest = $c == 1 ? "@r@b" : "@r";
    $" = ' ';
    #print STDERR "$res_rest\n";
    return ($res_quotes, $res_rest);
}

our %qty_unit_regex_subs;
sub qty_unit_regex_sub {
    my ($self, $bnd, $qty_unit) = @_;
    my $key = "$bnd $qty_unit";
    unless (exists $qty_unit_regex_subs{$key}) {
        sub quoteslash {
            my $text = shift;
            return $text =~ s/\//\\\//gr;
        }
        my $str = 'sub { my $text = shift; return $text =~ /(?:'.quoteslash($bnd).')((?:'.quoteslash($qty_unit).')(?:[\s]?[*xх][\s]?'.quoteslash($qty_unit).')*)/ig }';
        $qty_unit_regex_subs{$key} = eval "$str";
        die $@ if $@;
    }
    return $qty_unit_regex_subs{$key};
}

# находит все паттерны количество\s*единца_измерения, возвращает их массив и строку БЕЗ них
# удаляет также квадратные и круглые скобки, если паттерн входил в них (размеры 5х5мм)
# ЕИ - case-sensitive
sub cut_qty_base {
    my ($self, $metrical, $currency) = @_;
    $currency ||= 0;
    $metrical ||= [];
    my $text = $self->text;
    my @qty = ();
    my @qty_currency = ();
    my $res = "";
#    print STDERR "Input of cut_qty: $text\n";
#    print STDERR Dumper ( ['MTR',$metrical] );

    my $metricalre = @$metrical ? join("|", sort { length $b <=> length $a } @$metrical) : $dict_metricalre;

#kostyl' technology (=
#    my $slash_constr = "\[\\s\\ \\/\]\+";
#    $metricalre =~ s/\//$slash_constr/gi;

    my $metrical_nospc_re = "gb|GB|Gb";

    my $bnd = '(?:[^\w\d\.\-]|^|$)'; # граница слова для ", \b не подходит, т.к. " сами явл. границей слова
    my $num = '(?:[-+]?[\d]+[, .]?[\d]+|[\d])'; # число, простое или с десятичной частью или цифра, допускаются пробелы между разрядами
    my $num_or_range = "$num(?:[ ]?[–*\/xх-]?[ ]?$num)*"; # число или диапазон чисел, с различными разделителями
    my $num_nospc = '(?:[-+]?[\d]+[,.]?[\d]+|[\d])'; # число, простое или с десятичной частью или цифра, НЕ допускаются пробелы между разрядами
    my $num_or_range_nospc = "$num_nospc(?:[–*\/xх-]?$num_nospc)*"; # число или диапазон чисел, с различными разделителями
    if ( $text =~ /\b$num_or_range\s*(?:$metrical_nospc_re)\b/ ){
        $num = $num_nospc;
        $num_or_range = $num_or_range_nospc;
    }
    my $qty_unit = "\\b$num_or_range\\s?(?:$metricalre)\\b"; # число или диапазон + еи
#    print STDERR "$qty_unit\n";
    my $marker = '<=====>'; # сначала заменим на маркер
    if ($currency){# обрабатываем символы валюты
        my $metricalre_currency = join("|", sort { length $b <=> length $a } @$currency );
        my $qty_unit_currency = "$num_or_range\\s?(?:$metricalre_currency)";
        my $qty_currency;
        @qty_currency = $text =~ /(?:$bnd)((?:$qty_unit_currency)(?:[\s]?[*xх][\s]?$qty_unit_currency)*)/g;
        if (@qty_currency){
            @qty_currency = map { quotemeta $_ } @qty_currency;
            $qty_currency =  join "|", @qty_currency;
            $text =~ s/($qty_currency)/$marker/g
        }
    }
    # собираем в массив паттерны количество-еи
    @qty = $self->qty_unit_regex_sub($bnd,$qty_unit)->($text); # числа и диапазоны + еи тоже могут повторяться
#    print STDERR Dumper (['qty',\@qty]);

    # ...дюймы, проценты - отдельно
    my @inches = $text =~ /$bnd($num_or_range[\s]?(?:\"|\%|\*))$bnd/g;
    my @sinches = @inches;
    s/([\%\*\"\+])/\\$1/g for @sinches;

    # если были звездочки, экранируем их для будущих замен
    @qty = (@qty, @qty_currency);
    my @sqty;
    @sqty = map { my $t = $_; $t =~ s/(\*|\$)/\\$1/g; $t } @qty if @qty; ##no critic
   # my @sqty = map { my $t = $_; $t =~ s/(\$)/\\$1/g; $t } @qty if @qty;

    #my @sinches = map { my $t = $_; $t =~ s/(\*|\$)/\\$1/g; $t } @inches if @inches;
    #my $inches = join "|", @sinches if @sinches;
    my $inches = join "|", @sinches;
    my $qty;
    $qty = join "|", @sqty if @sqty;

    # заменяем паттерны маркером
    #print "[$qty] | [$inches] [$text]\n";
    $text =~ s/\b($qty)\b/$marker/g if $qty;
    #$text =~ s/\b($inches)(?=$bnd)/$marker/g if $inches;
    $text =~ s/((?<=[^\w\d-])|(?<=^))(?:$inches)(?=$bnd)/$marker/g if $inches;
    
    # теперь выделяем паттерны 5х20х3.6 без ЕИ
    my @xqty = $text =~ /\b(\d[\d.,]*(?:(?:x|х)\d[\d.,]*){1,})\b/g;
    if (@xqty) {
        my $xqty = join "|", @xqty;
        $text =~ s/\b($xqty)\b/$marker/g;
    }

    # если была замена внутри скобок, такие скобки полностью удаляем
    my @brackets = ();
    if ( @brackets = $text =~ /(\([^)]*$marker[^)]*\))/g ) {
        #@brackets = map { s#\(#\\(#g; $_ } @brackets;
        #@brackets = map { s#\)#\\)#g; $_ } @brackets;
    }    
    if ( my @sbrackets = $text =~ /(\[[^\[]*$marker[^\]]*\])/g ) {
        #@sbrackets = map { s#\[#\\[#g; $_ } @sbrackets;
        #@sbrackets = map { s#\]#\\]#g; $_ } @sbrackets;
        push @brackets, @sbrackets;
    }
    if ( @brackets ) {
        s/([\[\]\(\)\\\+\*\{\}\?\!])/\\$1/g for @brackets;
        #say STDERR Dumper(\@brackets);
        my $brackets = join "|", @brackets;
        $text =~ s/$brackets/ /g;
    }
    $text =~ s/$marker[.]?/ /g;
    $text =~ s/\s+/ /g;
    $text =~ s/(^\s+|\s+$)//g;

    s/\\([\%\$\*\"\+])/$1/g for @inches;
    push @qty, @inches if @inches;
#    print STDERR Dumper (['qty2',\@qty]);
    push @qty, @xqty if @xqty;
#    print STDERR Dumper (['qty3',\@qty]);

    $res = join ":",  @qty if @qty;
    #$res = join ":", map { my $x = $_; $x =~ s/([а-яa-zА-ЯA-Z])[\s\ \/]+([а-яa-zА-ЯA-Z])/$1\/$2/; $x }  @qty if @qty;
    return $text, $res;
}

sub cut_time {
    my ( $self ) = @_;
    my @metrics = qw/day|нед|second|день|дней|лет|мес|месяц|мин|минут|минута|ночь|ночей|секунда|секунд|сут|сутки|тысячелетие|дня|дн|недели|недель|месяц|месяцев|года|часов|часа|час|ч|минуты/;
    return $self->cut_qty_base( [@metrics] );
}

sub cut_qty {
    my ( $self ) = @_;
    my @metrics = qw/баррель|баррелей|ведро|ведер|линия|модель|моделей|сигар|точка|упаковка|упаковок|чарка|ящик|ящиков|товаров|опций|звезд|гостей|раз|отзыв|отзыва|отзывов|объявлений|сотки|соток|сотка|адрес|адреса|адресов|канал|канала|каналов|место|этаж|этажи|этажах|контрактов|шт|штук|штуки|штоф|ступеней|окружность|диаметр|коробок|предмет|предмета|предметов/;    
    return $self->cut_qty_base( [@metrics] );
}

sub cut_money {
    my ( $self ) = @_;
    my $quantifiers = 'тыс\.?|млн\.?|тысяч[аиу]?|миллион(?:а|ов)?';

    my $money_units = {
        dollar_units   => 'доллар(?:а|ов)|долл\.?',
        euro_units     => 'евро|euro|eur\.?',
        ruble_units    => 'рубл(?:ь|я|ей)|руб\.?|р\.?',
        hryvnias_units => 'гривн[аы]?|гривен|грн\.?',
        other          => 'тенге|тнг\.?'
    };

    my $money_units_re = join('|', map {$money_units->{$_}} keys %$money_units);
    my $money_re = "(?:$quantifiers)?\\s*(?:$money_units_re)";
    my $metrics = [$money_re];
    my @currency = qw/\p{Sc}|\x{20BD}/;
    return $self->cut_qty_base( $metrics, [@currency] );
}

sub cut_vol {
    my ( $self ) = @_;
    return $self->cut_qty_base;
}

sub cut_all_metrical {
    my ( $self ) = @_;
    my %res = ();
    my $txt = $self->text;
    ( $txt, $res{vol} ) = $self->proj->phrase($txt)->cut_vol;
#    print STDERR Dumper ( ['after cut_vol',$txt,$res{vol}] );
    ( $txt, $res{qty} ) = $self->proj->phrase($txt)->cut_qty;
#    print STDERR Dumper ( ['after cut_qty',$txt,$res{qty}] );
    ( $txt, $res{money} ) = $self->proj->phrase($txt)->cut_money;
#    print STDERR Dumper ( ['after cut_money',$txt,$res{money}] );
    ( $txt, $res{time} ) = $self->proj->phrase($txt)->cut_time;
#    print STDERR Dumper ( ['after cut_time',$txt,$res{time}] );
    return ( $txt, %res );
}

sub qtre {
    my ( $self, $text ) = @_;
    $text =~ s/([\+\*\?\!\.\[\]\{\}\(\)])/\\$1/g;
    return $text;
}

sub replace_regexp_symbols {
    my ( $text ) = @_;
    return '' unless $text;
    $text =~ s/\(/__left_bracket__/g;
    $text =~ s/\)/__right_bracket__/g;
    $text =~ s/([\+\*\?\!\.\[\]\{\}])/\\$1/g;
    return $text;
}

sub unreplace_regexp_symbols {
    my ( $text ) = @_;
    return '' unless $text;
    $text =~ s/__left_bracket__/\(/g;
    $text =~ s/__right_bracket__/\)/g;
    $text =~ s/\\([\+\*\?\!\.\[\]\{\}])/$1/g;
    return $text;
}

sub cut_article {
    my ( $self ) = @_;
    my $text = $self->text;
    my $article;

    $text = replace_regexp_symbols( $text );
    if ( $text =~ /(\b(?:артикул|арт)[.]?[:]?[\s]*)([\d\w][\d\w\.\-]*[\d\w])/i ) {
        my ($beforeart, $art) = (replace_regexp_symbols($1), replace_regexp_symbols($2));
        $article = $art if $art =~ /\d/;
        $text =~ s/((__left_bracket__.*)$beforeart$art(.*?__right_bracket__)|$beforeart$art)/ /;
    }
    $text = unreplace_regexp_symbols( $text );
    return ( $article, $text );
}


sub cut_code {
    my ( $self ) = @_;
    my $text = $self->text;
    my $code;

    $text = replace_regexp_symbols( $text );
    if ( $text =~ /(?:\W|__|^)((?:(код|код товара|код скидки)\b)[:]?[\s]*([a-zа-яё]*\s*(?:__left_bracket__)?\d+(?:__right_bracket__)|[\d\w][\d\w\.\-]*[\d\w]))(\s+|$)/i ) {
        $code = replace_regexp_symbols($1);
        $text =~ s/((__left_bracket__.*)$code(.*?__right_bracket__)|$code)/ /;
        #print STDERR "$phr => $code\n";
    }
    $text = unreplace_regexp_symbols( $text );
    $code = unreplace_regexp_symbols( $code );
    return ( $code, $text );
}

sub cut_size {
    my ( $self ) = @_;
    my $text = $self->text;
    my $size;

    $text = replace_regexp_symbols( $text );
    if ( $text =~ /(?:\W|__|^)((?:(размер|size)\b)[:]?[\s]*([a-zа-яё]*\s*(?:__left_bracket__)?\d+(?:__right_bracket__)|[\d\w][\d\w\.\-]*[\d\w]))(\s+|$)/i ) {
        $size = replace_regexp_symbols($1);
        $text =~ s/((__left_bracket__.*)$size(.*?__right_bracket__)|$size)/ /;
        #print STDERR "$phr => $code\n";
    }
    $text = unreplace_regexp_symbols( $text );
    $size = unreplace_regexp_symbols( $size );
    return ( $size, $text );
}

sub cut_season {
    my ( $self ) = @_;
    my $text = $self->text;
    my $season = '';
    # ключевые слова
    my $re = "весна|лето|осень|зима|весенн(?:ий|яя|ее|ие)|летн(?:ий|яя|ее|ие)|осенн(?:ий|яя|ее|ие)|зимн(?:ий|яя|ее|ие)|демисезон|демемисезонн(?:ый|ая|ое|ые)";
    # сначала ищем многословные конструкции вида весна-осень, весна-осень 2015/2016
    if ( $text =~ /(?:^|[^\w-])((?:$re)(?:-(?:$re))?(?:(?:\s+20\d\d)(?:[\/-]20\d\d)?)?)(?:$|[^\w-])/i ){
        $season = $1;
    }
    # затем - однословники
    if ( !$season && $text =~ /(?:^|[^\w-])($re)(?:$|[^\w-])/i ){
        $season = $1;
    }
    $text = replace_regexp_symbols( $text );
    if ( $season ) {
        $text =~ s/((__left_bracket__.*)$season(.*?__right_bracket__)|$season)/ /;
    }
    $text = unreplace_regexp_symbols( $text );
#    print Dumper ([$text,$season]);
    return ( $season, $text );
}

sub cut_sex {
    my ( $self ) = @_;
    return () unless $self->text;
    my @sex = ();
    my @text = ();
    my $hmap = { 'мужчина'=>'мужской', 'женщина'=>'женский' };
    for my $pair ( $self->normwords_pairs ){
        if ( in_array($pair->[1], [qw/мужской женский унисекс мужчина женщина/]) ){
            push @sex, $hmap->{$pair->[1]} || $pair->[1];
        } else {
            push @text, $pair->[0];
        }
    }
    return ( join(':', uniq_array(@sex)), join(' ', @text) );    
}

sub cut_age_old {
    my ( $self ) = @_;
    my @age = ();
    my @text = ();
    my $hmap = { 'подростковый'=>'подросток', 'ребенок'=>'детский', 'девочка'=>'для девочки', 'мальчик'=>'для мальчика' };
    for my $word ( split /\s+/,$self->text ){
        my $normword = lc $self->proj->phrase($word)->norm_phr;
        if ( in_array($normword, [qw/взрослый детский подростковый подросток ребенок девочка мальчик/]) ){
            push @age, $hmap->{$normword} || $normword;
        } else {
            push @text, $word;
        }
    }
    return ( join(':', uniq_array(@age)), join(' ', @text) );    
}

our $hchildmap = {'детский' => {'whose' => 'детский', 'adj' => 1},
                  'ребенок' => {'whose' => 'детский', 'adj' => 0},
                  'девочка' => {'whose' => 'для девочки', 'adj' => 0},
                  'мальчик' => {'whose' => 'для мальчика', 'adj' => 0},
                  'подростковый' => {'whose' => 'для подростка', 'adj' => 1},
                  'малыш' => {'whose' => 'для малышей', 'adj' => 0},
                  'новорожденный' => {'whose' => 'для новорожденных', 'adj' => 0},
                  'грудной' => {'whose' => 'для грудных', 'adj' => 1},
                  'грудничок' => {'whose' => 'для грудничков', 'adj' => 0},
                  };

our $hchildmap_phl;

sub cut_age {
    my ( $self ) = @_;
    my @age = ();
    $hchildmap_phl //= $self->proj->phrase_list( [ keys %$hchildmap ] );
    my $source_text = '';
    $source_text = join (' ', $self->words );

    my $phl_found = $hchildmap_phl->search_subphrases_in_phrase( $self->proj->phrase( $source_text ) );
    if ( $phl_found && $phl_found->count ){
        # возвращаем age в канонической форме с притяжательным падежом
        @age = map { $hchildmap->{$_}->{whose}} @{$phl_found->perl_array};
    }

    # SUPBL-454: если перед существительным, означающего принадлежность к ребёнку (см. $hchildmap) стоит предлог 'для',
    # вырезаем этот предлог тоже. Но если этот предлог стоит перед прилагательным, то ничего не вырезаем (иначе в ситуации,
    # например, 'костюм для детского праздника' останется 'костюм праздника')
    my @cuttext = ();
    for my $word (split/\s+/,$self->text) {
        my $snorm_word_phr = $self->proj->phrase($word)->snorm_phr || '';
        if (!$hchildmap->{$snorm_word_phr}) {
            push @cuttext, $word;

        # 'adj' определяет, является ли ключ из $hchildmap прилагательным
        } elsif (($cuttext[-1] && $cuttext[-1] =~ /^($preps_re)$/i) && (! $hchildmap->{$snorm_word_phr}->{adj})) {
            pop @cuttext;
        }
    }

    return ( join(':', uniq_array(@age)), join(' ', @cuttext) );
}

sub cut_month {
    my ( $self ) = @_;
    my $text = $self->text;
    my $month_word;
    $text = replace_regexp_symbols( $text );
    if ( $text =~ /(\b[\d]+(?:\s*-?\s*\d+)+\s+(?:января|февраля|марта|апреля|мая|июня|июля|августа|сентября|октября|ноября|декабря|янв|фев|мар|апр|авг|сен|сент|окт|ноя|дек)[.]?)(\s+|$)/i ) {
        my $month_word = $1;
        $text =~ s/((__left_bracket__.*)$month_word(.*?__right_bracket__)|$month_word)/ /;
        #print STDERR "$phr => $month_word\n";
    }
    $text = unreplace_regexp_symbols( $text );
    $month_word = unreplace_regexp_symbols( $month_word );
    return ( $month_word, $text );
}

our @colors_norm = ();

# TODO переписать по-нормальному
sub cut_colors {
    my ( $self ) = @_;
    return ('','') unless $self->text;
    my $text = $self->text;
    my @colors;
    # сначала пробуем регуляркой, она умеет комбинировать префиксы
    my $re = $self->color_regexp;
    if ( @colors = $text =~ /(?:^|\s)($re)(?:\s|$)/gi ){
        $text =~ s/(^|\s+)$_(\s+|$)/ / for ( uniq_array(@colors) );
        $text =~ s/\s+/ /g;
        $text =~ s/(^\s+|\s+$)//;
#        @colors = map { $hcolors_map{lc($_)} || $_ } @colors;
#        return ( join(':', @colors), $text );
    }
    # пословный матчинг по нормам
    unless ( @colors_norm ){
        my $dict_colors = $self->proj->dict_manager->get_dict("colors");
        @colors_norm = keys %{$dict_colors->norm_hash};
        push @colors_norm, map { ('темно-'.$_, 'светло-'.$_) } @colors_norm;
        push @colors_norm, 'мультиколор';
    }
    my @words = split( /\s+/, $text );
    my @temp = ();
    for my $word ( @words ){
        if ( in_array($self->proj->phrase($word)->norm_phr, \@colors_norm) ){
            push @colors, $word;
        } else {
            push @temp, $word;
        }
    }
    $text = join( ' ', @temp );
    @colors = grep { !/(?:мульти)/ } uniq_array(@colors);
#    print STDERR Dumper ( [ $self->text, $text, \@words, \@colors ]);
    return ( join(':', @colors), $text );
}

our $wearcolors = {
    modif => 'фиолетово|кремово|зелено|красно|черно|бело|желто|оранжево|малиново|сине|светло|темно|кирпично|розово|лимонно|лазурно|джинсовый|потертый|нежно|потерто|серебристо|кобальтово|коричнево|ярко|ярко|дымчато|серо|шоколадный|ежевичный|кремовый|клетка|клетка сине|отделка|отд|шоколадно|бежево|песочно|винно|жемчужно|ультрамариново|бело|рубиново|пастельно|мятно|красно|желто|золотисто|бледно|небесно',
    colors => 'аква|белый|бурый|белая|белые|бежевые|бежевый|бирюзовые|бирюзовая|бирюзовое|бирюзовый|блестящие|бордовый|бордовые|бордовая|бордовое|бронзовые|бронзовый|бронзовая|бронзовое|ванильный|вишневый|голубые|голубая|голубая набивка|голубое|голубой|горчичный|горчично-желтый|графитный|желтый|желтое|желтые|золотистый|золотой|золотые|зеленые|зеленая|зеленый|зеленое|земляничный|изумрудный|индиго|камуфляжный|кремово-белый|коричневое|коричневые|коричневая|коричневый|коралловый|королевский синий|красные|красный|красная|красные|красный молочный|кремовый|кукурузный желтый|лавандовая набивка|\&quot;леопардовый\&quot;|леопардовые|леопардовая|леопардовый|лиловый|лимонный|лососевый|малиновый|меланжевый|металлик|молочно-белый|мятный|небесно-голубой|неоновый лососевый|оливковый|оранжевый|оранжевые|оттенки коричневого|оттенки кофейного|охра|песочный|песочные|песочная|песочное|пудровый|пурпурный|разноцветный|разноцветная|разноцветные|розовые|розовый|розовый неоновый|рыжие|рыжее|рыжая|рыжий|салатовый|салатовая|салатовое|салатовые|серебристый|серебряные|серебряный|серебряное|серебряная|серный|серые|серый|серая|серый меланжевый|серые|сиреневый|сиреневые|сиреневая|сиреневое|синие|синее|синяя|синий|синий &quot;потертый&quot;|синий потертый|сливовый|слоновая кость|терракотовый|телесный|фуксия|фисташковый|фиолетовое|фиолетовый|фиолетовые|хаки|хромовый|цвет(?:а)? белой шерсти|цвет(?:а)? бронзы|цвет(?:а)? голубого льда|цвет(?:а)? лайма|цвет(?:а)? мальвы|цвет(?:а)? морской волны|цвет(?:а)? розового золота|цвет(?:а?) слоновой кости|цветной|цикламен|черное|черные|черный|черная|черный \&quot;леопардовый\&quot;|черный леопардовый|экрю|ягодный|яркие',
    postfix => 'в горох|в горошек|в клетку|в клеточку|в полоску|в полосочку|с цветочным рисунком|в цветочек|с цветами|с застежкой-молнией|с рисунком|с принтом|комбинированный|нефритовый|однотонный|\&quot;потертый\&quot;|&quot;варенка&quot;|варенка|с золотистой звездой|деним|кремовый|цветной|шоколадный|&quot;состаренный&quot;|состаренный|&quot;стираный&quot;|стираный|с &quot;леопардовым&quot; принтом|с матовым покрытием|с [^ ]+ рисунком|с рюшами|со стразами|с кружевом|на выпускной|с короткими рукавами|с рукавом летучая мышь|с вышивкой|на бретелях|с длинными рукавами|с баской|на молнии|с рукавом фонарик|с бантом|с тонкими бретельками|со складками|с рукавом реглан|с рукавами 3 4|с камнями|без бретелек|с завышенной талией|с воротником стойка|без рукавов|с карманами|с капюшоном|кружевные|без подкладки|Несколько цветов|клетка серо-черная|клетка серо-синяя|синий горошек|(?:горошек|журавли|цветы)\s+на\s+(?:белом|сиреневом|сером|синем)|Короткие рукава|длинные рукава|высокий воротник|в байкерском стиле|с \d карманами|высокий воротник|воротник-стойка|воротник-поло|цветочный принт|с бахромой|с капюшоном и поясом|на синтепоне с капюшоном|с поясом|на синтепоне|2 варианта пояса|с разрезом|с кружевной кокеткой',
    psdcolors => 'мультиколор',
};

sub color_regexp {
#    my $reclr = '(?:[Цц]вет\s*[-:]\s*)?(?:(?:(?:'.$wearcolors->{modif}.')[-/ ])*?(?:'.$wearcolors->{colors}.'))';
    my $reclr = '(?:[Цц]вет\s*[-:]\s*)?((?:(?:'.$wearcolors->{modif}.')[-/ ])*?(?:'.$wearcolors->{colors}.'))';
    return $reclr;
}

sub wide_color_regexp :GLOBALCACHE {
    my $reclr = '(?:\d[xх]\s+|[Цц]вет\s*[-:]\s*)?(?:(?:(?:'.$wearcolors->{modif}.')[-/ ])*?(?:'.$wearcolors->{colors}.'|\/\/   дальше не цвета   \/\/|'.$wearcolors->{postfix}.'|'.$wearcolors->{psdcolors}.')\s*(?:[,\+\/\-и]\s*)?)+(?:\(\d+\))?';
    my $re = '^(?:'.$reclr.')$';
    return $re;
}

sub has_color {
    my ($self) = @_;
    my $re = $self->color_regexp();
    return $self->text =~ /$re/;
}

sub cut_wear_property_deprecated {
    my ( $self ) = @_;
    my $text = $self->text;
    return ('','') unless $text;
    my @wp;
    my $re = $self->wear_postfix_regexp;
    if ( @wp = $text =~ /$re/gi ){
        @wp = map { $_=~s/^\s+//; $_ } @wp; ##no critic
        $text =~ s/(^|\s+)$_(\s+|$)/ / for @wp;
        $text =~ s/\s+/ /g;
        $text =~ s/(^\s+|\s+$)//;
        return ( join(':', @wp), $text );
    }
    return ( '', $text );
}

sub cut_wear_property {
    my ( $self ) = @_;
    my $text = $self->text;
    if ($wear_propre eq '')
    {
        $wear_propre = join '|', ($wearcolors->{postfix}, $wear_propre_shorten);
        $wear_propre = qr{$wear_propre};
    }
    my @wp;
    if ( @wp = $text =~ /(?:^|\b)($wear_propre)(?:\b|$)/gi ){
        @wp = map { $_=~s/^\s+//; $_ } @wp; ##no critic
        $text =~ s/(^|\s+)$_(\s+|$)/ / for @wp;
        $text =~ s/\s+/ /g;
        $text =~ s/(^\s+|\s+$)//;
        return ( join(':', @wp), $text );
    }
    return ( '', $text );
}

sub get_lemmer_inf {
    my ($self) = @_;
    my %res = ();
    my %ps = ();
    my %cs = ();
    my $text = $self->text;

    my $lemmer_out = $self->proj->lemmer_test->analyze($text);
    #print STDERR "$lemmer_out\n";
    my @lines = split "\n", $lemmer_out;
    for my $line (@lines) {
        next unless $line;
        my $word = '';
        if ( $line =~ /([^\s]+)(?=\s\d+\s\d+\s(mis|ru|en|uk|tt|be|kk|tr|ro|bas-en))/ ) {
            $word = $1;
        }
        my ($part, $case) = ('', '');
        if ( $line =~ /(?:Good|Bastard|Good\|BadRequest|Sob|Sob\|Prefixoid)(?:\|LLL)?\s?([A-Z]+)\,?(?:\S+)?\s?(\S+)?/ ) {
            ($part, $case) = ($1, $2);
        }
        $res{$word}{parts}{$part}++ if $part;
        my @arr = ();
        @arr = $case =~ /(loc|acc|gen|ins|dat|nom)/g if $case;
        $res{$word}{cases}{$_}++ for (@arr);
    }
    #print STDERR "$word\n";
    #print STDERR Dumper(\%res);
    return %res;
}

# вычисление словности
sub get_model_wc {
    my ($self) = @_;
    my @words = split " ", $self->text;

    my $n = 0;
    for my $i (0..$#words) {
        # односимвольные слова не считаем
        $n++ if $words[$i] !~ /^[\s]*[$RU$LA][\s]*$/;
    }
    # двухсловники считаем однословниками, если ни одно из конечных слов не цифро-буквенное
    $n = 1 if ($n == 0 ||  $n == 2 && $words[$#words-1] !~ /([\d].*[^\d]|[^\d].*[\d])/i 
                                   && $words[$#words]   !~ /([\d].*[^\d]|[^\d].*[\d])/i);
    return $n;
}

# убирает из фразы дубликаты
# с учетом нормализации norm=>1
sub fix_dup_words {
    my ($self, %par) = @_;
    return '' unless $self->text;
    my $text = $self->text;
    return $text if $text !~ / /;
    my @words = split / /, $text;
    my @res = ();
    for my $i (0..$#words) {
        my $is_valid = 1;
        my $nword_i = $par{norm} ? $self->proj->phrase($words[$i])->norm_phr || $words[$i] : $words[$i];
        for my $j (0..$#words) {
            my $nword_j = $par{norm} ? $self->proj->phrase($words[$j])->norm_phr || $words[$j] : $words[$j];
            $is_valid = 0 if ($nword_i eq $nword_j && $i > $j)
        }
        push @res, $words[$i] if $is_valid;
    }
    my $res = join " ", @res;
    #print STDERR "$res\n";
    return $res;
}

# то, что не вошло в префильтр по баннерам, делаем отдельным методом
sub post_prefilter_bnr {
    my ( $self ) = @_;
    my $txt = $self->text;
#    $txt =~ s/(?:\s|^|[,])(?:quot|за \d+|pink|\d{4} год(а)?|(?:c|до)\s+\d\d?\.\d\d?(?:\.\d\d\d{2}?)?|от производителя|от \d+|trade-in|trade in|планируете|хотите|в кредит|опт и розн|быстро и легко|где|предлагает|найти|поиск|беспокоит|хотите|нуж(?:на|ны|ный|)|нужен|ище(?:те|шь|м)|узна(?:й|йте|ть)|недорого|где можно|продать|купи(?:те|шь|ть)?|купля|прода(?:жа|ются|ется)|доставка|получ(?:ить|ение)|заказ(?:ать)?|(?:(?:до|на|свыше|более)\s+)\d*_percent|официальный дилер|белый_ветер|бытовая техника|бытовой техники|(?:dmn_)?холодильник.ру|доставка по россии|(?:[а-я]{2}\s+)?(?:предпраздничн\w+\s+)?(?:скидк|ставк)\w+(?:\s+[а-я]{2}?)|выгодный|выгодная(?: доставка)?(?: в ваш регион)?|дешев\w+|специальное_предложение|(спец\.?\s*)?предложение( на)?|качественный_сервис|(?:(?:в|от)\s+)?(?:dmn_\w+|[\w\.]+(?:ru|ua|com|org|net|by))|интернет_магазин|белый_ветер|220_instrument|под_ключ|сотмаркет|(?:в )?евросет(?:ь|и)|самовывоз|24_часа|в_подарок|ситилинк|здесь|sale(?: на)?)(?=(\s|[.,!:?]|$))/ /g;
    my $re = get_post_prefilter_bannerre;
    $txt =~ s/$re/ /g;
    $txt = $self->proj->phrase($txt)->norm_phr;
    my $dict_colors = $self->proj->dict_manager->get_dict("colors");
    my @colors_norm = keys %{$dict_colors->norm_hash};
    $txt = join( ' ', map { in_array($_, \@colors_norm) ? () : $_ } $self->proj->phrase($txt)->words );
    ( $txt ) = $self->proj->phrase($txt)->cut_all_metrical;
    return $self->proj->phrase($txt);
}

sub postfilter_subphrase_for_infusion {
    my ( $self ) = @_;
    my $phrzero = $self->proj->phrase;
#    return $phrzero unless $self->get_bnr_count;
    my $sub_text = $self->text;
    my $brand = $self->proj->phrase($sub_text)->get_brand;
    @local_wides = map { s/_/ /g; $_ } @local_wides; ##no critic
    $sub_text =~ s/ -+/ /g; 
    $sub_text =~ s/^-+//;
    $sub_text =~ s/-+ / /g; 
    $sub_text =~ s/-+$//;
    $sub_text =~ s/\"/ /g; 
    $sub_text =~ s/\\/ /g; 
    $sub_text =~ s/(^|\s+)(руб|купить)(\s+|$)/ /g; # слова которых не может быть в моделях 
    $sub_text =~ s/(?:^|\s+)(?:intel|celeron|core(?: i(?:3|5|7))?|amd|pentium|atom|athlon|retina|kit|i3|i5|i7|bluetooth|accessory|windows)(?:\s+|$)/ /g;
    $sub_text =~ s/^\s+|\s+$//g;
    my %h = ();
    $sub_text = join (' ', grep { $h{$_}++; $h{$_} == 1 } split /\s+/,$sub_text); # удаляем дубли
    my $sub_text_copy = $sub_text;
    $sub_text_copy =~ s/(\d)[.,](\d)/$1$2/g;
    return $phrzero if ( $sub_text_copy =~ / / && length($sub_text_copy)<6 || length($sub_text_copy)<4 );
#    return $phrzero if ( $sub_text_copy =~ /^(apple )?(iphone|ipad).* ([a-zа-яё]{3,}|\d{3,})$/ ); # iphone 5 жми, iphone 5 300 итд.
    return $phrzero if ( $sub_text_copy =~ /(?:apple|iphone|ipad)/ ); # весь эппл фильтруем
    return $phrzero if ( $sub_text_copy =~ /(^|\s+)(toys|yuton|ютон|berlingo|octavia|chevy|toyota|prado|land cruiser|bmw|series 5|opel|volvo|passat|golf|hummer|mazda|hyundai|pyker|уаз|ferrari|brilliance china auto|santa fe|suzuki|ikco|bentley|peugeot|land cruiser|range rover|alpha romeo|roewe|mitsuoka|daihatsu|dacia|hummer|ac cars|tata|pajero sport|x-trail|renault|lincoln|dodge|chevrolet|aveo|chery|mazda|koenigsegg|hafei|ford|focus|civic|ceed|almera|primera|donkervoort|acura|volvo|toyota|land rover|cadillac|chrysler|maserati|mitsubishi|fiat|daimler|jeep|alfa romeo|audi|volkswagen|jaguar|morgan|ssang yong|honda|freelander|maybach|infiniti|fx35|buick|subaru|lancia|opel|mercedes-benz|mercedes|mersedes|sprinter|hyundai|kia|lamborghini|nissan|samsung motors|bugatti|ssangyong|maruti|citroen|great wall|callaway|caterham|porsche|skoda|noble|lifan|saab|rolls-royce|mercury|abarth|bmw|mercedes|gmc|тагаз|aston martin|hover|geely|daf|даф|ваз|газ)(\s+|$)/ ); # игрушки машины нах
# категоризовать товарное предложение Маркетом и сверить исходную категорию с полученной
    return $phrzero if ( $sub_text_copy =~ /^\w\d{3}$/ ); # c320, слишком омонимично
    return $phrzero if ( $sub_text_copy =~ /^\d+ ?(w|v)$/ ); # 750w
    return $phrzero if ( $sub_text_copy =~ /^([a-zа-яё -]+\s+\d{1,4}|\d{1,4}\s+[a-zа-яё -]+)$/ ); # asus 4384, 24 порт, 40 светодиодный итп.
    return $phrzero if ( $sub_text_copy =~ /^[a-zа-яё -]+ \d$/i ); # shantou gepai 2
    return $phrzero if ( $sub_text_copy =~ /^([a-zа-яё]{1,2}\s+\d{1,3}|\d{1,3}\s+[a-zа-яё]{1,2})$/ ); # 123 ab, ab 123
    return $phrzero if ( $sub_text_copy =~ /^[\d,.]+(x|х)[\d,.]+$/ ); # 24x123
    return $phrzero if ( $sub_text_copy =~ /^[а-яё ]+$/i ); # только кириллица
    return $phrzero if (( $sub_text_copy !~ /\d/ && scalar( @{[split /\s+/, $sub_text_copy]} ) < 3 ) || $sub_text_copy !~ /[a-zа-яё]/); # christian dior addict, 123456
    return $phrzero if ( $sub_text_copy =~ /^(?:[a-zа-яё]+\s+\d0{0,3}|\d0{0,3}\s+[a-zа-яё]+)$/ ); # player 2000, 400 reader
    return $phrzero if ( $sub_text_copy =~ /^(?:[a-zа-яё]+\s+\d\s+\d|\d\s+\d\s+[a-zа-яё]+|\d\s+[a-zа-яё]+\s+\d)$/ ); # pocketbook 1 7, 1 pocketbook 4, 1 7 pocketbook
    return $phrzero if ( $sub_text_copy =~ /^(?:[a-zа-яё]+\s+[a-zа-яё]+\s+\d0{0,2}|[a-zа-яё]+\s+\d0{0,2}\s+[a-zа-яё]+|\d0{0,2}\s+[a-zа-яё]+\s+[a-zа-яё]+)$/ ); # 1 license net, license 1 net, license net 1
    # убираем трехзначные числа и числа с нулями на конце и возвращаем пустоту, если остаток широкая фраза
    $sub_text_copy =~ s/(^|\s)\d{1,3}0{0,2}(\s|$)/ /;
    $sub_text_copy =~ s/^\s+|\s+$//g;
    return $phrzero if ( length($sub_text_copy) != length($sub_text) && ( $self->proj->phrase($sub_text_copy)->is_wide_phrase || in_array($sub_text_copy, \@local_wides) ) );
    $sub_text_copy = $sub_text;
    # убираем все широкие фразы, брэнд и проверяем остаток
    $sub_text_copy =~ s/(\d)[.,](\d)/$1$2/g; # удаляем точку между цифрами
    $sub_text_copy =~ s/[.,-]/ /g; # все возможные разделители заменяем пробелом
    $sub_text_copy = join ( ' ', grep { !in_array($_,[@local_wides, @model_wides]) } split /\s+/, $sub_text_copy );
#    print STDERR "[$sub_text_copy]\n";
    $sub_text_copy =~ s/(^|\s)($brand)(\s|$)/ /g if $brand;
    $sub_text_copy =~ s/^\s+|\s+$//g;
    return $phrzero if length($sub_text_copy) < 4;

    my $re = getnoisere;
    $sub_text_copy =~ s/$re/ /gi;
    $sub_text_copy =~ s/\s+/ /;
    return $phrzero if length($sub_text_copy) < 4;

    return $phrzero unless ( $sub_text_copy =~ /\d/ && $sub_text_copy =~ /[a-z]/ );
#    $sub_text =~ s/$brand/\<$brand\>/g if $brand;


    return $self->proj->phrase($sub_text);
}

# для фразы возвращает два варианта склейки - сплошняком и через дефис
sub get_glued {
    my ( $self ) = @_;
    my @res = ();

    my ( $model, $additional_text ) = split /\t/, $self->text;
    my $phr_model = $self->proj->phrase($model);
    my @words = $phr_model->words;
    return [] if @words < 2;
    my $brand = $self->get_brand; # брэнд склеивать не будем
    $brand = ( $words[0] eq $brand ) ? shift @words : '';
    return [] if @words < 2;
    my @prefix = ();
    push @prefix, $brand if $brand;
    if ( $words[0] !~ /\d/ && length($words[0]) > 4 ){ # первое слово из букв тоже склеивать не будем, если достаточно длинное (aspire)
        my $firstword = shift @words;
        push @prefix, $firstword if $firstword;
    }
    return [] if @words < 2;
    
    my $without_brand = "@words";
    my $glued = $without_brand;
    $glued =~ s/(?<=\d) (?=\d+(\s|$))/\*/g;
    $glued =~ s/\s+|\-//g;
    $glued =~ s/\*/\-/g;
    $glued .= " $additional_text" if $additional_text;
    $glued = "@prefix $glued" if @prefix;
    push @res, $glued;
    $glued = $without_brand;
    $glued =~ s/\s+/-/g;
    $glued .= " $additional_text" if $additional_text;
    $glued = "@prefix $glued" if @prefix;
    push @res, $glued;

    return \@res;
}

# не сохраняет знаки препинания и спецсимволы, возвращает массив из 2-х строк - строка без регионов и строка только из регионов через двоеточие
sub cut_regions {
    my ( $self ) = @_;
    my @wordres = ();
    my %regres = (); # города иногда дублируются в словаре, а нам дубли не нужны
    my $text = $self->text;
    $text =~ s/\-/ /g;
    my $ph = $self->proj->phrase($text);
#    print STDERR ">>>>>".$self->text."\n";
    my %hgeo = ();
    for my $hreg ( $ph->get_regions ){
#        print STDERR Dumper ( $hreg );
        my $normtown = $self->proj->phrase($hreg->{town})->norm_phr;
        my $town = join (' ',$self->proj->phrase($hreg->{town})->words);
        $regres{lc $town}++;
        my $normname = $self->proj->phrase($hreg->{name})->norm_phr;
        $hgeo{$_}++ for ( split /\s+/, $normtown );
        $hgeo{$_}++ for ( split /\s+/, $normname );
    }
    return ( $self->text, '' ) if ( !%hgeo && $self->text );
    my @ww = $self->normwords_pairs;
    for my $i ( 0 .. $#ww ){
        my ( $src, $norm, $is_stop ) = @{$ww[$i]};
        my @arr_norm = split /\-/, $norm;
#        print STDERR "$src -> $norm -> $is_stop\n";
        if ( $is_stop && ($i<=$#ww-1) ){
            my ( $src_next, $norm_next, $is_stop_next ) = @{$ww[$i+1]};
            my @arr_norm_next = split /\-/, $norm_next;
#            print STDERR "$src_next => $norm_next => $is_stop_next\n";
            push @wordres, $src if ( $is_stop && ( @arr_norm_next == 1 && !$hgeo{$norm_next} || @arr_norm_next == 2 && ( !$hgeo{$arr_norm_next[0]} || !$hgeo{$arr_norm_next[1]} ) ) );
            next;
        }
        push @wordres, $src if ( @arr_norm == 1 && !$hgeo{$norm} || @arr_norm == 2 && ( !$hgeo{$arr_norm[0]} || !$hgeo{$arr_norm[1]} ) );
    }
    my @res = ();
    @res = ( join(' ', @wordres), join(':', keys %regres) ) if ( @wordres || %regres );
#    print STDERR Dumper ( \@res );
    return @res;
}

sub parse_model {
    my ($self, %par) = @_;

    my $DEBUG = 0;

    my $proj = $self->proj;
    my $text = $self->text;
    my $model = '';
    #print "text0: $text\n" if $DEBUG;

    my $toponyms = $self->get_toponyms;
    if ($toponyms) {
        $text = join(' ', $self->substitute_words( { map { $_ => "" } split /\:/, $toponyms }));
    }

    my $metricalre = $BM::PhraseParser::dict_metricalre;
    $metricalre .= "|шт|мм";

    $text =~ s/[!?+\\\/\[\]\(\)&]/ /g;
    $text =~ s/[",]//g;
    $text =~ s/\.$//;
    $text =~ s/\s+/ /g;

    #print "text1: $text\n" if $DEBUG;
    my $num = '(?:[-+]?[\d]+[, .]?[\d]+|[\d])';
    my $num_or_range = "$num(?:[ ]?[–*\/xх-]?[ ]?$num)*";
    $text =~ s/\b$num_or_range\s?(?:$metricalre)\b/ /g;
    my $size = '\d+(x|х)\d+((x|х)\d+)*';
    $text =~ s/(^|\s)$size($|\s)/ /g;
    $text =~ s/\s+/ /g;
    #print "text2: $text\n" if $DEBUG;

    (my $colors, $text) = $proj->phrase($text)->cut_colors;
    my $brand = $proj->phrase($text)->get_brand(use_context => 1) || $proj->phrase($text)->get_acc_brand;
    #print "brand: $brand\n" if $DEBUG;
    my ($before_brand, $after_brand);
    if ($brand and ($text =~ /^\s*(?<bb>.*?)\b$brand\b(?<ab>.*?)\s*$/)) {
        ($before_brand, $after_brand) = ($+{"bb"}, $+{"ab"});
        $before_brand =~ s/\s+$//;
        $after_brand =~ s/^\s+//;
        unless ($after_brand) {
            $after_brand = $before_brand;
        }
    }

    unless ($after_brand) {
        $after_brand = $text;
    }
    my @after_words = split(/\s+/, $after_brand);

    #print "after_brand: $after_brand\n" if $DEBUG;
    #print "before_brand: $before_brand\n" if $DEBUG;

    if ($brand and scalar @after_words > 0) {
        $model = shift @after_words if $after_words[0] =~ /^[A-ZА-ЯЁ]/ and $after_words[0] =~ /\d/;
        unless ($model) {
            if ($after_words[1] and $after_words[1] =~ /^\d+$/) {
                $model = "$after_words[0] $after_words[1]" if $after_words[0] =~ /[a-z\d]/i;
                shift @after_words;
                shift @after_words;
            }
        }
        $model .= ' ' if $model;
    }
    #print "model0: $model\n" if $DEBUG;

    my @after_words_modeled;
    for my $word (@after_words) {
        if ($word !~ /[^А-ЯЁа-яё-]/ && $word !~ /[А-ЯЁ]{2,}/ && $word !~ /(^|\s+)[А-ЯЁ](\s+|$)/) {
            $before_brand .= " $word";
        } elsif ( $word !~ /^[А-ЯЁ][-а-яё]{0,2}$/ ) {
            push @after_words_modeled, $word;
        }
    }
    $model .= join(' ', @after_words_modeled) if scalar @after_words_modeled < 9;
    #print "model1: $model\n" if $DEBUG;

    unless ($model) {
        # try to find in $before_brand
        my @before_brand = $before_brand ? (split /\s+/, $before_brand) : ();
        if (@before_brand) {
            my @model2 = ();
            @model2 = grep { $_ !~ /[А-ЯЁа-яё]/ && $_=~ /[^\s]/ } @before_brand;
            $model = join (' ', @model2) if @model2;
        }
    }
    #print "model2: $model\n" if $DEBUG;

    my $re = getmodelsre;
    unless ($model) {
        $model = $1 if $before_brand and $before_brand =~ /($re)/;
        unless ($model) {
            $model = $1 if $after_brand =~ /($re)/;
        }
    }
    #print "model3: $model\n" if $DEBUG;

    my $good = $self->proj->phrase($model)->get_goods_simple;
    $model =~ s/$good// if $good;
    #print "model4: $model\n" if $DEBUG;

    if ($model) {
        $model = '' if $model =~ /^(\d+|\d+\.\d+)$/;
    }
    #print "model5: $model\n" if $DEBUG;
    return '' if $model !~ /[A-ZА-ЯЁ]/i;
    return '' if $model =~ /^[a-zА-ЯЁ]+$/i;
    $model =~ s/\s+$//;
    return $model;
}

sub parse {
    my ($self, %par) = @_;
    my $logger = $self->proj->logger;
    $logger->debug("phrase parse beg");

    $use_goods_basis =  (exists($par{'use_goods_basis'})) ? $par{'use_goods_basis'}: 1;
    my $proj = $self->proj;

    my @keys = qw /type model model_for model_weak quotes surnames brand brand_for class bank prep toponyms model_dirty colors sex age season/;
    my %res = ();
    $res{$_} = '' for @keys;
    $res{src} = $self->text;

    return %res unless $self->text;

    my $phr = $self->text;

    $logger->debug("phrase parse 1:", \%res);

    # кавычки
    (my $quotes, my $rest) = $self->proj->phrase($phr)->get_quotes;
    if ( $quotes ){
        $quotes =~ s/\:.*$//;
        $res{quotes} = $self->proj->phrase( $quotes )->get_prefiltered_line->text;
    }

    # фамилии
    $res{surnames} = $self->proj->phrase($phr)->get_surnames; 

    $logger->debug("phrase parse 2:", \%res);

    # хеш со всеми сущностями ЕИ
    my %hmetrical = ();

    if ($res{surnames} && $res{quotes}) {

        my @surnames_ru = split /\s+/, $res{surnames};
        my @type =  @surnames_ru > 2 ? @surnames_ru[0..2] : @surnames_ru;
        $res{type} = join (' ', @type);
        my @model = split /\:/, $res{quotes};
        @model = @model[0..3] if @model > 3;
        $res{model} = join (' ', @model);
        $res{class} = "author-media";
        $logger->debug("phrase parse 3:", \%res);
    } else {
        # фраза без кавычек и их содержимого
        $phr = $rest if $rest;

        # вырезаем топоним, для слабых моделей не трогаем 
        $res{toponyms} = $self->proj->phrase($phr)->get_toponyms;
        if ( defined $res{toponyms} && !$par{weak} ){
            $phr = join( ' ', $self->proj->phrase( $phr )->substitute_words( { map { $_ => "" } split /\:/, $res{toponyms} } ) );
        }
        $logger->debug("phrase parse 4:", \%res, $phr);
        # вырезаем артикул, код
        ($res{article}, $phr) = $self->proj->phrase( $phr )->cut_article;
        (undef, $phr) = $self->proj->phrase( $phr )->cut_code;
        # вырезаем размер
        (undef, $phr) = $self->proj->phrase( $phr )->cut_size;
        # вырезаем слова с месяцем
        (undef, $phr) = $self->proj->phrase( $phr )->cut_month;
        # вырезаем сезон
        ($res{season}, undef) = $self->proj->phrase( $phr )->cut_season;
        # вырезаем пол
        ($res{sex}, undef) = $self->proj->phrase( $phr )->cut_sex;
        # вырезаем возраст
        ($res{age}, undef) = $self->proj->phrase( $phr )->cut_age;
        # вырезаем паттерны количество\s*единицы измерения
        $logger->debug("phrase parse 5:", \%res, $phr);

        $phr =~ s/[!?]/ /g;

        ($phr, %hmetrical) = $self->proj->phrase($phr)->cut_all_metrical;
        if ( %hmetrical ){
            # пока берем только первый найденный паттерн
#            $res{$_} = (split/\:/,$hmetrical{$_})[0] for ( keys %hmetrical );
            $res{$_} = $hmetrical{$_} for ( keys %hmetrical );
        }
        # вырезаем скидки
        if ( $phr =~ /((скидка|скидки)[:-]?(?:[\d\s]*[\d])?)(\s+|$)/i ) {
            my $disc = $1;
            $phr =~ s/$disc/ /i;
        }
        # вырезаем топы
        if ( $phr =~ /\s(топ\s*[-]?\s*\d+)(?:\s|$)/i ) {
            my $top = $1;
            $phr =~ s/$top/ /i;
        }
        $phr =~ s/(^|\s)[2-4]\s*(в|in)\s*1(\s|$)/ /gi;
        $phr =~ s/(^|\s)(два|три|четыре|пять|шесть)\s+в\s+(одном|1)(\s|$)/ /gi;

        # фильтруем и делим на слова
        $phr =~ s/\+\s*подарок.*$//i;
        $phr = $self->proj->phrase($phr)->get_prefiltered_line->text;
        my @phr = split /\s+/, $phr;
        $logger->debug("phrase parse 6:", \%res, $phr);

        # выпарсиваем бренд
        $res{brand} = $self->proj->phrase($phr)->get_brand(use_context => 1);

        # удаляем дубли
        $phr = $self->proj->phrase($phr)->fix_dup_words;

        # удаляем шумовые слова, если у нас > 3 слов
        my @phr_orig = split /\s+/, $self->text; 
        if (@phr_orig > 3) {
            my $re = getnoisere;
            $phr =~ s/$re/ /gi;
            $phr =~ s/\s+/ /;
        }

        # вырезаем банк 
        $res{bank} = $self->proj->phrase($phr)->get_bank;
        $phr =~ s/(?<=\b)$res{bank}(?=\b)//gi;
        $logger->debug("phrase parse 7:", \%res, $phr);

        # парсинг слабых моделей
        my $parse_phr = $self->proj->phrase( $phr ); #экономим вызов get_goods, используя одну и ту же фразу для парсингов
        %res = $parse_phr->_parse_weak( %res, %par );
        if ( $par{weak} ){
            $res{$_} = $self->clear_text($res{$_}) for ( qw /type model_weak quotes/ );
            return %res;
        }
        $logger->debug("phrase parse 8:", \%res, $phr);

        # парсим
        my $aprep = $self->is_accessory;
        if ( $aprep ){
            $res{prep} = $aprep;
            $res{class} = "accessory";
            %res = $parse_phr->_parse_accessory( %res );
            $logger->debug("phrase parse 9a:", \%res, $phr, $parse_phr);
        } else {
            $res{class} = "goods";
            %res = $parse_phr->_parse_goods( %res );
            $logger->debug("phrase parse 9b:", \%res, $phr, $parse_phr);
        }

        $logger->debug("phrase parse 10:", \%res, $phr);
    }

    if ( $res{type} ){
        (undef, $res{type}) = $self->proj->phrase( $res{type} )->cut_season;
        (undef, $res{type}) = $self->proj->phrase( $res{type} )->cut_sex;
        (undef, $res{type}) = $self->proj->phrase( $res{type} )->cut_age;
    }

    $res{model_dirty} = $res{model};
    if ( !$res{model_dirty} || length($res{model_dirty})<3 ){
        $res{model_dirty} = $phr;
        $res{model_dirty} =~ s/^((?:\S+\s+){3}\S+).*$/$1/;
    }

    $logger->debug("phrase parse 11:", \%res, $phr);

    # из модели вычитаем брэнд
    $res{model} = ($self->proj->phrase($res{model}) % $self->proj->phrase($res{brand}) )->text if ( $res{model} && $res{brand} );
    $res{model} = ($self->proj->phrase($res{model}) % $self->proj->phrase($res{type}) )->text if ( $res{model} && $res{type} );
    $res{model_for} = ($self->proj->phrase($res{model_for}) % $self->proj->phrase($res{type}) )->text if ( $res{model_for} && $res{type} );

    $logger->debug("phrase parse 12:", \%res);

    # из типа вычитаем модель и брэнд
#    $res{type} = ($self->proj->phrase($res{type}) % $self->proj->phrase($res{model}) )->text if ( $res{type} && $res{model} );
#    $res{type} = ($self->proj->phrase($res{type}) % $self->proj->phrase($res{model}) )->text if ( $res{type} && $res{model_for} );
    $res{type} = ($self->proj->phrase($res{type}) % $self->proj->phrase($res{brand}) )->text if ( $res{type} && $res{brand} );
    $res{type} = ($self->proj->phrase($res{type}) % $self->proj->phrase($res{brand}) )->text if ( $res{type} && $res{brand_for} );
    if ( my @ct = $self->proj->phrase($res{type})->cut_colors ){ # цвет может оказаться в типе, взятом из dict_goods
        $res{colors} = $ct[0] if $ct[0];
        $res{type} = $ct[1] if $ct[0];
    }

#    %res = $self->filter_and_check_model( \%res ) if ( $res{model} && $res{class} ne 'accessory' );
    %res = $self->filter_and_check_model( \%res ) if ( $res{model} || $res{model_for} );
#    $res{$_} = lc($res{$_}) for @keys;
    $res{$_} = $self->clear_text($res{$_}) for ( qw /type model model_for model_dirty model_weak quotes/ );

    $logger->debug("phrase parse 13:", \%res);

    # сортируем размеры по близости к модели, брэнду, типу
    my @temp = ();
    @temp = map { $_ eq 'type' ? (split/\s+/,$res{type})[0] : $res{$_} } grep { $res{$_} } qw/model model_for brand type model_dirty/;
    if ( %hmetrical && @temp ){
        for my $metr (keys %hmetrical){
            if ( $res{$metr} ){
                next unless $res{$metr} =~ /\:/;
                my $metr_srt = $self->sort_arr_by_distance( $temp[0], [split/\:/, $res{$metr}] );
                $res{$metr} = join(':',@$metr_srt) if @$metr_srt;
            }
        }
    }

    $logger->debug("phrase parse 14:", \%res);
    
    return %res;
}

sub is_accessory {
    my ( $self ) = @_;
    my $logger = $self->proj->logger;
    my ( $type, $is_accessory ) = $self->get_goods( all_fields => 1 );
    $type = $self->proj->phrase( $type )->norm_phr if $type;
    $is_accessory //= 0;
    $logger->debug('inside is_accessory: text="'.$self->text.'"', "type='$type', is_accessory='$is_accessory' dict_goods{type}=", $dict_goods{$type});

    if ( $self->text =~ /(?:\s+)($acc_preps_re)(?:\s+)(.*$)/i && ( !$type || $is_accessory ) ) {
        my $prep = $1;
        my $after_prep = $2;
        if ($prep !~ /под/i) {
            return $prep;
        }
        # если после предлога 'под' идёт творительный падеж, не считаем это аксессуаром (пример: 'дуб под маслом')
        my $case_num = $self->proj->phrase($after_prep)->get_gender_number_case->{case};
        if ($case_num ne 'ins') {
            return $prep;
        }
    }

    return '';
}

sub is_fixed_in_dict_goods {
    my ( $self ) = @_;
    my $text = ''.($self->text);
    my $res =   $text &&
                %dict_goods &&
                $dict_goods{$self->norm_phr} &&
                $dict_goods{$self->norm_phr}{denorm} &&
                ($dict_goods{$self->norm_phr}{denorm} eq $text || $dict_goods{$self->norm_phr}{denorm_plural} eq $text);
    return (defined $res) ? (int $res) : 0;
}

sub _parse_accessory {
    my ( $self, %res ) = @_;
    my $phr = $self->text;

    sub seems_like_model {
        my ( $txt ) = @_;
        my $slm = get_model_by_regexp($txt); # пробный вариант, время покажет
        $slm = join ' ', grep { $_ !~ /^[А-ЯЁа-яё][а-яё]+$/ } (split /\s+/, $slm);
#        print STDERR ">>> $txt => $slm\n";
        return $slm if $slm;
        my @arr = $txt =~ /[0-9a-z-\. ]{2,}/gi;
        return (sort {length($b)<=>length($a)} @arr)[0];
    }

    sub get_model_by_regexp {
        my ( $txt ) = @_;
        my $re = getmodelsre;
        if ( $txt =~ /($re)/ ) {
            return $1;
        }
        return '';
    }

    my ( $before_prep, $after_prep ) = split( /(?:\s*)(?:$acc_preps_re)(?:\s+)/i, $phr );
    my $brand_before = $self->proj->phrase($before_prep)->get_brand || $self->proj->phrase($before_prep)->get_acc_brand;
#    print STDERR Dumper ([$before_prep, $after_prep]);
    $before_prep = ($self->proj->phrase($before_prep) % $self->proj->phrase($brand_before))->text;
    ( $res{colors}, $before_prep ) = $self->proj->phrase($before_prep)->cut_colors;
    ( $res{wear_prop}, $before_prep ) = $self->proj->phrase($before_prep)->cut_wear_property;
#    print STDERR Dumper ([$before_prep, $after_prep]);
    my $slm_before = seems_like_model($before_prep);
    #$slm_before =~ s/(^|\s+)$brand_before(\s+|$)/ /i if $brand_before;
    if ( $brand_before ){
#        print STDERR Dumper (['QQQ',$brand_before,$slm_before]);
        $res{brand} = $brand_before;
        $res{model} = $slm_before || '';
    } else {
        $res{model} = seems_like_model( $before_prep ) || '';
    }
#    print STDERR Dumper ([$slm_before, $brand_before, $res{model}]);

    my $first_brand_after = $self->proj->phrase($after_prep)->get_brand || '';
    my $temp = $after_prep;
    $temp =~ s/(^|\s+)$first_brand_after(\s+|$)/ /i if $first_brand_after;
    my $second_brand_after = $self->proj->phrase($temp)->get_brand || '';

    my $tmp = '';
    ( $tmp, $after_prep ) = $self->proj->phrase($after_prep)->cut_colors;
    $res{colors} = join( ':', map { $_ || () } ( $res{colors}, $tmp ) );
    ( $tmp, $after_prep ) = $self->proj->phrase($after_prep)->cut_wear_property;
    $res{wear_prop} = join( ':', map { $_ || () } ( $res{wear_prop}, $tmp ) );

    if ( $first_brand_after && $second_brand_after ){
        my ( $before_fb, $after_fb, $after_sb ) = ('','','');
        if ( $phr =~ /для[а-я- ](.*)$first_brand_after(.*)$second_brand_after(.*)$/i ){
            ( $before_fb, $after_fb, $after_sb ) = ($1, $2, $3);
        }
        if ( $self->proj->phrase($first_brand_after)->get_acc_brand ){
            $res{brand} = $first_brand_after;
            my $mdl = seems_like_model($after_fb) || '';
            $res{model} = $mdl || '';
        } else {
            $res{brand} ||= $second_brand_after;
            my $mdl = seems_like_model($after_sb) || '';
            $res{model} ||= $mdl if $mdl;
            $res{brand_for} = $first_brand_after;
            $res{model_for} = seems_like_model($after_fb) || '';
        }
    } elsif ( $first_brand_after && !$second_brand_after ){
        my ( $before_fb, $after_fb ) = ('','');
        if ( $phr =~ /для[а-я- ](.*)$first_brand_after(.*)$/i ){
            ( $before_fb, $after_fb ) = ($1, $2 );
        }
        my $bmf = seems_like_model($before_fb) || '';
#        print STDERR Dumper ( ['WWWWW',$phr,$before_fb, $after_fb, $bmf] );
        if ( $bmf ){
            $res{model_for} = $bmf;
            $res{brand} = $first_brand_after;
            $res{model} = seems_like_model($after_fb) || '';
        } elsif ( $self->proj->phrase($first_brand_after)->get_acc_brand ){
            $res{brand} = $first_brand_after;
            $res{model} = seems_like_model($after_fb) || '';
        } else {
            if ( !$res{model} && !$res{brand} && $after_prep =~ /^\s*[а-яё-]+\s/i ){ # если после для русское слово, то это основная модель и брэнд
                $res{model} = seems_like_model($after_fb) || '';
                $res{brand} = $first_brand_after;
            } else { # иначе длядские
                $res{model_for} = seems_like_model($after_fb) || '';
                $res{brand_for} = $first_brand_after;
            }
        }
    } else {
        $res{model_for} = seems_like_model($after_prep) || '';
    }

    $res{type} ||= $self->proj->phrase($phr)->get_goods;
    if ($use_goods_basis && !$res{type} && ($res{model} || $res{model_for} || $res{brand} || $res{brand_for})){ #тип имеет смысл, только если это товар
        my %basis = $self->proj->phrase($phr)->get_goods_basis;
        $res{type} = $basis{type};
        $res{type} = '' unless $self->proj->phrase($phr)->_is_type_correct( $res{type} );
    }
    for my $key ( keys %res ){
        my $v = $res{$key};
        next unless $v;
        $v =~ s/(^\s+|\s+$)//g;
        $v =~ s/\s+/ /g;
        $res{$key} = $v;
    }
    $res{model} =~ s/^((?:\S+\s+){3}\S+)\s.*/$1/ if $res{model};
    $res{model_for} =~ s/^((?:\S+\s+){2}\S+)\s.*/$1/ if $res{model_for};
    $res{model} = '' if $self->proj->phrase($res{model})->pweight < 16;
    $res{model_for} = '' if $self->proj->phrase($res{model_for})->pweight < 13;
        
#    print STDERR Dumper ( \%res );
#    print pweight($res{model})."\t".$res{model}."\n" if $res{model};
#    print pweight($res{model_for})."\t".$res{model_for}."\n" if $res{model_for};

    return %res;
}

sub pweight {
    my ( $self ) = @_;
    my $txt = $self->text;
    my $digits = @{[$txt =~ /[1-9]/g]} || 0;
    my $zeroes = @{[$txt =~ /[0]/g]} || 0;
    my $letters = @{[$txt =~ /[a-z]/gi]} || 0;
    my $ru_letters = @{[$txt =~ /[а-я]/gi]} || 0;
    my $spaces = @{[$txt =~ /\s/g]} || 0;
    my $w = $digits*10 + $letters*8 + $ru_letters*6 + $zeroes*6 - $spaces*5;
    $w = 0.75*$w if ($txt !~ /\d/ || $txt !~ /[a-zа-яё]/i);
    return $w;
}


sub _parse_brand_model {
    my ( $self, %res ) = @_;
    my $logger = $self->proj->logger;
    my $phr = $self->text;
    $logger->debug("phrase _parse_brand_model beg", \%res, $phr);

    # выделяем брэнд, бьем фразу на до и после брэнда
    my ( $brand, $model, $before_brand, $after_brand ) = ( '', '', '', '' );
    $brand = $res{brand} || $self->get_brand(use_context => 1) || $self->get_acc_brand;
    $logger->debug("phrase _parse_brand_model 1:", "brand=[$brand]");

    if ($brand) {        
        if ( $phr =~ /^(.*?)$brand(.*)$/i ) {
            $before_brand = $1." " if $1;
            $after_brand = $2 if $2;
        }
    }
    unless ($after_brand) { # если после брэнда ничего нет
        $after_brand = $phr; # то берем фразу целиком
        $after_brand =~ s/(^|\s)$brand($|\s)/ /; # но без брэнда
    }
    $after_brand =~ s/(^\s+|\s+$)//;
    ( $res{colors}, $before_brand ) = $self->proj->phrase($before_brand)->cut_colors;
    ( $res{wear_prop}, $before_brand ) = $self->proj->phrase($before_brand)->cut_wear_property;
    my $tmp = '';
    ( $tmp, $after_brand ) = $self->proj->phrase($after_brand)->cut_colors;
    $res{colors} = join( ':', map { $_ || () } ( $res{colors}, $tmp ) );
    ( $tmp, $after_brand ) = $self->proj->phrase($after_brand)->cut_wear_property;
    $res{wear_prop} = join( ':', map { $_ || () } ( $res{wear_prop}, $tmp ) );

    # в before дописываем русскую часть after, в модель - все остальное
    my @after_brand = split /\s+/, $after_brand;
    @after_brand = @after_brand[0..3] if ( $brand && scalar( @after_brand ) > 4 );
#   print STDERR "after brand array: $after_brand\n";
    if ( $brand ){
        # если 1 слово после брэнда с большой буквы и есть хотя бы одна цифра, то это модель
        if ( @after_brand && $after_brand[0] =~ /^[A-ZА-ЯЁ]/ && $after_brand[0] =~ /\d/ ){
            my $text = quotemeta($brand.' '.$after_brand[0]); #доп.проверка: если между ними еще что-то, то это не модель
            if ( $self->text =~ /$text/ ) {
                $model .= shift @after_brand;
                $model .= ' ';
            }
        }
        # если 2 слово после брэнда цифровое, то первое - модель
        if ( $after_brand[1] && $after_brand[1] =~ /^\d+$/ ) { 
            my $ab1 = shift @after_brand;
            my $ab2 = shift @after_brand;
            $model .= "$ab1 $ab2 " if ( $ab1 =~ /[a-z]/i || $ab1 =~ /\d/ );
        }
        $logger->debug("phrase _parse_brand_model 2 (has brand):", "model=[$model]");
    }

    for my $word (@after_brand) {
        # не включаем если есть нерусские буквы или верхний регистр >2 букв или одинокие заглавные русские
        if ($word !~ /[^А-ЯЁа-яё-]/ && $word !~ /[А-ЯЁ]{2,}/ && $word !~ /(^|\s+)[А-ЯЁ](\s+|$)/) {
            $before_brand .= $word.' ' if (length($word) > 2);
        } elsif ( $word !~ /^[А-ЯЁ][-а-яё]{0,2}$/ ) {
            $model .= $word.' ';
        }
    }
    $logger->debug("phrase _parse_brand_model 3:", "model=[$model]");

    # в некоторых случаях модель идет перед брэндом
    my @before_brand = ();
    @before_brand = split /\s+/, $before_brand if $before_brand;
    if ( @before_brand ){
        my @model2 = ();
        @model2 = grep { $_ !~ /[А-ЯЁа-яё]/ && $_=~ /[^\s]/ } @before_brand;
        $model = join (' ', @model2) if ( @model2 && !$model );
    }

    $res{brand} = $brand if $brand;
    $res{model} = $model;
    $logger->debug("phrase _parse_brand_model 3:", \%res);
    # последняя попытка - пытаемся найти регуляркой
    unless ( $res{model} ) {
        my $temp = $phr;
        $temp =~ s/(^|\s+)$res{brand}(\s+|$)/ /i if $res{brand};
        $temp =~ s/(^\s+|\s+$)//g;
        my $re = getmodelsre;
        if ( $temp =~ /($re)/ ) {
            $res{model} = $1;
        }
        $logger->debug("phrase _parse_brand_model 3b:", \%res);
    }
    return %res;
}

sub _parse_goods {
    my ( $self, %res ) = @_;
    my $phr = $self->text;
    my $phr_changed = 0;

    %res = $self->_parse_brand_model(%res);

    if (0 ){
        my %basis = $self->proj->phrase($phr)->get_goods_basis;
        print $basis{type}."\n" unless $self->proj->phrase($phr)->get_goods;
    }
    my $brand = $res{brand};
    if ($brand) {
        $phr =~ s/(^|\s+)$brand(\s+|$)/ /i;
        $phr_changed = 1;
    }

    $res{type} ||= $phr_changed ? $self->proj->phrase($phr)->get_goods : $self->get_goods; #экономим вызов get_goods, если фраза не изменилась

#    print STDERR Dumper ( [$phr,$res{type}] );
    if ($use_goods_basis && !$res{type} && ($res{model} || $res{model_for} || $res{brand} || $res{brand_for})){ # тип имеет смысл, только если это товар
        my %basis = $self->proj->phrase($phr)->get_goods_basis;
        $res{type} = $basis{type};
        $res{type} = join(' ', grep { $_=~/^[а-яё\-]+$/i } split( /\s+/, $res{type} )) if $res{type};
#       вывод словосочетания для майнинга  dict_goods
#       print $res{type}, "\n" if length($res{type}) > 3;
    }

    if ( $res{type} ){
        $res{type} = '' unless $self->proj->phrase($phr)->_is_type_correct( $res{type} );
    }
#    print STDERR Dumper ( \%res );
#    print STDERR "====".$res{model}."\n";

    return %res;
}

sub _parse_weak {
    my ( $self, %res, %par ) = @_;
#print STDERR Dumper ([$self->text, \%res]);

    my $brand = $self->get_brand || '';
    my $type = $self->get_goods || '';
    

    my $pre = '';
    $pre = $type if ( $brand && $type && $self->text =~ /$brand\s+$type/i );        
    $pre ||= $brand || $type;

    # если нет ни брэнда, ни типа
    unless ( $pre ){
        # одно или несколько слов прописными
        $res{model_weak} ||= $1 if ( $self->text =~ /\b([A-ZА-ЯЁ]+(?:\s+[A-ZА-ЯЁ0-9]+){0,3})\b/ );

        # несколько слов с заглавной
        $res{model_weak} ||= $1 if ( $self->text =~ /\b([A-ZА-ЯЁ]\S+(?:\s+[A-ZА-ЯЁ0-9]\S+){1,3})\b/ );

        # первая заглавная, дальше несколько слов латиница-цифры
        $res{model_weak} ||= $1 if ( $self->text =~ /\b([А-ЯЁ]\S+)(?:\s+[a-zA-Z0-9]+){0,3}\b/ );
        
        # первая с заглавной, дальше латиница-цифры-пробелы
        $res{model_weak} ||= $1 if ( $self->text =~ /\b([A-Z][- a-zA-Z0-9]+)\b/ );
    }
#    print "---".$self->text."\n";

    # после $pre

    # буквенно-цифровые
    $res{model_weak} ||= $1 if ( $self->text =~ /(?:(?i)$pre)((?:\s+$re_mruseng){1,3})/ );

    # одно или несколько слов прописными
    $res{model_weak} ||= $1 if ( $self->text =~ /(\s+[A-ZА-ЯЁ]+(?:\s+[A-ZА-ЯЁ0-9]+){0,3})\b/ );
    
    # несколько слов с заглавной
    $res{model_weak} ||= $1 if ( $self->text =~ /(?:(?i)$pre)(\s+[A-ZА-ЯЁ][a-zа-яё0-9]+(?:\s+[A-ZА-ЯЁ][a-zа-яё0-9]+){1,3})\b/ );

    # первое с заглавной, дальше латиница
    $res{model_weak} ||= $1 if ( $self->text =~ /(?:(?i)$pre)(\s+[A-ZА-Я][-a-zа-яё]+(?:\s+[-a-z0-9]+){1,2})\b/ );

    # первая латинская заглавная, дальше смесь цифр и латиницы
    $res{model_weak} ||= $1 if ( $self->text =~ /(?:(?i)$pre)([A-Z][ -a-zA-Z0-9]+)\b/ );

    # от 1 до 3 слов с заглавной на латинице
    $res{model_weak} ||= $1 if ( $self->text =~ /(?:(?i)$pre)((?:\s+[A-Z][-A-Za-z0-9]+){1,3})\b/ );

    # от 2 до 3 слов с заглавной на кириллице
    $res{model_weak} ||= $1 if ( $self->text =~ /(?:(?i)$pre)((?:\s+[А-Я][-А-Яа-яё0-9]+){2,3})\b/ );

    # 1-е слово с заглавной, дальше 1 или 2 с маленькой
    $res{model_weak} ||= $1 if ( $self->text =~ /(?:(?i)$pre)((?:\s+[А-Я][-А-Яа-яё0-9]+)\s+([а-яё0-9]+){1,2})\b/ );

    # сжимаем пробелы
    $res{model_weak} = $self->proj->phrase( $res{model_weak} )->pack_spaces;
    
    # 2-х, 3-х итд.
    $res{model_weak} = '' if $res{model_weak} =~ /^\d\-[хx](?:\s|$)/;

    # n-спальный
    $res{model_weak} = '' if $res{model_weak} =~ /[.1-9]+\-спальн.+$/;

    # срезаем предлог в конце
    $res{model_weak} =~ s/\s+(?:$preps_re)$//;

    # не должна пересекаться с типом
    if ( $res{model_weak} && $type ){
        $res{model_weak} = '' if $self->proj->phrase_list( [$res{model_weak},$type] )->intersection( norm => 1);
    }

    # не должна пересекаться с брэндом
    if ( $res{model_weak} && $brand ){
        $res{model_weak} = '' if $self->proj->phrase_list( [$res{model_weak},$brand] )->intersection( norm => 1);
    }

    if ( $par{weak} ){
        $res{type} = $type;
        $res{brand} = $brand;
    }

    return %res;
}

sub _is_type_correct {
    my ( $self, $type ) = @_;
    # тип не должен входить полностью в часть после предлога (Орматек Е550 с чехлом)
    if ( $self->text =~ /\s+($preps_re)\s+/i ){
        my $prep = $1;
        my ( $before_prep, $after_prep ) = split /\s+$prep\s+/, $self->text, 2;
        my $typenorm = $self->proj->phrase($type)->norm_phr;
        my $before_prep_norm = $self->proj->phrase($before_prep)->norm_phr;
        return 1 if scalar(@{array_intersection([split/\s+/,$typenorm], [split/\s+/,$before_prep_norm])});

        my $after_prep_norm = $self->proj->phrase($after_prep)->norm_phr;
        my $temp = array_intersection([split/\s+/,$typenorm], [split/\s+/,$after_prep_norm]);
        return 0 if scalar(@$temp) == scalar(split/\s+/,$typenorm);
    }

    return 1;
}

sub get_surnames {
    my $self = shift;
    my $res = "";
    # определяем глобально фразу как массив слов
    my @phr = $self->pluswords;
    # ...и как хеш (нумеруем слова)
    my %phr = ();
    for my $i (0..$#phr) {
        my $word = $phr[$i];
        $phr{$word} ||= $i;
    }
    my @surnames_ru = ();
    # сначала ищем в основном словаре
    @surnames_ru = $self->match_itemdict(\%dict_surnames_ru, item_maxsize=>1);
    return '' unless @surnames_ru;
    my @surnames_filtered = ();
    for my $i (0..$#surnames_ru) {
        my $word = $surnames_ru[$i];
        next if length($word) < 3;
        my $valid = 0;
        if ($dict_surnames_wide_ru{lc($word)}) { # если фамилия есть в словаре широких фамилий, снимаем омонимию
            next if $word !~ /^[А-ЯЁ][а-яё]/; # должна начинаться с заглавной
            $valid = 1 if $self->text =~ /((^|[,.;\s])([А-Я]\s*\.){1,}\s*$word|$word\s*([А-Я]\s*\.){1,2})/; # берем если находим инициал с точкой
            my $wnum = $phr{lc($word)};
            next unless $wnum;
            # если это отчество, пропускаем
            next if (       $word =~ /(вич|вна)$/i 
                        && (        ($wnum-2 >= 0)     && $dict_surnames_ru{lc($phr[$wnum-2])} # фамилия либо на 2 слова назад
                                ||  ($wnum+1 <= $#phr) && $dict_surnames_ru{lc($phr[$wnum+1])} # либо на 1 слово вперед
                           ) 
                    );
            # берем, если перед ней имя с заглавной
            $valid = 1 if ($wnum-1 >= 0 && $dict_names_ru{lc($phr[$wnum-1])} && $phr[$wnum-1] =~ /^[А-ЯЁ]/);
            # берем, если перед ней имя и отчество
            $valid = 1 if ($wnum-2 >= 0 && $dict_names_ru{lc($phr[$wnum-2])} && $phr[$wnum-2] =~ /^[А-ЯЁ]/ && $phr[$wnum-1] =~ /(вич|вна)$/i);
            # берем, если после нее имя
            $valid = 1 if ($wnum+1 <= $#phr && $dict_names_ru{lc($phr[$wnum+1])} && $phr[$wnum+1] =~ /^[А-ЯЁ]/);            
        }    
        else { 
            $valid = 1; # если фамилия не широкая, берем всегда
        }
        push @surnames_filtered, $word if $valid;
    }
    $res = join (":", @surnames_filtered);
    return $res;    
}

sub _denorm_text { # по нормализованному мешку слов возвращает согласованный текст из исходной фразы
    my ( $self, $text ) = @_;
    sub fill_gaps { # заполняет в массиве цедых чисел дырки размером в 1
        my ( @a ) = @_;
        return map{($_ && $a[$_] == $a[$_-1] + 2) ? ($a[$_]-1, $a[$_]) : $a[$_]} 0..$#a;
    }
    my @words = $self->pluswords;
    return '' unless @words;
#    print STDERR Dumper ([\@words,$text]);
    my %h = map { my $key = $self->proj->phrase($words[$_])->norm_phr || $words[$_]; $key => $_ } reverse (0..$#words);
    my @text_indexes = sort map { my $t = $self->proj->phrase($_)->norm_phr || $_; $h{$t} // () } ( split /\s+/, $text );
    return '' unless @text_indexes;
    my @updated_indexes = fill_gaps( @text_indexes );
    if ( defined $h{'для'} && $h{'для'} > $updated_indexes[0] && $h{'для'} < $updated_indexes[-1] && !in_array($h{'для'},\@updated_indexes) ){
       push @updated_indexes, $h{'для'};
       @updated_indexes = sort { $a <=> $b } @updated_indexes;
    }
    # из fill_gaps пропускаем только предлоги
    my @res = map { in_array($_,\@text_indexes) ? $words[$_] : $words[$_]=~/^($preps_re)$/i ? $words[$_] : () } @updated_indexes;
#    print STDERR Dumper ( [ $text, \%h, \@words, \@updated_indexes, \@res ] );
    return join (' ', @res);
}

sub sort_arr_by_distance {
    my ( $self, $word, $arr_words ) = @_;
    return [] unless @$arr_words;
    my $text = lc $self->text;
    my $reword = $self->qtre(lc($word));
#    print STDERR Dumper ( [ $text, $word, $reword ] );
    my @temp = split/$reword/, $text;
    return [] if ( length($temp[0]) eq length( $text ) );
    my $word_pos = length( $temp[0] );
    my %harr_words_distance = ();
    for my $aw ( @$arr_words ){
        $aw = lc $aw;
        # экранируем спецсимволы
        my $reaw = $self->qtre($aw);
        my @parts = split( /$reaw/, $text );
        next if ( length( $parts[0] ) eq length( $text ) );
        my $aw_pos = length ( $parts[0] );
        my $distance = $aw_pos - $word_pos;
        $distance = abs(1.1*$distance) if $distance < 0;
        $harr_words_distance{$aw} = $distance;
    }
#    print STDERR Dumper ( [$arr_words, $word_pos, \%harr_words_distance] );
    return [ sort { $harr_words_distance{$a} <=> $harr_words_distance{$b} } keys %harr_words_distance ];    
}

1;
