#!/usr/bin/perl -w -T
package BM::PrefProjSrv;

use strict;

use utf8;
use open ":utf8";
no warnings 'utf8';
binmode(STDIN,  ":utf8");
binmode(STDOUT, ":utf8");
binmode(STDERR, ":utf8");

use Encode;
use POSIX ":sys_wait_h";
use Time::HiRes qw/gettimeofday tv_interval/;

use base qw(Net::Server::PreFork);

use FindBin;
use lib "$FindBin::Bin/../lib";
use lib "$FindBin::Bin/../wlib";
use Utils::Common;
use CatalogiaMediaProject;
use Project;
use Utils::Sys qw(mem_usage);
use BM::SolomonClient;
use Utils::Hosts qw( get_curr_host );

use constant MAX_CHILD_MEMORY => 5 * 1024 * 1024 * 1024;

sub child_init_hook {
    my ($self) = @_;
    my $proj = CatalogiaMediaProject->new({
        no_auth => 1,
        no_form => 1,
        nrmsrv => 0,
        load_dicts => 1,
        load_minicategs_light => 1,
        load_languages => ["en", "tr"]
    });
    $proj->post_init;
    $self->{proj} = $proj;

    $self->{proj}->categs_tree->never_write_categs_cache(1);
    $self->{proj}->categs_tree->never_read_categs_cache(1);

    my $h = $Utils::Common::options->{PrefProjSrv_params};
    $self->{$_} = $h->{$_} for keys %$h;

    $proj->phrase("toyota corolla")->get_minicategs(); # to load categories first time

    $self->log("ready!");
}

sub pid_file {
    my ($self) = @_;
    
    return $Utils::Common::options->{PrefProjSrv_params}{pid_file};
}

sub post_bind_hook {
    my ($self) = @_;

    # запоминаем pid
    open F, "> " . $self->pid_file or die($!);
    print F $$;
    close F;
}

sub proj {
    my ($self) = @_;

    if(!$self->{proj}) {
        $self->{proj} = Project->new({});
    }

    return $self->{proj};
}

sub pid2query {
    my ($self) = @_;

    return ($self->{pid2query} ||= {});
}

sub send_metric_to_solomon {
    my ($self, $metric) = @_;
    if (! defined ($self->{curr_host})) {
        $self->{curr_host} = get_curr_host();
    }
    if (! defined ($self->{solomon_client})) {
        $self->{solomon_client} = BM::SolomonClient->new();
    }

    $self->{solomon_client}->push_single_sensor({
        cluster => "host_info",
        service => "fcgi_server",
        sensor  => "prefprojsrv",
        labels  => {
            host        => $self->{curr_host},
            metric_name => $metric,
        },
        value   => 1,
    });
}

sub child_is_talking_hook {
    my ($self, $sock) = @_;
    my $line = <$sock>;
    return if !$line;

    chomp $line;
    $line = Encode::decode("UTF-8", $line);
    $self->proj->log("child_is_talking: $line");

    my ($type, $pid) = $line =~ /^(BEGIN|END)\s(\d+)/;
    if($type eq "BEGIN") {
        $self->pid2query->{$pid} = { start_time => time(), last_alarm => time() };
        $self->send_metric_to_solomon("query_processing_begin_count");
    } elsif($type eq "END") {
        my $duration = time - $self->pid2query->{$pid}{start_time};
        my $child_duration = ($line =~ m/ duration: ([^\s]+) /)[0] // '';
        if (int($duration - $child_duration) > 3) {
            $self->send_metric_to_solomon("strange_duration_count");
        }
        $self->send_metric_to_solomon("query_processing_end_count");
        $self->pid2query->{$pid} = undef;
    }
}

sub process_request {
    my $self = shift;
    
    # здесь stdin и stdout пришли из Net::Server, поэтому они в байтовом binmode
    while (<STDIN>) {
        #my $text = decode_utf8($_);
        my $text = Encode::decode('UTF-8', $_);
        my $length = length($text);
        my $text_out_debug = substr($text, 0, 100);
        $text_out_debug =~ s/[^a-zA-Z0-9_\- \t,\.;:'"\/\\\|\[\]\{\}\(\)]/./g;
        #$text_out_debug =~ s/[а-яА-Я]/./g;
        my $start_time = [gettimeofday];
        my $begin_time = time();
        $self->{server}{parent_sock}->send("BEGIN $$ time: $begin_time length: $length text: $text_out_debug\n");
        my $res = $self->{proj}->server_command->DoCommand($text);
        print Encode::encode('UTF-8', $res), "\n";
        my $res_length = length($res);
        my $duration = tv_interval($start_time);
        my $end_time = time();
        $self->{server}{parent_sock}->send("END $$ begin_time: $begin_time end_time: $end_time duration: $duration res_length: $res_length length: $length text: $text_out_debug\n");
    }
}

sub idle_loop_hook {
    my ($self) = @_;

    my @dead_pids = grep{waitpid($_, WNOHANG) != 0} keys %{$self->pid2query};
    delete $self->pid2query->{$_} for @dead_pids;

    my $curr_time = time();
    for my $pid (keys %{$self->pid2query}) {
        my $curr_bytes = mem_usage($pid);

        if($curr_bytes > MAX_CHILD_MEMORY) {
            # убиваем процесс, съевший слишком много памяти
            $self->proj->log("child $pid is killed (memory: $curr_bytes bytes)");
            $self->pid2query->{$pid} = undef;
            kill "KILL", $pid;
        } else {
            # пишем в лог информацию о долгих запросах
            my $query = $self->pid2query->{$pid};
            if($query) {
                my ($start_time, $last_alarm) = map{$query->{$_}} qw(start_time last_alarm);

                $self->pid2query->{$pid}{duration} = $curr_time - $start_time;

                if($curr_time - $last_alarm > 10) {
                    my $delta = $curr_time - $start_time;
                    my $msg = "LONG QUERY: pid='$pid' delta='$delta'";
                    unless ($query->{is_long}) {
                        $query->{is_long} = 1;
                        $self->send_metric_to_solomon("long_query_begin_count");
                    }
                    $self->proj->log($msg);
                    $query->{last_alarm} = $curr_time;
                }
            }
        }
    }

    # если свободных процессов осталось слишком мало, убиваем самые долгие задачи
    my $busy_forks = grep{defined($_)} values %{$self->pid2query};
    my $forks_to_kill = $busy_forks + 3 - 25;#$self->{min_free_forks} - $self->{num_forks};
    if($forks_to_kill > 0) {
        $self->proj->log("$busy_forks processes are busy, $forks_to_kill processes need to be killed");

        my @pids = sort{$self->pid2query->{$b}{duration} <=> $self->pid2query->{$a}{duration}} grep{$_ && $self->pid2query->{$_}} keys %{$self->pid2query};
        for my $pid (@pids[0..($forks_to_kill - 1)]) {
            $self->proj->log("child $pid is killed (last request duration: " . $self->pid2query->{$pid}{duration} . ")");
            kill "KILL", $pid;
            $self->pid2query->{$pid} = undef;
        }
    }
}

1;

