package BM::Prefilter;

use strict;
use utf8;
use open ':utf8';

use base qw(ObjLib::Obj);

use Data::Dumper;
use Utils::Common;

__PACKAGE__->mk_accessors(qw(
    file
    language
));

our $dont_use_prefilter = 0;
our $dont_use_lowercase = 0;

sub my_func {
    my $self = $_[0];
    print "lang = " . $self->{lang} . "\n";
}

sub init {
    my ($self) = @_;
   
    # $self->{language} - ссылка на язык, для которого создается объект
    # $self->{file} - имя файла в директории dicts, с которого берутся префильтры
    
    my ($language, $file) = ($self->{language}, $self->{file});
    
    # дефолтный файл с префильтрами
    $file ||= $Utils::Common::options->{dict_prefilters};

    open F, $file or die($!);
    while (<F>) {
        chomp;
        next if /^\#/;
        my @tab_list = /(\t)/g;
        my $tab_count = @tab_list;
        $language->proj->log("WARNING: prefilter \"$_\" contains $tab_count tabs") if $tab_count != 2;
        my @arr = /(\\*\(?\??\w)/gi;
        for my $match (@arr) {
            my @match_list = split //, $match;
            my $last_char = pop @match_list;
            next if @match_list > 1 && $match_list[-1] eq '?' && $match_list[-2] eq '(';
            $language->proj->log("WARNING: prefilter \"$_\" contains uppercase character $last_char") if !(@match_list & 1) && ($last_char ne lc($last_char)) ;
        }
        my ($lang, $value, $key) = split /\t/, $_;
        next if $lang ne $language->name; 
        push @{$self->{replace2patterns}{$key}}, $value;
    }
    close F;

    $self->{outdated} = 1;
    $self->build;
    $language->proj->log("/ prefilter: $file");
}

sub build {
    my $self = shift;

    return if !$self->{outdated};

    my %prefilter_dict;
    my %regex_cache;

    for my $replace (keys %{ $self->{replace2patterns} }) {
        my @pieces = sort { length($b) <=> length($a) } @{ $self->{replace2patterns}->{$replace} };
        my $pattern = '(' . join('|', @pieces) . ')';
        $prefilter_dict{$replace} = $pattern;
        $regex_cache{$pattern} //= qr{$pattern};
    }
    
    $self->{prefilter_dict} = \%prefilter_dict;
    $self->{regex_cache} = \%regex_cache;

    $self->{outdated} = 0;
}

sub push_prefilter {
    my ($self, $pattern, $replace) = @_;

    push @{$self->{replace2patterns}{$replace}}, $pattern;
    $self->{outdated} = 1;
}

sub push_prefilters {
    my ($self, @prefilters) = @_;

    for my $prefilter (@prefilters) {
        my ($pattern, $replace) = @$prefilter;
        $self->push_prefilter($pattern, $replace);
    }
}

sub prefilter_text_by_dict {
    my ($self, $text, %parameters) = @_;

    $self->build;

    my $phr = $self->language->phrase($text);

    my $save_debug = sub {
        my $pattern = $1;
        my $replace = $_[0];
        my @regex_arr = sort { length($b) <=> length($a) } @{$self->{replace2patterns}{$replace}};
        my $regex = '';
        foreach ( @regex_arr ) {
            my $cur_regex = qr{$_};
            if ( " $pattern " =~ /$cur_regex/ig ) {
                $regex = $_;
                last;
            }
        }
        push @{$self->{debug_info}}, {pattern => $pattern, replace => $replace, regex => $regex };
        return '';
    };

    my $s = $phr->fltrd_text;
    my $orig = $s;
    foreach my $replacing_pattern (keys %{$self->{prefilter_dict}}) {
        my $pattern = $self->{prefilter_dict}->{$replacing_pattern};
        my $reg_expr = $self->{regex_cache}->{$pattern} // qr{$pattern};

        #в дебаг-режиме проверяем все префильтры, которые могли бы сработать
        if ($parameters{debug}) {
            my @regex_arr = @{$self->{replace2patterns}{$replacing_pattern}};
            foreach my $regex_text ( @regex_arr ) {
                my $cur_regex = qr{$regex_text};
                my @matches;
                my $match_text = $orig;
                while ($match_text =~ /($cur_regex)/i) {
                    push @matches, $1;
                    $match_text =~ s/($cur_regex)//i;
                }
                foreach ( @matches ) {
                    push @{$self->{debug_info}}, {pattern => $_, replace => $replacing_pattern, regex => $regex_text, all => 1 };
                }
            }
        }

        my $match_res = $replacing_pattern =~ /[^\\]\$|^\$/;
        if (!$match_res) {
            if ($parameters{debug}) {
                $s =~ s/$reg_expr/$save_debug->($replacing_pattern) . "$replacing_pattern"/ge;
            } else {
                if ($dont_use_lowercase) {
                    $s =~ s/$pattern/$replacing_pattern/ig;
                } else {
                    # Эта строка занимает существенное время при категоризации
                    $s =~ s/$reg_expr/$replacing_pattern/g;
                }
            }
        } else {
            if ($parameters{debug}) {
                eval "\$s =~ s/$pattern/\$save_debug->(\$replacing_pattern) . \"$replacing_pattern\"/ge;";
            } else {
                if ($dont_use_lowercase) {
                    eval "\$s =~ s/$pattern/$replacing_pattern/ig;"
                } else {
                    eval "\$s =~ s/$pattern/$replacing_pattern/g;"
                }
            }
        }
    }
    return $s;
}


sub get_prefiltered_phrase {
    my ($self, $text, %parameters) = @_;

    my $language = $self->language;
    my $proj = $language->proj;
    my $phr = $language->phrase($text);

    return $phr if $dont_use_prefilter;

    $self->{debug_info} = [];

    #предварительная обработка
    my $s = $self->prefilter_preprocess($text);
    
    #Тестовый вызов для подсчёта диффа для изменения префильтра
    $s = $proj->{'test_banner_categ_prefilter'}->($proj, $s) if defined($proj->{'test_banner_categ_prefilter'});

    #применение регулярок из веб-интерфейса
    $s = $self->prefilter_text_by_dict($s, %parameters);

    #дополнительная попытка достать информацию, которая не поддалась парсингу.
    #выполняется после регулярок, так как происходит не замена, а добавление текста в конец, и нет гарантии на порядок слов. 
    $s = $self->prefilter_postprocess($text, $s);

    return $language->phrase($s);
}

#1) вычисление некоторых флагов модерации. Выполняется до применения регулярок, так как регулярки могут потереть то, на что они матчатся
#2) предварительная обработка фразы: lowercase с турецким фиксом, расцепление слов, сцепленных разделителем. 
sub prefilter_preprocess {
    my ($self, $text) = @_;

    my $language = $self->language;
    my $proj = $language->proj;

    #Специальное слово для доставки, чтобы это можно было учесть безотносительно остальных фильтров
    my $specword = '';
    my @matches = ();
    if ((@matches = $text =~ /(\bдостав[кил])/i) && $text !~ /(?:доставк|доставл|достави)(?:[А-Яа-я]+)\s*в\s*(?:перекресток|магазины|магазин|магазины\s*сети|бахетле|ашан|окей|карусель|билла|магнит|викторию|бехетле|дикси|магнолию|пятерочку|атак|азбуку\s*вкуса|(?:ближайшую)?\s*аптеку)/i )
{  
        $specword = '__delivery_pattern';
        push @{$self->{'debug_info'}}, {pattern => $matches[0], replace => $specword, regex => 'delivery_pattern'};
    }

    my $phr = $language->phrase($text);
    $phr = $language->phrase($phr->ends_of_banners_filter); #Удаляем частотные хвостики баннеров
    my $s = $phr->fltrd_text;

    $s =~ s/Акция/ /g if $language->name eq 'ru'; #это должно сработать до lc
    $s =~ tr/Iİ/ıi/ if $language->name eq 'tr'; # а это турецкий фикс для lc

    sub _rpl_bad_smbl {
        my $t = $_[0];
        return $t if $t =~ /domain/;
        my $t2 = $t;
        $t2 =~ s/[\/\#\_\&\-]/ /g;
        return "$t $t2";
    }
    $s =~ s/(\w+([\/\#\_\&\-](?=quot;)?\w+)+)/_rpl_bad_smbl($1)/eg;

    $s = lc($s) unless $dont_use_lowercase;

    #Если есть подозрительные слова - добавляем маркер
    if( $s =~ /[А-Яа-я][A-Za-z]|[A-Za-z][А-Яа-я]/ ){
        $s .= " moderatebadwordtype";
        my @arr = grep {$_} ($s =~ /([A-Za-z]+)([А-Яа-я]+)|([А-Яа-я]+)([A-Za-z]+)/g);
        $s .= join('', map {" $_"} @arr);
    }

    #и другой маркер, тоже после подготовительных логик, чтобы его не обработало
    $s .= " $specword";

    return $s;
}

#Пытаемся дополнительно вытащить осмысленные слова из текста, спрятанные всякими не поддающимися парсингу способами. 
#Найденные слова добавляются в конец текста
sub prefilter_postprocess {
    my ($self, $text, $s) = @_;

    my $language = $self->language;
    my $proj = $language->proj;

    if ($language->name eq 'ru') {
        #Борьба со злодеями
        if( $s =~ /[А-Яа-я][A-Za-z]|[A-Za-z][А-Яа-я]/ ){
            $s = BM::Phrase::_fix_bad_text($s);
        }
        # запись с разделителем через букву п.р.о.е.к.т.о.р
        # берём сочетание из исходного текста, а не из модифицированного, так как были проблемы c 'б/у' на конце фраз, которые превращались в 'бубу'
        #так как lookbehind не работает с началом строки, закостылим, поставив в начале текста пробел
        for( " $text" =~ /(?<= ^|\s|[^A-Za-zА-Яа-я0-9])(\w{1,2}(?:[ \t\/\#\_\&\-]\w{1,2}){3,})(?= \W|\s|$)/g ){
            my $addtext = $_;
            $addtext =~ s/\W//g;
            $s .= " $addtext";
        }
        for( $s =~ /([A-Za-z]+0[A-Za-z]+)/g ){ # 0 вместо O
            my $addtext = $_;
            $addtext =~ s/0/o/g;
            $s .= " $addtext";
        }
        for( $s =~ /([А-Яа-я]+0[А-Яа-я]+)/g ){ # 0 вместо О
            my $addtext = $_;
            $addtext =~ s/0/o/g;
            $s .= " $addtext";
        }
        for( $s =~ /([А-Яа-яA-Za-z]+(?:\#[А-Яа-яA-Za-z]+)+)/g ){ #Использование # как убираемого разделителя
            my $addtext = $_;
            $addtext =~ s/\#//g;
            $s .= " $addtext";
        }
        #/Борьба со злодеями

        $s =~ s/&quot;/"/g;
    }
    return $s;
}

1;
