package BM::RPC;
use strict;

use JSON;

use base qw(ObjLib::ProjPart);

__PACKAGE__->mk_accessors(qw(
    todo
    cdict_requests
    options
));

#
# Remote Procedure Calls for BroadMatch (useful for running bm stuff on YT)
#

# rpc-вызовы получения данных из cdict (название rpc обычно совпадает с названием метода $phl)
# формат: $method_name => $method_inf, это хэш с полями:
#   fld => $name  -  поле, по которому в объекте хранится закешированный ответ из cdict-а   (либо одно из полей)
#       по наличию этого поля также определяем, нужно ли делать rpc-запрос!
#   method => $name  -  метод cdict_client-а, который вызывается для получения данных
#       is_cache_method => $bool  -  соотв. метод сам кладёт данные в кэш объекта
#   namespace => $name  -  название namespace, по которому лежат данные (и в железном cdict, и в yt)
#       is_snorm_key => $bool  -  lookup key получается снормилизацией
my %cdict_conf = (
    cache_search_count => {
        fld => 'search_count',
        method => 'get_count',
        namespace => 'count',
    },
    cache_search_countg => {  # соотв. $phl->cache_regions_count
        fld => 'regions_count',
        method => 'get_regions_count',
        namespace => 'countg',
    },
    cache_search_countm => {  # matches $phl->cache_mobile_counts
        fld => 'cdict_countm',
        method => 'get_mobile_count',
        namespace => 'countm',
    },
    cache_search_tail => {
        fld => 'search_tail',
        method => 'get_tail',
        namespace => 'tail',
    },
    cache_search_query_count => {
        fld => 'search_query_count',
        method => 'get_query_count',
        namespace => 'countq',
    },
    cache_cdict_regions_phrases => {
        fld => 'cdict_regions_phrases',
        method => 'cache_regions_phrases', is_cache_method => 1,
        namespace => 'regions', is_snorm_key => 1,
    },
    cache_cdict_tail_categs => {
        fld => 'tail2categs',  #  одно из полей!
        method => 'cache_cdict_tail_categs', is_cache_method => 1,
        namespace => 'tail',
    },
    cache_cdict_minicategs => {
        fld => 'cdict_minicategs',
        method => 'get_categs',
        namespace => 'categs', is_snorm_key => 1,
    },
    cache_search_syns => {
        fld => 'search_syns',
        method => 'get_syns',
        namespace => 'syns', is_snorm_key => 1,
    },
);
sub init {
    my $self = shift;
    $self->todo([]);

    $self->options({});
    $self->cdict_requests({});

    if ($ENV{RPC_IRON}) {
        # нужно, чтобы со включённой опцией MR_BROADMATCH ходить в железный cdict
        $self->options->{is_local} = 1;
    } elsif ($ENV{RPC_SIMPLE}) {
        $self->options->{on_missing} = 'check';
        $self->options->{local_cdict} = 1;
        $self->options->{on_add} = 'eval';
    } elsif ($ENV{RPC_BANNERLAND}) {
        $self->options->{on_missing} = 'die';
        $self->options->{local_cdict} = 1;
        $self->options->{on_add} = 'check_fld';
    } elsif ($ENV{MR_BROADMATCH}) {
        $self->options->{on_missing} = 'die';
        $self->options->{local_cdict} = 1;
        $self->options->{on_add} = 'always';
    } else {
        $self->options->{is_local} = 1;
    }
}


# This will be deprecated
# вызвать функцию, внутри которой есть rpc, и за-eval-ить эти вызовы (локально)
# параметры:
#   $func  -  функция с rpc-вызовом
#             возвращает
#             - либо ($result,undef), если завершила работу, где $result - defined scalar
#             - либо (undef,$continuation) с rpc-вызовами и контекстом; поля хэша $continuation:
#                 ctx => контекст для $func и eval_rpc/yield_rpc
#                 rpc => [список rpc-вызовов], см. eval_rpc
#             $func принимает на вход параметры, в том числе keyword-параметр ctx -- контекст вызова
#   $args  -  ссылка на массив параметров для вызова функции; будет вызвано $func->(@$args, ctx=>$ctx)
#   verbose => 0|1|2  -  логировать доп. инфу
sub call_with_rpc {
    my $self = shift;
    my $func = shift;
    my $args = shift // [];
    my %par  = @_;

    my $ctx  = {};  # изначально контекст пустой

    while () {
        my ($result, $todo) = $func->(@$args, ctx => $ctx);  # modify ctx inplace
        my $verbose = $par{verbose} // 0;
        if ($verbose > 1) {
            $self->log("call_with_rpc: ".(defined $result ? "finished!" : "todo ".$self->proj->smart_json_encode($todo)));
        } elsif ($verbose) {
            $self->log("call_with_rpc: ".(defined $result ? "finished!" : "continues"));
        }
        return $result if defined $result;

        $self->eval_rpc($todo);
    }
}

# вызовы rpc
# параметры:
#   $rpc  -  список rpc-вызовов вида [$method,$param1,$param2,..]
#       method - название rpc-вызова
#       param1,.. - параметры для этого вызова, их кол-во и смысл может отличаться
#   у не-кэширующих rpc-вызовов первым аргументом идёт пара [key=>$h], результат пишется в $h->{key}
sub eval_rpc {
    my ($self, $rpc) = @_;
    my $proj = $self->proj;
    my $cdict_client = $self->proj->cdict_client;
    for my $h (@$rpc) {
        my ($method, @params) = @$h;
        my $mconf = $cdict_conf{$method};
        if ($mconf) {
            my ($phl) = @params;
            my $cdict_method = $mconf->{method};
            if ($mconf->{is_cache_method}) {
                $cdict_client->$cdict_method(@$phl);
            } else {
                my $fld = $mconf->{fld};
                my @resp = $self->proj->cdict_client->$cdict_method(@$phl);
                my @phrs = @$phl;
                if ($method eq 'cache_search_syns') {
                    for my $r (@resp) {
                        my ($i, $resp) = @$r;
                        $phrs[$i]->{$fld} = $phrs[$i]->restore_spec_words($resp);
                    }
                } else {
                    for my $r (@resp) {
                        my ($i, $resp) = @$r;
                        $phrs[$i]->{$fld} = $resp;
                    }
                }
            }
        } elsif ($method eq "jumbocache") {
            # pass
        } else {
            die "eval_rpc undefined for $method!\n";
        }
    }
}

# сгенерировать "запросы" для получения данных для rpc-вызова
# параметры (в хэше):
#   $rpc  -  список rpc-вызовов, см. eval_rpc
#   { rpc_type => $yield_func }
#       хэш с указателями на функции, используемые для yield-а
#       им приходят на вход соотв. "запросы" -- хэш $y с нужными полями
sub yield_rpc {
    my ($self, $rpc, $yield) = @_;
    my $gen_req = {
        cdict => sub {
            my ($ns, $key, $lang) = @_;
            return {
                # cdict_namespace cdict_key cdict_value
                cdict_namespace => $ns . ((!$lang || $lang eq 'ru') ? "" : "_$lang"),
                cdict_key => $key,
            };
        },
        jumbocache => sub {
            my ($ns, $key) = @_;
            return {
                cdict_namespace => "jumbo.$ns",
                cdict_key => $key,
            };
        },
        simpgraph => sub {
            my ($sg, $snorm_phr) = @_;
            return {
                cdict_namespace => "simpgraph.$sg",
                cdict_key => $snorm_phr,
            };
        },
        advq => sub {
            my ($key) = @_;
            return {
                text => $key,
            };
        },
    };

    # уникализация только в рамках одного вызова yield!
    my %seen;
    my $seen_max_size = 1e6;
    my $do_yield = sub {
        my $req_type = shift;
        for my $y ($gen_req->{$req_type}->(@_)) {
            %seen = () if keys(%seen) > $seen_max_size;  # prevent memory overflow

            # нет смысла генерить одинаковые запросы
            my $str = join($;, $req_type, map { ($_, $y->{$_}) } sort keys %$y);
            next if $seen{$str}++;

            $yield->{$req_type}->($y);
        }
    };

    for my $h (@$rpc) {
        my ($method, @params) = @$h;
        my $mconf = $cdict_conf{$method};
        if ($mconf) {
            my ($phl) = @params;
            my $ns = $mconf->{namespace};
            my $is_snorm = $mconf->{is_snorm_key};
            for my $phr (@$phl) {
                my $key =  $is_snorm ? $phr->cdict_key_snorm : $phr->cdict_key_norm;
                $do_yield->('cdict', $ns, $key, $phr->lang);
            }
        } elsif ($method eq "external_cdict_get") {
            my ($v, $data) = @params;
            for my $h (@$data) {
                my ($ns, $key) = @$h;
                $do_yield->('cdict', $ns, $key);
            }
        } elsif ($method eq "jumbocache") {
            # jumbocache => { ns1 => [keys1], ns2 => [keys2], }
            my ($jumboreq) = @params;
            my @nses = keys %$jumboreq;
            for my $ns (@nses) {
                $do_yield->('jumbocache', $ns, $_) for @{ $jumboreq->{$ns} };
                delete $jumboreq->{$ns};
            }
        } else {
            die "yield_rpc undefined for $method!\n";
        }
    }
}

sub add {
    my $self = shift;
    my $method  = shift;
    my $obj = shift;
    my %par = @_;

    my $rpc;
    if ($par{force}) {
        $rpc = [$method, $obj];
        $self->eval_rpc([$rpc]);
        return;
    }
    my $on_add = $self->options->{on_add};
    if ($cdict_conf{$method}) {
        my $todo = $obj;
        my $fld = $cdict_conf{$method}{fld};
        if ($self->options->{is_local} or $on_add eq 'check_fld') {
            # локальный режим, если данные уже есть в объекте - делать ничего не нужно
            $todo = $todo->lgrep(sub { !defined $_->{$fld} });
        } elsif ($on_add eq 'eval') {
            # в "простом" режиме возможно повторное создание объектов и вызов cache
            # поэтому храним весь micro_cdict

            # сначала пытаемся достать данные из объекта
            $todo = $todo->lgrep(sub { !defined $_->{$fld} });

            # теперь из micro_cdict
            # нужно запомнить, для каких запросов не было данных (их будем yield-ить позже)
            $self->options->{on_missing} = 'store';
            $self->eval_rpc([[$method, $todo]]);
            $self->options->{on_missing} = 'check';
            $todo = $todo->lgrep(sub { !defined $_->{$fld} });
        } elsif ($on_add eq 'always') {
            # режим broad_mr
            # если данные не сериализуются (см. BM::Phrase::FREEZE), то нужно готовить rpc-запросы для всех объектов!
            if ($method eq 'cache_search_count') {
                $todo = $todo->lgrep(sub { !defined $_->{$fld} });
            }
        }
        return if !$todo->count;
        $rpc = [$method, $todo];
    } else {
        die "Not implemented!";
    }

    if (!$self->options->{is_local}) {
        # накапливаем rpc до первого использования, чтобы минимизировать кол-во прерываний
        push @{$self->todo}, $rpc;
    } else {
        # сразу выполняем
        $self->eval_rpc([$rpc]);
    }
}

# проверка, что нет необработанных rpc запросов
# если есть, кидаем штатное исключение
sub check {
    my $self = shift;
    $self->die if @{$self->todo};
}

sub die {
    my $self = shift;
    my $msg = $self->die_msg;
    die $msg;
}
sub die_msg {
    return "BMYT_rpc_todo";
}

sub flush_todo {
    my $self = shift;
    $self->todo([]);
}
sub flush_cdict_requests {
    my $self = shift;
    $self->cdict_requests({});
}


1;
