package BM::Resources;

use strict;

use utf8;
use open ':utf8';

# все функции и настройки для работы с ресурсами

use Utils::Common;
use Utils::Sys qw(print_err uniq dir_files);
use Utils::Hosts qw(get_hosts get_host_info get_host_role get_curr_host get_host_group);
use File::Temp qw(tempdir);
use Time::HiRes qw(gettimeofday tv_interval);
use JSON qw(to_json);

use base qw(Exporter);
our @EXPORT = (
    '%resources',             # словарь ресурсов
    #'%required_resources',    # словарь необходимых ресурсов для хоста
    'get_resource',           # скачать данный ресурс
    'get_resource_producer',  # с какого хоста скачивать (вспомогательная функция)
    'get_required_resources', # список требуемых ресурсов
    'get_required_tables',    # список требуемых таблиц mysql
    'get_required_resources_db', # список требуемых ресурсов базы данных (названий ресурсов, вида tbl_ТАБЛИЦА)
    'get_produced_resources', # список производимых ресурсов
    'get_produced_tables',    # список производимых таблиц
);
our @EXPORT_OK = (
    'get_produced_sb_resources',
    'update_resource_in_sandbox',
);

# Флаг extended_db должен быть выставлен для всех таблиц, кроме этого списка исключений
# Должно быть согласовано с bm-utils/yandex-bm-mysql-catalogia-media/catalogia-media.cnf
# !!! Таблицы в этот список добавлять не будем! Если всё же понадобится добавить таблицы в этот список, уточнить у rivik@ или emurav@
# !!! При добавлении новых таблиц этот список НЕ меняем!!!
our @tables_in_catmedia_bm_dbh = (qw[
    BannerReverseDomain Campaigns
]); # !!! При добавлении новых таблиц этот список НЕ меняем!!!

my $options = $Utils::Common::options;

# resources:
#   producer => роль (или хост), которая генерит ресурс и с которой будут скачивать этот ресурс
#   aux_producer => роль (или хост), которая генерит ресурс, но отсюда не скачиваем
#   files => список файлов, из которых он состоит
#   dir => директория с файлами ресурса (дополнительно к files)
#   min_bytes => мин. кол-во байт в каждом файле (для проверки корректности)
#   db_path - куда копировать таблицы
#   dont_check_mtime => не проверять время обновления в мониторинге
#   dont_check_mtime_files => не проверять время обновления для некоторых файлов
#   max_allowed_hours_old => максимальный допустимый возраст файла (или самого старшего из файлов, если их несколько) - для проверки в мониторинге
#   project => проект (BM/CatMedia/...) - для мониторинга
our %resources = (

    tests_feed_files => {
        producer => 'catalogia-media-dev02e.yandex.ru',
        dir => $options->{dirs}{tests_feed_files},
        min_bytes => 1,
        recursive => 1,
    },

    dyn_stat => {
        producer => 'sandbox',
        files => [  $options->{dyn_stat_params}->{file_bad_dyn_phrases}, ],
        sandbox => {
            min_time => 600,
        },
    },

    marketdata_id_category_vendor_model => {
        producer => 'sandbox',
        files => [
            $options->{MarketData_params}{id_category_vendor_model},
        ],
        project => 'bmcollect',
        sandbox => {
            min_time => 1200,
        },
        max_allowed_hours_old => 120,
        warning_hours_old => 80,
    },

    marketdata_subphraser => {
        producer => 'sandbox',
        files => [
            (map { $options->{subphraser_market}{$_} } qw(comptrie_file staticmap_file)),
        ],
        project => 'bmcollect',
        sandbox => {
            min_time => 1200,
        },
        max_allowed_hours_old => 120,
        warning_hours_old => 80,
    },
    simpgraphs => {
        # frozen (see IRT-1911)
        producer => 'sandbox',
        files => [ (values %{$options->{simpgraph_files}}) ],
        dont_check_mtime_files => [ @{$options->{simpgraph_files}}{qw(Tags SerpClicks)} ],
        project => 'DRF',
    },
    models => {
        producer => 'sandbox',
        files => [
            @{$options->{Models_params}}{qw(wide_models gen_vendors_file gen_modeltypes_file gen_modelsyns_file)},
        ],
        min_bytes => 1000,
        sandbox => {
            min_time => 1200,
        },
    },

    mediacontent => {
        # заморожен, см. IRT-1986 IRT-1886
        producer => 'sandbox',
        files => [
            $options->{MoviesCategories_params}{categs_file_gen}
        ],
        dont_check_mtime => 1,
        min_bytes => 1000,
    },

    generated_dicts => {
        producer => 'sandbox',
        files => [
            $options->{DictEquivDomains}{file},
            $options->{DictTragicword}{file},
            @{$options->{DictDirectMod}}{qw(file file_full)},
            (map { $options->{Words_params}{$_} } qw(norm_dict word2snorm_dict)),
            (map { $_->[0] } @{$options->{Words_params}{syn_dicts}}),
            $options->{DictMulWordSyns}{misspell_file},

            $options->{DictCSyns}{smap_translit_ru2en_file},
            $options->{DictCSyns}{smap_translit_en2ru_file},
            $options->{DictCSyns}{smap_translit_en2ru_strict_file},
        ],
        min_bytes => 1000,
        sandbox => {
            min_time => 2400,
        },
    },

    # эти ресурсы не заносим в required_resources из-за большого трафика (файл обновляется ежечасно)
    # ресурс забирается явным вызовом get_resource
    chronicle_data => {
        producer => 'bmcdict-gen',
        files => [
            $options->{ChronicleCdict_params}{single_file},
        ]
    },
    datoteka_data => {
        producer => 'bmcdict-gen',
        files => [
            $options->{DatotekaCdict_params}{single_file},
        ]
    },
    banners_bender_data => {
        producer => "sandbox",
        files => [ map{$options->{DistrBannersBender_params}{binary_file} . "_$_"} 1..scalar(@{$options->{DistrBannersBender_params}{hosts}}) ],
    },
);

# ресурсы из дампов таблиц
while (my ($group, $opt) = each %{$options->{db_dump}}) {
    my %h = (
        producer => $opt->{producer},
        files => [ values %{$opt->{table_file}} ],
        min_bytes => 1,
    );
    $h{project} = $opt->{project} if $opt->{project};
    $resources{"db_dump_$group"} = \%h;
}

$resources{"db_dump_bannerland"}{sandbox} = {
    min_time => 60,
};
$resources{"db_dump_bannerland"}{dont_check_mtime} = 1;

# required_resources -- список необходимых ресурсов для каждой роли
# ключом является роль (или полное имя хоста)
my %required_resources = (
    "bmcdict-front02" => [ qw(chronicle_data generated_dicts wordcount_dict) ],

    bmapi => [ qw(
        models
        generated_dicts
        marketdata_id_category_vendor_model marketdata_subphraser
        tests_feed_files
    )
    ],
    bmfront => [ qw(
        generated_dicts models
        models
    ) ],
    bannerland => [ qw(
        generated_dicts
        models
        marketdata_id_category_vendor_model marketdata_subphraser
        dyn_stat
    ) ],
    'bannerland-preprod' => [ qw(
        generated_dicts
        models
        marketdata_id_category_vendor_model marketdata_subphraser
        dyn_stat
    ) ],

    catmedia => [
    qw(
        generated_dicts models
    ),
    	'mediacontent', # для infuse
    ],
    'catalogia-media-scripts' => [ qw(
            generated_dicts
        ),
        qw[ models ],  # для категоризации
    ],
    'catalogia-media-scripts_idle' => [ qw(
            generated_dicts
        ),
        qw[ models ],  # для категоризации
    ],
    "catalogia-media-tasks" => [ qw(
        generated_dicts models
    ) ],
    "catalogia-media-auto" => [ qw(
        generated_dicts models
    ) ],
    "catalogia-media-front" => [ qw(
        generated_dicts models
    ) ],
    'catalogia-media-gen' => [ qw(
        generated_dicts
    ) ],
    "bmcdict-gen" => [ qw(
        generated_dicts
        models
    ) ],
    "bmcdict-front" => [ qw(generated_dicts wordcount_dict models datoteka_data) ],
    "bmbender-front" => [ qw(generated_dicts models banners_bender_data) ],

    'bm-dev' => [ qw(
        generated_dicts
    ) ],

    bmcache => [qw(
        generated_dicts
    )],
);

$required_resources{'catmedia-dev'} = $required_resources{catmedia};
$required_resources{bannerland_idle} = $required_resources{bannerland};


# хэш производителей таблиц { table => \%info }, в info - производитель, доп. инфа (напр., tgz vs lz4)
my %tbl_producer = (
);

for my $table (keys %tbl_producer) {
    $tbl_producer{$table}{tgz_file} //= $options->{db_uploader_params}{work_dir} . "/$table.tgz";   # Для db_uploader
    $tbl_producer{$table}{files} = [ $tbl_producer{$table}{tgz_file} ];     # Для транспорта ресурсов

    # Флаг extended_db должен быть выставлен для всех таблиц, кроме этого списка исключений
    $tbl_producer{$table}{extended_db} //= 1    if not grep { $table eq $_ } @tables_in_catmedia_bm_dbh;

    $resources{"tbl_$table"} = $tbl_producer{$table};
}

for my $key (keys %resources) {
    my $default_min_bytes = 10000;
    $resources{$key}{min_bytes} //= $default_min_bytes;
}

# хэш потребителей таблиц { role => \@tables }:
my %tbl_required = (
    'catalogia-media-gen' => [

        qw[
        ],
    ],
);


# проверка с учётом флага :master
sub is_producer {
    my $host = shift;
    my $producer = shift;
    return 1 if $host eq $producer;

    my ($producer_role, $master_flag) = split /:/, $producer;
    my $info = get_host_info($host);
    return 0 if !$info->{role};
    return 0 if $info->{role} ne $producer_role;
    return 0 if $master_flag and $master_flag eq 'master' and not $info->{master};

    return 1;
}

sub is_master_producer {
    my $host = shift;
    my $producer = shift;
    return 1 if $host eq $producer;  # точное соответствие

    my $info = get_host_info($host);
    return 0 if !$info->{master};
    my ($producer_role, $master_flag) = split /:/, $producer;
    return 0 if !$info->{role};
    return 0 if $info->{role} ne $producer_role;

    return 1;
}

sub get_required_resources {
    my $proj = shift;
    my $host = shift || get_curr_host();
    my $role = get_host_info($host)->{role} or return;
    my @res;

    push @res, @{$required_resources{$role} || []};
    push @res, @{$required_resources{$host} || []};

    @res = uniq @res;

    return @res;
}

sub get_required_tables {
    my $host = shift || get_curr_host();
    my $role = get_host_info($host)->{role} or return;
    my @res = uniq (
        @{$tbl_required{$role} || []},
        @{$tbl_required{$host} || []},
    );
    return @res;
}

sub get_required_resources_db {
    my $host = shift || get_curr_host();
    my @res = map {"tbl_$_"} get_required_tables($host);
    return @res;
}

sub get_produced_resources {
    my $host = shift || get_curr_host();
    my @res;
    while (my ($resource, $data) = each %resources) {
        if (grep { $_ and is_producer($host, $_) } $data->{producer}, $data->{aux_producer}) {
            push @res, $resource;
        }
    }
    return @res;
}

# здесь важно заливать только с мастера!
sub get_produced_sb_resources {
    my $host = shift || get_curr_host();
    my @res;
    while (my ($resource, $data) = each %resources) {
        next if !$data->{sandbox};
        if (is_master_producer($host, $data->{producer})) {
            push @res, $resource;
        }
    }
    return @res;
}

sub get_produced_tables {
    my $host = shift || get_curr_host();
    my @res;
    while (my ($tbl, $data) = each %tbl_producer) {
        if (grep { $_ and is_producer($host, $_) } $data->{producer}, $data->{aux_producer}) {
            push @res, $tbl;
        }
    }
    return @res;
}

# get_resource($res_name, %par)
# $res_name: название ресурса
# ключи %par:
#   host:  скачивать с указанного хоста (по умолчанию 0 - producer)
#   dir:   класть файлы в указанную директорию (по умолчанию 0 - кладет где взял)
#   bwlimit:  макс. скорость - опция для rsync (по умолчанию 70_000)
#   local_host:  хост потребителя (по умолчанию, локалхост)
#   rsync_add_opt:  дополнительные опции для rsync (массив строк вида ['-W','-B'])
# возвращает 0 или undef в случае ошибки, 1 иначе
sub get_resource {
    my $res_name = shift;
    my %par = (
        bwlimit => 70_000,
        @_,
    );
    print_err("get_resource... " . to_json(\%par));

    my $resource = $resources{$res_name};
    if (not defined $resource) {
        print_err("ERROR: get_resource '$res_name': resource not in BM::Resources!");
        return undef;
    }

    # определяем хост producer-а
    my $hostname = $par{local_host} || get_curr_host();
    my $producer_host;
    if (defined $par{host}) {
        $producer_host = $par{host};
    } else {
        $producer_host = get_resource_producer($hostname, $resource->{producer});
    }

    if (!defined $producer_host) {
        print_err("ERROR: get_resource '$res_name': can't determine producer for $hostname");
        return undef;
    } elsif ($producer_host eq $hostname) {
        print_err("WARN: get_resource '$res_name': host = producer_host");
        return 1;
    }

    print_err("get_resource $res_name $producer_host ...");

    my $sandbox_transfer = $producer_host eq "sandbox";
    my @transfer;
    if ($sandbox_transfer) {
        push @transfer, [ $res_name, $options->{dirs}{arcadia} ]
    }
    else {
        if ($resource->{files}) {
            for my $file (@{$resource->{files}}) {
                my ($dir, $name) = ($file =~ /^(.*)\/([^\/]*)$/);
                my $out_dir = $par{dir} ? $par{dir} : $dir;
                system "mkdir -p $out_dir" if ! -d $out_dir;
                push @transfer, [ $name, "$out_dir/$name" ];
            }
        }
        if ($resource->{dir}) {
            my ($name) = ($resource->{dir} =~ /^.*\/([^\/]*)$/);
            $name .= '_directory';
            my $out_dir = $par{dir} ? $par{dir} : $resource->{dir};
            system "mkdir -p $out_dir" if ! -d $out_dir;
            push @transfer, [ $name.'/*', $out_dir ];
        }
    }

    unless (@transfer) {
        print_err("WARN: void transfer for $res_name");
    }

    my @errors;
    my @tmpdirs;
    for my $data (@transfer) {
        my ($name, $output) = @$data;
        # скачиваем
        my $min_size = $resource->{min_bytes};
        my $tmpdir = $options->{dirs}->{temp} . '/rsync-temp-'. ($name =~ s/\W//gr);
        system "rm -r $tmpdir" if -d $tmpdir;
        system "mkdir -p $tmpdir";
        push @tmpdirs, $tmpdir;
        my @rsync_args;
        if ($sandbox_transfer) {
            @rsync_args = (
                $options->{dirs}{scripts}."/get-sandbox-resource/get-sandbox-resource",
                "--resource_name=$name",
                "--arcadia_root=$output",
                "--temp_dir=$tmpdir",
            );
        } else {
            @rsync_args = (
                'rsync',
                '--copy-links',
                '--bwlimit='.$par{bwlimit},
                '--timeout=120',
                '--contimeout=120',
                '--times',  # preserve modification times
                '--update', # skip files that are newer on the receiver
                '--temp-dir=' . $tmpdir,
            );
            push @rsync_args, "-r" if $resource->{recursive};
            push @rsync_args, "--min-size=$min_size" if $min_size;
            push @rsync_args, @{$par{rsync_add_opt} // []};

            push @rsync_args, "rsync://$producer_host:/bmexport/$name";
            push @rsync_args, $output;
        }
        my $rsync_cmd = join(" ", @rsync_args);
        my $res = Utils::Sys::do_sys_cmd($rsync_cmd, timeout => 24*3600, kill_after => 60, no_die => 1, no_error => 1, );
        next if $res;

        if ($sandbox_transfer) {
            print_err("WARNING: sky ($producer_host, $name) failed (".($?>>8)."): $!, can't fetch resource!");
        } else {
            print_err("WARNING: rsync ($producer_host, $name) failed (".($?>>8)."): $!, fall back to IPv4");
            # возможно, не сработал ipv6, форсируем ipv4
            # или наоборот, если до этого был указан ipv4
            if (grep {$_ eq '-4'} @rsync_args) {
                @rsync_args = grep {$_ ne '-4'} @rsync_args;
                push @rsync_args, '-6';
            } else {
                 push @rsync_args, '-4';
            }
            $rsync_cmd = join(" ", @rsync_args);
            $res = Utils::Sys::do_sys_cmd($rsync_cmd, timeout => 24*3600, kill_after => 60, no_die => 1, no_error => 1, );
            next if $res;
            print_err("WARNING: rsync ($producer_host, $name, -4) failed (".($?>>8)."): $!, can't fetch resource!");
        }

        push @errors, $name;
        next;
    }
    #TODO implement File::Temp::tmpdir
    system "rm -r $_" foreach grep { $_ && -d $_ } @tmpdirs;
    if (@errors) {
        print_err("WARNING: failed ($producer_host): " . join(" ", @errors));
        return 0;
    } else {
        return 1;   # OK
    }
}


# определение хоста-производителя
# параметры:
#   consumer - потребитель (хост)
#   producer - производитель (хост или роль. Роль может быть вида "роль:master" )
# логика выбора производителя:
#   если producer - хост (есть в списке get_hosts()), берем его
#   иначе, пытаемся взять хост-производитель со значением флага resources в соотв. с resources_priority(consumer) || resources(consumer)
#   иначе, берем где получится
sub get_resource_producer {
    my ($consumer, $producer) = @_;

    return "sandbox" if ($producer eq "sandbox");

    # случай, когда производитель - хост
    my %is_host = map { $_ => 1 } get_hosts();
    if ($is_host{$producer}) {
        return $producer;
    }

    # берем хост с нужным флагом resources, в порядке, установленным resource_priority
    my $consumer_info = get_host_info($consumer) || {};
    my @priority = ();
    if ($consumer_info->{resources_priority}) {
        @priority = @{$consumer_info->{resources_priority}};
    } elsif ($consumer_info->{resources}) {
        @priority = ($consumer_info->{resources});
    };

    my ($role, $master_flag) = split /:/, $producer;
    my %hosts_prm = (
        role => $role,
        (($master_flag // '') eq 'master'  ?  (master => 1) : ()),
    );

    my $cur_host = get_curr_host();
    my $group = get_host_group($cur_host);
    for my $prm (
        (map { { resources => $_ } } @priority),
        ($group ? { group => $group } : ()),
        { },
    ) {
        my @hosts = get_hosts(%hosts_prm, %$prm);
        return $cur_host   if grep { $_ eq $cur_host } @hosts;
        return $hosts[0] if @hosts;
    }

    return undef;
}

sub update_resource_in_sandbox {
    my $proj = shift;
    my $res_name = shift;
    my $resource = $resources{$res_name};
    my %spec = (
        type => 'BM_GENDICT',
        attrs => { sub_type => $res_name },
    );
    my %par;
    if (ref($resource->{sandbox})) {
        for my $k (qw(min_time max_time)) {
            my $v = $resource->{sandbox}{$k};
            $par{$k} = $v if defined $v;
        }
    }
    my @files = @{$resource->{files}};
    $proj->sandbox_client->update_tgz_resource(\@files, $res_name, \%spec, %par);
}

1;
