package BM::SandboxClient;

use strict;
use utf8;
use open ':utf8';

use base qw(ObjLib::ProjPart);

use Cwd qw(abs_path getcwd);
use Time::Local qw(timegm);
use File::Copy qw/ /;
use File::Slurp qw(read_file);
use File::Temp;
use File::Basename;
use JSON;
use LWP::UserAgent;
use URI;
use Utils::CompileTime;
use Utils::Sys;

__PACKAGE__->mk_accessors(qw(
    api_url
    proxy_url
    temp_dir
    work_dir
));

use constant DEFAULT_TRIES => 5;
use constant DEFAULT_SLEEP => 60;

sub get_resource_info {
    my $self = shift;
    my %opts = @_;

    my $user_agent = LWP::UserAgent->new;
    my $url = URI->new($self->api_url . '/resource');
    $url->query_form(%opts);
    my $response;
    Utils::Sys::do_safely(
        sub {
            $response = $user_agent->get($url);
            die($response->code . ': ' . $response->message) if !$response->is_success;
        },
        tries               => $opts{'tries'} // DEFAULT_TRIES,
        sleep_between_tries => $opts{'sleep_between_tries'} // DEFAULT_SLEEP,
    );

    return JSON::from_json($response->decoded_content);
}

sub download_resource {
    my $self = shift;
    my $resource_id = shift;
    my $file = shift;
    my %opts = @_;

    my $resource_info = $self->get_resource_info(
        limit   => 1,
        id      => $resource_id,
    )->{'items'}[0];
    my $resource = $resource_info->{'type'};

    return if $resource_id <= $self->get_current_resource_id($resource) && -e($file);

    my $user_agent = LWP::UserAgent->new;
    my $url = URI->new($self->proxy_url);
    $url->path($resource_id);
    $url->query_form(%opts);
    my $temp_file = Utils::Sys::get_tempfile('', DIR => $self->temp_dir, UNLINK => 1);
    $self->proj->log("started download $resource");
    my $response;
    Utils::Sys::do_safely(
        sub {
            $response = $user_agent->get($url, ':content_file' => $temp_file);
            die($response->code . ': ' . $response->message) if !$response->is_success;
        },
        tries               => $opts{'tries'} // DEFAULT_TRIES,
        sleep_between_tries => $opts{'sleep_between_tries'} // DEFAULT_SLEEP,
    );
    $self->proj->log("finished dowload $resource");
    if ($opts{'stream'} eq 'tgz') {
        my $unpacked_temp_file = Utils::Sys::get_tempfile('', DIR => $self->temp_dir, UNLINK => 1);
        Utils::Sys::do_sys_cmd("gunzip -c $temp_file | tar xf - -O > $unpacked_temp_file");
        File::Copy::move($unpacked_temp_file, $temp_file) or die($!);
    }
    File::Copy::move($temp_file, $file) or die($!);
    $self->set_current_resource_id($resource, $resource_id);
}

sub get_current_resource_id {
    my $self = shift;
    my $resource = shift;

    my $resource_file = $self->work_dir . "/$resource";
    return 0 if !-e($resource_file);

    open(my $fh, '<', $resource_file) or die($!);
    my $resource_id = <$fh>;
    chomp($resource_id);
    close($fh) or die($!);

    return $resource_id;
}

sub set_current_resource_id {
    my $self = shift;
    my $resource = shift;
    my $resource_id = shift;

    my $resource_file = $self->work_dir . "/$resource";
    open(my $fh, '>', $resource_file) or die($!);
    print $fh "$resource_id\n";
    close($fh) or die($!);
}

sub get_token {
    my $self = shift;

    unless (defined $self->{token}) {
        $self->{token} = read_file($self->{token_path});
        chomp $self->{token};
    }
    
    return $self->{token};
}

# параметры:
#   $work_dir, $file_names  -  директория и список имён файлов для заливки
#   $spec   -  описание ресурса, хэш с полями: ttl => ..., attrs => ...
sub upload {
    my $self = shift;
    my $file_names = shift;
    my $spec = shift;

    local $ENV{OAUTH_TOKEN} = $self->get_token;
    my $ttl = $spec->{ttl} // 365;
    my $ya = $self->proj->options->{ya_path};

    my @command = (
        "$ya upload",
        @$file_names,
        '--type', $spec->{type},
        '--user', $self->{user},
        '--owner', $self->{owner},
        $spec->{description} ? ('--description', '"'.$spec->{description}.'"') : (),
        '--ttl', $ttl,
        '--token', '$OAUTH_TOKEN',
    );
    my %attrs = %{$spec->{attrs} // {}};
    while (my ($k, $v) = each %attrs) {
        push @command, ('-A', "$k=$v");
    }
    if (!$self->{debug_mode}) {
        $self->do_sys_cmd("@command");
    } else {
        $self->log("Debug mode! Will not upload: @command");
    }
}

# пакуем и перезаливаем ресурс, если требуется (пришло время, изменился md5, ...)
# параметры:
#   $files:  список файлов
#   $archive_name:  файл будет называться $archive_name.tgz
#   $spec:   описание ресурса, href с полями:
#       type => тип ресурса
#       attrs => хэш атрибутов; они сохраняются в ресурсе, и по ним определяется последний ресурс!
#       ttl => subj
#       ...
# доп. параметры:
#   min_time =>    минимальное кол-во секунд между обновлением ресурса (default: 3600)
#   max_time =>    максимальное (default: 3600 * 24 * 30)
sub update_tgz_resource {
    my $self = shift;
    my $files = shift;
    my $archive_name = shift;
    my $spec = shift;
    my %par  = (
        min_time => 3600,
        max_time => 3600 * 24 * 30,
        @_,
    );

    # first, we check creation time
    my $type = $spec->{type};
    my $attrs_json = JSON::to_json($spec->{attrs} // {});
    my $proj = Project->new;
    my $inf = $self->get_resource_info(
        limit => 1,
        type => $type,
        order => '-created',  # ищем последний
        attrs => $attrs_json,
        state => 'READY',
    );
    $self->log("Update SB resource $type with attrs: $attrs_json ...");
    my $last_sbr;
    my $last_age;
    if ($inf and $inf->{items} and @{$inf->{items}}) {
        $last_sbr = $inf->{items}[0];
        $self->proj->logger->info('last resource info:', $last_sbr);
        my $created_time = Utils::Sys::time_yt2unix($last_sbr->{time}{created}) or die "Wrong time format!";
        $last_age = time() - $created_time;
        if ($last_age < $par{min_time}) {
            $self->log("Resource update not required: updated $last_age seconds ago!");
            return;
        }
    } else {
        $self->log("Can't find previous version of resource!");
    }

    my $rev = Utils::CompileTime::get_revision();
    if ($last_sbr and $last_sbr->{attributes}{svn_revision} > $rev) {
        die "Resource $type has newer svn_revision, something went wrong!";
    }

    my $temp = File::Temp::tempdir("output.$$.XXXX", DIR => $self->temp_dir, CLEANUP => 1);
    my $file_tgz = "$temp/$archive_name.tgz";
    Utils::Sys::pack_files_from_arcadia_root($files, $file_tgz);
    if ($last_sbr and $last_age < $par{max_time}) {
        # проверка по md5
        my $md5 = $self->proj->file($file_tgz)->md5_hex;
        if ($last_sbr->{md5} eq $md5) {
            $self->log("Resource update not required: equal md5: $md5");
            return;
        }
    }
    $spec->{attrs} //= {};
    $spec->{attrs}{svn_revision} = $rev;
    $self->upload( ["$file_tgz"], $spec);
}

1;
