package BM::SolomonClient;

use strict;
use utf8;

use BaseProject;
use Utils::Hosts qw(
    get_curr_host
    get_host_role
    is_cur_host_in_cloud
    get_host_datacenter
    is_cur_host_in_deploy
    get_deploy_project_id
    get_deploy_stage_id
    get_deploy_unit_id
    get_deploy_workload_id
);
use Utils::Sys qw(print_err);
use HTTP::Request;
use JSON::XS qw(encode_json);
use LWP::UserAgent;


sub new {
    _new_instance(@_);
}

sub _new_instance {
    my ($class, %options) = @_;
    return bless {
        %{ $Utils::Common::options->{SolomonClient_params} },
        %options,
    }, $class;
};


# Отправляем сенсор в Соломон-агент
sub push_single_sensor {
    my ($self, $sensor_params) = @_;

    # Проверяем наличие необходимых ключей
    for my $must_init_key ("cluster", "service", "sensor", "value") {
        if (! defined $sensor_params->{$must_init_key}) {
            print_err("ERROR: invalid configuration of Solomon sensor!");
            return 1;
        }
    }

    # Преобразовываем сенсора из понятного нам формата в формат агента
    my $agent_sensor = {
        labels => {
            project_id => "irt",
            cluster_id => $sensor_params->{"cluster"},
            service_id => $sensor_params->{"service"},
            sensor     => $sensor_params->{"sensor"},
        },
        value  => 1.0 * $sensor_params->{"value"},
    };

    if (defined $sensor_params->{project}) {
        $agent_sensor->{labels}->{project_id} = $sensor_params->{project};
    }

    if (defined $sensor_params->{ts_datetime}) {
        $agent_sensor->{ts} = $sensor_params->{ts_datetime};
    }

    if (defined $sensor_params->{labels}) {
        my $init_labels = $agent_sensor->{labels};
        my $user_labels = $sensor_params->{labels};
        $agent_sensor->{labels} = {%$init_labels, %$user_labels};
    }
    if ((! exists $agent_sensor->{labels}->{datacenter}) && (exists $agent_sensor->{labels}->{host})) {
        my $dc = get_host_datacenter($agent_sensor->{labels}->{host});
        if ($dc) {
            $agent_sensor->{labels}->{datacenter} = $dc;
        }
    }
    if (is_cur_host_in_deploy()) {
        $agent_sensor->{labels}->{deploy_project_id} = get_deploy_project_id();
        $agent_sensor->{labels}->{deploy_stage_id} = get_deploy_stage_id();
        $agent_sensor->{labels}->{deploy_unit_id} = get_deploy_unit_id();
        $agent_sensor->{labels}->{deploy_workload_id} = get_deploy_workload_id();
    }

    my $sensors_json = encode_json({sensors => [$agent_sensor]});

    # Отправляем сенсор в агент (с 3-х попыток)
    my $ua = LWP::UserAgent->new();
    my $req = HTTP::Request->new(POST => $self->{push_host} . ":" . $self->{agent_port});
    $req->content_type('application/json');
    $req->content($sensors_json);

    my $total_attempts = 3;
    my $attempt_num = 0;
    while ($attempt_num < $total_attempts) {
        if ($attempt_num > 0) {
            print_err("Pushing to Solomon: attempt number " . $attempt_num);
        }

        my $res = $ua->request($req);
        if ($res->is_success) {
            print_err("Successful send to Solomon:'${sensors_json}'");
            return 0;
        }
        $attempt_num += 1;
    }

    # Не было ни одного успешного запроса - возвращаем ненулевой код.
    return 1;
};


# Отправляем в Соломон метку, о том, что скрипт завершился успешно
sub set_success_script_finish {
    my ($self, $script_name) = @_;

    my $labels = {
        host        => get_curr_host(),
        script_name => $script_name,
    };
    if (is_cur_host_in_cloud()) {
        $labels->{host_role} = get_host_role();
    }

    my $push_res = $self->push_single_sensor(
        {
            cluster => "host_info",
            service => "scripts",
            sensor  => "success_finish",
            value   => 1,
            labels  => $labels,
        }
    );

    if ($push_res != 0) {
        print_err("ERROR: Can't send Solomon sensor for success finish of the script.")
    }
}

1
