package BM::YQL::Client;
use strict;
use base qw(Exporter);

use Utils::Common;

use Utils::Sys qw(
    do_safely
    do_sys_cmd
    get_bash_cmd
    read_sys_cmd
);

use utf8;
use open ":utf8";

binmode STDIN, ':utf8';
binmode STDOUT, ':utf8';

use constant DEFAULT_YQL_TIMEOUT_HOUR => 6;

our @EXPORT_OK = (
    'make_yql_request',
    # Выполняет YQL запрос. Возвращает информацию о таблицах, в которые записан результат:
    #
    # my @yql_result = (
    #     {
    #         yt_cluster => 'philosopher'  # имя кластера YT: 'hahn', 'banach', 'hume', etc
    #         table_path => '//path/to/table'  # путь на ноде ко временной таблице, в которую записан результат
    #     },
    #     ...
    # )
    #
    # Для дальнейшей работы с таблицами с результатами поспользуйтесь BM::YT::Client / $proj->yt_client()
);

=head1
Arguments:
    request =text of YQL request
Keyword Arguments:
    $token => token of YQL user
Simple examples of usage:
    $request = "select answer from hahn.[table_name] where answer = 42;";
    @info_about_tables = make_yql_request($request);
    for my $table_info (@info_about_tables) {
        download_table_from_yt_or_do_something_else($table_info->{table_path});
    }
    If you want to pass token:
        $token = "MyToken123";
        @tables = make_yql_request($request, token => $token);
=cut

sub make_yql_request {
    my ($request_str, %options) = @_;

    my $token;
    $token = $options{token};
    if (defined ($token)) {
        $token = "--token $token";
    } else {
        $token = "";
    }

    my $timeout_hour = "--timeout_hour " .($options{timeout_hour} // DEFAULT_YQL_TIMEOUT_HOUR);

    my $python_script = "$Utils::Common::options->{'dirs'}{'scripts'}/yql/make_yql_request/make_yql_request";

    my $temp_file_name = Utils::Sys::get_tempfile("temp_file_for_request", UNLINK => 1);
    open my $temp_file, ">", $temp_file_name or die("Can't open temp_file $temp_file_name for read. Error is ($!)");
    print $temp_file $request_str;
    close $temp_file;

    my $cmd = "$python_script $timeout_hour <$temp_file_name;";
    my $res_str = read_sys_cmd($cmd);

    my @info_about_tables;
    for my $line (split("\n", $res_str)) {
        chomp $line;
        my ($yt_cluster, $table_path) = split("\t", $line);
        $table_path = "//" . $table_path;
        my $table_info = {yt_cluster => $yt_cluster, table_path => $table_path};
        push @info_about_tables, $table_info;
    }

    return @info_about_tables;
}

sub __table2url {
    my ($yt_cluster, $table, $columns) = @_;
    $table =~ s/\//%2F/g;
    my $url = "https://$yt_cluster.yt.yandex-team.ru/api/v3/read_table?path=$table&output_format[\$value]=schemaful_dsv&output_format[\$attributes][missing_value_mode]=fail&output_format[\$attributes][enable_column_names_header]=true&dump_error_into_response=true";
    for my $column (@$columns) {
        $url .= "&output_format[\$attributes][columns][]=$column";
    }

    $url =~ s/\[/%5B/g;
    $url =~ s/\]/%5D/g;
    $url =~ s/\$/%24/g;

    return $url;
}

1;
