package BM::YQL::Helpers;

use strict;
use warnings;

use utf8;
use open ":utf8";

binmode STDIN, ':utf8';
binmode STDOUT, ':utf8';

use base qw(Exporter);

use Utils::Sys qw(print_log print_err);
use Utils::Common;
use BM::YQL::Client qw(make_yql_request);

our @EXPORT_OK = (
    'get_k_random_items',
    'get_k_random_banners',
);


sub get_k_random_items {
    my %opts = @_;

    my $yt_client = $opts{yt_client} // die "provide yt_client";
    my $yt_cluster = $opts{yt_cluster} // $yt_client->params()->{cluster};

    my $table_name = $opts{table_name} // die "provide table_name";
    my $id_field = $opts{id_field} // die "provide id_field";
    my $count = $opts{count} // die "provide count";
    my $columns = $opts{columns};
    my $extra_condition = $opts{extra_condition};

    print_err "Get $count random rows from table '$table_name'...";

    if (defined $columns) {
        # сейчас YQL не поддерживает запись вида "select ..., some_field as alias where Random(alias) < 100500 order by Random(alias)"
        # и некоторых других логичных способов записать эту мысль тоже не поддерживает
        # поэтому, чтобы мочь обращаться к $id_field и из where, и из order by, нам нужно, чтобы он был перечислен в select буквально, без алиаса
        # перечислять его без алиаса два раза нельзя, поэтому мы проверяем, перечислен ли он в $columns, и если нет - добавляем
        # если когда-нибудь в будущем YQL это поправит, лучше заменить на обращение по alias'у
        # breqwas@, 09.10.2017

        $columns .= ", `$id_field`" unless $columns =~ m/(^|,)\s*$id_field\s*(,|$)/;
    } else {
        $columns = "*";
    }

    my $total_row_count;
    if (defined $extra_condition) {
        print_err "Extra condition provided, scan table to determine correct total_row_count";

        my ($yql_result) = make_yql_request("
            select count(*) as FilteredRowCount
            from $yt_cluster.`$table_name`
            where ($extra_condition)
        ");

        my ($selected_row) = @{ $yt_client->read_table($yql_result->{table_path}) };
        die "No result while scanning '$table_name' with extra condition" unless $selected_row && $selected_row->{FilteredRowCount};

        $total_row_count = $selected_row->{FilteredRowCount};
    } else {
        print_err "No extra condition provided, use table attribute as total_row_count";

        $extra_condition = "2*2 == 4";
        $total_row_count = $yt_client->get_attribute($table_name, "row_count");

        die "Couldn't get row count from '$table_name'" unless defined $total_row_count;
    }

    my $yql = "
        select $columns
        from $yt_cluster.`$table_name`
        where
            Random(`$id_field`) < ( 3.0 * cast($count as Double) / cast($total_row_count as Double) )
            and ($extra_condition)
        order by Random(`$id_field`)
        limit $count
    ";

    # 'where' в этом запросе нужен для того, чтобы при выполнении запроса YT сортировал не таблицу $table_name целиком, а выборку из неё
    # теоретически, при выборе малого количества строк есть небольшая вероятность получить меньше строк, чем запросили
    # на практике, функция используется для получения больших выборок из очень больших таблиц, и эту вероятность можно в расчёт не брать

    my ($res) = make_yql_request($yql);

    return $res;
}

sub get_k_random_banners {
    my %opts = @_;

    my $yt_client = $opts{yt_client} // die "provide yt_client";
    my $banner_count = $opts{banner_count} // die "provide banner_count";
    my $banners_table = $opts{banners_table} // "//home/catalogia/banners_extended";
    my $lang = $opts{lang};
    my $category_id = $opts{category_id};
    my $minicateg_id = $opts{minicateg_id};
    my $extra_condition = $opts{extra_condition};

    print_err "Get $banner_count random banners...";

    my @fields_mapping = @{ $Utils::Common::options->{yt_direct_banners_mapping} };
    #временный костыль, пока не будет полноценной поддержки title_extension везде
    push @fields_mapping, { yt_field => 'title_extension', mysql_field => 'title_extension' };

    my $columns = join(", ", map { "`$_->{yt_field}` as `$_->{mysql_field}`" } @fields_mapping);

    my @extra_conditions;
    push @extra_conditions, $extra_condition if defined $extra_condition;
    push @extra_conditions," lang = '$lang'" if defined $lang;
    push @extra_conditions, "CategoryIDs regexp '(^|,)$category_id(,|\$)'" if defined $category_id;
    push @extra_conditions, "minicategs_ids regexp '(^|,)$minicateg_id(,|\$)'" if defined $minicateg_id;
    push @extra_conditions, "title is not null and body is not null";  # в banners_extended такие попадаются

    $extra_condition = join(" and ", map { "( $_ )" } @extra_conditions) if scalar @extra_conditions;

    my $yql_result = get_k_random_items(
        yt_client => $yt_client,
        table_name => $banners_table,
        id_field => "bid",
        count => $banner_count,
        extra_condition => $extra_condition,
        columns => $columns,
    );

    print_err "Get $banner_count random banners done, results are in '$yql_result->{table_path}' at cluster '$yql_result->{yt_cluster}'";

    return $yql_result;
}

