package BM::YT::Client;

use strict;

use utf8;
use open ':utf8';

use base qw(ObjLib::ProjPart);

use JSON;
use Data::Dumper;
use B::Deparse;
use Digest;
use Time::Piece;

use Utils::Sys;
use IPC::Open2;
use YtProjectLaunchHelper;

=head1 AUTHORS

<F<serkh@yandex-team.ru>>
<F<malykhin@yandex-team.ru>>

=cut

#
# Общие операции для работы с YT
#
__PACKAGE__->mk_accessors(qw(
    temp_dir
    work_dir
    raw_data_dir
    bin
    params
    __json
));

sub init {
    my $self = shift;
    $self->{__json} = JSON->new->allow_nonref;
}

# команда yt с необходимыми настройками
sub get_cmd {
    my $self = shift;
    return join(' ',
        'YT_PROXY=' . $self->params->{'proxy'},
        'YT_TOKEN_PATH=' . $self->params->{'token_path'},
        'YT_RETRY_READ=1',
        $self->params->{'pool'} ? 'YT_POOL=' . $self->params->{'pool'} : (),
        $self->bin,
    );
}
sub do_cmd {
    my $self = shift;

    # указать дефолтную транзакцию, если она существует и явно не указана другая
    my $tx = ($self->{default_transaction} &&
              scalar(grep { $_ =~ /\-\-tx[\=\ \$]/ } @_) == 0
             ) ? "--tx=".$self->{default_transaction} : '';
	# tx перед аргументами чтобы корректно отрабатывал пайплайн (если его передают, например "yt read ... | grep 'abc'")
    my $command = join(' ', $self->get_cmd, $tx, @_);
    Utils::Sys::do_safely(
        sub {
            $self->proj->do_sys_cmd($command);
        },
        tries               => $self->params->{'tries'},
        sleep_between_tries => $self->params->{'sleep_between_tries'},
    );
}

sub read_cmd {
    my $self = shift;
    my $cmd = shift;
    my $cmd_params = shift // {};

    # указать дефолтную транзакцию, если она существует и явно не указана другая
    my $tx = ($self->{default_transaction} &&
              $cmd !~ /\-\-tx[\=\ \$]/
             ) ? "--tx=".$self->{default_transaction} : '';
	# tx перед аргументами чтобы корректно отрабатывал пайплайн (если его передают, например "yt read ... | grep 'abc'")
    my $command = join(' ', $self->get_cmd, $tx, $cmd) .'; true';
    my $result = '';
    Utils::Sys::do_safely(
        sub {
            $result = $self->proj->read_sys_cmd($command);
        },
        tries               => $self->params->{'tries'},
        sleep_between_tries => $self->params->{'sleep_between_tries'},
        timeout             => $cmd_params->{'timeout'} // 0,
    );

    return $result;
}

sub do_perl_cmd {
    my $self = shift;
    my $yt_cmd = shift;
    my $perl_cmd = shift;
    my @args = @_;

    # можно ли полагаться на то, что tempfile отдаст относительный путь?
    my $reducer = $self->proj->get_tempfile('yt_perl', DIR => '.', UNLINK => 1);

    open my $fh, '>', $reducer
        or die "Can't open reducer file: $!";
    print $fh $perl_cmd."\n";
    close $fh;
    $self->do_cmd(join(' ',
        $yt_cmd,
        "'perl ./$reducer'",
        @args,
        "--local-file=$reducer",
    ));
}

sub do_project_cmd {
    my $self = shift;

    my $helper = YtProjectLaunchHelper->new({
        temp_dir => $Utils::Common::options->{'dirs'}{'temp'},
        resource_url => 'https://proxy.sandbox.yandex-team.ru/last/BROADMATCH_MR_CATALOGIA?attrs={%22production%22:%221%22}',

        # sandbox_token => $self->proj->sandbox_client->get_token(),
        # sandbox однажды отрубил анонимные запросы. Когда он их врубил, запросы с токеном работать почему-то перестали. Разбираться, почему они не работают, руки не дашли, просто отключили токен тогда. Можно разобраться, почему они не работают (если еще не работают), починить и раскомментировать.
        # apovetkin@, 15.02.2018

        sandbox_token => '',
    });

    #загружаем ресурс
    Utils::Sys::do_safely(
        sub {
            $self->proj->do_sys_cmd($helper->resource_download_cmd);
        },
        tries               => $self->params->{'tries'},
        sleep_between_tries => $self->params->{'sleep_between_tries'},
    );

    #исполняем команду с доп.параметрами
    $self->do_cmd(@_, $helper->additional_console_params);
}

sub set_params {
    my $self = shift;
    my %params = @_;

    for my $key (keys %params) {
        $self->params->{$key} = $params{$key};
    }

    return $self;
}

# create command from hash of parameters
# {ignore_existing => 1, recursive => 1} -> '--ignore-existing --recursive'
sub __params2cmd {
    my $self = shift;
    my $params = shift;

    my %boolean_parameters = map {$_ => 1} qw(
        ignore-existing recursive force waitable
    );
    my %key_val_parameters = map {$_ => 1} qw(
        tx wait-for timeout mode
    );
    # TODO other parameters

    my @cmds;
    while (my ($param_name, $param_value) = each %$params) {
        $param_name =~ s/ignore_existing/ignore-existing/;
        $param_name =~ s/wait_for/wait-for/;
        $param_name = 'tx' if $param_name eq 'transaction';
        if (defined $boolean_parameters{$param_name}) {
            push @cmds, "--$param_name" if $param_value;
        }
        elsif (defined $key_val_parameters{$param_name}) {
            push @cmds, "--$param_name=$param_value";
        }
        else {
            warn "unknown parameter: '$param_name'";
        }
    }
    return join(" ", @cmds);
}

=head1 COMMON METHODS

=cut

=head2 get_attribute

Get attribute of node

Arguments:
    path
    attribute

Return value:
    undef if attribute doesn't exist, scalar (string, arrayref or hashref) otherwise

=cut

sub get_attribute {
    my $self = shift;
    my $path = shift;
    my $attr = shift;

    # handle /path/to/table[:#10] syntax
    $path =~ s/\[.+\]$//;

    my $attr_path = "$path/\@$attr";
    return undef unless $self->exists($attr_path);

    my $cmd = "get $attr_path --format json 2>/dev/null";
    my $output = $self->read_cmd( $cmd, {'timeout' => 5} ) or return undef;
    my $json_output;
    eval { $json_output = $self->__json->decode($output); };
    if ($@) {
        return undef;
    } else {
        return $json_output;
    }
}

=head2 set_attribute

Set attribute for node

Arguments:
    path
    attribute
    value

Return value:
    success flag (0/1)
=cut

sub set_attribute {
    my $self = shift;
    my $path = shift;
    my $attr = shift;
    my $attr_value = shift;
    my %par = @_;

    # handle /path/to/table[:#10] syntax
    $path =~ s/\[.+\]$//;

    my $attr_path = "$path/\@$attr";
    return 0 unless $self->exists($path);
    my $attr_value_json = JSON->new->allow_nonref->encode($attr_value);

    my $cmd = qq~set $attr_path '$attr_value_json' --format json ~ . $self->__params2cmd(\%par);
    my $output = $self->do_cmd( $cmd ) or return 0;

    return 1;
}

=head2 create
Create Cypress object

Arguments:
    type
    path

Keyword arguments:
    ignore_existing => 1        do not fail if node exists
=cut

sub create {
    my $self = shift;
    my $type = shift;
    my $path = shift;
    my %par = (ignore_existing => 1, @_);

    my $command = qq~create $type "$path" ~ . $self->__params2cmd(\%par);
    $self->do_cmd($command);
}
=head2 mkdir

Create Cypress map node (directory)

Arguments:
    path

Keyword arguments:
    ignore_existing => 1        do not fail if node exists

=cut

sub mkdir {
    my $self = shift;
    my $path = shift;
    my %par = (ignore_existing => 1, @_);

    my $command = qq~create map_node "$path" ~ . $self->__params2cmd(\%par);
    $self->do_cmd($command);
}

=head2 create_table

Create Cypress table

Arguments:
    table

Keyword arguments:
    ignore_existing => 1        do not fail if node exists
    recursive => 1              create intermediate nodes if necessary

=cut

sub create_table {
    my $self = shift;
    my $table = shift;
    my %par = (recursive => 1, ignore_existing => 1, @_);

    my $command = qq~create table "$table" ~ . $self->__params2cmd(\%par);
    $self->do_cmd($command);
}

=head2 remove

Remove Cypress node

Arguments:
    node

Keyword arguments:
    recursive => 1              remove sub-nodes
    force => 1                  do not fail if node doesn't exist

=cut

sub remove {
    my $self = shift;
    my $node = shift;
    my %par = (recursive => 1, force => 1, @_);

    my $command = qq~remove "$node" ~ . $self->__params2cmd(\%par);
    $self->do_cmd($command);
}

=head2 exists

Check if node exists

Arguments:
    node

Return value:
    1/0

=cut

sub exists {
    my $self = shift;
    my $node = shift;

    my $exists = $self->read_cmd("exists '$node'", {'timeout' => 5}); chomp $exists;
    return $exists eq 'true' ? 1 : 0;
}

=head2 list

Sub-nodes of Cypress node

Arguments:
    path

Return value:
    list of names of sub-nodes (undef if node does not exist)

=cut

sub list {
    my $self = shift;
    my $path = shift;

    return undef unless $self->exists($path);
    my $out = $self->read_cmd("list '$path'"); chomp $out;
    return [split /\n/, $out];
}

=head2 get

Get Cypress node content

Arguments:
    path

Return value:
    hashref (recursive structure of all sub-nodes)

=cut

sub get {
    my $self = shift;
    my $path = shift;

    return undef unless $self->exists($path);
    my $out = $self->read_cmd("get '$path' --format json"); chomp $out;
    my $out_json;
    eval {
        $out_json = from_json($out);
    };
    return undef if $@;
    return $out_json;
}

=head2 move

Move node to another node

Arguments:
    src_node
    dst_node

Keyword arguments:
    recursive => 0/1
    force => 0/1

=cut

sub move {
    my $self = shift;
    my $src = shift;
    my $dst = shift;
    my %par = (force => 1, recursive => 1, @_);

    my $command = qq~move $src $dst ~ . $self->__params2cmd(\%par);
    $self->do_cmd($command);
}

sub __format_options2cmd {
    my $self = shift;
    my $format = shift;
    my $format_options = shift;

    if ($format eq 'dsv' || $format eq 'tskv') {
        return "--format dsv";
    } elsif ($format eq 'json') {
        my $encode_utf8 = $format_options->{encode_utf8} ? 'true' : 'false';
        return "--format '<encode_utf8=$encode_utf8>json'";
    } elsif ($format eq 'yamr') {
        my $has_subkey = $format_options->{has_subkey} ? 'true' : 'false';
        return "--format '<has_subkey=$has_subkey>yamr'";
    } elsif ($format eq 'schemaful_dsv') {
        my $columns = $format_options->{columns};
        if (!defined $columns || ref($columns) ne 'ARRAY') {
            warn "No columns provided for format '$format'!";
            return undef;
        }
        my $fmt = "<columns=[" . join(";", @$columns) . ";]>schemaful_dsv";
        return "--format '$fmt'";
    } else {
        warn "Unknown format: '$format', use as is";
        return "--format $format";
    }
}

=head2 write_table_from_file

Write into table from file

Arguments:
    filename
    table
    format      dsv/tskv/yamr

Keyword arguments:
    append => 0/1
    format_options => {
        For 'schemaful_dsv':
            columns => ['col_name_1', 'col_name_2', ...],
        For 'yamr':
            has_subkey => 0/1
    }

=cut

sub write_table_from_file {
    my $self = shift;
    my $filename = shift;
    my $table = shift;
    my $format = shift;
    my %par = @_;

    if ($par{append}) {
        $table = "<append=true>$table";
    }

    my $format_options = $par{format_options} // {};
    my @cmds = ("write '$table'");
    my $format_cmd = $self->__format_options2cmd($format, $format_options);
    die "Error in format: '$format'" unless defined $format_cmd;
    push @cmds, $format_cmd;
    push @cmds, "< $filename";

    $self->do_cmd(join(" ", @cmds));
}

=head2 write_tsv_file

Upload tab seprated file into table with specified columns.
If types is defined upload in such a way that YQL supports specified types.

Arguments:
    file
    table
    columns = ['col_name_1', 'col_name_2', ...]
    types   = ['col_type_1', 'col_type_2', ...]

=cut

sub write_tsv_file {
    my $self = shift;
    my $file = shift;
    my $table = shift;
    my $columns = shift;
    my $types = shift;
    my $pool = shift // 'catalogia';

    my $format = 'schemaful_dsv';
    my $format_options = {columns => $columns};
    my $format_text = $self->__format_options2cmd($format, $format_options);
    # TODO: make tmp dir as a parameter of client
    my $temp_table = '//home/catalogia/tmp/' . Digest::MD5::md5_hex($table);
    $self->do_cmd(join(' ', 'write', $temp_table, $format_text, "< $file"));
    if (@$types) {
        my @string_columns = map {$columns->[$_]} grep {$types->[$_] eq 'string'} (0..$#$types);
        my @int64_columns = map {$columns->[$_]} grep {$types->[$_] eq 'int64'} (0..$#$types);
        my @uint64_columns = map {$columns->[$_]} grep {$types->[$_] eq 'uint64'} (0..$#$types);
        my $int64_columns_regexp = join('|', @int64_columns);
        my $uint64_columns_regexp = join('|', @uint64_columns);
        my $perl_code = qq#
            while (my \$line = <STDIN>) {
                \$line =~ s/("($int64_columns_regexp)"=)"(\\d+)"/\$1\$3/g;
                \$line =~ s/("($uint64_columns_regexp)"=)"(\\d+)"/\$1\$3u/g;
                print \$line;
            }
        #;
        $self->do_perl_cmd('map', $perl_code,
           "--src=$temp_table",
           "--dst=$temp_table",
           "--format='<format=text>'yson",
           "--spec='{pool=$pool}'"
        );
        my @string_schema = map {'{"name"="' . $_ .'";"type"="string";};'} @string_columns;
        my @int64_schema = map {'{"name"="' . $_ .'";"type"="int64";};'} @int64_columns;
        my @uint64_schema = map {'{"name"="' . $_ .'";"type"="uint64";};'} @uint64_columns;
        my $read_schema = '<"strict"="false";>[' . join('', @string_schema, @int64_schema, @uint64_schema) . ']';
        my $set_schema_command = join(' ', "echo '$read_schema' |", $self->get_cmd(), 'set', "$temp_table/\@_read_schema");
        Utils::Sys::do_sys_cmd($set_schema_command);
    }
    $self->do_cmd(join(' ', 'move', '--force', '--no-preserve-account', $temp_table, $table));
    $self->set_upload_time($table);
}

sub lock {
    my ($self, $path, %par) = @_;

    die 'tx_id must be provided (OR default must be set)!' unless $par{tx_id} || $self->{default_transaction};

    my $cmd = qq~lock $path ~ . $self->__params2cmd(\%par) . qq~ 2>&1 ~;
    my $stderr = $self->read_cmd($cmd);
    if ($stderr =~ /lock is taken by concurrent transaction/) {
        # it is not critical error
        return 0;
    }
    if ($stderr =~ /error/i) {
        # this one is critical
        warn $stderr;
        return 0;
    }
    return 1;
}

=head2 read_table_to_file

Read from table to file

Arguments:
    table
    filename
    format      dsv/tskv/yamr

Keyword arguments:
    format_options => {
        For 'schemaful_dsv':
            columns => ['col_name_1', 'col_name_2', ...],
        For 'yamr':
            has_subkey => 0/1
    }

=cut
# key - скачать только строки с определенным значением ключа (по которому таблица отсортирована)
# range - скачать строки из диапазона переданных номеров
# additional_cmds - комманды для постобработки output'а ридера (например, grep или sed)
sub read_table_to_file {
    my $self = shift;
    my $table = shift;
    my $filename = shift;
    my $format = shift;
    my %par = @_;

    my $format_options = $par{format_options} // {};
    my $index = '';
    if ($par{key}) {
        $index = "[\"".$par{key}."\"]";
    } elsif ($par{range}) {
        $index = '['.$par{range}.']';
    }
    my @cmds = ("read '$table$index'");
    my $format_cmd = $self->__format_options2cmd($format, $format_options);
    die "Error in format: '$format'" unless defined $format_cmd;
    push @cmds, $format_cmd;
    # параллельное чтение
    if ($par{parallel_num}) {
        push @cmds, " --config \"{read_parallel={enable=%true;max_thread_count=$par{parallel_num};}}\" ";
    }
    if ($par{additional_cmds}) {
        push @cmds, '|'.join('|', @{$par{additional_cmds}});
    }
    push @cmds, "> $filename";

    $self->do_cmd(join(" ", @cmds));
}

# Читаем YT-таблицу в память.
# Для определенных последовательностей строк из-за бага в Encode неправильно считается длина строки, и она считается отрицательной.
# В связи с этим YT-данные не читаем сразу в память и не делаем сплит считанной строки по "\n" (см. https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=835990).
# Для этого вначале используем чтение YT-данных во временный файл.
sub read_table {
    my $self = shift;
    my $table = shift;
    my $format = shift // "json";

    my $tmpfile = $self->proj->get_tempfile('yt_data_tmp', DIR => '.', UNLINK => 1);
    $self->read_table_to_file($table, $tmpfile, $format);

    my $res = [];
    open my $fin, '<', $tmpfile or die "Can't open tmp file for the reading of YT-data: $!";
    while (<$fin>) {
        push $res, JSON::from_json($_);
    }
    close $fin;
    unlink $tmpfile;

    return $res;
}


sub __generate_map_script_code {
    my $self = shift;
    my $subroutine = shift;
    my %par = @_;

    my $deparse = B::Deparse->new("-sC");
    my $code = $deparse->coderef2text($subroutine);

    # default prepare function uses fields from JSON
    my $prepare = sub {
        my $line = shift;
        return from_json($line);
    };
    my %meta_format2fields;
    if ($par{src}) {
        for my $table (@{$par{src}}) {
            my $bs_meta = $self->get_bs_meta($table);
            if ($bs_meta) {
                my $__parse_field = sub {
                    my $field = shift;
                    return [map {[split ":", $_]} split ",", $field];
                };
                for my $meta_item (@$bs_meta) {
                    my ($key, $subkey, $value) = split /\\t/, $meta_item->[2];
                    $meta_format2fields{$meta_item->[0]}{key} = $__parse_field->($key);
                    $meta_format2fields{$meta_item->[0]}{subkey} = $__parse_field->($subkey);
                    $meta_format2fields{$meta_item->[0]}{value} = $__parse_field->($value);
                }
            }
        }
    }
    my $meta_format2fields_str;
    {
        local $Data::Dumper::Indent = 0;
        $meta_format2fields_str = Data::Dumper->Dump([\%meta_format2fields], [qw(*meta_format2fields)]);
    }
    if (keys %meta_format2fields) {
        # (format => {key => [[field1_name, field1_type], [field2_name, field2_type], ... ], subkey => [[...], [...], ...]}, value => [[...], [...], ...]})
        # redefine prepare code
        # %meta_format2fields is visible in mapper script!
        $prepare = sub {
            my $line = shift;
            my $rec = from_json($line);
            my %out_rec;

            my @key_fields = split ",", $rec->{key}, -1;
            my @subkey_fields = split ",", $rec->{subkey}, -1;
            my @value_fields = split /\t/, $rec->{value}, -1;

            # there should be only 1 field in subkey: FORMAT
            if (scalar(@subkey_fields) != 1) {
                die "wrong subkey: ".join(",", @subkey_fields);
            }
            my $format_id = $subkey_fields[0];
            my $format_fields = $meta_format2fields{$format_id};
            if (!defined $format_fields) {
                die "unknown format: $format_id; avaiable formats: ".join(",", keys %meta_format2fields);
            }

            my @key_format = @{ $format_fields->{key} };
            my @value_format = @{ $format_fields->{value} };
            if (scalar(@key_format) != scalar(@key_fields)) {
                die "wrong number of fields in key";
            }
            if (scalar(@value_format) != scalar(@value_fields)) {
                die "wrong number of fields in value";
            }

            my @all_formats = (@key_format, @value_format);
            my @all_fields = (@key_fields, @value_fields);
            for my $i (0..$#all_fields) {
                my $field = $all_fields[$i];
                my ($name, $type) = @{ $all_formats[$i] };
                if ($type eq 'int' or $type eq 'long' or $type eq 'float') {
                    $field = $field + 0;  # "cast" to a number
                }
                $out_rec{$name} = $field;
            }

#            %out_rec = (Options => "mobile-app", "BannerID" => 42, "OrderID" => 0);
            return \%out_rec;
        };
    }
    my $prepare_code = $deparse->coderef2text($prepare);
    $prepare_code =~ s/\s*package .+?;//;

    my $total_code = "#!/usr/bin/perl
use strict;
use utf8;
use open ':utf8';
use v5.10;
use JSON;
binmode STDIN, ':utf8';
binmode STDOUT, ':utf8';

my $meta_format2fields_str
my \$prepare = sub $prepare_code;

my \$map = sub $code;
while(<STDIN>) {
    chomp;
    my \$in_record = \$prepare->(\$_);
    my \$out_records = \$map->(\$in_record) // [];
    for my \$out_record (\@\$out_records) {
        say STDOUT to_json(\$out_record);
    }
}
";
    #print $total_code;
    return $total_code;
}

=head2 do_map

Run map operation

Arguments:
    src             source table(s) (string or arrayref)
    dst             table
    subroutine      function with hasref as input and arrayref of hashrefs as output

Keyword arguments:

=cut

sub do_map {
    my $self = shift;
    my $src = shift;
    my $dst = shift;
    my $subroutine = shift;
    my %par = @_;

    my $code = $self->__generate_map_script_code($subroutine, src => ref($src) eq 'ARRAY' ? $src : [$src]);
    my $code_file = $self->proj->get_tempfile('yt_client_mapper', UNLINK => 1);
    my $code_filename = pop [split "/", $code_file];
    open my $tF, '>', $code_file or die $!;
    print $tF $code;
    close $tF or die $!;
    my $format_cmd = '--format json';
    my $cmd = join(" ",
        "map",
        "'perl $code_filename'",
        ref($src) eq 'ARRAY' ? map {"--src '$_'"} @$src : "--src '$src'",
        "--dst '$dst'",
        "--local-file $code_file",
        $format_cmd,
    );
    $self->do_cmd($cmd);
}

=head2 get_node_unixtimes

Return hash { access_time => ..., creation_time => ..., modification_time => ...} with unixtimes as values

Arguments:
    node

=cut
sub get_node_unixtimes {
    my $self = shift;
    my $node = shift;
    return undef unless $self->exists($node);

    my %times;
    for my $attr (qw(access_time creation_time modification_time)) {
        my $raw_time = $self->get_attribute($node, $attr);
        next unless defined $raw_time;
        $times{$attr} = Utils::Sys::time_yt2unix($raw_time);
    }

    return \%times;
}


=head1 BS UTILS

=cut

=head2 get_bs_meta

Get meta-information about table

Arguments:
    table

Return:
    arrayref of (key, subkey, value) triples

=cut

sub get_bs_meta {
    my $self = shift;
    my $table = shift;
    my %par = @_;

    my $meta = $self->get_attribute($table, '_yql_read_udf_type_config');
    return undef unless defined $meta;

    my $meta_arr = from_json($meta);
    # in attributes, subkey is written last...
    my @correctly_ordered_meta = map {[$_->[0], $_->[2], $_->[1]]} @$meta_arr;

    return \@correctly_ordered_meta;
}

=head2 write_bs_meta

TODO

=cut

sub write_bs_meta {
    my $self = shift;
    my $table = shift;
    my $meta = shift;
    my %par = @_;

    # TODO
    warn __PACKAGE__ . "::write_meta is not implemented yet";
}

=head1 BM UTILS

=cut

sub open_read {
    my $self = shift;
    my $source_table = shift;
    my $format = shift;

    my $read_cmd = join(' ', $self->get_cmd(), 'read', $source_table, "--format=$format");
    open(my $fh, '-|', $read_cmd) or die($!);

    return $fh;
}

sub open_write {
    my $self = shift;
    my $target_table = shift;
    my $format = shift;

    my $write_cmd = join(' ', $self->get_cmd(), 'write', $target_table, "--format=$format");
    open(my $fh, '|-', $write_cmd) or die($!);

    return $fh;
}

sub shuffle {
    my $self = shift;
    my $src_table = shift;
    my $dst_table = shift;

    my $temp_table = "//tmp/$$.shuffle";
    $self->do_cmd(
        'map',
        q/"perl -MJSON -ne '
            chomp(\$_);
            \$json = JSON::from_json(\$_);
            \$json->{"_random_id"} = int(rand(1e18));
            print JSON::to_json(\$json), \"\\n\";
        '"/,
        "--src=$src_table",
        "--dst=$temp_table",
        '--format=json',
    );
    $self->do_cmd('sort', "--src=$temp_table", "--dst=$dst_table", '--sort-by=_random_id');
}

sub sort {
    my ($self, $table, $field) = @_;
    $self->do_cmd('sort', "--src=$table", "--dst=$table", "--sort-by=$field");
}

sub set_upload_time {
    my $self = shift;
    my $table = shift // die "provide table";
    my $timestamp = shift // time();

    my $time = gmtime($timestamp)->datetime();
    my $time_json = JSON->new()->allow_nonref()->encode($time);

    return $self->do_cmd(
        'set',
        "$table/\@upload_time",
        '--format=json',
        "--value='$time_json'",
    );
}

1;
