package BM::YT::DynTablesProxyClient;

use strict;
use warnings;

use utf8;
use open ':utf8';

use base qw(ObjLib::ProjPart);


use LWP::UserAgent;
use HTTP::Request;
use JSON::XS qw(encode_json decode_json);
use Time::HiRes;
use List::Util qw(shuffle);

use Utils::Hosts qw(get_curr_host);
use Utils::CompileTime;


sub init {
    my $self = shift;
    $self->{ua} = LWP::UserAgent->new();
    $self->{proxy_port} = $self->{params}->{proxy_port};
    $self->{max_client_timeout} = $self->{params}->{max_client_timeout};
    $self->{local_proxy_hosts} = { map { $_ => 1 } @{ $self->{params}->{local_proxy_hosts} } };
    $self->{public_proxy_hosts} = { map { $_ => 1 } @{ $self->{params}->{public_proxy_hosts} } };
}

sub _meta {
    return __PACKAGE__ . " at " . get_curr_host() . ", script: '$0', pid: $$, svn revision: " . (Utils::CompileTime::get_revision() // "(unknown)");
}

sub _get_hosts {
    my $self = shift;
    my $curr_host = shift // get_curr_host();
    
    my @hosts;
    push @hosts, "localhost" if $self->{local_proxy_hosts}->{$curr_host};
    push @hosts, shuffle grep { $_ ne $curr_host } keys %{ $self->{public_proxy_hosts} };

    return @hosts;
}

sub do_select {
    my $self = shift;
    my %opts = @_;

    my $proj = $self->proj;
    my $timeout = $self->{max_client_timeout};
    my @hosts = $self->_get_hosts();

    die "No known hosts with proxies" unless scalar @hosts;

    my $attempt = 0;
    my $started = Time::HiRes::time();
    while ("true") {
        
        # опрашиваем хосты по кругу
        my $host = shift @hosts; push @hosts, $host;
        my $res = eval { $self->_do_select(proxy_host => $host, %opts) };

        if ($@) {
            my $errmsg = $@;            
            
            # про некоторые ошибки мы знаем, что перезапрос не поможет
            if (
                $errmsg =~ m/provide \S+/
                || $errmsg =~ m/Error while parsing query/
                || $errmsg =~ m/Query terminated prematurely due to excessive input/
            ) {
                die __PACKAGE__ . ", fatal error: " . $errmsg;
            }
            
            chomp $errmsg; $proj->log($errmsg);
            $proj->log(__PACKAGE__ . ": attempt $attempt with proxy host = '$host' failed, wait and retry");

            # первый retry делаем сразу и безусловно, последующие - с увеличивающимся шагом, и проверяя таймаут
            if ($attempt > 0) {
                my $sleep_time = 2 ** ($attempt - 1);
                last if Time::HiRes::time() + $sleep_time - $started > $timeout;
                sleep($sleep_time);
            }
        } else {
            return $res;
        }

        $attempt++;
    }

    die __PACKAGE__ . ", fatal error: could not get data from dt_proxy after $attempt attempts, give up";
}

sub _do_select {
    my $self = shift;
    my %opts = @_;

    my $proxy_host = $opts{proxy_host} // die "provide proxy_host";
    my $table = $opts{table} // die "provide table";
    my $fields = $opts{fields} // die "provide fields";
    my $condition = $opts{condition} // die "provide condition";
    my $limit = $opts{limit} // 0;
    my $slice_result = $opts{slice_result} // 1;
    my $fields_mapping = $opts{fields_mapping} // {};

    my $all_started = Time::HiRes::time();
    
    my $query = {
        action => "select",
        what => join(", ", @$fields),
        from => $table,
        where => $condition,
        meta => $self->_meta(),
        limit => 0 + $limit,
    };

    for my $key (qw/where meta/) {
        $query->{$key} =~ s/\s*\n\s*/ /g;
    }
    
    my $url = "http://" . $proxy_host . ":" . $self->{proxy_port} . "/";
    my $req = HTTP::Request->new(POST => $url);
    $req->content(encode_json($query));

    my $request_started = Time::HiRes::time();
    my $resp = $self->{ua}->request($req);
    my $request_elapsed = Time::HiRes::time() - $request_started;

    my $content = eval { decode_json($resp->decoded_content()) };
    my $rpc_elapsed = defined($content->{milliseconds_elapsed}) ? ($content->{milliseconds_elapsed} / 1000) : 0;

    unless ($resp->is_success()) {
        my $errmsg = ref($content) ? $content->{errmsg} : $resp->decoded_content();
        die "Request failed: " . $resp->status_line()
            . ($errmsg ? ("\n" . $errmsg) : "");
    }

    unless ($content && $content->{data}) {
        die "Bad response: no data in content"
            . "\n" . $resp->decoded_content();
    }

    unless ($slice_result) {
        return $content->{data};
    }
    
    my @result;
    for my $row (@{ $content->{data} }) {
        if (scalar(@$fields) != scalar(@$row)) {
            die "Bad response: row count and fields count do not match"
                . "\n" . $resp->decoded_content();
        }

        my %sliced_row = map {
            my $field_name = $fields->[$_];
            ($fields_mapping->{$field_name} // $field_name) => $row->[$_];
        } (0 .. $#$fields);
        
        push @result, \%sliced_row;
    }

    my $all_elapsed = Time::HiRes::time() - $all_started;

    my $time_stats_str = sprintf(
        "of %.3f seconds in total: %.3f in rpc call, %.3f in request to proxy, %.3f in perl code",
        $all_elapsed,
        $rpc_elapsed,
        $request_elapsed - $rpc_elapsed,
        $all_elapsed - $request_elapsed,
    );

    if ($all_elapsed > $self->{params}->{log_requests_longer_than}) {
        $self->proj->log("Long _do_select in " . __PACKAGE__ . "; $time_stats_str");
    } else {
        $self->proj->logger->debug("Normal _do_select in " . __PACKAGE__ . "; $time_stats_str");
    }
    
    return \@result;
}

sub do_select_by_key {
    my $self = shift;
    my %opts = @_;
    
    my $table = $opts{table} // die "provide table";
    my $fields = $opts{fields} // die "provide fields";
    my $id_field = $opts{id_field} // die "provide id_field";
    my $key_condition = $opts{key_condition} // die "provide key_condition";
    my $fields_mapping = $opts{fields_mapping} // {};

    my $condition = $opts{condition} // "True";
    my $slice_result = $opts{slice_result} // 1;

    my $key_table = $opts{key_table};
    my $key_field = $opts{key_field};
    die "provide key_table or key_field" unless defined($key_table) || defined($key_field);

    $key_table //= "${table}_by_${key_field}";

    my @ids = map { $_->[0] } @{ $self->do_select(
        table => $key_table,
        fields => [ $id_field ],
        condition => $key_condition,
        slice_result => 0,
    ) };

    my @result;
    while (my @chunk = splice(@ids, 0, 500_000)) {
        push @result, @{ $self->do_select(
            table => $table,
            fields => $fields,
            condition => "$id_field in (" . join(",", @chunk) . ") and ($condition)",
            slice_result => $slice_result,
            fields_mapping => $fields_mapping,
        ) };
    }

    return \@result;
}

1;
