package BaseProject;

use strict;
use utf8;
use open ':utf8';

use base qw(ObjLib::Obj);

use Storable;
use Data::Dumper;
use Scalar::Util qw(weaken);
use Term::ANSIColor qw(colored colorvalid);

use Cache::Memcached::Fast;

use ObjLib::Timer;
use ObjLib::FileLogger;

use Utils::Common;
use Utils::Words qw();
use Utils::Urls qw();
use Utils::Sys;
use Utils::Hosts;
use Utils::CompileTime;

# ломаются тесты broadmatching/test
# use Utils::URLValidator::URLValidator;

use Utils::Sys qw(mem_usage uniq);
use Scalar::Util qw(set_prototype);

use BM::Phrase;
use BM::PhraseCategs;
use BM::PhraseParser;
use BM::PhraseModif;
use BM::PhraseList;
use BM::PhraseListFast;
use BM::PhraseProjSrv;
use BM::PhraseListProjSrv;
use BM::Models;
use BM::Homonyms;
use BM::Language;
use BM::LemmerTest;
use BM::Geo;
use BM::RPC;
use BM::BMClient::PrefProjSrvClient;

use BM::Banners::BannerFactory;

use BM::ContextSyns::ContextLinks;
use BM::ContextSyns::Hyperonyms;
use BM::ContextSyns::MulWordSyns;
use BM::ContextSyns::DBSyns;

use BM::Banners::LBannerAnalysis;

use BM::CategoriesTree;
use BM::Categories::PagesCategories;
use BM::Categories::LocalLayer;
use BM::Categories::RemoteLayer;
use BM::Categories::CategoryGroups;
use BM::CategoriesTextFields;
use BM::Markers;

use BM::Dicts::Norm;
use BM::Dicts::Multiwords;
use BM::Dicts::DictManager;

use BM::BMClient::CdictClient;
use BM::BMClient::SimpgraphCdict;
use BM::BMClient::SubphrasesClient;
use BM::BMClient::ComptrieSubphrasesClient;

use BM::Zora;
use BM::BroadKyoto;
use BM::EmptyKyoto;

use BM::Filter;
use BM::Filters;

use BM::ContextFiltering::BannersCategories;

use BM::Categories;

use BM::YT::DynTablesProxyClient;

use BM::DetectCharset;
use BM::Pages::Page;
use BM::Pages::PageLite;
use BM::Pages::PageProjSrv;
use BM::Pages::PageListHierarchy;
use BM::Pages::PageListProjSrv;
use BM::Pages::Image;
use BM::Pages::ImageList;
use BM::Pages::Site;
use BM::Pages::SiteProjSrv;
use BM::Pages::PageHierarchy;
use BM::Pages::PageSnippet;

use BM::XMLParser;
use BM::LRUCache;

use BM::File;

use Time::HiRes qw/time gettimeofday tv_interval/;

use DataSource::ElemFactory;

use Dates;
use JSON::XS;
use HTTP::Cookies;


########################################################
#Доступ к полям
########################################################

__PACKAGE__->mk_accessors((
    'temp_dir',
    'work_dir',
    'log_file',
    #@returns ObjLib::FileLogger
    'logger',
    #@returns ObjLib::Timer
    'timer',
    #@returns BM::Banners::BannerFactory
    'bf_base',
    #@returns BM::ContextFiltering::BannersCategories
    'banners_categories',
    #@returns Dates
    'dates',
    #unused?
    'pages_learn',
    #@returns BM::Homonyms
    'homonyms',
    'context_syns',
    #@returns BM::Categories
    'categs',
    #@returns BM::Categories
    'categs_atoms',
    #@returns BM::Categories
    'categs_dim2',
    #@returns BM::YT::DynTablesProxyClient
    'dt_proxy_client',
    #@returns BM::Dicts::Multiwords
    'multiwords',
    'subphrases_client',
    'perfect_subphrases_client',
    #@returns BM::DetectCharset
    'detect_charset',
    'languages',
    #@returns BM::Language
    'default_language',
    'default_lang',
    'current_lang',
    'current_region',
    'options',
    'categs_diff',
    #@returns BM::CategoriesTree
    'categs_tree',
    #@returns BM::Zora
    'zora_client',
    #@returns BM::Zora
    'zora_fast_client',
    #@returns BM::BMClient::CdictClient
    'cdict_client',
    #@returns BM::BMClient::CdictClient
    'datoteka_client',
    #@returns BM::Dicts::DictManager
    'dict_manager',
    #unused?
    'categs_mapper',
    #@returns BM::Categories::CategoryGroups
    'mediagroups',
    #@returns BM::Categories::CategoryGroups
    'videodirectgroups',
    #@returns BM::Categories::CategoryGroups
    'interests_mapping',
    'chronicle_cdict',
    #@returns BM::BMClient::CdictClient
    'datoteka_cdict',
    'xml_parser',
    #@returns BM::Geo
    'geo',
    #@returns BM::RPC
    'rpc',
    #@returns BM::Markers
    'catalogia_markers',
    #@returns DataSource::ElemFactory
    'elem_factory',
    #@returns BM::LRUCache
    'lru_cache',
));

#   models                      объект для работы с моделями
#   phrase                      из строки делает объект фразы
#   normed_phrase               то же, только с оптимизацией (если в строке уже нормализованная фраза)
#   snormed_phrase              то же для snorm
#   phrase_list                 по хэшу для конструктора генерит объект списка фраз
#   banner                      по хэшу для конструктора генерит объект баннера

########################################################
# Инициализация
########################################################

# флажки при создании Project-а:
# load_dicts                    загружать словари: нормализатор, синонимы, экв. домены, плохие/широкие слова и т.п.
# load_minicategs               загружать категории (всю информацию, включая категорийные фразы)
# load_minicategs_light         загружать дерево категорий (при категоризации использовать subphraser)
# allow_lazy_dicts              разрешить ленивую загрузку словарей (поддерживающих, такую возможность)
# dont_use_srvgraph             использовать обычные симпграфы в памяти, а не BM-сервер
# main_dbh                      выбор основной базы
# use_local_syns                использовать локальную версию синонимов (syn_cells_*.local)
# projsrv, nrmsrv(устар.)       использовать prefprojserver
# memlog                        выводить информацию о потреблении памяти

# устаревшие флажки, не нужно использовать:
# load_counts

sub init {
    my $self = shift;
    return if $self->get_calls_count_and_inc();

    my $perl_var = $_; # Сохраняем и восстанавливаем $_, иначе некорректно работают выражения типа   my $pr;  print Dumper([ map { $pr //= Project->new({});  "$_;" } @a ]);

    my $started = Time::HiRes::time();

    $self->{timelogpath} = [];
    $self->{timelogtree} = {};

    # слабая ссылка на Project; все объекты ProjPart содержат сильную ссылку на proxy_ref
    # это нужно, чтобы избежать большого кол-ва слабых сслылок - в этой ситуации сборка мусора
    # очень тормозит (n**2) из-за кривой реализации слабых ссылок
    my $weak_ref = $self;
    weaken($weak_ref);
    $self->{proxy_ref} = \$weak_ref;

    # для инициализации объектов ProjPart нужно, чтобы был задан current_lang
    my $default_lang_name = "ru";
    $self->{current_lang} = $default_lang_name;
    $self->{default_lang} = $default_lang_name;


    #Создаём прототип объекта фразы для ускорения создания объектов
    $self->{_phrase_prot} = $self->use_projsrv ? BM::PhraseProjSrv->new_lite({}) : BM::PhraseModif->new_lite({});

    if($self->{timelogpackages}){
        $self->timelogpackage($_) for @{$self->{timelogpackages}};
    }

    my $options = $self->{options} || { %{$Utils::Common::options} };
    $self->{options} = $options;

    Utils::Sys::include_cpan();

    # create common dirs ...
    -d $_ || mkdir $_ for map { $options->{dirs}{$_} } qw(work temp log lock);

    $self->dates( Dates->new );

    $self->lru_cache(BM::LRUCache->new( { proj => $self, conf => $options->{'lru_cache'} } ));

    if (!$self->logger) {
        # use standard logger
        my %log_conf;
        if ($self->{memlog}) {
            $log_conf{preprocess}{mem_usage} = 1;
        }
        if ($self->log_file) {
            $log_conf{log_file} = $self->log_file;
        } else {
            $log_conf{log_fh} = \*STDERR;
        }
        $self->logger(ObjLib::FileLogger->new(\%log_conf));
    }
    $self->timer(ObjLib::Timer->new);  # global timer, use carefully

    # текущий регион
    $self->{current_region} = 0;

    # работа с различными языками
    $self->{languages} = { map{$_ => ""} ($default_lang_name, @{$self->{load_languages} || []}) };
    for my $lang (keys %{$self->languages}) {
        $self->languages->{$lang} = BM::Language->new({
            proj        => $self,
            name        => $lang,
            is_default  => ($lang eq $default_lang_name),
            is_loaded   => 1,
            %{$options->{Language_params}{$lang} || {}}
        });
    }
    $self->{default_language} = $self->languages->{$default_lang_name};

    # медиагруппы
    $self->{mediagroups} = BM::Categories::CategoryGroups->new({
        proj    => $self,
        %{$options->{MediaGroups_params}}
    });

    # видеоподложки
    $self->{videodirectgroups} = BM::Categories::CategoryGroups->new({
        proj    => $self,
        %{$options->{VideodirectGroups_params}}
    });

    # маппинг для интересов в дзене
    $self->{interests_mapping} = BM::Categories::CategoryGroups->new({
        proj    => $self,
        %{$options->{InterestsMapping_params}}
    });

    # категоризаторы страниц для разных языков
    $self->{pages_categories} = {};
    for my $lang (keys %{$self->{languages}}) {
        $self->{pages_categories}{$lang} = BM::Categories::PagesCategories->new({
            proj        => $self,
            language    => $self->{languages}{$lang},
            %{$options->{PagesCategories_params}}
        });
    }

    # директории для static_map
    (-d $_ || mkdir $_) for @{$Utils::Common::options->{'static_map_params'}}{qw/work_dir temp_dir/};

    # клиент cdict
    $self->{cdict_client} = BM::BMClient::CdictClient->new({
        proj => $self,
        %{$options->{CdictClient_params}},
        %{$options->{cdict_params}},
        %{$options->{ChronicleCdict_params}}
    });
    (-d $_ || mkdir $_)  for ($self->cdict_client->temp_dir, $self->cdict_client->work_dir);

    $self->{datoteka_client} = BM::BMClient::CdictClient->new({
        proj => $self,
        %{$options->{cdict_params}},
        %{$options->{DatotekaCdict_params}}
    });

    $self->{chronicle_cdict} = BM::BMClient::CdictClient->new({
        proj => $self,
        %{$options->{cdict_params}},
        %{$options->{ChronicleCdict_params}}
    });

    $self->{datoteka_cdict} = BM::BMClient::CdictClient->new({
        proj => $self,
        %{$options->{cdict_params}},
        %{$options->{DatotekaCdict_params}}
    });

    # клиент для подфраз категорий
    if ($self->{comptrie_subphraser}) {
        $self->{subphrases_client} = BM::BMClient::ComptrieSubphrasesClient->new({
            proj => $self,
            %{$options->{subphraser_params}},
            comptrie_file => $self->{comptrie_subphraser},
        });
    } else {
        $self->{subphrases_client} = BM::BMClient::SubphrasesClient->new({
            proj => $self,
            %{$options->{subphraser_params}}
        });
    }

    if ($self->{'use_comptrie_subphraser'}) {
        $self->{'subphrases_client'} = BM::BMClient::ComptrieSubphrasesClient->new({
            proj => $self,
            %{$options->{'subphraser_params'}},
        });
    }

    $self->{perfect_subphrases_client} = BM::BMClient::SubphrasesClient->new({
        proj => $self,
        %{$options->{perfect_subphraser_params}}
    });
    (-d $_ || mkdir $_) for ($self->subphrases_client->temp_dir,
                             $self->subphrases_client->work_dir,
                            );

    # клиенты для zora
    $self->{zora_client} = BM::Zora->new({
        proj => $self,
        %{$options->{zora_params}},
        custom_cookies => HTTP::Cookies->new({}),
    });
    (-d $_ || mkdir $_)  for ($self->zora_client->temp_dir, $self->zora_client->work_dir);
    for my $cookie_option (@{$options->{custom_cookies}}) {
        $self->{zora_client}->{custom_cookies}->set_cookie(
            0,
            $cookie_option->{cookie_key},
            $cookie_option->{cookie_value},
            $cookie_option->{path},
            $cookie_option->{domain}
        );
    }

    $self->{zora_fast_client} = BM::Zora->new({
        proj => $self,
        %{$options->{zora_fast_params}},
        custom_cookies => $self->{zora_client}->{custom_cookies},
    });
    (-d $_ || mkdir $_)  for ($self->zora_fast_client->temp_dir, $self->zora_fast_client->work_dir);

    # менеджер словарей
    $self->{dict_manager} = BM::Dicts::DictManager->new({
        proj => $self,
        %{$options->{DictManager_params}}
    });

    # Категории баннеров
    $self->{"banners_categories"} = BM::ContextFiltering::BannersCategories->new({
        proj => $self,
        %{$options->{BannersCategories_params}}
    });

    # Омонимы
    $self->{"homonyms"} = BM::Homonyms->new({
        proj => $self,
        %{$options->{Homonyms_params}}
    });

    # контекстные синонимы
    $self->{context_syns} = {
        'hyperonyms'      => BM::ContextSyns::Hyperonyms->new({proj => $self, %{$options->{ContextSyns}{Hyperonyms}}}),
        'mulword_syns'    => BM::ContextSyns::MulWordSyns->new({proj => $self, %{$options->{ContextSyns}{MulWordSyns}}}),
        'catmedia_csyns'  => BM::ContextSyns::DBSyns->new({proj => $self, %{$options->{ContextSyns}{CatmediaContextSyns}}}),
    };
    $self->{context_syns}{$_} //= BM::ContextSyns::ContextLinks->new({proj => $self, %{$options->{ContextSyns}{$_}}})
        for (qw[ merchant_sim precise precise_tr careful medium medium_exp medium_tr brands_assoc diacr orfovars_tr translit_ru2en translit_en2ru translit_en2ru_strict translit_smpg smpg_spell smpg_medium ]);
    $self->{context_syns}{synonyms} = $self->{context_syns}{precise};  # для совместимости

    # Мультислова
    $self->{multiwords} = BM::Dicts::Multiwords->new({proj => $self});

    # Словари
    if($self->{load_dicts}) {
        $self->load_dicts;
    }

    $self->{categs_tree} = BM::CategoriesTree->new({ proj => $self });

    # Категоризаторы
    if($self->{load_categs}) {
        for my $name ("categs", "categs_atoms", "categs_dim2") {
            $self->{$name} = BM::Categories->new({
                proj => $self,
                name => $name,
                %{$options->{$name."_params"}}
            });
        }
    }

    #Маркеры модерации
    $self->{catalogia_markers} = BM::Markers->new({ proj => $self });

    $self->bf_base(BM::Banners::BannerFactory->new({proj => $self}));

    # подключаем BroadmatchDirectPhrases
    -d $_ || mkdir $_ for @{$options->{'BroadmatchDirectPhrases_params'}}{qw(work_dir temp_dir)};

    -d $_ || mkdir $_ for @{$options->{DictEquivDomains}}{qw(work_dir temp_dir)};
    -d $_ || mkdir $_ for @{$options->{DictNorm}}{qw(work_dir temp_dir)};
    -d $_ || mkdir $_ for @{$options->{DictSynonyms}}{qw(work_dir)};
    -d $_ || mkdir $_ for $options->{DictGrammar}{temp_dir};

    # ContextFiltering
    -d $_ || mkdir $_ for map { $_.'/ContextFiltering' } @{$options->{dirs}}{qw(work temp)};

    # /ContextFiltering

    #Прокидываем часть настроек из опций
    $self->{work_dir} = $options->{dirs}{work};
    $self->{temp_dir} = $options->{dirs}{temp};

    $self->{detect_charset} = BM::DetectCharset->new({
        proj => $self,
        symbfile => $options->{symbfile},
    });

    # rpc-proxy для динтаблиц YT
    $self->dt_proxy_client(BM::YT::DynTablesProxyClient->new({
        proj => $self,
        params => $options->{dt_proxy_params},
    }));

    # директория для кэширования страниц сайтов
    mkdir $options->{'pageinf'}{'cache_dir'} unless -d $options->{'pageinf'}{'cache_dir'};

    # директория для очереди генерации баннеров
    mkdir $options->{'pageinf'}{'banner_maker_dir'} unless -d $options->{'pageinf'}{'banner_maker_dir'};
    mkdir $options->{'pageinf'}{'banner_maker_dir'}.'/IN'  unless -d $options->{'pageinf'}{'banner_maker_dir'}.'/IN' ;
    mkdir $options->{'pageinf'}{'banner_maker_dir'}.'/OUT' unless -d $options->{'pageinf'}{'banner_maker_dir'}.'/OUT';

    $self->{xml_parser} = BM::XMLParser->new({ proj => $self });

    # география - геобаза и наше дерево регионов
    $self->geo(BM::Geo->new({ proj => $self }));

    $self->rpc(BM::RPC->new({ proj => $self }));

    $self->elem_factory(DataSource::ElemFactory->new({ proj => $self }));

    # Тесты
    -d $_ || mkdir $_ for $options->{dirs}{QTests};

    # Итог
    my $elapsed = Time::HiRes::time() - $started;
    $self->log("Project::init() for $0 done, duration: $elapsed, svn_revision: " . (Utils::CompileTime::get_revision() // "UNKNOWN"));
    $self->log("Running in debug mode") if $options->{show_debug_mode_warnings};

    $_ = $perl_var; # восстанавливаем $_
}

sub post_init {
    my $self = shift;
}

# alias, temporary ?
sub bf {
    my $self = shift;
    return $self->bf_base;
}

########################################################
#Вспомогательные методы
########################################################

sub get_curr_host {
    my $self = shift;
    return Utils::Hosts::get_curr_host();
}

sub host_info {
    my $self = shift;
    return Utils::Hosts::get_host_info() || {};
}

sub host_role {
    my $self = shift;
    return $self->host_info->{role};
}

########################################################
#Методы
########################################################

sub get_new_timer {
    my $self = shift;
    return ObjLib::Timer->new;
}

sub proxy_ref {
    my ($self) = @_;
    return $self->{proxy_ref};
}

sub use_projsrv {
    my ($self) = @_;
    return $self->{nrmsrv} || $self->{projsrv};
}

# используется ли subphraser
sub is_subphraser_used {
    my ($self) = @_;

    return $self->{load_minicategs_light};
}

sub create_layer {
    my ($self, %opts) = @_;
    my $h = { %opts, proj => $self };

    if($opts{is_local} || !$self->is_subphraser_used) {
        return BM::Categories::LocalLayer->new($h);
    } else {
        return BM::Categories::RemoteLayer->new($h);
    }
}

sub get_languages_list {
    my ($self) = @_;

    return grep{$_->is_loaded} map{$self->languages->{$_}} keys %{$self->languages};
}

sub models {
    my ($self) = @_;
    unless ($self->{'models'}) {
        $self->{'models'} = BM::Models->new({
                proj => $self,
                %{$self->{options}{Models_params}},
            });
        for my $dt (qw(work_dir temp_dir dict_dir)) {
            my $dir = $self->{'models'}{$dt};
            mkdir $dir unless -d $dir;
        }
        $self->{'models'}->load_models;
    }
    return $self->{'models'};
}

# memcached client -- deprecated
# now this is alias for broad_kyoto
sub memclient {
     my ( $self ) = @_;
     return $self->broad_kyoto;
}

# broad kyoto client
sub broad_kyoto {
    my ( $self ) = @_;
    if ( !defined($self->{broad_kyoto}) ) {
        if ($ENV{MR_BROADMATCH}) {
            $self->{broad_kyoto} = BM::EmptyKyoto->new({proj => $self});
        } else {
            $self->{broad_kyoto} = BM::BroadKyoto->new({
                proj => $self,
                kyoto => new Cache::Memcached::Fast({
                      servers                 => $Utils::Common::options->{broad_kyoto}{servers},
                      max_size                => $Utils::Common::options->{broad_kyoto}{max_size},
                      utf8                    => 1,
                }),
                %{ $Utils::Common::options->{broad_kyoto} },
            });
        }
    }
    return $self->{broad_kyoto};
}

# kyoto client
sub ktclient {
    my ( $self ) = @_;
    if ( !($self->{ktclient}) ) {
        if ($ENV{MR_BROADMATCH}) {
            $self->{ktclient} = BM::EmptyKyoto->new({proj => $self});
        } else {
            $self->{ktclient} = new Cache::Memcached::Fast({
                    servers                 => $Utils::Common::options->{kyoto}{servers},
                    max_size                => $Utils::Common::options->{kyoto}{max_size},
            });
        }
    };
    return $self->{ktclient};
}

sub validate_url {
    my ($self, $url) = @_;
    my $validator = $self->url_validator;

    my $res = eval { $validator->validate_url($url) };
    if ($@) {
        warn "validator failed at $url: $@";
        return 0;
    }
    return 0 if !$res;

    my $pun_url = eval { Utils::Urls::url_to_punycode($url) };
    if ($@) {
        warn "url_to_punycode failed at $url: $@";
        return 0;
    }

    $res = eval { $validator->validate_url($pun_url) };
    if ($@) {
        warn "validator failed at $url: $@";
        return 0;
    }
    return 0 if !$res;

    return 1;
}

sub url_validator {
    my ($self) = @_;

    unless ($self->{__url_validator}) {
        require Utils::URLValidator::URLValidator;
        $self->{__url_validator} = Utils::URLValidator::URLValidator->new( );
    }

    return $self->{__url_validator};
}

sub projsrv_cl {
    my $self = shift;
    my $parameters = shift;
    if (!$self->{projsrv_cl}) {
        $self->{projsrv_cl} = BM::BMClient::PrefProjSrvClient->new($parameters);
    }
    return $self->{projsrv_cl};
}

# загрузка файла по урлу
sub load_file_by_url {
    my ($self, $url, $res_file, %params) = @_;
    my $tries = 3;
    my $done = 0;
    while ($tries > 0 && !$done) {
        my $page = $self->page( $url );
        $page->timeout($params{timeout} || 120);
        $page->{fix_page_charset} = $params{fix_page_charset} // 1;
        $page->{$_} = $params{$_} for grep { $params{$_} } qw/login pass/;
        $page->result_file($res_file);
        $page->tt; # записали в файл
        $done = $page->{download_failed} ? 0 : 1;
        $tries--;
    }
    return $done;
}

sub page {
    my ($self, $url, $name, $ptext) = @_;
    if(ref($url) eq 'HASH'){ #Формат создания объекта через хэш инициализации
        my $data = $url;
        $data->{proj} = $self;
        $data->{opts} = $self->{'options'}{'pageinf'};
        $data->{lang} = $self->{current_lang};
        return BM::Pages::PageProjSrv->new($data) if $self->use_projsrv;
        return BM::Pages::PageHierarchy->new($data);
    }
    $name = '' unless $name;
    return BM::Pages::PageProjSrv->new({ proj => $self, url => $url, name => $name, opts => $self->{'options'}{'pageinf'}, lang => $self->{current_lang}, pagetext => $ptext, }) if $self->use_projsrv;
    return BM::Pages::PageHierarchy->new({ proj => $self, url => $url, name => $name, opts => $self->{'options'}{'pageinf'}, lang => $self->{current_lang}, pagetext => $ptext, });
}


sub page_list {
    my ($self, $data) = @_;
    $data = [] unless defined $data;
    my $newdata = { proj => $self, pages => $data };
    return BM::Pages::PageListProjSrv->new($newdata) if $self->use_projsrv;
    return BM::Pages::PageListHierarchy->new($newdata);
}

sub file {
    my ($self, $data) = @_;
    my $newdata;
    if( ref($data) eq 'HASH' ){
        $newdata = $data;
        $newdata->{proj} = $self;
    }else{
        $newdata = { proj => $self, name => $data };
    }
    return BM::File->new($newdata);
}

sub image {
    my ($self, $url, $name) = @_;
    $name = '' unless $name;
    return BM::Pages::Image->new({ proj => $self, url => $url, name => $name, opts => $self->{'options'}{'pageinf'}, lang => $self->{current_lang} });
}

sub image_list {
    my ($self, $data) = @_;
    $data = [] unless defined $data;
    my $newdata = { proj => $self, data => $data };
    return BM::Pages::ImageList->new($newdata);
}

sub zpage {
    my ($self, %opts) = @_;

    my $data = {
        proj        => $self,
        url         => $opts{url} || "",
        name        => $opts{name} || "",
        language    => $opts{language} || $self->get_language($opts{lang}) || $self->default_language,
        site_proxy_ref => $opts{site_proxy_ref} || "",
        inf         => $opts{'inf'} || "",
    };
    $data->{opts} = $self->{'options'}{'pageinf'};

    return BM::Pages::PageProjSrv->new($data) if $self->use_projsrv;
    return BM::Pages::PageHierarchy->new($data);
}

sub site {
    my ($self, $url) = @_;
    my $data = { proj => $self, url => $url, opts => $self->{'options'}{'pageinf'} };
    return BM::Pages::SiteProjSrv->new($data) if $self->use_projsrv;
    my $s = BM::Pages::Site->new($data);
    return $s;
}

sub filter {
    my ( $self, $params ) = @_;
    return BM::Filter->new({proj=> $self, %$params});
}

sub filters {
    my ( $self, $params ) = @_;
    return BM::Filters->new({proj=> $self, params => $params});
}

sub prefilter {
   my ($self, $file) = @_;
   return $self->get_language($self->{current_lang})->prefilter($file);
}

sub categories_text_fields_dict {
    my $self = shift;

    $self->{categories_text_fields_dict} = BM::CategoriesTextFields->new({proj => $self})
        if !$self->{categories_text_fields_dict};

    return $self->{categories_text_fields_dict};
}

#@returns BM::PhraseModif
sub phrase {
    my $self = shift;
    return $self->{_phrase_prot}->new_lite({ proj_proxy_ref => $self->{'proxy_ref'}, text => $_[0], inf => $_[1], lang => ($_[2] || $self->{current_lang}) }) if $_[1];
    return $self->{_phrase_prot}->new_lite({ proj_proxy_ref => $self->{'proxy_ref'}, text => $_[0], lang => ($_[2] || $self->{current_lang}) });
}

sub normed_phrase {
    my $self = shift;
    return $self->{_phrase_prot}->new_lite({ proj_proxy_ref => $self->{'proxy_ref'}, text => $_[0], normed => 1, inf => $_[1], lang => $self->{current_lang} }) if $_[1];
    return $self->{_phrase_prot}->new_lite({ proj_proxy_ref => $self->{'proxy_ref'}, text => $_[0], normed => 1, lang => $self->{current_lang} });
}

sub snormed_phrase {
    my $self = shift;
    return $self->{_phrase_prot}->new_lite({ proj_proxy_ref => $self->{'proxy_ref'}, text => $_[0], snormed => 1, inf => $_[1], lang => $self->{current_lang} }) if $_[1];
    return $self->{_phrase_prot}->new_lite({ proj_proxy_ref => $self->{'proxy_ref'}, text => $_[0], snormed => 1, lang => $self->{current_lang} });
}

#@returns BM::PhraseListFast
sub _phrase_list_orig {
    my ($self, $data) = @_;
    my $r = ref $data;
    if($r eq 'HASH'){
        $data->{'proj'} = $self;
        $data->{lang} //= $self->current_lang;
        return BM::PhraseListFast->new($data);
    }elsif($r eq 'ARRAY'){
        my @sclrs = grep { ! ref($_) } @$data;
        return BM::PhraseListFast->new( { phrases_list => $data, proj => $self, lang => $self->current_lang } ) unless @sclrs;
        return BM::PhraseListFast->new( { phrases_arr  => \@sclrs, proj => $self, lang => $self->current_lang } ) if @sclrs == @$data;
        my @objs  = @sclrs < @$data ? (grep { ref($_) } @$data) : ();
        my $phl1 =  BM::PhraseListFast->new( { phrases_arr  => \@sclrs, proj => $self, lang => $self->current_lang } );
        my $phl =  BM::PhraseListFast->new( { phrases_list => [ @objs, $phl1->phrases ],  proj => $self, lang => $self->current_lang } );
        return $phl;
    }elsif( (! $r) && ($data || ( defined($data) && ($data eq '')))){
        return BM::PhraseListFast->new({ phrases_text => $data, proj => $self, lang => $self->current_lang });
    }else{
        return BM::PhraseListFast->new( { phrases_list  => [], proj => $self, lang => $self->current_lang } );
    }
}

#@returns BM::PhraseListFast
sub phrase_list {
    my ($self, $data, $lang) = @_;
    $lang //= $self->current_lang;
    my $phl = $self->_phrase_list_orig($data);
    return $phl unless $self->use_projsrv;
    return BM::PhraseListProjSrv->new( { phrases_list => [ @{$phl} ], 'proj' => $self, lang => $lang });
}

sub log_dump {
    my ($self, $obj) = @_;
    local $Data::Dumper::Terse=1;
    my $text = Dumper($obj);
    $text =~ s/^/[DUMP]\t[$$]\t/mg;
    $self->log("$text");
}

sub dd { my $self = shift; print $self->dump(\@_); }
sub ddst { my $self = shift; print $self->dump(\@_, $self->_stack_trace); }

sub load_dicts {
    my ($self) = @_;
    my $options = $self->{options};

    $self->log('load_dicts');

    if ($self->{use_local_syns}) {
        $self->log("using local synonyms!");
        $options->{Words_params}{syn_dicts} = [ map { [ $_->[0].'.local', $_->[1] ] } @{$options->{Words_params}{syn_dicts}} ];
    }

    #Инициализируем классы, подгружающие информацию - важен порядок!
    Utils::Words->class_init({
        logger => $self,
        %{$options->{Words_params}},
    });
    my @cl_init_before_lang = (
        [ 'BM::Phrase', [ qw{
            widephrs notwidephrs dict_not_wide
            directmodphrs porn_words obsc_words jets jets_exclusions
            biwords mulword_syns
            celebrities
            grammar_dict
        } ] ],
    );
    my @cl_init_after_lang = (
        [ 'BM::PhraseCategs', [ qw{
            minuswords
            minicategs minicategsaddphrs categs_antiwords categs_minus categs_minus_old categs_plus categs_siblings categs_siblings_add categs_nephews categs_never_sibl categs_diff categs_flags categs_direct_ids categs_addphr categs_uncertain categs_virtual dynbanners_categs_mapping
            dict_categskeys_with_minuswords
        } ] ],
        [ 'BM::PhraseParser', [ qw{
            dict_brands dict_brands_market dict_context_brands dict_noises dict_acc_brands dict_metrical dict_goods dict_surnames_ru dict_surnames_en dict_surnames_wide_ru dict_names_ru dict_banks dict_notmodels dict_geo dict_wear_properties
        } ] ],
        [ 'BM::Banners::LBannerAnalysis', [ qw{diff_words goods_words service_words info_words hyponyms_words minus_media dict_towns catalogia_markers dict_domain_flags} ] ],
        [ 'Utils::Urls', [ qw{equiv_domains} ] ],
    );
    for my $cl (@cl_init_before_lang){
        $cl->[0]->class_init({
            logger => $self,
            proj   => $self,
            ( map { $_ => $options->{$_} } @{$cl->[1]} ),
            allow_lazy_dicts => $self->{allow_lazy_dicts},
        });
    }
    for my $lang (sort keys %{$self->languages}) {
        $self->languages->{$lang}->load_dicts;
    }
    for my $cl (@cl_init_after_lang){
        $cl->[0]->class_init({
            logger => $self,
            proj   => $self,
            ( map { $_ => $options->{$_} } @{$cl->[1]} ),
            allow_lazy_dicts => $self->{allow_lazy_dicts},
        });
    }
}

sub log{
    my $self = shift;
    $self->logger->info(@_);
}

sub read_sys_cmd_bash_remote {
    my ( $self, $host, $command ) = @_;
    $command =~ s/'/'"'"'/g;
    $command =~ s/"/\\"/g;
    $command =~ s/\$/\\\$/g;
    $command = qq[ssh -n -o StrictHostKeyChecking=no $host "/bin/bash -c 'set -o pipefail; ] . $command . q['"];
    $self->log("cmd {{ $command }}");
    my $start_time = [gettimeofday];

    open my $fh, $command . '|'
        or do { $self->log("ERROR: read cmd '$command' failed: $!"); return };
    my $output;
    while (<$fh>) {
        $output .= $_;
    }
    close $fh
        or do { $self->log("ERROR: read cmd '$command' failed: $!"); return };

    my $duration = tv_interval($start_time);
    my $msg = "read cmd {{ $command }} done. Duration: $duration";
    $self->log($msg);

    return $output;
}

sub do_sys_cmd_bash {
    my ($self, $command, %prm) = @_;
    Utils::Sys::do_sys_cmd($command, logger => $self, shell => 'bash', %prm);
}

sub do_sys_cmd {
    my ($self, $command, %prm) = @_;
    Utils::Sys::do_sys_cmd($command, logger => $self, %prm);
}

sub read_sys_cmd {
    my ($self, $command, %prm) = @_;
    Utils::Sys::read_sys_cmd($command, logger => $self, %prm);
}

sub get_tempfile {
    my $self = shift;
    my $name = shift || 'proj_tmp';
    my %par  = (
        DIR => $self->temp_dir,
        @_,
    );
    return Utils::Sys::get_tempfile($name, %par);
}

# если вызывается с аргументами - пересоздает леммер
sub lemmer_test {
    my $self = shift;
    my @args = @_;
    my $args_str = join("\t", @args);
    # пересоздаем леммер, только если новые аргументы отличаются от старых
    if (!$self->{lemmer_test} or (@args and $args_str ne $self->{lemmer_test_args_str})) {
        $self->{lemmer_test} = BM::LemmerTest->new({proj => $self, args => \@args});
        $self->{lemmer_test_args_str} = $args_str;
    }
    return $self->{lemmer_test};
}

sub procmem {
#    return qx/ps --format size --no-headers -p $$/;
    return mem_usage($$);
    open(STATUS, "< /proc/$$/status");
    my $mem = 0;
    while(my $l = <STATUS>){
        $mem = $1, last if $l =~ /^VmSize:\s+(\d+)/;
    }
    close(STATUS);
    return $mem;
}

sub timelogbeg {
    my ($self, $name) = @_;
    push(@{$self->{timelogpath}}, $name);
    $self->{timelogbegh}{$name} = time;
    if($self->{memlog}){
        $self->{memlogbegh}{$name} = $self->procmem;
    }
}

sub timelogend {
    my ($self, $name) = @_;
    my $now = time;
    my $worktime = $now - $self->{timelogbegh}{$name};
    $self->{timeloghash}{$name}{'=ttime'} += $worktime;
    $self->{timeloghash}{$name}{'=cnt'}++;
    pop(@{$self->{timelogpath}});
    my $cur = $self->{timelogtree};
    for(grep{defined($_)} @{$self->{timelogpath}}){
        $cur->{$_} = {} unless $cur->{$_};
        $cur = $cur->{$_};
    }
    $cur->{$name}{'=ttime'} += $worktime;
    $cur->{$name}{'=cnt'}++;
    if($self->{memlog}){
        my $nowmem = $self->procmem;
        my $workmem = $nowmem - $self->{memlogbegh}{$name};
        $self->{memloghash}{$name} += $workmem;
        $cur->{$name}{'=mem'} += $workmem;
    }
}

sub timeloginf {
    my ($self, $tt) = @_;
    $self->{timeloghash} = $tt if $tt;
    return $self->{timeloghash};
}

sub timelogtree {
    my ($self, $tt) = @_;
    $self->{timelogtree} = $tt if $tt;
    return $self->{timelogtree};
}

sub timelogclear {
    my ($self) = @_;
    $self->timeloginf({});
    $self->timelogtree({});
}

sub _tltree {
    my ($h, $pref) = @_;
    my $text = '';
    for my $l ( sort { $h->{$b}{'=ttime'} <=> $h->{$a}{'=ttime'} } grep {!/^=/} keys %$h ){
        $text .= "$pref$l = ".$h->{$l}{'=ttime'};
        $text .= " count=".$h->{$l}{'=cnt'} if $h->{$l}{'=cnt'};
        $text .= "\n" ;
        $text .=  _tltree($h->{$l}, "  $pref");
    }
    return $text;
}

#С фильтрацией по значимости
sub _tltree_fltd {
    my ($h, $pref, $prtm) = @_;
    my $text = '';
    for my $l ( sort { $h->{$b}{'=ttime'} <=> $h->{$a}{'=ttime'} } grep {!/^=/} keys %$h ){
        next if $prtm && ( $h->{$l}{'=ttime'} < $prtm * 0.1);
        $text .= "$pref$l = ".$h->{$l}{'=ttime'};
        $text .= " count=".$h->{$l}{'=cnt'} if $h->{$l}{'=cnt'};
        $text .= " mem=".$h->{$l}{'=mem'} if $h->{$l}{'=mem'};
        $text .="\n" ;
        $text .=  _tltree_fltd($h->{$l}, "  $pref", $h->{$l}{'=ttime'});
    }
    return $text;
}

sub timelogreport {
    my ($self) = @_;
    my $text = "";
    my $ti = $self->timeloginf;
    $text .= "timeloginf:\n";
    $text .= "  $_ = ".$ti->{$_}{'=ttime'}." ".$ti->{$_}{'=cnt'}."\n" for sort {$ti->{$b}{'=ttime'} <=> $ti->{$a}{'=ttime'}} keys %$ti;
    if($self->{memloghash}){
        my $mi = $self->{memloghash};
        $text .= "\nmemloginf:\n";
        $text .= "  $_ = ".$mi->{$_}."\n" for sort {$mi->{$b} <=> $mi->{$a}} keys %$mi;
    }
    $text .= "\n\ntimelogtree:\n";
    $text .= _tltree($self->timelogtree, '  ')."\n";
    $text .= "\n\ntimelogtree_fltd:\n";
    $text .= _tltree_fltd($self->timelogtree, '  ')."\n";
    return $text;
}

sub timelogpackage {
    my ($self, $pn) = @_;
    return if $self->{'timelogpackage_list'} && $self->{'timelogpackage_list'}{$pn};
    my $mtds = $self->package_methods($pn);
    for my $k ( keys %$mtds){
        my $m = $mtds->{$k};
        my $tt = scalar(*$m);
        $tt =~ /^\*(.+::)?([^:]+)$/;
        my $package = $1;
        my $name = $2;
        no warnings 'redefine';
        my $referent = ref $m eq 'CODE'
                       ? $m
                       : *$m{CODE};
        my $proj_proxy_ref = $self->proxy_ref;
        my $prot = prototype $tt;
        my $tmpmtd = sub {
            my $vsl = ${$proj_proxy_ref}->{visual};
            my $bg = 0;
            if($vsl){
                $bg = time;
            }
            ${$proj_proxy_ref}->timelogbeg("sub $package $name");
            my @res = $referent->(@_);
            if($vsl){
                my $tm = time - $bg;
                print STDERR "VSL:".join("/", @{ ${$proj_proxy_ref}->{timelogpath}})." $tm\n" if $tm > 10;
                if( time - ${$proj_proxy_ref}->{VSLTM} > 60 ){
                    #print "VSLTM:".join("/", @{ ${$proj_proxy_ref}->{timelogpath}})." $tm\n";
                    print STDERR "VSLTM: $tm\n".join("", map { "    $_\n" } @{ ${$proj_proxy_ref}->{timelogpath}});
                    ${$proj_proxy_ref}->{VSLTM} = time;
                }
            }
            ${$proj_proxy_ref}->timelogend("sub $package $name");
            return wantarray ? @res : exists( $res[0] ) ? $res[0] : undef;
        };
        set_prototype \&$tmpmtd, $prot;
        no strict 'refs';
        my $ttt = $package.$name;
        *{ $ttt } = $tmpmtd;
    }
    $self->{'timelogpackage_list'}{$pn}++;
}

sub package_methods {
    my ($self, $pn) = @_;
    my %h = ();
    eval "\%h = \%${pn}::";
    my %hh = map { $_ => $h{$_} } grep { defined &{$h{$_}} } grep {! ref $h{$_}} grep {! /^isa|isweak|weaken|import|can|get_methods|new|Dumper|isweak|isa|init|proj|refaddr|tainted|dualvar|isvstring|blessed|readonly|reftype|set_prototype|looks_like_number|gettimeofday|uniq|phrases_arr|.{2}_DECnet$/} grep { /[a-z]/ } keys %h;
    return \%hh;
}

sub get_language {
    my ($self, $name) = @_;
    $name //= 'ru';

    return $self->languages->{$name} ||= BM::Language->new({
        proj        => $self,
        name        => $name,
    });
}

# эксперименты на всем хосте
# experiment_info нет, т.к. по большому счету это костыль
sub experiment {
    my $self = shift;
    my %exper;

    # дополнительные сиблинги
    $exper{add_siblings} = 1 if $self->options->{use_experiment}{add_siblings};

    return \%exper;
}

sub get_transl_hash {
    my ($self) = @_;
    return $self->{'transl_hash'} if $self->{'transl_hash'};
    my $h = {};
    #open(F,"<".$self->{options}{'transl_intrf'});
    open(F, $self->{options}{'cmd_transl_intrf'});
    while(<F>){
        chomp;
        next unless /\S/;
        my ($ru, $en) = split("\t", $_);
        $h->{$ru} = $en;
    }
    close(F);
    $self->{'transl_hash'} = $h;
    return $h;
}

#Функция для перехвата SIG DIE
our $error_stack_trace = '';
sub __error_sig_die {
    $error_stack_trace = join "", map { $_->{subroutine}." <= ".$_->{package}." ".$_->{line}."\n" } reverse @{&ObjLib::Obj::_stack_trace};
}

sub catch_exception {
    my ($self, $code, $name) = @_;
    $name ||= '';
    local $SIG{'__DIE__'} = \&__error_sig_die;
    eval { $code->() };
    if ( $@ ) {
        return $error_stack_trace."ERROR: $@";
    }
    return '';
}

sub dumper_text {
    my ($self) = @_;
    $_[0] = \'Project';
    return @_;
}

sub dumper_text_lite {
    my ($self) = @_;
    $_[0] = \'Project';
    return @_;
}

sub json_magic {
    return '<--#JsonSep#-->';
}

#@returns JSON::XS
sub json_obj {
    my $self = shift;
    return $self->{'json_obj'} if $self->{'json_obj'};
    my $json = JSON::XS->new;
    $json->utf8(0);
    $json->indent(0);
    $self->{'json_obj'} = $json;
    return $json;
}

sub serial {
    my ($self, $arr) = @_;
    my $json = $self->json_obj;
    return $json->encode($arr) if ref($arr) ne 'ARRAY';
    my $magic = $self->json_magic;
    return join($magic, map { $json->encode($_) } @$arr);
}

sub deserial {
    my ($self, $str) = @_;
    my $json = $self->json_obj;
    my $magic = $self->json_magic;
    my @str = split /$magic/, $str;
    return [ map { $json->decode($_) } @str ];
}

#Структуру данный в JSON
sub d2j {
    my ($self, $data) = @_;
    return $self->json_obj->encode($data);
}

#JSON в структуру данных
sub j2d {
    my ($self, $data) = @_;
    return $self->json_obj->decode($data);
}

sub encode {
    my ($self, $obj) = @_;
    return $self->sereal_encode($obj);
}
sub decode {
    my ($self, $str) = @_;
    return $self->sereal_decode($str);
}

use BM::SmartJson;

sub _smart_json_obj {
    my ($self) = @_;

    return $self->{_smart_json_obj} if $self->{_smart_json_obj};

    $self->{_smart_json_obj} = BM::SmartJson->new({proj => $self});

    return $self->{_smart_json_obj};
}

sub smart_json_encode {
    my $self = shift;

    return $self->_smart_json_obj->encode(@_);
}

sub smart_json_decode {
    my $self = shift;

    return $self->_smart_json_obj->decode(@_);
}

sub smart_json_encode_cache {
    my $self = shift;

    return $self->_smart_json_obj->encode_cache(@_);
}

sub _init_sereal {
    my $proj = shift;

    require Sereal::Encoder;
    require Sereal::Decoder;

    $proj->{_sereal_encoder} = Sereal::Encoder->new({freeze_callbacks => 1});
    $proj->{_sereal_decoder} = Sereal::Decoder->new;
}

sub sereal_encode {
    my ($proj, $obj) = @_;
    my $encoder = $proj->{_sereal_encoder};
    if (!$encoder) {
        $proj->_init_sereal;
        $encoder = $proj->{_sereal_encoder};
    };

    # FREEZE method defined for:
    # ObjLib::ProjPart (use with derived classes carefully!)
    # Phrase, PhraseList, BM::Banners::Campaign, BM::Banners::LBannerBM

    $encoder->encode($obj);
}

sub sereal_decode {
    my ($proj, $str) = @_;
    my $decoder = $proj->{_sereal_decoder};
    if (!$decoder) {
        $proj->_init_sereal;
        $decoder = $proj->{_sereal_decoder};
    }

    # т.к. конструктор фраз вызывается очень часто, максимально его оптизирируем
    local *BM::Phrase::THAW = sub {
        my ($class, $serializer, $data) = @_;
        $data->{proj_proxy_ref} = $proj->{proxy_ref};
        $data->{proj_current_lang} = $proj->{current_lang};
        return $proj->{_phrase_prot}->new_lite($data);
    };

    local *ObjLib::ProjPart::THAW = sub {
        my ($class, $serializer, $data) = @_;
        $data->{proj} = $proj;
        return $class->new($data, no_init => 1);  # init был вызван до сериализации при первом создании объекта
    };

    $decoder->decode($str);
}


##################################################################################
# / Работа с ClientID
##################################################################################

sub direct_users {
    my ($self) = @_;
    return $self->dbtable('DirectUsers', '', 'catalogia_media_dbh');
}

sub categ2mediagroup_hash {
    my ($self) = @_;
    return $self->mediagroups->categ2group;
}

sub curdate {
    my $self = shift;
    return $self->dates->cur_date('db');
}

sub curtime {
    my $self = shift;
    return $self->dates->cur_date('db_time');
}

# хеш обновляемых/генерируемых словарей
sub generated_dicts {
    my ($self) = @_;
    unless ($self->{generated_dicts}) {
        my $options = $self->{options};
        $self->{generated_dicts} = {
            'norm'              => BM::Dicts::Norm->new({proj => $self, %{$options->{DictNorm}}}),
        };
    }
    return $self->{generated_dicts};
}

#хотфикс, надо переделать эти опции, чтобы были в common вместо bannerland
sub _bannerland_allowed_naming_chars :CACHE {
    my ( $self ) = @_;
    return $self->options->{bannerland_allowed_naming_chars};
}

sub _bannerland_punctuation_not_valid_for_naming :CACHE {
    my ( $self ) = @_;
    return $self->options->{bannerland_punctuation_not_valid_for_naming};
}

my %stack_params;
if ($ENV{DIE_STACK_TRACE} || $ENV{STACK_TRACE_DIE}) {
    $stack_params{DIE} = {stack_trace => 1};
}
if ($ENV{WARN_STACK_TRACE} || $ENV{STACK_TRACE_WARN}) {
    $stack_params{WARN} = {stack_trace => 1};
}
Utils::Sys::handle_errors(%stack_params) if %stack_params;

1;
