package DataSource::Elem;

use std;

use Time::HiRes qw(gettimeofday tv_interval);

use base qw(ObjLib::ProjPart ObjLib::Error ObjLib::DS);

use DataSource::Filter::Filter_sum;

#use Utils::TimeLog;

our $long_requests_logging_min_duration; # Для логирования долгих запросов в List_SQL - Минимальная продолжительность выполнения запроса, для которой писать в лог

########################################################
# Инициализация
########################################################

sub init {
    my $self = shift;
    $self->id_field('ID') unless $self->id_field;
}

########################################################
#Доступ к полям
########################################################

__PACKAGE__->mk_accessors(
    'db_table',             #Рабочая таблица
    'dbh',                  #Доступ к базе
    'dbh_read',             #Доступ к базе на чтение
    'dbh_heavy',            #Доступ к базе на чтение
    'dont_use_dbh_read',    #Может быть полезно, когда есть вероятность запаздывания реплики
    'use_dbh_heavy',        #Может быть полезно, когда есть вероятность запаздывания реплики
    'editing_fields',       #Поля, которые могут редактироваться
    'adding_fields',        #Поля, которые могут быть при добавлении
    'listing_fields',       #Поля, которые могут быть при получении списка
    'id_field',             #Поле-идентификатор
    'read_only',            #Только на чтение
);

########################################################
#Вспомогательные функции
########################################################

sub get_last_id {
    my ($self) = @_;
    my $dbh = $self->dbh;
    return $dbh->last_insert_id(undef,undef,undef,undef);
}

########################################################
#Методы
########################################################

#Возвращает фильтр
sub create_f {
    my $self = shift;
    my $f = DataSource::Filter::Filter_sum->new;
    $f->add_f(@$_) for @_;
    return $f;
}

#Экранирует поле
sub FieldScreening {
    my ($self, $field) = @_;
    return $field !~ /[*)(\/]/ ? '`'.$field.'`' : $field;
#    return $field ne '*' ? '`'.$field.'`' : $field;
}


#Экранирует поля в массиве 
sub FieldsScreening {
    my ($self, @fields) = @_;
    return map { $self->FieldScreening( $_ ) } @fields;
}

#Экранирует поля, перечисленные в строке через запятую
sub FieldsStringScreening {
    my ($self, $fields_string) = @_;
    return $fields_string  if $fields_string =~ /[*`)(-]/;
    my @fields = split /\s*,\s*/, $fields_string;
    # В строке order_by может встретиться   -myField  или  myField desc.  "myField desc"  экранируем, остальное не трогаем. TODO  экранировать -myField 
    @fields = map {
        /^\s*(\w+)(\s+DESC)\s*$/i  ?  $self->FieldScreening( $1 ).' '.$2  :
        /^\s*(\w+)\s*$/  ?  $self->FieldScreening( $1 )  :
        $_
    } @fields;
    return join(',', @fields);
}


sub Add {
    my ($self, $data, $opt) = @_;

    return undef if $self->read_only;

    return $self->Add_multi($data, $opt) if ref($data) eq 'ARRAY'; #Множественная вставка
#    return undef unless ref($data) eq 'HASH';

    my $fields = $self->adding_fields;
    $fields = [ keys %$data ] unless $fields;
    $self->CleanseFields($data);

    my $db_table = $self->db_table;

    my $ignore = $opt->{ignore} ? 'IGNORE' : '';
    my $delayed = $opt->{delayed} ? 'DELAYED' : ''; #не работает для InnoDB
    my $sqltype = 'INSERT';
    $sqltype = 'REPLACE' if $opt->{replace};
    my $SQL = "$sqltype $delayed $ignore INTO $db_table (".join(', ', $self->FieldsScreening( @$fields ) ).') VALUES ( '.('?, ' x  (@$fields-1)).'?)';

    my $attempts = 10;
    while(!$self->dbh->do( $SQL, undef, @{$data}{@$fields})) {
        $attempts--;
        if($DBI::err == 2013 && $attempts > 0) {
            print STDERR $DBI::errstr . ", $attempts left";
        } else {
            die("Can't Add: $DBI::err, $DBI::errstr. \n SQL: $SQL");
        }
    }

    return $self->get_last_id;
}

#Множественная вставка
sub Add_multi {
    my ($self, $data, $opt) = @_;

    my $dbh = $self->dbh;

    return undef if $self->read_only;
    return undef unless ref($data) eq 'ARRAY';
    return undef unless @$data;

    #Если большое количество элементов - дробим на пачки по 1000
    my $max_count = 1000;
    while(@$data > $max_count){
        my @new = splice(@$data, 0, $max_count);
        $self->Add_multi(\@new, $opt);
    }

    my $fields = $self->adding_fields;
    $fields = [ keys %{$data->[0]} ] unless $fields;

    my $db_table = $self->db_table;

    my $ignore = $opt->{ignore} ? 'IGNORE' : '';
    my $sqltype = 'INSERT';
    $sqltype = 'REPLACE' if $opt->{replace};
    my $SQL = "$sqltype $ignore INTO $db_table (".join(', ', $self->FieldsScreening( @$fields ) ).') VALUES '
        .(
            ( 
                '( '.('?, ' x  (@$fields-1)).'?),'
            ) x (@$data-1)
         )
        .'( '.('?, ' x  (@$fields-1)).'?)';
        
    if ($opt->{update}) {
        $SQL .= " ON DUPLICATE KEY UPDATE " . join(', ', map { "$_=VALUES($_)" } $self->FieldsScreening(grep {$_ ne $self->id_field} @$fields));
    }
    
    my $SQL_ = $SQL;
    if($self->{'sql_log'}) {
        for my $el (map { @{$_}{@$fields} } @$data){
            $SQL_ =~ s/\?(,|\))/$dbh->quote($el).$1/e;
        }
    }
    if($self->{'sql_log'} && $self->{'sql_log'} < 3) {
        print STDERR "$SQL_\n";
    } 

    $dbh->do( $SQL, undef, (map { @{$_}{@$fields} } @$data) ) || die($self->proj->stack_trace('Add_multi error    ')."\n".'Add_multi error: '.$DBI::err .' => '. $DBI::errstr."\n".$SQL);

#    return $self->get_last_id;

#$SQL =~ s/\?/$_/ for map { @{$_}{@$fields} } @$data;
#$self->proj->dd($SQL);


    return undef;
}

sub Del {
    my ($self, $id) = @_;

    return undef if $self->read_only;

    my $db_table = $self->db_table;

    my $idf = $self->FieldScreening($self->id_field);

    my $SQL = "delete from $db_table where $idf=".$self->dbh->quote($id);

    $self->dbh->do($SQL);
}

sub DelList {
    my ($self, $filter) = @_;

    return undef if $self->read_only;

    my ($where_d, $arr) = $self->filter2str($filter);

    my @arr_v = @$arr;

    my $db_table = $self->db_table;

    my $idf = $self->id_field;

    if( $where_d ){
        my $SQL = "delete from $db_table where $where_d";
        my $sth = $self->dbh->prepare($SQL);
        $sth->execute(@arr_v);
    }
}

sub Edit {
    my ($self, $IDS, $data) = @_;
#    LogDump([$IDS, ref($IDS)]);
    if(!ref($IDS)) { # Если не ссылка значит засунуть в массив.
        $IDS = [ $IDS ];
    }
    return undef if $self->read_only;

    my $fields = $self->editing_fields;
    $fields = [ keys %$data ] unless $fields;
    $self->CleanseFields($data);

    return undef unless @$fields;

    my $SQL = sprintf("UPDATE %s SET %s WHERE %s IN (%s) ", 
        $self->db_table, 
        join( ', ', map { $self->FieldScreening($_).'=?' } @$fields ), 
        $self->id_field, 
        join(',', map{"?"} @$IDS)
    );

    my $LOG = $SQL;
    if($self->{'sql_log'}) {
        for my $el (map { $data->{$_} } @$fields){
            $LOG =~ s/\?/"'".$el."'"/e;
        }
        for my $el (@$IDS){
            $LOG =~ s/\?/"'".$el."'"/e;
        }
        print STDERR "Edit: $LOG\n";
    } 

    $self->dbh->do($SQL, undef,
        @{$data}{@$fields}, @$IDS
    );
}

#Получает фильтр, возвращает строку ограничений и указатель на массив значений
#Приводит к SQL-строке с массивом значений к ней
sub filter2str {
    my ($self, $filter, $pref) = @_;
    my $f = $self->create_f([$filter, $pref]);
    return $f->filter2str;
}

# число попыток сделать запрос
our $TRIES = 5;
sub _sth_execute {
    my ($dbh, $sth) = (shift, shift);
    my ($err, $errstr);
    for my $try (1 .. $TRIES) {
        my $rv = eval { $sth->execute(@_) };
        return $rv if $rv;

        $err = $sth->err // $dbh->err // $DBI::err // -1;
        $errstr = $sth->errstr // $dbh->errstr // $DBI::errstr // '?';

        if ($err == 2013) { #2013, Lost connection to MySQL server during query.
            # здесь делать reconnect нельзя, почему-то потом не определяется $dbh->err
            # по-видимому, reconnect делается автоматически, за счёт mysql_auto_reconnect = 1
            print STDERR "WARN: try $try of $TRIES failed: ($err, $errstr)\n";
        } elsif ($err != 1205 and $err != 1213) {
            print STDERR $dbh->proj->stack_trace('  SQL ERROR   ');
            die("Can't execute the $sth->{Statement}: $err, $errstr.");
        }
        sleep 5;
    }
    die("Can't execute the $sth->{Statement} after $TRIES retries: $err, $errstr.");
}

sub LogDump {
    my ($self, @arr) = @_;
    $self->proj->dd(\@arr);
}

#Получения списка по SQL запросу и списку значений
sub List_SQL {
    my ($self, $SQL, $arr) = @_;

    if ($ENV{_DISALLOW_SQL} || $ENV{_EMPTY_SQL}) {
        my $c = 0;
        my @caller = caller($c);
        while (
            $caller[0] =~ /DataSource::/
            || $caller[3] =~ /::[^:]*SQL[^:]*$/
            || ($caller[0] =~ /Project/ && $caller[2] > 3280 && $caller[2] < 3320)
        ) {
            $c++;
            @caller = caller($c);
        }
        warn ">>> List_SQL FROM ". join " : ", @caller[0 .. 4], "\n";

        for (1 .. 3) {
            $c++;
            @caller = caller($c);
            warn ">>>          FROM ". join " : ", @caller[0 .. 4], "\n";
        }
    }
    if ($ENV{_DISALLOW_SQL}) {
        die "SQL DISALLOWED!!!\n$SQL\n" . eval { defined $arr ? "@$arr" : "@" };
    }
    if ($ENV{_EMPTY_SQL}) {
        return [];
    }

    my $start_time;
    $start_time = [gettimeofday]   if defined $long_requests_logging_min_duration;

    my @arr_v = $arr ? @$arr : ();

    $self->LogDump([$SQL, $arr]) if $self->{'sql_log'} && ($self->{'sql_log'} == 1);

    my $dbh = $self->dbh_read;
    $dbh = $self->dbh if ((! $dbh) || $self->dont_use_dbh_read);
    $dbh = $self->dbh_heavy if ($self->use_dbh_heavy);
    
    #Отмечаем время до начала запроса
    my $lasttime = gettimeofday;
    #Предварительный вывод запроса
    my $SQL_ = $SQL;
    if($self->{'sql_log'}||$self->{'sql_dd_log'}) {
        for my $el (@$arr){
            $SQL_ =~ s/\?/"'".$el."'"/e;
        }
    }
    if($self->{'sql_log'} && $self->{'sql_log'} < 3) {
        print STDERR "$SQL_\n";
    } 

    my $sth = $dbh->prepare($SQL);
    _sth_execute($dbh, $sth, @arr_v);

    my $cur = gettimeofday;
    my $t = $cur - $lasttime;

    my @w = ();
    while (my $x = $sth->fetchrow_hashref) {
        push @w, $x;
    }

    #Пишем в лог, если запрос долго отрабатывался/доставался
    if($self->{'sql_log'} && ($self->{'sql_log'} == 3)){
        print STDERR "$SQL_\n";
    }

    if($self->{'sql_dd_log'}){
        $self->proj->dd( $t, $SQL, \@arr_v, $SQL_ );
    }

    if (defined $long_requests_logging_min_duration) {
        my $duration = tv_interval($start_time);
        if ($duration >= $long_requests_logging_min_duration) {
            my $max_length = 2000;
            my $sql_str = substr($SQL, 0, $max_length);
            $sql_str =~ s/\s{2,}/  /g;
            $self->log("Long List_SQL: duration: $duration SQL: $sql_str");
        }
    }

    return \@w;
}

sub update {
    my ($self, $flt, $data) = @_;
    my ($where_d, $arr) = $self->filter2str($flt);

    my $fields = $self->editing_fields;
    $fields = [ keys %$data ] unless $fields;
    $self->CleanseFields($data);

    my $SQL = sprintf("UPDATE %s SET %s ", 
        $self->db_table, 
        join( ', ', map { $self->FieldScreening($_).'=?' } @$fields ), 
    );
    $arr = [@{$data}{@$fields}, @$arr];

    if( $where_d ){
        $SQL .= " where $where_d ";
    }

    #print STDERR Dumper([$SQL, $arr]);
    print STDERR Dumper([$SQL, $arr]) if $self->{sql_log};
    $self->Do_SQL($SQL, $arr);
    
    #$self->List_Filter_Data(\%opts);
}

#Получения списка с приведённым к стандартному виду фильтром
#(строка фильтра, массив для фильтра, выводимые поля)
sub List_Filter_Data {
    my ($self, $opts) = @_;

    my $db_table = $self->db_table;

    my $gfields = $opts->{'gfields'};
    $gfields = $self->listing_fields unless $gfields;
    $gfields = ['*'] unless $gfields;
    
    my $select_fields;
    
    if ($opts->{count}) {
        my $count = $opts->{'count'};
        if( ref($count) eq 'ARRAY' ){
             $count = 'DISTINCT '.join(', ', $self->FieldsScreening(@$count));
        }elsif(defined($count)&&($count>0)){
             $count = '*';
        }
        $select_fields = "count($count) c";
    }
    elsif (my $dist = $opts->{distinct}) {
        $select_fields = 'DISTINCT ' . join(', ', $self->FieldsScreening(@$dist)); 
    }
    elsif (my $sum_filter = $opts->{sum_filter}) {
        my $comma = '';
        my @arr;
        while (my ($name, $filter) = each %$sum_filter) {
            my ($where, $arr) = $self->filter2str($filter);
            push(@arr, @$arr);
            $select_fields .= $comma . "SUM(IF($where,1,0))" . ' ' . $self->FieldScreening($name);
            $comma = ', ';
        }
        unshift(@{$opts->{arr}}, @arr);
    }
    else {
        my $comma = "";
        my @arr;
        foreach my $field (@$gfields) {
            if (ref($field) eq 'ARRAY') {
                my ($alias, $filter) = @$field;
                
                my $f = $self->create_f;
                $f->add_f($filter);
                
                my ($where, $arr) = $f->filter2str;
                $select_fields .= $comma . $where . ' ' . $alias;
                push(@arr, @$arr);
            }
            else {
                $select_fields .= $comma . $self->FieldScreening($field);
            }
            $comma = ", ";
        }
        unshift(@{$opts->{arr}}, @arr);
    }

    my $SQL = "select $select_fields from $db_table";
    $SQL .= " where ".$opts->{'where_d'} if $opts->{'where_d'};
    $SQL .= " group by " . $self->FieldsStringScreening($opts->{'group_by'}) if $opts->{'group_by'};
    $SQL .= " order by " . $self->FieldsStringScreening($opts->{'order_by'}) if $opts->{'order_by'};
    $SQL .= " limit ".$opts->{'limit'} if $opts->{'limit'};

    my $result = $self->List_SQL($SQL, $opts->{'arr'});
    return $result->[0] if $opts->{sum_filter};
    return $result;
}

# List2(filter => [], gfields => [], limit => "", order_by => "", group_by => "")
sub List2 {
    my ($self, %opts) = @_;
    
    if ($opts{filter}) {
        ($opts{where_d}, $opts{arr}) = $self->filter2str(delete $opts{filter});
    }
    return $self->List_Filter_Data(\%opts);
}

sub List {
    my ($self, $filter, $gfields, $lim, $order_by) = @_;
    my ($where_d, $arr) = $self->filter2str($filter);
    return $self->List_Filter_Data({'where_d' => $where_d, 'arr' => $arr, 'gfields' => $gfields, limit => $lim, order_by => $order_by });
}

sub Count {
    my ($self, $filter, $gfields) = @_;

    my ($where_d, $arr) = $self->filter2str($filter);
    my $count = $gfields ? $gfields : 1;
    return $self->List_Filter_Data({'where_d' => $where_d, 'arr' => $arr, 'gfields' => $gfields, 'count' => $count })->[0]->{'c'};
}

#Получить первый элемент по идентификационному полю или 
#по указанному полю, если передаётся два параметра
sub Get {
    my $self = shift;
    my ($idf, $id);
    if(@_ == 1){
        $idf = $self->id_field;
        $id = $_[0];
    }elsif(@_ > 1){
        ($idf, $id) = @_;
    }else{
        return undef;
    }

    my $rows = $self->List({ $idf => $id }, ['*']);

    unless(@$rows){
        return undef;
    }else{
        return $rows->[0];
    }
}

sub CleanseFields {
    my ($self, $data) = @_;
    $data->{$_} = $self->CleanseField($_, $data->{$_}) for keys %$data;
    return $data;
}

sub CleanseField {
    my ($self, $name, $value) = @_;
    return $value;
}

#Получает хеш изменений и возвращает список полей, которые менялись
sub CheckChanges {
    my ($self, $id, $data) = @_;

    my @changes = ();

    my $d = $self->Get($id);

    for(keys %$data){
        push(@changes, $_) if $self->CleanseField($_, $d->{$_} || '') ne $self->CleanseField($_, $data->{$_} || '');
    }

    return \@changes;
}

sub Do_SQL {
    my $self = shift;
    my ($SQL, $arr) = @_;

    #local $SIG{'__DIE__'} = sub { print STDERR $self->proj->stack_trace('  SQL ERROR   '); };

    $self->dbh->do($SQL, undef, @{$arr}) || do {
        print STDERR $self->proj->stack_trace('  Do_SQL error  ')."\n";
        #print STDERR 'Do_SQL error DBI: '.$DBI::err .' => '. $DBI::errstr."\n";
        #print STDERR 'Do_SQL error SQL: '.$SQL."\n";
        die('Do_SQL error: '.$DBI::err .' => '. $DBI::errstr."    SQL: ".$SQL);
    };
};

sub Do_SQL2 {
    my $self = shift;
    my ($SQL, $arr) = @_;

    local $SIG{'__DIE__'} = sub { print STDERR $self->proj->stack_trace('  SQL ERROR   '); };

    $self->dbh_read->do($SQL, undef, @{$arr});
};

#Получить колонку данных
sub Column {
    my ($self, $field, $filter, %opts) = @_;
    my $fldname = $field;
    $fldname=~s/^.+?\s+as\s+//si; # Чтобы уметь обрабатывать конструкции 'SUM(DistrSerpShows) as DistrSerpShows'
    return [ map { $_->{$fldname} } @{$self->List2(filter => $filter, gfields => [$field], %opts)} ];
}

#Начало транзакции
sub begin_work {
    my ($self) = @_;
    $self->dbh->begin_work;
}

#Завершение транзакции
sub commit {    
    my ($self) = @_;
    $self->dbh->commit;
}

#Откат транзакции
sub rollback {  
    my ($self) = @_;
    $self->dbh->rollback;
}

sub FormatSQL {
    my ($self, $sql)=@_;
    $sql=~s/(FROM|WHERE|AS [\w\.`]+,)/$1\n/sg;
    return $sql;
}

sub table_status {
    my ($self) = @_;
    my $res = $self->List_SQL("show table status like '".$self->db_table."'");
    return @$res > 0 ? $res->[0] : {};
}

sub GetRand {
    my ($self, $cc, $filter) = @_;
    $cc += 0; #Приводим к числу
    $cc ||= 1;
    my $SQL = 'SELECT r1.* FROM '.$self->db_table.' AS r1 JOIN (  SELECT (RAND() * (SELECT MAX('.$self->id_field.') FROM '.$self->db_table.')) AS '.$self->id_field.' ) AS r2 WHERE r1.'.$self->id_field.' >= r2.'.$self->id_field.' ORDER BY r1.'.$self->id_field.' ASC LIMIT 1';
    if( $filter ){
        my ($where_d, $arr) = $self->filter2str($filter);
        $arr = [ map { $self->dbh->quote($_) } @$arr ];
        $where_d =~ s/\?/$_/ for @$arr; 

        $SQL = 'SELECT r1.* FROM '.$self->db_table.' AS r1 JOIN (  SELECT (RAND() * (SELECT MAX('.$self->id_field.') FROM '.$self->db_table.')) AS '.$self->id_field.' ) AS r2 WHERE r1.'.$self->id_field.' >= r2.'.$self->id_field.' AND '.$where_d.' ORDER BY r1.'.$self->id_field.' ASC LIMIT 1';
        print "SQL: $SQL\n"; 
    }
    if( $cc == 1 ){
        return $self->List_SQL($SQL);
    }else{
        #Быстрый, но проблема разжеренности
        my $SQL_M = "($SQL) ".( " UNION ($SQL)" x (2*$cc) );
        $SQL_M = "select * from ($SQL_M) tb limit $cc";
        return $self->List_SQL($SQL_M);

        #Очень медленный
        #my $l = $self->List_SQL('SELECT count(*) cc FROM '.$self->db_table);
        #my $mx = $l->[0]{cc};
        #my $llp =  ($cc + 2) / ($mx + 1);
        #my $SQL_M = "select * from ".$self->db_table." tb where RAND() < $llp limit $cc";
        #return $self->List_SQL($SQL_M);

        #my $l = $self->List_SQL('SELECT count(*) cc FROM '.$self->db_table);
        #my $mx = $l->[0]{cc};
        #my $SQL_L = 'SELECT * FROM '.$self->db_table." limit RAND() * $mx , 1";
        #my $SQL_M = "($SQL_L) ".( " UNION ($SQL_L)" x (2*$cc) );
        #$SQL_M = "select * from ($SQL_M) tb limit $cc";
        #return $self->List_SQL($SQL_M);
    }
}

########################################################
#Перегрузка операторов
########################################################

########################################################
#Деструктор
########################################################

sub DESTROY {
}

########################################################

#Завершающий код
END{
}




1;

