package DataSource::ElemFactory;

use std;

use base qw(ObjLib::ProjPart);
use File::Slurp qw(read_file write_file);
use File::Path qw(make_path);
use DBI;
use DataSource::Elem;

my %db_dump_elems;

sub elem_from_db_dump {
    my ($self, $group, $table) = @_;
    if ( !exists $db_dump_elems{$group}{$table} ) {
        my $opts = $self->proj->options->{db_dump}->{$group};
        my $elem = $self->elem_from_json_file(join('_',$group,$table), $opts->{table_file}{$table});
        $db_dump_elems{$group}{$table} = $elem;
    }
    return $db_dump_elems{$group}{$table};
}

sub elem_from_json_file {
    my ($self,$table, $filename) = @_;
    my $rows = $self->_load_rows_from_json_file($filename);
    my $elem = $self->elem_from_arrayref($table, $rows);
    return $elem;
}

sub elem_from_arrayref {
    my ($self, $table, $rows) = @_;
    return 'empty' if !@$rows;  # temporary
    my %fields_hash = map {$_ => 1} map {keys %{$_}} @$rows;
    my @fields = keys %fields_hash;
    my $dbh = $self->_sqlite_dbh;
    $dbh->do(join (' ',
        'create table',
        $dbh->quote_identifier($table),
        '(',
            join (', ', map {$dbh->quote($_)} @fields),
        ')',
    ));
    my $elem = DataSource::Elem->new({
        dbh => $dbh,
        dbh_read => $dbh,   
        proj => $self->proj,
        dbhname => 'sqlite',
        db_table => $table,
        adding_fields => \@fields,
        editing_fields => \@fields,
        listing_fields => \@fields,
    });
    $elem->Add($_) foreach @$rows;
    return $elem;
}

sub table_to_db_dump {
    my ($self, $group, $table) = @_;
    my $opts = $self->proj->options->{db_dump}{$group};
    my $rows = $self->_load_rows_from_mysql($opts->{dbh_name}, $table);

    my $dir = $opts->{dir};
    make_path($dir);

    $self->_save_rows_to_json_file($rows, $opts->{table_file}{$table})
}

sub _load_rows_from_json_file {
    my ($self, $filename) = @_;
    my $rows_json = read_file($filename)
        or die "Can't read file $filename: $!";
    my $rows = $self->_json->decode($rows_json);
    return $rows;
}

sub _save_rows_to_json_file {
    my ($self, $rows, $filename) = @_;
    my $proj = $self->proj;
    my $rows_json = $self->_json->encode($rows);
    if (-f $filename and read_file($filename) eq $rows_json) {
        $proj->log("file $filename: not changed");
        return;
    }
    write_file("$filename.tmp", $rows_json)
        or die "Can't write file $filename: $!";
    rename("$filename.tmp", $filename)
        or die "Can't rename $filename: $!";

    $proj->log("written new data to $filename");
}

sub _load_rows_from_mysql {
    my ($self, $dbh_name, $table) = @_;
    my $db_elem = $self->proj->dbtable($table, undef, $dbh_name);
    return $db_elem->List();
}

sub _sqlite_dbh :CACHE {
    my $self = shift;
    my $dbh = DBI->connect("dbi:SQLite:dbname=:memory:","","");
    return $dbh;
}

sub _json :CACHE {
    my $self = shift;
    return JSON->new->utf8(1)->canonical(1);
}

1;

