package Dates;

#Модуль для преобразования форматов дат

use utf8;
use std;

use base qw(ObjLib::Obj);

use POSIX qw(strftime);
use Time::Local;

use Time::HiRes qw/time/;

use Date::Calc qw(Days_in_Month Day_of_Week Add_Delta_DHMS Add_Delta_YMD Week_of_Year Monday_of_Week check_date);

########################################################
#Список методов
########################################################

#     delta_time($self, $d1, $d2, $i_format, $o_format)        Разница по времени между датами, $o_format - seconds, minutes, hours
#     cur_date($self,$f2,$d)                                   Текущая дата, можно указать формат и смещение в секундах

########################################################
#Вспомогательные переменные
########################################################

our ($months, $hcs, $week_days, $holidays);

sub gettext { return @_; }

sub init_vars {
    $months = {
         '1' => { c => '31', n => 'января',   ni => gettext('январь'),  nr => gettext('января'),   nd => 'январю',   nv => 'январь',   nt => 'январём',    np => 'январе',    nc => 'янв',  ncap => 'Январь',       },
         '2' => { c => '28', n => 'февраля',  ni => gettext('февраль'), nr => gettext('февраля'),  nd => 'февралю',  nv => 'февраль',  nt => 'февралём',   np => 'феврале',   nc => 'фев',  ncap => 'Февраль',      },
         '3' => { c => '31', n => 'марта',    ni => gettext('март'),    nr => gettext('марта'),    nd => 'марту',    nv => 'март',     nt => 'мартом',     np => 'марте',     nc => 'мар',  ncap => 'Март',         },
         '4' => { c => '30', n => 'апреля',   ni => gettext('апрель'),  nr => gettext('апреля'),   nd => 'апрелю',   nv => 'апрель',   nt => 'апрелем',    np => 'апреле',    nc => 'апр',  ncap => 'Апрель',       },
         '5' => { c => '31', n => 'мая',      ni => gettext('май'),     nr => gettext('мая'),      nd => 'маю',      nv => 'май',      nt => 'маем',       np => 'мае',       nc => 'мая',  ncap => 'Май',          },
         '6' => { c => '30', n => 'июня',     ni => gettext('июнь'),    nr => gettext('июня'),     nd => 'июню',     nv => 'июнь',     nt => 'июнем',      np => 'июне',      nc => 'июн',  ncap => 'Июнь',         },
         '7' => { c => '31', n => 'июля',     ni => gettext('июль'),    nr => gettext('июля'),     nd => 'июлю',     nv => 'июль',     nt => 'июлем',      np => 'июле',      nc => 'июл',  ncap => 'Июль',         },
         '8' => { c => '31', n => 'августа',  ni => gettext('август'),  nr => gettext('августа'),  nd => 'августу',  nv => 'август',   nt => 'августот',   np => 'августе',   nc => 'авг',  ncap => 'Август',       },
         '9' => { c => '30', n => 'сентября', ni => gettext('сентябрь'),nr => gettext('сентября'), nd => 'сентябрю', nv => 'сентябрь', nt => 'сентябрём',  np => 'сентябре',  nc => 'сен',  ncap => 'Сентябрь',     },
        '10' => { c => '31', n => 'октября',  ni => gettext('октябрь'), nr => gettext('октября'),  nd => 'октябрю',  nv => 'октябрь',  nt => 'октябрём',   np => 'октябре',   nc => 'окт',  ncap => 'Октябрь',      },
        '11' => { c => '30', n => 'ноября',   ni => gettext('ноябрь'),  nr => gettext('ноября'),   nd => 'ноябрю',   nv => 'ноябрь',   nt => 'ноябрём',    np => 'ноябре',    nc => 'ноя',  ncap => 'Ноябрь',       },
        '12' => { c => '31', n => 'декабря',  ni => gettext('декабрь'), nr => gettext('декабря'),  nd => 'декабрю',  nv => 'декабрь',  nt => 'декабрём',   np => 'декабре',   nc => 'дек',  ncap => 'Декабрь',      },
    };

    $hcs = {
        wd => '0',  # weekend
        hl => '1',  # holiday
    };

    $week_days = [
    '0',
    'пн',
    'вт',
    'ср',
    'чт',
    'пт',
    'сб',
    'вс',
];

    #Праздники
    $holidays = {
        '^\d{4}-02-23'       => $hcs->{'hl'},
        '^\d{4}-03-08'       => $hcs->{'hl'},
        '^\d{4}-05-01'       => $hcs->{'hl'},
        '^\d{4}-05-09'       => $hcs->{'hl'},
        '^\d{4}-06-12'       => $hcs->{'hl'},
        '^\d{4}-11-04'       => $hcs->{'hl'},
        '^2006-01-(0\d|09)'  => $hcs->{'hl'},
        '^2006-02-24'        => $hcs->{'wd'}, #Перенос выходного
        '^2006-02-26'        => '',
        '^2006-05-06'        => '', #Перенос выходного
        '^2006-05-08'        => $hcs->{'wd'},
        '^2006-11-06'        => $hcs->{'wd'}, #Перенос выходного
        '^2007-06-09'        => '', #Перенос выходного
        '^2007-06-11'        => $hcs->{'wd'},
    };
}    


########################################################
# Инициализация
########################################################

sub init {
    my $self = shift;
    $self->init_vars;

    my $month_r = sub { $self->{'months'} };

    $self->{trs} = {
        'norm'   => {
                    '>' => sub { return [ @{$_[0]} ]; },
                    '<' => sub { return [ @{$_[0]} ]; },
        },
        'form'   => {
                    '>' => sub { my $m = [ @{$_[0]}[0..2] ]; $m->[0] -= 2000; $m->[0] = '0'.$m->[0] if $m->[0] < 10; return $m; },
                    '<' => sub { my $m = [ @{$_[0]}, 0,0,0 ]; $m->[0] += 2000; return $m; },
        },
        'bs'     => { 
                    '>' => sub { return sprintf('%04d%02d%02d000000', @{$_[0]}); },
                    '<' => sub { if(shift()=~/(\d{4})(\d{2})(\d{2})000000/){ return [$1,$2,$3,0,0,0]; }else{ return []; } },
        },
        'bs_time'     => { 
                    '>' => sub { return sprintf('%04d%02d%02d%02d%02d%02d', @{$_[0]}); },
                    '<' => sub { if(shift()=~/(\d{4})(\d{2})(\d{2})(\d{2})(\d{2})(\d{2})/){ return [$1,$2,$3,$4,$5,$6]; }else{ return []; } },
        },
        'direct' => { 
                    '>' => sub { return sprintf('%04d%02d%02d', @{$_[0]}); },
                    '<' => sub { if(shift()=~/(\d{4})(\d{2})(\d{2})/){ return [$1,$2,$3,0,0,0]; }else{ return []; } },
        },
        'market' => {
                    '>' => sub { return sprintf('%02d/%02d/%04d', reverse @{$_[0]}[0..2]); },
                    '<' => sub { if( shift() =~ /(\d{2})\/(\d{2})\/(\d{4})/ ){ return [$3,$2,$1,0,0,0]; }else{ return []; } },
        },
        'user'   => {
                    '>' => sub { return sprintf('%02d.%02d.%04d', reverse @{$_[0]}[0..2]); },
                    '<' => sub { if( shift() =~ /(\d{2})\.(\d{2})\.(\d{4})/ ){ return [$3,$2,$1,0,0,0]; }else{ return []; } },
        },
        'user_full'=>{
                    '>' => sub { return sprintf('%02d.%02d.%04d %02d:%02d:%02d', reverse(@{$_[0]}[0..2]), @{$_[0]}[3..5] ); },
                    '<' => sub { if( shift() =~ /(\d{2})\.(\d{2})\.(\d{4}) (\d{2})\:(\d{2})\:(\d{2})/ ){ return [$6,$5,$4,$3,$2,$1]; }else{ return []; } },
        },
        'user_txt' => {
                    '>' => sub { return sprintf('%02d %s %04d', $_[0][2], $months->{int($_[0][1])}{'nc'}, $_[0][0]); },
                    '<' => sub {my %hs = map {$months->{$_}{'nc'} => $_} (keys %$months); if( shift() =~ /(\d{2}) (.{3}) (\d{4})/ && $hs{$2}){ return [$3,$hs{$2},$1,0,0,0]; }else{ return []; } },
        },
        'user_txt_long_fixed' => {
                    '>' => sub { return sprintf('%d %s %04d', $_[0][2], $months->{int($_[0][1])}{'nr'}, $_[0][0]); },
                    '<' => sub {my %hs = map {$months->{$_}{'nc'} => $_} (keys %$months); if( shift() =~ /(\d{2}) (.{3}) (\d{4})/ && $hs{$2}){ return [$3,$hs{$2},$1,0,0,0]; }else{ return []; } },
        },
        'user_txt_long' => {
                    '>' => sub { return sprintf('%02d %s %04d', $_[0][2], $months->{int($_[0][1])}{'nr'}, $_[0][0]); },
                    '<' => sub {my %hs = map {$months->{$_}{'nc'} => $_} (keys %$months); if( shift() =~ /(\d{2}) (.{3}) (\d{4})/ && $hs{$2}){ return [$3,$hs{$2},$1,0,0,0]; }else{ return []; } },
        },
        'user_txt_short' => {
                    '>' => sub { return sprintf('%d %s', $_[0][2], $months->{int($_[0][1])}{'nr'}, $_[0][0]); },
                    '<' => sub {my %hs = map {$months->{$_}{'nc'} => $_} (keys %$months); if( shift() =~ /(\d{2}) (.{3})/ && $hs{$2}){ return [(localtime)[5],$hs{$2},$1,0,0,0]; }else{ return []; } },
        },
        'user_full_txt' => {
                    '>' => sub { return sprintf('%02d %s %04d %02d:%02d:%02d', $_[0][2], $months->{int($_[0][1])}{'nc'}, $_[0][0], @{$_[0]}[3..5]); },
                    '<' => sub {my %hs = map {$months->{$_}{'nc'} => $_} (keys %$months); if( shift() =~ /(\d{2}) (.{3}) (\d{4}) (\d{2})\:(\d{2})\:(\d{2})/ && $hs{$2}){ return [$3,$hs{$2},$1,$4,$5,$6]; }else{ return []; } },
        },
        'user_os_full_txt' => {
                    '>' => sub { return sprintf('%02d %s %04d в %02d:%02d', $_[0][2], $months->{int($_[0][1])}{'nc'}, $_[0][0], @{$_[0]}[3..4]); },
                    '<' => sub {my %hs = map {$months->{$_}{'nc'} => $_} (keys %$months); if( shift() =~ /(\d{2}) (.{3}) (\d{4}) в (\d{2})\:(\d{2})/ && $hs{$2}){ return [$3,$hs{$2},$1,$4,$5,0]; }else{ return []; } },
        },
        'db'     => {
                    '>' => sub { return sprintf('%04d-%02d-%02d', @{$_[0]}); },
                    '<' => sub { if( shift() =~ /(\d{4})-(\d{2})-(\d{2})(?:\s+|T\d{1,2}:\d{1,2}:\d{1,2})?/ ){ return [$1,$2,$3,0,0,0]; }else{ return []; } },
        },
        'sec'   => {
#                    '>' => sub { return (shift() =~ /(\d{4})-(\d{2})-(\d{2})(\s|T)(\d{1,2}):(\d{1,2}):(\d{1,2})/ ? timelocal($7, $6, $5, $3, ($2 - 1), ($1 - 1900)) : 0 ); },
#                    '<' => sub { my @dt = localtime(shift()); return sprintf("%04d-%02d-%02d %02d:%02d:%02d", $dt[5]+1900, $dt[4]+1, $dt[3], $dt[2], $dt[1], $dt[0]); },
                    '>' => sub { return $_[0]->[1] < 1 ? 0 : timelocal($_[0]->[5] || 0, $_[0]->[4] || 0, $_[0]->[3] || 0, $_[0]->[2], ($_[0]->[1] - 1), ($_[0]->[0] - 1900)); },
                    '<' => sub { my @dt = localtime(shift()); return [ $dt[5]+1900, $dt[4]+1, $dt[3], $dt[2], $dt[1], $dt[0] ]; },
        },
        'mpcrmjson'   => {
                    '>' => sub { return $_[0]->[1] < 1 ? 0 : timelocal($_[0]->[5] || 0, $_[0]->[4] || 0, $_[0]->[3] || 0, $_[0]->[2], ($_[0]->[1] - 1), ($_[0]->[0] - 1900)); },
                    '<' => sub { if( $_[0] =~ /(\d+)\d{3}/ ){ my @dt = localtime($1); return [ $dt[5]+1900, $dt[4]+1, $dt[3], $dt[2], $dt[1], $dt[0] ]; } else { return [] } },
        },
        'db_time' => {
                    '>' => sub { return sprintf("%04d-%02d-%02d %02d:%02d:%02d", @{$_[0]} ); },
                    '<' => sub { return (shift() =~ /(\d{4})-(\d{2})-(\d{2})(?:(?:\s+|T)(\d{1,2}):(\d{1,2}):(\d{1,2}))?/ ? [ $1, $2, $3, $4, $5, $6 ] : [] ); },
        },
        'timings' => {
                    '>' => sub { return sprintf("%04d-%02d-%02d_%02d:%02d:%02d", @{$_[0]} ); },
                    '<' => sub { return (shift() =~ /(\d{4})-(\d{2})-(\d{2})(?:(?:_)(\d{1,2}):(\d{1,2}):(\d{1,2}))?/ ? [ $1, $2, $3, $4, $5, $6 ] : [] ); },
        },
        'db_time_hr' => {
                    '>' => sub { return sprintf("%04d-%02d-%02d %02d:%02d:%02.4f", @{$_[0]}[0 .. 4], $_[0][5]+($_[0][6]||0) ); },
                    '<' => sub { return (shift() =~ /(\d{4})-(\d{2})-(\d{2})(?:(?:\s+|T)(\d{1,2}):(\d{1,2}):(\d{1,2}))?/ ? [ $1, $2, $3, $4, $5, $6 ] : [] ); },
        },
        #Формирует строку, но в обратную сторону принимает хеш
        'cgi1'     => { 
                    '>' => sub { return sprintf('&y1=%02d&m1=%02d&d1=%02d', ($_[0]->[0]-2000 , @{$_[0]}[1,2] )); },
                    '<' => sub { return [$_[0]->{'y1'}+2000,$_[0]->{'m1'},$_[0]->{'d1'},0,0,0]; },
        },
        #Формирует строку, но в обратную сторону принимает хеш
        'cgi2'     => { 
                    '>' => sub { return sprintf('&y2=%02d&m2=%02d&d2=%02d', ($_[0]->[0]-2000 , @{$_[0]}[1,2] )); },
                    '<' => sub { return [$_[0]->{'y2'}+2000,$_[0]->{'m2'},$_[0]->{'d2'},0,0,0]; },
        },
        'month' => {
                    '>' => sub { return $_[0]->[0].'-'.$_[0]->[1]; },
                    '<' => sub { return [$_[0]->{'y2'}+2000,$_[0]->{'m2'},$_[0]->{'d2'},0,0,0]; },
        },
        'week' => {
                    '>' => sub {
                            my $d = $_[0];
                            
                            my $timet = timelocal(0, 0, 0, $d->[2], ($d->[1] - 1), ($d->[0] - 1900));
                            my $weekd = strftime("%u", localtime($timet)) - 1;
                            
                            my $from = strftime("%02y.%m.%d", localtime($timet - ($weekd*86400)) ) ;
                            my $to   = strftime("%02y.%m.%d", localtime($timet + ((6 - $weekd)*86400)) );
                            
                            my $key = "$from\&nbsp;-\&nbsp;$to";
                            
                            return $key;
                    },
                    '<' => sub { return [] },
        },
        'week_txt' => {
                    '>' => sub {
                            my $d = $_[0];
                            
                            my $timet = timelocal(0, 0, 0, $d->[2], ($d->[1] - 1), ($d->[0] - 1900));
                            my $weekd = strftime("%u", localtime($timet)) - 1;
                            
                            my $from = strftime("%Y-%m-%d", localtime($timet - ($weekd*86400)) ) ;
                            my $to   = strftime("%Y-%m-%d", localtime($timet + ((6 - $weekd)*86400)) );
                            
                            my $key = "$from - $to";
                            
                            return $key;
                    },
                    '<' => sub { return [] },
        },
        #сжатый формат
        'week2' => {
                    '>' => sub {
                            my $d = $_[0];
                            
                            my $timet = timelocal(0, 0, 0, $d->[2], ($d->[1] - 1), ($d->[0] - 1900));
                            my $weekd = strftime("%u", localtime($timet)) - 1;
                            
                            my $from = strftime("%d.%m.%y", localtime($timet - ($weekd*86400)) ) ;
                            my $to   = strftime("%d.%m.%y", localtime($timet + ((6 - $weekd)*86400)) );
                            
                            my $key = "$from\&nbsp;-\&nbsp;$to";
                            
                            return $key;
                    },
                    '<' => sub { return [] },
        },
        ## Дату norm в строку %x%v - как в MySQL
        'db_week' => {
                    '>' => sub { return sprintf("%04d%02d", reverse Week_of_Year($_[0][0],$_[0][1],$_[0][2])); },
                    '<' => sub { my $text = shift; if( $text=~/(\d{4})(\d{2})/ ){return [Monday_of_Week($2, $1)];}else{ return []} },
        },
        'db_week_monday' => {
                    '>' => sub { return sprintf("%04d-%02d-%02d", 
                        Monday_of_Week(
                            (Week_of_Year($_[0][0],$_[0][1],$_[0][2]))[0],
                            $_[0][0]
                            )
                        ); },
                    '<' => sub { return [] },
        },
        'month_txt' => {
                    '>' => sub { $_[0][1]=~s/^0?//; return $months->{$_[0][1]}->{'ni'}; },
                    '<' => sub { return [] },
        },
        'month_txt_capitalized' => {
                    '>' => sub { $_[0][1]=~s/^0?//; return $months->{$_[0][1]}->{'ncap'}; },
                    '<' => sub { return [] },
        },
        'month_txt_r' => {
                    '>' => sub { $_[0][1]=~s/^0?//; return $months->{$_[0][1]}->{'nr'}; },
                    '<' => sub { return [] },
        },
        #Для актов
        'acts' => {
                    '>' => sub { return $_[0]->[2].' '.$months->{$_[0]->[1]+0}->{'nr'}.' '.$_[0]->[0].' года'; },
                    '<' => sub { return []; },
        },
        #Для актов
        'acts_month_i' => {
                    '>' => sub { return $months->{$_[0]->[1]+0}->{'ni'}.' '.$_[0]->[0].' года'; },
                    '<' => sub { return []; },
        },
        #Для актов
        'acts_month_p' => {
                    '>' => sub { return $months->{$_[0]->[1]+0}->{'np'}.' '.$_[0]->[0].' года'; },
                    '<' => sub { return []; },
        },
        #Для актов
        'acts_month_r' => {
                    '>' => sub { return $months->{$_[0]->[1]+0}->{'nr'}.' '.$_[0]->[0].' года'; },
                    '<' => sub { return []; },
        },
        'MM/YY'     => {
                    '>' => sub { return sprintf('%02d/%02d', $_[0]->[1], $_[0]->[0] % 100); },
                    '<' => sub { return []; },
        },
        'YYYYMMDDHH' => {
                    '>' => sub { return sprintf('%04d%02d%02d%02d', @{$_[0]} ); },
                    '<' => sub { return (shift() =~ /(\d{4})(\d{2})(\d{2})(\d{2})/ ? [ $1, $2, $3, $4, 0, 0 ] : [] ); },
        },
        'balance' => {
                    '>' => sub { return sprintf("%04d%02d%02dT%02d:%02d:%02d", @{$_[0]} ); },
                    '<' => sub { return (shift() =~ /(\d{4})(\d{2})(\d{2})(?:(?:\s+|T)(\d{1,2}):(\d{1,2}):(\d{1,2}))?/ ? [ $1, $2, $3, $4, $5, $6 ] : [] ); },
        },
        #Для определения праздников и выходных
        'holiday' => {
                    '>' => sub {
                                 my $d = sprintf("%04d-%02d-%02d", @{$_[0]} );
                                 for my $k ( keys %$holidays ){
                                     return $holidays->{$k} if $d =~ /$k/;
                                 }
                                 return if not check_date($_[0]->[0], $_[0]->[1], $_[0]->[2]);
                                 my $wd = Day_of_Week($_[0]->[0], $_[0]->[1], $_[0]->[2]);
                                 return (($wd == 6) || ($wd == 7)) ? $hcs->{'hl'} : $hcs->{'wd'};
                    },
                    '<' => sub { return []; },
        },
        #Для определения дней недели
        'week_day' => {
                    '>' => sub {
                                 return if not check_date($_[0]->[0], $_[0]->[1], $_[0]->[2]);
                                 return $week_days->[ Day_of_Week($_[0]->[0], $_[0]->[1], $_[0]->[2]) ];
                    },
                    '<' => sub { return []; },
        },
        yt_attr => {
                    '>' => sub { return sprintf("%04d-%02d-%02dT%02d:%02d:%02d", @{$_[0]} ); },
                    '<' => sub { return (shift() =~ /(\d{4})-(\d{2})-(\d{2})T(\d{2}):(\d{2}):(\d{2})/ ? [ $1, $2, $3, $4, $5, $6 ] : [] ); },
        }
    };

}

########################################################
#Методы
########################################################

#Преобразуем месяц в количество дней
sub DaysInMonth {
    my ($self, $y, $m) = @_;
    return Days_in_Month($y, $m);
#    return $self->{'months'}->{$m} if ($m != 2) || (! $y);
#    return ( ( ($y % 4) == 0 ) && ( ($y % 400) != 0 ) )  ? $self->{'months'}->{$m}+1 : $self->{'months'}->{$m};
}

# Current Time
sub ctime {
    my $self = shift;
    my $timeres = $_[0];
    my %ctime = ();
    my $time = ($timeres) ? $timeres : time;
    my $mls = $time - int($time);
    ($ctime{sec},$ctime{min},$ctime{hour},$ctime{day},$ctime{month},$ctime{year},$ctime{wday},$ctime{yday},$ctime{isdst}) = localtime($time);
    $ctime{year} += 1900;
    ++$ctime{month};
    for ('day','month','hour','min','sec') {
      $ctime{$_} = "0".$ctime{$_} if $ctime{$_} =~ /^.$/;
      }
    $ctime{mydat} = join "-", ($ctime{year}, $ctime{month}, $ctime{day});
    $ctime{dat} = join ".", ($ctime{day}, $ctime{month}, $ctime{year});
    $ctime{time} = join ":", ($ctime{hour}, $ctime{min},$ctime{sec});
    $ctime{datetime} = join " ", $ctime{dat}, $ctime{time};
    $ctime{mydatetime} = join " ", $ctime{mydat}, $ctime{time};
    $ctime{norm} = [$ctime{year}, $ctime{month}, $ctime{day}, $ctime{hour}, $ctime{min},$ctime{sec}, $mls];
    $ctime{sec}  = $time;
#    $ctime{mydatetime}
    return \%ctime;
}

#Добавление трансформатора
#имя, указ.на функцию [yyyy,mm,dd] в дату, указ.на функцию дата в [yyyy,mm,dd]
sub add_tr {
    my $self = shift;
    $self->{trs}->{$_[0]} = { '>' => $_[1], '<' => $_[2] };
}

#Преобразование дат
#из какого, в какой, дата
sub trdate {
    my ($self,$f1,$f2,$date) = @_;
#    eval {
        return $self->{trs}->{$f2}->{'>'}->($self->{trs}->{$f1}->{'<'}->($date));
#    };
#    if($@){
#        print STDERR caller();
#        print STDERR "\n";
#    }
}

#Текущая дата, можно указать формат и смещение в секундах
sub cur_date {
    my ($self,$f2,$d) = @_;
    my $dt = $self->ctime();
    $f2 ||= 'db_time';
    if( $d ){
        my $cur = $dt->{sec};
        $cur += $d;   
        return $self->trdate('sec',$f2,$cur);
    } 
    return $self->trdate('norm', $f2,$self->ctime()->{norm});
}

#Получает дату и возвращает следующий после неё день (это решение выбрано супротип прибавления секунд из-за существования переходов на летнее время)
sub next_date {
    my $self = shift;
    return $self->next_n_date(1, @_);
}

sub next_n_date {
    my $self = shift;
    my $n = shift;
    my ($f1,$f2,$date) = (undef,undef,undef);
#LogDump(\@_);
    ($f1,$f2,$date) = @_                    if @_ == 3;                     #вх_тип, вых_тип, дата
    ($f1,$f2,$date) = ('norm', @_)          if @_ == 2;                     #вых_тип, дата
    ($f1,$f2,$date) = ('norm', @_, $self->cur_date('norm'))  if @_ == 1;    #вых_тип
    ($f1,$f2,$date) = ('norm', 'norm', $self->cur_date('norm')) if @_ == 0; #_ничего_
    my $inpdate = $f1 eq 'norm' ? $date : $self->trdate($f1,'norm', $date);
    $inpdate = [ @{$inpdate}[0 .. 5] ];
    my $newdate = [ Add_Delta_DHMS( @{$inpdate}, $n, 0, 0, 0) ];
    my $outdate = $f2 eq 'norm' ? $newdate : $self->trdate('norm',$f2, $newdate);
    return $outdate;
}

sub add_dhms {
    my ($self, $date, $dhms, $o_format, $i_format) = @_;
    $i_format ||= 'norm';
    $o_format ||= 'norm';

    my $inpdate = $i_format eq 'norm' ? $date : $self->trdate($i_format,'norm', $date);
    my $newdate = [ Add_Delta_DHMS( @{$inpdate}, @$dhms) ];

    my $outdate = $o_format eq 'norm' ? $newdate : $self->trdate('norm',$o_format, $newdate);
    return $outdate;
};

#Получает дату и возвращает дату на 30 дней больше
sub next_month_30 {
    my ($self, $date, $o_format, $i_format) = @_;
    $i_format ||= 'norm';
    $o_format ||= 'norm';

    my $inpdate = $i_format eq 'norm' ? $date : $self->trdate($i_format,'norm', $date);
    my $newdate = [ Add_Delta_DHMS( @{$inpdate}, 30, 0, 0, 0) ];

    my $outdate = $o_format eq 'norm' ? $newdate : $self->trdate('norm',$o_format, $newdate);
    return $outdate;
};

#Получает дату и возвращает дату на год больше
sub next_year {
    my ($self, $date, $o_format, $i_format) = @_;
    $i_format ||= 'norm';
    $o_format ||= 'norm';
    
    $date = $self->trdate($i_format, 'norm', $date);
    $date->[0]++;
    if ($date->[1] ==2 && $date->[2] == 29) {
        $date->[1] = 3;
        $date->[2] = 1;
    };
    return $self->trdate('norm', $o_format, $date);
};

#Получает дату и возвращает дату на месяц больше
sub next_month {
    my ($self, $date, $o_format, $i_format) = @_;
    $i_format ||= 'norm';
    $o_format ||= 'norm';
    
    $date = $self->trdate($i_format, 'norm', $date);
    $date->[1]++;
    
    if ($date->[1] == 13) {
        $date->[1] = 1;
        $date->[0]++;
    };
    
    my $last_day = $self->DaysInMonth($date->[0], $date->[1]);
    $date->[1] = $last_day if $date->[1] > $last_day;

    return $self->trdate('norm', $o_format, $date);
};


#Получает дату и возвращает дату на месяц меньше
sub before_month {
    my ($self, $date, $o_format, $i_format) = @_;
    $date ||= $self->cur_date('norm');
    $i_format ||= 'norm';
    $o_format ||= 'norm';
    
    $date = $self->trdate($i_format, 'norm', $date);
    $date->[1]--;
    
    if ($date->[1] == 0) {
        $date->[1] = 12;
        $date->[0]--;
    };
    
    my $last_day = $self->DaysInMonth($date->[0], $date->[1]);
    $date->[1] = $last_day if $date->[1] > $last_day;

    return $self->trdate('norm', $o_format, $date);
};


#Получает дату и возвращает дату на 30 дней меньше
sub before_month_30 {
    my ($self, $date, $o_format, $i_format) = @_;
    $i_format ||= 'norm';
    $o_format ||= 'norm';
    
    $date = $self->trdate($i_format, 'norm', $date);
    my $newdate = [ Add_Delta_DHMS( @$date, -30, 0, 0, 0) ];
    return $self->trdate('norm', $o_format, $newdate);
};

#Получает дату и возвращает предыдущий день (это решение выбрано супротип прибавления секунд из-за существования переходов на летнее время)
sub before_date {
    my $self = shift;
    my ($f1,$f2,$date) = (undef,undef,undef);
    ($f1,$f2,$date) = @_                    if @_ == 3;                     #вх_тип, вых_тип, дата
    ($f1,$f2,$date) = ('norm', @_)          if @_ == 2;                     #вых_тип, дата
    ($f1,$f2,$date) = ('norm', @_, $self->cur_date('norm'))  if @_ == 1;    #вых_тип
    ($f1,$f2,$date) = ('norm', 'norm', $self->cur_date('norm')) if @_ == 0; #_ничего_
    my $inpdate = $f1 eq 'norm' ? $date : $self->trdate($f1,'norm', $date);
    my $newdate = [ Add_Delta_DHMS( @{$inpdate}[0..5], -1, 0, 0, 0) ];
    my $outdate = $f2 eq 'norm' ? $newdate : $self->trdate('norm',$f2, $newdate);
    return $outdate;
}

#Массив день, месяц, год в строку недели
sub date2week {
    my $d = $_[0];

    my $timet = timelocal(0, 0, 0, $d->[0], ($d->[1] - 1), ($d->[2] - 1900));
    my $weekd = strftime("%u", localtime($timet)) - 1;
    
    my $from = strftime("%d.%m.%y", localtime($timet - ($weekd*86400)) ) ;
    my $to   = strftime("%d.%m.%y", localtime($timet + ((6 - $weekd)*86400)) );
    
    my $key = "$from\&nbsp;-\&nbsp;$to";

    return $key;
}

sub name2dates {
    my ($self,$name,$fd,$fd2,$opt) = @_;

    my %cdt = %{$self->ctime};

    my ($filter_date, $filter_date2) = ('',''); #Ограничение на период
    $filter_date  = $self->cur_date('db',-3600*24)    if $name eq 'yesterday';
    $filter_date2 = $filter_date                      if $name eq 'yesterday';
    $filter_date  = $self->cur_date('db')             if $name eq 'today';
    $filter_date2  = $self->cur_date('db')            if $name eq 'today';
    $filter_date  = $self->cur_date('db',-3600*24*6)  if $name eq 'week';
    $filter_date2 = $self->cur_date('db')             if $name eq 'week';
    
    my $c = $self->cur_date('norm');
    if( $name eq 'lastmonth' || $name eq 'pastmonth' ){
        my ($y, $m) = @{$c}[0,1];
        $m -= 1;
        ($m, $y) = (12, $y-1) unless $m;
        $filter_date  = $self->trdate('norm','db',[$y,$m,1]);
        $filter_date2 = $self->trdate('norm','db',[$y,$m,$self->DaysInMonth($y,$m)]);
    }elsif( $name eq 'month' ){
        my ($y, $m) = @{$c}[0,1];
        $filter_date  = $self->trdate('norm','db',[$y,$m,1]);
        $filter_date2 = $self->trdate('norm','db',[$y,$m,$self->DaysInMonth($y,$m)]);
    }elsif ($name eq 'thismonth') {
        my ($y, $m) = @{$c}[0,1];
        $filter_date  = $self->trdate('norm','db',[$y,$m,1]);
        $filter_date2 = $self->cur_date('db'); 
    }elsif( $name eq 'dates' || $name eq 'range' ){
        $filter_date  = $self->trdate('db','db',$fd);  #На всякий случай проверяем формат
        $filter_date2 = $self->trdate('db','db',$fd2); #На всякий случай проверяем формат
    }elsif ($name eq 'year' || $name eq 'thisyear') {
        $filter_date  = "$cdt{'year'}-01-01";
        $filter_date2 = "$cdt{'year'}-12-31";
    } elsif ($name eq 'tendays') {
        my $timet = timelocal(0, 0, 0, $cdt{'day'}, ($cdt{'month'} - 1), ($cdt{'year'} - 1900));
        $filter_date  = strftime("%Y-%m-%d", localtime($timet - 9*86400) );
        $filter_date2 = strftime("%Y-%m-%d", localtime($timet));
    } elsif ($name eq '30days') {
        my $timet = timelocal(0, 0, 0, $cdt{'day'}, ($cdt{'month'} - 1), ($cdt{'year'} - 1900));
        $filter_date  = strftime("%Y-%m-%d", localtime($timet - 29*86400) );
        $filter_date2 = strftime("%Y-%m-%d", localtime($timet));
    } elsif ($name eq 'sevendays' || $name eq 'last7days') {
        my $timet = timelocal(0, 0, 0, $cdt{'day'}, ($cdt{'month'} - 1), ($cdt{'year'} - 1900));
        $filter_date  = strftime("%Y-%m-%d", localtime($timet - 6*86400) );
        $filter_date2 = strftime("%Y-%m-%d", localtime($timet));
    } elsif ($name eq 'thisweek') {
        my $timet = timelocal(0, 0, 0, $cdt{'day'}, ($cdt{'month'} - 1), ($cdt{'year'} - 1900));
        my $weekd = strftime("%u", localtime($timet)) - 1;
        $filter_date  = strftime("%Y-%m-%d", localtime($timet - ($weekd*86400)) ) ;
        $filter_date2 = $self->cur_date('db'); #strftime("%Y-%m-%d", localtime($timet + ((6 - $weekd)*86400)) );
    } elsif ($name eq 'lastweek' || $name eq 'pastweek') {
        my $timet = timelocal(0, 0, 0, $cdt{'day'}, ($cdt{'month'} - 1), ($cdt{'year'} - 1900));
        my $weekd = strftime("%u", localtime($timet)) - 1;
        $filter_date  = strftime("%Y-%m-%d", localtime($timet - (($weekd + 7)*86400)) ) ;
        $filter_date2 = strftime("%Y-%m-%d", localtime($timet + ((6 - $weekd - 7)*86400)) );
    } elsif ($name eq 'pastpastweek') {
        my $timet = timelocal(0, 0, 0, $cdt{'day'}, ($cdt{'month'} - 1), ($cdt{'year'} - 1900));
        my $weekd = strftime("%u", localtime($timet)) - 1;
        $filter_date  = strftime("%Y-%m-%d", localtime($timet - (($weekd + 14)*86400)) ) ;
        $filter_date2 = strftime("%Y-%m-%d", localtime($timet + ((6 - $weekd - 14)*86400)) );
    } elsif ($name eq 'daybeforeyesterday') {
        $filter_date = $filter_date2 = $self->trdate('norm', 'db', [Add_Delta_DHMS(@$c, -2, 0, 0, 0)] );
    } elsif ($name eq 'lasttoday') {
        $filter_date = $filter_date2 = $self->trdate('norm', 'db', [Add_Delta_DHMS(@$c, -7, 0, 0, 0)] );
    } elsif ($name eq 'lastyesterday') {
        $filter_date = $filter_date2 = $self->trdate('norm', 'db', [Add_Delta_DHMS(@$c, -8, 0, 0, 0)] );
    } elsif ($name eq 'past7days') {
        $filter_date  = $self->trdate('norm', 'db', [Add_Delta_DHMS(@$c, -13, 0, 0, 0)] );
        $filter_date2 = $self->trdate('norm', 'db', [Add_Delta_DHMS(@$c, -7, 0, 0, 0)] );
    } elsif ($name eq 'pastpastmonth') {
        $filter_date  = $self->trdate('norm', 'db', [Add_Delta_YMD(@$c[0..2], 0, -2, 1 - $cdt{day})] );
        $filter_date2 = $self->trdate('norm', 'db', [Add_Delta_YMD(@$c[0..2], 0, -1, -$cdt{day})] );
    } elsif ($name eq 'pastyear') {
        my $y = $cdt{year} - 1;
        $filter_date  = "$y-01-01";
        $filter_date2 = "$y-12-31";
    }


    if($opt->{'time'}){
        $filter_date  .= ' 00:00:00' if $filter_date  ne '';
        $filter_date2 .= ' 23:59:59' if $filter_date2 ne '';
    }
    return ($filter_date, $filter_date2);
}

#Превращает две даты в массив дат между ними
sub days2array {
    my ($self,$d1,$d2,$fmt) = @_;
    return [] unless $d1 && $d2;
    
    ($d1, $d2) = map { $self->trdate($fmt, 'db', $_) } $d1, $d2 if $fmt;
    
    ($d1, $d2)=$self->check_dates_order($d1, $d2, 'db'); # Проверяем последовательность, формат и все остальное.
    my $s1 = $self->trdate('db','db',$d1); #защита от неправильного формата
    my $s2 = $self->trdate('db','db',$d2); #защита от неправильного формата
    my $sec1 = $self->trdate('db','sec',$s1); #защита от неправильного формата
    my $sec2 = $self->trdate('db','sec',$s2); #защита от неправильного формата
    
    return [] unless $sec1 && $sec2;
    my $s = $s1;
    my @days = ();
=h
    my $days_sec = 24 * 3600;
    while( $s <= $s2 ){
        push @days, $self->trdate('sec','db',$s);
        $s += $days_sec;
    }
=cut
    while( $s le $s2 ){
        push @days, $self->trdate('db','db',$s);
        $s = $self->next_date('db','db',$s);
    }

    @days = map { $self->trdate('db', $fmt, $_) } @days if $fmt;

    return @days;
}

sub weeks2array {
    my ($self,$d1,$d2) = @_;
    my %weeks=( 
                map { $self->trdate('db', 'db_week', $_) => $_ } $self->days2array($d1, $d2) 
            );
    my @weeks=(sort keys %weeks);
    return @weeks;
}

sub month_name {
    my $self = shift;
    my ($n,$t) = @_;
    return $months->{$n}->{$t};
};

#Возвращает дату - последний день месяца
sub end_month {
    my ($self, $date, $i_format, $o_format) = @_;
   
    $date = $self->trdate($i_format, 'norm', $date) if $i_format; 
    
    $date->[2] = $self->DaysInMonth(@$date);
    $date->[3] = 23;
    $date->[4] = $date->[5] = 59;
    
    $date = $self->trdate('norm', $o_format, $date) if $o_format;
    return $date; 
};

# Проверяет даты на правильность и последовательность.
# Если даты не распознались, выставляет 2005-08-01 или текущую
# Если конец раньше начала, меняет местами
# Запускать так:
# ($start, $end)=$proj->dates->check_dates_order($start, $end, 'db');
# При любых обстоятельстах должна возвращать валидную дату
sub check_dates_order {
    my($self, $start, $end, $format) = @_;
    $format ||= 'norm';
    $start ||= $self->cur_date($format);
    $end ||= $self->cur_date($format);
    # Следующее используется для входящих проверок. Блокировка перегрузки счетчика юникса
    $start = $self->correct_date($start) if($format eq 'db');
    $end = $self->correct_date($end) if($format eq 'db');

    $start=$self->trdate($format, 'db_time', $start);
    $end=$self->trdate($format, 'db_time', $end);
    if($start eq '0000-00-00 00:00:00') {
        $start='2005-08-01 00:00:00';
    }
    if($end eq '0000-00-00 00:00:00') {
        $end=$self->cur_date('db_time');
    }
    if($start gt $end) {
        ($start, $end)=($end, $start)
    }
    $start=$self->trdate('db_time', $format, $start);
    $end=$self->trdate('db_time', $format, $end);
    return ($start, $end);
}
1;

# На вход ест только формат DB
sub correct_date {
    my($self, $dt) = @_;
    # Режем мусор по краям
    $dt=~s/^\D//;
    $dt=~s/\D$//;
    $dt = [ split(/\D/, $dt) ];
    return $self->cur_date('db') if $#{$dt} < 2;
    $dt=[2030, 1, 1] if $dt->[0] > 2030;
    $dt=[2005, 8, 1] if $dt->[0] < 2005;
    $dt->[1]=12 if $dt->[1] > 12;
    $dt->[1]=1 if $dt->[1] < 1;
    $dt->[2]=1 if $dt->[2] < 1;
    $dt->[2]=Date::Calc::Days_in_Month(@{$dt}[0, 1]) if $dt->[2] > Date::Calc::Days_in_Month(@{$dt}[0, 1]) ;
    return sprintf("%04d-%02d-%02d", @{$dt});
}

#sub is_valid_date {
#    return Date::Calc::check_date split(/-/, $_[1]);
#}

sub delta_days {
    my ($self, $d1, $d2, $i_format) = @_;
    $i_format ||= 'norm';
    
    $d1 = $self->trdate($i_format, 'norm', $d1);
    $d2 = $self->trdate($i_format, 'norm', $d2);
    
    return Date::Calc::Delta_Days(@$d1[0..2], @$d2[0..2]); 
}    


sub delta_time {
    my ($self, $d1, $d2, $i_format, $o_format) = @_;
    $i_format ||= 'norm';
    $o_format ||= 'seconds';
    $d1 ||= $self->cur_date($i_format);
    $d2 ||= $self->cur_date($i_format);
#    LogDump([$d1, $d2, $i_format, $o_format]);
    
    $d1 = $self->trdate($i_format, 'sec', $d1);
    $d2 = $self->trdate($i_format, 'sec', $d2);
    my $delta = $d2-$d1;
    if($o_format eq 'seconds') {
        return $delta;
    }
    elsif($o_format eq 'minutes') {
        return int($delta/60);
    }
    elsif($o_format eq 'hours') {
        return int($delta/3600);
    }
    elsif($o_format eq 'days') {
        return int($delta/3600/24);
    }
}


1;    
