package Deploy;
use utf8;
use open ':utf8';
no warnings 'utf8';

use base qw(Exporter);
our @EXPORT_OK = qw(
    run_deploy
    DbSave
);

use Time::HiRes qw(gettimeofday);
use POSIX ":sys_wait_h";
use File::Temp qw(tempdir);
use JSON qw(to_json from_json);
use POSIX qw(ceil);

use Utils::Common;
use Utils::Hosts qw(
    get_hosts
    get_curr_host
    get_host_role
    get_host_info
    get_host_datacenter
    get_from_conductor_api
    get_hosts_domains
);
use Utils::DB;
use Utils::Sys qw(
    do_safely
    get_file_lock
    release_file_lock
    wait_for_file_lock
    uniq
    wait_children
    print_err
);


binmode(STDIN,  ":utf8");
binmode(STDOUT, ":utf8");
binmode(STDERR, ":utf8");


my $DEBUG = $ENV{DEBUG};
my $infuse_host = 'catalogia-media01e';
my $REPO = 'svn+ssh://robot-bm-admin@arcadia.yandex.ru';
my $RedButton_params = $Utils::Common::options->{RedButton_params};

my $ya = "/opt/arcadia/ya"; # TODO: использовать $Utils::Common::options->{ya_path} ?


# деплой бродматча
# на входе: хэш опций, построенный в RedButton.pm
# на выходе: хэш опций для сохранения в базу (DbSave)
sub run_deploy {
    my ($proj, $options) = @_;

    $|++;
    my $options_save;
    $options_save->{DeployParams} = to_json({ map { $_ => $options->{$_} } qw[ macro macro_par command hosts paths revision ] });

    # macro definitions
    my $macro = $options->{macro};
    if ($macro) {
        if ($macro eq 'cmedia-light') {
            $options->{fork} = 1;
            $options->{cmd_array} = [
                {
                    hosts   => 'catalogia-media-front01[ei]',
                    command => 'cmedia-light',
                    sequential => 1,
                },
            ];
        } elsif ($macro eq 'bmfront-up') {
            $options->{hosts}   = [get_hosts( role => 'bmfront' )];
            $options->{command} = 'svnupcrontab';
        } elsif ($macro eq 'bmfront-up-myself') {
            my $curr_host = get_curr_host() // "";
            die "Using macro 'bmfront-up-myself' with not 'bmfront' host!" unless (get_host_role($curr_host) eq "bmfront");
            $options->{hosts}   = [ $curr_host ];
            $options->{command} = 'svnupcrontab';
        } elsif ($macro eq 'bmcdict-front-up') {
            $options->{hosts} = [get_hosts(role => 'bmcdict-front')];
            $options->{command} = 'svnup';
        } elsif ($macro eq 'cmedia-gen-up') {
            $options->{hosts} = [get_hosts(role => 'catalogia-media-gen')];
            $options->{command} = 'svnup';
        } elsif ($macro eq 'catalogia-media-auto01-up') {
            $options->{hosts} = [grep { m/01/ } get_hosts(role => 'catalogia-media-auto')];
            $options->{command} = 'svnup';
        } elsif ($macro eq 'cmedia-short') {
            $options->{hosts}   = 'catalogia-media-front01[ei]';
            $options->{command} = 'cmedia-short';
            $options->{sequential} = 1;
        } elsif ($macro eq 'cmedia-subphraser') {
            $options->{cmd_array} = [
                {
                    hosts   => 'catalogia-media-front01[ei]',
                    command => 'svnup',
                },
                {
                    hosts   => 'catalogia-media-front01[ei]',
                    command => 'spcommands:subphraser_prepare_data',
                },
                {
                    hosts   => 'catalogia-media-front01[ei]',
                    command => 'spcommands:switch_catmedia_host_under_balancer_down,subphraser_stop_server_perfect,subphraser_start_server_perfect,sleep_30,switch_catmedia_host_under_balancer_up',
                    sequential => 1,
                },
            ];
        } elsif ($macro eq 'infuse-cmedia'
            or    $macro eq 'infuse-cmedia-light'
            or    $macro eq 'infuse-cmedia-heavy'
        ) {
            get_file_lock('deploy_infuse-cmedia')
                or die "infuse-cmedia* is already runned!";
            $options->{restart_only_if_need} = $macro eq 'infuse-cmedia-heavy'  ?  0 : 1;
            print_err("infuse. macro: $macro, restart_only_if_need: " . $options->{restart_only_if_need});

            my %exclude_hosts = map { $_ => 1 } ('catalogia01t.yandex.ru', $infuse_host . '.yandex.ru');
            my @hosts_for_svn_up_add = grep { not $exclude_hosts{$_} } sort( uniq(
                    get_hosts(
                        role => [qw[
                                catalogia-media-auto
                                catalogia-media-scripts
                                catalogia-media-tasks
                                catmedia
                        ]],
                    ),
            ));

            my $frontends_for_update = [
                [qw[
                    catalogia-media-front01e
                    catalogia-media-front01i
                ]],
            ]; # двухуровневый список фронтендов, которым нужен рестарт. Списки запускаются параллельно, а серверы, которые внутри одного списка - последовательно.
            my @update_cmd_array;
            foreach my $updatelist (@$frontends_for_update) {
                my $hosts = [grep { $_ ne $infuse_host } @$updatelist]; #хост, на котором был инфьюз, уже рестартовал ранее
                if (@$hosts) {
                    push @update_cmd_array, {
                        hosts   => $hosts,
                        command => 'cmedia-prepare-and-restart-services',
                        sequential  => 1,
                    };
                }
            }
            push @update_cmd_array, {
                hosts   => [@hosts_for_svn_up_add],
                command => 'svnup',
            };

            #обновим сендбокс (этот хост не должен входить в список на svn_up)
            push @update_cmd_array, {
                hosts   => [get_hosts(role => 'bmfront', master => 1 )],
                command => 'publish_catalogia_to_sandbox',
            };

            $options->{cmd_array} = [
                { # infuse - только на одном хосте
                    hosts   => $infuse_host,
                    command => $macro,
                },
                {
                    fork => 1,
                    cmd_array => [
                        { # update_catmedia_db - только на одном хосте
                            hosts   => $infuse_host,
                            command => 'update_catmedia_db',
                        },

                        { # Во время cmedia-restart-services в DoPrepareSubphraser запустить subphraser/prepare_data.pl с флагами "--prepare-comptrie-file --upload-to-sandbox"
                          # "--prepare-comptrie-file --upload-to-sandbox" - только на одном хостe
                            # TODO Нужно ли делать cmedia-restart-services на $infuse_host ?
                            hosts   => $infuse_host,
                            command => 'cmedia-restart-services',
                        },

                        #infuse-cmedia-light выполняется без обновления продакшена
                        $macro ne 'infuse-cmedia-light' ? @update_cmd_array : (),
                    ],
                },
            ];
        } elsif ($macro eq 'publish_catalogia_to_sandbox_ALL') {
            $options->{hosts}   = [get_hosts(role => 'bmfront', master => 1 )];
            $options->{command} = 'publish_catalogia_to_sandbox';
        } elsif ($macro eq 'publish_catalogia_to_sandbox_selected') {
            $options->{hosts}   = [get_hosts(role => 'bmfront', master => 1 )];
            $options->{command} = 'publish_catalogia_to_sandbox';
            my $macro_par = $options->{macro_par} // {};
            my @bootstraps;
            while (my ($key, $val) = each %$macro_par) {
                push @bootstraps, $key if $val;
            }
            die "Select some sb resource!" if !@bootstraps;
            $options->{command_par} = {bootstraps => [sort @bootstraps]};
        } elsif ($macro eq 'update_catmedia_db') {
            $options->{cmd_array} = [
                {
                    hosts   => $infuse_host,
                    command => 'svnup',
                },
                {
                    hosts   => $infuse_host,
                    command => 'update_catmedia_db',
                },
            ];
        } elsif ($macro eq 'qloud') {
            $options->{hosts} = [get_hosts(role => 'bmfront', master => 1 )];
            $options->{command} = 'qloud';
            my @qloud_params;
            my %mpar = %{$options->{macro_par} // {}};
            if ($mpar{bmapi}) {
                push @qloud_params, {
                    resource_type => 'BROADMATCH_MR_CATALOGIA_RAW',
                    application => 'bmapi-ql',
                    environments => ["test", "prod"],
                    components => ['backend'],
                };
            }
            unless (@qloud_params) {
                die "No qloud env has been selected";
            }
            $options->{command_par} = { qloud_par => \@qloud_params };
        } elsif ($macro eq 'bannerland') {
            my %mpar = %{$options->{macro_par} // {}};

            my @regen_bannerland;
            push @regen_bannerland, 'dyn' if $mpar{iron_dyn_regen};
            push @regen_bannerland, 'perf' if $mpar{iron_perf_regen};
            my @regen = (@regen_bannerland);
            if (@regen) {
                push @cmd_array, {
                    command => 'set_svn_revision',
                    command_par => {
                        task_types => \@regen,
                        todo => ['export_offers'],
                    },
                    hosts => [get_hosts(role => 'bmfront', master => 1 )],
                };
            }

            my @hosts;
            if ($mpar{preprod}) {
                push @hosts, get_hosts(role => [ 'bannerland-preprod' ]);
            }
            if ($mpar{iron} or @regen_bannerland) {
                push @hosts, get_hosts(role => [ 'bannerland' ]);
            }
            if ($mpar{YT}) {
                push @hosts, get_hosts(role => [ 'bannerland-yt', 'bannerland-yt-net' ]);
            }
            if (@hosts) {
                push @cmd_array, {
                    hosts => \@hosts,
                    command => 'svnupcrontab',
                };
            }
            if ($mpar{make_pocket}) {
                push @cmd_array, {
                    hosts => [get_hosts(role => 'bmfront', master => 1 )],
                    command => 'publish_make_pocket',
                };
            }
            $options->{cmd_array} = \@cmd_array;
            $options->{fork} = 0 if @regen;
        } else {
            print_err("ERROR: unknown macro:" . $macro);
            die("Unknown macro:" . $macro);
        }
    } else {
        $options->{macro} = "nomacro_".$options->{command}; # Для записи в таблицу лога - для просмотра в интерфейсе. Можно добавить сюда дополнительную информацию.
        # Команды, для которых не нужен номер ревизии svn
        my %commands_without_GetRevision = map { $_ => 1 }  qw[ check_logs qtests_checking crontab_checking svn_diff setcrontab ];
        if ($commands_without_GetRevision{ $options->{command} }) {
            $options->{dont_GetRevision} = 1;
        }
    }

    # макросов больше нет, работаем со списком команд
    $options_save->{macro} = delete($options->{macro});
    $options_save->{macro_par} = delete($options->{macro_par});

    if (!exists($options->{cmd_array})) {
        # в $options задаются либо глобальные опции + cmd_array, либо глобальные опции + опции одного макроса
        # во втором случае нужно отделить одно от другого (и записать в cmd_array)
        my @global_options = qw(dont_GetRevision revision webusername fork make_clear);
        my @cmd_options = qw(command command_par hosts paths sequential);

        my (%cmd, %global);
        $cmd{$_} = delete $options->{$_} for grep { exists $options->{$_} } @cmd_options;
        $global{$_} = delete $options->{$_} for grep { exists $options->{$_} } @global_options;

        die "Unknown options: ".join(',', keys %$options) if keys %$options;

        $options = { %global, cmd_array => [ \%cmd ] };

        # если cmd один, то пишем инфу о нём в базу
        $options_save = { %$options_save, %cmd };
    }

    $options->{username} = getlogin() || $options->{webusername} || die("no user!");
    $options->{paths} ||= '';

    $options->{BeginTime} = $proj->dates->cur_date('db_time');

    if ($options->{dont_GetRevision}) {
        if ($options->{revision}) {
            print_err("WARN: do not use revision! options->{revision}=".$options->{revision});
            delete $options->{revision};
            delete $options->{make_clear};
        }
    } else {
        print_err("options->{revision}: " . ($options->{revision} // ''));
        if ($options->{revision}) {
            $options->{UseCustomRevision} = 1;
        }
        $options->{revision} = GetRevision($proj, revision => $options->{revision}) || do {
            print_err("ERROR: Could not get good svn revision!");
            exit(2);
        };
        # Поле $options->{revision} - непустое
        print_err("revision: " . $options->{revision});
        print_err("make_clear: " . ($options->{make_clear} // ''));
    }

    my @cmd_array_fld = qw(fork cmd_array);
    my %cmd_array_options;
    $cmd_array_options{$_} = delete($options->{$_}) for grep { exists $options->{$_} } @cmd_array_fld;
    do_safely(
        sub { run_cmd_array($proj, $options, \%cmd_array_options) },
        no_die => 1,
    ) or print_err("ERROR: run_cmd_array failed"); # TODO don't return 1

    $options_save = { %$options_save, %$options };
    return $options_save;
}

# На входе:
#   $hosts - [ список хостов ]
#   $count - на сколько наборов хостов делить
# На выходе:
#   @hosts_sets - список наборов хостов (строк, через запятую)
sub split_hosts_to_sets {
    my ($hosts, $sets_count) = @_;

    my @hosts = sort @$hosts;
    my $count_in_set = ceil(@hosts / $sets_count); # TODO
    print ": $count_in_set " . scalar @hosts . "\n";
    my @hosts_sets;
    while (@hosts) {
        my @set;
        push @set, pop @hosts   for 1 .. $count_in_set;
        @set = grep { $_ } @set;
        push @hosts_sets, join(",", @set);

    }
    print_err("hosts_sets: " . to_json(\@hosts_sets));
    return @hosts_sets;
}

# на вход принимается:
# $options -  хэш глобальных настроек, передаётся в run_cmd и run_function
# $cmd_options -  хэш с полями:
#   fork =>  subj
#   cmd_array =>  список хэшей { command => .. }
sub run_cmd_array {
    my ($proj, $options, $cmd_options) = @_;
    print_err("run_cmd_array ... options: ".to_json($options).", cmd_options: ".to_json($cmd_options));

    my $cmd_array = $cmd_options->{cmd_array};
    if (!$cmd_options->{fork} or @$cmd_array == 1) {
        # Если только одна команда, не делаем fork. Тогда изменения $options, сделанные в run_function, будут доступны в следующих cmd_item
        for my $cmd_item (@$cmd_array) {
            if ($cmd_item->{cmd_array}) {
                run_cmd_array($proj, $options, $cmd_item);
            } else {
                run_cmd($proj, $options, $cmd_item);
            }
        }
    } else {
        for my $cmd_item (@$cmd_array) {
            my $pid = fork();
            if ($pid) {
                print_err("child pid: $pid   cmd_item: " . to_json($cmd_item));
            } else {
                if ($cmd_item->{cmd_array}) {
                    run_cmd_array($proj, $options, $cmd_item);
                } else {
                    run_cmd($proj, $options, $cmd_item);
                }
                exit(0);
            }
        }
        wait_children() or print_err("ERROR: wait_children failed");
    }
    print_err("run_cmd_array done!");
}

sub run_cmd {
    my ($proj, $options, $cmd_item) = @_;
    my $start_time = gettimeofday();
    print_err("run_cmd ...   options: " . to_json($options) . ", cmd_item: " . to_json($cmd_item));

    my @hosts = expand_hosts($cmd_item->{hosts});

    print_err("hosts: " . join(" ", @hosts));
    print_err("options: " . to_json($options));
    print_err("cmd_item: " . to_json($cmd_item));
    print_err("sequential: " . ($cmd_item->{sequential} // ''));
    # Если хост только один, можно не делать fork. Тогда изменения $options, сделанные в run_function, будут доступны в следующих cmd_item (используется в infuse)
    if ($cmd_item->{sequential} or @hosts == 1) {
        for my $host (@hosts) {
            run_function($proj, $host, $options, $cmd_item);
        }
    } else {
        print_err("Fork by hosts");
        for my $host (@hosts) {
            my $pid = fork();
            print_err("host: $host  pid: $pid");
            if ($pid == 0) {
                # это потомок
                run_function($proj, $host, $options, $cmd_item);
                exit(0);
            } else {
                print_err("child pid: $pid   host: $host");
            }
        }
        # ждем завершения всех потомков
        sleep(2);
        wait_children() or print_err("ERROR: wait_children failed");
    }
    my $duration = sprintf "%.1f sec", gettimeofday() - $start_time;
    print_err("run_cmd done. Duration: $duration.  options: " . to_json($options) . "cmd_item: " . to_json($cmd_item));
    return 1;
}

sub run_function {
    my ($proj, $host, $opts, $cmd_opts ) = @_;

    my %svnup_prm = map { $_ => $opts->{$_} } qw[ revision make_clear ];
    my $command = $cmd_opts->{command};

    print_err("[host:$host] [$command] started. " . to_json(\%svnup_prm));
    print_err("opts: " . to_json($opts));
    print_err("cmd_opts: " . to_json($cmd_opts));

    if ($DEBUG) {
        print_err("DEBUG mode: run_function does not check ssh at $host");
    } else {
        # TODO Нужна ли эта проверка?
        my $command_result = `ssh -o ConnectTimeout=10 -o StrictHostKeyChecking=no $host echo connection ok`;
        for my $command_result_line (split /[\r\n]+/, $command_result) {
            print_err( join("\t","[host:$host]", $command_result_line));
        }
        if ($command_result !~ /ok/) {
            my $msg = "[host:$host] ERROR:ssh failed. YOU HAVE FORGOTTEN KEYS ON [$host] FOR BMCLIENT! OR ERROR IN AUTHORIZED KEYS. THIS ERROR CAN BE FIXED ONLY USING YOUR HANDS!!! NO AUTOMATION!!! $!";
            print_err($msg);
            die $msg;
        }
    }

    check_is_deploy_allowed($host);

    my $bm_root_path = $RedButton_params->{svnup_prm}{bm_path};

    # хэш по командам
    if ($command eq 'svnup') {
        DoSvnUp( $host, %svnup_prm );
    } elsif ($command eq 'svnup_clear') {
        DoSvnUp( $host, %svnup_prm, make_opts_str => '--clear');
    } elsif ($command eq 'recheckout_arcadia') {
        my $work_dir = "/opt/recheckout_arcadia_" . time;
        my $revision_str = $svnup_prm{revision} ? "-r ".$svnup_prm{revision} : "";
        for my $cmd (
            "bash -c '[[ ! -d $work_dir ]]'",
            "mkdir $work_dir",
            "cd $work_dir && svn cat $revision_str $REPO/arc/trunk/arcadia/ya | python - clone --prefer-svn-from-sandbox --repo $REPO arcadia",
        ) {
            DoRemoteCommand($host, $cmd);
        }
        DoSvnUp($host, arcadia_path => "$work_dir/arcadia/", %svnup_prm, no_test => 1,);

        DoRemoteCommand($host, "echo | crontab -"); # Не `crontab -r`, т.к. фейлится при повторном вызове
        for my $sig ("", "-9") {
            DoRemoteCommand($host, "ps aux | grep '^bmclient ' | grep -v -P ' (sshd: |bash |sh |ps |grep |xargs |DONT_KILL_THIS_PROCESS|kill)' | perl -wlne 'q[ DONT_KILL_THIS_PROCESS ] if 0; print m/([0-9]+)/' | xargs --no-run-if-empty kill $sig");
            DoRemoteCommand($host, "sleep 5")  if $sig eq "";
        }

        DoRemoteCommand($host, "$work_dir/arcadia/rt-research/broadmatching/scripts/utils/arcadia-setup/arcadia-setup.pl create_data_symlinks_in_arcadia --arcadia-dir $work_dir/arcadia --bm-dir /opt/broadmatching");
        DoRemoteCommand($host, "$work_dir/arcadia/rt-research/broadmatching/scripts/tests/proj.pl");

        # Путь к arcadia из $bm_root_path без '/..' . Будет работать, только если $bm_root_path существует. TODO сделать менее костыльно
        my $arcadia_path = (`ssh -o StrictHostKeyChecking=no $host 'cd $bm_root_path/../../ && pwd'` =~ m/(\S+)/)[0] || die "Could not get arcadia path from $bm_root_path";

        DoRemoteCommand($host, "mv $arcadia_path $work_dir/arcadia-old  &&  mv $work_dir/arcadia $arcadia_path");

        DoRemoteCommand($host, "$bm_root_path/scripts/tests/proj.pl");

        DoRemoteCommand($host, "$bm_root_path/scripts/set-crontab.pl --set");

    } elsif ($command eq 'reconfigure_runit') {
        DoRemoteCommand( $host, "$bm_root_path/scripts/reconfigure_runit.pl" );
    } elsif ($command eq 'svnup_revert_static_zmap') {
        #one-shot kostyl after r2861925 etc
        # cd scripts/cpp-source/ && ya tool svn </dev/null up static_zmap --accept p && rm -r static_zmap && ya tool svn </dev/null cleanup && ya tool svn </dev/null revert -R static_zmap && ya tool svn </dev/null up static_zmap && ya make static_zmap
        #DoSvnUp( $host, %svnup_prm );
        my $revision = $svnup_prm{revision};
        # TODO make_clear   ?
        my $revision_str = $revision ? "-r $revision" : "";
        my $cmd = join(' && ',
            "cd $bm_root_path/scripts/cpp-source/",
            "$ya tool svn </dev/null up static_zmap --accept p $revision_str",
            "rm -r static_zmap",
            "$ya tool svn </dev/null cleanup",
            "$ya tool svn </dev/null revert -R static_zmap",
            "$ya tool svn </dev/null up static_zmap $revision_str",
            "$ya make static_zmap",
        );
        DoRemoteCommand( $host, $cmd );
        DoSvnUp( $host, %svnup_prm );
    } elsif ($command eq 'setcrontab') {
        DoSetCrontab( $host, $bm_root_path );
    } elsif ($command eq 'svnupcrontab') {
        DoSvnUp( $host, %svnup_prm );
        DoSetCrontab( $host, $bm_root_path );
    } elsif ($command eq 'kill_solomon_agent') {
        for my $sig ("-TERM", "-KILL") {
            DoRemoteCommand($host, "pgrep -u bmclient -f '/opt/arcadia/solomon/agent/bin/solomon-agent\\s+--config\\s+/opt/arcadia/rt-research/monitoring/solomon/agent/\\w+config\$' | xargs --no-run-if-empty kill $sig");
            DoRemoteCommand($host, "sleep 5")  if $sig eq "-TERM";
        }
    } elsif ($command eq 'svnup_getresources') {
        DoSvnUp( $host, %svnup_prm );
        DoSpecialCmd( $host, $bm_root_path, 'get_resources' );
    } elsif ($command eq 'infuse-cmedia'  or  $command eq 'infuse-cmedia-heavy' or  $command eq 'infuse-cmedia-light') {
        DoSvnUp( $host, %svnup_prm );

        DoUpdateDictsFromGendicts( $host, $bm_root_path );
        DoInfuse( $host, $bm_root_path );

        { # Определяем $opts->{need_restart_subphraser} и $opts->{need_update_catmedia_db}
            #Файл: $Utils::Common::options->{infuse_web_result_file}
            my $remote_file_act = change_bm_path($Utils::Common::options->{infuse_web_result_file}, new_bm_path => $bm_root_path);
            my $act_code = ReadRemoteFile($host, $remote_file_act);
            #В начале файла должно быть записано число. Биты: 1=рестарт сабфрейзера, 2=update_catmedia_db
            if ($DEBUG) {
                $act_code = 3;
                print_err("DEBUG mode: Set act_code = $act_code");
            }
            if ($act_code =~ /^(\d+)/) {
                $act_code = $1;
            } else {
                die "ERROR in infuse_web_result_file: ($act_code)";
            }
            $opts->{need_restart_subphraser} = 1;   # TODO убрать флаг need_restart_subphraser? prepare-data нужен при любом infuse для заливки словарей в sandbox
            print_err("need_restart_subphraser always is 1");
            print_err("act_code: $act_code");
            #$opts->{need_restart_subphraser} = ($act_code & 1) ? 1 : 0;
            $opts->{need_update_catmedia_db} = ($act_code & 2) ? 1 : 0;
            print_err("command: " . $command . " need_restart_subphraser: " . $opts->{need_restart_subphraser} . ", need_update_catmedia_db: " . $opts->{need_update_catmedia_db});
        }

        if (not $opts->{restart_only_if_need}  or  $opts->{need_restart_subphraser}) {
            DoPrepareSubphraser( $host, $bm_root_path, upload_categorization_data_to_sandbox => 1, (map { $_ => $svnup_prm{$_} } qw[ make_clear ]) );
        }

        DoRemoteCommand( $host, "$bm_root_path/scripts/sandbox/upload_categories_suppression_dict.pl >> $bm_root_path/log/upload_categories_suppression_dict.log 2>>$bm_root_path/log/upload_categories_suppression_dict.err");

        DoRemoteCommand( $host, "$bm_root_path/scripts/sandbox/update_cmake_lists.pl >> $bm_root_path/log/update_cmake_lists.log 2>>$bm_root_path/log/update_cmake_lists.err");

        # DoSvnCiDicts - после DoPrepareSubphraser, upload_categories_suppression_dict.pl, и update_cmake_lists.pl т.к. там загружаем словари в sandbox и получаем id этих словарей
        DoSvnCiDicts( $host, $bm_root_path );  # После этого нужно поменять $revision

        # TODO
        $opts->{revision} = GetRevision($proj) || do {   # Сохраняем изменившийся номер ревизии для DbSave и последующих svn up
            print_err("ERROR: Could not get good svn revision!");
            exit(2);    # TODO  die
        };
        print_err("New revision: " . $opts->{revision});

    } elsif ($command eq 'update_catmedia_db') {
        # Только на catalogia-media01e. Можно делать параллельно с restart_subphraser (cmedia-restart-services или cmedia-prepare-and-restart-services)
        print_err("command: $command");
        print_err(join(" ", map {"$_: " . $opts->{$_} // ''} qw[ restart_only_if_need need_update_catmedia_db ]));
        if (not $opts->{restart_only_if_need}  or  $opts->{need_update_catmedia_db}) {
            DoUpdateBmfrontCategs( $host, $bm_root_path );
        }
    } elsif ($command eq 'upload_categories_suppression_dict') {
        # Только на catalogia-media01e. Можно делать параллельно с restart_subphraser (cmedia-restart-services или cmedia-prepare-and-restart-services)
        my $cmd = "$bm_root_path/scripts/sandbox/upload_categories_suppression_dict.pl >> $bm_root_path/log/upload_categories_suppression_dict.log 2>>$bm_root_path/log/upload_categories_suppression_dict.err";
        DoRemoteCommand( $host, $cmd );
    } elsif ($command eq 'cmedia-prepare-and-restart-services'
        or    $command eq 'cmedia-restart-services'
    ) {
        # На catalogia-media01e и catalogia-media-front*
        print_err("command: $command");
        print_err(join(" ", map {"$_: " . $opts->{$_} // ''} qw[ restart_only_if_need need_restart_subphraser ]));
        switch_catmedia_host_under_balancer($host, 0);
        DoSvnUp( $host, %svnup_prm );
        my $pid;
        if (not $opts->{restart_only_if_need}  or  $opts->{need_restart_subphraser}) {
            $pid = fork();
            if ($pid == 0) {
                if ($command eq 'cmedia-prepare-and-restart-services') {
                    DoPrepareSubphraser( $host, $bm_root_path );
                }
                DoRestartSubphraser( $host, $bm_root_path );
                exit(0);
            }
        }
        $pid = fork(); if ($pid == 0) { DoRestartPrevproj( $host, $bm_root_path ); exit(0); }
        unless ($host =~ m/^catalogia-media01e/) { # На catalogia-media01e не нужен FCGI - https://st.yandex-team.ru/CATALOGIA-981
            $pid = fork(); if ($pid == 0) { DoRestartFcgiNginxBeta( $host, $bm_root_path ); exit(0); }
        }

        wait_children() or print_err("ERROR: wait_children failed");
        switch_catmedia_host_under_balancer($host, 1);
    } elsif ($command eq 'cmedia-short') {
        switch_catmedia_host_under_balancer($host, 0);
        DoSvnUp( $host, %svnup_prm );
        DoSetCrontab( $host, $bm_root_path );
        DoRestartPrevproj( $host, $bm_root_path );
        DoRestartFcgiNginxBeta( $host, $bm_root_path );
        switch_catmedia_host_under_balancer($host, 1);
    } elsif ($command eq 'cmedia-light') {
        switch_catmedia_host_under_balancer($host, 0);
        DoSvnUp( $host, %svnup_prm );
        DoSetCrontab( $host, $bm_root_path );
        DoRestartFcgiNginxBeta( $host, $bm_root_path );
        switch_catmedia_host_under_balancer($host, 1);
    } elsif ($command eq 'publish_make_pocket') {
        DoSvnUp($host, %svnup_prm);
        DoPublishBannerlandBinMakePocket($proj);
    } elsif ($command eq 'publish_catalogia_to_sandbox') {
        my %command_par = %{$cmd_opts->{command_par} // {}};
        my $lock_name = "deploy-$command";
        wait_for_file_lock($lock_name);

        # Проверяем, что копия Аркадии "arcadia-bootstrap", либо создаём новую со счекаутенным "rt-research/broadmatching"
        my $work_dir = "/opt";
        my $arcadia_name = "arcadia-bootstrap";
        my $bootstrap_path = "$work_dir/$arcadia_name/rt-research/broadmatching";
        my $cmd = "[ -d $work_dir/$arcadia_name ] || " .
            "(" . join (" && ",
                "cd $work_dir",
                "svn cat $REPO/arc/trunk/arcadia/ya | python - clone --prefer-svn-from-sandbox --repo $REPO $arcadia_name",
                "cd $work_dir/$arcadia_name",
                "./ya make --checkout $bootstrap_path",
                "ln -s /opt/broadmatching/secrets $bootstrap_path/secrets",
            ) . ")";
        DoRemoteCommand($host, $cmd);

        DoSvnUp( $host, %svnup_prm, bm_path => $bootstrap_path, );
        DoPublishCatalogiaToSandbox($host, $bootstrap_path, %command_par);
        release_file_lock($lock_name);
    } elsif ($command eq 'set_svn_revision') {
        my $lock_name = 'deploy-'.$command;
        wait_for_file_lock($lock_name);

        DoSvnUp($host, %svnup_prm, bm_path => $bm_root_path);

        my %command_par = %{$cmd_opts->{command_par}};
        my $svn = get_svn_cmd();
        my $dir = "$bm_root_path/scripts/dyn-smart-banners";
        my @cmd;
        my $todo_list = join(',', @{$command_par{todo}});
        for my $task_type (@{$command_par{task_types}}) {
            push @cmd, "$dir/set_svn_revision.pl --task-type $task_type $todo_list";
        }
        push @cmd, "$svn commit -m 'update actual svn revision SKIP_CHECK' $dir";
        push @cmd, "echo update done";
        my $cmd = join(" && ", @cmd);
        DoRemoteCommand($host, $cmd);
        release_file_lock($lock_name);
    } elsif ($command eq 'qloud') {
        DoSvnUp($host, %svnup_prm);
        my $ql_project = 'bm'; # Always bm for now, no need for parametrization
        # Update every option chosen in RB
        my %command_par = %{$cmd_opts->{command_par} // {}};
        for my $params (@{$command_par{qloud_par}}) {
            # It is possible that app has multiple environments, that we want to update i.e. test & prod
            for my $ql_env (@{$params->{environments}}) {
                DoUpdateQloudEnv(
                    $host, $bm_root_path, $params->{resource_type},
                    $ql_project, $params->{application},
                    $ql_env, $params->{components},
                    $params->{resource_attrs} // {},
                );
            }
        }
    } elsif (grep { $_ eq $command }  qw[ check_logs qtests_checking crontab_checking svn_diff ]) {
        DoSpecialCmd( $host, $bm_root_path, $command, %svnup_prm );
    } elsif ($command =~ m/^spcommands:\s*(.+)/) {
        my $scmds = $1;
        DoSpecialCmd( $host, $bm_root_path, $scmds, %svnup_prm );
    } elsif ($command =~ m/^spcommands_json:\s*(.+)/) {
        my $scmds_json = $1;
        my $scmds_list = from_json($scmds_json);
        DoSpecialCmd( $host, $bm_root_path, $scmds_list, %svnup_prm );
    } else {
        print_err( "ERROR: Unknown command! [host:$host] [$command]");
        # TODO   die ?
    }

    print_err( "[host:$host] [$command] done");
}


# Проверяем, не запрещен ли деплой для этой машины
# Если на машине есть файл /opt/NO-DEPLOY, красная кнопка не будет на ней ничего делать (и упадет с ошибкой, т.к. зафейлится `bash -c ...` в DoRemoteCommand)
sub check_is_deploy_allowed {
    my ($host) = @_;
    print_err("check_is_deploy_allowed($host) done");
    my $no_deploy_file = '/opt/NO-DEPLOY';
    DoRemoteCommand($host, "bash -c '[[ ! -f $no_deploy_file ]]'");
    print_err("check_is_deploy_allowed($host) done");
}

sub DbSave {
    my $proj = shift;
    my $opts = shift;

    my $dbtable = $proj->dbtable($RedButton_params->{log_table}, 'ID', $RedButton_params->{dbh_name});
    my $hosts_str = ref($opts->{hosts}) ? join(',', @{$opts->{hosts}}) : $opts->{hosts};
    my $macro_str = $opts->{macro};
    if (keys %{$opts->{macro_par} // {}}) {
        $macro_str .= '('.to_json($opts->{macro_par}, {canonical=>1}).')';
    }
    my %h = (
        BeginTime => $opts->{BeginTime},
        UpdateTime => $proj->dates->cur_date('db_time'),
        Username => $opts->{username},
        Command => $opts->{command},
        Hosts => $hosts_str,
        Paths => $opts->{paths},
        Macro => $macro_str,
        Revision => $opts->{revision},
        UseCustomRevision => $opts->{UseCustomRevision},
        DeployParams => $opts->{DeployParams},
    );
    $h{$_} //= ''  for keys %h;
    $h{Hosts} =~ s/\.yandex\.ru\b//g;   # TODO  хосты вида HOST01k.da.yandex.ru

    print_err("DbSave: ".to_json(\%h));
    if ($DEBUG) {
        print_err("DEBUG mode: DbSave does not add. " . to_json(\%h));
        return 1;
    }

    my $id = $dbtable->Add(\%h);
    print_err($RedButton_params->{RedButtonLog_ID_label}.":$id");

    print_err("DbSave done");
    return 1;
}

sub DoUpdateBmfrontCategs {
    my ( $host, $bm_path ) = @_;
    DoRemoteCommand( $host, "[ -d $bm_path ] && echo \"update catmedia db...\" && $bm_path/scripts/update_catmedia_db.pl >> $bm_path/log/update_catmedia_db.log 2>> $bm_path/log/update_catmedia_db.err && echo \"update catmedia db done\"");
    return 1;
}

sub DoPublishBannerlandBinMakePocket {
    my ($proj) = @_;
    my $arc = $proj->options->{dirs}{arcadia};
    my $tmpdir = tempdir("build-make-pocket.XXXX", DIR => $proj->options->{dirs}{temp}, CLEANUP => 1);
    $proj->do_sys_cmd("/usr/bin/env python $arc/rt-research/bannerland/bin/make_pocket/build_resource.py $tmpdir");
    my $spec = {
        description => 'make_pocket from red button',
        type => 'BANNERLAND_BIN_MAKE_POCKET',
        attrs => {},  # TODO: add 'released' ?
    };
    $proj->sandbox_client->upload([$tmpdir], $spec);
}

sub DoPublishCatalogiaToSandbox {
    my ($host, $bm_path, %par) = @_;
    my @publish_args = (
        "--waitlock",
        "--prod",
    );
    push @publish_args, "--bootstraps=".join(',', @{$par{bootstraps}}) if $par{bootstraps};

    my $cmd = join(" && ",
        "[ -d $bm_path ]",
        "echo \"update sandbox start...\"",
        "$bm_path/scripts/publish_catalogia_to_sandbox.pl @publish_args",
        "echo \"update sandbox done\"",
    );
    DoRemoteCommand( $host, $cmd );
    return 1;
}

sub DoSvnupAndRestartService {
    my ($host, %prm) = @_;
    my $wait_for_alive_service = $prm{wait_for_alive_service} // '';
    my $stop_command = $prm{stop_command} // '';
    my $start_command = $prm{start_command} // '';
    my $switch_host_under_balancer = $prm{switch_host_under_balancer};
    my $after_switch_on_under_balancer_command = $prm{after_switch_on_under_balancer_command};
    print_err("DoSvnupAndRestartService($host, wait_for_alive_service => $wait_for_alive_service) ...");

    if ($switch_host_under_balancer) {
        print_err("close under balancer ...");
        # TODO other (not iptruler)
        DoRemoteCommand($host, "sudo /usr/sbin/iptruler all down"),
        print_err("sleep 10 ...");
        sleep 10;
    }

    DoSvnUp( $host, (map { $_ => $prm{$_} } qw[ bm_path paths revision make_clear ]) );

    my $bm_path = $prm{bm_path} // $RedButton_params->{svnup_prm}{bm_path};

    if ($stop_command) {
        print_err("stop_command ...");
        DoRemoteCommand( $host, "cd $bm_path && $stop_command" );
        print_err("sleep 10 ...");
        sleep 10;
    }

    if ($start_command) {
        print_err("stop_command ...");
        DoRemoteCommand( $host, "cd $bm_path && $start_command" );
        print_err("sleep 10 ...");
        sleep 10;
    }

    if ($wait_for_alive_service) {
        print_err("wait_for_alive_service ...");
        DoRemoteCommand($host, "$bm_path/scripts/utils/wait-for-service-alive.pl --name $wait_for_alive_service >> $bm_path/log/wait-for-service-alive_$wait_for_alive_service.log 2>> $bm_path/log/wait-for-service-alive_$wait_for_alive_service.err" );
        print_err("sleep 40 ...");
        sleep 40;
    }

    if ($switch_host_under_balancer) {
        print_err("open under balancer ...");
        # TODO other (not iptruler)
        DoRemoteCommand($host, "sudo /usr/sbin/iptruler all up"),
        print_err("sleep 10 ...");
        sleep 10;

        if ($after_switch_on_under_balancer_command) {
            DoRemoteCommand($host, $after_switch_on_under_balancer_command),
        }
    }

    print_err("DoSvnupAndRestartService($host, wait_for_alive_service => $wait_for_alive_service) done");
    return 1;
}

# svn up с номером ревизии, заданным в $prm{revision} (номер ревизии должен быть непустым - для update-svn-revision-file.pl)
# bm_path - путь к arcadia/rt-research/broadmatching
# arcadia_path - путь к arcadia
# paths - либо пустая строка (апать от корня аркадии), либо массив путей от корня arcadia/rt-research/broadmatching/
# TODO задавать пути от arcadia ??
# target-ы для make определяются по конфигу хоста
sub DoSvnUp {
    my ( $host, %prm ) = @_;
    my $revision = $prm{revision} || die 'Void revision!';
    my %default_prm = %{ $RedButton_params->{svnup_prm} // {} };
    my $bm_path = $prm{bm_path} // $default_prm{bm_path};
    my $arcadia_path = $prm{arcadia_path} // "$bm_path/../..";
    my $paths = $prm{paths} // $default_prm{paths};
    my $make_opts_str = $prm{make_opts_str} // '';
    my $make_clear_str = $prm{make_clear} ? '--clear' : '';

    my @default_make_targets = @{ $default_prm{make_targets} };

    my $ya_path = "$arcadia_path/ya";

    my @make_targets;
    my $info = get_host_info($host) // {};
    if ($info->{make_targets}) {
        @make_targets = @{$info->{make_targets}};
    } elsif ($info->{make_targets_add}) {
        @make_targets = (@default_make_targets, @{$info->{make_targets_add}});
    } else {
        @make_targets = @default_make_targets;
    }

    my $svn = get_svn_cmd(ya_path => $ya_path);
    my $revision_str = "-r $revision";
    my $paths_str = $paths ? join(' ',@$paths) : "";

    # '--accept', чтобы в случае конфликтов svn не оставлял в файлах строки типа '<<<<<<< .mine'
    if ($paths_str) {
        DoRemoteCommand( $host, join(" && ",
                "[ -d $bm_path ]",
                "echo \"svnup...\"",
                "cd $bm_path",
                "$svn revert -R $paths_str",
                "$svn up --accept theirs-full $revision_str $paths_str",
                "echo \"svnup done\"",
        ));

        # Если $paths - выборочно, то не делаем ya make
    } else {
        DoRemoteCommand( $host, join(" && ",
                "[ -d $arcadia_path ]",
                "echo \"svnup...\"",
                "cd $arcadia_path",
                "$svn revert -R ./",
                "$svn up --accept theirs-full $revision_str",
                "echo \"svnup done\"",
        ));

        DoRemoteCommand( $host, join(" && ",
                "echo ya make ...",
                "cd $arcadia_path",
                "$ya_path make -r --checkout $make_clear_str $make_opts_str @make_targets",
                "echo ya make done",
        ));

        DoRemoteCommand( $host, "$bm_path/scripts/utils/update-svn-revision-file.pl $revision" );

        # Если сделали svn up, но что-то пошло не так (например, https://st.yandex-team.ru/CATALOGIA-545), хотим узнать об этом сразу от красной кнопки
        # Проверка ревизии до выкладки остается необходима, но ее может быть недостаточно, например, из-за различий между пакетами (https://st.yandex-team.ru/CATALOGIA-545)
        # Если scripts/tests/proj.pl зафейлился, падаем. Возможно, стоит сделать флаг "не падать, если после svn up тест зафейлился" - для задач типа svnup_revert_static_zmap
        DoRemoteCommand( $host, join(" && ",
                "echo Tests after update ...",
                "$bm_path/scripts/tests/proj.pl",
                "echo Tests after update done",
        )) unless $prm{no_test};
    }

    return 1;
}

# ya make --checkout
# $arcadia_path - путь к arcadia
# $paths - массив путей от корня arcadia/
sub DoYaMake {
    my ( $host, $arcadia_path, $paths, %prm ) = @_;
    my $make_clear_str = $prm{make_clear} ? '--clear' : '';

    my $paths_str = $paths ? join(' ',@$paths) : "";
    unless ($paths_str =~ m/[^\s]/) {
        die "Void paths in DoYamake!"
    }

    DoRemoteCommand( $host, join(" && ",
            "cd $arcadia_path",
            "YA_USER=robot-bm-admin $ya make -r --checkout $make_clear_str $paths_str",
    ));

    return 1;
}

sub DoSvnCiDicts {
    my ($host, $bm_path) = @_;

    my $paths_str = "dicts bin-dicts/categorization";
    my $svn = get_svn_cmd();
    my $cmd = join(" && ",
        "[ -d $bm_path ]",
        "echo \"svn commit dicts...\"",
        "cd $bm_path",
        "$svn commit -m\"CATALOGIA-1268 auto infuse commit [mergeto:bmgen] SKIP_CHECK\" $paths_str",
        "echo \"svn commit done\"",
    );
    DoRemoteCommand( $host, $cmd );

    return 1;
}

sub DoUpdateQloudEnv {
    my ($host,
        $bm_path,
        $resource_type,
        $ql_project,
        $ql_application,
        $ql_environment,
        $ql_components,
        $resource_attrs,
    ) = @_;
    my @resource_opts;
    while (my ($k, $v) = each %$resource_attrs) {
        push @resource_opts, "--resource-attr $k=$v";
    }
    my $cmd = join(" && ",
        "[ -d $bm_path ]",
        "echo 'updating qloud env'",
        join(" ",
            "$bm_path/scripts/update_qloud_env.pl",
            "--resource-type $resource_type",
            "--project $ql_project",
            "--application $ql_application",
            "--environment $ql_environment",
            join(" ", map {"--component $_"} @$ql_components),
            join(" ", @resource_opts),
        ),
        "echo 'Updated qloud env'",
    );
    DoRemoteCommand($host, $cmd);
    return 1;
}

sub DoSetCrontab {
    my ( $host, $bm_path ) = @_;
    DoRemoteCommand( $host, "[ -d $bm_path ] && echo \"set crontab...\" && $bm_path/scripts/set-crontab.pl --set && echo \"set crontab done\"");
    return 1;
}

sub DoGetResources {
    my ( $host, $bm_path ) = @_;
    DoRemoteCommand( $host, "[ -d $bm_path ] && echo \"get resources...\" && $bm_path/scripts/get-resources.pl > $bm_path/log/deploy.log 2>&1 && cat $bm_path/log/deploy.log && echo \"get resources done\"");
    return 1;
}

sub DoPutResources {
    my ( $host, $bm_path ) = @_;
    DoRemoteCommand( $host, "[ -d $bm_path ] && echo \"put resources...\" && $bm_path/scripts/put-resources.pl > $bm_path/log/deploy.log 2>&1 && cat $bm_path/log/deploy.log && echo \"put resources done\"");
    return 1;
}

sub DoPrepareSubphraser {
    my ( $host, $bm_path, %prm ) = @_;

    my $prepare_data_prm_str = "";
    if ($prm{upload_categorization_data_to_sandbox}) {
        my $arcadia_path = "$bm_path/../../";
        DoYaMake($host, $arcadia_path, ['ads/quality/bm_subphraser/bin/converter'], (map { $_ => $prm{$_} } qw[ make_clear ]));
        $prepare_data_prm_str = "--prepare-comptrie-file --upload-to-sandbox";
    }

    my $cmd = join(" && ",
            "[ -d $bm_path ]",
            "echo \"subphraser prepare ...\"",
            "$bm_path/scripts/subphraser/prepare_data.pl $prepare_data_prm_str >> $bm_path/log/subphraser-prepare-data.log 2>> $bm_path/log/subphraser-prepare-data.err",
            "echo \"subphraser prepare done\"",
    );
    DoRemoteCommand( $host, $cmd );
    return 1;
}

sub DoRestartSubphraser {
    my ( $host, $bm_path, %prm ) = @_;

    # Предполагаем, что subphraser запущен на том же хосте (это могло быть переопределено настройкой subphraser_host в Hosts, при этом DoRestartSubphraser будет работать некорректно) TODO
    my $cmd = join(" && ",
            "[ -d $bm_path ]",
            "echo \"subphraser stop...\"",
            "$bm_path/scripts/subphraser/stop_server.pl >> $bm_path/log/subphraser-stop-server.log 2>> $bm_path/log/subphraser-stop-server.err",
            "echo \"sleep ...\"",
            "sleep 30",
            "echo \"wait for alive...\"",
            "$bm_path/scripts/utils/wait-for-service-alive.pl --name subphraser >> $bm_path/log/wait-for-service-alive_subphraser.log 2>> $bm_path/log/wait-for-service-alive_subphraser.err",
            "echo \"subphraser restart done\"",
    );
    DoRemoteCommand( $host, $cmd );
    return 1;
}

sub DoUpdateDictsFromGendicts {
    my ( $host, $bm_path ) = @_;

    # Привезти ресурсы, нужные для update-caddphr-dicts-from-gen-dicts.pl
    # TODO не ждать --waitlock
    # Отключаем, чтобы в красной кнопке не ждать, пока приедут все ресурсы. TODO
    #DoRemoteCommand( $host, "$bm_path/scripts/get-resources.pl --list=models,mediacontent --waitlock >> $bm_path/log/get-resources.log 2>> $bm_path/log/get-resources.err");

    my $cmd = join(" && ",
        "set -o pipefail",  # Для использования " | tee -a "
        "[ -d $bm_path ]",
        "echo \"update-caddphr-dicts-from-gen-dicts ...\"",
        "$bm_path/scripts/update-caddphr-dicts-from-gen-dicts.pl 2>&1 | tee -a $bm_path/log/update-caddphr-dicts-from-gen-dicts.log 2>&1",
        "echo \"update-caddphr-dicts-from-gen-dicts done\"",
    );
    DoRemoteCommand( $host, $cmd);
    return 1;
}

sub DoInfuse {
    my ( $host, $bm_path ) = @_;

    my $categs_diff_file = change_bm_path($Utils::Common::options->{categs_diff_file}, new_bm_path => $bm_path);
    my $cmd = join(" && ",
        "set -o pipefail",  # Для использования " | tee -a "
        "[ -d $bm_path ]",
        "echo \"infuse_web_categs...\"",
        "$bm_path/scripts/utils/infuse_web_categs.pl 2>&1 | tee -a $bm_path/log/infuse_categs.log 2>&1",
        "echo \"enumerate_categs...\"",
        "$bm_path/scripts/enumerate_categs.pl 2>&1 | tee -a $bm_path/log/enumerate_categs.log 2>&1",
        "echo \"update_categories_tree_md5...\"",
        "$bm_path/scripts/update_categories_tree_md5.pl $categs_diff_file 2>&1 | tee -a $bm_path/log/update_categories_tree_md5.log 2>&1",
        "echo \"update_categs_tree_from_infuse...\"",
        "$bm_path/scripts/update_categs_tree_from_infuse.pl 2>&1 | tee -a $bm_path/log/update_categs_tree_from_infuse.log 2>&1",

        "echo \"infuse done\"",
    );
    DoRemoteCommand( $host, $cmd);
    return 1;
}

sub DoRestartFcgiNginxBeta {
    my ( $host, $bm_path ) = @_;
    my $cmd = join(" && ",
        "[ -d $bm_path ]",
        "echo \"restart fcgi nginx\"",
        "$bm_path/scripts/broadmatching-server/fcgi-nginx-server-beta restart >> $bm_path/log/fcgi-server-beta.log 2>> $bm_path/log/fcgi-server-beta.log",
        "echo \"sleep ...\"",
        "sleep 10",
        "echo \"wait for alive...\"",
        "$bm_path/scripts/utils/wait-for-service-alive.pl --name fcgi >> $bm_path/log/wait-for-service-alive_fcgi.log 2>> $bm_path/log/wait-for-service-alive_fcgi.err",
        "echo \"restart fcgi nginx done\"",
    );

    DoRemoteCommand( $host, $cmd );
    return 1;
}

sub DoRestartPrevproj {
    my ( $host, $bm_path ) = @_;
    my $cmd = join(" && ",
        "[ -d $bm_path ]",
        "echo \"restart prevproj\"",
        "$bm_path/scripts/prefprojsrv/prefprojsrv.pl restart >> $bm_path/log/prefprojsrv-restart-server.log 2>> $bm_path/log/prefprojsrv-restart-server.err",
        "echo \"sleep ...\"",
        "sleep 30",
        "echo \"wait for alive...\"",
        "$bm_path/scripts/utils/wait-for-service-alive.pl --name prefprojsrv >> $bm_path/log/wait-for-service-alive_prefprojsrv.log 2>> $bm_path/log/wait-for-service-alive_prefprojsrv.err",
        "echo \"restart prevproj done\"",
    );
    DoRemoteCommand( $host, $cmd);
    return 1;
}

# Выполняет одну или несколько из специальных команд.
# На входе $spcmd:
#       строка из одной или нескольких команд через запятую.
#   или ссылка на массив хэшей { spcmd => ..., prm => ... }
# (описание команд - в хэше %spcmd2cmdarr)
sub DoSpecialCmd {
    my ( $host, $bm_path, $spcmd_in ) = @_;
    my $start_time = gettimeofday();
    my @spcmds_list = ref($spcmd_in) eq 'ARRAY'  ?  @{$spcmd_in}  :  map {{ spcmd => $_ }} split /,/, $spcmd_in;
    print_err("DoSpecialCmd $host:$bm_path " . to_json(\@spcmds_list) . " ...");

    my $svn = get_svn_cmd();

    my $spcmd2cmdarr = sub {
        my ($spcmd, $prm) = @_;
        $prm //= {};

        my %spcmd2cmdarr = (
            subphraser_prepare_data => ["$bm_path/scripts/subphraser/prepare_data.pl >> $bm_path/log/subphraser-prepare-data.log 2>> $bm_path/log/subphraser-prepare-data.err"],
            subphraser_stop  => ["$bm_path/scripts/subphraser/stop_server.pl  >> $bm_path/log/subphraser-stop-server.log  2>> $bm_path/log/subphraser-stop-server.err"],
            subphraser_start => [
                "bash -c '$bm_path/scripts/subphraser/start_server.pl >> $bm_path/log/subphraser-start-server.log 2>> $bm_path/log/subphraser-start-server.err & ' ",
                "echo \"wait-for-service-alive subphraser ...\" && $bm_path/scripts/utils/wait-for-service-alive.pl --name subphraser >> $bm_path/log/wait-for-service-alive_subphraser.log 2>> $bm_path/log/wait-for-service-alive_subphraser.err",
            ],
            subphraser_stop_server_perfect  => ["$bm_path/scripts/subphraser/stop_server.pl  perfect_subphrases_client >> $bm_path/log/subphraser-stop-server-perfect.log  2>> $bm_path/log/subphraser-stop-server-perfect.err"],
            subphraser_start_server_perfect => ["$bm_path/scripts/subphraser/start_server.pl perfect_subphrases_client >> $bm_path/log/subphraser-start-server-perfect.log 2>> $bm_path/log/subphraser-start-server-perfect.err"],
            switch_host_under_balancer_down => ["sudo /usr/sbin/iptruler all down"],
            switch_host_under_balancer_up => ["sudo /usr/sbin/iptruler all up"],

            switch_catmedia_host_under_balancer_down => sub { my ($host, $bm_path) = @_;   switch_catmedia_host_under_balancer($host, 0); },
            switch_catmedia_host_under_balancer_up   => sub { my ($host, $bm_path) = @_;   switch_catmedia_host_under_balancer($host, 1); },

            sleep => ["sleep " . ((ref($prm) eq 'HASH' ? $prm : {})->{time} // 1)],
            sleep_3 => ["sleep 3"],
            sleep_30 => ["sleep 30"],

            qtests_checking => ["$bm_path/scripts/run-cron.pl qtests_checking --waitlock"],
            check_logs => ["$bm_path/scripts/monitors/check-logs.pl --print"],
            crontab_checking => ["$bm_path/scripts/set-crontab.pl --check"],
            svn_diff => ["cd $bm_path; $svn diff"],
            get_resources => ["$bm_path/scripts/get-resources.pl --waitlock >> $bm_path/log/get-resources.log 2>> $bm_path/log/get-resources.err"],
            restart_fcgi_nginx => [
                "$bm_path/scripts/broadmatching-server/fcgi-nginx-server-beta restart >> $bm_path/log/fcgi-server-beta.log 2>> $bm_path/log/fcgi-server-beta.log",
                "echo \"sleep ...\"",
                "sleep 30",
                "echo \"wait for alive...\"",
                "$bm_path/scripts/utils/wait-for-service-alive.pl --name fcgi >> $bm_path/log/wait-for-service-alive_fcgi.log 2>> $bm_path/log/wait-for-service-alive_fcgi.err",
            ],

            wait_for_alive_subphraser => [
                "$bm_path/scripts/utils/wait-for-service-alive.pl --name subphraser >> $bm_path/log/wait-for-service-alive_subphraser.log 2>> $bm_path/log/wait-for-service-alive_subphraser.err",
            ],
        );
        unless ($host =~ m/catalogia-media-front/) {
            $spcmd2cmdarr{$_} = [] for qw[switch_catmedia_host_under_balancer_up switch_catmedia_host_under_balancer_down];  # TODO
        }

        my $cmds = $spcmd2cmdarr{$spcmd};
        return $cmds;
    };

    # Для этих команд не нужен $bm_path - не нужно проверять [ -d $bm_path ]
    my %dont_check_bmpath_spcmds = ();

    for (@spcmds_list) {
        my ($spcmd, $prm) = @{$_}{qw[ spcmd prm ]};
        #$prm //= {};
        print_err("[host:$host] spcmd: $spcmd (" . (ref($prm) eq 'HASH' ? to_json($prm) : ($prm // '')) . ") ...");
        my $start_time_part = gettimeofday();
        my $cmds = $spcmd2cmdarr->( $spcmd, $prm )
            // die "Bad SpecialCmd: $spcmd ($host:$bm_path)";

        if (ref($cmds) eq 'ARRAY') { # Массив команд для выполнения на удаленном хосте
            my $remote_cmd = join(" && ",
                ($dont_check_bmpath_spcmds{$spcmd} ? () : "[ -d $bm_path ]"),
                (   qq[echo "$spcmd ..."],
                    @$cmds,
                    qq[echo "$spcmd done"],
                ),
            );
            DoRemoteCommand( $host, $remote_cmd );
        } elsif (ref($cmds) eq 'CODE') {
            $cmds->($host, $bm_path);
        } else {
            die "Bad SpecialCmd: $spcmd ($host:$bm_path): $cmds";
        }

        my $duration = sprintf "%.1f sec", gettimeofday() - $start_time_part;
        print_err("[host:$host] spcmd: $spcmd (" . (ref($prm) eq 'HASH' ? to_json($prm) : ($prm // '')) . ") done. Duration: $duration");
    }

    #my $remote_cmd = join(" && ",
    #    "[ -d $bm_path ]",
    #    (map {
    #        my $spcmd = $_;
    #        my $cmds = $spcmd2cmdarr{$spcmd}
    #            or die "Bad SpecialCmd: $spcmd ($host:$bm_path)";
    #        (   qq[echo "$spcmd ..."],
    #            @$cmds,
    #            qq[echo "$spcmd done"],
    #        ),
    #    } @$spcmds),
    #);
    #DoRemoteCommand( $host, $remote_cmd );

    my $duration = sprintf "%.1f sec", gettimeofday() - $start_time;
    print_err("DoSpecialCmd $host:$bm_path " . to_json(\@spcmds_list) . " done. Duration: $duration");
    return 1;
}

# make remote command on remote server
sub DoRemoteCommand {
    my ( $host, $command ) = @_;

    my $time_begin = time;

    $command =~ s/\"/\\\"/g;
    print_err("DoRemoteCommand ...  ( $host, { $command })");

    if ($DEBUG) {
        print_err("DEBUG mode: DoRemoteCommand does nothing. ( $host, { $command } )");
        return 1;
    }

    open my $fh, "( ssh -o StrictHostKeyChecking=no $host \"$command\" || ( echo ERROR occured at ssh $host; false ) ) 2>&1 | "
        or die "ERROR in DoRemoteCommand($host, { $command })";
    while (<$fh>) {
        chomp;
        print_err(join("\t","[host:$host]", $_));
    }
    close $fh
        or die "ERROR in DoRemoteCommand($host, { $command }), cannot close!";

    my $time_end = time;
    my $duration = $time_end - $time_begin;
    print_err("DoRemoteCommand done, $duration s ( $host, { $command })");
    return 1;
}

sub ReadRemoteFile {
    my ( $host, $file ) = @_;
    if ($DEBUG) {
        print_err("DEBUG mode: WARN: ReadRemoteFile( $host, $file ) returns empty result!");
        return "";
    }
    print_err("ReadRemoteFile ...  ( $host, $file )");
    my $result = `ssh  -o StrictHostKeyChecking=no  $host 'cat "$file"'`;
    print_err("ReadRemoteFile done ( $host, $file ), RESULT($result)ENDRESULT");
    return $result;
}

# host0[12][fg] -> host01f, host02f, host01g, host02g
sub expand {
    my $item = shift;
    my @variants = ('');
    my @parts = ($item =~ m/([^\]\[]+|\[[^\[\]]+\])/g);
    for my $part (@parts) {
        if ($part !~ /^\[/) {
            $_ .= $part for @variants;
        } else {
            $part =~ s/^\[//;
            $part =~ s/\]$//;
            my @c = split //, $part;
            @variants = map { my $old = $_; map { $old . $_ } @c } @variants;
        }
    }
    return @variants;
}

sub expand_hosts {
    my ($input_hosts) = @_;
    $input_hosts = [$input_hosts] if !ref($input_hosts);  # допускается задавать строку, а не список
    my @hosts;
    my @hosts_bad;

    for my $hosts_str (@$input_hosts) {
        for my $def ( map { s/(^\s+|\s+$)//g; $_ || () } split( /,/, $hosts_str) ) {
            # Роль хоста или выражение для expand
            my @h = get_hosts( role => $def );

            # Если есть хосты по роли, добавляем их (предварительно отсеяв alias) и идём дальше
            if (@h) {
                for my $host_var (@h) {
                    push @hosts, $host_var unless get_host_info($host_var)->{is_alias};
                }
                next;
            }

            # Если это не роль, пробуем расширить через "expand". Далее проверяем, есть ли такие хосты
            @h = expand($def);
            push @h, $def unless @h;

            for my $host_var (@h) {
                my $in_hosts_pm = 0;
                my $host_domains = get_hosts_domains();
                for my $host_domain (@$host_domains, "") {
                    my $host_postfix = $host_domain ? "." . $host_domain : $host_domain;
                    my $host_info = get_host_info($host_var . $host_postfix);
                    if ($host_info->{host}) {
                        if ($host_info->{is_alias}) {
                            print_err ("WARN: No deploy for '$host_info->{host}', because this is alias");
                        } else {
                            push @hosts, $host_info->{host};
                        }
                        $in_hosts_pm = 1;
                        last;
                    }
                }
                push @hosts_bad, $host_var unless $in_hosts_pm;
            }
        }
    }

    # Если есть хосты, не найденные в Hosts.pm, падаем
    if (@hosts_bad) {
        print_err("ERROR: Unknown hosts (not found in Hosts.pm): " . join(", ", @hosts_bad));
        exit(2);
    }

    @hosts = uniq @hosts;
    return @hosts;
}

# Получить номер последней ревизии svn, если для нее проходят тесты. Если тесты не проходят или произошла ошибка, возвращает undef или 0
# Если передан номер ревизии, то проверяет эту ревизию и возвращает этот номер, если успешно; иначе - undef или 0
sub GetRevision {
    my ($proj, %prm) = @_;
    my $revision = $prm{revision};
    my $mode = $revision ? 'custom' : 'last';
    $proj->log("GetRevision $mode" . (defined $revision ? "revision: $revision" : ""));

    if ($DEBUG) {
        if ($revision) {
            $proj->log("DEBUG mode: GetRevision do nothing");
            return $revision;
        } else {
            $revision = 66666666;
            $proj->log("DEBUG mode: WARN: GetRevision returns $revision");
            return $revision;
        }
    }

    my $dir_bm = $Utils::Common::options->{dirs}{root};

    # Тесты для заданной ревизии - в отдельной директории, чтобы не замедлять выкладку с последней ревизией
    my $dir_test = "$dir_bm/work/deploy/GetRevision-$mode";

    my $yamake_threadsnumber = $mode eq 'last' ? 16 : 4;

    my $lock_name = "deploy_GetRevision_$mode";
    if (1) {
        $proj->log("wait_for_file_lock($lock_name) ...");
        wait_for_file_lock($lock_name);
    } else {
        # TODO remove? Т.к. ya make не будет работать в двух директориях одновременно?
        get_file_lock($lock_name) or do {
            # Лок занят, если там работает другой процесс deploy.pl. Делаем временную директорию.
            $proj->log("$dir_test is busy. Get a temp dir for svn.");
            my $dir_temp = "$dir_bm/temp/deploy";   # TODO
            $proj->do_sys_cmd("mkdir -p $dir_temp");
            $dir_test = tempdir("deploy-GetRevision-$mode.XXXX", DIR => $dir_temp, CLEANUP => 1);
            $yamake_threadsnumber = 4;
        };
    }
    $proj->log("dir_test: $dir_test");

    my $revision_str = $revision ? "-r $revision" : "";

    $proj->do_sys_cmd("mkdir -p $dir_test");
    $proj->do_sys_cmd("cd $dir_test; [ -d arcadia ] || svn cat $REPO/arc/trunk/arcadia/ya | python - clone --prefer-svn-from-sandbox --repo $REPO arcadia");
    # TODO use DoSvnUp and update-svn-revision-file.pl
    # TODO make_clear
    # '--accept', чтобы в случае конфликтов svn не оставлял в файлах строки типа '<<<<<<< .mine'
    my @make_targets = @{$RedButton_params->{svnup_prm}{make_targets}};
    $proj->do_sys_cmd("cd $dir_test/arcadia && ./ya tool svn revert -R ./ && ./ya tool svn </dev/null up --accept theirs-conflict $revision_str && ./ya make -r --checkout @make_targets -j$yamake_threadsnumber");
    $proj->log("checkout done");

    my $dir_test_bm = "$dir_test/arcadia/rt-research/broadmatching";
    my $dir_log = "$dir_bm/log";
    do_safely(sub{ CheckTests($proj, "$dir_test_bm/", $dir_log) }, no_die => 1, ) || do {
        $proj->log("Tests failed!");
        release_file_lock($lock_name);
        return;
    };
    $proj->log("Tests done");

    # Тесты прошли успешно, можно вернуть номер ревизии
    unless ($revision) {
        my $svn = get_svn_cmd();
        my $last_revision_command = "$svn info $dir_test_bm | grep 'Revision:' | sed 's/Revision: //' ";
        my $last_revision = int(`$last_revision_command`);
        $revision = $last_revision;
    }
    $proj->log("$mode revision is OK: $revision");
    release_file_lock($lock_name);
    return $revision;
}

sub CheckTests {
    my ($proj, $dir_root, $dir_log) = @_;

    my @cmds = (
        "$dir_root/scripts/tests/proj.pl --bm",
        # TODO  "$dir_root/scripts/run-cron.pl qtests_checking >> $dir_root/log/qtests_checking.log 2>> $dir_root/log/qtests_checking.err",
    );

    for my $cmd (@cmds) {
        $proj->log("Run test: ( $cmd )");
        my $res = `$cmd 2>&1`;
        my $tests_exit_code = $?;
        $proj->log("Test done ( $cmd ), res:( $res )");

        my $res_str = join("  ", split /\n/, $res);
        if ($tests_exit_code) {
            $proj->log("ERROR: Test failed. cmd: { $cmd } Exit code: $tests_exit_code Output: $res_str");
            return;
        } else {
            $proj->log("Test done. cmd: { $cmd } Exit code: $tests_exit_code  Output: $res_str");
        }
    }

    return 1;
}

# В пути к файлу заменить $Utils::Common::options->{dirs}{root} на new_bm_path
sub change_bm_path {
    my ($file, %prm) = @_;
    my $new_bm_path = $prm{new_bm_path};

    my $local_bm_path = $Utils::Common::options->{dirs}{root};
    do { $_ = "$_/";  s!/+!/!g; }  for ($local_bm_path, $new_bm_path);
    $file =~ s/^$local_bm_path/$new_bm_path/;
    return $file;
}

# Для хостов catmedia.yandex.ru, которые используются через балансер (catalogia-media-front01[ei])
# state - 1/0 (включить/выключить хост)
sub switch_catmedia_host_under_balancer {
    my ($host, $state) = @_;
    my $mode = $state ? 'up' : 'down';

    return 1   unless $host =~ m/catalogia-media-front/;  # TODO

    my $lock_name = catmedia_front_lock_name();
    if ($mode eq 'down') {
        print_err("wait_for_file_lock($lock_name) ...");
        wait_for_file_lock($lock_name);
        print_err("wait_for_file_lock($lock_name) done");
    }
    DoRemoteCommand($host, "sudo /usr/sbin/iptruler all $mode" );
    sleep 10;
    if ($mode eq 'up') {
        print_err("release_file_lock($lock_name)");
        release_file_lock($lock_name);
    }
    return 1;
}

sub catmedia_front_lock_name {
    return "catmedia_front_lock";
}

sub get_svn_cmd {
    my (%prm) = @_;
    my $ya_path = $prm{ya_path} // $ya;
    return "$ya_path tool svn </dev/null";
}

1;
