package ObjLib::BaseLogger;
use strict;

use utf8;
use open ':utf8';

use base qw(ObjLib::Obj);

use Scalar::Util;
use Data::Dumper;
use Utils::Sys qw(mem_usage log_msg_fmt);

use constant ERROR => 40;
use constant WARNING => 30;
use constant INFO => 20;
use constant DEBUG => 10;

my %loglevel_str = (
    ERROR() => 'ERROR',
    WARNING() => 'WARNING',
    INFO() => 'INFO',
    DEBUG() => 'DEBUG',
);
my %loglevel_int = reverse %loglevel_str;

# устанавливаем loglevel глобально
my $LOGLEVEL;

if (defined $ENV{BM_LOGLEVEL}) {
    $LOGLEVEL = $ENV{BM_LOGLEVEL};
} elsif (defined $ENV{BM_LOGLEVEL_STR}) {
    $LOGLEVEL = $loglevel_int{ $ENV{BM_LOGLEVEL_STR} };
} else {
    $LOGLEVEL = INFO;
}

sub init {
    my $self = shift;
    $self->{preprocess} //= {};
}

sub set_loglevel {
    my $class = shift;
    my $level = shift;
    $LOGLEVEL = $level;
}

sub get_loglevel {
    my $class = shift;
    return $LOGLEVEL;
}

sub preprocess {
    my $self = shift;
    my $level = shift;
    my $text = shift;

    my $conf = $self->{preprocess};

    if ($conf->{loglevel}) {
        my $level_str = $loglevel_str{$level} // "LOGLEVEL $level";
        $text = "$level_str: $text";
    }
    if ($conf->{decorator}) {
        $text = $conf->{decorator}->($text);
    }
    if ($conf->{mem_usage}) {
        my $mem_str = '[MEM:' . int(mem_usage() / (1024 * 1024)) . ']';
        $text = $mem_str . "\t" . $text;
    }
    if ($conf->{log_msg_fmt}) {
        $text = log_msg_fmt($text);
    }
    if (defined $conf->{max_length}) {
        $text = substr($text, 0, $conf->{max_length});
    }
    return $text;
}

sub out {
    die "Define `out' method in derived class!\n";
}

sub freeze_recursive {
    my $obj = shift;
    my $cache = shift // {};

    my $ref = ref($obj);
    return $obj if !$ref;

    my $id = Scalar::Util::refaddr($obj);
    return $cache->{$id} if defined $cache->{$id};  # already seen

    my $res;
    if ($ref eq 'ARRAY') {
        $res = [ map { freeze_recursive($_, $cache) } @$obj ];
    } elsif ($ref eq 'HASH') {
        $res = { map { $_ => freeze_recursive($obj->{$_}, $cache) } keys %$obj };
    } elsif (Scalar::Util::blessed($obj) and $obj->can('FREEZE')) {
        $res = bless(freeze_recursive($obj->FREEZE), $ref);
    } else {
        $res = $obj;
    }
    $cache->{$id} = $res;
    return $res;
}

sub log {
    my $self = shift;
    my $level = shift;
    if ($level >= $LOGLEVEL) {
        my @texts;
        for my $msg (@_) {
            if (!ref($msg)) {
                # ok, already scalar
                push @texts, $msg;
            } else {
                local $Data::Dumper::Terse = 1;
                local $Data::Dumper::Indent = 0;
                my $to_dump = freeze_recursive($msg);
                push @texts, Data::Dumper::Dumper($to_dump);
            }
        }
        my $text = $self->preprocess($level, join(' ', @texts));
        $self->out($text);
    }
}

sub error {
    my $self = shift;
    $self->log(ERROR, @_);
}

sub warning {
    my $self = shift;
    $self->log(WARNING, @_);
}

sub info {
    my $self = shift;
    $self->log(INFO, @_);
}

sub debug {
    my $self = shift;
    $self->log(DEBUG, @_);
}

sub is_error { return (ERROR >= $LOGLEVEL); }
sub is_warning { return (WARNING >= $LOGLEVEL); }
sub is_info { return (INFO >= $LOGLEVEL); }
sub is_debug { return (DEBUG >= $LOGLEVEL); }

1;
