package ObjLib::ListObj;

# Абстрактный класс, от которого должны наследовать модули проекта
# Реализует доступ к объекту проекта

use std;
use base qw(ObjLib::ProjPart);
use Digest::MD5 qw(md5_hex);
use Utils::Sys qw( h2top );

use Utils::Sys; # для md5

########################################################
#Конструктор
########################################################

########################################################
# Интерфейс
########################################################

#  list_arrayref                      ! нужно переопределить, получение указателя на массив элементов
#  new_listobj                        ! нужно переопределить, создание нового объекта для работы со списками, должен уметь обрабатывать массивы
#  list2text                          ! нужно переопределить, получение текста из списка
#
#  new_list                             создание нового объекта для работы со списками, должен уметь обрабатывать указатели на массивы
#                                  
#  lfirst                               получить первый элемент
#  list                                 получение массива элементов
#  lmap                                 $lst = $lst->lmap(sub { $_, $_ }); вариант map для списка, возвращает массив элементов
#  lgrep                                $lst = $lst->lgrep(sub { $_->val > 5 }); вариант grep для списка 
#  lpush                                добавляем массив элементов (не списков)
#  llpush                               добавляем массив списков
#  divide                               ($lst_yes, $lst_no) = $lst->divide(sub { $_->val > 5 }); разделение по условию на 2 списка
#  triple                               ($r<0, 0, $r>0) = $lst->triple(sub { $_->val <=> 13 }); разделение по сравнению на 3 списка
#  split2hash                           my $h = $lst->split2hash(sub { $_->norm_phr }); получить хэш списков по вычисляемому ключу
#  pack_list                            $lst = $lst->pack_list(sub { $_->norm_phr }); убираем повторы в списке
#  smart_pack_list                      $lst = $lst->smart_pack_list(sub { $_->norm_phr }); убираем повторы, беря лучший среди повторяющихся
#  smart_best_item                    ! нужно переопределить, возвращает лучший элемент, используется в smart_pack_list
#  shuffle_list                         случайно перетасовывает список
#  splice_list                          получить часть списка splice_list($offset, $length)
#  get_frequent_top                     получить элементы, соответствующие самым частотным вариантам
#                                           $lst = $lst->get_frequent_top(sub { $_->urt_tmpl }, 0.9, 0.8, 3)
#                                           где 0.9 - порог от предыдущего, 0.8 - порог от первого места, 3 - абсолютный порог
                                   
#  take_every_third                     возвращает каждый n-й элемент, если не указано значение - каждый третий
#  distributed_elems                    возвращает указанное количество равномерно распределённых элементов, если не указано - 30
                                   
#  split_by_count                       разбивает на массив списков по указанному количеству элементов, возвращает массив (не указатель) списков
#  get_list_stat(sub1, sub2)            если только sub1 - возвращает указатель на хэш, где ключ - результат sub1, значение - количество элементов
#                                       если указана sub2, то ключ - результат sub1, значение - количество разных sub2
                                   
#  lsort                                сортирует по значениям, полученным от функции
#  lnsort                               сортирует как числа по значениям, полученным от функции
#  reverse                              меняет порядок элементов на обратный
#  rand_n($cc)                          сколько-то случайных
#  rand200                              200 случайных
#  first200                             200 первых
#  lhead($n)                            взять первые n элементов, если параметр не указан - 10
#  ltail($n)                            взять последние n элементов, если параметр не указан - 10
#  get_sublist($from, $to)              взять элементы с $from по $to, нумерация с 0

#  debug_print                          специальный отладочный вывод с префиксом

########################################################
#Доступ к полям
########################################################


#Метод получения указателя на массив элементов, должен быть переопределен
#Если при создании объекта использовать 'list_arrayref', то можно не переопределять 
sub list_arrayref {
    my ($self) = @_;
    return $self->{'new_list_arrayref'} if defined $self->{'new_list_arrayref'};
    $self->{'new_list_arrayref'} = [ @{$self->{'list_arrayref'} || []} ]; #Создаём новый массив, так как ссылки на старый позволяют его менять извне
    delete($self->{'list_arrayref'}); #Удаляем ссылку на старый, чтобы не мешать очистке памяти
    return $self->{'new_list_arrayref'}; 
} 

sub new_listobj { } #Создание нового объекта для работы со списками, должен уметь обрабатывать массивы
sub new_list { my $self = shift; $self->new_listobj(@_); } 
sub list2text { return 'list2text' } #Получение текста из списка

sub list { return @{ $_[0]->list_arrayref } } #Метод получения массива элементов

sub lfirst { return $_[0]->list_arrayref->[0] }

sub count {
    my $self = shift;
    my @a = @$self;
    return 0+@a;
}

#На любителя
sub size { $_[0]->count; }
sub number_of_elems { $_[0]->count; }

sub lmap($&) {
    my ($self, $func) = @_;
    return $self->new_listobj( [ map { &$func } @$self ] ) ;
}

sub llmap($&) {
    my ($self, $func) = @_;
    return $self->new_listobj( [ map { $_->list } map { &$func } @$self ] ) ;
}

sub lgrep($&) {
    my ($self, $tgrep) = @_;
#    return $self->new_listobj( [ grep { &$tgrep; } @$self ] ) ;
    my @arr = ();
    for( @$self ){
        my $el = $_;
        next unless &$tgrep;
        push(@arr, $el);
    }
    return $self->new_listobj( \@arr );
}

sub lsort($&) {
    my ($self, $tgrep) = @_;
    my @arr = map { [ $_, &$tgrep ] } @$self;
    @arr = sort { $a->[1] cmp $b->[1] } @arr;
    @arr = map { $_->[0] } @arr;
    return $self->new_listobj( \@arr ) ;
}

sub lnsort($&) {
    my ($self, $tgrep) = @_;
    my @arr = map { [ $_, &$tgrep ] } @$self;
    @arr = sort { $a->[1] <=> $b->[1] } @arr;
    @arr = map { $_->[0] } @arr;
    return $self->new_listobj( \@arr ) ;
}

sub lpush {
    my ($self, @newels) = @_;
    return $self->new_listobj( [ @$self, @newels ] );
}

sub llpush {
    my ($self, @newlsts) = @_;
    return $self->lpush( map {@$_} @newlsts );
}

sub reverse {
    my ($self) = @_;
    return $self->new_listobj( [ reverse @$self ] );
}

sub divide($&){
    my ($self, $func) = @_;
    my (@arr1, @arr2) = ();
    #&$func ? push(@arr1, $_) : push(@arr2, $_) for @$self;
    for( @$self ){
        my $el = $_;
        &$func ? push(@arr1, $el) : push(@arr2, $el);
        #$self->proj->dd([ $el, $_]), print Dumper({ title => $el->title, body => $el->body, phrases => [ map {$_->text} @{$el->phl} ]}) unless ref $_;
    }
    return $self->new_listobj(\@arr1), $self->new_listobj(\@arr2); 
}

sub triple($&){
    my ($self, $func) = @_;
    my (@arr1, @arr2, @arr3) = ();
    #&$func ? push(@arr1, $_) : push(@arr2, $_) for @$self;
    for( @$self ){
        my $el = $_;
        my $res = &$func;
        if($res < 0){
            push(@arr1, $el);
        }elsif($res > 0){
            push(@arr3, $el);
        }else{
            push(@arr2, $el);
        }
        #$self->proj->dd([ $el, $_]), print Dumper({ title => $el->title, body => $el->body, phrases => [ map {$_->text} @{$el->phl} ]}) unless ref $_;
    }
    return $self->new_listobj(\@arr1), $self->new_listobj(\@arr2), $self->new_listobj(\@arr3); 
}

sub divide2($&){
    my ($self, $func) = @_;
    return $self->new_listobj([ grep {&$func} @$self]), $self->new_listobj([ grep {! &$func} @$self]); 
}

sub split2hash {
    my ($self, $func) = @_;
    my %h = ();
    for( @$self ){
        my $el = $_;
        my $k = &$func;
        $h{$k} ||= [];
        push(@{$h{$k}}, $el);
    }
    $h{$_} = $self->new_listobj($h{$_}) for keys %h;
    return \%h;
}

sub pack_list {
    my ($self, $func) = @_;
    return $self->new_listobj([ values %{{ map { $_->[1] => $_->[0] } map { [ $_, &$func ] } @$self }} ]);
}

sub smart_best_item {
    my ($self) = @_;
    return undef unless $self->count; 
    return $self->list_arrayref->[0];
}

sub smart_pack_list {
    my ($self, $func) = @_;
    my $h = $self->split2hash($func);
    return $self->new_listobj([ map { $_->smart_best_obj } values %$h ]);
}

sub shuffle_list { 
    my ($self) = @_;
    no warnings;
    my @arr = map { $_->[1] } sort {$a->[0] cmp $b->[0]}
        map { [ md5_hex("".\$_), $_ ] } @$self;
    return $self->new_listobj(\@arr);
}

sub first200 {
    my ($self) = @_;
    return $self->splice_list(0, 200);
}

sub rand200 {
    my ($self) = @_;
    return $self->rand_items(200);
}

sub rand_n {
    my ($self, $cc) = @_;
    return $self->rand_items($cc);
}

sub splice_list {
    my ($self, $offset, $length) = @_;
    my @arr = @$self;
    if($length){
        @arr = splice(@arr, $offset, $length);
    }else{
        @arr = splice(@arr, $offset);
    }
    return $self->new_listobj(\@arr); 
}

sub lhead {
    my ($self, $n) = @_;
    $n ||= 10;
    return $self->splice_list(0, $n);
}

sub ltail {
    my ($self, $n) = @_;
    $n ||= 10;
    return $self->splice_list(0, -$n);
}

sub get_sublist {
    my ($self, $from, $to) = @_;
    my $cc = $self->count;
    return $self->new_listobj if $from > $cc;
    $to = $cc if $to > $cc;
    return $self->new_listobj([@{$self->list_arrayref}[$from .. $to]]);    
}

sub rand_items {
    my ($self, $cnt) = @_;
    $cnt ||= 100;
    return $self if $self->count <= $cnt;
    return $self->shuffle_list->splice_list(0, $cnt);
}

sub get_frequent_top {
    my ($self, $func, $prvlim, $toplim, $abslim, $cntlim) = @_;
    my $h = $self->split2hash($func);
    #$self->proj->dd($h);
    my $hh = {};
    $hh->{$_} = $h->{$_}->count for keys %$h;
    my @arr = h2top($hh, $prvlim, $toplim, $abslim);
    if ((defined $cntlim) and ($cntlim < scalar @arr)) {
        @arr = @arr[0 .. ($cntlim - 1)]; # Let's skip sorting because it's not strictly required and already performed in current version
    }
#$self->proj->dd(\@arr);
    return $self->new_listobj([ map { $h->{$_->[0]}->list }  @arr ]);   
}

sub get_list_stat {
    my ($self, $func, $func2) = @_;
    unless($func2){
        my $h = {};
        $h->{&$func}++ for @$self;
        return $h;
    }
    my $hh = {};
    $hh->{&$func}{&$func2}++ for @$self;
    $hh->{$_} = keys %{ $hh->{$_} } for keys %$hh;
    return $hh; 
}

sub split_by_count {
    my ($self, $length) = @_;
    $length ||= 1000;
    my @res = ();
    my @arr = @$self;
    while( my @sarr = splice(@arr, 0, $length) ){
        push(@res, $self->new_listobj(\@sarr));
    }
    return @res; 
}

#возвращает каждый n-й элемент
sub take_every_third {
    my ($self, $nmb) = @_;
    $nmb ||= 3;
    my $count = $self->count;
    my @arr = ();
    my $i = 0;
    for my $el (@$self){
        $i++;
        next if $i % $nmb;
        push(@arr, $el);
    }
    return $self->new_listobj(\@arr);
}

#возвращает указанное количество равномерно распределённых элементов, если не указано - 30
sub distributed_elems {
    my ($self, $ttl) = @_;
    $ttl ||= 30;
    my $count = $self->count;
    my $nmb = int( $count / $ttl );
    my $lst = $self->take_every_third($nmb)->lhead($ttl);
    return $lst;
}

#Очистка кэшей, прокидываем до элементов
sub _delete_cached_inf {
    my ($self) = @_;
    $_->_delete_cached_inf for @$self;
    $self->SUPER::_delete_cached_inf; 
}

#Специальный отладочный вывод с префиксом
sub debug_print {
    my ($self, $pref, $fh) = @_;
    $pref ||= 'dpnt';
    #print $fh "=================== $pref BEG ===================\n";
    if(defined $fh){
        print $fh "$pref: $_\n" for @$self;
    }else{
        print "$pref: $_\n" for @$self;
    }
    #print $fh "=================== $pref END ===================\n";
    return $self;
}

use overload
    "+" => sub {
             my ($self, $other, $swap) = @_;
             $other = $self->new_listobj($other) if (ref( $other ) eq 'ARRAY') || (! ref $other );
             return $self->new_listobj([ $self->list, $other->list ] );
        },
    '@{}' => sub {
             my ($self) = @_;
             return [@{$self->list_arrayref}];
        },
    'bool' => sub {
             my ($self) = @_;
             return $self->count;
        },
    '""' => sub { 
             my ($self) = @_;
             return $self->list2text;
        };


1;
