package ObjLib::UserAgent;
use strict;

use utf8;
use open ':encoding(utf8)';

use Net::HTTP;
use LWP::UserAgent;

use base qw(LWP::UserAgent);

use Utils::Sys qw(
    print_err
    print_log
    do_safely
);

use Utils::Urls qw(get_sec_level_domain);

sub patch_HTTP_methods {
    no warnings "all";
    *Net::HTTP::Methods::read_entity_body = \&read_entity_body;
    *Net::HTTP::Methods::my_read          = \&my_read;
    use warnings "all";

#    print STDERR "Patching done!\n";
    return 1;
}

sub init {
    my ($self) = @_;
    patch_HTTP_methods;
    $self->add_handler(response_done => \&_response_handler);
}

sub hard_timeout {
    my $self = shift;
    if (@_) {
        $self->{hard_timeout} = $_[0];
    }
    return $self->{hard_timeout};
}

# Если hard_timeout задан, то выполняется LWP::UserAgent::request с дополнительным ограничением по времени
# Если hard_timeout не задан, то выполняется LWP::UserAgent::request
sub request {
    my ($self, @args) = @_;

    my $res;
    if ($self->hard_timeout) {
        $res = do_safely(
            sub { LWP::UserAgent::request($self, @args) },
            timeout => $self->hard_timeout,
            no_die  => 1,
        );
    } else {
        $res = LWP::UserAgent::request($self, @args);
    }

    return $res;
}



################################
#HTTP module HOTFIXES
################################

sub my_read {
    die if @_ > 3;
    my $self = shift;
    my $len = $_[1];
    for (${*$self}{'http_buf'}) {
	    if (length) {
	        $_[0] = substr($_, 0, $len, "");
	        return length($_[0]);
	    } else {
#FIX PLACE
        # bug	 return $self->sysread($_[0], $len);
            my $n = $self->sysread($_[0], $len);
            redo if $!{EINTR};
            if ($!{EAGAIN}) {
                my $mask = '';
                vec($mask,$self->fileno,1) = 1;
                select($mask,undef,undef, ${*$self}{io_socket_timeout} || 0.1);
                redo;
            }
            return $n;
	    }
    }
}

sub read_entity_body {
    my $self = shift;
    my $buf_ref = \$_[0];
    my $size = $_[1];
    die "Offset not supported yet" if $_[2];

    my $chunked;
    my $bytes;

    if (${*$self}{'http_first_body'}) {
	${*$self}{'http_first_body'} = 0;
	delete ${*$self}{'http_chunked'};
	delete ${*$self}{'http_bytes'};
	my $method = shift(@{${*$self}{'http_request_method'}});
	my $status = ${*$self}{'http_status'};
	if ($method eq "HEAD") {
	    # this response is always empty regardless of other headers
	    $bytes = 0;
	}
	elsif (my $te = ${*$self}{'http_te'}) {

	    my @te = split(/\s*,\s*/, lc($te));
	    die "Chunked must be last Transfer-Encoding '$te'"
		unless pop(@te) eq "chunked";

	    for (@te) {
		if ($_ eq "deflate" && inflate_ok()) {
		    #require Compress::Raw::Zlib;
		    my ($i, $status) = Compress::Raw::Zlib::Inflate->new();
		    die "Can't make inflator: $status" unless $i;
		    $_ = sub { my $out; $i->inflate($_[0], \$out); $out }
		}
		elsif ($_ eq "gzip" && gunzip_ok()) {
		    #require IO::Uncompress::Gunzip;
		    my @buf;
		    $_ = sub {
			push(@buf, $_[0]);
			return "" unless $_[1];
			my $input = join("", @buf);
			my $output;
			IO::Uncompress::Gunzip::gunzip(\$input, \$output, Transparent => 0)
			    or die "Can't gunzip content: $IO::Uncompress::Gunzip::GunzipError";
			return \$output;
		    };
		}
		elsif ($_ eq "identity") {
		    $_ = sub { $_[0] };
		}
		else {
		    die "Can't handle transfer encoding '$te'";
		}
	    }

	    @te = reverse(@te);

	    ${*$self}{'http_te2'} = @te ? \@te : "";
	    $chunked = -1;
	}
	elsif (defined(my $content_length = ${*$self}{'http_content_length'})) {
	    $bytes = $content_length;
	}
        elsif ($status =~ /^(?:1|[23]04)/) {
            # RFC 2616 says that these responses should always be empty
            # but that does not appear to be true in practice [RT#17907]
            $bytes = 0;
        }
	else {
	    # XXX Multi-Part types are self delimiting, but RFC 2616 says we
	    # only has to deal with 'multipart/byteranges'

	    # Read until EOF
	}
    }
    else {
	$chunked = ${*$self}{'http_chunked'};
	$bytes   = ${*$self}{'http_bytes'};
    }

    if (defined $chunked) {
	# The state encoded in $chunked is:
	#   $chunked == 0:   read CRLF after chunk, then chunk header
        #   $chunked == -1:  read chunk header
	#   $chunked > 0:    bytes left in current chunk to read

	if ($chunked <= 0) {
	    my $line = Net::HTTP::Methods::my_readline($self, 'Entity body');
	    if ($chunked == 0) {
		die "Missing newline after chunk data: '$line'"
		    if !defined($line) || $line ne "";
		$line = Net::HTTP::Methods::my_readline($self, 'Entity body');
	    }
	    die "EOF when chunk header expected" unless defined($line);
	    my $chunk_len = $line;
	    $chunk_len =~ s/;.*//;  # ignore potential chunk parameters
	    unless ($chunk_len =~ /^([\da-fA-F]+)\s*$/) {
		die "Bad chunk-size in HTTP response: $line";
	    }
	    $chunked = hex($1);
	    if ($chunked == 0) {
		${*$self}{'http_trailers'} = [$self->_read_header_lines];
		$$buf_ref = "";

		my $n = 0;
		if (my $transforms = delete ${*$self}{'http_te2'}) {
		    for (@$transforms) {
			$$buf_ref = &$_($$buf_ref, 1);
		    }
		    $n = length($$buf_ref);
		}

		# in case somebody tries to read more, make sure we continue
		# to return EOF
		delete ${*$self}{'http_chunked'};
		${*$self}{'http_bytes'} = 0;

		return $n;
	    }
	}

	my $n = $chunked;
	$n = $size if $size && $size < $n;
	$n = Net::HTTP::Methods::my_read($self, $$buf_ref, $n);

#FIX_PLACE
	${*$self}{'http_chunked'} = $chunked - $n || 0;
    return undef unless defined $n;

# bug    ${*$self}{'http_chunked'} = $chunked - $n;

	if ($n > 0) {
	    if (my $transforms = ${*$self}{'http_te2'}) {
		for (@$transforms) {
		    $$buf_ref = &$_($$buf_ref, 0);
		}
		$n = length($$buf_ref);
		$n = -1 if $n == 0;
	    }
	}
	return $n;
    }
    elsif (defined $bytes) {
	unless ($bytes) {
	    $$buf_ref = "";
	    return 0;
	}
	my $n = $bytes;
	$n = $size if $size && $size < $n;
	$n = Net::HTTP::Methods::my_read($self, $$buf_ref, $n);
	return undef unless defined $n;
	${*$self}{'http_bytes'} = $bytes - $n;
	return $n;
    }
    else {
	# read until eof
	$size ||= 8*1024;
	return Net::HTTP::Methods::my_read($self, $$buf_ref, $size);
    }
}

sub forbid_outgoing_redirects {
    my $self = shift;
    if (@_) {
        $self->{forbid_outgoing_redirects} = $_[0];
    }
    return $self->{forbid_outgoing_redirects} // 0;
}

#хендлер с отключаемой блокировкой редиректов на другие домены (по дефолту отключена, включается только в get_internal_subpages). Нужен в обходчике, чтобы не скликивать рекламу
sub _response_handler {
    my ($res, $ua, $h) = @_;
    if ( $ua->forbid_outgoing_redirects && $res->is_redirect && get_sec_level_domain($res->{_request}{_uri}) ne get_sec_level_domain($res->{_headers}{location}) ) {

        #клонируем предыдущий ответ
        my $referral = $res->clone;
        #клонирование не клонирует previous, поэтому выставляем ссылку сами
        $referral->previous($res->previous);

        #отдаем пустой ответ, прицепляем к нему цепь предыдущих запросов, чтобы вызывающий код мог корректно обработать редирект
        $res->code(403);
        $res->message('Outgoing redirects are forbidden by ObjLib::UserAgent::forbid_outgoing_redirects');
        $res->previous($referral);
        $res->content('');
    }
}

#HOTFIXES ended


1;
