package StaticZmap;

use 5.008;
use strict;
use warnings;
use Utils::Common;
use Utils::Sys;
use File::Copy;
use File::Temp qw/tempfile/;
use File::Path qw/make_path/;
use Data::Dumper;
use Fcntl qw/:flock/;

require DynaLoader;

# Подключаем собранную so-шку по абсолютному пути
use Cwd "abs_path";
push @DynaLoader::dl_library_path, abs_path(__FILE__ . "/../Utils/StaticZmap");

our @ISA = qw(DynaLoader);
our $VERSION = '0.01';

bootstrap StaticZmap $VERSION;

sub set_missed_options {
    my $options = shift;

    $options->{elements_delim} //= "\n";
    $options->{fields_delim} //= "\t";
    $options->{words_delim} //= "/";
    $options->{keep_key_unencoded} = $options->{keep_key_unencoded} ? '--keep-key-unencoded' : '';
}

sub get_work_filename {
    my ($data_file, %options) = @_;

    set_missed_options(\%options);

    return Utils::Sys::md5int("$data_file\tid=$options{id}")
        if $options{id};

    my $options_text = join("\t", map {$_ . '=' . $options{$_}} sort keys %options);
    return Utils::Sys::md5int("$data_file\t$options_text");
}

sub build {
    my ($data_file, %options) = @_;

    set_missed_options(\%options);

    my $temp_dir = $Utils::Common::options->{static_zmap_params}{temp_dir};
    my $work_dir = $Utils::Common::options->{static_zmap_params}{work_dir};
    my $proj_dir = $Utils::Common::options->{static_zmap_params}{proj_dir};
    # do_sys_cmd parameter
    my $silent = 1;
    
    # create temp and work directories
    make_path $temp_dir unless -e $temp_dir;
    make_path $work_dir unless -e $work_dir;

    my $work_file = $options{target_file} ?
                    $options{target_file} :
                    "$work_dir/" . get_work_filename($data_file, %options);
    my $lock_file = "$temp_dir/" . Utils::Sys::md5int($work_file) . '.lock';

    open (my $lock_file_handle, '>', $lock_file)
        or die("can't write $lock_file");
    flock($lock_file_handle, LOCK_EX);

    # check necessary files
    if (-e $work_file && (stat $data_file)[9] < (stat $work_file)[9]) {
        my $static_zmap = StaticZmap->new($work_file, $options{words_delim}, $options{keep_key_unencoded});
        flock($lock_file_handle, LOCK_UN);
        return $static_zmap;
    }

    # get name for temp files and create them
    my $temp_file = Utils::Sys::get_tempfile('', DIR => $temp_dir, UNLINK => 1);
    my $header_file = Utils::Sys::get_tempfile('header', DIR => $temp_dir, UNLINK => 1);
    my $encoded_file = Utils::Sys::get_tempfile('encoded', DIR => $temp_dir, UNLINK => 1);
    my $huffman_tree_file = Utils::Sys::get_tempfile('huffman_tree', DIR => $temp_dir, UNLINK => 1);
    my $sorted_file = Utils::Sys::get_tempfile('sorted', DIR => $temp_dir, UNLINK => 1);
    my $transformed_data_file = Utils::Sys::get_tempfile('', DIR => $temp_dir, UNLINK => 1);
    if ($options{transform_function}) {
        $options{transform_function}->($data_file, $transformed_data_file);
    } else {
        $transformed_data_file = $data_file;
    }
    # run encode_data
    Utils::Sys::do_sys_cmd(
            "$proj_dir/encode_data \\
            --src-file=$transformed_data_file \\
            --header-file=$header_file \\
            --dst-file=$encoded_file \\
            --huffman-tree-file=$huffman_tree_file \\
            --src-ed=\$'$options{elements_delim}' \\
            --src-fd=\$'$options{fields_delim}' \\
            --src-wd=\$'$options{words_delim}' \\
            $options{keep_key_unencoded}",
            silent => $silent);
    # remove transformed file if created
    Utils::Sys::do_sys_cmd("rm $transformed_data_file", silent => $silent) if $options{transform_function};
    # sort encoded file
    Utils::Sys::do_sys_cmd("LC_ALL=C sort $encoded_file > $sorted_file", silent => $silent);
    # remove encoded file
    Utils::Sys::do_sys_cmd("rm $encoded_file", silent => $silent);
    # merge 3 files
    Utils::Sys::do_sys_cmd("cat $header_file > $temp_file", silent => $silent);
    Utils::Sys::do_sys_cmd("rm $header_file", silent => $silent);
    Utils::Sys::do_sys_cmd("cat $huffman_tree_file >> $temp_file", silent => $silent);
    Utils::Sys::do_sys_cmd("rm $huffman_tree_file", silent => $silent);
    Utils::Sys::do_sys_cmd("cat $sorted_file >> $temp_file", silent => $silent);
    Utils::Sys::do_sys_cmd("rm $sorted_file", silent => $silent);
    # move to work
    Utils::Sys::do_sys_cmd("mv $temp_file $work_file", silent => $silent);
    # set rights
    Utils::Sys::do_sys_cmd("chmod 644 $work_file", silent => $silent);
    my $static_zmap = StaticZmap->new($work_file, $options{words_delim}, $options{keep_key_unencoded});
    flock($lock_file_handle, LOCK_UN);
    
    return $static_zmap;
}

1;
