package Utils::Array;

use strict;
use base qw(Exporter);
use Data::Dumper;
use List::Util qw(sum min max);
use JSON qw(to_json);

our @EXPORT = (
    'split_array'           ,       #Разбивает массив хешей на хеш массивов хешей по какому-либо полю
    'sum_array'             ,       #Сжимает массив хешей до хеша, суммируя указанные поля
    'array_intersection'    ,       #Пересечение массивов
    'array_equal'           ,       #Возвращает true если 2 массива одинаковые иначе false; сортировка важна!
    'in_array'              ,       #Проверяет вхождение элемента в массив
    'uniq_array'            ,       #Возвращает уникальные элементы массива
    'is_intersect'          ,       #Проверяет пересечение двух массивов
    'sum_array_elements'    ,       #Суммирует весь вход
    'uniq_array_ref'        ,       #Возвращает уникальные элементы массива взятого по ссылке
    'uniq_array_ref_deep'   ,       #Возвращает уникальные элементы массива взятого по ссылке, поддерживает сложны струтуры
    'sort_strings_with_numbers'     ,    #Сортировка массива со строками вида ('mirror1', 'mirror2', 'mirror10')
    'array_permutations'    ,       #Все перестановки элементов массива, возвращает ссылку на двумерный массив
    'array_subsets_ordered'  ,       #Возврашает все варианты перестановок элементов массива с разным числом элементов и сохранением их порядка a b c => a, a b, b c, a b c
    'pairs_func'            ,       #[arr,func] Перебирает пары элементов из arr, выполняя для них func
    'product'         ,              #Декартово произведение входных массивов
    'array_minus'         ,              #Классическая разность 2 массивов: елементы первого, к-рые не содержатся во втором
    # векторные операции
    'dim',                          # Размерность вектора
    'norm',                         # Норма вектора
    'add',                          # Сложить два вектора
    'subtract',                     # Вычесть из первого вектора второй
    'dot',                          # Скалярное произведение векторов
    'distance',                     # Расстояние между векторами
    # numpy-like операции с массивами
    'mean',                         # Среднее арифметическое элементов массива
    'std',                          # Среднеквадратичное отклонение
    'median',                       # медиана

    'mix',                          # смесь массивов в разных пропорциях
);
our @EXPORT_OK = @EXPORT;

=head1 Методы

=cut

=head2 split_array

B<Параметры:> 1) $ ссылка на массив 2) $ с полем

B<Возвращаемое значение:>

Разбивает массив хешей на хеш массивов хешей по какому-либо полю

=cut

our $unordered_subsets_bit_masks;


sub split_array {
    my $array = shift;
    my $field = shift;

    my %hash     = ();
    for (@$array) {
        my $key = $_->{$field};
        if(exists $hash{$key}){
            push( @{$hash{$key}}, $_ );
        }else{
            $hash{$key} = [$_];
        }
    }

    return \%hash;
}

=head2 sum_array 

B<Параметры:>

B<Возвращаемое значение:>k

Сжимает массив хешей до хеша, суммируя указанные поля

=cut

sub sum_array {
    my ($array, $fields) = @_;

    # если поля не указаны, суммируем все
    unless ( defined $fields ){
        my %hallfields = ();
        for my $h ( @$array ){
            $hallfields{$_}++ for ( keys %$h );
        }
        $fields = [ grep { $_ } keys %hallfields ];
    }

    my %hash;
    $hash{$_} = 0 for @$fields;

    for my $h (@$array){
        my $affected_fields = array_intersection( $fields, [keys %$h] );
        $hash{$_} += $h->{$_} for(@$affected_fields);
    }

#    $hash{$_} = (int( $hash{$_} * 1000000 + 0.5) /1000000) for @$fields;
#    $hash{$_} = (int( $hash{$_} * 100 + 0.5) /100) for @$fields;

    return \%hash;
}

=head2 sum_array_elements

B<Параметры:> 1) @

B<Возвращаемое значение:> 1) $ с суммой всех элементов массива

=cut

sub sum_array_elements {
    my $res=0;
    $res+=$_ for (@_);
    return $res;
}

=head2 array_intersection

B<Параметры:>

B<Возвращаемое значение:>

Пересечение массивов

=cut


sub array_intersection {
  my %t = ();
  $t{$_}++ for map { uniq_array( @$_ ) } @_;
  return [ grep { $t{$_} == @_ } keys %t ];
}


=head2 is_intersect 

B<Параметры:>

B<Возвращаемое значение:>

Пересечение массивов

=cut

sub is_intersect {


  my $m1 = shift;
  my $m2 = shift;

  my @m = ();

  my %t = ();
  for(@$m1){ $t{$_}++; }
  for(@$m2){ return 1 if exists($t{$_}); }

  return 0;
}


sub array_minus {
  my $m1 = shift;
  my $m2 = shift;
  return [ grep { !in_array($_, $m2) } @$m1 ];
}

=head2 array_equal

B<Параметры:>

B<Возвращаемое значение:>

#Возвращает true если 2 массива одинаковые иначе false
#from http://perldoc.perl.org/perlfaq4.html#How-do-I-test-whether-two-arrays-or-hashes-are-equal?

=cut

sub array_equal {
    my ($first, $second) = @_;
    no warnings; # silence spurious -w undef complaints
    return 0 unless @$first == @$second;
    for (my $i = 0; $i < @$first; $i++) {
        return 0 if $first->[$i] ne $second->[$i];
    }
    return 1;
}

=head2 in_array

B<Параметры:> 1) $ с элементом 2) $ со ссылкой на массив

B<Возвращаемое значение:> 1) 1 если элемент есть в массиве; иначе 0

=cut

sub in_array {
  my ($el, $ar) = @_;

  return 0 unless defined $el;

  for (@$ar) {
    return 1 if $_ eq $el;
  }
  return 0;
}

=head2 uniq_array

B<Параметры:> 1) @

B<Возвращаемое значение:> 1) @

Сохраняет порядок следования элементов

=cut

sub uniq_array {
    my %h;
    return map { $h{$_}++ == 0 ? $_ : () } @_;
}

=head2 uniq_array_ref

B<Параметры:> 1) $ с ссылкой на список

B<Возвращаемое значение:> 1) $ с ссылкой на список

Сохраняет порядок следования элементов

=cut

sub uniq_array_ref {
    my $ref = shift;
    my %h;
    return [  map { $h{$_}++ == 0 ? $_ : () } @$ref ];
}

=head2 sort_strings_with_numbers

B<Параметры:> 1) @ со строками, которые нужно отсортировать 

B<Возвращаемое значение:> 1) @ с отсортированными строками

Иногда есть спики с элементами вида "значение_число". И есть необходимость их расположить в порядке
возрастания чисел. Для решения этой задачи написана эта саба.

    LogDump sort_strings_with_numbers('mirror2', 'mirror10', 'mirror1');
    $VAR1 = 'mirror1';
    $VAR2 = 'mirror2';
    $VAR3 = 'mirror10';

Решение подсмотрено на http://www.perlmonks.org/?node_id=483462 и http://www.perlmonks.org/?node_id=492976

=cut

sub sort_strings_with_numbers {

    my @sorted = @_[
            map { unpack "N", substr($_,-4) }
            sort
            map {
                my $key = $_[$_];
                $key =~ s[(\d+)][ pack "N", $1 ]ge;
                $key . pack "CNN", 0, 0, $_
            } 0..$#_
    ];

    return @sorted;
}

=head2 uniq_array_ref_deep

B<Параметры:> 1) $ - ссылка на массив с произвольными структурами, которые можно перегнать в json

B<Возвращаемое значение:> 1) $ - ссылка на массив уникальных значений из входного массива

=cut

sub uniq_array_ref_deep {
    my $arr = shift;
    my $h = {};
    my @res;
    for my $row (@$arr) {
        my $json_row = to_json($row, {canonical => 1, utf8 => 1});
        unless ($h->{$json_row}) {
            push @res, $row;
            $h->{$json_row} = 1;
        }
    }
    return \@res;

}

=head2 array_permutations

B<Параметры:> 1) $, ссылка на одномерный массив элементов

B<Возвращаемое значение:> $, ссылка на двумерный массив, сожержащий все комбинации элементов исходного массива

Все перестановки элементов массива

=cut
sub array_permutations {
    my ($arr, $p, $res) = @_;
    # внутренние параметры
    $p = [ () ] unless $p;
    $res = [ () ] unless $res;

    if ( @$p == @$arr ) {
        #print STDERR join(" ", @$p), "\n";
        push @$res, [@$p];
    }

L:  for my $i (0..$#$arr) {
        for my $j (0..$#$p) {
            next L if $$arr[$i] eq $$p[$j];
        }
        push @$p, $$arr[$i];
        array_permutations ($arr, $p, $res);
        pop @$p;
    }
    return $res;
}

sub array_subsets_ordered {
    my $arr = shift;
    my @res = ('');

    for my $el ( @$arr ) {
        my @temp = ();
        for my $r ( @res ) {
            push @temp, $r;
            push @temp, $r ? "$r $el" : "$el";
        }
        @res = @temp;
    }
    return [ @res[1..$#res] ];
}

sub pairs_func {
    my $arr = shift;
    my $func = shift;
    my $n = @$arr;
    for my $e1 (0 .. $n-1){
        for my $e2 ($e1+1 .. $n-1){
            $func->($arr->[$e1], $arr->[$e2]);
        }
    }
}

=head2 product

B<Параметры:> @, массив ссылок на одномерные массивы

B<Возвращаемое значение:> $, ссылка на многомерный массив

Декартово произведение массивов

=cut
sub product {
    my @arrays = @_;

    my $dim = scalar @arrays;
    my @curr_prod = ([]);
    for my $i (0..$dim-1) {
        my @new_prod;
        for my $current_point (@curr_prod) {
            for my $new_set ($arrays[$i]) {
                for (@$new_set) {
                    push @new_prod, [@$current_point, $_];
                }
            }
        }
        @curr_prod = @new_prod;
    }

    return \@curr_prod;
}

# векторные операции
sub dim {
    my $x = shift;
    return scalar @$x;
}

sub _equal_dimensions {
    # вспомогательная функция
    # для заданного спискам векторов проверяет, совпадают ли их размерности
    my @vectors = @_;

    my %sizes;
    for my $vector (@vectors) {
        $sizes{dim($vector)} = 1;
    }

    return int(scalar(keys %sizes) == 1);
}

sub norm {
    my $x = shift;
    my $p = shift;  # число или 'inf'

    $p //= 2;
    return undef if $p <= 0;
    if ($p eq 'inf') {
        return max(map {abs($_)} @$x);    
    } else {
        return (sum(map {abs($_)**$p} @$x))**(1/$p);
    }
}

sub add {
    my $x = shift;
    my $y = shift;

    return undef unless _equal_dimensions($x, $y);
    return [map {$x->[$_] + $y->[$_]} 0..dim($x)-1];
}

sub subtract {
    my $x = shift;
    my $y = shift;

    return undef unless _equal_dimensions($x, $y);
    return [map {$x->[$_] - $y->[$_]} 0..dim($x)-1];
}

sub dot {
    my $x = shift;
    my $y = shift;

    return undef unless _equal_dimensions($x, $y);
    return sum_array_elements(map {$x->[$_]*$y->[$_]} 0..dim($x)-1);
}

sub distance {
    my $x = shift;
    my $y = shift;
    my $distance_type = shift;
    $distance_type //= 'cosine';

    return undef unless _equal_dimensions($x, $y);
    if ($distance_type eq 'cosine') {
        return 1 - dot($x, $y) / norm($x) / norm($y);
    } elsif ($distance_type eq 'euclidian') {
        return norm(subtract($x, $y));
    } else {
        return undef;
    }
}

# numpy-like
sub mean {
    my $a = shift;

    return undef unless scalar @$a;
    return sum_array_elements(@$a) / @$a;
}

sub median {
    my $arr = shift;

    return undef unless scalar(@$arr);
    my @sorted = sort { $a <=> $b } @$arr;
    return $sorted[ int(@sorted/2) ];
}

sub std {
    my $a = shift;

    return undef unless scalar(@$a);
    my $Ea = mean($a);
    my $Ea2 = mean([map {$_**2} @$a]);
    return sqrt($Ea2 - $Ea**2);
}


# смешать несколько массивов, чтобы в начале элементы шли в разной пропорции
# mix([1,2,3,4,5] => 50, ['a','b','c','d','e','f'] => 40, ['X','Y','Z'] => 100);
# X 1 Y a Z 2 b 3 c 4 5 d e f
sub mix {
    my @input = @_;
    my @data;
    while (@input) {
        my $arr = shift @input;
        my $rate = shift @input;
        for my $i (1 .. @$arr) {
            push @data, [ $arr->[$i-1], $i / $rate ];
        }
    }

    return map { $_->[0] } sort { $a->[1] <=> $b->[1] } @data;
}

sub cartesian_product {
    my $array = shift;

    return () unless defined($array);
    return map {[$_]} @$array unless @_;

    my @subproduct = cartesian_product(@_);
    my @result = ();
    for my $elem (@$array) {
        push(@result, [$elem, @$_]) for @subproduct;
    }

    return @result;
}

1;
